/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dataexchange;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.dataexchange.model.AccessDeniedException;
import software.amazon.awssdk.services.dataexchange.model.CancelJobRequest;
import software.amazon.awssdk.services.dataexchange.model.CancelJobResponse;
import software.amazon.awssdk.services.dataexchange.model.ConflictException;
import software.amazon.awssdk.services.dataexchange.model.CreateDataSetRequest;
import software.amazon.awssdk.services.dataexchange.model.CreateDataSetResponse;
import software.amazon.awssdk.services.dataexchange.model.CreateJobRequest;
import software.amazon.awssdk.services.dataexchange.model.CreateJobResponse;
import software.amazon.awssdk.services.dataexchange.model.CreateRevisionRequest;
import software.amazon.awssdk.services.dataexchange.model.CreateRevisionResponse;
import software.amazon.awssdk.services.dataexchange.model.DataExchangeException;
import software.amazon.awssdk.services.dataexchange.model.DataExchangeRequest;
import software.amazon.awssdk.services.dataexchange.model.DeleteAssetRequest;
import software.amazon.awssdk.services.dataexchange.model.DeleteAssetResponse;
import software.amazon.awssdk.services.dataexchange.model.DeleteDataSetRequest;
import software.amazon.awssdk.services.dataexchange.model.DeleteDataSetResponse;
import software.amazon.awssdk.services.dataexchange.model.DeleteRevisionRequest;
import software.amazon.awssdk.services.dataexchange.model.DeleteRevisionResponse;
import software.amazon.awssdk.services.dataexchange.model.GetAssetRequest;
import software.amazon.awssdk.services.dataexchange.model.GetAssetResponse;
import software.amazon.awssdk.services.dataexchange.model.GetDataSetRequest;
import software.amazon.awssdk.services.dataexchange.model.GetDataSetResponse;
import software.amazon.awssdk.services.dataexchange.model.GetJobRequest;
import software.amazon.awssdk.services.dataexchange.model.GetJobResponse;
import software.amazon.awssdk.services.dataexchange.model.GetRevisionRequest;
import software.amazon.awssdk.services.dataexchange.model.GetRevisionResponse;
import software.amazon.awssdk.services.dataexchange.model.InternalServerException;
import software.amazon.awssdk.services.dataexchange.model.ListDataSetRevisionsRequest;
import software.amazon.awssdk.services.dataexchange.model.ListDataSetRevisionsResponse;
import software.amazon.awssdk.services.dataexchange.model.ListDataSetsRequest;
import software.amazon.awssdk.services.dataexchange.model.ListDataSetsResponse;
import software.amazon.awssdk.services.dataexchange.model.ListJobsRequest;
import software.amazon.awssdk.services.dataexchange.model.ListJobsResponse;
import software.amazon.awssdk.services.dataexchange.model.ListRevisionAssetsRequest;
import software.amazon.awssdk.services.dataexchange.model.ListRevisionAssetsResponse;
import software.amazon.awssdk.services.dataexchange.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.dataexchange.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.dataexchange.model.ResourceNotFoundException;
import software.amazon.awssdk.services.dataexchange.model.ServiceLimitExceededException;
import software.amazon.awssdk.services.dataexchange.model.StartJobRequest;
import software.amazon.awssdk.services.dataexchange.model.StartJobResponse;
import software.amazon.awssdk.services.dataexchange.model.TagResourceRequest;
import software.amazon.awssdk.services.dataexchange.model.TagResourceResponse;
import software.amazon.awssdk.services.dataexchange.model.ThrottlingException;
import software.amazon.awssdk.services.dataexchange.model.UntagResourceRequest;
import software.amazon.awssdk.services.dataexchange.model.UntagResourceResponse;
import software.amazon.awssdk.services.dataexchange.model.UpdateAssetRequest;
import software.amazon.awssdk.services.dataexchange.model.UpdateAssetResponse;
import software.amazon.awssdk.services.dataexchange.model.UpdateDataSetRequest;
import software.amazon.awssdk.services.dataexchange.model.UpdateDataSetResponse;
import software.amazon.awssdk.services.dataexchange.model.UpdateRevisionRequest;
import software.amazon.awssdk.services.dataexchange.model.UpdateRevisionResponse;
import software.amazon.awssdk.services.dataexchange.model.ValidationException;
import software.amazon.awssdk.services.dataexchange.paginators.ListDataSetRevisionsIterable;
import software.amazon.awssdk.services.dataexchange.paginators.ListDataSetsIterable;
import software.amazon.awssdk.services.dataexchange.paginators.ListJobsIterable;
import software.amazon.awssdk.services.dataexchange.paginators.ListRevisionAssetsIterable;
import software.amazon.awssdk.services.dataexchange.transform.CancelJobRequestMarshaller;
import software.amazon.awssdk.services.dataexchange.transform.CreateDataSetRequestMarshaller;
import software.amazon.awssdk.services.dataexchange.transform.CreateJobRequestMarshaller;
import software.amazon.awssdk.services.dataexchange.transform.CreateRevisionRequestMarshaller;
import software.amazon.awssdk.services.dataexchange.transform.DeleteAssetRequestMarshaller;
import software.amazon.awssdk.services.dataexchange.transform.DeleteDataSetRequestMarshaller;
import software.amazon.awssdk.services.dataexchange.transform.DeleteRevisionRequestMarshaller;
import software.amazon.awssdk.services.dataexchange.transform.GetAssetRequestMarshaller;
import software.amazon.awssdk.services.dataexchange.transform.GetDataSetRequestMarshaller;
import software.amazon.awssdk.services.dataexchange.transform.GetJobRequestMarshaller;
import software.amazon.awssdk.services.dataexchange.transform.GetRevisionRequestMarshaller;
import software.amazon.awssdk.services.dataexchange.transform.ListDataSetRevisionsRequestMarshaller;
import software.amazon.awssdk.services.dataexchange.transform.ListDataSetsRequestMarshaller;
import software.amazon.awssdk.services.dataexchange.transform.ListJobsRequestMarshaller;
import software.amazon.awssdk.services.dataexchange.transform.ListRevisionAssetsRequestMarshaller;
import software.amazon.awssdk.services.dataexchange.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.dataexchange.transform.StartJobRequestMarshaller;
import software.amazon.awssdk.services.dataexchange.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.dataexchange.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.dataexchange.transform.UpdateAssetRequestMarshaller;
import software.amazon.awssdk.services.dataexchange.transform.UpdateDataSetRequestMarshaller;
import software.amazon.awssdk.services.dataexchange.transform.UpdateRevisionRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link DataExchangeClient}.
 *
 * @see DataExchangeClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultDataExchangeClient implements DataExchangeClient {
    private static final Logger log = Logger.loggerFor(DefaultDataExchangeClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultDataExchangeClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * This operation cancels a job. Jobs can be cancelled only when they are in the WAITING state.
     * </p>
     *
     * @param cancelJobRequest
     * @return Result of the CancelJob operation returned by the service.
     * @throws ResourceNotFoundException
     *         404 response
     * @throws ThrottlingException
     *         429 response
     * @throws ValidationException
     *         400 response
     * @throws InternalServerException
     *         500 response
     * @throws ConflictException
     *         409 response
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DataExchangeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DataExchangeClient.CancelJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dataexchange-2017-07-25/CancelJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CancelJobResponse cancelJob(CancelJobRequest cancelJobRequest) throws ResourceNotFoundException, ThrottlingException,
            ValidationException, InternalServerException, ConflictException, AwsServiceException, SdkClientException,
            DataExchangeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CancelJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CancelJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataExchange");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelJob");

            return clientHandler.execute(new ClientExecutionParams<CancelJobRequest, CancelJobResponse>()
                    .withOperationName("CancelJob").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(cancelJobRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new CancelJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation creates a data set.
     * </p>
     *
     * @param createDataSetRequest
     *        The request body for CreateDataSet.
     * @return Result of the CreateDataSet operation returned by the service.
     * @throws ThrottlingException
     *         429 response
     * @throws ValidationException
     *         400 response
     * @throws InternalServerException
     *         500 response
     * @throws ServiceLimitExceededException
     *         402 response
     * @throws AccessDeniedException
     *         403 response
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DataExchangeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DataExchangeClient.CreateDataSet
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dataexchange-2017-07-25/CreateDataSet" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateDataSetResponse createDataSet(CreateDataSetRequest createDataSetRequest) throws ThrottlingException,
            ValidationException, InternalServerException, ServiceLimitExceededException, AccessDeniedException,
            AwsServiceException, SdkClientException, DataExchangeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateDataSetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateDataSetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDataSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataExchange");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDataSet");

            return clientHandler.execute(new ClientExecutionParams<CreateDataSetRequest, CreateDataSetResponse>()
                    .withOperationName("CreateDataSet").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createDataSetRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateDataSetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation creates a job.
     * </p>
     *
     * @param createJobRequest
     *        The request body for CreateJob.
     * @return Result of the CreateJob operation returned by the service.
     * @throws ResourceNotFoundException
     *         404 response
     * @throws ThrottlingException
     *         429 response
     * @throws ValidationException
     *         400 response
     * @throws InternalServerException
     *         500 response
     * @throws AccessDeniedException
     *         403 response
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DataExchangeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DataExchangeClient.CreateJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dataexchange-2017-07-25/CreateJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateJobResponse createJob(CreateJobRequest createJobRequest) throws ResourceNotFoundException, ThrottlingException,
            ValidationException, InternalServerException, AccessDeniedException, AwsServiceException, SdkClientException,
            DataExchangeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataExchange");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateJob");

            return clientHandler.execute(new ClientExecutionParams<CreateJobRequest, CreateJobResponse>()
                    .withOperationName("CreateJob").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createJobRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new CreateJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation creates a revision for a data set.
     * </p>
     *
     * @param createRevisionRequest
     *        The request body for CreateRevision.
     * @return Result of the CreateRevision operation returned by the service.
     * @throws ResourceNotFoundException
     *         404 response
     * @throws ThrottlingException
     *         429 response
     * @throws ValidationException
     *         400 response
     * @throws InternalServerException
     *         500 response
     * @throws AccessDeniedException
     *         403 response
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DataExchangeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DataExchangeClient.CreateRevision
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dataexchange-2017-07-25/CreateRevision" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateRevisionResponse createRevision(CreateRevisionRequest createRevisionRequest) throws ResourceNotFoundException,
            ThrottlingException, ValidationException, InternalServerException, AccessDeniedException, AwsServiceException,
            SdkClientException, DataExchangeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateRevisionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateRevisionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createRevisionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataExchange");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateRevision");

            return clientHandler.execute(new ClientExecutionParams<CreateRevisionRequest, CreateRevisionResponse>()
                    .withOperationName("CreateRevision").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createRevisionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateRevisionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation deletes an asset.
     * </p>
     *
     * @param deleteAssetRequest
     * @return Result of the DeleteAsset operation returned by the service.
     * @throws ValidationException
     *         400 response
     * @throws InternalServerException
     *         500 response
     * @throws AccessDeniedException
     *         403 response
     * @throws ResourceNotFoundException
     *         404 response
     * @throws ThrottlingException
     *         429 response
     * @throws ConflictException
     *         409 response
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DataExchangeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DataExchangeClient.DeleteAsset
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dataexchange-2017-07-25/DeleteAsset" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteAssetResponse deleteAsset(DeleteAssetRequest deleteAssetRequest) throws ValidationException,
            InternalServerException, AccessDeniedException, ResourceNotFoundException, ThrottlingException, ConflictException,
            AwsServiceException, SdkClientException, DataExchangeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteAssetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteAssetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAssetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataExchange");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAsset");

            return clientHandler.execute(new ClientExecutionParams<DeleteAssetRequest, DeleteAssetResponse>()
                    .withOperationName("DeleteAsset").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteAssetRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteAssetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation deletes a data set.
     * </p>
     *
     * @param deleteDataSetRequest
     * @return Result of the DeleteDataSet operation returned by the service.
     * @throws ValidationException
     *         400 response
     * @throws InternalServerException
     *         500 response
     * @throws AccessDeniedException
     *         403 response
     * @throws ResourceNotFoundException
     *         404 response
     * @throws ThrottlingException
     *         429 response
     * @throws ConflictException
     *         409 response
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DataExchangeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DataExchangeClient.DeleteDataSet
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dataexchange-2017-07-25/DeleteDataSet" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteDataSetResponse deleteDataSet(DeleteDataSetRequest deleteDataSetRequest) throws ValidationException,
            InternalServerException, AccessDeniedException, ResourceNotFoundException, ThrottlingException, ConflictException,
            AwsServiceException, SdkClientException, DataExchangeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteDataSetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteDataSetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDataSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataExchange");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDataSet");

            return clientHandler.execute(new ClientExecutionParams<DeleteDataSetRequest, DeleteDataSetResponse>()
                    .withOperationName("DeleteDataSet").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteDataSetRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteDataSetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation deletes a revision.
     * </p>
     *
     * @param deleteRevisionRequest
     * @return Result of the DeleteRevision operation returned by the service.
     * @throws ValidationException
     *         400 response
     * @throws InternalServerException
     *         500 response
     * @throws AccessDeniedException
     *         403 response
     * @throws ResourceNotFoundException
     *         404 response
     * @throws ThrottlingException
     *         429 response
     * @throws ConflictException
     *         409 response
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DataExchangeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DataExchangeClient.DeleteRevision
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dataexchange-2017-07-25/DeleteRevision" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteRevisionResponse deleteRevision(DeleteRevisionRequest deleteRevisionRequest) throws ValidationException,
            InternalServerException, AccessDeniedException, ResourceNotFoundException, ThrottlingException, ConflictException,
            AwsServiceException, SdkClientException, DataExchangeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteRevisionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteRevisionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteRevisionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataExchange");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteRevision");

            return clientHandler.execute(new ClientExecutionParams<DeleteRevisionRequest, DeleteRevisionResponse>()
                    .withOperationName("DeleteRevision").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteRevisionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteRevisionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation returns information about an asset.
     * </p>
     *
     * @param getAssetRequest
     * @return Result of the GetAsset operation returned by the service.
     * @throws ResourceNotFoundException
     *         404 response
     * @throws ThrottlingException
     *         429 response
     * @throws ValidationException
     *         400 response
     * @throws InternalServerException
     *         500 response
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DataExchangeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DataExchangeClient.GetAsset
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dataexchange-2017-07-25/GetAsset" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetAssetResponse getAsset(GetAssetRequest getAssetRequest) throws ResourceNotFoundException, ThrottlingException,
            ValidationException, InternalServerException, AwsServiceException, SdkClientException, DataExchangeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetAssetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetAssetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAssetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataExchange");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAsset");

            return clientHandler.execute(new ClientExecutionParams<GetAssetRequest, GetAssetResponse>()
                    .withOperationName("GetAsset").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getAssetRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new GetAssetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation returns information about a data set.
     * </p>
     *
     * @param getDataSetRequest
     * @return Result of the GetDataSet operation returned by the service.
     * @throws ResourceNotFoundException
     *         404 response
     * @throws ThrottlingException
     *         429 response
     * @throws ValidationException
     *         400 response
     * @throws InternalServerException
     *         500 response
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DataExchangeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DataExchangeClient.GetDataSet
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dataexchange-2017-07-25/GetDataSet" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetDataSetResponse getDataSet(GetDataSetRequest getDataSetRequest) throws ResourceNotFoundException,
            ThrottlingException, ValidationException, InternalServerException, AwsServiceException, SdkClientException,
            DataExchangeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetDataSetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetDataSetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDataSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataExchange");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDataSet");

            return clientHandler
                    .execute(new ClientExecutionParams<GetDataSetRequest, GetDataSetResponse>().withOperationName("GetDataSet")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getDataSetRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetDataSetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation returns information about a job.
     * </p>
     *
     * @param getJobRequest
     * @return Result of the GetJob operation returned by the service.
     * @throws ResourceNotFoundException
     *         404 response
     * @throws ThrottlingException
     *         429 response
     * @throws ValidationException
     *         400 response
     * @throws InternalServerException
     *         500 response
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DataExchangeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DataExchangeClient.GetJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dataexchange-2017-07-25/GetJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetJobResponse getJob(GetJobRequest getJobRequest) throws ResourceNotFoundException, ThrottlingException,
            ValidationException, InternalServerException, AwsServiceException, SdkClientException, DataExchangeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataExchange");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetJob");

            return clientHandler.execute(new ClientExecutionParams<GetJobRequest, GetJobResponse>().withOperationName("GetJob")
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(getJobRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new GetJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation returns information about a revision.
     * </p>
     *
     * @param getRevisionRequest
     * @return Result of the GetRevision operation returned by the service.
     * @throws ResourceNotFoundException
     *         404 response
     * @throws ThrottlingException
     *         429 response
     * @throws ValidationException
     *         400 response
     * @throws InternalServerException
     *         500 response
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DataExchangeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DataExchangeClient.GetRevision
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dataexchange-2017-07-25/GetRevision" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetRevisionResponse getRevision(GetRevisionRequest getRevisionRequest) throws ResourceNotFoundException,
            ThrottlingException, ValidationException, InternalServerException, AwsServiceException, SdkClientException,
            DataExchangeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetRevisionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetRevisionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getRevisionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataExchange");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRevision");

            return clientHandler.execute(new ClientExecutionParams<GetRevisionRequest, GetRevisionResponse>()
                    .withOperationName("GetRevision").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getRevisionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetRevisionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation lists a data set's revisions sorted by CreatedAt in descending order.
     * </p>
     *
     * @param listDataSetRevisionsRequest
     * @return Result of the ListDataSetRevisions operation returned by the service.
     * @throws ResourceNotFoundException
     *         404 response
     * @throws ThrottlingException
     *         429 response
     * @throws ValidationException
     *         400 response
     * @throws InternalServerException
     *         500 response
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DataExchangeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DataExchangeClient.ListDataSetRevisions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dataexchange-2017-07-25/ListDataSetRevisions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListDataSetRevisionsResponse listDataSetRevisions(ListDataSetRevisionsRequest listDataSetRevisionsRequest)
            throws ResourceNotFoundException, ThrottlingException, ValidationException, InternalServerException,
            AwsServiceException, SdkClientException, DataExchangeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListDataSetRevisionsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListDataSetRevisionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDataSetRevisionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataExchange");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDataSetRevisions");

            return clientHandler.execute(new ClientExecutionParams<ListDataSetRevisionsRequest, ListDataSetRevisionsResponse>()
                    .withOperationName("ListDataSetRevisions").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listDataSetRevisionsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListDataSetRevisionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation lists a data set's revisions sorted by CreatedAt in descending order.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listDataSetRevisions(software.amazon.awssdk.services.dataexchange.model.ListDataSetRevisionsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.dataexchange.paginators.ListDataSetRevisionsIterable responses = client.listDataSetRevisionsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.dataexchange.paginators.ListDataSetRevisionsIterable responses = client
     *             .listDataSetRevisionsPaginator(request);
     *     for (software.amazon.awssdk.services.dataexchange.model.ListDataSetRevisionsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.dataexchange.paginators.ListDataSetRevisionsIterable responses = client.listDataSetRevisionsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listDataSetRevisions(software.amazon.awssdk.services.dataexchange.model.ListDataSetRevisionsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listDataSetRevisionsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         404 response
     * @throws ThrottlingException
     *         429 response
     * @throws ValidationException
     *         400 response
     * @throws InternalServerException
     *         500 response
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DataExchangeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DataExchangeClient.ListDataSetRevisions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dataexchange-2017-07-25/ListDataSetRevisions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListDataSetRevisionsIterable listDataSetRevisionsPaginator(ListDataSetRevisionsRequest listDataSetRevisionsRequest)
            throws ResourceNotFoundException, ThrottlingException, ValidationException, InternalServerException,
            AwsServiceException, SdkClientException, DataExchangeException {
        return new ListDataSetRevisionsIterable(this, applyPaginatorUserAgent(listDataSetRevisionsRequest));
    }

    /**
     * <p>
     * This operation lists your data sets. When listing by origin OWNED, results are sorted by CreatedAt in descending
     * order. When listing by origin ENTITLED, there is no order and the maxResults parameter is ignored.
     * </p>
     *
     * @param listDataSetsRequest
     * @return Result of the ListDataSets operation returned by the service.
     * @throws ResourceNotFoundException
     *         404 response
     * @throws ThrottlingException
     *         429 response
     * @throws ValidationException
     *         400 response
     * @throws InternalServerException
     *         500 response
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DataExchangeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DataExchangeClient.ListDataSets
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dataexchange-2017-07-25/ListDataSets" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListDataSetsResponse listDataSets(ListDataSetsRequest listDataSetsRequest) throws ResourceNotFoundException,
            ThrottlingException, ValidationException, InternalServerException, AwsServiceException, SdkClientException,
            DataExchangeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListDataSetsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListDataSetsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDataSetsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataExchange");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDataSets");

            return clientHandler.execute(new ClientExecutionParams<ListDataSetsRequest, ListDataSetsResponse>()
                    .withOperationName("ListDataSets").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listDataSetsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListDataSetsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation lists your data sets. When listing by origin OWNED, results are sorted by CreatedAt in descending
     * order. When listing by origin ENTITLED, there is no order and the maxResults parameter is ignored.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listDataSets(software.amazon.awssdk.services.dataexchange.model.ListDataSetsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.dataexchange.paginators.ListDataSetsIterable responses = client.listDataSetsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.dataexchange.paginators.ListDataSetsIterable responses = client
     *             .listDataSetsPaginator(request);
     *     for (software.amazon.awssdk.services.dataexchange.model.ListDataSetsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.dataexchange.paginators.ListDataSetsIterable responses = client.listDataSetsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listDataSets(software.amazon.awssdk.services.dataexchange.model.ListDataSetsRequest)} operation.</b>
     * </p>
     *
     * @param listDataSetsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         404 response
     * @throws ThrottlingException
     *         429 response
     * @throws ValidationException
     *         400 response
     * @throws InternalServerException
     *         500 response
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DataExchangeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DataExchangeClient.ListDataSets
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dataexchange-2017-07-25/ListDataSets" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListDataSetsIterable listDataSetsPaginator(ListDataSetsRequest listDataSetsRequest) throws ResourceNotFoundException,
            ThrottlingException, ValidationException, InternalServerException, AwsServiceException, SdkClientException,
            DataExchangeException {
        return new ListDataSetsIterable(this, applyPaginatorUserAgent(listDataSetsRequest));
    }

    /**
     * <p>
     * This operation lists your jobs sorted by CreatedAt in descending order.
     * </p>
     *
     * @param listJobsRequest
     * @return Result of the ListJobs operation returned by the service.
     * @throws ResourceNotFoundException
     *         404 response
     * @throws ThrottlingException
     *         429 response
     * @throws ValidationException
     *         400 response
     * @throws InternalServerException
     *         500 response
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DataExchangeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DataExchangeClient.ListJobs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dataexchange-2017-07-25/ListJobs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListJobsResponse listJobs(ListJobsRequest listJobsRequest) throws ResourceNotFoundException, ThrottlingException,
            ValidationException, InternalServerException, AwsServiceException, SdkClientException, DataExchangeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListJobsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListJobsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataExchange");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListJobs");

            return clientHandler.execute(new ClientExecutionParams<ListJobsRequest, ListJobsResponse>()
                    .withOperationName("ListJobs").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listJobsRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new ListJobsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation lists your jobs sorted by CreatedAt in descending order.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listJobs(software.amazon.awssdk.services.dataexchange.model.ListJobsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.dataexchange.paginators.ListJobsIterable responses = client.listJobsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.dataexchange.paginators.ListJobsIterable responses = client.listJobsPaginator(request);
     *     for (software.amazon.awssdk.services.dataexchange.model.ListJobsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.dataexchange.paginators.ListJobsIterable responses = client.listJobsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listJobs(software.amazon.awssdk.services.dataexchange.model.ListJobsRequest)} operation.</b>
     * </p>
     *
     * @param listJobsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         404 response
     * @throws ThrottlingException
     *         429 response
     * @throws ValidationException
     *         400 response
     * @throws InternalServerException
     *         500 response
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DataExchangeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DataExchangeClient.ListJobs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dataexchange-2017-07-25/ListJobs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListJobsIterable listJobsPaginator(ListJobsRequest listJobsRequest) throws ResourceNotFoundException,
            ThrottlingException, ValidationException, InternalServerException, AwsServiceException, SdkClientException,
            DataExchangeException {
        return new ListJobsIterable(this, applyPaginatorUserAgent(listJobsRequest));
    }

    /**
     * <p>
     * This operation lists a revision's assets sorted alphabetically in descending order.
     * </p>
     *
     * @param listRevisionAssetsRequest
     * @return Result of the ListRevisionAssets operation returned by the service.
     * @throws ResourceNotFoundException
     *         404 response
     * @throws ThrottlingException
     *         429 response
     * @throws ValidationException
     *         400 response
     * @throws InternalServerException
     *         500 response
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DataExchangeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DataExchangeClient.ListRevisionAssets
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dataexchange-2017-07-25/ListRevisionAssets"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListRevisionAssetsResponse listRevisionAssets(ListRevisionAssetsRequest listRevisionAssetsRequest)
            throws ResourceNotFoundException, ThrottlingException, ValidationException, InternalServerException,
            AwsServiceException, SdkClientException, DataExchangeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListRevisionAssetsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListRevisionAssetsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRevisionAssetsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataExchange");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRevisionAssets");

            return clientHandler.execute(new ClientExecutionParams<ListRevisionAssetsRequest, ListRevisionAssetsResponse>()
                    .withOperationName("ListRevisionAssets").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listRevisionAssetsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListRevisionAssetsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation lists a revision's assets sorted alphabetically in descending order.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listRevisionAssets(software.amazon.awssdk.services.dataexchange.model.ListRevisionAssetsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.dataexchange.paginators.ListRevisionAssetsIterable responses = client.listRevisionAssetsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.dataexchange.paginators.ListRevisionAssetsIterable responses = client
     *             .listRevisionAssetsPaginator(request);
     *     for (software.amazon.awssdk.services.dataexchange.model.ListRevisionAssetsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.dataexchange.paginators.ListRevisionAssetsIterable responses = client.listRevisionAssetsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listRevisionAssets(software.amazon.awssdk.services.dataexchange.model.ListRevisionAssetsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listRevisionAssetsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         404 response
     * @throws ThrottlingException
     *         429 response
     * @throws ValidationException
     *         400 response
     * @throws InternalServerException
     *         500 response
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DataExchangeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DataExchangeClient.ListRevisionAssets
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dataexchange-2017-07-25/ListRevisionAssets"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListRevisionAssetsIterable listRevisionAssetsPaginator(ListRevisionAssetsRequest listRevisionAssetsRequest)
            throws ResourceNotFoundException, ThrottlingException, ValidationException, InternalServerException,
            AwsServiceException, SdkClientException, DataExchangeException {
        return new ListRevisionAssetsIterable(this, applyPaginatorUserAgent(listRevisionAssetsRequest));
    }

    /**
     * <p>
     * This operation lists the tags on the resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DataExchangeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DataExchangeClient.ListTagsForResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dataexchange-2017-07-25/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws AwsServiceException, SdkClientException, DataExchangeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataExchange");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation starts a job.
     * </p>
     *
     * @param startJobRequest
     * @return Result of the StartJob operation returned by the service.
     * @throws ValidationException
     *         400 response
     * @throws InternalServerException
     *         500 response
     * @throws AccessDeniedException
     *         403 response
     * @throws ResourceNotFoundException
     *         404 response
     * @throws ThrottlingException
     *         429 response
     * @throws ConflictException
     *         409 response
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DataExchangeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DataExchangeClient.StartJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dataexchange-2017-07-25/StartJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StartJobResponse startJob(StartJobRequest startJobRequest) throws ValidationException, InternalServerException,
            AccessDeniedException, ResourceNotFoundException, ThrottlingException, ConflictException, AwsServiceException,
            SdkClientException, DataExchangeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StartJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataExchange");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartJob");

            return clientHandler.execute(new ClientExecutionParams<StartJobRequest, StartJobResponse>()
                    .withOperationName("StartJob").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(startJobRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new StartJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation tags a resource.
     * </p>
     *
     * @param tagResourceRequest
     *        The request body for TagResource.
     * @return Result of the TagResource operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DataExchangeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DataExchangeClient.TagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dataexchange-2017-07-25/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws AwsServiceException, SdkClientException,
            DataExchangeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataExchange");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation removes one or more tags from a resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DataExchangeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DataExchangeClient.UntagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dataexchange-2017-07-25/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws AwsServiceException,
            SdkClientException, DataExchangeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataExchange");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation updates an asset.
     * </p>
     *
     * @param updateAssetRequest
     *        The request body for UpdateAsset.
     * @return Result of the UpdateAsset operation returned by the service.
     * @throws ValidationException
     *         400 response
     * @throws InternalServerException
     *         500 response
     * @throws AccessDeniedException
     *         403 response
     * @throws ResourceNotFoundException
     *         404 response
     * @throws ThrottlingException
     *         429 response
     * @throws ConflictException
     *         409 response
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DataExchangeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DataExchangeClient.UpdateAsset
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dataexchange-2017-07-25/UpdateAsset" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateAssetResponse updateAsset(UpdateAssetRequest updateAssetRequest) throws ValidationException,
            InternalServerException, AccessDeniedException, ResourceNotFoundException, ThrottlingException, ConflictException,
            AwsServiceException, SdkClientException, DataExchangeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateAssetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateAssetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateAssetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataExchange");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateAsset");

            return clientHandler.execute(new ClientExecutionParams<UpdateAssetRequest, UpdateAssetResponse>()
                    .withOperationName("UpdateAsset").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateAssetRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateAssetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation updates a data set.
     * </p>
     *
     * @param updateDataSetRequest
     *        The request body for UpdateDataSet.
     * @return Result of the UpdateDataSet operation returned by the service.
     * @throws ResourceNotFoundException
     *         404 response
     * @throws ThrottlingException
     *         429 response
     * @throws ValidationException
     *         400 response
     * @throws InternalServerException
     *         500 response
     * @throws AccessDeniedException
     *         403 response
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DataExchangeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DataExchangeClient.UpdateDataSet
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dataexchange-2017-07-25/UpdateDataSet" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateDataSetResponse updateDataSet(UpdateDataSetRequest updateDataSetRequest) throws ResourceNotFoundException,
            ThrottlingException, ValidationException, InternalServerException, AccessDeniedException, AwsServiceException,
            SdkClientException, DataExchangeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateDataSetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateDataSetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateDataSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataExchange");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDataSet");

            return clientHandler.execute(new ClientExecutionParams<UpdateDataSetRequest, UpdateDataSetResponse>()
                    .withOperationName("UpdateDataSet").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateDataSetRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateDataSetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation updates a revision.
     * </p>
     *
     * @param updateRevisionRequest
     *        The request body for UpdateRevision.
     * @return Result of the UpdateRevision operation returned by the service.
     * @throws ValidationException
     *         400 response
     * @throws InternalServerException
     *         500 response
     * @throws AccessDeniedException
     *         403 response
     * @throws ResourceNotFoundException
     *         404 response
     * @throws ThrottlingException
     *         429 response
     * @throws ConflictException
     *         409 response
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DataExchangeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DataExchangeClient.UpdateRevision
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dataexchange-2017-07-25/UpdateRevision" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateRevisionResponse updateRevision(UpdateRevisionRequest updateRevisionRequest) throws ValidationException,
            InternalServerException, AccessDeniedException, ResourceNotFoundException, ThrottlingException, ConflictException,
            AwsServiceException, SdkClientException, DataExchangeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateRevisionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateRevisionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateRevisionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataExchange");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateRevision");

            return clientHandler.execute(new ClientExecutionParams<UpdateRevisionRequest, UpdateRevisionResponse>()
                    .withOperationName("UpdateRevision").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateRevisionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateRevisionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(DataExchangeException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceLimitExceededException")
                                .exceptionBuilderSupplier(ServiceLimitExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends DataExchangeRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
