/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.databasemigration.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides information that defines a PostgreSQL endpoint.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PostgreSQLSettings implements SdkPojo, Serializable,
        ToCopyableBuilder<PostgreSQLSettings.Builder, PostgreSQLSettings> {
    private static final SdkField<String> AFTER_CONNECT_SCRIPT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AfterConnectScript").getter(getter(PostgreSQLSettings::afterConnectScript))
            .setter(setter(Builder::afterConnectScript))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AfterConnectScript").build())
            .build();

    private static final SdkField<Boolean> CAPTURE_DDLS_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("CaptureDdls").getter(getter(PostgreSQLSettings::captureDdls)).setter(setter(Builder::captureDdls))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CaptureDdls").build()).build();

    private static final SdkField<Integer> MAX_FILE_SIZE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("MaxFileSize").getter(getter(PostgreSQLSettings::maxFileSize)).setter(setter(Builder::maxFileSize))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxFileSize").build()).build();

    private static final SdkField<String> DATABASE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DatabaseName").getter(getter(PostgreSQLSettings::databaseName)).setter(setter(Builder::databaseName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DatabaseName").build()).build();

    private static final SdkField<String> DDL_ARTIFACTS_SCHEMA_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DdlArtifactsSchema").getter(getter(PostgreSQLSettings::ddlArtifactsSchema))
            .setter(setter(Builder::ddlArtifactsSchema))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DdlArtifactsSchema").build())
            .build();

    private static final SdkField<Integer> EXECUTE_TIMEOUT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("ExecuteTimeout").getter(getter(PostgreSQLSettings::executeTimeout))
            .setter(setter(Builder::executeTimeout))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExecuteTimeout").build()).build();

    private static final SdkField<Boolean> FAIL_TASKS_ON_LOB_TRUNCATION_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN).memberName("FailTasksOnLobTruncation")
            .getter(getter(PostgreSQLSettings::failTasksOnLobTruncation)).setter(setter(Builder::failTasksOnLobTruncation))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FailTasksOnLobTruncation").build())
            .build();

    private static final SdkField<Boolean> HEARTBEAT_ENABLE_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("HeartbeatEnable").getter(getter(PostgreSQLSettings::heartbeatEnable))
            .setter(setter(Builder::heartbeatEnable))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HeartbeatEnable").build()).build();

    private static final SdkField<String> HEARTBEAT_SCHEMA_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("HeartbeatSchema").getter(getter(PostgreSQLSettings::heartbeatSchema))
            .setter(setter(Builder::heartbeatSchema))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HeartbeatSchema").build()).build();

    private static final SdkField<Integer> HEARTBEAT_FREQUENCY_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("HeartbeatFrequency").getter(getter(PostgreSQLSettings::heartbeatFrequency))
            .setter(setter(Builder::heartbeatFrequency))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HeartbeatFrequency").build())
            .build();

    private static final SdkField<String> PASSWORD_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Password").getter(getter(PostgreSQLSettings::password)).setter(setter(Builder::password))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Password").build()).build();

    private static final SdkField<Integer> PORT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER).memberName("Port")
            .getter(getter(PostgreSQLSettings::port)).setter(setter(Builder::port))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Port").build()).build();

    private static final SdkField<String> SERVER_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ServerName").getter(getter(PostgreSQLSettings::serverName)).setter(setter(Builder::serverName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServerName").build()).build();

    private static final SdkField<String> USERNAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Username").getter(getter(PostgreSQLSettings::username)).setter(setter(Builder::username))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Username").build()).build();

    private static final SdkField<String> SLOT_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SlotName").getter(getter(PostgreSQLSettings::slotName)).setter(setter(Builder::slotName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SlotName").build()).build();

    private static final SdkField<String> PLUGIN_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PluginName").getter(getter(PostgreSQLSettings::pluginNameAsString)).setter(setter(Builder::pluginName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PluginName").build()).build();

    private static final SdkField<String> SECRETS_MANAGER_ACCESS_ROLE_ARN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("SecretsManagerAccessRoleArn")
            .getter(getter(PostgreSQLSettings::secretsManagerAccessRoleArn))
            .setter(setter(Builder::secretsManagerAccessRoleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecretsManagerAccessRoleArn")
                    .build()).build();

    private static final SdkField<String> SECRETS_MANAGER_SECRET_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SecretsManagerSecretId").getter(getter(PostgreSQLSettings::secretsManagerSecretId))
            .setter(setter(Builder::secretsManagerSecretId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecretsManagerSecretId").build())
            .build();

    private static final SdkField<Boolean> TRIM_SPACE_IN_CHAR_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("TrimSpaceInChar").getter(getter(PostgreSQLSettings::trimSpaceInChar))
            .setter(setter(Builder::trimSpaceInChar))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TrimSpaceInChar").build()).build();

    private static final SdkField<Boolean> MAP_BOOLEAN_AS_BOOLEAN_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("MapBooleanAsBoolean").getter(getter(PostgreSQLSettings::mapBooleanAsBoolean))
            .setter(setter(Builder::mapBooleanAsBoolean))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MapBooleanAsBoolean").build())
            .build();

    private static final SdkField<Boolean> MAP_JSONB_AS_CLOB_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("MapJsonbAsClob").getter(getter(PostgreSQLSettings::mapJsonbAsClob))
            .setter(setter(Builder::mapJsonbAsClob))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MapJsonbAsClob").build()).build();

    private static final SdkField<String> MAP_LONG_VARCHAR_AS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MapLongVarcharAs").getter(getter(PostgreSQLSettings::mapLongVarcharAsAsString))
            .setter(setter(Builder::mapLongVarcharAs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MapLongVarcharAs").build()).build();

    private static final SdkField<String> DATABASE_MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DatabaseMode").getter(getter(PostgreSQLSettings::databaseModeAsString))
            .setter(setter(Builder::databaseMode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DatabaseMode").build()).build();

    private static final SdkField<String> BABELFISH_DATABASE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("BabelfishDatabaseName").getter(getter(PostgreSQLSettings::babelfishDatabaseName))
            .setter(setter(Builder::babelfishDatabaseName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BabelfishDatabaseName").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(AFTER_CONNECT_SCRIPT_FIELD,
            CAPTURE_DDLS_FIELD, MAX_FILE_SIZE_FIELD, DATABASE_NAME_FIELD, DDL_ARTIFACTS_SCHEMA_FIELD, EXECUTE_TIMEOUT_FIELD,
            FAIL_TASKS_ON_LOB_TRUNCATION_FIELD, HEARTBEAT_ENABLE_FIELD, HEARTBEAT_SCHEMA_FIELD, HEARTBEAT_FREQUENCY_FIELD,
            PASSWORD_FIELD, PORT_FIELD, SERVER_NAME_FIELD, USERNAME_FIELD, SLOT_NAME_FIELD, PLUGIN_NAME_FIELD,
            SECRETS_MANAGER_ACCESS_ROLE_ARN_FIELD, SECRETS_MANAGER_SECRET_ID_FIELD, TRIM_SPACE_IN_CHAR_FIELD,
            MAP_BOOLEAN_AS_BOOLEAN_FIELD, MAP_JSONB_AS_CLOB_FIELD, MAP_LONG_VARCHAR_AS_FIELD, DATABASE_MODE_FIELD,
            BABELFISH_DATABASE_NAME_FIELD));

    private static final long serialVersionUID = 1L;

    private final String afterConnectScript;

    private final Boolean captureDdls;

    private final Integer maxFileSize;

    private final String databaseName;

    private final String ddlArtifactsSchema;

    private final Integer executeTimeout;

    private final Boolean failTasksOnLobTruncation;

    private final Boolean heartbeatEnable;

    private final String heartbeatSchema;

    private final Integer heartbeatFrequency;

    private final String password;

    private final Integer port;

    private final String serverName;

    private final String username;

    private final String slotName;

    private final String pluginName;

    private final String secretsManagerAccessRoleArn;

    private final String secretsManagerSecretId;

    private final Boolean trimSpaceInChar;

    private final Boolean mapBooleanAsBoolean;

    private final Boolean mapJsonbAsClob;

    private final String mapLongVarcharAs;

    private final String databaseMode;

    private final String babelfishDatabaseName;

    private PostgreSQLSettings(BuilderImpl builder) {
        this.afterConnectScript = builder.afterConnectScript;
        this.captureDdls = builder.captureDdls;
        this.maxFileSize = builder.maxFileSize;
        this.databaseName = builder.databaseName;
        this.ddlArtifactsSchema = builder.ddlArtifactsSchema;
        this.executeTimeout = builder.executeTimeout;
        this.failTasksOnLobTruncation = builder.failTasksOnLobTruncation;
        this.heartbeatEnable = builder.heartbeatEnable;
        this.heartbeatSchema = builder.heartbeatSchema;
        this.heartbeatFrequency = builder.heartbeatFrequency;
        this.password = builder.password;
        this.port = builder.port;
        this.serverName = builder.serverName;
        this.username = builder.username;
        this.slotName = builder.slotName;
        this.pluginName = builder.pluginName;
        this.secretsManagerAccessRoleArn = builder.secretsManagerAccessRoleArn;
        this.secretsManagerSecretId = builder.secretsManagerSecretId;
        this.trimSpaceInChar = builder.trimSpaceInChar;
        this.mapBooleanAsBoolean = builder.mapBooleanAsBoolean;
        this.mapJsonbAsClob = builder.mapJsonbAsClob;
        this.mapLongVarcharAs = builder.mapLongVarcharAs;
        this.databaseMode = builder.databaseMode;
        this.babelfishDatabaseName = builder.babelfishDatabaseName;
    }

    /**
     * <p>
     * For use with change data capture (CDC) only, this attribute has DMS bypass foreign keys and user triggers to
     * reduce the time it takes to bulk load data.
     * </p>
     * <p>
     * Example: <code>afterConnectScript=SET session_replication_role='replica'</code>
     * </p>
     * 
     * @return For use with change data capture (CDC) only, this attribute has DMS bypass foreign keys and user triggers
     *         to reduce the time it takes to bulk load data.</p>
     *         <p>
     *         Example: <code>afterConnectScript=SET session_replication_role='replica'</code>
     */
    public final String afterConnectScript() {
        return afterConnectScript;
    }

    /**
     * <p>
     * To capture DDL events, DMS creates various artifacts in the PostgreSQL database when the task starts. You can
     * later remove these artifacts.
     * </p>
     * <p>
     * If this value is set to <code>N</code>, you don't have to create tables or triggers on the source database.
     * </p>
     * 
     * @return To capture DDL events, DMS creates various artifacts in the PostgreSQL database when the task starts. You
     *         can later remove these artifacts.</p>
     *         <p>
     *         If this value is set to <code>N</code>, you don't have to create tables or triggers on the source
     *         database.
     */
    public final Boolean captureDdls() {
        return captureDdls;
    }

    /**
     * <p>
     * Specifies the maximum size (in KB) of any .csv file used to transfer data to PostgreSQL.
     * </p>
     * <p>
     * Example: <code>maxFileSize=512</code>
     * </p>
     * 
     * @return Specifies the maximum size (in KB) of any .csv file used to transfer data to PostgreSQL.</p>
     *         <p>
     *         Example: <code>maxFileSize=512</code>
     */
    public final Integer maxFileSize() {
        return maxFileSize;
    }

    /**
     * <p>
     * Database name for the endpoint.
     * </p>
     * 
     * @return Database name for the endpoint.
     */
    public final String databaseName() {
        return databaseName;
    }

    /**
     * <p>
     * The schema in which the operational DDL database artifacts are created.
     * </p>
     * <p>
     * Example: <code>ddlArtifactsSchema=xyzddlschema;</code>
     * </p>
     * 
     * @return The schema in which the operational DDL database artifacts are created.</p>
     *         <p>
     *         Example: <code>ddlArtifactsSchema=xyzddlschema;</code>
     */
    public final String ddlArtifactsSchema() {
        return ddlArtifactsSchema;
    }

    /**
     * <p>
     * Sets the client statement timeout for the PostgreSQL instance, in seconds. The default value is 60 seconds.
     * </p>
     * <p>
     * Example: <code>executeTimeout=100;</code>
     * </p>
     * 
     * @return Sets the client statement timeout for the PostgreSQL instance, in seconds. The default value is 60
     *         seconds.</p>
     *         <p>
     *         Example: <code>executeTimeout=100;</code>
     */
    public final Integer executeTimeout() {
        return executeTimeout;
    }

    /**
     * <p>
     * When set to <code>true</code>, this value causes a task to fail if the actual size of a LOB column is greater
     * than the specified <code>LobMaxSize</code>.
     * </p>
     * <p>
     * If task is set to Limited LOB mode and this option is set to true, the task fails instead of truncating the LOB
     * data.
     * </p>
     * 
     * @return When set to <code>true</code>, this value causes a task to fail if the actual size of a LOB column is
     *         greater than the specified <code>LobMaxSize</code>.</p>
     *         <p>
     *         If task is set to Limited LOB mode and this option is set to true, the task fails instead of truncating
     *         the LOB data.
     */
    public final Boolean failTasksOnLobTruncation() {
        return failTasksOnLobTruncation;
    }

    /**
     * <p>
     * The write-ahead log (WAL) heartbeat feature mimics a dummy transaction. By doing this, it prevents idle logical
     * replication slots from holding onto old WAL logs, which can result in storage full situations on the source. This
     * heartbeat keeps <code>restart_lsn</code> moving and prevents storage full scenarios.
     * </p>
     * 
     * @return The write-ahead log (WAL) heartbeat feature mimics a dummy transaction. By doing this, it prevents idle
     *         logical replication slots from holding onto old WAL logs, which can result in storage full situations on
     *         the source. This heartbeat keeps <code>restart_lsn</code> moving and prevents storage full scenarios.
     */
    public final Boolean heartbeatEnable() {
        return heartbeatEnable;
    }

    /**
     * <p>
     * Sets the schema in which the heartbeat artifacts are created.
     * </p>
     * 
     * @return Sets the schema in which the heartbeat artifacts are created.
     */
    public final String heartbeatSchema() {
        return heartbeatSchema;
    }

    /**
     * <p>
     * Sets the WAL heartbeat frequency (in minutes).
     * </p>
     * 
     * @return Sets the WAL heartbeat frequency (in minutes).
     */
    public final Integer heartbeatFrequency() {
        return heartbeatFrequency;
    }

    /**
     * <p>
     * Endpoint connection password.
     * </p>
     * 
     * @return Endpoint connection password.
     */
    public final String password() {
        return password;
    }

    /**
     * <p>
     * Endpoint TCP port. The default is 5432.
     * </p>
     * 
     * @return Endpoint TCP port. The default is 5432.
     */
    public final Integer port() {
        return port;
    }

    /**
     * <p>
     * The host name of the endpoint database.
     * </p>
     * <p>
     * For an Amazon RDS PostgreSQL instance, this is the output of <a
     * href="https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBInstances.html"
     * >DescribeDBInstances</a>, in the
     * <code> <a href="https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_Endpoint.html">Endpoint</a>.Address</code>
     * field.
     * </p>
     * <p>
     * For an Aurora PostgreSQL instance, this is the output of <a
     * href="https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBClusters.html"
     * >DescribeDBClusters</a>, in the <code>Endpoint</code> field.
     * </p>
     * 
     * @return The host name of the endpoint database. </p>
     *         <p>
     *         For an Amazon RDS PostgreSQL instance, this is the output of <a
     *         href="https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBInstances.html"
     *         >DescribeDBInstances</a>, in the
     *         <code> <a href="https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_Endpoint.html">Endpoint</a>.Address</code>
     *         field.
     *         </p>
     *         <p>
     *         For an Aurora PostgreSQL instance, this is the output of <a
     *         href="https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBClusters.html"
     *         >DescribeDBClusters</a>, in the <code>Endpoint</code> field.
     */
    public final String serverName() {
        return serverName;
    }

    /**
     * <p>
     * Endpoint connection user name.
     * </p>
     * 
     * @return Endpoint connection user name.
     */
    public final String username() {
        return username;
    }

    /**
     * <p>
     * Sets the name of a previously created logical replication slot for a change data capture (CDC) load of the
     * PostgreSQL source instance.
     * </p>
     * <p>
     * When used with the <code>CdcStartPosition</code> request parameter for the DMS API , this attribute also makes it
     * possible to use native CDC start points. DMS verifies that the specified logical replication slot exists before
     * starting the CDC load task. It also verifies that the task was created with a valid setting of
     * <code>CdcStartPosition</code>. If the specified slot doesn't exist or the task doesn't have a valid
     * <code>CdcStartPosition</code> setting, DMS raises an error.
     * </p>
     * <p>
     * For more information about setting the <code>CdcStartPosition</code> request parameter, see <a
     * href="https://docs.aws.amazon.com/dms/latest/userguide/CHAP_Task.CDC.html#CHAP_Task.CDC.StartPoint.Native"
     * >Determining a CDC native start point</a> in the <i>Database Migration Service User Guide</i>. For more
     * information about using <code>CdcStartPosition</code>, see <a
     * href="https://docs.aws.amazon.com/dms/latest/APIReference/API_CreateReplicationTask.html"
     * >CreateReplicationTask</a>, <a
     * href="https://docs.aws.amazon.com/dms/latest/APIReference/API_StartReplicationTask.html"
     * >StartReplicationTask</a>, and <a
     * href="https://docs.aws.amazon.com/dms/latest/APIReference/API_ModifyReplicationTask.html"
     * >ModifyReplicationTask</a>.
     * </p>
     * 
     * @return Sets the name of a previously created logical replication slot for a change data capture (CDC) load of
     *         the PostgreSQL source instance. </p>
     *         <p>
     *         When used with the <code>CdcStartPosition</code> request parameter for the DMS API , this attribute also
     *         makes it possible to use native CDC start points. DMS verifies that the specified logical replication
     *         slot exists before starting the CDC load task. It also verifies that the task was created with a valid
     *         setting of <code>CdcStartPosition</code>. If the specified slot doesn't exist or the task doesn't have a
     *         valid <code>CdcStartPosition</code> setting, DMS raises an error.
     *         </p>
     *         <p>
     *         For more information about setting the <code>CdcStartPosition</code> request parameter, see <a
     *         href="https://docs.aws.amazon.com/dms/latest/userguide/CHAP_Task.CDC.html#CHAP_Task.CDC.StartPoint.Native"
     *         >Determining a CDC native start point</a> in the <i>Database Migration Service User Guide</i>. For more
     *         information about using <code>CdcStartPosition</code>, see <a
     *         href="https://docs.aws.amazon.com/dms/latest/APIReference/API_CreateReplicationTask.html"
     *         >CreateReplicationTask</a>, <a
     *         href="https://docs.aws.amazon.com/dms/latest/APIReference/API_StartReplicationTask.html"
     *         >StartReplicationTask</a>, and <a
     *         href="https://docs.aws.amazon.com/dms/latest/APIReference/API_ModifyReplicationTask.html"
     *         >ModifyReplicationTask</a>.
     */
    public final String slotName() {
        return slotName;
    }

    /**
     * <p>
     * Specifies the plugin to use to create a replication slot.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #pluginName} will
     * return {@link PluginNameValue#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #pluginNameAsString}.
     * </p>
     * 
     * @return Specifies the plugin to use to create a replication slot.
     * @see PluginNameValue
     */
    public final PluginNameValue pluginName() {
        return PluginNameValue.fromValue(pluginName);
    }

    /**
     * <p>
     * Specifies the plugin to use to create a replication slot.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #pluginName} will
     * return {@link PluginNameValue#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #pluginNameAsString}.
     * </p>
     * 
     * @return Specifies the plugin to use to create a replication slot.
     * @see PluginNameValue
     */
    public final String pluginNameAsString() {
        return pluginName;
    }

    /**
     * <p>
     * The full Amazon Resource Name (ARN) of the IAM role that specifies DMS as the trusted entity and grants the
     * required permissions to access the value in <code>SecretsManagerSecret</code>. The role must allow the
     * <code>iam:PassRole</code> action. <code>SecretsManagerSecret</code> has the value of the Amazon Web Services
     * Secrets Manager secret that allows access to the PostgreSQL endpoint.
     * </p>
     * <note>
     * <p>
     * You can specify one of two sets of values for these permissions. You can specify the values for this setting and
     * <code>SecretsManagerSecretId</code>. Or you can specify clear-text values for <code>UserName</code>,
     * <code>Password</code>, <code>ServerName</code>, and <code>Port</code>. You can't specify both. For more
     * information on creating this <code>SecretsManagerSecret</code> and the <code>SecretsManagerAccessRoleArn</code>
     * and <code>SecretsManagerSecretId</code> required to access it, see <a
     * href="https://docs.aws.amazon.com/dms/latest/userguide/CHAP_Security.html#security-iam-secretsmanager">Using
     * secrets to access Database Migration Service resources</a> in the <i>Database Migration Service User Guide</i>.
     * </p>
     * </note>
     * 
     * @return The full Amazon Resource Name (ARN) of the IAM role that specifies DMS as the trusted entity and grants
     *         the required permissions to access the value in <code>SecretsManagerSecret</code>. The role must allow
     *         the <code>iam:PassRole</code> action. <code>SecretsManagerSecret</code> has the value of the Amazon Web
     *         Services Secrets Manager secret that allows access to the PostgreSQL endpoint.</p> <note>
     *         <p>
     *         You can specify one of two sets of values for these permissions. You can specify the values for this
     *         setting and <code>SecretsManagerSecretId</code>. Or you can specify clear-text values for
     *         <code>UserName</code>, <code>Password</code>, <code>ServerName</code>, and <code>Port</code>. You can't
     *         specify both. For more information on creating this <code>SecretsManagerSecret</code> and the
     *         <code>SecretsManagerAccessRoleArn</code> and <code>SecretsManagerSecretId</code> required to access it,
     *         see <a
     *         href="https://docs.aws.amazon.com/dms/latest/userguide/CHAP_Security.html#security-iam-secretsmanager"
     *         >Using secrets to access Database Migration Service resources</a> in the <i>Database Migration Service
     *         User Guide</i>.
     *         </p>
     */
    public final String secretsManagerAccessRoleArn() {
        return secretsManagerAccessRoleArn;
    }

    /**
     * <p>
     * The full ARN, partial ARN, or friendly name of the <code>SecretsManagerSecret</code> that contains the PostgreSQL
     * endpoint connection details.
     * </p>
     * 
     * @return The full ARN, partial ARN, or friendly name of the <code>SecretsManagerSecret</code> that contains the
     *         PostgreSQL endpoint connection details.
     */
    public final String secretsManagerSecretId() {
        return secretsManagerSecretId;
    }

    /**
     * <p>
     * Use the <code>TrimSpaceInChar</code> source endpoint setting to trim data on CHAR and NCHAR data types during
     * migration. The default value is <code>true</code>.
     * </p>
     * 
     * @return Use the <code>TrimSpaceInChar</code> source endpoint setting to trim data on CHAR and NCHAR data types
     *         during migration. The default value is <code>true</code>.
     */
    public final Boolean trimSpaceInChar() {
        return trimSpaceInChar;
    }

    /**
     * <p>
     * When true, lets PostgreSQL migrate the boolean type as boolean. By default, PostgreSQL migrates booleans as
     * <code>varchar(5)</code>.
     * </p>
     * 
     * @return When true, lets PostgreSQL migrate the boolean type as boolean. By default, PostgreSQL migrates booleans
     *         as <code>varchar(5)</code>.
     */
    public final Boolean mapBooleanAsBoolean() {
        return mapBooleanAsBoolean;
    }

    /**
     * <p>
     * When true, DMS migrates JSONB values as CLOB.
     * </p>
     * 
     * @return When true, DMS migrates JSONB values as CLOB.
     */
    public final Boolean mapJsonbAsClob() {
        return mapJsonbAsClob;
    }

    /**
     * <p>
     * When true, DMS migrates LONG values as VARCHAR.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #mapLongVarcharAs}
     * will return {@link LongVarcharMappingType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #mapLongVarcharAsAsString}.
     * </p>
     * 
     * @return When true, DMS migrates LONG values as VARCHAR.
     * @see LongVarcharMappingType
     */
    public final LongVarcharMappingType mapLongVarcharAs() {
        return LongVarcharMappingType.fromValue(mapLongVarcharAs);
    }

    /**
     * <p>
     * When true, DMS migrates LONG values as VARCHAR.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #mapLongVarcharAs}
     * will return {@link LongVarcharMappingType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #mapLongVarcharAsAsString}.
     * </p>
     * 
     * @return When true, DMS migrates LONG values as VARCHAR.
     * @see LongVarcharMappingType
     */
    public final String mapLongVarcharAsAsString() {
        return mapLongVarcharAs;
    }

    /**
     * <p>
     * Specifies whether to use default or custom replication behavior for PostgreSQL-compatible endpoints. You can use
     * this setting to specify replication behavior for endpoints that require additional configuration, such as
     * Babelfish endpoints.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #databaseMode} will
     * return {@link DatabaseMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #databaseModeAsString}.
     * </p>
     * 
     * @return Specifies whether to use default or custom replication behavior for PostgreSQL-compatible endpoints. You
     *         can use this setting to specify replication behavior for endpoints that require additional configuration,
     *         such as Babelfish endpoints.
     * @see DatabaseMode
     */
    public final DatabaseMode databaseMode() {
        return DatabaseMode.fromValue(databaseMode);
    }

    /**
     * <p>
     * Specifies whether to use default or custom replication behavior for PostgreSQL-compatible endpoints. You can use
     * this setting to specify replication behavior for endpoints that require additional configuration, such as
     * Babelfish endpoints.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #databaseMode} will
     * return {@link DatabaseMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #databaseModeAsString}.
     * </p>
     * 
     * @return Specifies whether to use default or custom replication behavior for PostgreSQL-compatible endpoints. You
     *         can use this setting to specify replication behavior for endpoints that require additional configuration,
     *         such as Babelfish endpoints.
     * @see DatabaseMode
     */
    public final String databaseModeAsString() {
        return databaseMode;
    }

    /**
     * <p>
     * The Babelfish for Aurora PostgreSQL database name for the endpoint.
     * </p>
     * 
     * @return The Babelfish for Aurora PostgreSQL database name for the endpoint.
     */
    public final String babelfishDatabaseName() {
        return babelfishDatabaseName;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(afterConnectScript());
        hashCode = 31 * hashCode + Objects.hashCode(captureDdls());
        hashCode = 31 * hashCode + Objects.hashCode(maxFileSize());
        hashCode = 31 * hashCode + Objects.hashCode(databaseName());
        hashCode = 31 * hashCode + Objects.hashCode(ddlArtifactsSchema());
        hashCode = 31 * hashCode + Objects.hashCode(executeTimeout());
        hashCode = 31 * hashCode + Objects.hashCode(failTasksOnLobTruncation());
        hashCode = 31 * hashCode + Objects.hashCode(heartbeatEnable());
        hashCode = 31 * hashCode + Objects.hashCode(heartbeatSchema());
        hashCode = 31 * hashCode + Objects.hashCode(heartbeatFrequency());
        hashCode = 31 * hashCode + Objects.hashCode(password());
        hashCode = 31 * hashCode + Objects.hashCode(port());
        hashCode = 31 * hashCode + Objects.hashCode(serverName());
        hashCode = 31 * hashCode + Objects.hashCode(username());
        hashCode = 31 * hashCode + Objects.hashCode(slotName());
        hashCode = 31 * hashCode + Objects.hashCode(pluginNameAsString());
        hashCode = 31 * hashCode + Objects.hashCode(secretsManagerAccessRoleArn());
        hashCode = 31 * hashCode + Objects.hashCode(secretsManagerSecretId());
        hashCode = 31 * hashCode + Objects.hashCode(trimSpaceInChar());
        hashCode = 31 * hashCode + Objects.hashCode(mapBooleanAsBoolean());
        hashCode = 31 * hashCode + Objects.hashCode(mapJsonbAsClob());
        hashCode = 31 * hashCode + Objects.hashCode(mapLongVarcharAsAsString());
        hashCode = 31 * hashCode + Objects.hashCode(databaseModeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(babelfishDatabaseName());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PostgreSQLSettings)) {
            return false;
        }
        PostgreSQLSettings other = (PostgreSQLSettings) obj;
        return Objects.equals(afterConnectScript(), other.afterConnectScript())
                && Objects.equals(captureDdls(), other.captureDdls()) && Objects.equals(maxFileSize(), other.maxFileSize())
                && Objects.equals(databaseName(), other.databaseName())
                && Objects.equals(ddlArtifactsSchema(), other.ddlArtifactsSchema())
                && Objects.equals(executeTimeout(), other.executeTimeout())
                && Objects.equals(failTasksOnLobTruncation(), other.failTasksOnLobTruncation())
                && Objects.equals(heartbeatEnable(), other.heartbeatEnable())
                && Objects.equals(heartbeatSchema(), other.heartbeatSchema())
                && Objects.equals(heartbeatFrequency(), other.heartbeatFrequency())
                && Objects.equals(password(), other.password()) && Objects.equals(port(), other.port())
                && Objects.equals(serverName(), other.serverName()) && Objects.equals(username(), other.username())
                && Objects.equals(slotName(), other.slotName())
                && Objects.equals(pluginNameAsString(), other.pluginNameAsString())
                && Objects.equals(secretsManagerAccessRoleArn(), other.secretsManagerAccessRoleArn())
                && Objects.equals(secretsManagerSecretId(), other.secretsManagerSecretId())
                && Objects.equals(trimSpaceInChar(), other.trimSpaceInChar())
                && Objects.equals(mapBooleanAsBoolean(), other.mapBooleanAsBoolean())
                && Objects.equals(mapJsonbAsClob(), other.mapJsonbAsClob())
                && Objects.equals(mapLongVarcharAsAsString(), other.mapLongVarcharAsAsString())
                && Objects.equals(databaseModeAsString(), other.databaseModeAsString())
                && Objects.equals(babelfishDatabaseName(), other.babelfishDatabaseName());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("PostgreSQLSettings").add("AfterConnectScript", afterConnectScript())
                .add("CaptureDdls", captureDdls()).add("MaxFileSize", maxFileSize()).add("DatabaseName", databaseName())
                .add("DdlArtifactsSchema", ddlArtifactsSchema()).add("ExecuteTimeout", executeTimeout())
                .add("FailTasksOnLobTruncation", failTasksOnLobTruncation()).add("HeartbeatEnable", heartbeatEnable())
                .add("HeartbeatSchema", heartbeatSchema()).add("HeartbeatFrequency", heartbeatFrequency())
                .add("Password", password() == null ? null : "*** Sensitive Data Redacted ***").add("Port", port())
                .add("ServerName", serverName()).add("Username", username()).add("SlotName", slotName())
                .add("PluginName", pluginNameAsString()).add("SecretsManagerAccessRoleArn", secretsManagerAccessRoleArn())
                .add("SecretsManagerSecretId", secretsManagerSecretId()).add("TrimSpaceInChar", trimSpaceInChar())
                .add("MapBooleanAsBoolean", mapBooleanAsBoolean()).add("MapJsonbAsClob", mapJsonbAsClob())
                .add("MapLongVarcharAs", mapLongVarcharAsAsString()).add("DatabaseMode", databaseModeAsString())
                .add("BabelfishDatabaseName", babelfishDatabaseName()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AfterConnectScript":
            return Optional.ofNullable(clazz.cast(afterConnectScript()));
        case "CaptureDdls":
            return Optional.ofNullable(clazz.cast(captureDdls()));
        case "MaxFileSize":
            return Optional.ofNullable(clazz.cast(maxFileSize()));
        case "DatabaseName":
            return Optional.ofNullable(clazz.cast(databaseName()));
        case "DdlArtifactsSchema":
            return Optional.ofNullable(clazz.cast(ddlArtifactsSchema()));
        case "ExecuteTimeout":
            return Optional.ofNullable(clazz.cast(executeTimeout()));
        case "FailTasksOnLobTruncation":
            return Optional.ofNullable(clazz.cast(failTasksOnLobTruncation()));
        case "HeartbeatEnable":
            return Optional.ofNullable(clazz.cast(heartbeatEnable()));
        case "HeartbeatSchema":
            return Optional.ofNullable(clazz.cast(heartbeatSchema()));
        case "HeartbeatFrequency":
            return Optional.ofNullable(clazz.cast(heartbeatFrequency()));
        case "Password":
            return Optional.ofNullable(clazz.cast(password()));
        case "Port":
            return Optional.ofNullable(clazz.cast(port()));
        case "ServerName":
            return Optional.ofNullable(clazz.cast(serverName()));
        case "Username":
            return Optional.ofNullable(clazz.cast(username()));
        case "SlotName":
            return Optional.ofNullable(clazz.cast(slotName()));
        case "PluginName":
            return Optional.ofNullable(clazz.cast(pluginNameAsString()));
        case "SecretsManagerAccessRoleArn":
            return Optional.ofNullable(clazz.cast(secretsManagerAccessRoleArn()));
        case "SecretsManagerSecretId":
            return Optional.ofNullable(clazz.cast(secretsManagerSecretId()));
        case "TrimSpaceInChar":
            return Optional.ofNullable(clazz.cast(trimSpaceInChar()));
        case "MapBooleanAsBoolean":
            return Optional.ofNullable(clazz.cast(mapBooleanAsBoolean()));
        case "MapJsonbAsClob":
            return Optional.ofNullable(clazz.cast(mapJsonbAsClob()));
        case "MapLongVarcharAs":
            return Optional.ofNullable(clazz.cast(mapLongVarcharAsAsString()));
        case "DatabaseMode":
            return Optional.ofNullable(clazz.cast(databaseModeAsString()));
        case "BabelfishDatabaseName":
            return Optional.ofNullable(clazz.cast(babelfishDatabaseName()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<PostgreSQLSettings, T> g) {
        return obj -> g.apply((PostgreSQLSettings) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, PostgreSQLSettings> {
        /**
         * <p>
         * For use with change data capture (CDC) only, this attribute has DMS bypass foreign keys and user triggers to
         * reduce the time it takes to bulk load data.
         * </p>
         * <p>
         * Example: <code>afterConnectScript=SET session_replication_role='replica'</code>
         * </p>
         * 
         * @param afterConnectScript
         *        For use with change data capture (CDC) only, this attribute has DMS bypass foreign keys and user
         *        triggers to reduce the time it takes to bulk load data.</p>
         *        <p>
         *        Example: <code>afterConnectScript=SET session_replication_role='replica'</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder afterConnectScript(String afterConnectScript);

        /**
         * <p>
         * To capture DDL events, DMS creates various artifacts in the PostgreSQL database when the task starts. You can
         * later remove these artifacts.
         * </p>
         * <p>
         * If this value is set to <code>N</code>, you don't have to create tables or triggers on the source database.
         * </p>
         * 
         * @param captureDdls
         *        To capture DDL events, DMS creates various artifacts in the PostgreSQL database when the task starts.
         *        You can later remove these artifacts.</p>
         *        <p>
         *        If this value is set to <code>N</code>, you don't have to create tables or triggers on the source
         *        database.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder captureDdls(Boolean captureDdls);

        /**
         * <p>
         * Specifies the maximum size (in KB) of any .csv file used to transfer data to PostgreSQL.
         * </p>
         * <p>
         * Example: <code>maxFileSize=512</code>
         * </p>
         * 
         * @param maxFileSize
         *        Specifies the maximum size (in KB) of any .csv file used to transfer data to PostgreSQL.</p>
         *        <p>
         *        Example: <code>maxFileSize=512</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxFileSize(Integer maxFileSize);

        /**
         * <p>
         * Database name for the endpoint.
         * </p>
         * 
         * @param databaseName
         *        Database name for the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder databaseName(String databaseName);

        /**
         * <p>
         * The schema in which the operational DDL database artifacts are created.
         * </p>
         * <p>
         * Example: <code>ddlArtifactsSchema=xyzddlschema;</code>
         * </p>
         * 
         * @param ddlArtifactsSchema
         *        The schema in which the operational DDL database artifacts are created.</p>
         *        <p>
         *        Example: <code>ddlArtifactsSchema=xyzddlschema;</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ddlArtifactsSchema(String ddlArtifactsSchema);

        /**
         * <p>
         * Sets the client statement timeout for the PostgreSQL instance, in seconds. The default value is 60 seconds.
         * </p>
         * <p>
         * Example: <code>executeTimeout=100;</code>
         * </p>
         * 
         * @param executeTimeout
         *        Sets the client statement timeout for the PostgreSQL instance, in seconds. The default value is 60
         *        seconds.</p>
         *        <p>
         *        Example: <code>executeTimeout=100;</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder executeTimeout(Integer executeTimeout);

        /**
         * <p>
         * When set to <code>true</code>, this value causes a task to fail if the actual size of a LOB column is greater
         * than the specified <code>LobMaxSize</code>.
         * </p>
         * <p>
         * If task is set to Limited LOB mode and this option is set to true, the task fails instead of truncating the
         * LOB data.
         * </p>
         * 
         * @param failTasksOnLobTruncation
         *        When set to <code>true</code>, this value causes a task to fail if the actual size of a LOB column is
         *        greater than the specified <code>LobMaxSize</code>.</p>
         *        <p>
         *        If task is set to Limited LOB mode and this option is set to true, the task fails instead of
         *        truncating the LOB data.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failTasksOnLobTruncation(Boolean failTasksOnLobTruncation);

        /**
         * <p>
         * The write-ahead log (WAL) heartbeat feature mimics a dummy transaction. By doing this, it prevents idle
         * logical replication slots from holding onto old WAL logs, which can result in storage full situations on the
         * source. This heartbeat keeps <code>restart_lsn</code> moving and prevents storage full scenarios.
         * </p>
         * 
         * @param heartbeatEnable
         *        The write-ahead log (WAL) heartbeat feature mimics a dummy transaction. By doing this, it prevents
         *        idle logical replication slots from holding onto old WAL logs, which can result in storage full
         *        situations on the source. This heartbeat keeps <code>restart_lsn</code> moving and prevents storage
         *        full scenarios.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder heartbeatEnable(Boolean heartbeatEnable);

        /**
         * <p>
         * Sets the schema in which the heartbeat artifacts are created.
         * </p>
         * 
         * @param heartbeatSchema
         *        Sets the schema in which the heartbeat artifacts are created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder heartbeatSchema(String heartbeatSchema);

        /**
         * <p>
         * Sets the WAL heartbeat frequency (in minutes).
         * </p>
         * 
         * @param heartbeatFrequency
         *        Sets the WAL heartbeat frequency (in minutes).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder heartbeatFrequency(Integer heartbeatFrequency);

        /**
         * <p>
         * Endpoint connection password.
         * </p>
         * 
         * @param password
         *        Endpoint connection password.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder password(String password);

        /**
         * <p>
         * Endpoint TCP port. The default is 5432.
         * </p>
         * 
         * @param port
         *        Endpoint TCP port. The default is 5432.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder port(Integer port);

        /**
         * <p>
         * The host name of the endpoint database.
         * </p>
         * <p>
         * For an Amazon RDS PostgreSQL instance, this is the output of <a
         * href="https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBInstances.html"
         * >DescribeDBInstances</a>, in the
         * <code> <a href="https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_Endpoint.html">Endpoint</a>.Address</code>
         * field.
         * </p>
         * <p>
         * For an Aurora PostgreSQL instance, this is the output of <a
         * href="https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBClusters.html"
         * >DescribeDBClusters</a>, in the <code>Endpoint</code> field.
         * </p>
         * 
         * @param serverName
         *        The host name of the endpoint database. </p>
         *        <p>
         *        For an Amazon RDS PostgreSQL instance, this is the output of <a
         *        href="https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBInstances.html"
         *        >DescribeDBInstances</a>, in the
         *        <code> <a href="https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_Endpoint.html">Endpoint</a>.Address</code>
         *        field.
         *        </p>
         *        <p>
         *        For an Aurora PostgreSQL instance, this is the output of <a
         *        href="https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBClusters.html"
         *        >DescribeDBClusters</a>, in the <code>Endpoint</code> field.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serverName(String serverName);

        /**
         * <p>
         * Endpoint connection user name.
         * </p>
         * 
         * @param username
         *        Endpoint connection user name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder username(String username);

        /**
         * <p>
         * Sets the name of a previously created logical replication slot for a change data capture (CDC) load of the
         * PostgreSQL source instance.
         * </p>
         * <p>
         * When used with the <code>CdcStartPosition</code> request parameter for the DMS API , this attribute also
         * makes it possible to use native CDC start points. DMS verifies that the specified logical replication slot
         * exists before starting the CDC load task. It also verifies that the task was created with a valid setting of
         * <code>CdcStartPosition</code>. If the specified slot doesn't exist or the task doesn't have a valid
         * <code>CdcStartPosition</code> setting, DMS raises an error.
         * </p>
         * <p>
         * For more information about setting the <code>CdcStartPosition</code> request parameter, see <a
         * href="https://docs.aws.amazon.com/dms/latest/userguide/CHAP_Task.CDC.html#CHAP_Task.CDC.StartPoint.Native"
         * >Determining a CDC native start point</a> in the <i>Database Migration Service User Guide</i>. For more
         * information about using <code>CdcStartPosition</code>, see <a
         * href="https://docs.aws.amazon.com/dms/latest/APIReference/API_CreateReplicationTask.html"
         * >CreateReplicationTask</a>, <a
         * href="https://docs.aws.amazon.com/dms/latest/APIReference/API_StartReplicationTask.html"
         * >StartReplicationTask</a>, and <a
         * href="https://docs.aws.amazon.com/dms/latest/APIReference/API_ModifyReplicationTask.html"
         * >ModifyReplicationTask</a>.
         * </p>
         * 
         * @param slotName
         *        Sets the name of a previously created logical replication slot for a change data capture (CDC) load of
         *        the PostgreSQL source instance. </p>
         *        <p>
         *        When used with the <code>CdcStartPosition</code> request parameter for the DMS API , this attribute
         *        also makes it possible to use native CDC start points. DMS verifies that the specified logical
         *        replication slot exists before starting the CDC load task. It also verifies that the task was created
         *        with a valid setting of <code>CdcStartPosition</code>. If the specified slot doesn't exist or the task
         *        doesn't have a valid <code>CdcStartPosition</code> setting, DMS raises an error.
         *        </p>
         *        <p>
         *        For more information about setting the <code>CdcStartPosition</code> request parameter, see <a
         *        href="https://docs.aws.amazon.com/dms/latest/userguide/CHAP_Task.CDC.html#CHAP_Task.CDC.StartPoint.Native"
         *        >Determining a CDC native start point</a> in the <i>Database Migration Service User Guide</i>. For
         *        more information about using <code>CdcStartPosition</code>, see <a
         *        href="https://docs.aws.amazon.com/dms/latest/APIReference/API_CreateReplicationTask.html"
         *        >CreateReplicationTask</a>, <a
         *        href="https://docs.aws.amazon.com/dms/latest/APIReference/API_StartReplicationTask.html"
         *        >StartReplicationTask</a>, and <a
         *        href="https://docs.aws.amazon.com/dms/latest/APIReference/API_ModifyReplicationTask.html"
         *        >ModifyReplicationTask</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder slotName(String slotName);

        /**
         * <p>
         * Specifies the plugin to use to create a replication slot.
         * </p>
         * 
         * @param pluginName
         *        Specifies the plugin to use to create a replication slot.
         * @see PluginNameValue
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PluginNameValue
         */
        Builder pluginName(String pluginName);

        /**
         * <p>
         * Specifies the plugin to use to create a replication slot.
         * </p>
         * 
         * @param pluginName
         *        Specifies the plugin to use to create a replication slot.
         * @see PluginNameValue
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PluginNameValue
         */
        Builder pluginName(PluginNameValue pluginName);

        /**
         * <p>
         * The full Amazon Resource Name (ARN) of the IAM role that specifies DMS as the trusted entity and grants the
         * required permissions to access the value in <code>SecretsManagerSecret</code>. The role must allow the
         * <code>iam:PassRole</code> action. <code>SecretsManagerSecret</code> has the value of the Amazon Web Services
         * Secrets Manager secret that allows access to the PostgreSQL endpoint.
         * </p>
         * <note>
         * <p>
         * You can specify one of two sets of values for these permissions. You can specify the values for this setting
         * and <code>SecretsManagerSecretId</code>. Or you can specify clear-text values for <code>UserName</code>,
         * <code>Password</code>, <code>ServerName</code>, and <code>Port</code>. You can't specify both. For more
         * information on creating this <code>SecretsManagerSecret</code> and the
         * <code>SecretsManagerAccessRoleArn</code> and <code>SecretsManagerSecretId</code> required to access it, see
         * <a href="https://docs.aws.amazon.com/dms/latest/userguide/CHAP_Security.html#security-iam-secretsmanager">
         * Using secrets to access Database Migration Service resources</a> in the <i>Database Migration Service User
         * Guide</i>.
         * </p>
         * </note>
         * 
         * @param secretsManagerAccessRoleArn
         *        The full Amazon Resource Name (ARN) of the IAM role that specifies DMS as the trusted entity and
         *        grants the required permissions to access the value in <code>SecretsManagerSecret</code>. The role
         *        must allow the <code>iam:PassRole</code> action. <code>SecretsManagerSecret</code> has the value of
         *        the Amazon Web Services Secrets Manager secret that allows access to the PostgreSQL endpoint.</p>
         *        <note>
         *        <p>
         *        You can specify one of two sets of values for these permissions. You can specify the values for this
         *        setting and <code>SecretsManagerSecretId</code>. Or you can specify clear-text values for
         *        <code>UserName</code>, <code>Password</code>, <code>ServerName</code>, and <code>Port</code>. You
         *        can't specify both. For more information on creating this <code>SecretsManagerSecret</code> and the
         *        <code>SecretsManagerAccessRoleArn</code> and <code>SecretsManagerSecretId</code> required to access
         *        it, see <a href=
         *        "https://docs.aws.amazon.com/dms/latest/userguide/CHAP_Security.html#security-iam-secretsmanager"
         *        >Using secrets to access Database Migration Service resources</a> in the <i>Database Migration Service
         *        User Guide</i>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secretsManagerAccessRoleArn(String secretsManagerAccessRoleArn);

        /**
         * <p>
         * The full ARN, partial ARN, or friendly name of the <code>SecretsManagerSecret</code> that contains the
         * PostgreSQL endpoint connection details.
         * </p>
         * 
         * @param secretsManagerSecretId
         *        The full ARN, partial ARN, or friendly name of the <code>SecretsManagerSecret</code> that contains the
         *        PostgreSQL endpoint connection details.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secretsManagerSecretId(String secretsManagerSecretId);

        /**
         * <p>
         * Use the <code>TrimSpaceInChar</code> source endpoint setting to trim data on CHAR and NCHAR data types during
         * migration. The default value is <code>true</code>.
         * </p>
         * 
         * @param trimSpaceInChar
         *        Use the <code>TrimSpaceInChar</code> source endpoint setting to trim data on CHAR and NCHAR data types
         *        during migration. The default value is <code>true</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder trimSpaceInChar(Boolean trimSpaceInChar);

        /**
         * <p>
         * When true, lets PostgreSQL migrate the boolean type as boolean. By default, PostgreSQL migrates booleans as
         * <code>varchar(5)</code>.
         * </p>
         * 
         * @param mapBooleanAsBoolean
         *        When true, lets PostgreSQL migrate the boolean type as boolean. By default, PostgreSQL migrates
         *        booleans as <code>varchar(5)</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder mapBooleanAsBoolean(Boolean mapBooleanAsBoolean);

        /**
         * <p>
         * When true, DMS migrates JSONB values as CLOB.
         * </p>
         * 
         * @param mapJsonbAsClob
         *        When true, DMS migrates JSONB values as CLOB.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder mapJsonbAsClob(Boolean mapJsonbAsClob);

        /**
         * <p>
         * When true, DMS migrates LONG values as VARCHAR.
         * </p>
         * 
         * @param mapLongVarcharAs
         *        When true, DMS migrates LONG values as VARCHAR.
         * @see LongVarcharMappingType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LongVarcharMappingType
         */
        Builder mapLongVarcharAs(String mapLongVarcharAs);

        /**
         * <p>
         * When true, DMS migrates LONG values as VARCHAR.
         * </p>
         * 
         * @param mapLongVarcharAs
         *        When true, DMS migrates LONG values as VARCHAR.
         * @see LongVarcharMappingType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LongVarcharMappingType
         */
        Builder mapLongVarcharAs(LongVarcharMappingType mapLongVarcharAs);

        /**
         * <p>
         * Specifies whether to use default or custom replication behavior for PostgreSQL-compatible endpoints. You can
         * use this setting to specify replication behavior for endpoints that require additional configuration, such as
         * Babelfish endpoints.
         * </p>
         * 
         * @param databaseMode
         *        Specifies whether to use default or custom replication behavior for PostgreSQL-compatible endpoints.
         *        You can use this setting to specify replication behavior for endpoints that require additional
         *        configuration, such as Babelfish endpoints.
         * @see DatabaseMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DatabaseMode
         */
        Builder databaseMode(String databaseMode);

        /**
         * <p>
         * Specifies whether to use default or custom replication behavior for PostgreSQL-compatible endpoints. You can
         * use this setting to specify replication behavior for endpoints that require additional configuration, such as
         * Babelfish endpoints.
         * </p>
         * 
         * @param databaseMode
         *        Specifies whether to use default or custom replication behavior for PostgreSQL-compatible endpoints.
         *        You can use this setting to specify replication behavior for endpoints that require additional
         *        configuration, such as Babelfish endpoints.
         * @see DatabaseMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DatabaseMode
         */
        Builder databaseMode(DatabaseMode databaseMode);

        /**
         * <p>
         * The Babelfish for Aurora PostgreSQL database name for the endpoint.
         * </p>
         * 
         * @param babelfishDatabaseName
         *        The Babelfish for Aurora PostgreSQL database name for the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder babelfishDatabaseName(String babelfishDatabaseName);
    }

    static final class BuilderImpl implements Builder {
        private String afterConnectScript;

        private Boolean captureDdls;

        private Integer maxFileSize;

        private String databaseName;

        private String ddlArtifactsSchema;

        private Integer executeTimeout;

        private Boolean failTasksOnLobTruncation;

        private Boolean heartbeatEnable;

        private String heartbeatSchema;

        private Integer heartbeatFrequency;

        private String password;

        private Integer port;

        private String serverName;

        private String username;

        private String slotName;

        private String pluginName;

        private String secretsManagerAccessRoleArn;

        private String secretsManagerSecretId;

        private Boolean trimSpaceInChar;

        private Boolean mapBooleanAsBoolean;

        private Boolean mapJsonbAsClob;

        private String mapLongVarcharAs;

        private String databaseMode;

        private String babelfishDatabaseName;

        private BuilderImpl() {
        }

        private BuilderImpl(PostgreSQLSettings model) {
            afterConnectScript(model.afterConnectScript);
            captureDdls(model.captureDdls);
            maxFileSize(model.maxFileSize);
            databaseName(model.databaseName);
            ddlArtifactsSchema(model.ddlArtifactsSchema);
            executeTimeout(model.executeTimeout);
            failTasksOnLobTruncation(model.failTasksOnLobTruncation);
            heartbeatEnable(model.heartbeatEnable);
            heartbeatSchema(model.heartbeatSchema);
            heartbeatFrequency(model.heartbeatFrequency);
            password(model.password);
            port(model.port);
            serverName(model.serverName);
            username(model.username);
            slotName(model.slotName);
            pluginName(model.pluginName);
            secretsManagerAccessRoleArn(model.secretsManagerAccessRoleArn);
            secretsManagerSecretId(model.secretsManagerSecretId);
            trimSpaceInChar(model.trimSpaceInChar);
            mapBooleanAsBoolean(model.mapBooleanAsBoolean);
            mapJsonbAsClob(model.mapJsonbAsClob);
            mapLongVarcharAs(model.mapLongVarcharAs);
            databaseMode(model.databaseMode);
            babelfishDatabaseName(model.babelfishDatabaseName);
        }

        public final String getAfterConnectScript() {
            return afterConnectScript;
        }

        public final void setAfterConnectScript(String afterConnectScript) {
            this.afterConnectScript = afterConnectScript;
        }

        @Override
        public final Builder afterConnectScript(String afterConnectScript) {
            this.afterConnectScript = afterConnectScript;
            return this;
        }

        public final Boolean getCaptureDdls() {
            return captureDdls;
        }

        public final void setCaptureDdls(Boolean captureDdls) {
            this.captureDdls = captureDdls;
        }

        @Override
        public final Builder captureDdls(Boolean captureDdls) {
            this.captureDdls = captureDdls;
            return this;
        }

        public final Integer getMaxFileSize() {
            return maxFileSize;
        }

        public final void setMaxFileSize(Integer maxFileSize) {
            this.maxFileSize = maxFileSize;
        }

        @Override
        public final Builder maxFileSize(Integer maxFileSize) {
            this.maxFileSize = maxFileSize;
            return this;
        }

        public final String getDatabaseName() {
            return databaseName;
        }

        public final void setDatabaseName(String databaseName) {
            this.databaseName = databaseName;
        }

        @Override
        public final Builder databaseName(String databaseName) {
            this.databaseName = databaseName;
            return this;
        }

        public final String getDdlArtifactsSchema() {
            return ddlArtifactsSchema;
        }

        public final void setDdlArtifactsSchema(String ddlArtifactsSchema) {
            this.ddlArtifactsSchema = ddlArtifactsSchema;
        }

        @Override
        public final Builder ddlArtifactsSchema(String ddlArtifactsSchema) {
            this.ddlArtifactsSchema = ddlArtifactsSchema;
            return this;
        }

        public final Integer getExecuteTimeout() {
            return executeTimeout;
        }

        public final void setExecuteTimeout(Integer executeTimeout) {
            this.executeTimeout = executeTimeout;
        }

        @Override
        public final Builder executeTimeout(Integer executeTimeout) {
            this.executeTimeout = executeTimeout;
            return this;
        }

        public final Boolean getFailTasksOnLobTruncation() {
            return failTasksOnLobTruncation;
        }

        public final void setFailTasksOnLobTruncation(Boolean failTasksOnLobTruncation) {
            this.failTasksOnLobTruncation = failTasksOnLobTruncation;
        }

        @Override
        public final Builder failTasksOnLobTruncation(Boolean failTasksOnLobTruncation) {
            this.failTasksOnLobTruncation = failTasksOnLobTruncation;
            return this;
        }

        public final Boolean getHeartbeatEnable() {
            return heartbeatEnable;
        }

        public final void setHeartbeatEnable(Boolean heartbeatEnable) {
            this.heartbeatEnable = heartbeatEnable;
        }

        @Override
        public final Builder heartbeatEnable(Boolean heartbeatEnable) {
            this.heartbeatEnable = heartbeatEnable;
            return this;
        }

        public final String getHeartbeatSchema() {
            return heartbeatSchema;
        }

        public final void setHeartbeatSchema(String heartbeatSchema) {
            this.heartbeatSchema = heartbeatSchema;
        }

        @Override
        public final Builder heartbeatSchema(String heartbeatSchema) {
            this.heartbeatSchema = heartbeatSchema;
            return this;
        }

        public final Integer getHeartbeatFrequency() {
            return heartbeatFrequency;
        }

        public final void setHeartbeatFrequency(Integer heartbeatFrequency) {
            this.heartbeatFrequency = heartbeatFrequency;
        }

        @Override
        public final Builder heartbeatFrequency(Integer heartbeatFrequency) {
            this.heartbeatFrequency = heartbeatFrequency;
            return this;
        }

        public final String getPassword() {
            return password;
        }

        public final void setPassword(String password) {
            this.password = password;
        }

        @Override
        public final Builder password(String password) {
            this.password = password;
            return this;
        }

        public final Integer getPort() {
            return port;
        }

        public final void setPort(Integer port) {
            this.port = port;
        }

        @Override
        public final Builder port(Integer port) {
            this.port = port;
            return this;
        }

        public final String getServerName() {
            return serverName;
        }

        public final void setServerName(String serverName) {
            this.serverName = serverName;
        }

        @Override
        public final Builder serverName(String serverName) {
            this.serverName = serverName;
            return this;
        }

        public final String getUsername() {
            return username;
        }

        public final void setUsername(String username) {
            this.username = username;
        }

        @Override
        public final Builder username(String username) {
            this.username = username;
            return this;
        }

        public final String getSlotName() {
            return slotName;
        }

        public final void setSlotName(String slotName) {
            this.slotName = slotName;
        }

        @Override
        public final Builder slotName(String slotName) {
            this.slotName = slotName;
            return this;
        }

        public final String getPluginName() {
            return pluginName;
        }

        public final void setPluginName(String pluginName) {
            this.pluginName = pluginName;
        }

        @Override
        public final Builder pluginName(String pluginName) {
            this.pluginName = pluginName;
            return this;
        }

        @Override
        public final Builder pluginName(PluginNameValue pluginName) {
            this.pluginName(pluginName == null ? null : pluginName.toString());
            return this;
        }

        public final String getSecretsManagerAccessRoleArn() {
            return secretsManagerAccessRoleArn;
        }

        public final void setSecretsManagerAccessRoleArn(String secretsManagerAccessRoleArn) {
            this.secretsManagerAccessRoleArn = secretsManagerAccessRoleArn;
        }

        @Override
        public final Builder secretsManagerAccessRoleArn(String secretsManagerAccessRoleArn) {
            this.secretsManagerAccessRoleArn = secretsManagerAccessRoleArn;
            return this;
        }

        public final String getSecretsManagerSecretId() {
            return secretsManagerSecretId;
        }

        public final void setSecretsManagerSecretId(String secretsManagerSecretId) {
            this.secretsManagerSecretId = secretsManagerSecretId;
        }

        @Override
        public final Builder secretsManagerSecretId(String secretsManagerSecretId) {
            this.secretsManagerSecretId = secretsManagerSecretId;
            return this;
        }

        public final Boolean getTrimSpaceInChar() {
            return trimSpaceInChar;
        }

        public final void setTrimSpaceInChar(Boolean trimSpaceInChar) {
            this.trimSpaceInChar = trimSpaceInChar;
        }

        @Override
        public final Builder trimSpaceInChar(Boolean trimSpaceInChar) {
            this.trimSpaceInChar = trimSpaceInChar;
            return this;
        }

        public final Boolean getMapBooleanAsBoolean() {
            return mapBooleanAsBoolean;
        }

        public final void setMapBooleanAsBoolean(Boolean mapBooleanAsBoolean) {
            this.mapBooleanAsBoolean = mapBooleanAsBoolean;
        }

        @Override
        public final Builder mapBooleanAsBoolean(Boolean mapBooleanAsBoolean) {
            this.mapBooleanAsBoolean = mapBooleanAsBoolean;
            return this;
        }

        public final Boolean getMapJsonbAsClob() {
            return mapJsonbAsClob;
        }

        public final void setMapJsonbAsClob(Boolean mapJsonbAsClob) {
            this.mapJsonbAsClob = mapJsonbAsClob;
        }

        @Override
        public final Builder mapJsonbAsClob(Boolean mapJsonbAsClob) {
            this.mapJsonbAsClob = mapJsonbAsClob;
            return this;
        }

        public final String getMapLongVarcharAs() {
            return mapLongVarcharAs;
        }

        public final void setMapLongVarcharAs(String mapLongVarcharAs) {
            this.mapLongVarcharAs = mapLongVarcharAs;
        }

        @Override
        public final Builder mapLongVarcharAs(String mapLongVarcharAs) {
            this.mapLongVarcharAs = mapLongVarcharAs;
            return this;
        }

        @Override
        public final Builder mapLongVarcharAs(LongVarcharMappingType mapLongVarcharAs) {
            this.mapLongVarcharAs(mapLongVarcharAs == null ? null : mapLongVarcharAs.toString());
            return this;
        }

        public final String getDatabaseMode() {
            return databaseMode;
        }

        public final void setDatabaseMode(String databaseMode) {
            this.databaseMode = databaseMode;
        }

        @Override
        public final Builder databaseMode(String databaseMode) {
            this.databaseMode = databaseMode;
            return this;
        }

        @Override
        public final Builder databaseMode(DatabaseMode databaseMode) {
            this.databaseMode(databaseMode == null ? null : databaseMode.toString());
            return this;
        }

        public final String getBabelfishDatabaseName() {
            return babelfishDatabaseName;
        }

        public final void setBabelfishDatabaseName(String babelfishDatabaseName) {
            this.babelfishDatabaseName = babelfishDatabaseName;
        }

        @Override
        public final Builder babelfishDatabaseName(String babelfishDatabaseName) {
            this.babelfishDatabaseName = babelfishDatabaseName;
            return this;
        }

        @Override
        public PostgreSQLSettings build() {
            return new PostgreSQLSettings(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
