/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.costandusagereport;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.costandusagereport.model.CostAndUsageReportException;
import software.amazon.awssdk.services.costandusagereport.model.CostAndUsageReportRequest;
import software.amazon.awssdk.services.costandusagereport.model.DeleteReportDefinitionRequest;
import software.amazon.awssdk.services.costandusagereport.model.DeleteReportDefinitionResponse;
import software.amazon.awssdk.services.costandusagereport.model.DescribeReportDefinitionsRequest;
import software.amazon.awssdk.services.costandusagereport.model.DescribeReportDefinitionsResponse;
import software.amazon.awssdk.services.costandusagereport.model.DuplicateReportNameException;
import software.amazon.awssdk.services.costandusagereport.model.InternalErrorException;
import software.amazon.awssdk.services.costandusagereport.model.ModifyReportDefinitionRequest;
import software.amazon.awssdk.services.costandusagereport.model.ModifyReportDefinitionResponse;
import software.amazon.awssdk.services.costandusagereport.model.PutReportDefinitionRequest;
import software.amazon.awssdk.services.costandusagereport.model.PutReportDefinitionResponse;
import software.amazon.awssdk.services.costandusagereport.model.ReportLimitReachedException;
import software.amazon.awssdk.services.costandusagereport.model.ValidationException;
import software.amazon.awssdk.services.costandusagereport.paginators.DescribeReportDefinitionsIterable;
import software.amazon.awssdk.services.costandusagereport.transform.DeleteReportDefinitionRequestMarshaller;
import software.amazon.awssdk.services.costandusagereport.transform.DescribeReportDefinitionsRequestMarshaller;
import software.amazon.awssdk.services.costandusagereport.transform.ModifyReportDefinitionRequestMarshaller;
import software.amazon.awssdk.services.costandusagereport.transform.PutReportDefinitionRequestMarshaller;

/**
 * Internal implementation of {@link CostAndUsageReportClient}.
 *
 * @see CostAndUsageReportClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultCostAndUsageReportClient implements CostAndUsageReportClient {
    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultCostAndUsageReportClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Deletes the specified report.
     * </p>
     *
     * @param deleteReportDefinitionRequest
     *        Deletes the specified report.
     * @return Result of the DeleteReportDefinition operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CostAndUsageReportException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CostAndUsageReportClient.DeleteReportDefinition
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cur-2017-01-06/DeleteReportDefinition" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteReportDefinitionResponse deleteReportDefinition(DeleteReportDefinitionRequest deleteReportDefinitionRequest)
            throws InternalErrorException, ValidationException, AwsServiceException, SdkClientException,
            CostAndUsageReportException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteReportDefinitionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteReportDefinitionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeleteReportDefinitionRequest, DeleteReportDefinitionResponse>()
                .withOperationName("DeleteReportDefinition").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteReportDefinitionRequest)
                .withMarshaller(new DeleteReportDefinitionRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists the AWS Cost and Usage reports available to this account.
     * </p>
     *
     * @param describeReportDefinitionsRequest
     *        Requests a list of AWS Cost and Usage reports owned by the account.
     * @return Result of the DescribeReportDefinitions operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CostAndUsageReportException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CostAndUsageReportClient.DescribeReportDefinitions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cur-2017-01-06/DescribeReportDefinitions" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeReportDefinitionsResponse describeReportDefinitions(
            DescribeReportDefinitionsRequest describeReportDefinitionsRequest) throws InternalErrorException,
            AwsServiceException, SdkClientException, CostAndUsageReportException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeReportDefinitionsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeReportDefinitionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<DescribeReportDefinitionsRequest, DescribeReportDefinitionsResponse>()
                        .withOperationName("DescribeReportDefinitions").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(describeReportDefinitionsRequest)
                        .withMarshaller(new DescribeReportDefinitionsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists the AWS Cost and Usage reports available to this account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeReportDefinitions(software.amazon.awssdk.services.costandusagereport.model.DescribeReportDefinitionsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.costandusagereport.paginators.DescribeReportDefinitionsIterable responses = client.describeReportDefinitionsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.costandusagereport.paginators.DescribeReportDefinitionsIterable responses = client
     *             .describeReportDefinitionsPaginator(request);
     *     for (software.amazon.awssdk.services.costandusagereport.model.DescribeReportDefinitionsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.costandusagereport.paginators.DescribeReportDefinitionsIterable responses = client.describeReportDefinitionsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeReportDefinitions(software.amazon.awssdk.services.costandusagereport.model.DescribeReportDefinitionsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeReportDefinitionsRequest
     *        Requests a list of AWS Cost and Usage reports owned by the account.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CostAndUsageReportException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CostAndUsageReportClient.DescribeReportDefinitions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cur-2017-01-06/DescribeReportDefinitions" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeReportDefinitionsIterable describeReportDefinitionsPaginator(
            DescribeReportDefinitionsRequest describeReportDefinitionsRequest) throws InternalErrorException,
            AwsServiceException, SdkClientException, CostAndUsageReportException {
        return new DescribeReportDefinitionsIterable(this, applyPaginatorUserAgent(describeReportDefinitionsRequest));
    }

    /**
     * <p>
     * Allows you to programatically update your report preferences.
     * </p>
     *
     * @param modifyReportDefinitionRequest
     * @return Result of the ModifyReportDefinition operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CostAndUsageReportException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CostAndUsageReportClient.ModifyReportDefinition
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cur-2017-01-06/ModifyReportDefinition" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ModifyReportDefinitionResponse modifyReportDefinition(ModifyReportDefinitionRequest modifyReportDefinitionRequest)
            throws InternalErrorException, ValidationException, AwsServiceException, SdkClientException,
            CostAndUsageReportException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ModifyReportDefinitionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ModifyReportDefinitionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ModifyReportDefinitionRequest, ModifyReportDefinitionResponse>()
                .withOperationName("ModifyReportDefinition").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(modifyReportDefinitionRequest)
                .withMarshaller(new ModifyReportDefinitionRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Creates a new report using the description that you provide.
     * </p>
     *
     * @param putReportDefinitionRequest
     *        Creates a Cost and Usage Report.
     * @return Result of the PutReportDefinition operation returned by the service.
     * @throws DuplicateReportNameException
     *         A report with the specified name already exists in the account. Specify a different report name.
     * @throws ReportLimitReachedException
     *         This account already has five reports defined. To define a new report, you must delete an existing
     *         report.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CostAndUsageReportException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CostAndUsageReportClient.PutReportDefinition
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cur-2017-01-06/PutReportDefinition" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PutReportDefinitionResponse putReportDefinition(PutReportDefinitionRequest putReportDefinitionRequest)
            throws DuplicateReportNameException, ReportLimitReachedException, InternalErrorException, ValidationException,
            AwsServiceException, SdkClientException, CostAndUsageReportException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutReportDefinitionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutReportDefinitionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<PutReportDefinitionRequest, PutReportDefinitionResponse>()
                .withOperationName("PutReportDefinition").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(putReportDefinitionRequest)
                .withMarshaller(new PutReportDefinitionRequestMarshaller(protocolFactory)));
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(CostAndUsageReportException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ReportLimitReachedException")
                                .exceptionBuilderSupplier(ReportLimitReachedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalErrorException")
                                .exceptionBuilderSupplier(InternalErrorException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DuplicateReportNameException")
                                .exceptionBuilderSupplier(DuplicateReportNameException::builder).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends CostAndUsageReportRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
