/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.awssdk.http.pipeline.stages;

import java.io.IOException;
import java.io.InputStream;
import java.util.Collections;
import java.util.Date;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import software.amazon.awssdk.RequestExecutionContext;
import software.amazon.awssdk.ResetException;
import software.amazon.awssdk.Response;
import software.amazon.awssdk.SdkBaseException;
import software.amazon.awssdk.SdkClientException;
import software.amazon.awssdk.event.ProgressEventType;
import software.amazon.awssdk.event.ProgressListener;
import software.amazon.awssdk.event.SdkProgressPublisher;
import software.amazon.awssdk.handlers.AwsHandlerKeys;
import software.amazon.awssdk.http.AmazonHttpClient;
import software.amazon.awssdk.http.HttpClientDependencies;
import software.amazon.awssdk.http.HttpResponse;
import software.amazon.awssdk.http.SdkHttpFullRequest;
import software.amazon.awssdk.http.pipeline.RequestPipeline;
import software.amazon.awssdk.metrics.spi.AwsRequestMetrics;
import software.amazon.awssdk.metrics.spi.MetricType;
import software.amazon.awssdk.retry.RetryUtils;
import software.amazon.awssdk.retry.v2.RetryPolicy;
import software.amazon.awssdk.retry.v2.RetryPolicyContext;
import software.amazon.awssdk.util.CapacityManager;
import software.amazon.awssdk.util.DateUtils;

public class AsyncRetryableStage<OutputT>
implements RequestPipeline<SdkHttpFullRequest, CompletableFuture<Response<OutputT>>> {
    private static final Log log = LogFactory.getLog(AsyncRetryableStage.class);
    private final RequestPipeline<SdkHttpFullRequest, CompletableFuture<Response<OutputT>>> requestPipeline;
    private final ScheduledExecutorService retrySubmitter;
    private final HttpClientDependencies dependencies;
    private final CapacityManager retryCapacity;
    private final RetryPolicy retryPolicy;

    public AsyncRetryableStage(HttpClientDependencies dependencies, RequestPipeline<SdkHttpFullRequest, CompletableFuture<Response<OutputT>>> requestPipeline) {
        this.dependencies = dependencies;
        this.retrySubmitter = dependencies.executorService();
        this.retryCapacity = dependencies.retryCapacity();
        this.retryPolicy = dependencies.retryPolicy();
        this.requestPipeline = requestPipeline;
    }

    @Override
    public CompletableFuture<Response<OutputT>> execute(SdkHttpFullRequest request, RequestExecutionContext context) throws Exception {
        context.awsRequestMetrics().addPropertyWith((MetricType)AwsRequestMetrics.Field.RequestType, (Object)context.requestConfig().getRequestType()).addPropertyWith((MetricType)AwsRequestMetrics.Field.ServiceName, request.handlerContext(AwsHandlerKeys.SERVICE_NAME)).addPropertyWith((MetricType)AwsRequestMetrics.Field.ServiceEndpoint, (Object)request.getEndpoint());
        return new RetryExecutor(request, context).execute();
    }

    private static void resetRequestInputStream(InputStream inputStream) throws ResetException {
        if (inputStream != null && inputStream.markSupported()) {
            try {
                inputStream.reset();
            }
            catch (IOException ex) {
                throw new ResetException("Failed to reset the request input stream", ex);
            }
        }
    }

    private static int parseClockSkewOffset(HttpResponse httpResponse) {
        Optional<String> dateHeader = Optional.ofNullable(httpResponse.getHeader("Date"));
        try {
            Date serverDate = dateHeader.filter(h -> !h.isEmpty()).map(DateUtils::parseRfc822Date).orElseThrow(() -> new RuntimeException("Unable to parse clock skew offset from response. Server Date header missing"));
            long diff = System.currentTimeMillis() - serverDate.getTime();
            return (int)(diff / 1000L);
        }
        catch (RuntimeException e) {
            log.warn((Object)("Unable to parse clock skew offset from response: " + dateHeader.orElse("")), (Throwable)e);
            return 0;
        }
    }

    private class RetryExecutor {
        private final SdkHttpFullRequest request;
        private final RequestExecutionContext context;
        private final ProgressListener progressListener;
        private final AwsRequestMetrics awsRequestMetrics;
        private Optional<SdkBaseException> retriedException;
        private RetryPolicyContext retryPolicyContext;
        private int requestCount;
        private long lastBackoffDelay;
        private boolean retryCapacityConsumed;

        private RetryExecutor(SdkHttpFullRequest request, RequestExecutionContext context) {
            this.request = request;
            this.context = context;
            this.progressListener = context.requestConfig().getProgressListener();
            this.awsRequestMetrics = context.awsRequestMetrics();
            this.retriedException = Optional.empty();
        }

        public CompletableFuture<Response<OutputT>> execute() throws Exception {
            CompletableFuture future = new CompletableFuture();
            this.execute(future);
            return future;
        }

        public void execute(CompletableFuture<Response<OutputT>> future) throws Exception {
            this.beforeExecute();
            this.doExecute().handle((resp, err) -> this.handle(future, (Response)resp, (Throwable)err));
        }

        private Void handle(CompletableFuture<Response<OutputT>> future, Response<OutputT> resp, Throwable err) {
            try {
                if (resp != null && resp.isSuccess()) {
                    this.releaseRetryCapacity();
                    future.complete(resp);
                } else if (resp != null) {
                    this.setRetriedException(this.handleSdkException(resp));
                    this.executeRetry(future);
                } else if (err instanceof IOException) {
                    this.setRetriedException(this.handleIoException((IOException)err));
                    this.executeRetry(future);
                } else {
                    future.completeExceptionally(err);
                }
            }
            catch (Exception e) {
                future.completeExceptionally(e);
            }
            return null;
        }

        private void executeRetry(CompletableFuture<Response<OutputT>> future) {
            long delay;
            SdkProgressPublisher.publishProgress(this.progressListener, ProgressEventType.CLIENT_REQUEST_RETRY_EVENT);
            this.awsRequestMetrics.startEvent((MetricType)AwsRequestMetrics.Field.RetryPauseTime);
            int retriesAttempted = this.requestCount - 2;
            this.lastBackoffDelay = delay = AsyncRetryableStage.this.retryPolicy.computeDelayBeforeNextRetry(this.retryPolicyContext);
            if (log.isDebugEnabled()) {
                log.debug((Object)("Retriable error detected, will retry in " + delay + "ms, attempt number: " + retriesAttempted));
            }
            AsyncRetryableStage.this.retrySubmitter.schedule(() -> {
                this.awsRequestMetrics.endEvent((MetricType)AwsRequestMetrics.Field.RetryPauseTime);
                this.execute(future);
                return null;
            }, delay, TimeUnit.MILLISECONDS);
        }

        private void releaseRetryCapacity() {
            if (this.isRetry() && this.retryCapacityConsumed) {
                AsyncRetryableStage.this.retryCapacity.release(5);
            } else {
                AsyncRetryableStage.this.retryCapacity.release();
            }
        }

        private void beforeExecute() {
            this.retryCapacityConsumed = false;
            this.context.awsRequestMetrics().setCounter((MetricType)AwsRequestMetrics.Field.RequestCount, (long)(++this.requestCount));
        }

        private CompletableFuture<Response<OutputT>> doExecute() throws Exception {
            if (this.isRetry()) {
                AsyncRetryableStage.resetRequestInputStream(this.request.getContent());
            }
            this.markInputStream(this.request.getContent());
            if (AmazonHttpClient.REQUEST_LOG.isDebugEnabled()) {
                AmazonHttpClient.REQUEST_LOG.debug((Object)((this.isRetry() ? "Retrying " : "Sending ") + "Request: " + this.request));
            }
            return (CompletableFuture)AsyncRetryableStage.this.requestPipeline.execute(this.addRetryInfoHeader(this.request), this.context);
        }

        private boolean isRetry() {
            return this.retriedException.isPresent();
        }

        private void setRetriedException(SdkBaseException e) {
            this.retriedException = Optional.of(e);
        }

        private SdkBaseException handleSdkException(Response<OutputT> response) {
            SdkBaseException exception = response.getException();
            if (!this.shouldRetry(response.getHttpResponse(), exception)) {
                throw exception;
            }
            if (RetryUtils.isClockSkewError(exception)) {
                int clockSkew = AsyncRetryableStage.parseClockSkewOffset(response.getHttpResponse());
                AsyncRetryableStage.this.dependencies.updateTimeOffset(clockSkew);
            }
            return exception;
        }

        private SdkBaseException handleIoException(IOException ioe) {
            SdkClientException sdkClientException = new SdkClientException("Unable to execute HTTP request: " + ioe.getMessage(), ioe);
            boolean willRetry = this.shouldRetry(null, sdkClientException);
            if (log.isDebugEnabled()) {
                log.debug((Object)(sdkClientException.getMessage() + (willRetry ? " Request will be retried." : "")), (Throwable)ioe);
            }
            if (!willRetry) {
                throw sdkClientException;
            }
            return sdkClientException;
        }

        private void markInputStream(InputStream originalContent) {
            if (originalContent != null && originalContent.markSupported()) {
                originalContent.mark(this.readLimit());
            }
        }

        private int readLimit() {
            return this.context.requestConfig().getRequestClientOptions().getReadLimit();
        }

        private boolean shouldRetry(HttpResponse httpResponse, SdkBaseException exception) {
            int retriesAttempted = this.requestCount - 1;
            if (!RetryUtils.isThrottlingException(exception)) {
                if (!AsyncRetryableStage.this.retryCapacity.acquire(5)) {
                    this.awsRequestMetrics.incrementCounter((MetricType)AwsRequestMetrics.Field.ThrottledRetryCount);
                    return false;
                }
                this.retryCapacityConsumed = true;
            }
            this.retryPolicyContext = RetryPolicyContext.builder().request(this.request).originalRequest(this.context.requestConfig().getOriginalRequest()).exception(exception).retriesAttempted(retriesAttempted).httpStatusCode(httpResponse == null ? null : Integer.valueOf(httpResponse.getStatusCode())).build();
            if (!AsyncRetryableStage.this.retryPolicy.shouldRetry(this.retryPolicyContext)) {
                if (this.retryCapacityConsumed) {
                    AsyncRetryableStage.this.retryCapacity.release(5);
                }
                return false;
            }
            return true;
        }

        private SdkHttpFullRequest addRetryInfoHeader(SdkHttpFullRequest request) throws Exception {
            int availableRetryCapacity = AsyncRetryableStage.this.retryCapacity.availableCapacity();
            return (SdkHttpFullRequest)((SdkHttpFullRequest.Builder)request.toBuilder()).header("amz-sdk-retry", Collections.singletonList(String.format("%s/%s/%s", this.requestCount - 1, this.lastBackoffDelay, availableRetryCapacity >= 0 ? Integer.valueOf(availableRetryCapacity) : ""))).build();
        }
    }
}

