/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.comprehendmedical;

import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.comprehendmedical.model.ComprehendMedicalException;
import software.amazon.awssdk.services.comprehendmedical.model.DescribeEntitiesDetectionV2JobRequest;
import software.amazon.awssdk.services.comprehendmedical.model.DescribeEntitiesDetectionV2JobResponse;
import software.amazon.awssdk.services.comprehendmedical.model.DescribePhiDetectionJobRequest;
import software.amazon.awssdk.services.comprehendmedical.model.DescribePhiDetectionJobResponse;
import software.amazon.awssdk.services.comprehendmedical.model.DetectEntitiesRequest;
import software.amazon.awssdk.services.comprehendmedical.model.DetectEntitiesResponse;
import software.amazon.awssdk.services.comprehendmedical.model.DetectEntitiesV2Request;
import software.amazon.awssdk.services.comprehendmedical.model.DetectEntitiesV2Response;
import software.amazon.awssdk.services.comprehendmedical.model.DetectPhiRequest;
import software.amazon.awssdk.services.comprehendmedical.model.DetectPhiResponse;
import software.amazon.awssdk.services.comprehendmedical.model.InferIcd10CmRequest;
import software.amazon.awssdk.services.comprehendmedical.model.InferIcd10CmResponse;
import software.amazon.awssdk.services.comprehendmedical.model.InferRxNormRequest;
import software.amazon.awssdk.services.comprehendmedical.model.InferRxNormResponse;
import software.amazon.awssdk.services.comprehendmedical.model.InternalServerException;
import software.amazon.awssdk.services.comprehendmedical.model.InvalidEncodingException;
import software.amazon.awssdk.services.comprehendmedical.model.InvalidRequestException;
import software.amazon.awssdk.services.comprehendmedical.model.ListEntitiesDetectionV2JobsRequest;
import software.amazon.awssdk.services.comprehendmedical.model.ListEntitiesDetectionV2JobsResponse;
import software.amazon.awssdk.services.comprehendmedical.model.ListPhiDetectionJobsRequest;
import software.amazon.awssdk.services.comprehendmedical.model.ListPhiDetectionJobsResponse;
import software.amazon.awssdk.services.comprehendmedical.model.ResourceNotFoundException;
import software.amazon.awssdk.services.comprehendmedical.model.ServiceUnavailableException;
import software.amazon.awssdk.services.comprehendmedical.model.StartEntitiesDetectionV2JobRequest;
import software.amazon.awssdk.services.comprehendmedical.model.StartEntitiesDetectionV2JobResponse;
import software.amazon.awssdk.services.comprehendmedical.model.StartPhiDetectionJobRequest;
import software.amazon.awssdk.services.comprehendmedical.model.StartPhiDetectionJobResponse;
import software.amazon.awssdk.services.comprehendmedical.model.StopEntitiesDetectionV2JobRequest;
import software.amazon.awssdk.services.comprehendmedical.model.StopEntitiesDetectionV2JobResponse;
import software.amazon.awssdk.services.comprehendmedical.model.StopPhiDetectionJobRequest;
import software.amazon.awssdk.services.comprehendmedical.model.StopPhiDetectionJobResponse;
import software.amazon.awssdk.services.comprehendmedical.model.TextSizeLimitExceededException;
import software.amazon.awssdk.services.comprehendmedical.model.TooManyRequestsException;
import software.amazon.awssdk.services.comprehendmedical.model.ValidationException;
import software.amazon.awssdk.services.comprehendmedical.transform.DescribeEntitiesDetectionV2JobRequestMarshaller;
import software.amazon.awssdk.services.comprehendmedical.transform.DescribePhiDetectionJobRequestMarshaller;
import software.amazon.awssdk.services.comprehendmedical.transform.DetectEntitiesRequestMarshaller;
import software.amazon.awssdk.services.comprehendmedical.transform.DetectEntitiesV2RequestMarshaller;
import software.amazon.awssdk.services.comprehendmedical.transform.DetectPhiRequestMarshaller;
import software.amazon.awssdk.services.comprehendmedical.transform.InferIcd10CmRequestMarshaller;
import software.amazon.awssdk.services.comprehendmedical.transform.InferRxNormRequestMarshaller;
import software.amazon.awssdk.services.comprehendmedical.transform.ListEntitiesDetectionV2JobsRequestMarshaller;
import software.amazon.awssdk.services.comprehendmedical.transform.ListPhiDetectionJobsRequestMarshaller;
import software.amazon.awssdk.services.comprehendmedical.transform.StartEntitiesDetectionV2JobRequestMarshaller;
import software.amazon.awssdk.services.comprehendmedical.transform.StartPhiDetectionJobRequestMarshaller;
import software.amazon.awssdk.services.comprehendmedical.transform.StopEntitiesDetectionV2JobRequestMarshaller;
import software.amazon.awssdk.services.comprehendmedical.transform.StopPhiDetectionJobRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link ComprehendMedicalAsyncClient}.
 *
 * @see ComprehendMedicalAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultComprehendMedicalAsyncClient implements ComprehendMedicalAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultComprehendMedicalAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultComprehendMedicalAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Gets the properties associated with a medical entities detection job. Use this operation to get the status of a
     * detection job.
     * </p>
     *
     * @param describeEntitiesDetectionV2JobRequest
     * @return A Java Future containing the result of the DescribeEntitiesDetectionV2Job operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request that you made is invalid. Check your request to determine why
     *         it's invalid and then retry the request.</li>
     *         <li>TooManyRequestsException You have made too many requests within a short period of time. Wait for a
     *         short time and then try your request again. Contact customer support for more information about a service
     *         limit increase.</li>
     *         <li>ResourceNotFoundException The resource identified by the specified Amazon Resource Name (ARN) was not
     *         found. Check the ARN and try your request again.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ComprehendMedicalException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ComprehendMedicalAsyncClient.DescribeEntitiesDetectionV2Job
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/comprehendmedical-2018-10-30/DescribeEntitiesDetectionV2Job"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeEntitiesDetectionV2JobResponse> describeEntitiesDetectionV2Job(
            DescribeEntitiesDetectionV2JobRequest describeEntitiesDetectionV2JobRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeEntitiesDetectionV2JobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeEntitiesDetectionV2JobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeEntitiesDetectionV2JobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeEntitiesDetectionV2JobRequest, DescribeEntitiesDetectionV2JobResponse>()
                            .withOperationName("DescribeEntitiesDetectionV2Job")
                            .withMarshaller(new DescribeEntitiesDetectionV2JobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeEntitiesDetectionV2JobRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the properties associated with a protected health information (PHI) detection job. Use this operation to get
     * the status of a detection job.
     * </p>
     *
     * @param describePhiDetectionJobRequest
     * @return A Java Future containing the result of the DescribePHIDetectionJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request that you made is invalid. Check your request to determine why
     *         it's invalid and then retry the request.</li>
     *         <li>TooManyRequestsException You have made too many requests within a short period of time. Wait for a
     *         short time and then try your request again. Contact customer support for more information about a service
     *         limit increase.</li>
     *         <li>ResourceNotFoundException The resource identified by the specified Amazon Resource Name (ARN) was not
     *         found. Check the ARN and try your request again.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ComprehendMedicalException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ComprehendMedicalAsyncClient.DescribePHIDetectionJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/comprehendmedical-2018-10-30/DescribePHIDetectionJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribePhiDetectionJobResponse> describePHIDetectionJob(
            DescribePhiDetectionJobRequest describePhiDetectionJobRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribePhiDetectionJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribePhiDetectionJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribePhiDetectionJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribePhiDetectionJobRequest, DescribePhiDetectionJobResponse>()
                            .withOperationName("DescribePHIDetectionJob")
                            .withMarshaller(new DescribePhiDetectionJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describePhiDetectionJobRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>DetectEntities</code> operation is deprecated. You should use the <a>DetectEntitiesV2</a> operation
     * instead.
     * </p>
     * <p>
     * Inspects the clinical text for a variety of medical entities and returns specific information about them such as
     * entity category, location, and confidence score on that information .
     * </p>
     *
     * @param detectEntitiesRequest
     * @return A Java Future containing the result of the DetectEntities operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ServiceUnavailableException The Amazon Comprehend Medical service is temporarily unavailable. Please
     *         wait and then retry your request.</li>
     *         <li>TooManyRequestsException You have made too many requests within a short period of time. Wait for a
     *         short time and then try your request again. Contact customer support for more information about a service
     *         limit increase.</li>
     *         <li>InvalidRequestException The request that you made is invalid. Check your request to determine why
     *         it's invalid and then retry the request.</li>
     *         <li>InvalidEncodingException The input text was not in valid UTF-8 character encoding. Check your text
     *         then retry your request.</li>
     *         <li>TextSizeLimitExceededException The size of the text you submitted exceeds the size limit. Reduce the
     *         size of the text or use a smaller document and then retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ComprehendMedicalException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ComprehendMedicalAsyncClient.DetectEntities
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/comprehendmedical-2018-10-30/DetectEntities"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DetectEntitiesResponse> detectEntities(DetectEntitiesRequest detectEntitiesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DetectEntitiesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DetectEntitiesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DetectEntitiesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DetectEntitiesRequest, DetectEntitiesResponse>()
                            .withOperationName("DetectEntities")
                            .withMarshaller(new DetectEntitiesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(detectEntitiesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Inspects the clinical text for a variety of medical entities and returns specific information about them such as
     * entity category, location, and confidence score on that information. Amazon Comprehend Medical only detects
     * medical entities in English language texts.
     * </p>
     * <p>
     * The <code>DetectEntitiesV2</code> operation replaces the <a>DetectEntities</a> operation. This new action uses a
     * different model for determining the entities in your medical text and changes the way that some entities are
     * returned in the output. You should use the <code>DetectEntitiesV2</code> operation in all new applications.
     * </p>
     * <p>
     * The <code>DetectEntitiesV2</code> operation returns the <code>Acuity</code> and <code>Direction</code> entities
     * as attributes instead of types.
     * </p>
     *
     * @param detectEntitiesV2Request
     * @return A Java Future containing the result of the DetectEntitiesV2 operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ServiceUnavailableException The Amazon Comprehend Medical service is temporarily unavailable. Please
     *         wait and then retry your request.</li>
     *         <li>TooManyRequestsException You have made too many requests within a short period of time. Wait for a
     *         short time and then try your request again. Contact customer support for more information about a service
     *         limit increase.</li>
     *         <li>InvalidRequestException The request that you made is invalid. Check your request to determine why
     *         it's invalid and then retry the request.</li>
     *         <li>InvalidEncodingException The input text was not in valid UTF-8 character encoding. Check your text
     *         then retry your request.</li>
     *         <li>TextSizeLimitExceededException The size of the text you submitted exceeds the size limit. Reduce the
     *         size of the text or use a smaller document and then retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ComprehendMedicalException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ComprehendMedicalAsyncClient.DetectEntitiesV2
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/comprehendmedical-2018-10-30/DetectEntitiesV2"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DetectEntitiesV2Response> detectEntitiesV2(DetectEntitiesV2Request detectEntitiesV2Request) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DetectEntitiesV2Response> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DetectEntitiesV2Response::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DetectEntitiesV2Response> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DetectEntitiesV2Request, DetectEntitiesV2Response>()
                            .withOperationName("DetectEntitiesV2")
                            .withMarshaller(new DetectEntitiesV2RequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(detectEntitiesV2Request));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Inspects the clinical text for protected health information (PHI) entities and returns the entity category,
     * location, and confidence score for each entity. Amazon Comprehend Medical only detects entities in English
     * language texts.
     * </p>
     *
     * @param detectPhiRequest
     * @return A Java Future containing the result of the DetectPHI operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ServiceUnavailableException The Amazon Comprehend Medical service is temporarily unavailable. Please
     *         wait and then retry your request.</li>
     *         <li>TooManyRequestsException You have made too many requests within a short period of time. Wait for a
     *         short time and then try your request again. Contact customer support for more information about a service
     *         limit increase.</li>
     *         <li>InvalidRequestException The request that you made is invalid. Check your request to determine why
     *         it's invalid and then retry the request.</li>
     *         <li>InvalidEncodingException The input text was not in valid UTF-8 character encoding. Check your text
     *         then retry your request.</li>
     *         <li>TextSizeLimitExceededException The size of the text you submitted exceeds the size limit. Reduce the
     *         size of the text or use a smaller document and then retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ComprehendMedicalException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ComprehendMedicalAsyncClient.DetectPHI
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/comprehendmedical-2018-10-30/DetectPHI" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DetectPhiResponse> detectPHI(DetectPhiRequest detectPhiRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DetectPhiResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DetectPhiResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DetectPhiResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DetectPhiRequest, DetectPhiResponse>().withOperationName("DetectPHI")
                            .withMarshaller(new DetectPhiRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(detectPhiRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * InferICD10CM detects medical conditions as entities listed in a patient record and links those entities to
     * normalized concept identifiers in the ICD-10-CM knowledge base from the Centers for Disease Control. Amazon
     * Comprehend Medical only detects medical entities in English language texts.
     * </p>
     *
     * @param inferIcd10CmRequest
     * @return A Java Future containing the result of the InferICD10CM operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ServiceUnavailableException The Amazon Comprehend Medical service is temporarily unavailable. Please
     *         wait and then retry your request.</li>
     *         <li>TooManyRequestsException You have made too many requests within a short period of time. Wait for a
     *         short time and then try your request again. Contact customer support for more information about a service
     *         limit increase.</li>
     *         <li>InvalidRequestException The request that you made is invalid. Check your request to determine why
     *         it's invalid and then retry the request.</li>
     *         <li>InvalidEncodingException The input text was not in valid UTF-8 character encoding. Check your text
     *         then retry your request.</li>
     *         <li>TextSizeLimitExceededException The size of the text you submitted exceeds the size limit. Reduce the
     *         size of the text or use a smaller document and then retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ComprehendMedicalException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ComprehendMedicalAsyncClient.InferICD10CM
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/comprehendmedical-2018-10-30/InferICD10CM" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<InferIcd10CmResponse> inferICD10CM(InferIcd10CmRequest inferIcd10CmRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<InferIcd10CmResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    InferIcd10CmResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<InferIcd10CmResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<InferIcd10CmRequest, InferIcd10CmResponse>()
                            .withOperationName("InferICD10CM").withMarshaller(new InferIcd10CmRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(inferIcd10CmRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * InferRxNorm detects medications as entities listed in a patient record and links to the normalized concept
     * identifiers in the RxNorm database from the National Library of Medicine. Amazon Comprehend Medical only detects
     * medical entities in English language texts.
     * </p>
     *
     * @param inferRxNormRequest
     * @return A Java Future containing the result of the InferRxNorm operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ServiceUnavailableException The Amazon Comprehend Medical service is temporarily unavailable. Please
     *         wait and then retry your request.</li>
     *         <li>TooManyRequestsException You have made too many requests within a short period of time. Wait for a
     *         short time and then try your request again. Contact customer support for more information about a service
     *         limit increase.</li>
     *         <li>InvalidRequestException The request that you made is invalid. Check your request to determine why
     *         it's invalid and then retry the request.</li>
     *         <li>InvalidEncodingException The input text was not in valid UTF-8 character encoding. Check your text
     *         then retry your request.</li>
     *         <li>TextSizeLimitExceededException The size of the text you submitted exceeds the size limit. Reduce the
     *         size of the text or use a smaller document and then retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ComprehendMedicalException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ComprehendMedicalAsyncClient.InferRxNorm
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/comprehendmedical-2018-10-30/InferRxNorm" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<InferRxNormResponse> inferRxNorm(InferRxNormRequest inferRxNormRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<InferRxNormResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    InferRxNormResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<InferRxNormResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<InferRxNormRequest, InferRxNormResponse>()
                            .withOperationName("InferRxNorm").withMarshaller(new InferRxNormRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(inferRxNormRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets a list of medical entity detection jobs that you have submitted.
     * </p>
     *
     * @param listEntitiesDetectionV2JobsRequest
     * @return A Java Future containing the result of the ListEntitiesDetectionV2Jobs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request that you made is invalid. Check your request to determine why
     *         it's invalid and then retry the request.</li>
     *         <li>ValidationException The filter that you specified for the operation is invalid. Check the filter
     *         values that you entered and try your request again.</li>
     *         <li>TooManyRequestsException You have made too many requests within a short period of time. Wait for a
     *         short time and then try your request again. Contact customer support for more information about a service
     *         limit increase.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ComprehendMedicalException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ComprehendMedicalAsyncClient.ListEntitiesDetectionV2Jobs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/comprehendmedical-2018-10-30/ListEntitiesDetectionV2Jobs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListEntitiesDetectionV2JobsResponse> listEntitiesDetectionV2Jobs(
            ListEntitiesDetectionV2JobsRequest listEntitiesDetectionV2JobsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListEntitiesDetectionV2JobsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListEntitiesDetectionV2JobsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListEntitiesDetectionV2JobsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListEntitiesDetectionV2JobsRequest, ListEntitiesDetectionV2JobsResponse>()
                            .withOperationName("ListEntitiesDetectionV2Jobs")
                            .withMarshaller(new ListEntitiesDetectionV2JobsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listEntitiesDetectionV2JobsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets a list of protected health information (PHI) detection jobs that you have submitted.
     * </p>
     *
     * @param listPhiDetectionJobsRequest
     * @return A Java Future containing the result of the ListPHIDetectionJobs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request that you made is invalid. Check your request to determine why
     *         it's invalid and then retry the request.</li>
     *         <li>ValidationException The filter that you specified for the operation is invalid. Check the filter
     *         values that you entered and try your request again.</li>
     *         <li>TooManyRequestsException You have made too many requests within a short period of time. Wait for a
     *         short time and then try your request again. Contact customer support for more information about a service
     *         limit increase.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ComprehendMedicalException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ComprehendMedicalAsyncClient.ListPHIDetectionJobs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/comprehendmedical-2018-10-30/ListPHIDetectionJobs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListPhiDetectionJobsResponse> listPHIDetectionJobs(
            ListPhiDetectionJobsRequest listPhiDetectionJobsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPhiDetectionJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListPhiDetectionJobsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPhiDetectionJobsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPhiDetectionJobsRequest, ListPhiDetectionJobsResponse>()
                            .withOperationName("ListPHIDetectionJobs")
                            .withMarshaller(new ListPhiDetectionJobsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listPhiDetectionJobsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts an asynchronous medical entity detection job for a collection of documents. Use the
     * <code>DescribeEntitiesDetectionV2Job</code> operation to track the status of a job.
     * </p>
     *
     * @param startEntitiesDetectionV2JobRequest
     * @return A Java Future containing the result of the StartEntitiesDetectionV2Job operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request that you made is invalid. Check your request to determine why
     *         it's invalid and then retry the request.</li>
     *         <li>TooManyRequestsException You have made too many requests within a short period of time. Wait for a
     *         short time and then try your request again. Contact customer support for more information about a service
     *         limit increase.</li>
     *         <li>ResourceNotFoundException The resource identified by the specified Amazon Resource Name (ARN) was not
     *         found. Check the ARN and try your request again.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ComprehendMedicalException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ComprehendMedicalAsyncClient.StartEntitiesDetectionV2Job
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/comprehendmedical-2018-10-30/StartEntitiesDetectionV2Job"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartEntitiesDetectionV2JobResponse> startEntitiesDetectionV2Job(
            StartEntitiesDetectionV2JobRequest startEntitiesDetectionV2JobRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartEntitiesDetectionV2JobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartEntitiesDetectionV2JobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartEntitiesDetectionV2JobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartEntitiesDetectionV2JobRequest, StartEntitiesDetectionV2JobResponse>()
                            .withOperationName("StartEntitiesDetectionV2Job")
                            .withMarshaller(new StartEntitiesDetectionV2JobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(startEntitiesDetectionV2JobRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts an asynchronous job to detect protected health information (PHI). Use the
     * <code>DescribePHIDetectionJob</code> operation to track the status of a job.
     * </p>
     *
     * @param startPhiDetectionJobRequest
     * @return A Java Future containing the result of the StartPHIDetectionJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request that you made is invalid. Check your request to determine why
     *         it's invalid and then retry the request.</li>
     *         <li>TooManyRequestsException You have made too many requests within a short period of time. Wait for a
     *         short time and then try your request again. Contact customer support for more information about a service
     *         limit increase.</li>
     *         <li>ResourceNotFoundException The resource identified by the specified Amazon Resource Name (ARN) was not
     *         found. Check the ARN and try your request again.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ComprehendMedicalException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ComprehendMedicalAsyncClient.StartPHIDetectionJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/comprehendmedical-2018-10-30/StartPHIDetectionJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartPhiDetectionJobResponse> startPHIDetectionJob(
            StartPhiDetectionJobRequest startPhiDetectionJobRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartPhiDetectionJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartPhiDetectionJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartPhiDetectionJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartPhiDetectionJobRequest, StartPhiDetectionJobResponse>()
                            .withOperationName("StartPHIDetectionJob")
                            .withMarshaller(new StartPhiDetectionJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(startPhiDetectionJobRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Stops a medical entities detection job in progress.
     * </p>
     *
     * @param stopEntitiesDetectionV2JobRequest
     * @return A Java Future containing the result of the StopEntitiesDetectionV2Job operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request that you made is invalid. Check your request to determine why
     *         it's invalid and then retry the request.</li>
     *         <li>ResourceNotFoundException The resource identified by the specified Amazon Resource Name (ARN) was not
     *         found. Check the ARN and try your request again.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ComprehendMedicalException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ComprehendMedicalAsyncClient.StopEntitiesDetectionV2Job
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/comprehendmedical-2018-10-30/StopEntitiesDetectionV2Job"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StopEntitiesDetectionV2JobResponse> stopEntitiesDetectionV2Job(
            StopEntitiesDetectionV2JobRequest stopEntitiesDetectionV2JobRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopEntitiesDetectionV2JobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StopEntitiesDetectionV2JobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StopEntitiesDetectionV2JobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopEntitiesDetectionV2JobRequest, StopEntitiesDetectionV2JobResponse>()
                            .withOperationName("StopEntitiesDetectionV2Job")
                            .withMarshaller(new StopEntitiesDetectionV2JobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(stopEntitiesDetectionV2JobRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Stops a protected health information (PHI) detection job in progress.
     * </p>
     *
     * @param stopPhiDetectionJobRequest
     * @return A Java Future containing the result of the StopPHIDetectionJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request that you made is invalid. Check your request to determine why
     *         it's invalid and then retry the request.</li>
     *         <li>ResourceNotFoundException The resource identified by the specified Amazon Resource Name (ARN) was not
     *         found. Check the ARN and try your request again.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ComprehendMedicalException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ComprehendMedicalAsyncClient.StopPHIDetectionJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/comprehendmedical-2018-10-30/StopPHIDetectionJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StopPhiDetectionJobResponse> stopPHIDetectionJob(
            StopPhiDetectionJobRequest stopPhiDetectionJobRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopPhiDetectionJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StopPhiDetectionJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StopPhiDetectionJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopPhiDetectionJobRequest, StopPhiDetectionJobResponse>()
                            .withOperationName("StopPHIDetectionJob")
                            .withMarshaller(new StopPhiDetectionJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(stopPhiDetectionJobRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ComprehendMedicalException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestException")
                                .exceptionBuilderSupplier(InvalidRequestException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidEncodingException")
                                .exceptionBuilderSupplier(InvalidEncodingException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TextSizeLimitExceededException")
                                .exceptionBuilderSupplier(TextSizeLimitExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyRequestsException")
                                .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
