/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.comprehendmedical;

import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.comprehendmedical.model.ComprehendMedicalException;
import software.amazon.awssdk.services.comprehendmedical.model.DescribeEntitiesDetectionV2JobRequest;
import software.amazon.awssdk.services.comprehendmedical.model.DescribeEntitiesDetectionV2JobResponse;
import software.amazon.awssdk.services.comprehendmedical.model.DescribePhiDetectionJobRequest;
import software.amazon.awssdk.services.comprehendmedical.model.DescribePhiDetectionJobResponse;
import software.amazon.awssdk.services.comprehendmedical.model.DetectEntitiesRequest;
import software.amazon.awssdk.services.comprehendmedical.model.DetectEntitiesResponse;
import software.amazon.awssdk.services.comprehendmedical.model.DetectEntitiesV2Request;
import software.amazon.awssdk.services.comprehendmedical.model.DetectEntitiesV2Response;
import software.amazon.awssdk.services.comprehendmedical.model.DetectPhiRequest;
import software.amazon.awssdk.services.comprehendmedical.model.DetectPhiResponse;
import software.amazon.awssdk.services.comprehendmedical.model.InferIcd10CmRequest;
import software.amazon.awssdk.services.comprehendmedical.model.InferIcd10CmResponse;
import software.amazon.awssdk.services.comprehendmedical.model.InferRxNormRequest;
import software.amazon.awssdk.services.comprehendmedical.model.InferRxNormResponse;
import software.amazon.awssdk.services.comprehendmedical.model.InternalServerException;
import software.amazon.awssdk.services.comprehendmedical.model.InvalidEncodingException;
import software.amazon.awssdk.services.comprehendmedical.model.InvalidRequestException;
import software.amazon.awssdk.services.comprehendmedical.model.ListEntitiesDetectionV2JobsRequest;
import software.amazon.awssdk.services.comprehendmedical.model.ListEntitiesDetectionV2JobsResponse;
import software.amazon.awssdk.services.comprehendmedical.model.ListPhiDetectionJobsRequest;
import software.amazon.awssdk.services.comprehendmedical.model.ListPhiDetectionJobsResponse;
import software.amazon.awssdk.services.comprehendmedical.model.ResourceNotFoundException;
import software.amazon.awssdk.services.comprehendmedical.model.ServiceUnavailableException;
import software.amazon.awssdk.services.comprehendmedical.model.StartEntitiesDetectionV2JobRequest;
import software.amazon.awssdk.services.comprehendmedical.model.StartEntitiesDetectionV2JobResponse;
import software.amazon.awssdk.services.comprehendmedical.model.StartPhiDetectionJobRequest;
import software.amazon.awssdk.services.comprehendmedical.model.StartPhiDetectionJobResponse;
import software.amazon.awssdk.services.comprehendmedical.model.StopEntitiesDetectionV2JobRequest;
import software.amazon.awssdk.services.comprehendmedical.model.StopEntitiesDetectionV2JobResponse;
import software.amazon.awssdk.services.comprehendmedical.model.StopPhiDetectionJobRequest;
import software.amazon.awssdk.services.comprehendmedical.model.StopPhiDetectionJobResponse;
import software.amazon.awssdk.services.comprehendmedical.model.TextSizeLimitExceededException;
import software.amazon.awssdk.services.comprehendmedical.model.TooManyRequestsException;
import software.amazon.awssdk.services.comprehendmedical.model.ValidationException;
import software.amazon.awssdk.services.comprehendmedical.transform.DescribeEntitiesDetectionV2JobRequestMarshaller;
import software.amazon.awssdk.services.comprehendmedical.transform.DescribePhiDetectionJobRequestMarshaller;
import software.amazon.awssdk.services.comprehendmedical.transform.DetectEntitiesRequestMarshaller;
import software.amazon.awssdk.services.comprehendmedical.transform.DetectEntitiesV2RequestMarshaller;
import software.amazon.awssdk.services.comprehendmedical.transform.DetectPhiRequestMarshaller;
import software.amazon.awssdk.services.comprehendmedical.transform.InferIcd10CmRequestMarshaller;
import software.amazon.awssdk.services.comprehendmedical.transform.InferRxNormRequestMarshaller;
import software.amazon.awssdk.services.comprehendmedical.transform.ListEntitiesDetectionV2JobsRequestMarshaller;
import software.amazon.awssdk.services.comprehendmedical.transform.ListPhiDetectionJobsRequestMarshaller;
import software.amazon.awssdk.services.comprehendmedical.transform.StartEntitiesDetectionV2JobRequestMarshaller;
import software.amazon.awssdk.services.comprehendmedical.transform.StartPhiDetectionJobRequestMarshaller;
import software.amazon.awssdk.services.comprehendmedical.transform.StopEntitiesDetectionV2JobRequestMarshaller;
import software.amazon.awssdk.services.comprehendmedical.transform.StopPhiDetectionJobRequestMarshaller;

/**
 * Internal implementation of {@link ComprehendMedicalClient}.
 *
 * @see ComprehendMedicalClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultComprehendMedicalClient implements ComprehendMedicalClient {
    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultComprehendMedicalClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Gets the properties associated with a medical entities detection job. Use this operation to get the status of a
     * detection job.
     * </p>
     *
     * @param describeEntitiesDetectionV2JobRequest
     * @return Result of the DescribeEntitiesDetectionV2Job operation returned by the service.
     * @throws InvalidRequestException
     *         The request that you made is invalid. Check your request to determine why it's invalid and then retry the
     *         request.
     * @throws TooManyRequestsException
     *         You have made too many requests within a short period of time. Wait for a short time and then try your
     *         request again. Contact customer support for more information about a service limit increase.
     * @throws ResourceNotFoundException
     *         The resource identified by the specified Amazon Resource Name (ARN) was not found. Check the ARN and try
     *         your request again.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ComprehendMedicalException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ComprehendMedicalClient.DescribeEntitiesDetectionV2Job
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/comprehendmedical-2018-10-30/DescribeEntitiesDetectionV2Job"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeEntitiesDetectionV2JobResponse describeEntitiesDetectionV2Job(
            DescribeEntitiesDetectionV2JobRequest describeEntitiesDetectionV2JobRequest) throws InvalidRequestException,
            TooManyRequestsException, ResourceNotFoundException, InternalServerException, AwsServiceException,
            SdkClientException, ComprehendMedicalException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeEntitiesDetectionV2JobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeEntitiesDetectionV2JobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<DescribeEntitiesDetectionV2JobRequest, DescribeEntitiesDetectionV2JobResponse>()
                        .withOperationName("DescribeEntitiesDetectionV2Job").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(describeEntitiesDetectionV2JobRequest)
                        .withMarshaller(new DescribeEntitiesDetectionV2JobRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Gets the properties associated with a protected health information (PHI) detection job. Use this operation to get
     * the status of a detection job.
     * </p>
     *
     * @param describePhiDetectionJobRequest
     * @return Result of the DescribePHIDetectionJob operation returned by the service.
     * @throws InvalidRequestException
     *         The request that you made is invalid. Check your request to determine why it's invalid and then retry the
     *         request.
     * @throws TooManyRequestsException
     *         You have made too many requests within a short period of time. Wait for a short time and then try your
     *         request again. Contact customer support for more information about a service limit increase.
     * @throws ResourceNotFoundException
     *         The resource identified by the specified Amazon Resource Name (ARN) was not found. Check the ARN and try
     *         your request again.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ComprehendMedicalException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ComprehendMedicalClient.DescribePHIDetectionJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/comprehendmedical-2018-10-30/DescribePHIDetectionJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribePhiDetectionJobResponse describePHIDetectionJob(DescribePhiDetectionJobRequest describePhiDetectionJobRequest)
            throws InvalidRequestException, TooManyRequestsException, ResourceNotFoundException, InternalServerException,
            AwsServiceException, SdkClientException, ComprehendMedicalException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribePhiDetectionJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribePhiDetectionJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DescribePhiDetectionJobRequest, DescribePhiDetectionJobResponse>()
                .withOperationName("DescribePHIDetectionJob").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describePhiDetectionJobRequest)
                .withMarshaller(new DescribePhiDetectionJobRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * The <code>DetectEntities</code> operation is deprecated. You should use the <a>DetectEntitiesV2</a> operation
     * instead.
     * </p>
     * <p>
     * Inspects the clinical text for a variety of medical entities and returns specific information about them such as
     * entity category, location, and confidence score on that information .
     * </p>
     *
     * @param detectEntitiesRequest
     * @return Result of the DetectEntities operation returned by the service.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws ServiceUnavailableException
     *         The Amazon Comprehend Medical service is temporarily unavailable. Please wait and then retry your
     *         request.
     * @throws TooManyRequestsException
     *         You have made too many requests within a short period of time. Wait for a short time and then try your
     *         request again. Contact customer support for more information about a service limit increase.
     * @throws InvalidRequestException
     *         The request that you made is invalid. Check your request to determine why it's invalid and then retry the
     *         request.
     * @throws InvalidEncodingException
     *         The input text was not in valid UTF-8 character encoding. Check your text then retry your request.
     * @throws TextSizeLimitExceededException
     *         The size of the text you submitted exceeds the size limit. Reduce the size of the text or use a smaller
     *         document and then retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ComprehendMedicalException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ComprehendMedicalClient.DetectEntities
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/comprehendmedical-2018-10-30/DetectEntities"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DetectEntitiesResponse detectEntities(DetectEntitiesRequest detectEntitiesRequest) throws InternalServerException,
            ServiceUnavailableException, TooManyRequestsException, InvalidRequestException, InvalidEncodingException,
            TextSizeLimitExceededException, AwsServiceException, SdkClientException, ComprehendMedicalException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DetectEntitiesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DetectEntitiesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DetectEntitiesRequest, DetectEntitiesResponse>()
                .withOperationName("DetectEntities").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(detectEntitiesRequest)
                .withMarshaller(new DetectEntitiesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Inspects the clinical text for a variety of medical entities and returns specific information about them such as
     * entity category, location, and confidence score on that information. Amazon Comprehend Medical only detects
     * medical entities in English language texts.
     * </p>
     * <p>
     * The <code>DetectEntitiesV2</code> operation replaces the <a>DetectEntities</a> operation. This new action uses a
     * different model for determining the entities in your medical text and changes the way that some entities are
     * returned in the output. You should use the <code>DetectEntitiesV2</code> operation in all new applications.
     * </p>
     * <p>
     * The <code>DetectEntitiesV2</code> operation returns the <code>Acuity</code> and <code>Direction</code> entities
     * as attributes instead of types.
     * </p>
     *
     * @param detectEntitiesV2Request
     * @return Result of the DetectEntitiesV2 operation returned by the service.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws ServiceUnavailableException
     *         The Amazon Comprehend Medical service is temporarily unavailable. Please wait and then retry your
     *         request.
     * @throws TooManyRequestsException
     *         You have made too many requests within a short period of time. Wait for a short time and then try your
     *         request again. Contact customer support for more information about a service limit increase.
     * @throws InvalidRequestException
     *         The request that you made is invalid. Check your request to determine why it's invalid and then retry the
     *         request.
     * @throws InvalidEncodingException
     *         The input text was not in valid UTF-8 character encoding. Check your text then retry your request.
     * @throws TextSizeLimitExceededException
     *         The size of the text you submitted exceeds the size limit. Reduce the size of the text or use a smaller
     *         document and then retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ComprehendMedicalException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ComprehendMedicalClient.DetectEntitiesV2
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/comprehendmedical-2018-10-30/DetectEntitiesV2"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DetectEntitiesV2Response detectEntitiesV2(DetectEntitiesV2Request detectEntitiesV2Request)
            throws InternalServerException, ServiceUnavailableException, TooManyRequestsException, InvalidRequestException,
            InvalidEncodingException, TextSizeLimitExceededException, AwsServiceException, SdkClientException,
            ComprehendMedicalException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DetectEntitiesV2Response> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DetectEntitiesV2Response::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DetectEntitiesV2Request, DetectEntitiesV2Response>()
                .withOperationName("DetectEntitiesV2").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(detectEntitiesV2Request)
                .withMarshaller(new DetectEntitiesV2RequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Inspects the clinical text for protected health information (PHI) entities and returns the entity category,
     * location, and confidence score for each entity. Amazon Comprehend Medical only detects entities in English
     * language texts.
     * </p>
     *
     * @param detectPhiRequest
     * @return Result of the DetectPHI operation returned by the service.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws ServiceUnavailableException
     *         The Amazon Comprehend Medical service is temporarily unavailable. Please wait and then retry your
     *         request.
     * @throws TooManyRequestsException
     *         You have made too many requests within a short period of time. Wait for a short time and then try your
     *         request again. Contact customer support for more information about a service limit increase.
     * @throws InvalidRequestException
     *         The request that you made is invalid. Check your request to determine why it's invalid and then retry the
     *         request.
     * @throws InvalidEncodingException
     *         The input text was not in valid UTF-8 character encoding. Check your text then retry your request.
     * @throws TextSizeLimitExceededException
     *         The size of the text you submitted exceeds the size limit. Reduce the size of the text or use a smaller
     *         document and then retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ComprehendMedicalException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ComprehendMedicalClient.DetectPHI
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/comprehendmedical-2018-10-30/DetectPHI" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DetectPhiResponse detectPHI(DetectPhiRequest detectPhiRequest) throws InternalServerException,
            ServiceUnavailableException, TooManyRequestsException, InvalidRequestException, InvalidEncodingException,
            TextSizeLimitExceededException, AwsServiceException, SdkClientException, ComprehendMedicalException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DetectPhiResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DetectPhiResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DetectPhiRequest, DetectPhiResponse>()
                .withOperationName("DetectPHI").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(detectPhiRequest)
                .withMarshaller(new DetectPhiRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * InferICD10CM detects medical conditions as entities listed in a patient record and links those entities to
     * normalized concept identifiers in the ICD-10-CM knowledge base from the Centers for Disease Control. Amazon
     * Comprehend Medical only detects medical entities in English language texts.
     * </p>
     *
     * @param inferIcd10CmRequest
     * @return Result of the InferICD10CM operation returned by the service.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws ServiceUnavailableException
     *         The Amazon Comprehend Medical service is temporarily unavailable. Please wait and then retry your
     *         request.
     * @throws TooManyRequestsException
     *         You have made too many requests within a short period of time. Wait for a short time and then try your
     *         request again. Contact customer support for more information about a service limit increase.
     * @throws InvalidRequestException
     *         The request that you made is invalid. Check your request to determine why it's invalid and then retry the
     *         request.
     * @throws InvalidEncodingException
     *         The input text was not in valid UTF-8 character encoding. Check your text then retry your request.
     * @throws TextSizeLimitExceededException
     *         The size of the text you submitted exceeds the size limit. Reduce the size of the text or use a smaller
     *         document and then retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ComprehendMedicalException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ComprehendMedicalClient.InferICD10CM
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/comprehendmedical-2018-10-30/InferICD10CM" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public InferIcd10CmResponse inferICD10CM(InferIcd10CmRequest inferIcd10CmRequest) throws InternalServerException,
            ServiceUnavailableException, TooManyRequestsException, InvalidRequestException, InvalidEncodingException,
            TextSizeLimitExceededException, AwsServiceException, SdkClientException, ComprehendMedicalException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<InferIcd10CmResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                InferIcd10CmResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<InferIcd10CmRequest, InferIcd10CmResponse>()
                .withOperationName("InferICD10CM").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(inferIcd10CmRequest)
                .withMarshaller(new InferIcd10CmRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * InferRxNorm detects medications as entities listed in a patient record and links to the normalized concept
     * identifiers in the RxNorm database from the National Library of Medicine. Amazon Comprehend Medical only detects
     * medical entities in English language texts.
     * </p>
     *
     * @param inferRxNormRequest
     * @return Result of the InferRxNorm operation returned by the service.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws ServiceUnavailableException
     *         The Amazon Comprehend Medical service is temporarily unavailable. Please wait and then retry your
     *         request.
     * @throws TooManyRequestsException
     *         You have made too many requests within a short period of time. Wait for a short time and then try your
     *         request again. Contact customer support for more information about a service limit increase.
     * @throws InvalidRequestException
     *         The request that you made is invalid. Check your request to determine why it's invalid and then retry the
     *         request.
     * @throws InvalidEncodingException
     *         The input text was not in valid UTF-8 character encoding. Check your text then retry your request.
     * @throws TextSizeLimitExceededException
     *         The size of the text you submitted exceeds the size limit. Reduce the size of the text or use a smaller
     *         document and then retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ComprehendMedicalException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ComprehendMedicalClient.InferRxNorm
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/comprehendmedical-2018-10-30/InferRxNorm" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public InferRxNormResponse inferRxNorm(InferRxNormRequest inferRxNormRequest) throws InternalServerException,
            ServiceUnavailableException, TooManyRequestsException, InvalidRequestException, InvalidEncodingException,
            TextSizeLimitExceededException, AwsServiceException, SdkClientException, ComprehendMedicalException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<InferRxNormResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                InferRxNormResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<InferRxNormRequest, InferRxNormResponse>()
                .withOperationName("InferRxNorm").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(inferRxNormRequest)
                .withMarshaller(new InferRxNormRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Gets a list of medical entity detection jobs that you have submitted.
     * </p>
     *
     * @param listEntitiesDetectionV2JobsRequest
     * @return Result of the ListEntitiesDetectionV2Jobs operation returned by the service.
     * @throws InvalidRequestException
     *         The request that you made is invalid. Check your request to determine why it's invalid and then retry the
     *         request.
     * @throws ValidationException
     *         The filter that you specified for the operation is invalid. Check the filter values that you entered and
     *         try your request again.
     * @throws TooManyRequestsException
     *         You have made too many requests within a short period of time. Wait for a short time and then try your
     *         request again. Contact customer support for more information about a service limit increase.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ComprehendMedicalException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ComprehendMedicalClient.ListEntitiesDetectionV2Jobs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/comprehendmedical-2018-10-30/ListEntitiesDetectionV2Jobs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListEntitiesDetectionV2JobsResponse listEntitiesDetectionV2Jobs(
            ListEntitiesDetectionV2JobsRequest listEntitiesDetectionV2JobsRequest) throws InvalidRequestException,
            ValidationException, TooManyRequestsException, InternalServerException, AwsServiceException, SdkClientException,
            ComprehendMedicalException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListEntitiesDetectionV2JobsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListEntitiesDetectionV2JobsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<ListEntitiesDetectionV2JobsRequest, ListEntitiesDetectionV2JobsResponse>()
                        .withOperationName("ListEntitiesDetectionV2Jobs").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(listEntitiesDetectionV2JobsRequest)
                        .withMarshaller(new ListEntitiesDetectionV2JobsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Gets a list of protected health information (PHI) detection jobs that you have submitted.
     * </p>
     *
     * @param listPhiDetectionJobsRequest
     * @return Result of the ListPHIDetectionJobs operation returned by the service.
     * @throws InvalidRequestException
     *         The request that you made is invalid. Check your request to determine why it's invalid and then retry the
     *         request.
     * @throws ValidationException
     *         The filter that you specified for the operation is invalid. Check the filter values that you entered and
     *         try your request again.
     * @throws TooManyRequestsException
     *         You have made too many requests within a short period of time. Wait for a short time and then try your
     *         request again. Contact customer support for more information about a service limit increase.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ComprehendMedicalException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ComprehendMedicalClient.ListPHIDetectionJobs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/comprehendmedical-2018-10-30/ListPHIDetectionJobs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListPhiDetectionJobsResponse listPHIDetectionJobs(ListPhiDetectionJobsRequest listPhiDetectionJobsRequest)
            throws InvalidRequestException, ValidationException, TooManyRequestsException, InternalServerException,
            AwsServiceException, SdkClientException, ComprehendMedicalException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListPhiDetectionJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListPhiDetectionJobsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListPhiDetectionJobsRequest, ListPhiDetectionJobsResponse>()
                .withOperationName("ListPHIDetectionJobs").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listPhiDetectionJobsRequest)
                .withMarshaller(new ListPhiDetectionJobsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Starts an asynchronous medical entity detection job for a collection of documents. Use the
     * <code>DescribeEntitiesDetectionV2Job</code> operation to track the status of a job.
     * </p>
     *
     * @param startEntitiesDetectionV2JobRequest
     * @return Result of the StartEntitiesDetectionV2Job operation returned by the service.
     * @throws InvalidRequestException
     *         The request that you made is invalid. Check your request to determine why it's invalid and then retry the
     *         request.
     * @throws TooManyRequestsException
     *         You have made too many requests within a short period of time. Wait for a short time and then try your
     *         request again. Contact customer support for more information about a service limit increase.
     * @throws ResourceNotFoundException
     *         The resource identified by the specified Amazon Resource Name (ARN) was not found. Check the ARN and try
     *         your request again.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ComprehendMedicalException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ComprehendMedicalClient.StartEntitiesDetectionV2Job
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/comprehendmedical-2018-10-30/StartEntitiesDetectionV2Job"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartEntitiesDetectionV2JobResponse startEntitiesDetectionV2Job(
            StartEntitiesDetectionV2JobRequest startEntitiesDetectionV2JobRequest) throws InvalidRequestException,
            TooManyRequestsException, ResourceNotFoundException, InternalServerException, AwsServiceException,
            SdkClientException, ComprehendMedicalException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartEntitiesDetectionV2JobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartEntitiesDetectionV2JobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<StartEntitiesDetectionV2JobRequest, StartEntitiesDetectionV2JobResponse>()
                        .withOperationName("StartEntitiesDetectionV2Job").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(startEntitiesDetectionV2JobRequest)
                        .withMarshaller(new StartEntitiesDetectionV2JobRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Starts an asynchronous job to detect protected health information (PHI). Use the
     * <code>DescribePHIDetectionJob</code> operation to track the status of a job.
     * </p>
     *
     * @param startPhiDetectionJobRequest
     * @return Result of the StartPHIDetectionJob operation returned by the service.
     * @throws InvalidRequestException
     *         The request that you made is invalid. Check your request to determine why it's invalid and then retry the
     *         request.
     * @throws TooManyRequestsException
     *         You have made too many requests within a short period of time. Wait for a short time and then try your
     *         request again. Contact customer support for more information about a service limit increase.
     * @throws ResourceNotFoundException
     *         The resource identified by the specified Amazon Resource Name (ARN) was not found. Check the ARN and try
     *         your request again.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ComprehendMedicalException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ComprehendMedicalClient.StartPHIDetectionJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/comprehendmedical-2018-10-30/StartPHIDetectionJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartPhiDetectionJobResponse startPHIDetectionJob(StartPhiDetectionJobRequest startPhiDetectionJobRequest)
            throws InvalidRequestException, TooManyRequestsException, ResourceNotFoundException, InternalServerException,
            AwsServiceException, SdkClientException, ComprehendMedicalException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartPhiDetectionJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartPhiDetectionJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<StartPhiDetectionJobRequest, StartPhiDetectionJobResponse>()
                .withOperationName("StartPHIDetectionJob").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(startPhiDetectionJobRequest)
                .withMarshaller(new StartPhiDetectionJobRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Stops a medical entities detection job in progress.
     * </p>
     *
     * @param stopEntitiesDetectionV2JobRequest
     * @return Result of the StopEntitiesDetectionV2Job operation returned by the service.
     * @throws InvalidRequestException
     *         The request that you made is invalid. Check your request to determine why it's invalid and then retry the
     *         request.
     * @throws ResourceNotFoundException
     *         The resource identified by the specified Amazon Resource Name (ARN) was not found. Check the ARN and try
     *         your request again.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ComprehendMedicalException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ComprehendMedicalClient.StopEntitiesDetectionV2Job
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/comprehendmedical-2018-10-30/StopEntitiesDetectionV2Job"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StopEntitiesDetectionV2JobResponse stopEntitiesDetectionV2Job(
            StopEntitiesDetectionV2JobRequest stopEntitiesDetectionV2JobRequest) throws InvalidRequestException,
            ResourceNotFoundException, InternalServerException, AwsServiceException, SdkClientException,
            ComprehendMedicalException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StopEntitiesDetectionV2JobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StopEntitiesDetectionV2JobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<StopEntitiesDetectionV2JobRequest, StopEntitiesDetectionV2JobResponse>()
                        .withOperationName("StopEntitiesDetectionV2Job").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(stopEntitiesDetectionV2JobRequest)
                        .withMarshaller(new StopEntitiesDetectionV2JobRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Stops a protected health information (PHI) detection job in progress.
     * </p>
     *
     * @param stopPhiDetectionJobRequest
     * @return Result of the StopPHIDetectionJob operation returned by the service.
     * @throws InvalidRequestException
     *         The request that you made is invalid. Check your request to determine why it's invalid and then retry the
     *         request.
     * @throws ResourceNotFoundException
     *         The resource identified by the specified Amazon Resource Name (ARN) was not found. Check the ARN and try
     *         your request again.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ComprehendMedicalException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ComprehendMedicalClient.StopPHIDetectionJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/comprehendmedical-2018-10-30/StopPHIDetectionJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StopPhiDetectionJobResponse stopPHIDetectionJob(StopPhiDetectionJobRequest stopPhiDetectionJobRequest)
            throws InvalidRequestException, ResourceNotFoundException, InternalServerException, AwsServiceException,
            SdkClientException, ComprehendMedicalException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StopPhiDetectionJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StopPhiDetectionJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<StopPhiDetectionJobRequest, StopPhiDetectionJobResponse>()
                .withOperationName("StopPHIDetectionJob").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(stopPhiDetectionJobRequest)
                .withMarshaller(new StopPhiDetectionJobRequestMarshaller(protocolFactory)));
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ComprehendMedicalException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestException")
                                .exceptionBuilderSupplier(InvalidRequestException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidEncodingException")
                                .exceptionBuilderSupplier(InvalidEncodingException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TextSizeLimitExceededException")
                                .exceptionBuilderSupplier(TextSizeLimitExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyRequestsException")
                                .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
