/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cognitosync;

import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.cognitosync.model.AlreadyStreamedException;
import software.amazon.awssdk.services.cognitosync.model.BulkPublishRequest;
import software.amazon.awssdk.services.cognitosync.model.BulkPublishResponse;
import software.amazon.awssdk.services.cognitosync.model.CognitoSyncException;
import software.amazon.awssdk.services.cognitosync.model.ConcurrentModificationException;
import software.amazon.awssdk.services.cognitosync.model.DeleteDatasetRequest;
import software.amazon.awssdk.services.cognitosync.model.DeleteDatasetResponse;
import software.amazon.awssdk.services.cognitosync.model.DescribeDatasetRequest;
import software.amazon.awssdk.services.cognitosync.model.DescribeDatasetResponse;
import software.amazon.awssdk.services.cognitosync.model.DescribeIdentityPoolUsageRequest;
import software.amazon.awssdk.services.cognitosync.model.DescribeIdentityPoolUsageResponse;
import software.amazon.awssdk.services.cognitosync.model.DescribeIdentityUsageRequest;
import software.amazon.awssdk.services.cognitosync.model.DescribeIdentityUsageResponse;
import software.amazon.awssdk.services.cognitosync.model.DuplicateRequestException;
import software.amazon.awssdk.services.cognitosync.model.GetBulkPublishDetailsRequest;
import software.amazon.awssdk.services.cognitosync.model.GetBulkPublishDetailsResponse;
import software.amazon.awssdk.services.cognitosync.model.GetCognitoEventsRequest;
import software.amazon.awssdk.services.cognitosync.model.GetCognitoEventsResponse;
import software.amazon.awssdk.services.cognitosync.model.GetIdentityPoolConfigurationRequest;
import software.amazon.awssdk.services.cognitosync.model.GetIdentityPoolConfigurationResponse;
import software.amazon.awssdk.services.cognitosync.model.InternalErrorException;
import software.amazon.awssdk.services.cognitosync.model.InvalidConfigurationException;
import software.amazon.awssdk.services.cognitosync.model.InvalidLambdaFunctionOutputException;
import software.amazon.awssdk.services.cognitosync.model.InvalidParameterException;
import software.amazon.awssdk.services.cognitosync.model.LambdaThrottledException;
import software.amazon.awssdk.services.cognitosync.model.LimitExceededException;
import software.amazon.awssdk.services.cognitosync.model.ListDatasetsRequest;
import software.amazon.awssdk.services.cognitosync.model.ListDatasetsResponse;
import software.amazon.awssdk.services.cognitosync.model.ListIdentityPoolUsageRequest;
import software.amazon.awssdk.services.cognitosync.model.ListIdentityPoolUsageResponse;
import software.amazon.awssdk.services.cognitosync.model.ListRecordsRequest;
import software.amazon.awssdk.services.cognitosync.model.ListRecordsResponse;
import software.amazon.awssdk.services.cognitosync.model.NotAuthorizedException;
import software.amazon.awssdk.services.cognitosync.model.RegisterDeviceRequest;
import software.amazon.awssdk.services.cognitosync.model.RegisterDeviceResponse;
import software.amazon.awssdk.services.cognitosync.model.ResourceConflictException;
import software.amazon.awssdk.services.cognitosync.model.ResourceNotFoundException;
import software.amazon.awssdk.services.cognitosync.model.SetCognitoEventsRequest;
import software.amazon.awssdk.services.cognitosync.model.SetCognitoEventsResponse;
import software.amazon.awssdk.services.cognitosync.model.SetIdentityPoolConfigurationRequest;
import software.amazon.awssdk.services.cognitosync.model.SetIdentityPoolConfigurationResponse;
import software.amazon.awssdk.services.cognitosync.model.SubscribeToDatasetRequest;
import software.amazon.awssdk.services.cognitosync.model.SubscribeToDatasetResponse;
import software.amazon.awssdk.services.cognitosync.model.TooManyRequestsException;
import software.amazon.awssdk.services.cognitosync.model.UnsubscribeFromDatasetRequest;
import software.amazon.awssdk.services.cognitosync.model.UnsubscribeFromDatasetResponse;
import software.amazon.awssdk.services.cognitosync.model.UpdateRecordsRequest;
import software.amazon.awssdk.services.cognitosync.model.UpdateRecordsResponse;
import software.amazon.awssdk.services.cognitosync.transform.BulkPublishRequestMarshaller;
import software.amazon.awssdk.services.cognitosync.transform.DeleteDatasetRequestMarshaller;
import software.amazon.awssdk.services.cognitosync.transform.DescribeDatasetRequestMarshaller;
import software.amazon.awssdk.services.cognitosync.transform.DescribeIdentityPoolUsageRequestMarshaller;
import software.amazon.awssdk.services.cognitosync.transform.DescribeIdentityUsageRequestMarshaller;
import software.amazon.awssdk.services.cognitosync.transform.GetBulkPublishDetailsRequestMarshaller;
import software.amazon.awssdk.services.cognitosync.transform.GetCognitoEventsRequestMarshaller;
import software.amazon.awssdk.services.cognitosync.transform.GetIdentityPoolConfigurationRequestMarshaller;
import software.amazon.awssdk.services.cognitosync.transform.ListDatasetsRequestMarshaller;
import software.amazon.awssdk.services.cognitosync.transform.ListIdentityPoolUsageRequestMarshaller;
import software.amazon.awssdk.services.cognitosync.transform.ListRecordsRequestMarshaller;
import software.amazon.awssdk.services.cognitosync.transform.RegisterDeviceRequestMarshaller;
import software.amazon.awssdk.services.cognitosync.transform.SetCognitoEventsRequestMarshaller;
import software.amazon.awssdk.services.cognitosync.transform.SetIdentityPoolConfigurationRequestMarshaller;
import software.amazon.awssdk.services.cognitosync.transform.SubscribeToDatasetRequestMarshaller;
import software.amazon.awssdk.services.cognitosync.transform.UnsubscribeFromDatasetRequestMarshaller;
import software.amazon.awssdk.services.cognitosync.transform.UpdateRecordsRequestMarshaller;

/**
 * Internal implementation of {@link CognitoSyncClient}.
 *
 * @see CognitoSyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultCognitoSyncClient implements CognitoSyncClient {
    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultCognitoSyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Initiates a bulk publish of all existing datasets for an Identity Pool to the configured stream. Customers are
     * limited to one successful bulk publish per 24 hours. Bulk publish is an asynchronous request, customers can see
     * the status of the request via the GetBulkPublishDetails operation.
     * </p>
     * <p>
     * This API can only be called with developer credentials. You cannot call this API with the temporary user
     * credentials provided by Cognito Identity.
     * </p>
     *
     * @param bulkPublishRequest
     *        The input for the BulkPublish operation.
     * @return Result of the BulkPublish operation returned by the service.
     * @throws NotAuthorizedException
     *         Thrown when a user is not authorized to access the requested resource.
     * @throws InvalidParameterException
     *         Thrown when a request parameter does not comply with the associated constraints.
     * @throws ResourceNotFoundException
     *         Thrown if the resource doesn't exist.
     * @throws InternalErrorException
     *         Indicates an internal service error.
     * @throws DuplicateRequestException
     *         An exception thrown when there is an IN_PROGRESS bulk publish operation for the given identity pool.
     * @throws AlreadyStreamedException
     *         An exception thrown when a bulk publish operation is requested less than 24 hours after a previous bulk
     *         publish operation completed successfully.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CognitoSyncException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CognitoSyncClient.BulkPublish
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-sync-2014-06-30/BulkPublish" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public BulkPublishResponse bulkPublish(BulkPublishRequest bulkPublishRequest) throws NotAuthorizedException,
            InvalidParameterException, ResourceNotFoundException, InternalErrorException, DuplicateRequestException,
            AlreadyStreamedException, AwsServiceException, SdkClientException, CognitoSyncException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<BulkPublishResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                BulkPublishResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<BulkPublishRequest, BulkPublishResponse>()
                .withOperationName("BulkPublish").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(bulkPublishRequest)
                .withMarshaller(new BulkPublishRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes the specific dataset. The dataset will be deleted permanently, and the action can't be undone. Datasets
     * that this dataset was merged with will no longer report the merge. Any subsequent operation on this dataset will
     * result in a ResourceNotFoundException.
     * </p>
     * <p>
     * This API can be called with temporary user credentials provided by Cognito Identity or with developer
     * credentials.
     * </p>
     *
     * @param deleteDatasetRequest
     *        A request to delete the specific dataset.
     * @return Result of the DeleteDataset operation returned by the service.
     * @throws NotAuthorizedException
     *         Thrown when a user is not authorized to access the requested resource.
     * @throws InvalidParameterException
     *         Thrown when a request parameter does not comply with the associated constraints.
     * @throws ResourceNotFoundException
     *         Thrown if the resource doesn't exist.
     * @throws InternalErrorException
     *         Indicates an internal service error.
     * @throws TooManyRequestsException
     *         Thrown if the request is throttled.
     * @throws ResourceConflictException
     *         Thrown if an update can't be applied because the resource was changed by another call and this would
     *         result in a conflict.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CognitoSyncException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CognitoSyncClient.DeleteDataset
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-sync-2014-06-30/DeleteDataset" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteDatasetResponse deleteDataset(DeleteDatasetRequest deleteDatasetRequest) throws NotAuthorizedException,
            InvalidParameterException, ResourceNotFoundException, InternalErrorException, TooManyRequestsException,
            ResourceConflictException, AwsServiceException, SdkClientException, CognitoSyncException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteDatasetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteDatasetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeleteDatasetRequest, DeleteDatasetResponse>()
                .withOperationName("DeleteDataset").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteDatasetRequest)
                .withMarshaller(new DeleteDatasetRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Gets meta data about a dataset by identity and dataset name. With Amazon Cognito Sync, each identity has access
     * only to its own data. Thus, the credentials used to make this API call need to have access to the identity data.
     * </p>
     * <p>
     * This API can be called with temporary user credentials provided by Cognito Identity or with developer
     * credentials. You should use Cognito Identity credentials to make this API call.
     * </p>
     *
     * @param describeDatasetRequest
     *        A request for meta data about a dataset (creation date, number of records, size) by owner and dataset
     *        name.
     * @return Result of the DescribeDataset operation returned by the service.
     * @throws NotAuthorizedException
     *         Thrown when a user is not authorized to access the requested resource.
     * @throws InvalidParameterException
     *         Thrown when a request parameter does not comply with the associated constraints.
     * @throws ResourceNotFoundException
     *         Thrown if the resource doesn't exist.
     * @throws InternalErrorException
     *         Indicates an internal service error.
     * @throws TooManyRequestsException
     *         Thrown if the request is throttled.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CognitoSyncException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CognitoSyncClient.DescribeDataset
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-sync-2014-06-30/DescribeDataset" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeDatasetResponse describeDataset(DescribeDatasetRequest describeDatasetRequest) throws NotAuthorizedException,
            InvalidParameterException, ResourceNotFoundException, InternalErrorException, TooManyRequestsException,
            AwsServiceException, SdkClientException, CognitoSyncException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeDatasetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeDatasetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DescribeDatasetRequest, DescribeDatasetResponse>()
                .withOperationName("DescribeDataset").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeDatasetRequest)
                .withMarshaller(new DescribeDatasetRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Gets usage details (for example, data storage) about a particular identity pool.
     * </p>
     * <p>
     * This API can only be called with developer credentials. You cannot call this API with the temporary user
     * credentials provided by Cognito Identity.
     * </p>
     *
     * @param describeIdentityPoolUsageRequest
     *        A request for usage information about the identity pool.
     * @return Result of the DescribeIdentityPoolUsage operation returned by the service.
     * @throws NotAuthorizedException
     *         Thrown when a user is not authorized to access the requested resource.
     * @throws InvalidParameterException
     *         Thrown when a request parameter does not comply with the associated constraints.
     * @throws ResourceNotFoundException
     *         Thrown if the resource doesn't exist.
     * @throws InternalErrorException
     *         Indicates an internal service error.
     * @throws TooManyRequestsException
     *         Thrown if the request is throttled.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CognitoSyncException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CognitoSyncClient.DescribeIdentityPoolUsage
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-sync-2014-06-30/DescribeIdentityPoolUsage"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeIdentityPoolUsageResponse describeIdentityPoolUsage(
            DescribeIdentityPoolUsageRequest describeIdentityPoolUsageRequest) throws NotAuthorizedException,
            InvalidParameterException, ResourceNotFoundException, InternalErrorException, TooManyRequestsException,
            AwsServiceException, SdkClientException, CognitoSyncException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeIdentityPoolUsageResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeIdentityPoolUsageResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<DescribeIdentityPoolUsageRequest, DescribeIdentityPoolUsageResponse>()
                        .withOperationName("DescribeIdentityPoolUsage").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(describeIdentityPoolUsageRequest)
                        .withMarshaller(new DescribeIdentityPoolUsageRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Gets usage information for an identity, including number of datasets and data usage.
     * </p>
     * <p>
     * This API can be called with temporary user credentials provided by Cognito Identity or with developer
     * credentials.
     * </p>
     *
     * @param describeIdentityUsageRequest
     *        A request for information about the usage of an identity pool.
     * @return Result of the DescribeIdentityUsage operation returned by the service.
     * @throws NotAuthorizedException
     *         Thrown when a user is not authorized to access the requested resource.
     * @throws InvalidParameterException
     *         Thrown when a request parameter does not comply with the associated constraints.
     * @throws ResourceNotFoundException
     *         Thrown if the resource doesn't exist.
     * @throws InternalErrorException
     *         Indicates an internal service error.
     * @throws TooManyRequestsException
     *         Thrown if the request is throttled.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CognitoSyncException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CognitoSyncClient.DescribeIdentityUsage
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-sync-2014-06-30/DescribeIdentityUsage"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeIdentityUsageResponse describeIdentityUsage(DescribeIdentityUsageRequest describeIdentityUsageRequest)
            throws NotAuthorizedException, InvalidParameterException, ResourceNotFoundException, InternalErrorException,
            TooManyRequestsException, AwsServiceException, SdkClientException, CognitoSyncException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeIdentityUsageResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeIdentityUsageResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DescribeIdentityUsageRequest, DescribeIdentityUsageResponse>()
                .withOperationName("DescribeIdentityUsage").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeIdentityUsageRequest)
                .withMarshaller(new DescribeIdentityUsageRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Get the status of the last BulkPublish operation for an identity pool.
     * </p>
     * <p>
     * This API can only be called with developer credentials. You cannot call this API with the temporary user
     * credentials provided by Cognito Identity.
     * </p>
     *
     * @param getBulkPublishDetailsRequest
     *        The input for the GetBulkPublishDetails operation.
     * @return Result of the GetBulkPublishDetails operation returned by the service.
     * @throws NotAuthorizedException
     *         Thrown when a user is not authorized to access the requested resource.
     * @throws InvalidParameterException
     *         Thrown when a request parameter does not comply with the associated constraints.
     * @throws ResourceNotFoundException
     *         Thrown if the resource doesn't exist.
     * @throws InternalErrorException
     *         Indicates an internal service error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CognitoSyncException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CognitoSyncClient.GetBulkPublishDetails
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-sync-2014-06-30/GetBulkPublishDetails"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetBulkPublishDetailsResponse getBulkPublishDetails(GetBulkPublishDetailsRequest getBulkPublishDetailsRequest)
            throws NotAuthorizedException, InvalidParameterException, ResourceNotFoundException, InternalErrorException,
            AwsServiceException, SdkClientException, CognitoSyncException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetBulkPublishDetailsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetBulkPublishDetailsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetBulkPublishDetailsRequest, GetBulkPublishDetailsResponse>()
                .withOperationName("GetBulkPublishDetails").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getBulkPublishDetailsRequest)
                .withMarshaller(new GetBulkPublishDetailsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Gets the events and the corresponding Lambda functions associated with an identity pool.
     * </p>
     * <p>
     * This API can only be called with developer credentials. You cannot call this API with the temporary user
     * credentials provided by Cognito Identity.
     * </p>
     *
     * @param getCognitoEventsRequest
     *        A request for a list of the configured Cognito Events
     * @return Result of the GetCognitoEvents operation returned by the service.
     * @throws InvalidParameterException
     *         Thrown when a request parameter does not comply with the associated constraints.
     * @throws ResourceNotFoundException
     *         Thrown if the resource doesn't exist.
     * @throws NotAuthorizedException
     *         Thrown when a user is not authorized to access the requested resource.
     * @throws InternalErrorException
     *         Indicates an internal service error.
     * @throws TooManyRequestsException
     *         Thrown if the request is throttled.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CognitoSyncException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CognitoSyncClient.GetCognitoEvents
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-sync-2014-06-30/GetCognitoEvents" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetCognitoEventsResponse getCognitoEvents(GetCognitoEventsRequest getCognitoEventsRequest)
            throws InvalidParameterException, ResourceNotFoundException, NotAuthorizedException, InternalErrorException,
            TooManyRequestsException, AwsServiceException, SdkClientException, CognitoSyncException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetCognitoEventsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetCognitoEventsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetCognitoEventsRequest, GetCognitoEventsResponse>()
                .withOperationName("GetCognitoEvents").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getCognitoEventsRequest)
                .withMarshaller(new GetCognitoEventsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Gets the configuration settings of an identity pool.
     * </p>
     * <p>
     * This API can only be called with developer credentials. You cannot call this API with the temporary user
     * credentials provided by Cognito Identity.
     * </p>
     *
     * @param getIdentityPoolConfigurationRequest
     *        The input for the GetIdentityPoolConfiguration operation.
     * @return Result of the GetIdentityPoolConfiguration operation returned by the service.
     * @throws NotAuthorizedException
     *         Thrown when a user is not authorized to access the requested resource.
     * @throws InvalidParameterException
     *         Thrown when a request parameter does not comply with the associated constraints.
     * @throws ResourceNotFoundException
     *         Thrown if the resource doesn't exist.
     * @throws InternalErrorException
     *         Indicates an internal service error.
     * @throws TooManyRequestsException
     *         Thrown if the request is throttled.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CognitoSyncException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CognitoSyncClient.GetIdentityPoolConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-sync-2014-06-30/GetIdentityPoolConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetIdentityPoolConfigurationResponse getIdentityPoolConfiguration(
            GetIdentityPoolConfigurationRequest getIdentityPoolConfigurationRequest) throws NotAuthorizedException,
            InvalidParameterException, ResourceNotFoundException, InternalErrorException, TooManyRequestsException,
            AwsServiceException, SdkClientException, CognitoSyncException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetIdentityPoolConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetIdentityPoolConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<GetIdentityPoolConfigurationRequest, GetIdentityPoolConfigurationResponse>()
                        .withOperationName("GetIdentityPoolConfiguration").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(getIdentityPoolConfigurationRequest)
                        .withMarshaller(new GetIdentityPoolConfigurationRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists datasets for an identity. With Amazon Cognito Sync, each identity has access only to its own data. Thus,
     * the credentials used to make this API call need to have access to the identity data.
     * </p>
     * <p>
     * ListDatasets can be called with temporary user credentials provided by Cognito Identity or with developer
     * credentials. You should use the Cognito Identity credentials to make this API call.
     * </p>
     *
     * @param listDatasetsRequest
     *        Request for a list of datasets for an identity.
     * @return Result of the ListDatasets operation returned by the service.
     * @throws NotAuthorizedException
     *         Thrown when a user is not authorized to access the requested resource.
     * @throws InvalidParameterException
     *         Thrown when a request parameter does not comply with the associated constraints.
     * @throws InternalErrorException
     *         Indicates an internal service error.
     * @throws TooManyRequestsException
     *         Thrown if the request is throttled.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CognitoSyncException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CognitoSyncClient.ListDatasets
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-sync-2014-06-30/ListDatasets" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListDatasetsResponse listDatasets(ListDatasetsRequest listDatasetsRequest) throws NotAuthorizedException,
            InvalidParameterException, InternalErrorException, TooManyRequestsException, AwsServiceException, SdkClientException,
            CognitoSyncException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListDatasetsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListDatasetsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListDatasetsRequest, ListDatasetsResponse>()
                .withOperationName("ListDatasets").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listDatasetsRequest)
                .withMarshaller(new ListDatasetsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Gets a list of identity pools registered with Cognito.
     * </p>
     * <p>
     * ListIdentityPoolUsage can only be called with developer credentials. You cannot make this API call with the
     * temporary user credentials provided by Cognito Identity.
     * </p>
     *
     * @param listIdentityPoolUsageRequest
     *        A request for usage information on an identity pool.
     * @return Result of the ListIdentityPoolUsage operation returned by the service.
     * @throws NotAuthorizedException
     *         Thrown when a user is not authorized to access the requested resource.
     * @throws InvalidParameterException
     *         Thrown when a request parameter does not comply with the associated constraints.
     * @throws InternalErrorException
     *         Indicates an internal service error.
     * @throws TooManyRequestsException
     *         Thrown if the request is throttled.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CognitoSyncException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CognitoSyncClient.ListIdentityPoolUsage
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-sync-2014-06-30/ListIdentityPoolUsage"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListIdentityPoolUsageResponse listIdentityPoolUsage(ListIdentityPoolUsageRequest listIdentityPoolUsageRequest)
            throws NotAuthorizedException, InvalidParameterException, InternalErrorException, TooManyRequestsException,
            AwsServiceException, SdkClientException, CognitoSyncException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListIdentityPoolUsageResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListIdentityPoolUsageResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListIdentityPoolUsageRequest, ListIdentityPoolUsageResponse>()
                .withOperationName("ListIdentityPoolUsage").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listIdentityPoolUsageRequest)
                .withMarshaller(new ListIdentityPoolUsageRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Gets paginated records, optionally changed after a particular sync count for a dataset and identity. With Amazon
     * Cognito Sync, each identity has access only to its own data. Thus, the credentials used to make this API call
     * need to have access to the identity data.
     * </p>
     * <p>
     * ListRecords can be called with temporary user credentials provided by Cognito Identity or with developer
     * credentials. You should use Cognito Identity credentials to make this API call.
     * </p>
     *
     * @param listRecordsRequest
     *        A request for a list of records.
     * @return Result of the ListRecords operation returned by the service.
     * @throws InvalidParameterException
     *         Thrown when a request parameter does not comply with the associated constraints.
     * @throws NotAuthorizedException
     *         Thrown when a user is not authorized to access the requested resource.
     * @throws TooManyRequestsException
     *         Thrown if the request is throttled.
     * @throws InternalErrorException
     *         Indicates an internal service error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CognitoSyncException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CognitoSyncClient.ListRecords
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-sync-2014-06-30/ListRecords" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListRecordsResponse listRecords(ListRecordsRequest listRecordsRequest) throws InvalidParameterException,
            NotAuthorizedException, TooManyRequestsException, InternalErrorException, AwsServiceException, SdkClientException,
            CognitoSyncException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListRecordsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListRecordsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListRecordsRequest, ListRecordsResponse>()
                .withOperationName("ListRecords").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listRecordsRequest)
                .withMarshaller(new ListRecordsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Registers a device to receive push sync notifications.
     * </p>
     * <p>
     * This API can only be called with temporary credentials provided by Cognito Identity. You cannot call this API
     * with developer credentials.
     * </p>
     *
     * @param registerDeviceRequest
     *        A request to RegisterDevice.
     * @return Result of the RegisterDevice operation returned by the service.
     * @throws NotAuthorizedException
     *         Thrown when a user is not authorized to access the requested resource.
     * @throws InvalidParameterException
     *         Thrown when a request parameter does not comply with the associated constraints.
     * @throws ResourceNotFoundException
     *         Thrown if the resource doesn't exist.
     * @throws InternalErrorException
     *         Indicates an internal service error.
     * @throws InvalidConfigurationException
     * @throws TooManyRequestsException
     *         Thrown if the request is throttled.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CognitoSyncException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CognitoSyncClient.RegisterDevice
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-sync-2014-06-30/RegisterDevice" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public RegisterDeviceResponse registerDevice(RegisterDeviceRequest registerDeviceRequest) throws NotAuthorizedException,
            InvalidParameterException, ResourceNotFoundException, InternalErrorException, InvalidConfigurationException,
            TooManyRequestsException, AwsServiceException, SdkClientException, CognitoSyncException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RegisterDeviceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                RegisterDeviceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<RegisterDeviceRequest, RegisterDeviceResponse>()
                .withOperationName("RegisterDevice").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(registerDeviceRequest)
                .withMarshaller(new RegisterDeviceRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Sets the AWS Lambda function for a given event type for an identity pool. This request only updates the key/value
     * pair specified. Other key/values pairs are not updated. To remove a key value pair, pass a empty value for the
     * particular key.
     * </p>
     * <p>
     * This API can only be called with developer credentials. You cannot call this API with the temporary user
     * credentials provided by Cognito Identity.
     * </p>
     *
     * @param setCognitoEventsRequest
     *        A request to configure Cognito Events"
     * @return Result of the SetCognitoEvents operation returned by the service.
     * @throws InvalidParameterException
     *         Thrown when a request parameter does not comply with the associated constraints.
     * @throws ResourceNotFoundException
     *         Thrown if the resource doesn't exist.
     * @throws NotAuthorizedException
     *         Thrown when a user is not authorized to access the requested resource.
     * @throws InternalErrorException
     *         Indicates an internal service error.
     * @throws TooManyRequestsException
     *         Thrown if the request is throttled.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CognitoSyncException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CognitoSyncClient.SetCognitoEvents
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-sync-2014-06-30/SetCognitoEvents" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public SetCognitoEventsResponse setCognitoEvents(SetCognitoEventsRequest setCognitoEventsRequest)
            throws InvalidParameterException, ResourceNotFoundException, NotAuthorizedException, InternalErrorException,
            TooManyRequestsException, AwsServiceException, SdkClientException, CognitoSyncException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<SetCognitoEventsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                SetCognitoEventsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<SetCognitoEventsRequest, SetCognitoEventsResponse>()
                .withOperationName("SetCognitoEvents").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(setCognitoEventsRequest)
                .withMarshaller(new SetCognitoEventsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Sets the necessary configuration for push sync.
     * </p>
     * <p>
     * This API can only be called with developer credentials. You cannot call this API with the temporary user
     * credentials provided by Cognito Identity.
     * </p>
     *
     * @param setIdentityPoolConfigurationRequest
     *        The input for the SetIdentityPoolConfiguration operation.
     * @return Result of the SetIdentityPoolConfiguration operation returned by the service.
     * @throws NotAuthorizedException
     *         Thrown when a user is not authorized to access the requested resource.
     * @throws InvalidParameterException
     *         Thrown when a request parameter does not comply with the associated constraints.
     * @throws ResourceNotFoundException
     *         Thrown if the resource doesn't exist.
     * @throws InternalErrorException
     *         Indicates an internal service error.
     * @throws TooManyRequestsException
     *         Thrown if the request is throttled.
     * @throws ConcurrentModificationException
     *         Thrown if there are parallel requests to modify a resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CognitoSyncException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CognitoSyncClient.SetIdentityPoolConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-sync-2014-06-30/SetIdentityPoolConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public SetIdentityPoolConfigurationResponse setIdentityPoolConfiguration(
            SetIdentityPoolConfigurationRequest setIdentityPoolConfigurationRequest) throws NotAuthorizedException,
            InvalidParameterException, ResourceNotFoundException, InternalErrorException, TooManyRequestsException,
            ConcurrentModificationException, AwsServiceException, SdkClientException, CognitoSyncException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<SetIdentityPoolConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, SetIdentityPoolConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<SetIdentityPoolConfigurationRequest, SetIdentityPoolConfigurationResponse>()
                        .withOperationName("SetIdentityPoolConfiguration").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(setIdentityPoolConfigurationRequest)
                        .withMarshaller(new SetIdentityPoolConfigurationRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Subscribes to receive notifications when a dataset is modified by another device.
     * </p>
     * <p>
     * This API can only be called with temporary credentials provided by Cognito Identity. You cannot call this API
     * with developer credentials.
     * </p>
     *
     * @param subscribeToDatasetRequest
     *        A request to SubscribeToDatasetRequest.
     * @return Result of the SubscribeToDataset operation returned by the service.
     * @throws NotAuthorizedException
     *         Thrown when a user is not authorized to access the requested resource.
     * @throws InvalidParameterException
     *         Thrown when a request parameter does not comply with the associated constraints.
     * @throws ResourceNotFoundException
     *         Thrown if the resource doesn't exist.
     * @throws InternalErrorException
     *         Indicates an internal service error.
     * @throws InvalidConfigurationException
     * @throws TooManyRequestsException
     *         Thrown if the request is throttled.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CognitoSyncException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CognitoSyncClient.SubscribeToDataset
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-sync-2014-06-30/SubscribeToDataset"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public SubscribeToDatasetResponse subscribeToDataset(SubscribeToDatasetRequest subscribeToDatasetRequest)
            throws NotAuthorizedException, InvalidParameterException, ResourceNotFoundException, InternalErrorException,
            InvalidConfigurationException, TooManyRequestsException, AwsServiceException, SdkClientException,
            CognitoSyncException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<SubscribeToDatasetResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, SubscribeToDatasetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<SubscribeToDatasetRequest, SubscribeToDatasetResponse>()
                .withOperationName("SubscribeToDataset").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(subscribeToDatasetRequest)
                .withMarshaller(new SubscribeToDatasetRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Unsubscribes from receiving notifications when a dataset is modified by another device.
     * </p>
     * <p>
     * This API can only be called with temporary credentials provided by Cognito Identity. You cannot call this API
     * with developer credentials.
     * </p>
     *
     * @param unsubscribeFromDatasetRequest
     *        A request to UnsubscribeFromDataset.
     * @return Result of the UnsubscribeFromDataset operation returned by the service.
     * @throws NotAuthorizedException
     *         Thrown when a user is not authorized to access the requested resource.
     * @throws InvalidParameterException
     *         Thrown when a request parameter does not comply with the associated constraints.
     * @throws ResourceNotFoundException
     *         Thrown if the resource doesn't exist.
     * @throws InternalErrorException
     *         Indicates an internal service error.
     * @throws InvalidConfigurationException
     * @throws TooManyRequestsException
     *         Thrown if the request is throttled.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CognitoSyncException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CognitoSyncClient.UnsubscribeFromDataset
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-sync-2014-06-30/UnsubscribeFromDataset"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UnsubscribeFromDatasetResponse unsubscribeFromDataset(UnsubscribeFromDatasetRequest unsubscribeFromDatasetRequest)
            throws NotAuthorizedException, InvalidParameterException, ResourceNotFoundException, InternalErrorException,
            InvalidConfigurationException, TooManyRequestsException, AwsServiceException, SdkClientException,
            CognitoSyncException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UnsubscribeFromDatasetResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UnsubscribeFromDatasetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UnsubscribeFromDatasetRequest, UnsubscribeFromDatasetResponse>()
                .withOperationName("UnsubscribeFromDataset").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(unsubscribeFromDatasetRequest)
                .withMarshaller(new UnsubscribeFromDatasetRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Posts updates to records and adds and deletes records for a dataset and user.
     * </p>
     * <p>
     * The sync count in the record patch is your last known sync count for that record. The server will reject an
     * UpdateRecords request with a ResourceConflictException if you try to patch a record with a new value but a stale
     * sync count.
     * </p>
     * <p>
     * For example, if the sync count on the server is 5 for a key called highScore and you try and submit a new
     * highScore with sync count of 4, the request will be rejected. To obtain the current sync count for a record, call
     * ListRecords. On a successful update of the record, the response returns the new sync count for that record. You
     * should present that sync count the next time you try to update that same record. When the record does not exist,
     * specify the sync count as 0.
     * </p>
     * <p>
     * This API can be called with temporary user credentials provided by Cognito Identity or with developer
     * credentials.
     * </p>
     *
     * @param updateRecordsRequest
     *        A request to post updates to records or add and delete records for a dataset and user.
     * @return Result of the UpdateRecords operation returned by the service.
     * @throws InvalidParameterException
     *         Thrown when a request parameter does not comply with the associated constraints.
     * @throws LimitExceededException
     *         Thrown when the limit on the number of objects or operations has been exceeded.
     * @throws NotAuthorizedException
     *         Thrown when a user is not authorized to access the requested resource.
     * @throws ResourceNotFoundException
     *         Thrown if the resource doesn't exist.
     * @throws ResourceConflictException
     *         Thrown if an update can't be applied because the resource was changed by another call and this would
     *         result in a conflict.
     * @throws InvalidLambdaFunctionOutputException
     *         The AWS Lambda function returned invalid output or an exception.
     * @throws LambdaThrottledException
     *         AWS Lambda throttled your account, please contact AWS Support
     * @throws TooManyRequestsException
     *         Thrown if the request is throttled.
     * @throws InternalErrorException
     *         Indicates an internal service error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CognitoSyncException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CognitoSyncClient.UpdateRecords
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-sync-2014-06-30/UpdateRecords" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateRecordsResponse updateRecords(UpdateRecordsRequest updateRecordsRequest) throws InvalidParameterException,
            LimitExceededException, NotAuthorizedException, ResourceNotFoundException, ResourceConflictException,
            InvalidLambdaFunctionOutputException, LambdaThrottledException, TooManyRequestsException, InternalErrorException,
            AwsServiceException, SdkClientException, CognitoSyncException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateRecordsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateRecordsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UpdateRecordsRequest, UpdateRecordsResponse>()
                .withOperationName("UpdateRecords").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(updateRecordsRequest)
                .withMarshaller(new UpdateRecordsRequestMarshaller(protocolFactory)));
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(CognitoSyncException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConcurrentModification")
                                .exceptionBuilderSupplier(ConcurrentModificationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidConfiguration")
                                .exceptionBuilderSupplier(InvalidConfigurationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameter")
                                .exceptionBuilderSupplier(InvalidParameterException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFound")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DuplicateRequest")
                                .exceptionBuilderSupplier(DuplicateRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceConflict")
                                .exceptionBuilderSupplier(ResourceConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LambdaThrottled")
                                .exceptionBuilderSupplier(LambdaThrottledException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotAuthorizedError")
                                .exceptionBuilderSupplier(NotAuthorizedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalError")
                                .exceptionBuilderSupplier(InternalErrorException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyRequests")
                                .exceptionBuilderSupplier(TooManyRequestsException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidLambdaFunctionOutput")
                                .exceptionBuilderSupplier(InvalidLambdaFunctionOutputException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceeded")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AlreadyStreamed")
                                .exceptionBuilderSupplier(AlreadyStreamedException::builder).httpStatusCode(400).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
