/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cognitoidentityprovider.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents the request to register a user.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SignUpRequest extends CognitoIdentityProviderRequest implements
        ToCopyableBuilder<SignUpRequest.Builder, SignUpRequest> {
    private final String clientId;

    private final String secretHash;

    private final String username;

    private final String password;

    private final List<AttributeType> userAttributes;

    private final List<AttributeType> validationData;

    private SignUpRequest(BuilderImpl builder) {
        super(builder);
        this.clientId = builder.clientId;
        this.secretHash = builder.secretHash;
        this.username = builder.username;
        this.password = builder.password;
        this.userAttributes = builder.userAttributes;
        this.validationData = builder.validationData;
    }

    /**
     * <p>
     * The ID of the client associated with the user pool.
     * </p>
     * 
     * @return The ID of the client associated with the user pool.
     */
    public String clientId() {
        return clientId;
    }

    /**
     * <p>
     * A keyed-hash message authentication code (HMAC) calculated using the secret key of a user pool client and
     * username plus the client ID in the message.
     * </p>
     * 
     * @return A keyed-hash message authentication code (HMAC) calculated using the secret key of a user pool client and
     *         username plus the client ID in the message.
     */
    public String secretHash() {
        return secretHash;
    }

    /**
     * <p>
     * The user name of the user you wish to register.
     * </p>
     * 
     * @return The user name of the user you wish to register.
     */
    public String username() {
        return username;
    }

    /**
     * <p>
     * The password of the user you wish to register.
     * </p>
     * 
     * @return The password of the user you wish to register.
     */
    public String password() {
        return password;
    }

    /**
     * <p>
     * An array of name-value pairs representing user attributes.
     * </p>
     * <p>
     * For custom attributes, you must prepend the <code>custom:</code> prefix to the attribute name.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return An array of name-value pairs representing user attributes.</p>
     *         <p>
     *         For custom attributes, you must prepend the <code>custom:</code> prefix to the attribute name.
     */
    public List<AttributeType> userAttributes() {
        return userAttributes;
    }

    /**
     * <p>
     * The validation data in the request to register a user.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The validation data in the request to register a user.
     */
    public List<AttributeType> validationData() {
        return validationData;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(clientId());
        hashCode = 31 * hashCode + Objects.hashCode(secretHash());
        hashCode = 31 * hashCode + Objects.hashCode(username());
        hashCode = 31 * hashCode + Objects.hashCode(password());
        hashCode = 31 * hashCode + Objects.hashCode(userAttributes());
        hashCode = 31 * hashCode + Objects.hashCode(validationData());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SignUpRequest)) {
            return false;
        }
        SignUpRequest other = (SignUpRequest) obj;
        return Objects.equals(clientId(), other.clientId()) && Objects.equals(secretHash(), other.secretHash())
                && Objects.equals(username(), other.username()) && Objects.equals(password(), other.password())
                && Objects.equals(userAttributes(), other.userAttributes())
                && Objects.equals(validationData(), other.validationData());
    }

    @Override
    public String toString() {
        return ToString.builder("SignUpRequest").add("ClientId", clientId()).add("SecretHash", secretHash())
                .add("Username", username()).add("Password", password()).add("UserAttributes", userAttributes())
                .add("ValidationData", validationData()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ClientId":
            return Optional.ofNullable(clazz.cast(clientId()));
        case "SecretHash":
            return Optional.ofNullable(clazz.cast(secretHash()));
        case "Username":
            return Optional.ofNullable(clazz.cast(username()));
        case "Password":
            return Optional.ofNullable(clazz.cast(password()));
        case "UserAttributes":
            return Optional.ofNullable(clazz.cast(userAttributes()));
        case "ValidationData":
            return Optional.ofNullable(clazz.cast(validationData()));
        default:
            return Optional.empty();
        }
    }

    public interface Builder extends CognitoIdentityProviderRequest.Builder, CopyableBuilder<Builder, SignUpRequest> {
        /**
         * <p>
         * The ID of the client associated with the user pool.
         * </p>
         * 
         * @param clientId
         *        The ID of the client associated with the user pool.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientId(String clientId);

        /**
         * <p>
         * A keyed-hash message authentication code (HMAC) calculated using the secret key of a user pool client and
         * username plus the client ID in the message.
         * </p>
         * 
         * @param secretHash
         *        A keyed-hash message authentication code (HMAC) calculated using the secret key of a user pool client
         *        and username plus the client ID in the message.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secretHash(String secretHash);

        /**
         * <p>
         * The user name of the user you wish to register.
         * </p>
         * 
         * @param username
         *        The user name of the user you wish to register.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder username(String username);

        /**
         * <p>
         * The password of the user you wish to register.
         * </p>
         * 
         * @param password
         *        The password of the user you wish to register.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder password(String password);

        /**
         * <p>
         * An array of name-value pairs representing user attributes.
         * </p>
         * <p>
         * For custom attributes, you must prepend the <code>custom:</code> prefix to the attribute name.
         * </p>
         * 
         * @param userAttributes
         *        An array of name-value pairs representing user attributes.</p>
         *        <p>
         *        For custom attributes, you must prepend the <code>custom:</code> prefix to the attribute name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userAttributes(Collection<AttributeType> userAttributes);

        /**
         * <p>
         * An array of name-value pairs representing user attributes.
         * </p>
         * <p>
         * For custom attributes, you must prepend the <code>custom:</code> prefix to the attribute name.
         * </p>
         * 
         * @param userAttributes
         *        An array of name-value pairs representing user attributes.</p>
         *        <p>
         *        For custom attributes, you must prepend the <code>custom:</code> prefix to the attribute name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userAttributes(AttributeType... userAttributes);

        /**
         * <p>
         * An array of name-value pairs representing user attributes.
         * </p>
         * <p>
         * For custom attributes, you must prepend the <code>custom:</code> prefix to the attribute name.
         * </p>
         * This is a convenience that creates an instance of the {@link List<AttributeType>.Builder} avoiding the need
         * to create one manually via {@link List<AttributeType>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<AttributeType>.Builder#build()} is called immediately and
         * its result is passed to {@link #userAttributes(List<AttributeType>)}.
         * 
         * @param userAttributes
         *        a consumer that will call methods on {@link List<AttributeType>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #userAttributes(List<AttributeType>)
         */
        Builder userAttributes(Consumer<AttributeType.Builder>... userAttributes);

        /**
         * <p>
         * The validation data in the request to register a user.
         * </p>
         * 
         * @param validationData
         *        The validation data in the request to register a user.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder validationData(Collection<AttributeType> validationData);

        /**
         * <p>
         * The validation data in the request to register a user.
         * </p>
         * 
         * @param validationData
         *        The validation data in the request to register a user.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder validationData(AttributeType... validationData);

        /**
         * <p>
         * The validation data in the request to register a user.
         * </p>
         * This is a convenience that creates an instance of the {@link List<AttributeType>.Builder} avoiding the need
         * to create one manually via {@link List<AttributeType>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<AttributeType>.Builder#build()} is called immediately and
         * its result is passed to {@link #validationData(List<AttributeType>)}.
         * 
         * @param validationData
         *        a consumer that will call methods on {@link List<AttributeType>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #validationData(List<AttributeType>)
         */
        Builder validationData(Consumer<AttributeType.Builder>... validationData);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends CognitoIdentityProviderRequest.BuilderImpl implements Builder {
        private String clientId;

        private String secretHash;

        private String username;

        private String password;

        private List<AttributeType> userAttributes = DefaultSdkAutoConstructList.getInstance();

        private List<AttributeType> validationData = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(SignUpRequest model) {
            super(model);
            clientId(model.clientId);
            secretHash(model.secretHash);
            username(model.username);
            password(model.password);
            userAttributes(model.userAttributes);
            validationData(model.validationData);
        }

        public final String getClientId() {
            return clientId;
        }

        @Override
        public final Builder clientId(String clientId) {
            this.clientId = clientId;
            return this;
        }

        public final void setClientId(String clientId) {
            this.clientId = clientId;
        }

        public final String getSecretHash() {
            return secretHash;
        }

        @Override
        public final Builder secretHash(String secretHash) {
            this.secretHash = secretHash;
            return this;
        }

        public final void setSecretHash(String secretHash) {
            this.secretHash = secretHash;
        }

        public final String getUsername() {
            return username;
        }

        @Override
        public final Builder username(String username) {
            this.username = username;
            return this;
        }

        public final void setUsername(String username) {
            this.username = username;
        }

        public final String getPassword() {
            return password;
        }

        @Override
        public final Builder password(String password) {
            this.password = password;
            return this;
        }

        public final void setPassword(String password) {
            this.password = password;
        }

        public final Collection<AttributeType.Builder> getUserAttributes() {
            return userAttributes != null ? userAttributes.stream().map(AttributeType::toBuilder).collect(Collectors.toList())
                    : null;
        }

        @Override
        public final Builder userAttributes(Collection<AttributeType> userAttributes) {
            this.userAttributes = AttributeListTypeCopier.copy(userAttributes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder userAttributes(AttributeType... userAttributes) {
            userAttributes(Arrays.asList(userAttributes));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder userAttributes(Consumer<AttributeType.Builder>... userAttributes) {
            userAttributes(Stream.of(userAttributes).map(c -> AttributeType.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setUserAttributes(Collection<AttributeType.BuilderImpl> userAttributes) {
            this.userAttributes = AttributeListTypeCopier.copyFromBuilder(userAttributes);
        }

        public final Collection<AttributeType.Builder> getValidationData() {
            return validationData != null ? validationData.stream().map(AttributeType::toBuilder).collect(Collectors.toList())
                    : null;
        }

        @Override
        public final Builder validationData(Collection<AttributeType> validationData) {
            this.validationData = AttributeListTypeCopier.copy(validationData);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder validationData(AttributeType... validationData) {
            validationData(Arrays.asList(validationData));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder validationData(Consumer<AttributeType.Builder>... validationData) {
            validationData(Stream.of(validationData).map(c -> AttributeType.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setValidationData(Collection<AttributeType.BuilderImpl> validationData) {
            this.validationData = AttributeListTypeCopier.copyFromBuilder(validationData);
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public SignUpRequest build() {
            return new SignUpRequest(this);
        }
    }
}
