/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cognitoidentityprovider.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.services.cognitoidentityprovider.transform.ResourceServerTypeMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A container for information about a resource server for a user pool.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ResourceServerType implements StructuredPojo,
        ToCopyableBuilder<ResourceServerType.Builder, ResourceServerType> {
    private final String userPoolId;

    private final String identifier;

    private final String name;

    private final List<ResourceServerScopeType> scopes;

    private ResourceServerType(BuilderImpl builder) {
        this.userPoolId = builder.userPoolId;
        this.identifier = builder.identifier;
        this.name = builder.name;
        this.scopes = builder.scopes;
    }

    /**
     * <p>
     * The user pool ID for the user pool that hosts the resource server.
     * </p>
     * 
     * @return The user pool ID for the user pool that hosts the resource server.
     */
    public String userPoolId() {
        return userPoolId;
    }

    /**
     * <p>
     * The identifier for the resource server.
     * </p>
     * 
     * @return The identifier for the resource server.
     */
    public String identifier() {
        return identifier;
    }

    /**
     * <p>
     * The name of the resource server.
     * </p>
     * 
     * @return The name of the resource server.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * A list of scopes that are defined for the resource server.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A list of scopes that are defined for the resource server.
     */
    public List<ResourceServerScopeType> scopes() {
        return scopes;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(userPoolId());
        hashCode = 31 * hashCode + Objects.hashCode(identifier());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(scopes());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ResourceServerType)) {
            return false;
        }
        ResourceServerType other = (ResourceServerType) obj;
        return Objects.equals(userPoolId(), other.userPoolId()) && Objects.equals(identifier(), other.identifier())
                && Objects.equals(name(), other.name()) && Objects.equals(scopes(), other.scopes());
    }

    @Override
    public String toString() {
        return ToString.builder("ResourceServerType").add("UserPoolId", userPoolId()).add("Identifier", identifier())
                .add("Name", name()).add("Scopes", scopes()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "UserPoolId":
            return Optional.ofNullable(clazz.cast(userPoolId()));
        case "Identifier":
            return Optional.ofNullable(clazz.cast(identifier()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Scopes":
            return Optional.ofNullable(clazz.cast(scopes()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        ResourceServerTypeMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, ResourceServerType> {
        /**
         * <p>
         * The user pool ID for the user pool that hosts the resource server.
         * </p>
         * 
         * @param userPoolId
         *        The user pool ID for the user pool that hosts the resource server.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userPoolId(String userPoolId);

        /**
         * <p>
         * The identifier for the resource server.
         * </p>
         * 
         * @param identifier
         *        The identifier for the resource server.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder identifier(String identifier);

        /**
         * <p>
         * The name of the resource server.
         * </p>
         * 
         * @param name
         *        The name of the resource server.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * A list of scopes that are defined for the resource server.
         * </p>
         * 
         * @param scopes
         *        A list of scopes that are defined for the resource server.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scopes(Collection<ResourceServerScopeType> scopes);

        /**
         * <p>
         * A list of scopes that are defined for the resource server.
         * </p>
         * 
         * @param scopes
         *        A list of scopes that are defined for the resource server.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scopes(ResourceServerScopeType... scopes);

        /**
         * <p>
         * A list of scopes that are defined for the resource server.
         * </p>
         * This is a convenience that creates an instance of the {@link List<ResourceServerScopeType>.Builder} avoiding
         * the need to create one manually via {@link List<ResourceServerScopeType>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<ResourceServerScopeType>.Builder#build()} is called
         * immediately and its result is passed to {@link #scopes(List<ResourceServerScopeType>)}.
         * 
         * @param scopes
         *        a consumer that will call methods on {@link List<ResourceServerScopeType>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #scopes(List<ResourceServerScopeType>)
         */
        Builder scopes(Consumer<ResourceServerScopeType.Builder>... scopes);
    }

    static final class BuilderImpl implements Builder {
        private String userPoolId;

        private String identifier;

        private String name;

        private List<ResourceServerScopeType> scopes = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ResourceServerType model) {
            userPoolId(model.userPoolId);
            identifier(model.identifier);
            name(model.name);
            scopes(model.scopes);
        }

        public final String getUserPoolId() {
            return userPoolId;
        }

        @Override
        public final Builder userPoolId(String userPoolId) {
            this.userPoolId = userPoolId;
            return this;
        }

        public final void setUserPoolId(String userPoolId) {
            this.userPoolId = userPoolId;
        }

        public final String getIdentifier() {
            return identifier;
        }

        @Override
        public final Builder identifier(String identifier) {
            this.identifier = identifier;
            return this;
        }

        public final void setIdentifier(String identifier) {
            this.identifier = identifier;
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final Collection<ResourceServerScopeType.Builder> getScopes() {
            return scopes != null ? scopes.stream().map(ResourceServerScopeType::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder scopes(Collection<ResourceServerScopeType> scopes) {
            this.scopes = ResourceServerScopeListTypeCopier.copy(scopes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder scopes(ResourceServerScopeType... scopes) {
            scopes(Arrays.asList(scopes));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder scopes(Consumer<ResourceServerScopeType.Builder>... scopes) {
            scopes(Stream.of(scopes).map(c -> ResourceServerScopeType.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setScopes(Collection<ResourceServerScopeType.BuilderImpl> scopes) {
            this.scopes = ResourceServerScopeListTypeCopier.copyFromBuilder(scopes);
        }

        @Override
        public ResourceServerType build() {
            return new ResourceServerType(this);
        }
    }
}
