/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cognitoidentityprovider.model;

import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.services.cognitoidentityprovider.transform.IdentityProviderTypeMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A container for information about an identity provider.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class IdentityProviderType implements StructuredPojo,
        ToCopyableBuilder<IdentityProviderType.Builder, IdentityProviderType> {
    private final String userPoolId;

    private final String providerName;

    private final String providerType;

    private final Map<String, String> providerDetails;

    private final Map<String, String> attributeMapping;

    private final List<String> idpIdentifiers;

    private final Instant lastModifiedDate;

    private final Instant creationDate;

    private IdentityProviderType(BuilderImpl builder) {
        this.userPoolId = builder.userPoolId;
        this.providerName = builder.providerName;
        this.providerType = builder.providerType;
        this.providerDetails = builder.providerDetails;
        this.attributeMapping = builder.attributeMapping;
        this.idpIdentifiers = builder.idpIdentifiers;
        this.lastModifiedDate = builder.lastModifiedDate;
        this.creationDate = builder.creationDate;
    }

    /**
     * <p>
     * The user pool ID.
     * </p>
     * 
     * @return The user pool ID.
     */
    public String userPoolId() {
        return userPoolId;
    }

    /**
     * <p>
     * The identity provider name.
     * </p>
     * 
     * @return The identity provider name.
     */
    public String providerName() {
        return providerName;
    }

    /**
     * <p>
     * The identity provider type.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #providerType} will
     * return {@link IdentityProviderTypeType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #providerTypeAsString}.
     * </p>
     * 
     * @return The identity provider type.
     * @see IdentityProviderTypeType
     */
    public IdentityProviderTypeType providerType() {
        return IdentityProviderTypeType.fromValue(providerType);
    }

    /**
     * <p>
     * The identity provider type.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #providerType} will
     * return {@link IdentityProviderTypeType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #providerTypeAsString}.
     * </p>
     * 
     * @return The identity provider type.
     * @see IdentityProviderTypeType
     */
    public String providerTypeAsString() {
        return providerType;
    }

    /**
     * <p>
     * The identity provider details, such as <code>MetadataURL</code> and <code>MetadataFile</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The identity provider details, such as <code>MetadataURL</code> and <code>MetadataFile</code>.
     */
    public Map<String, String> providerDetails() {
        return providerDetails;
    }

    /**
     * <p>
     * A mapping of identity provider attributes to standard and custom user pool attributes.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A mapping of identity provider attributes to standard and custom user pool attributes.
     */
    public Map<String, String> attributeMapping() {
        return attributeMapping;
    }

    /**
     * <p>
     * A list of identity provider identifiers.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A list of identity provider identifiers.
     */
    public List<String> idpIdentifiers() {
        return idpIdentifiers;
    }

    /**
     * <p>
     * The date the identity provider was last modified.
     * </p>
     * 
     * @return The date the identity provider was last modified.
     */
    public Instant lastModifiedDate() {
        return lastModifiedDate;
    }

    /**
     * <p>
     * The date the identity provider was created.
     * </p>
     * 
     * @return The date the identity provider was created.
     */
    public Instant creationDate() {
        return creationDate;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(userPoolId());
        hashCode = 31 * hashCode + Objects.hashCode(providerName());
        hashCode = 31 * hashCode + Objects.hashCode(providerTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(providerDetails());
        hashCode = 31 * hashCode + Objects.hashCode(attributeMapping());
        hashCode = 31 * hashCode + Objects.hashCode(idpIdentifiers());
        hashCode = 31 * hashCode + Objects.hashCode(lastModifiedDate());
        hashCode = 31 * hashCode + Objects.hashCode(creationDate());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof IdentityProviderType)) {
            return false;
        }
        IdentityProviderType other = (IdentityProviderType) obj;
        return Objects.equals(userPoolId(), other.userPoolId()) && Objects.equals(providerName(), other.providerName())
                && Objects.equals(providerTypeAsString(), other.providerTypeAsString())
                && Objects.equals(providerDetails(), other.providerDetails())
                && Objects.equals(attributeMapping(), other.attributeMapping())
                && Objects.equals(idpIdentifiers(), other.idpIdentifiers())
                && Objects.equals(lastModifiedDate(), other.lastModifiedDate())
                && Objects.equals(creationDate(), other.creationDate());
    }

    @Override
    public String toString() {
        return ToString.builder("IdentityProviderType").add("UserPoolId", userPoolId()).add("ProviderName", providerName())
                .add("ProviderType", providerTypeAsString()).add("ProviderDetails", providerDetails())
                .add("AttributeMapping", attributeMapping()).add("IdpIdentifiers", idpIdentifiers())
                .add("LastModifiedDate", lastModifiedDate()).add("CreationDate", creationDate()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "UserPoolId":
            return Optional.ofNullable(clazz.cast(userPoolId()));
        case "ProviderName":
            return Optional.ofNullable(clazz.cast(providerName()));
        case "ProviderType":
            return Optional.ofNullable(clazz.cast(providerTypeAsString()));
        case "ProviderDetails":
            return Optional.ofNullable(clazz.cast(providerDetails()));
        case "AttributeMapping":
            return Optional.ofNullable(clazz.cast(attributeMapping()));
        case "IdpIdentifiers":
            return Optional.ofNullable(clazz.cast(idpIdentifiers()));
        case "LastModifiedDate":
            return Optional.ofNullable(clazz.cast(lastModifiedDate()));
        case "CreationDate":
            return Optional.ofNullable(clazz.cast(creationDate()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        IdentityProviderTypeMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, IdentityProviderType> {
        /**
         * <p>
         * The user pool ID.
         * </p>
         * 
         * @param userPoolId
         *        The user pool ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userPoolId(String userPoolId);

        /**
         * <p>
         * The identity provider name.
         * </p>
         * 
         * @param providerName
         *        The identity provider name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder providerName(String providerName);

        /**
         * <p>
         * The identity provider type.
         * </p>
         * 
         * @param providerType
         *        The identity provider type.
         * @see IdentityProviderTypeType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IdentityProviderTypeType
         */
        Builder providerType(String providerType);

        /**
         * <p>
         * The identity provider type.
         * </p>
         * 
         * @param providerType
         *        The identity provider type.
         * @see IdentityProviderTypeType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IdentityProviderTypeType
         */
        Builder providerType(IdentityProviderTypeType providerType);

        /**
         * <p>
         * The identity provider details, such as <code>MetadataURL</code> and <code>MetadataFile</code>.
         * </p>
         * 
         * @param providerDetails
         *        The identity provider details, such as <code>MetadataURL</code> and <code>MetadataFile</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder providerDetails(Map<String, String> providerDetails);

        /**
         * <p>
         * A mapping of identity provider attributes to standard and custom user pool attributes.
         * </p>
         * 
         * @param attributeMapping
         *        A mapping of identity provider attributes to standard and custom user pool attributes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributeMapping(Map<String, String> attributeMapping);

        /**
         * <p>
         * A list of identity provider identifiers.
         * </p>
         * 
         * @param idpIdentifiers
         *        A list of identity provider identifiers.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder idpIdentifiers(Collection<String> idpIdentifiers);

        /**
         * <p>
         * A list of identity provider identifiers.
         * </p>
         * 
         * @param idpIdentifiers
         *        A list of identity provider identifiers.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder idpIdentifiers(String... idpIdentifiers);

        /**
         * <p>
         * The date the identity provider was last modified.
         * </p>
         * 
         * @param lastModifiedDate
         *        The date the identity provider was last modified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastModifiedDate(Instant lastModifiedDate);

        /**
         * <p>
         * The date the identity provider was created.
         * </p>
         * 
         * @param creationDate
         *        The date the identity provider was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationDate(Instant creationDate);
    }

    static final class BuilderImpl implements Builder {
        private String userPoolId;

        private String providerName;

        private String providerType;

        private Map<String, String> providerDetails = DefaultSdkAutoConstructMap.getInstance();

        private Map<String, String> attributeMapping = DefaultSdkAutoConstructMap.getInstance();

        private List<String> idpIdentifiers = DefaultSdkAutoConstructList.getInstance();

        private Instant lastModifiedDate;

        private Instant creationDate;

        private BuilderImpl() {
        }

        private BuilderImpl(IdentityProviderType model) {
            userPoolId(model.userPoolId);
            providerName(model.providerName);
            providerType(model.providerType);
            providerDetails(model.providerDetails);
            attributeMapping(model.attributeMapping);
            idpIdentifiers(model.idpIdentifiers);
            lastModifiedDate(model.lastModifiedDate);
            creationDate(model.creationDate);
        }

        public final String getUserPoolId() {
            return userPoolId;
        }

        @Override
        public final Builder userPoolId(String userPoolId) {
            this.userPoolId = userPoolId;
            return this;
        }

        public final void setUserPoolId(String userPoolId) {
            this.userPoolId = userPoolId;
        }

        public final String getProviderName() {
            return providerName;
        }

        @Override
        public final Builder providerName(String providerName) {
            this.providerName = providerName;
            return this;
        }

        public final void setProviderName(String providerName) {
            this.providerName = providerName;
        }

        public final String getProviderType() {
            return providerType;
        }

        @Override
        public final Builder providerType(String providerType) {
            this.providerType = providerType;
            return this;
        }

        @Override
        public final Builder providerType(IdentityProviderTypeType providerType) {
            this.providerType(providerType.toString());
            return this;
        }

        public final void setProviderType(String providerType) {
            this.providerType = providerType;
        }

        public final Map<String, String> getProviderDetails() {
            return providerDetails;
        }

        @Override
        public final Builder providerDetails(Map<String, String> providerDetails) {
            this.providerDetails = ProviderDetailsTypeCopier.copy(providerDetails);
            return this;
        }

        public final void setProviderDetails(Map<String, String> providerDetails) {
            this.providerDetails = ProviderDetailsTypeCopier.copy(providerDetails);
        }

        public final Map<String, String> getAttributeMapping() {
            return attributeMapping;
        }

        @Override
        public final Builder attributeMapping(Map<String, String> attributeMapping) {
            this.attributeMapping = AttributeMappingTypeCopier.copy(attributeMapping);
            return this;
        }

        public final void setAttributeMapping(Map<String, String> attributeMapping) {
            this.attributeMapping = AttributeMappingTypeCopier.copy(attributeMapping);
        }

        public final Collection<String> getIdpIdentifiers() {
            return idpIdentifiers;
        }

        @Override
        public final Builder idpIdentifiers(Collection<String> idpIdentifiers) {
            this.idpIdentifiers = IdpIdentifiersListTypeCopier.copy(idpIdentifiers);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder idpIdentifiers(String... idpIdentifiers) {
            idpIdentifiers(Arrays.asList(idpIdentifiers));
            return this;
        }

        public final void setIdpIdentifiers(Collection<String> idpIdentifiers) {
            this.idpIdentifiers = IdpIdentifiersListTypeCopier.copy(idpIdentifiers);
        }

        public final Instant getLastModifiedDate() {
            return lastModifiedDate;
        }

        @Override
        public final Builder lastModifiedDate(Instant lastModifiedDate) {
            this.lastModifiedDate = lastModifiedDate;
            return this;
        }

        public final void setLastModifiedDate(Instant lastModifiedDate) {
            this.lastModifiedDate = lastModifiedDate;
        }

        public final Instant getCreationDate() {
            return creationDate;
        }

        @Override
        public final Builder creationDate(Instant creationDate) {
            this.creationDate = creationDate;
            return this;
        }

        public final void setCreationDate(Instant creationDate) {
            this.creationDate = creationDate;
        }

        @Override
        public IdentityProviderType build() {
            return new IdentityProviderType(this);
        }
    }
}
