/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cognitoidentityprovider.model;

import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The request to respond to the authentication challenge, as an administrator.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AdminRespondToAuthChallengeRequest extends CognitoIdentityProviderRequest implements
        ToCopyableBuilder<AdminRespondToAuthChallengeRequest.Builder, AdminRespondToAuthChallengeRequest> {
    private final String userPoolId;

    private final String clientId;

    private final String challengeName;

    private final Map<String, String> challengeResponses;

    private final String session;

    private AdminRespondToAuthChallengeRequest(BuilderImpl builder) {
        super(builder);
        this.userPoolId = builder.userPoolId;
        this.clientId = builder.clientId;
        this.challengeName = builder.challengeName;
        this.challengeResponses = builder.challengeResponses;
        this.session = builder.session;
    }

    /**
     * <p>
     * The ID of the Amazon Cognito user pool.
     * </p>
     * 
     * @return The ID of the Amazon Cognito user pool.
     */
    public String userPoolId() {
        return userPoolId;
    }

    /**
     * <p>
     * The app client ID.
     * </p>
     * 
     * @return The app client ID.
     */
    public String clientId() {
        return clientId;
    }

    /**
     * <p>
     * The challenge name. For more information, see <a href="API_AdminInitiateAuth.html">AdminInitiateAuth</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #challengeName}
     * will return {@link ChallengeNameType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #challengeNameAsString}.
     * </p>
     * 
     * @return The challenge name. For more information, see <a href="API_AdminInitiateAuth.html">AdminInitiateAuth</a>.
     * @see ChallengeNameType
     */
    public ChallengeNameType challengeName() {
        return ChallengeNameType.fromValue(challengeName);
    }

    /**
     * <p>
     * The challenge name. For more information, see <a href="API_AdminInitiateAuth.html">AdminInitiateAuth</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #challengeName}
     * will return {@link ChallengeNameType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #challengeNameAsString}.
     * </p>
     * 
     * @return The challenge name. For more information, see <a href="API_AdminInitiateAuth.html">AdminInitiateAuth</a>.
     * @see ChallengeNameType
     */
    public String challengeNameAsString() {
        return challengeName;
    }

    /**
     * <p>
     * The challenge responses. These are inputs corresponding to the value of <code>ChallengeName</code>, for example:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>SMS_MFA</code>: <code>SMS_MFA_CODE</code>, <code>USERNAME</code>, <code>SECRET_HASH</code> (if app client
     * is configured with client secret).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PASSWORD_VERIFIER</code>: <code>PASSWORD_CLAIM_SIGNATURE</code>, <code>PASSWORD_CLAIM_SECRET_BLOCK</code>,
     * <code>TIMESTAMP</code>, <code>USERNAME</code>, <code>SECRET_HASH</code> (if app client is configured with client
     * secret).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ADMIN_NO_SRP_AUTH</code>: <code>PASSWORD</code>, <code>USERNAME</code>, <code>SECRET_HASH</code> (if app
     * client is configured with client secret).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NEW_PASSWORD_REQUIRED</code>: <code>NEW_PASSWORD</code>, any other required attributes,
     * <code>USERNAME</code>, <code>SECRET_HASH</code> (if app client is configured with client secret).
     * </p>
     * </li>
     * </ul>
     * <p>
     * The value of the <code>USERNAME</code> attribute must be the user's actual username, not an alias (such as email
     * address or phone number). To make this easier, the <code>AdminInitiateAuth</code> response includes the actual
     * username value in the <code>USERNAMEUSER_ID_FOR_SRP</code> attribute, even if you specified an alias in your call
     * to <code>AdminInitiateAuth</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The challenge responses. These are inputs corresponding to the value of <code>ChallengeName</code>, for
     *         example:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>SMS_MFA</code>: <code>SMS_MFA_CODE</code>, <code>USERNAME</code>, <code>SECRET_HASH</code> (if app
     *         client is configured with client secret).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>PASSWORD_VERIFIER</code>: <code>PASSWORD_CLAIM_SIGNATURE</code>,
     *         <code>PASSWORD_CLAIM_SECRET_BLOCK</code>, <code>TIMESTAMP</code>, <code>USERNAME</code>,
     *         <code>SECRET_HASH</code> (if app client is configured with client secret).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ADMIN_NO_SRP_AUTH</code>: <code>PASSWORD</code>, <code>USERNAME</code>, <code>SECRET_HASH</code>
     *         (if app client is configured with client secret).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NEW_PASSWORD_REQUIRED</code>: <code>NEW_PASSWORD</code>, any other required attributes,
     *         <code>USERNAME</code>, <code>SECRET_HASH</code> (if app client is configured with client secret).
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         The value of the <code>USERNAME</code> attribute must be the user's actual username, not an alias (such
     *         as email address or phone number). To make this easier, the <code>AdminInitiateAuth</code> response
     *         includes the actual username value in the <code>USERNAMEUSER_ID_FOR_SRP</code> attribute, even if you
     *         specified an alias in your call to <code>AdminInitiateAuth</code>.
     */
    public Map<String, String> challengeResponses() {
        return challengeResponses;
    }

    /**
     * <p>
     * The session which should be passed both ways in challenge-response calls to the service. If
     * <code>InitiateAuth</code> or <code>RespondToAuthChallenge</code> API call determines that the caller needs to go
     * through another challenge, they return a session with other challenge parameters. This session should be passed
     * as it is to the next <code>RespondToAuthChallenge</code> API call.
     * </p>
     * 
     * @return The session which should be passed both ways in challenge-response calls to the service. If
     *         <code>InitiateAuth</code> or <code>RespondToAuthChallenge</code> API call determines that the caller
     *         needs to go through another challenge, they return a session with other challenge parameters. This
     *         session should be passed as it is to the next <code>RespondToAuthChallenge</code> API call.
     */
    public String session() {
        return session;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(userPoolId());
        hashCode = 31 * hashCode + Objects.hashCode(clientId());
        hashCode = 31 * hashCode + Objects.hashCode(challengeNameAsString());
        hashCode = 31 * hashCode + Objects.hashCode(challengeResponses());
        hashCode = 31 * hashCode + Objects.hashCode(session());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AdminRespondToAuthChallengeRequest)) {
            return false;
        }
        AdminRespondToAuthChallengeRequest other = (AdminRespondToAuthChallengeRequest) obj;
        return Objects.equals(userPoolId(), other.userPoolId()) && Objects.equals(clientId(), other.clientId())
                && Objects.equals(challengeNameAsString(), other.challengeNameAsString())
                && Objects.equals(challengeResponses(), other.challengeResponses()) && Objects.equals(session(), other.session());
    }

    @Override
    public String toString() {
        return ToString.builder("AdminRespondToAuthChallengeRequest").add("UserPoolId", userPoolId()).add("ClientId", clientId())
                .add("ChallengeName", challengeNameAsString()).add("ChallengeResponses", challengeResponses())
                .add("Session", session()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "UserPoolId":
            return Optional.ofNullable(clazz.cast(userPoolId()));
        case "ClientId":
            return Optional.ofNullable(clazz.cast(clientId()));
        case "ChallengeName":
            return Optional.ofNullable(clazz.cast(challengeNameAsString()));
        case "ChallengeResponses":
            return Optional.ofNullable(clazz.cast(challengeResponses()));
        case "Session":
            return Optional.ofNullable(clazz.cast(session()));
        default:
            return Optional.empty();
        }
    }

    public interface Builder extends CognitoIdentityProviderRequest.Builder,
            CopyableBuilder<Builder, AdminRespondToAuthChallengeRequest> {
        /**
         * <p>
         * The ID of the Amazon Cognito user pool.
         * </p>
         * 
         * @param userPoolId
         *        The ID of the Amazon Cognito user pool.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userPoolId(String userPoolId);

        /**
         * <p>
         * The app client ID.
         * </p>
         * 
         * @param clientId
         *        The app client ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientId(String clientId);

        /**
         * <p>
         * The challenge name. For more information, see <a href="API_AdminInitiateAuth.html">AdminInitiateAuth</a>.
         * </p>
         * 
         * @param challengeName
         *        The challenge name. For more information, see <a
         *        href="API_AdminInitiateAuth.html">AdminInitiateAuth</a>.
         * @see ChallengeNameType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ChallengeNameType
         */
        Builder challengeName(String challengeName);

        /**
         * <p>
         * The challenge name. For more information, see <a href="API_AdminInitiateAuth.html">AdminInitiateAuth</a>.
         * </p>
         * 
         * @param challengeName
         *        The challenge name. For more information, see <a
         *        href="API_AdminInitiateAuth.html">AdminInitiateAuth</a>.
         * @see ChallengeNameType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ChallengeNameType
         */
        Builder challengeName(ChallengeNameType challengeName);

        /**
         * <p>
         * The challenge responses. These are inputs corresponding to the value of <code>ChallengeName</code>, for
         * example:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>SMS_MFA</code>: <code>SMS_MFA_CODE</code>, <code>USERNAME</code>, <code>SECRET_HASH</code> (if app
         * client is configured with client secret).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>PASSWORD_VERIFIER</code>: <code>PASSWORD_CLAIM_SIGNATURE</code>,
         * <code>PASSWORD_CLAIM_SECRET_BLOCK</code>, <code>TIMESTAMP</code>, <code>USERNAME</code>,
         * <code>SECRET_HASH</code> (if app client is configured with client secret).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ADMIN_NO_SRP_AUTH</code>: <code>PASSWORD</code>, <code>USERNAME</code>, <code>SECRET_HASH</code> (if
         * app client is configured with client secret).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NEW_PASSWORD_REQUIRED</code>: <code>NEW_PASSWORD</code>, any other required attributes,
         * <code>USERNAME</code>, <code>SECRET_HASH</code> (if app client is configured with client secret).
         * </p>
         * </li>
         * </ul>
         * <p>
         * The value of the <code>USERNAME</code> attribute must be the user's actual username, not an alias (such as
         * email address or phone number). To make this easier, the <code>AdminInitiateAuth</code> response includes the
         * actual username value in the <code>USERNAMEUSER_ID_FOR_SRP</code> attribute, even if you specified an alias
         * in your call to <code>AdminInitiateAuth</code>.
         * </p>
         * 
         * @param challengeResponses
         *        The challenge responses. These are inputs corresponding to the value of <code>ChallengeName</code>,
         *        for example:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>SMS_MFA</code>: <code>SMS_MFA_CODE</code>, <code>USERNAME</code>, <code>SECRET_HASH</code> (if
         *        app client is configured with client secret).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>PASSWORD_VERIFIER</code>: <code>PASSWORD_CLAIM_SIGNATURE</code>,
         *        <code>PASSWORD_CLAIM_SECRET_BLOCK</code>, <code>TIMESTAMP</code>, <code>USERNAME</code>,
         *        <code>SECRET_HASH</code> (if app client is configured with client secret).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ADMIN_NO_SRP_AUTH</code>: <code>PASSWORD</code>, <code>USERNAME</code>, <code>SECRET_HASH</code>
         *        (if app client is configured with client secret).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NEW_PASSWORD_REQUIRED</code>: <code>NEW_PASSWORD</code>, any other required attributes,
         *        <code>USERNAME</code>, <code>SECRET_HASH</code> (if app client is configured with client secret).
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        The value of the <code>USERNAME</code> attribute must be the user's actual username, not an alias
         *        (such as email address or phone number). To make this easier, the <code>AdminInitiateAuth</code>
         *        response includes the actual username value in the <code>USERNAMEUSER_ID_FOR_SRP</code> attribute,
         *        even if you specified an alias in your call to <code>AdminInitiateAuth</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder challengeResponses(Map<String, String> challengeResponses);

        /**
         * <p>
         * The session which should be passed both ways in challenge-response calls to the service. If
         * <code>InitiateAuth</code> or <code>RespondToAuthChallenge</code> API call determines that the caller needs to
         * go through another challenge, they return a session with other challenge parameters. This session should be
         * passed as it is to the next <code>RespondToAuthChallenge</code> API call.
         * </p>
         * 
         * @param session
         *        The session which should be passed both ways in challenge-response calls to the service. If
         *        <code>InitiateAuth</code> or <code>RespondToAuthChallenge</code> API call determines that the caller
         *        needs to go through another challenge, they return a session with other challenge parameters. This
         *        session should be passed as it is to the next <code>RespondToAuthChallenge</code> API call.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder session(String session);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends CognitoIdentityProviderRequest.BuilderImpl implements Builder {
        private String userPoolId;

        private String clientId;

        private String challengeName;

        private Map<String, String> challengeResponses = DefaultSdkAutoConstructMap.getInstance();

        private String session;

        private BuilderImpl() {
        }

        private BuilderImpl(AdminRespondToAuthChallengeRequest model) {
            super(model);
            userPoolId(model.userPoolId);
            clientId(model.clientId);
            challengeName(model.challengeName);
            challengeResponses(model.challengeResponses);
            session(model.session);
        }

        public final String getUserPoolId() {
            return userPoolId;
        }

        @Override
        public final Builder userPoolId(String userPoolId) {
            this.userPoolId = userPoolId;
            return this;
        }

        public final void setUserPoolId(String userPoolId) {
            this.userPoolId = userPoolId;
        }

        public final String getClientId() {
            return clientId;
        }

        @Override
        public final Builder clientId(String clientId) {
            this.clientId = clientId;
            return this;
        }

        public final void setClientId(String clientId) {
            this.clientId = clientId;
        }

        public final String getChallengeName() {
            return challengeName;
        }

        @Override
        public final Builder challengeName(String challengeName) {
            this.challengeName = challengeName;
            return this;
        }

        @Override
        public final Builder challengeName(ChallengeNameType challengeName) {
            this.challengeName(challengeName.toString());
            return this;
        }

        public final void setChallengeName(String challengeName) {
            this.challengeName = challengeName;
        }

        public final Map<String, String> getChallengeResponses() {
            return challengeResponses;
        }

        @Override
        public final Builder challengeResponses(Map<String, String> challengeResponses) {
            this.challengeResponses = ChallengeResponsesTypeCopier.copy(challengeResponses);
            return this;
        }

        public final void setChallengeResponses(Map<String, String> challengeResponses) {
            this.challengeResponses = ChallengeResponsesTypeCopier.copy(challengeResponses);
        }

        public final String getSession() {
            return session;
        }

        @Override
        public final Builder session(String session) {
            this.session = session;
            return this;
        }

        public final void setSession(String session) {
            this.session = session;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public AdminRespondToAuthChallengeRequest build() {
            return new AdminRespondToAuthChallengeRequest(this);
        }
    }
}
