/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cognitoidentityprovider.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class UpdateIdentityProviderRequest extends CognitoIdentityProviderRequest implements
        ToCopyableBuilder<UpdateIdentityProviderRequest.Builder, UpdateIdentityProviderRequest> {
    private static final SdkField<String> USER_POOL_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("UserPoolId").getter(getter(UpdateIdentityProviderRequest::userPoolId))
            .setter(setter(Builder::userPoolId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UserPoolId").build()).build();

    private static final SdkField<String> PROVIDER_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ProviderName").getter(getter(UpdateIdentityProviderRequest::providerName))
            .setter(setter(Builder::providerName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ProviderName").build()).build();

    private static final SdkField<Map<String, String>> PROVIDER_DETAILS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("ProviderDetails")
            .getter(getter(UpdateIdentityProviderRequest::providerDetails))
            .setter(setter(Builder::providerDetails))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ProviderDetails").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Map<String, String>> ATTRIBUTE_MAPPING_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("AttributeMapping")
            .getter(getter(UpdateIdentityProviderRequest::attributeMapping))
            .setter(setter(Builder::attributeMapping))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AttributeMapping").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<List<String>> IDP_IDENTIFIERS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("IdpIdentifiers")
            .getter(getter(UpdateIdentityProviderRequest::idpIdentifiers))
            .setter(setter(Builder::idpIdentifiers))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IdpIdentifiers").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(USER_POOL_ID_FIELD,
            PROVIDER_NAME_FIELD, PROVIDER_DETAILS_FIELD, ATTRIBUTE_MAPPING_FIELD, IDP_IDENTIFIERS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private final String userPoolId;

    private final String providerName;

    private final Map<String, String> providerDetails;

    private final Map<String, String> attributeMapping;

    private final List<String> idpIdentifiers;

    private UpdateIdentityProviderRequest(BuilderImpl builder) {
        super(builder);
        this.userPoolId = builder.userPoolId;
        this.providerName = builder.providerName;
        this.providerDetails = builder.providerDetails;
        this.attributeMapping = builder.attributeMapping;
        this.idpIdentifiers = builder.idpIdentifiers;
    }

    /**
     * <p>
     * The user pool ID.
     * </p>
     * 
     * @return The user pool ID.
     */
    public final String userPoolId() {
        return userPoolId;
    }

    /**
     * <p>
     * The IdP name.
     * </p>
     * 
     * @return The IdP name.
     */
    public final String providerName() {
        return providerName;
    }

    /**
     * For responses, this returns true if the service returned a value for the ProviderDetails property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasProviderDetails() {
        return providerDetails != null && !(providerDetails instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The scopes, URLs, and identifiers for your external identity provider. The following examples describe the
     * provider detail keys for each IdP type. These values and their schema are subject to change. Social IdP
     * <code>authorize_scopes</code> values must match the values listed here.
     * </p>
     * <dl>
     * <dt>OpenID Connect (OIDC)</dt>
     * <dd>
     * <p>
     * Amazon Cognito accepts the following elements when it can't discover endpoint URLs from <code>oidc_issuer</code>:
     * <code>attributes_url</code>, <code>authorize_url</code>, <code>jwks_uri</code>, <code>token_url</code>.
     * </p>
     * <p>
     * Create or update request:
     * <code>"ProviderDetails": { "attributes_request_method": "GET", "attributes_url": "https://auth.example.com/userInfo", "authorize_scopes": "openid profile email", "authorize_url": "https://auth.example.com/authorize", "client_id": "1example23456789", "client_secret": "provider-app-client-secret", "jwks_uri": "https://auth.example.com/.well-known/jwks.json", "oidc_issuer": "https://auth.example.com", "token_url": "https://example.com/token" }</code>
     * </p>
     * <p>
     * Describe response:
     * <code>"ProviderDetails": { "attributes_request_method": "GET", "attributes_url": "https://auth.example.com/userInfo", "attributes_url_add_attributes": "false", "authorize_scopes": "openid profile email", "authorize_url": "https://auth.example.com/authorize", "client_id": "1example23456789", "client_secret": "provider-app-client-secret", "jwks_uri": "https://auth.example.com/.well-known/jwks.json", "oidc_issuer": "https://auth.example.com", "token_url": "https://example.com/token" }</code>
     * </p>
     * </dd>
     * <dt>SAML</dt>
     * <dd>
     * <p>
     * Create or update request with Metadata URL:
     * <code>"ProviderDetails": { "IDPInit": "true", "IDPSignout": "true", "EncryptedResponses" : "true", "MetadataURL": "https://auth.example.com/sso/saml/metadata", "RequestSigningAlgorithm": "rsa-sha256" }</code>
     * </p>
     * <p>
     * Create or update request with Metadata file:
     * <code>"ProviderDetails": { "IDPInit": "true", "IDPSignout": "true", "EncryptedResponses" : "true", "MetadataFile": "[metadata XML]", "RequestSigningAlgorithm": "rsa-sha256" }</code>
     * </p>
     * <p>
     * The value of <code>MetadataFile</code> must be the plaintext metadata document with all quote (") characters
     * escaped by backslashes.
     * </p>
     * <p>
     * Describe response:
     * <code>"ProviderDetails": { "IDPInit": "true", "IDPSignout": "true", "EncryptedResponses" : "true", "ActiveEncryptionCertificate": "[certificate]", "MetadataURL": "https://auth.example.com/sso/saml/metadata", "RequestSigningAlgorithm": "rsa-sha256", "SLORedirectBindingURI": "https://auth.example.com/slo/saml", "SSORedirectBindingURI": "https://auth.example.com/sso/saml" }</code>
     * </p>
     * </dd>
     * <dt>LoginWithAmazon</dt>
     * <dd>
     * <p>
     * Create or update request:
     * <code>"ProviderDetails": { "authorize_scopes": "profile postal_code", "client_id": "amzn1.application-oa2-client.1example23456789", "client_secret": "provider-app-client-secret"</code>
     * </p>
     * <p>
     * Describe response:
     * <code>"ProviderDetails": { "attributes_url": "https://api.amazon.com/user/profile", "attributes_url_add_attributes": "false", "authorize_scopes": "profile postal_code", "authorize_url": "https://www.amazon.com/ap/oa", "client_id": "amzn1.application-oa2-client.1example23456789", "client_secret": "provider-app-client-secret", "token_request_method": "POST", "token_url": "https://api.amazon.com/auth/o2/token" }</code>
     * </p>
     * </dd>
     * <dt>Google</dt>
     * <dd>
     * <p>
     * Create or update request:
     * <code>"ProviderDetails": { "authorize_scopes": "email profile openid", "client_id": "1example23456789.apps.googleusercontent.com", "client_secret": "provider-app-client-secret" }</code>
     * </p>
     * <p>
     * Describe response:
     * <code>"ProviderDetails": { "attributes_url": "https://people.googleapis.com/v1/people/me?personFields=", "attributes_url_add_attributes": "true", "authorize_scopes": "email profile openid", "authorize_url": "https://accounts.google.com/o/oauth2/v2/auth", "client_id": "1example23456789.apps.googleusercontent.com", "client_secret": "provider-app-client-secret", "oidc_issuer": "https://accounts.google.com", "token_request_method": "POST", "token_url": "https://www.googleapis.com/oauth2/v4/token" }</code>
     * </p>
     * </dd>
     * <dt>SignInWithApple</dt>
     * <dd>
     * <p>
     * Create or update request:
     * <code>"ProviderDetails": { "authorize_scopes": "email name", "client_id": "com.example.cognito", "private_key": "1EXAMPLE", "key_id": "2EXAMPLE", "team_id": "3EXAMPLE" }</code>
     * </p>
     * <p>
     * Describe response:
     * <code>"ProviderDetails": { "attributes_url_add_attributes": "false", "authorize_scopes": "email name", "authorize_url": "https://appleid.apple.com/auth/authorize", "client_id": "com.example.cognito", "key_id": "1EXAMPLE", "oidc_issuer": "https://appleid.apple.com", "team_id": "2EXAMPLE", "token_request_method": "POST", "token_url": "https://appleid.apple.com/auth/token" }</code>
     * </p>
     * </dd>
     * <dt>Facebook</dt>
     * <dd>
     * <p>
     * Create or update request:
     * <code>"ProviderDetails": { "api_version": "v17.0", "authorize_scopes": "public_profile, email", "client_id": "1example23456789", "client_secret": "provider-app-client-secret" }</code>
     * </p>
     * <p>
     * Describe response:
     * <code>"ProviderDetails": { "api_version": "v17.0", "attributes_url": "https://graph.facebook.com/v17.0/me?fields=", "attributes_url_add_attributes": "true", "authorize_scopes": "public_profile, email", "authorize_url": "https://www.facebook.com/v17.0/dialog/oauth", "client_id": "1example23456789", "client_secret": "provider-app-client-secret", "token_request_method": "GET", "token_url": "https://graph.facebook.com/v17.0/oauth/access_token" }</code>
     * </p>
     * </dd>
     * </dl>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasProviderDetails} method.
     * </p>
     * 
     * @return The scopes, URLs, and identifiers for your external identity provider. The following examples describe
     *         the provider detail keys for each IdP type. These values and their schema are subject to change. Social
     *         IdP <code>authorize_scopes</code> values must match the values listed here.</p>
     *         <dl>
     *         <dt>OpenID Connect (OIDC)</dt>
     *         <dd>
     *         <p>
     *         Amazon Cognito accepts the following elements when it can't discover endpoint URLs from
     *         <code>oidc_issuer</code>: <code>attributes_url</code>, <code>authorize_url</code>, <code>jwks_uri</code>,
     *         <code>token_url</code>.
     *         </p>
     *         <p>
     *         Create or update request:
     *         <code>"ProviderDetails": { "attributes_request_method": "GET", "attributes_url": "https://auth.example.com/userInfo", "authorize_scopes": "openid profile email", "authorize_url": "https://auth.example.com/authorize", "client_id": "1example23456789", "client_secret": "provider-app-client-secret", "jwks_uri": "https://auth.example.com/.well-known/jwks.json", "oidc_issuer": "https://auth.example.com", "token_url": "https://example.com/token" }</code>
     *         </p>
     *         <p>
     *         Describe response:
     *         <code>"ProviderDetails": { "attributes_request_method": "GET", "attributes_url": "https://auth.example.com/userInfo", "attributes_url_add_attributes": "false", "authorize_scopes": "openid profile email", "authorize_url": "https://auth.example.com/authorize", "client_id": "1example23456789", "client_secret": "provider-app-client-secret", "jwks_uri": "https://auth.example.com/.well-known/jwks.json", "oidc_issuer": "https://auth.example.com", "token_url": "https://example.com/token" }</code>
     *         </p>
     *         </dd>
     *         <dt>SAML</dt>
     *         <dd>
     *         <p>
     *         Create or update request with Metadata URL:
     *         <code>"ProviderDetails": { "IDPInit": "true", "IDPSignout": "true", "EncryptedResponses" : "true", "MetadataURL": "https://auth.example.com/sso/saml/metadata", "RequestSigningAlgorithm": "rsa-sha256" }</code>
     *         </p>
     *         <p>
     *         Create or update request with Metadata file:
     *         <code>"ProviderDetails": { "IDPInit": "true", "IDPSignout": "true", "EncryptedResponses" : "true", "MetadataFile": "[metadata XML]", "RequestSigningAlgorithm": "rsa-sha256" }</code>
     *         </p>
     *         <p>
     *         The value of <code>MetadataFile</code> must be the plaintext metadata document with all quote (")
     *         characters escaped by backslashes.
     *         </p>
     *         <p>
     *         Describe response:
     *         <code>"ProviderDetails": { "IDPInit": "true", "IDPSignout": "true", "EncryptedResponses" : "true", "ActiveEncryptionCertificate": "[certificate]", "MetadataURL": "https://auth.example.com/sso/saml/metadata", "RequestSigningAlgorithm": "rsa-sha256", "SLORedirectBindingURI": "https://auth.example.com/slo/saml", "SSORedirectBindingURI": "https://auth.example.com/sso/saml" }</code>
     *         </p>
     *         </dd>
     *         <dt>LoginWithAmazon</dt>
     *         <dd>
     *         <p>
     *         Create or update request:
     *         <code>"ProviderDetails": { "authorize_scopes": "profile postal_code", "client_id": "amzn1.application-oa2-client.1example23456789", "client_secret": "provider-app-client-secret"</code>
     *         </p>
     *         <p>
     *         Describe response:
     *         <code>"ProviderDetails": { "attributes_url": "https://api.amazon.com/user/profile", "attributes_url_add_attributes": "false", "authorize_scopes": "profile postal_code", "authorize_url": "https://www.amazon.com/ap/oa", "client_id": "amzn1.application-oa2-client.1example23456789", "client_secret": "provider-app-client-secret", "token_request_method": "POST", "token_url": "https://api.amazon.com/auth/o2/token" }</code>
     *         </p>
     *         </dd>
     *         <dt>Google</dt>
     *         <dd>
     *         <p>
     *         Create or update request:
     *         <code>"ProviderDetails": { "authorize_scopes": "email profile openid", "client_id": "1example23456789.apps.googleusercontent.com", "client_secret": "provider-app-client-secret" }</code>
     *         </p>
     *         <p>
     *         Describe response:
     *         <code>"ProviderDetails": { "attributes_url": "https://people.googleapis.com/v1/people/me?personFields=", "attributes_url_add_attributes": "true", "authorize_scopes": "email profile openid", "authorize_url": "https://accounts.google.com/o/oauth2/v2/auth", "client_id": "1example23456789.apps.googleusercontent.com", "client_secret": "provider-app-client-secret", "oidc_issuer": "https://accounts.google.com", "token_request_method": "POST", "token_url": "https://www.googleapis.com/oauth2/v4/token" }</code>
     *         </p>
     *         </dd>
     *         <dt>SignInWithApple</dt>
     *         <dd>
     *         <p>
     *         Create or update request:
     *         <code>"ProviderDetails": { "authorize_scopes": "email name", "client_id": "com.example.cognito", "private_key": "1EXAMPLE", "key_id": "2EXAMPLE", "team_id": "3EXAMPLE" }</code>
     *         </p>
     *         <p>
     *         Describe response:
     *         <code>"ProviderDetails": { "attributes_url_add_attributes": "false", "authorize_scopes": "email name", "authorize_url": "https://appleid.apple.com/auth/authorize", "client_id": "com.example.cognito", "key_id": "1EXAMPLE", "oidc_issuer": "https://appleid.apple.com", "team_id": "2EXAMPLE", "token_request_method": "POST", "token_url": "https://appleid.apple.com/auth/token" }</code>
     *         </p>
     *         </dd>
     *         <dt>Facebook</dt>
     *         <dd>
     *         <p>
     *         Create or update request:
     *         <code>"ProviderDetails": { "api_version": "v17.0", "authorize_scopes": "public_profile, email", "client_id": "1example23456789", "client_secret": "provider-app-client-secret" }</code>
     *         </p>
     *         <p>
     *         Describe response:
     *         <code>"ProviderDetails": { "api_version": "v17.0", "attributes_url": "https://graph.facebook.com/v17.0/me?fields=", "attributes_url_add_attributes": "true", "authorize_scopes": "public_profile, email", "authorize_url": "https://www.facebook.com/v17.0/dialog/oauth", "client_id": "1example23456789", "client_secret": "provider-app-client-secret", "token_request_method": "GET", "token_url": "https://graph.facebook.com/v17.0/oauth/access_token" }</code>
     *         </p>
     *         </dd>
     */
    public final Map<String, String> providerDetails() {
        return providerDetails;
    }

    /**
     * For responses, this returns true if the service returned a value for the AttributeMapping property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAttributeMapping() {
        return attributeMapping != null && !(attributeMapping instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The IdP attribute mapping to be changed.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAttributeMapping} method.
     * </p>
     * 
     * @return The IdP attribute mapping to be changed.
     */
    public final Map<String, String> attributeMapping() {
        return attributeMapping;
    }

    /**
     * For responses, this returns true if the service returned a value for the IdpIdentifiers property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasIdpIdentifiers() {
        return idpIdentifiers != null && !(idpIdentifiers instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of IdP identifiers.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasIdpIdentifiers} method.
     * </p>
     * 
     * @return A list of IdP identifiers.
     */
    public final List<String> idpIdentifiers() {
        return idpIdentifiers;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(userPoolId());
        hashCode = 31 * hashCode + Objects.hashCode(providerName());
        hashCode = 31 * hashCode + Objects.hashCode(hasProviderDetails() ? providerDetails() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasAttributeMapping() ? attributeMapping() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasIdpIdentifiers() ? idpIdentifiers() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof UpdateIdentityProviderRequest)) {
            return false;
        }
        UpdateIdentityProviderRequest other = (UpdateIdentityProviderRequest) obj;
        return Objects.equals(userPoolId(), other.userPoolId()) && Objects.equals(providerName(), other.providerName())
                && hasProviderDetails() == other.hasProviderDetails()
                && Objects.equals(providerDetails(), other.providerDetails())
                && hasAttributeMapping() == other.hasAttributeMapping()
                && Objects.equals(attributeMapping(), other.attributeMapping())
                && hasIdpIdentifiers() == other.hasIdpIdentifiers() && Objects.equals(idpIdentifiers(), other.idpIdentifiers());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("UpdateIdentityProviderRequest").add("UserPoolId", userPoolId())
                .add("ProviderName", providerName()).add("ProviderDetails", hasProviderDetails() ? providerDetails() : null)
                .add("AttributeMapping", hasAttributeMapping() ? attributeMapping() : null)
                .add("IdpIdentifiers", hasIdpIdentifiers() ? idpIdentifiers() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "UserPoolId":
            return Optional.ofNullable(clazz.cast(userPoolId()));
        case "ProviderName":
            return Optional.ofNullable(clazz.cast(providerName()));
        case "ProviderDetails":
            return Optional.ofNullable(clazz.cast(providerDetails()));
        case "AttributeMapping":
            return Optional.ofNullable(clazz.cast(attributeMapping()));
        case "IdpIdentifiers":
            return Optional.ofNullable(clazz.cast(idpIdentifiers()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("UserPoolId", USER_POOL_ID_FIELD);
        map.put("ProviderName", PROVIDER_NAME_FIELD);
        map.put("ProviderDetails", PROVIDER_DETAILS_FIELD);
        map.put("AttributeMapping", ATTRIBUTE_MAPPING_FIELD);
        map.put("IdpIdentifiers", IDP_IDENTIFIERS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<UpdateIdentityProviderRequest, T> g) {
        return obj -> g.apply((UpdateIdentityProviderRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends CognitoIdentityProviderRequest.Builder, SdkPojo,
            CopyableBuilder<Builder, UpdateIdentityProviderRequest> {
        /**
         * <p>
         * The user pool ID.
         * </p>
         * 
         * @param userPoolId
         *        The user pool ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userPoolId(String userPoolId);

        /**
         * <p>
         * The IdP name.
         * </p>
         * 
         * @param providerName
         *        The IdP name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder providerName(String providerName);

        /**
         * <p>
         * The scopes, URLs, and identifiers for your external identity provider. The following examples describe the
         * provider detail keys for each IdP type. These values and their schema are subject to change. Social IdP
         * <code>authorize_scopes</code> values must match the values listed here.
         * </p>
         * <dl>
         * <dt>OpenID Connect (OIDC)</dt>
         * <dd>
         * <p>
         * Amazon Cognito accepts the following elements when it can't discover endpoint URLs from
         * <code>oidc_issuer</code>: <code>attributes_url</code>, <code>authorize_url</code>, <code>jwks_uri</code>,
         * <code>token_url</code>.
         * </p>
         * <p>
         * Create or update request:
         * <code>"ProviderDetails": { "attributes_request_method": "GET", "attributes_url": "https://auth.example.com/userInfo", "authorize_scopes": "openid profile email", "authorize_url": "https://auth.example.com/authorize", "client_id": "1example23456789", "client_secret": "provider-app-client-secret", "jwks_uri": "https://auth.example.com/.well-known/jwks.json", "oidc_issuer": "https://auth.example.com", "token_url": "https://example.com/token" }</code>
         * </p>
         * <p>
         * Describe response:
         * <code>"ProviderDetails": { "attributes_request_method": "GET", "attributes_url": "https://auth.example.com/userInfo", "attributes_url_add_attributes": "false", "authorize_scopes": "openid profile email", "authorize_url": "https://auth.example.com/authorize", "client_id": "1example23456789", "client_secret": "provider-app-client-secret", "jwks_uri": "https://auth.example.com/.well-known/jwks.json", "oidc_issuer": "https://auth.example.com", "token_url": "https://example.com/token" }</code>
         * </p>
         * </dd>
         * <dt>SAML</dt>
         * <dd>
         * <p>
         * Create or update request with Metadata URL:
         * <code>"ProviderDetails": { "IDPInit": "true", "IDPSignout": "true", "EncryptedResponses" : "true", "MetadataURL": "https://auth.example.com/sso/saml/metadata", "RequestSigningAlgorithm": "rsa-sha256" }</code>
         * </p>
         * <p>
         * Create or update request with Metadata file:
         * <code>"ProviderDetails": { "IDPInit": "true", "IDPSignout": "true", "EncryptedResponses" : "true", "MetadataFile": "[metadata XML]", "RequestSigningAlgorithm": "rsa-sha256" }</code>
         * </p>
         * <p>
         * The value of <code>MetadataFile</code> must be the plaintext metadata document with all quote (") characters
         * escaped by backslashes.
         * </p>
         * <p>
         * Describe response:
         * <code>"ProviderDetails": { "IDPInit": "true", "IDPSignout": "true", "EncryptedResponses" : "true", "ActiveEncryptionCertificate": "[certificate]", "MetadataURL": "https://auth.example.com/sso/saml/metadata", "RequestSigningAlgorithm": "rsa-sha256", "SLORedirectBindingURI": "https://auth.example.com/slo/saml", "SSORedirectBindingURI": "https://auth.example.com/sso/saml" }</code>
         * </p>
         * </dd>
         * <dt>LoginWithAmazon</dt>
         * <dd>
         * <p>
         * Create or update request:
         * <code>"ProviderDetails": { "authorize_scopes": "profile postal_code", "client_id": "amzn1.application-oa2-client.1example23456789", "client_secret": "provider-app-client-secret"</code>
         * </p>
         * <p>
         * Describe response:
         * <code>"ProviderDetails": { "attributes_url": "https://api.amazon.com/user/profile", "attributes_url_add_attributes": "false", "authorize_scopes": "profile postal_code", "authorize_url": "https://www.amazon.com/ap/oa", "client_id": "amzn1.application-oa2-client.1example23456789", "client_secret": "provider-app-client-secret", "token_request_method": "POST", "token_url": "https://api.amazon.com/auth/o2/token" }</code>
         * </p>
         * </dd>
         * <dt>Google</dt>
         * <dd>
         * <p>
         * Create or update request:
         * <code>"ProviderDetails": { "authorize_scopes": "email profile openid", "client_id": "1example23456789.apps.googleusercontent.com", "client_secret": "provider-app-client-secret" }</code>
         * </p>
         * <p>
         * Describe response:
         * <code>"ProviderDetails": { "attributes_url": "https://people.googleapis.com/v1/people/me?personFields=", "attributes_url_add_attributes": "true", "authorize_scopes": "email profile openid", "authorize_url": "https://accounts.google.com/o/oauth2/v2/auth", "client_id": "1example23456789.apps.googleusercontent.com", "client_secret": "provider-app-client-secret", "oidc_issuer": "https://accounts.google.com", "token_request_method": "POST", "token_url": "https://www.googleapis.com/oauth2/v4/token" }</code>
         * </p>
         * </dd>
         * <dt>SignInWithApple</dt>
         * <dd>
         * <p>
         * Create or update request:
         * <code>"ProviderDetails": { "authorize_scopes": "email name", "client_id": "com.example.cognito", "private_key": "1EXAMPLE", "key_id": "2EXAMPLE", "team_id": "3EXAMPLE" }</code>
         * </p>
         * <p>
         * Describe response:
         * <code>"ProviderDetails": { "attributes_url_add_attributes": "false", "authorize_scopes": "email name", "authorize_url": "https://appleid.apple.com/auth/authorize", "client_id": "com.example.cognito", "key_id": "1EXAMPLE", "oidc_issuer": "https://appleid.apple.com", "team_id": "2EXAMPLE", "token_request_method": "POST", "token_url": "https://appleid.apple.com/auth/token" }</code>
         * </p>
         * </dd>
         * <dt>Facebook</dt>
         * <dd>
         * <p>
         * Create or update request:
         * <code>"ProviderDetails": { "api_version": "v17.0", "authorize_scopes": "public_profile, email", "client_id": "1example23456789", "client_secret": "provider-app-client-secret" }</code>
         * </p>
         * <p>
         * Describe response:
         * <code>"ProviderDetails": { "api_version": "v17.0", "attributes_url": "https://graph.facebook.com/v17.0/me?fields=", "attributes_url_add_attributes": "true", "authorize_scopes": "public_profile, email", "authorize_url": "https://www.facebook.com/v17.0/dialog/oauth", "client_id": "1example23456789", "client_secret": "provider-app-client-secret", "token_request_method": "GET", "token_url": "https://graph.facebook.com/v17.0/oauth/access_token" }</code>
         * </p>
         * </dd>
         * </dl>
         * 
         * @param providerDetails
         *        The scopes, URLs, and identifiers for your external identity provider. The following examples describe
         *        the provider detail keys for each IdP type. These values and their schema are subject to change.
         *        Social IdP <code>authorize_scopes</code> values must match the values listed here.</p>
         *        <dl>
         *        <dt>OpenID Connect (OIDC)</dt>
         *        <dd>
         *        <p>
         *        Amazon Cognito accepts the following elements when it can't discover endpoint URLs from
         *        <code>oidc_issuer</code>: <code>attributes_url</code>, <code>authorize_url</code>,
         *        <code>jwks_uri</code>, <code>token_url</code>.
         *        </p>
         *        <p>
         *        Create or update request:
         *        <code>"ProviderDetails": { "attributes_request_method": "GET", "attributes_url": "https://auth.example.com/userInfo", "authorize_scopes": "openid profile email", "authorize_url": "https://auth.example.com/authorize", "client_id": "1example23456789", "client_secret": "provider-app-client-secret", "jwks_uri": "https://auth.example.com/.well-known/jwks.json", "oidc_issuer": "https://auth.example.com", "token_url": "https://example.com/token" }</code>
         *        </p>
         *        <p>
         *        Describe response:
         *        <code>"ProviderDetails": { "attributes_request_method": "GET", "attributes_url": "https://auth.example.com/userInfo", "attributes_url_add_attributes": "false", "authorize_scopes": "openid profile email", "authorize_url": "https://auth.example.com/authorize", "client_id": "1example23456789", "client_secret": "provider-app-client-secret", "jwks_uri": "https://auth.example.com/.well-known/jwks.json", "oidc_issuer": "https://auth.example.com", "token_url": "https://example.com/token" }</code>
         *        </p>
         *        </dd>
         *        <dt>SAML</dt>
         *        <dd>
         *        <p>
         *        Create or update request with Metadata URL:
         *        <code>"ProviderDetails": { "IDPInit": "true", "IDPSignout": "true", "EncryptedResponses" : "true", "MetadataURL": "https://auth.example.com/sso/saml/metadata", "RequestSigningAlgorithm": "rsa-sha256" }</code>
         *        </p>
         *        <p>
         *        Create or update request with Metadata file:
         *        <code>"ProviderDetails": { "IDPInit": "true", "IDPSignout": "true", "EncryptedResponses" : "true", "MetadataFile": "[metadata XML]", "RequestSigningAlgorithm": "rsa-sha256" }</code>
         *        </p>
         *        <p>
         *        The value of <code>MetadataFile</code> must be the plaintext metadata document with all quote (")
         *        characters escaped by backslashes.
         *        </p>
         *        <p>
         *        Describe response:
         *        <code>"ProviderDetails": { "IDPInit": "true", "IDPSignout": "true", "EncryptedResponses" : "true", "ActiveEncryptionCertificate": "[certificate]", "MetadataURL": "https://auth.example.com/sso/saml/metadata", "RequestSigningAlgorithm": "rsa-sha256", "SLORedirectBindingURI": "https://auth.example.com/slo/saml", "SSORedirectBindingURI": "https://auth.example.com/sso/saml" }</code>
         *        </p>
         *        </dd>
         *        <dt>LoginWithAmazon</dt>
         *        <dd>
         *        <p>
         *        Create or update request:
         *        <code>"ProviderDetails": { "authorize_scopes": "profile postal_code", "client_id": "amzn1.application-oa2-client.1example23456789", "client_secret": "provider-app-client-secret"</code>
         *        </p>
         *        <p>
         *        Describe response:
         *        <code>"ProviderDetails": { "attributes_url": "https://api.amazon.com/user/profile", "attributes_url_add_attributes": "false", "authorize_scopes": "profile postal_code", "authorize_url": "https://www.amazon.com/ap/oa", "client_id": "amzn1.application-oa2-client.1example23456789", "client_secret": "provider-app-client-secret", "token_request_method": "POST", "token_url": "https://api.amazon.com/auth/o2/token" }</code>
         *        </p>
         *        </dd>
         *        <dt>Google</dt>
         *        <dd>
         *        <p>
         *        Create or update request:
         *        <code>"ProviderDetails": { "authorize_scopes": "email profile openid", "client_id": "1example23456789.apps.googleusercontent.com", "client_secret": "provider-app-client-secret" }</code>
         *        </p>
         *        <p>
         *        Describe response:
         *        <code>"ProviderDetails": { "attributes_url": "https://people.googleapis.com/v1/people/me?personFields=", "attributes_url_add_attributes": "true", "authorize_scopes": "email profile openid", "authorize_url": "https://accounts.google.com/o/oauth2/v2/auth", "client_id": "1example23456789.apps.googleusercontent.com", "client_secret": "provider-app-client-secret", "oidc_issuer": "https://accounts.google.com", "token_request_method": "POST", "token_url": "https://www.googleapis.com/oauth2/v4/token" }</code>
         *        </p>
         *        </dd>
         *        <dt>SignInWithApple</dt>
         *        <dd>
         *        <p>
         *        Create or update request:
         *        <code>"ProviderDetails": { "authorize_scopes": "email name", "client_id": "com.example.cognito", "private_key": "1EXAMPLE", "key_id": "2EXAMPLE", "team_id": "3EXAMPLE" }</code>
         *        </p>
         *        <p>
         *        Describe response:
         *        <code>"ProviderDetails": { "attributes_url_add_attributes": "false", "authorize_scopes": "email name", "authorize_url": "https://appleid.apple.com/auth/authorize", "client_id": "com.example.cognito", "key_id": "1EXAMPLE", "oidc_issuer": "https://appleid.apple.com", "team_id": "2EXAMPLE", "token_request_method": "POST", "token_url": "https://appleid.apple.com/auth/token" }</code>
         *        </p>
         *        </dd>
         *        <dt>Facebook</dt>
         *        <dd>
         *        <p>
         *        Create or update request:
         *        <code>"ProviderDetails": { "api_version": "v17.0", "authorize_scopes": "public_profile, email", "client_id": "1example23456789", "client_secret": "provider-app-client-secret" }</code>
         *        </p>
         *        <p>
         *        Describe response:
         *        <code>"ProviderDetails": { "api_version": "v17.0", "attributes_url": "https://graph.facebook.com/v17.0/me?fields=", "attributes_url_add_attributes": "true", "authorize_scopes": "public_profile, email", "authorize_url": "https://www.facebook.com/v17.0/dialog/oauth", "client_id": "1example23456789", "client_secret": "provider-app-client-secret", "token_request_method": "GET", "token_url": "https://graph.facebook.com/v17.0/oauth/access_token" }</code>
         *        </p>
         *        </dd>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder providerDetails(Map<String, String> providerDetails);

        /**
         * <p>
         * The IdP attribute mapping to be changed.
         * </p>
         * 
         * @param attributeMapping
         *        The IdP attribute mapping to be changed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributeMapping(Map<String, String> attributeMapping);

        /**
         * <p>
         * A list of IdP identifiers.
         * </p>
         * 
         * @param idpIdentifiers
         *        A list of IdP identifiers.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder idpIdentifiers(Collection<String> idpIdentifiers);

        /**
         * <p>
         * A list of IdP identifiers.
         * </p>
         * 
         * @param idpIdentifiers
         *        A list of IdP identifiers.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder idpIdentifiers(String... idpIdentifiers);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends CognitoIdentityProviderRequest.BuilderImpl implements Builder {
        private String userPoolId;

        private String providerName;

        private Map<String, String> providerDetails = DefaultSdkAutoConstructMap.getInstance();

        private Map<String, String> attributeMapping = DefaultSdkAutoConstructMap.getInstance();

        private List<String> idpIdentifiers = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(UpdateIdentityProviderRequest model) {
            super(model);
            userPoolId(model.userPoolId);
            providerName(model.providerName);
            providerDetails(model.providerDetails);
            attributeMapping(model.attributeMapping);
            idpIdentifiers(model.idpIdentifiers);
        }

        public final String getUserPoolId() {
            return userPoolId;
        }

        public final void setUserPoolId(String userPoolId) {
            this.userPoolId = userPoolId;
        }

        @Override
        public final Builder userPoolId(String userPoolId) {
            this.userPoolId = userPoolId;
            return this;
        }

        public final String getProviderName() {
            return providerName;
        }

        public final void setProviderName(String providerName) {
            this.providerName = providerName;
        }

        @Override
        public final Builder providerName(String providerName) {
            this.providerName = providerName;
            return this;
        }

        public final Map<String, String> getProviderDetails() {
            if (providerDetails instanceof SdkAutoConstructMap) {
                return null;
            }
            return providerDetails;
        }

        public final void setProviderDetails(Map<String, String> providerDetails) {
            this.providerDetails = ProviderDetailsTypeCopier.copy(providerDetails);
        }

        @Override
        public final Builder providerDetails(Map<String, String> providerDetails) {
            this.providerDetails = ProviderDetailsTypeCopier.copy(providerDetails);
            return this;
        }

        public final Map<String, String> getAttributeMapping() {
            if (attributeMapping instanceof SdkAutoConstructMap) {
                return null;
            }
            return attributeMapping;
        }

        public final void setAttributeMapping(Map<String, String> attributeMapping) {
            this.attributeMapping = AttributeMappingTypeCopier.copy(attributeMapping);
        }

        @Override
        public final Builder attributeMapping(Map<String, String> attributeMapping) {
            this.attributeMapping = AttributeMappingTypeCopier.copy(attributeMapping);
            return this;
        }

        public final Collection<String> getIdpIdentifiers() {
            if (idpIdentifiers instanceof SdkAutoConstructList) {
                return null;
            }
            return idpIdentifiers;
        }

        public final void setIdpIdentifiers(Collection<String> idpIdentifiers) {
            this.idpIdentifiers = IdpIdentifiersListTypeCopier.copy(idpIdentifiers);
        }

        @Override
        public final Builder idpIdentifiers(Collection<String> idpIdentifiers) {
            this.idpIdentifiers = IdpIdentifiersListTypeCopier.copy(idpIdentifiers);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder idpIdentifiers(String... idpIdentifiers) {
            idpIdentifiers(Arrays.asList(idpIdentifiers));
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public UpdateIdentityProviderRequest build() {
            return new UpdateIdentityProviderRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
