/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codestarconnections;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.SdkClient;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.regions.ServiceMetadata;
import software.amazon.awssdk.services.codestarconnections.model.CodeStarConnectionsException;
import software.amazon.awssdk.services.codestarconnections.model.CreateConnectionRequest;
import software.amazon.awssdk.services.codestarconnections.model.CreateConnectionResponse;
import software.amazon.awssdk.services.codestarconnections.model.DeleteConnectionRequest;
import software.amazon.awssdk.services.codestarconnections.model.DeleteConnectionResponse;
import software.amazon.awssdk.services.codestarconnections.model.GetConnectionRequest;
import software.amazon.awssdk.services.codestarconnections.model.GetConnectionResponse;
import software.amazon.awssdk.services.codestarconnections.model.LimitExceededException;
import software.amazon.awssdk.services.codestarconnections.model.ListConnectionsRequest;
import software.amazon.awssdk.services.codestarconnections.model.ListConnectionsResponse;
import software.amazon.awssdk.services.codestarconnections.model.ResourceNotFoundException;
import software.amazon.awssdk.services.codestarconnections.paginators.ListConnectionsIterable;

/**
 * Service client for accessing AWS CodeStar connections. This can be created using the static {@link #builder()}
 * method.
 *
 * <p>
 * This AWS CodeStar Connections API Reference provides descriptions and usage examples of the operations and data types
 * for the AWS CodeStar Connections API. You can use the Connections API to work with connections and installations.
 * </p>
 * <p>
 * <i>Connections</i> are configurations that you use to connect AWS resources to external code repositories. Each
 * connection is a resource that can be given to services such as CodePipeline to connect to a third-party repository
 * such as Bitbucket. For example, you can add the connection in CodePipeline so that it triggers your pipeline when a
 * code change is made to your third-party code repository. Each connection is named and associated with a unique ARN
 * that is used to reference the connection.
 * </p>
 * <p>
 * When you create a connection, the console initiates a third-party connection handshake. <i>Installations</i> are the
 * apps that are used to conduct this handshake. For example, the installation for the Bitbucket provider type is the
 * Bitbucket Cloud app. When you create a connection, you can choose an existing installation or create one.
 * </p>
 * <p>
 * You can work with connections by calling:
 * </p>
 * <ul>
 * <li>
 * <p>
 * <a>CreateConnection</a>, which creates a uniquely named connection that can be referenced by services such as
 * CodePipeline.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>DeleteConnection</a>, which deletes the specified connection.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>GetConnection</a>, which returns information about the connection, including the connection status.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>ListConnections</a>, which lists the connections associated with your account.
 * </p>
 * </li>
 * </ul>
 * <p>
 * For information about how to use AWS CodeStar Connections, see the <a
 * href="https://docs.aws.amazon.com/codepipeline/latest/userguide/welcome.html">AWS CodePipeline User Guide</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public interface CodeStarConnectionsClient extends SdkClient {
    String SERVICE_NAME = "codestar-connections";

    /**
     * Create a {@link CodeStarConnectionsClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static CodeStarConnectionsClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link CodeStarConnectionsClient}.
     */
    static CodeStarConnectionsClientBuilder builder() {
        return new DefaultCodeStarConnectionsClientBuilder();
    }

    /**
     * <p>
     * Creates a connection that can then be given to other AWS services like CodePipeline so that it can access
     * third-party code repositories. The connection is in pending status until the third-party connection handshake is
     * completed from the console.
     * </p>
     *
     * @param createConnectionRequest
     * @return Result of the CreateConnection operation returned by the service.
     * @throws LimitExceededException
     *         Exceeded the maximum limit for connections.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeStarConnectionsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeStarConnectionsClient.CreateConnection
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codestar-connections-2019-12-01/CreateConnection"
     *      target="_top">AWS API Documentation</a>
     */
    default CreateConnectionResponse createConnection(CreateConnectionRequest createConnectionRequest)
            throws LimitExceededException, AwsServiceException, SdkClientException, CodeStarConnectionsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates a connection that can then be given to other AWS services like CodePipeline so that it can access
     * third-party code repositories. The connection is in pending status until the third-party connection handshake is
     * completed from the console.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateConnectionRequest.Builder} avoiding the need
     * to create one manually via {@link CreateConnectionRequest#builder()}
     * </p>
     *
     * @param createConnectionRequest
     *        A {@link Consumer} that will call methods on {@link CreateConnectionInput.Builder} to create a request.
     * @return Result of the CreateConnection operation returned by the service.
     * @throws LimitExceededException
     *         Exceeded the maximum limit for connections.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeStarConnectionsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeStarConnectionsClient.CreateConnection
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codestar-connections-2019-12-01/CreateConnection"
     *      target="_top">AWS API Documentation</a>
     */
    default CreateConnectionResponse createConnection(Consumer<CreateConnectionRequest.Builder> createConnectionRequest)
            throws LimitExceededException, AwsServiceException, SdkClientException, CodeStarConnectionsException {
        return createConnection(CreateConnectionRequest.builder().applyMutation(createConnectionRequest).build());
    }

    /**
     * <p>
     * The connection to be deleted.
     * </p>
     *
     * @param deleteConnectionRequest
     * @return Result of the DeleteConnection operation returned by the service.
     * @throws ResourceNotFoundException
     *         Resource not found. Verify the connection resource ARN and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeStarConnectionsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeStarConnectionsClient.DeleteConnection
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codestar-connections-2019-12-01/DeleteConnection"
     *      target="_top">AWS API Documentation</a>
     */
    default DeleteConnectionResponse deleteConnection(DeleteConnectionRequest deleteConnectionRequest)
            throws ResourceNotFoundException, AwsServiceException, SdkClientException, CodeStarConnectionsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * The connection to be deleted.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteConnectionRequest.Builder} avoiding the need
     * to create one manually via {@link DeleteConnectionRequest#builder()}
     * </p>
     *
     * @param deleteConnectionRequest
     *        A {@link Consumer} that will call methods on {@link DeleteConnectionInput.Builder} to create a request.
     * @return Result of the DeleteConnection operation returned by the service.
     * @throws ResourceNotFoundException
     *         Resource not found. Verify the connection resource ARN and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeStarConnectionsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeStarConnectionsClient.DeleteConnection
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codestar-connections-2019-12-01/DeleteConnection"
     *      target="_top">AWS API Documentation</a>
     */
    default DeleteConnectionResponse deleteConnection(Consumer<DeleteConnectionRequest.Builder> deleteConnectionRequest)
            throws ResourceNotFoundException, AwsServiceException, SdkClientException, CodeStarConnectionsException {
        return deleteConnection(DeleteConnectionRequest.builder().applyMutation(deleteConnectionRequest).build());
    }

    /**
     * <p>
     * Returns the connection ARN and details such as status, owner, and provider type.
     * </p>
     *
     * @param getConnectionRequest
     * @return Result of the GetConnection operation returned by the service.
     * @throws ResourceNotFoundException
     *         Resource not found. Verify the connection resource ARN and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeStarConnectionsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeStarConnectionsClient.GetConnection
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codestar-connections-2019-12-01/GetConnection"
     *      target="_top">AWS API Documentation</a>
     */
    default GetConnectionResponse getConnection(GetConnectionRequest getConnectionRequest) throws ResourceNotFoundException,
            AwsServiceException, SdkClientException, CodeStarConnectionsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns the connection ARN and details such as status, owner, and provider type.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetConnectionRequest.Builder} avoiding the need to
     * create one manually via {@link GetConnectionRequest#builder()}
     * </p>
     *
     * @param getConnectionRequest
     *        A {@link Consumer} that will call methods on {@link GetConnectionInput.Builder} to create a request.
     * @return Result of the GetConnection operation returned by the service.
     * @throws ResourceNotFoundException
     *         Resource not found. Verify the connection resource ARN and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeStarConnectionsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeStarConnectionsClient.GetConnection
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codestar-connections-2019-12-01/GetConnection"
     *      target="_top">AWS API Documentation</a>
     */
    default GetConnectionResponse getConnection(Consumer<GetConnectionRequest.Builder> getConnectionRequest)
            throws ResourceNotFoundException, AwsServiceException, SdkClientException, CodeStarConnectionsException {
        return getConnection(GetConnectionRequest.builder().applyMutation(getConnectionRequest).build());
    }

    /**
     * <p>
     * Lists the connections associated with your account.
     * </p>
     *
     * @param listConnectionsRequest
     * @return Result of the ListConnections operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeStarConnectionsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeStarConnectionsClient.ListConnections
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codestar-connections-2019-12-01/ListConnections"
     *      target="_top">AWS API Documentation</a>
     */
    default ListConnectionsResponse listConnections(ListConnectionsRequest listConnectionsRequest) throws AwsServiceException,
            SdkClientException, CodeStarConnectionsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists the connections associated with your account.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListConnectionsRequest.Builder} avoiding the need
     * to create one manually via {@link ListConnectionsRequest#builder()}
     * </p>
     *
     * @param listConnectionsRequest
     *        A {@link Consumer} that will call methods on {@link ListConnectionsInput.Builder} to create a request.
     * @return Result of the ListConnections operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeStarConnectionsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeStarConnectionsClient.ListConnections
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codestar-connections-2019-12-01/ListConnections"
     *      target="_top">AWS API Documentation</a>
     */
    default ListConnectionsResponse listConnections(Consumer<ListConnectionsRequest.Builder> listConnectionsRequest)
            throws AwsServiceException, SdkClientException, CodeStarConnectionsException {
        return listConnections(ListConnectionsRequest.builder().applyMutation(listConnectionsRequest).build());
    }

    /**
     * <p>
     * Lists the connections associated with your account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listConnections(software.amazon.awssdk.services.codestarconnections.model.ListConnectionsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codestarconnections.paginators.ListConnectionsIterable responses = client.listConnectionsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.codestarconnections.paginators.ListConnectionsIterable responses = client
     *             .listConnectionsPaginator(request);
     *     for (software.amazon.awssdk.services.codestarconnections.model.ListConnectionsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codestarconnections.paginators.ListConnectionsIterable responses = client.listConnectionsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listConnections(software.amazon.awssdk.services.codestarconnections.model.ListConnectionsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listConnectionsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeStarConnectionsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeStarConnectionsClient.ListConnections
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codestar-connections-2019-12-01/ListConnections"
     *      target="_top">AWS API Documentation</a>
     */
    default ListConnectionsIterable listConnectionsPaginator(ListConnectionsRequest listConnectionsRequest)
            throws AwsServiceException, SdkClientException, CodeStarConnectionsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists the connections associated with your account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listConnections(software.amazon.awssdk.services.codestarconnections.model.ListConnectionsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codestarconnections.paginators.ListConnectionsIterable responses = client.listConnectionsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.codestarconnections.paginators.ListConnectionsIterable responses = client
     *             .listConnectionsPaginator(request);
     *     for (software.amazon.awssdk.services.codestarconnections.model.ListConnectionsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codestarconnections.paginators.ListConnectionsIterable responses = client.listConnectionsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listConnections(software.amazon.awssdk.services.codestarconnections.model.ListConnectionsRequest)}
     * operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link ListConnectionsRequest.Builder} avoiding the need
     * to create one manually via {@link ListConnectionsRequest#builder()}
     * </p>
     *
     * @param listConnectionsRequest
     *        A {@link Consumer} that will call methods on {@link ListConnectionsInput.Builder} to create a request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeStarConnectionsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeStarConnectionsClient.ListConnections
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codestar-connections-2019-12-01/ListConnections"
     *      target="_top">AWS API Documentation</a>
     */
    default ListConnectionsIterable listConnectionsPaginator(Consumer<ListConnectionsRequest.Builder> listConnectionsRequest)
            throws AwsServiceException, SdkClientException, CodeStarConnectionsException {
        return listConnectionsPaginator(ListConnectionsRequest.builder().applyMutation(listConnectionsRequest).build());
    }

    static ServiceMetadata serviceMetadata() {
        return ServiceMetadata.of("codestar-connections");
    }
}
