/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codeguruprofiler;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.codeguruprofiler.model.CodeGuruProfilerException;
import software.amazon.awssdk.services.codeguruprofiler.model.CodeGuruProfilerRequest;
import software.amazon.awssdk.services.codeguruprofiler.model.ConfigureAgentRequest;
import software.amazon.awssdk.services.codeguruprofiler.model.ConfigureAgentResponse;
import software.amazon.awssdk.services.codeguruprofiler.model.ConflictException;
import software.amazon.awssdk.services.codeguruprofiler.model.CreateProfilingGroupRequest;
import software.amazon.awssdk.services.codeguruprofiler.model.CreateProfilingGroupResponse;
import software.amazon.awssdk.services.codeguruprofiler.model.DeleteProfilingGroupRequest;
import software.amazon.awssdk.services.codeguruprofiler.model.DeleteProfilingGroupResponse;
import software.amazon.awssdk.services.codeguruprofiler.model.DescribeProfilingGroupRequest;
import software.amazon.awssdk.services.codeguruprofiler.model.DescribeProfilingGroupResponse;
import software.amazon.awssdk.services.codeguruprofiler.model.GetPolicyRequest;
import software.amazon.awssdk.services.codeguruprofiler.model.GetPolicyResponse;
import software.amazon.awssdk.services.codeguruprofiler.model.GetProfileRequest;
import software.amazon.awssdk.services.codeguruprofiler.model.GetProfileResponse;
import software.amazon.awssdk.services.codeguruprofiler.model.InternalServerException;
import software.amazon.awssdk.services.codeguruprofiler.model.ListProfileTimesRequest;
import software.amazon.awssdk.services.codeguruprofiler.model.ListProfileTimesResponse;
import software.amazon.awssdk.services.codeguruprofiler.model.ListProfilingGroupsRequest;
import software.amazon.awssdk.services.codeguruprofiler.model.ListProfilingGroupsResponse;
import software.amazon.awssdk.services.codeguruprofiler.model.PostAgentProfileRequest;
import software.amazon.awssdk.services.codeguruprofiler.model.PostAgentProfileResponse;
import software.amazon.awssdk.services.codeguruprofiler.model.PutPermissionRequest;
import software.amazon.awssdk.services.codeguruprofiler.model.PutPermissionResponse;
import software.amazon.awssdk.services.codeguruprofiler.model.RemovePermissionRequest;
import software.amazon.awssdk.services.codeguruprofiler.model.RemovePermissionResponse;
import software.amazon.awssdk.services.codeguruprofiler.model.ResourceNotFoundException;
import software.amazon.awssdk.services.codeguruprofiler.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.codeguruprofiler.model.ThrottlingException;
import software.amazon.awssdk.services.codeguruprofiler.model.UpdateProfilingGroupRequest;
import software.amazon.awssdk.services.codeguruprofiler.model.UpdateProfilingGroupResponse;
import software.amazon.awssdk.services.codeguruprofiler.model.ValidationException;
import software.amazon.awssdk.services.codeguruprofiler.paginators.ListProfileTimesIterable;
import software.amazon.awssdk.services.codeguruprofiler.paginators.ListProfilingGroupsIterable;
import software.amazon.awssdk.services.codeguruprofiler.transform.ConfigureAgentRequestMarshaller;
import software.amazon.awssdk.services.codeguruprofiler.transform.CreateProfilingGroupRequestMarshaller;
import software.amazon.awssdk.services.codeguruprofiler.transform.DeleteProfilingGroupRequestMarshaller;
import software.amazon.awssdk.services.codeguruprofiler.transform.DescribeProfilingGroupRequestMarshaller;
import software.amazon.awssdk.services.codeguruprofiler.transform.GetPolicyRequestMarshaller;
import software.amazon.awssdk.services.codeguruprofiler.transform.GetProfileRequestMarshaller;
import software.amazon.awssdk.services.codeguruprofiler.transform.ListProfileTimesRequestMarshaller;
import software.amazon.awssdk.services.codeguruprofiler.transform.ListProfilingGroupsRequestMarshaller;
import software.amazon.awssdk.services.codeguruprofiler.transform.PostAgentProfileRequestMarshaller;
import software.amazon.awssdk.services.codeguruprofiler.transform.PutPermissionRequestMarshaller;
import software.amazon.awssdk.services.codeguruprofiler.transform.RemovePermissionRequestMarshaller;
import software.amazon.awssdk.services.codeguruprofiler.transform.UpdateProfilingGroupRequestMarshaller;

/**
 * Internal implementation of {@link CodeGuruProfilerClient}.
 *
 * @see CodeGuruProfilerClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultCodeGuruProfilerClient implements CodeGuruProfilerClient {
    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultCodeGuruProfilerClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p/>
     *
     * @param configureAgentRequest
     *        The structure representing the configureAgentRequest.
     * @return Result of the ConfigureAgent operation returned by the service.
     * @throws InternalServerException
     *         The server encountered an internal error and is unable to complete the request.
     * @throws ValidationException
     *         The parameter is not valid.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         The resource specified in the request does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeGuruProfilerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeGuruProfilerClient.ConfigureAgent
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeguruprofiler-2019-07-18/ConfigureAgent"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ConfigureAgentResponse configureAgent(ConfigureAgentRequest configureAgentRequest) throws InternalServerException,
            ValidationException, ThrottlingException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            CodeGuruProfilerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ConfigureAgentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ConfigureAgentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ConfigureAgentRequest, ConfigureAgentResponse>()
                .withOperationName("ConfigureAgent").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(configureAgentRequest)
                .withMarshaller(new ConfigureAgentRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Creates a profiling group.
     * </p>
     *
     * @param createProfilingGroupRequest
     *        The structure representing the createProfiliingGroupRequest.
     * @return Result of the CreateProfilingGroup operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         You have exceeded your service quota. To perform the requested action, remove some of the relevant
     *         resources, or use <a href="https://docs.aws.amazon.com/servicequotas/latest/userguide/intro.html">Service
     *         Quotas</a> to request a service quota increase.
     * @throws InternalServerException
     *         The server encountered an internal error and is unable to complete the request.
     * @throws ConflictException
     *         The requested operation would cause a conflict with the current state of a service resource associated
     *         with the request. Resolve the conflict before retrying this request.
     * @throws ValidationException
     *         The parameter is not valid.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeGuruProfilerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeGuruProfilerClient.CreateProfilingGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeguruprofiler-2019-07-18/CreateProfilingGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateProfilingGroupResponse createProfilingGroup(CreateProfilingGroupRequest createProfilingGroupRequest)
            throws ServiceQuotaExceededException, InternalServerException, ConflictException, ValidationException,
            ThrottlingException, AwsServiceException, SdkClientException, CodeGuruProfilerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateProfilingGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateProfilingGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<CreateProfilingGroupRequest, CreateProfilingGroupResponse>()
                .withOperationName("CreateProfilingGroup").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(createProfilingGroupRequest)
                .withMarshaller(new CreateProfilingGroupRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes a profiling group.
     * </p>
     *
     * @param deleteProfilingGroupRequest
     *        The structure representing the deleteProfilingGroupRequest.
     * @return Result of the DeleteProfilingGroup operation returned by the service.
     * @throws InternalServerException
     *         The server encountered an internal error and is unable to complete the request.
     * @throws ValidationException
     *         The parameter is not valid.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         The resource specified in the request does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeGuruProfilerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeGuruProfilerClient.DeleteProfilingGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeguruprofiler-2019-07-18/DeleteProfilingGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteProfilingGroupResponse deleteProfilingGroup(DeleteProfilingGroupRequest deleteProfilingGroupRequest)
            throws InternalServerException, ValidationException, ThrottlingException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, CodeGuruProfilerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteProfilingGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteProfilingGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeleteProfilingGroupRequest, DeleteProfilingGroupResponse>()
                .withOperationName("DeleteProfilingGroup").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteProfilingGroupRequest)
                .withMarshaller(new DeleteProfilingGroupRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Describes a profiling group.
     * </p>
     *
     * @param describeProfilingGroupRequest
     *        The structure representing the describeProfilingGroupRequest.
     * @return Result of the DescribeProfilingGroup operation returned by the service.
     * @throws InternalServerException
     *         The server encountered an internal error and is unable to complete the request.
     * @throws ValidationException
     *         The parameter is not valid.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         The resource specified in the request does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeGuruProfilerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeGuruProfilerClient.DescribeProfilingGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeguruprofiler-2019-07-18/DescribeProfilingGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeProfilingGroupResponse describeProfilingGroup(DescribeProfilingGroupRequest describeProfilingGroupRequest)
            throws InternalServerException, ValidationException, ThrottlingException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, CodeGuruProfilerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeProfilingGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeProfilingGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DescribeProfilingGroupRequest, DescribeProfilingGroupResponse>()
                .withOperationName("DescribeProfilingGroup").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeProfilingGroupRequest)
                .withMarshaller(new DescribeProfilingGroupRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Gets the profiling group policy.
     * </p>
     *
     * @param getPolicyRequest
     *        The structure representing the getPolicyRequest.
     * @return Result of the GetPolicy operation returned by the service.
     * @throws InternalServerException
     *         The server encountered an internal error and is unable to complete the request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         The resource specified in the request does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeGuruProfilerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeGuruProfilerClient.GetPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeguruprofiler-2019-07-18/GetPolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetPolicyResponse getPolicy(GetPolicyRequest getPolicyRequest) throws InternalServerException, ThrottlingException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, CodeGuruProfilerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetPolicyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetPolicyRequest, GetPolicyResponse>()
                .withOperationName("GetPolicy").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getPolicyRequest)
                .withMarshaller(new GetPolicyRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Gets the aggregated profile of a profiling group for the specified time range. If the requested time range does
     * not align with the available aggregated profiles, it is expanded to attain alignment. If aggregated profiles are
     * available only for part of the period requested, the profile is returned from the earliest available to the
     * latest within the requested time range.
     * </p>
     * <p>
     * For example, if the requested time range is from 00:00 to 00:20 and the available profiles are from 00:15 to
     * 00:25, the returned profile will be from 00:15 to 00:20.
     * </p>
     * <p>
     * You must specify exactly two of the following parameters: <code>startTime</code>, <code>period</code>, and
     * <code>endTime</code>.
     * </p>
     *
     * @param getProfileRequest
     *        The structure representing the getProfileRequest.
     * @return Result of the GetProfile operation returned by the service.
     * @throws InternalServerException
     *         The server encountered an internal error and is unable to complete the request.
     * @throws ValidationException
     *         The parameter is not valid.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         The resource specified in the request does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeGuruProfilerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeGuruProfilerClient.GetProfile
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeguruprofiler-2019-07-18/GetProfile" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetProfileResponse getProfile(GetProfileRequest getProfileRequest) throws InternalServerException,
            ValidationException, ThrottlingException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            CodeGuruProfilerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(false).build();

        HttpResponseHandler<GetProfileResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetProfileResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetProfileRequest, GetProfileResponse>()
                .withOperationName("GetProfile").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getProfileRequest)
                .withMarshaller(new GetProfileRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * List the start times of the available aggregated profiles of a profiling group for an aggregation period within
     * the specified time range.
     * </p>
     *
     * @param listProfileTimesRequest
     *        The structure representing the listProfileTimesRequest.
     * @return Result of the ListProfileTimes operation returned by the service.
     * @throws InternalServerException
     *         The server encountered an internal error and is unable to complete the request.
     * @throws ValidationException
     *         The parameter is not valid.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         The resource specified in the request does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeGuruProfilerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeGuruProfilerClient.ListProfileTimes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeguruprofiler-2019-07-18/ListProfileTimes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListProfileTimesResponse listProfileTimes(ListProfileTimesRequest listProfileTimesRequest)
            throws InternalServerException, ValidationException, ThrottlingException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, CodeGuruProfilerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListProfileTimesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListProfileTimesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListProfileTimesRequest, ListProfileTimesResponse>()
                .withOperationName("ListProfileTimes").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listProfileTimesRequest)
                .withMarshaller(new ListProfileTimesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * List the start times of the available aggregated profiles of a profiling group for an aggregation period within
     * the specified time range.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listProfileTimes(software.amazon.awssdk.services.codeguruprofiler.model.ListProfileTimesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codeguruprofiler.paginators.ListProfileTimesIterable responses = client.listProfileTimesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.codeguruprofiler.paginators.ListProfileTimesIterable responses = client
     *             .listProfileTimesPaginator(request);
     *     for (software.amazon.awssdk.services.codeguruprofiler.model.ListProfileTimesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codeguruprofiler.paginators.ListProfileTimesIterable responses = client.listProfileTimesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listProfileTimes(software.amazon.awssdk.services.codeguruprofiler.model.ListProfileTimesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listProfileTimesRequest
     *        The structure representing the listProfileTimesRequest.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         The server encountered an internal error and is unable to complete the request.
     * @throws ValidationException
     *         The parameter is not valid.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         The resource specified in the request does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeGuruProfilerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeGuruProfilerClient.ListProfileTimes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeguruprofiler-2019-07-18/ListProfileTimes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListProfileTimesIterable listProfileTimesPaginator(ListProfileTimesRequest listProfileTimesRequest)
            throws InternalServerException, ValidationException, ThrottlingException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, CodeGuruProfilerException {
        return new ListProfileTimesIterable(this, applyPaginatorUserAgent(listProfileTimesRequest));
    }

    /**
     * <p>
     * Lists profiling groups.
     * </p>
     *
     * @param listProfilingGroupsRequest
     *        The structure representing the listProfilingGroupsRequest.
     * @return Result of the ListProfilingGroups operation returned by the service.
     * @throws InternalServerException
     *         The server encountered an internal error and is unable to complete the request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeGuruProfilerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeGuruProfilerClient.ListProfilingGroups
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeguruprofiler-2019-07-18/ListProfilingGroups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListProfilingGroupsResponse listProfilingGroups(ListProfilingGroupsRequest listProfilingGroupsRequest)
            throws InternalServerException, ThrottlingException, AwsServiceException, SdkClientException,
            CodeGuruProfilerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListProfilingGroupsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListProfilingGroupsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListProfilingGroupsRequest, ListProfilingGroupsResponse>()
                .withOperationName("ListProfilingGroups").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listProfilingGroupsRequest)
                .withMarshaller(new ListProfilingGroupsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists profiling groups.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listProfilingGroups(software.amazon.awssdk.services.codeguruprofiler.model.ListProfilingGroupsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codeguruprofiler.paginators.ListProfilingGroupsIterable responses = client.listProfilingGroupsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.codeguruprofiler.paginators.ListProfilingGroupsIterable responses = client
     *             .listProfilingGroupsPaginator(request);
     *     for (software.amazon.awssdk.services.codeguruprofiler.model.ListProfilingGroupsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codeguruprofiler.paginators.ListProfilingGroupsIterable responses = client.listProfilingGroupsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listProfilingGroups(software.amazon.awssdk.services.codeguruprofiler.model.ListProfilingGroupsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listProfilingGroupsRequest
     *        The structure representing the listProfilingGroupsRequest.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         The server encountered an internal error and is unable to complete the request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeGuruProfilerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeGuruProfilerClient.ListProfilingGroups
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeguruprofiler-2019-07-18/ListProfilingGroups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListProfilingGroupsIterable listProfilingGroupsPaginator(ListProfilingGroupsRequest listProfilingGroupsRequest)
            throws InternalServerException, ThrottlingException, AwsServiceException, SdkClientException,
            CodeGuruProfilerException {
        return new ListProfilingGroupsIterable(this, applyPaginatorUserAgent(listProfilingGroupsRequest));
    }

    /**
     * <p/>
     *
     * @param postAgentProfileRequest
     *        The structure representing the postAgentProfileRequest.
     * @return Result of the PostAgentProfile operation returned by the service.
     * @throws InternalServerException
     *         The server encountered an internal error and is unable to complete the request.
     * @throws ValidationException
     *         The parameter is not valid.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         The resource specified in the request does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeGuruProfilerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeGuruProfilerClient.PostAgentProfile
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeguruprofiler-2019-07-18/PostAgentProfile"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PostAgentProfileResponse postAgentProfile(PostAgentProfileRequest postAgentProfileRequest)
            throws InternalServerException, ValidationException, ThrottlingException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, CodeGuruProfilerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PostAgentProfileResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PostAgentProfileResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<PostAgentProfileRequest, PostAgentProfileResponse>()
                .withOperationName("PostAgentProfile").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(postAgentProfileRequest)
                .withMarshaller(new PostAgentProfileRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Provides permission to the principals. This overwrites the existing permissions, and is not additive.
     * </p>
     *
     * @param putPermissionRequest
     *        The structure representing the putPermissionRequest.
     * @return Result of the PutPermission operation returned by the service.
     * @throws InternalServerException
     *         The server encountered an internal error and is unable to complete the request.
     * @throws ConflictException
     *         The requested operation would cause a conflict with the current state of a service resource associated
     *         with the request. Resolve the conflict before retrying this request.
     * @throws ValidationException
     *         The parameter is not valid.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         The resource specified in the request does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeGuruProfilerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeGuruProfilerClient.PutPermission
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeguruprofiler-2019-07-18/PutPermission" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public PutPermissionResponse putPermission(PutPermissionRequest putPermissionRequest) throws InternalServerException,
            ConflictException, ValidationException, ThrottlingException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, CodeGuruProfilerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutPermissionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PutPermissionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<PutPermissionRequest, PutPermissionResponse>()
                .withOperationName("PutPermission").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(putPermissionRequest)
                .withMarshaller(new PutPermissionRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Removes statement for the provided action group from the policy.
     * </p>
     *
     * @param removePermissionRequest
     *        The structure representing the removePermissionRequest.
     * @return Result of the RemovePermission operation returned by the service.
     * @throws InternalServerException
     *         The server encountered an internal error and is unable to complete the request.
     * @throws ConflictException
     *         The requested operation would cause a conflict with the current state of a service resource associated
     *         with the request. Resolve the conflict before retrying this request.
     * @throws ValidationException
     *         The parameter is not valid.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         The resource specified in the request does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeGuruProfilerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeGuruProfilerClient.RemovePermission
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeguruprofiler-2019-07-18/RemovePermission"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public RemovePermissionResponse removePermission(RemovePermissionRequest removePermissionRequest)
            throws InternalServerException, ConflictException, ValidationException, ThrottlingException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, CodeGuruProfilerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RemovePermissionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                RemovePermissionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<RemovePermissionRequest, RemovePermissionResponse>()
                .withOperationName("RemovePermission").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(removePermissionRequest)
                .withMarshaller(new RemovePermissionRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Updates a profiling group.
     * </p>
     *
     * @param updateProfilingGroupRequest
     *        The structure representing the updateProfilingGroupRequest.
     * @return Result of the UpdateProfilingGroup operation returned by the service.
     * @throws InternalServerException
     *         The server encountered an internal error and is unable to complete the request.
     * @throws ConflictException
     *         The requested operation would cause a conflict with the current state of a service resource associated
     *         with the request. Resolve the conflict before retrying this request.
     * @throws ValidationException
     *         The parameter is not valid.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         The resource specified in the request does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeGuruProfilerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeGuruProfilerClient.UpdateProfilingGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeguruprofiler-2019-07-18/UpdateProfilingGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateProfilingGroupResponse updateProfilingGroup(UpdateProfilingGroupRequest updateProfilingGroupRequest)
            throws InternalServerException, ConflictException, ValidationException, ThrottlingException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, CodeGuruProfilerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateProfilingGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateProfilingGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UpdateProfilingGroupRequest, UpdateProfilingGroupResponse>()
                .withOperationName("UpdateProfilingGroup").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(updateProfilingGroupRequest)
                .withMarshaller(new UpdateProfilingGroupRequestMarshaller(protocolFactory)));
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(CodeGuruProfilerException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends CodeGuruProfilerRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
