/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codecommit.model;

import java.beans.Transient;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateCommitRequest extends CodeCommitRequest implements
        ToCopyableBuilder<CreateCommitRequest.Builder, CreateCommitRequest> {
    private static final SdkField<String> REPOSITORY_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("repositoryName").getter(getter(CreateCommitRequest::repositoryName))
            .setter(setter(Builder::repositoryName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("repositoryName").build()).build();

    private static final SdkField<String> BRANCH_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("branchName").getter(getter(CreateCommitRequest::branchName)).setter(setter(Builder::branchName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("branchName").build()).build();

    private static final SdkField<String> PARENT_COMMIT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("parentCommitId").getter(getter(CreateCommitRequest::parentCommitId))
            .setter(setter(Builder::parentCommitId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("parentCommitId").build()).build();

    private static final SdkField<String> AUTHOR_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("authorName").getter(getter(CreateCommitRequest::authorName)).setter(setter(Builder::authorName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("authorName").build()).build();

    private static final SdkField<String> EMAIL_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("email")
            .getter(getter(CreateCommitRequest::email)).setter(setter(Builder::email))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("email").build()).build();

    private static final SdkField<String> COMMIT_MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("commitMessage").getter(getter(CreateCommitRequest::commitMessage))
            .setter(setter(Builder::commitMessage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("commitMessage").build()).build();

    private static final SdkField<Boolean> KEEP_EMPTY_FOLDERS_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("keepEmptyFolders").getter(getter(CreateCommitRequest::keepEmptyFolders))
            .setter(setter(Builder::keepEmptyFolders))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("keepEmptyFolders").build()).build();

    private static final SdkField<List<PutFileEntry>> PUT_FILES_FIELD = SdkField
            .<List<PutFileEntry>> builder(MarshallingType.LIST)
            .memberName("putFiles")
            .getter(getter(CreateCommitRequest::putFiles))
            .setter(setter(Builder::putFiles))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("putFiles").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<PutFileEntry> builder(MarshallingType.SDK_POJO)
                                            .constructor(PutFileEntry::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<DeleteFileEntry>> DELETE_FILES_FIELD = SdkField
            .<List<DeleteFileEntry>> builder(MarshallingType.LIST)
            .memberName("deleteFiles")
            .getter(getter(CreateCommitRequest::deleteFiles))
            .setter(setter(Builder::deleteFiles))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("deleteFiles").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<DeleteFileEntry> builder(MarshallingType.SDK_POJO)
                                            .constructor(DeleteFileEntry::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<SetFileModeEntry>> SET_FILE_MODES_FIELD = SdkField
            .<List<SetFileModeEntry>> builder(MarshallingType.LIST)
            .memberName("setFileModes")
            .getter(getter(CreateCommitRequest::setFileModes))
            .setter(setter(Builder::setFileModes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("setFileModes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<SetFileModeEntry> builder(MarshallingType.SDK_POJO)
                                            .constructor(SetFileModeEntry::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(REPOSITORY_NAME_FIELD,
            BRANCH_NAME_FIELD, PARENT_COMMIT_ID_FIELD, AUTHOR_NAME_FIELD, EMAIL_FIELD, COMMIT_MESSAGE_FIELD,
            KEEP_EMPTY_FOLDERS_FIELD, PUT_FILES_FIELD, DELETE_FILES_FIELD, SET_FILE_MODES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private final String repositoryName;

    private final String branchName;

    private final String parentCommitId;

    private final String authorName;

    private final String email;

    private final String commitMessage;

    private final Boolean keepEmptyFolders;

    private final List<PutFileEntry> putFiles;

    private final List<DeleteFileEntry> deleteFiles;

    private final List<SetFileModeEntry> setFileModes;

    private CreateCommitRequest(BuilderImpl builder) {
        super(builder);
        this.repositoryName = builder.repositoryName;
        this.branchName = builder.branchName;
        this.parentCommitId = builder.parentCommitId;
        this.authorName = builder.authorName;
        this.email = builder.email;
        this.commitMessage = builder.commitMessage;
        this.keepEmptyFolders = builder.keepEmptyFolders;
        this.putFiles = builder.putFiles;
        this.deleteFiles = builder.deleteFiles;
        this.setFileModes = builder.setFileModes;
    }

    /**
     * <p>
     * The name of the repository where you create the commit.
     * </p>
     * 
     * @return The name of the repository where you create the commit.
     */
    public final String repositoryName() {
        return repositoryName;
    }

    /**
     * <p>
     * The name of the branch where you create the commit.
     * </p>
     * 
     * @return The name of the branch where you create the commit.
     */
    public final String branchName() {
        return branchName;
    }

    /**
     * <p>
     * The ID of the commit that is the parent of the commit you create. Not required if this is an empty repository.
     * </p>
     * 
     * @return The ID of the commit that is the parent of the commit you create. Not required if this is an empty
     *         repository.
     */
    public final String parentCommitId() {
        return parentCommitId;
    }

    /**
     * <p>
     * The name of the author who created the commit. This information is used as both the author and committer for the
     * commit.
     * </p>
     * 
     * @return The name of the author who created the commit. This information is used as both the author and committer
     *         for the commit.
     */
    public final String authorName() {
        return authorName;
    }

    /**
     * <p>
     * The email address of the person who created the commit.
     * </p>
     * 
     * @return The email address of the person who created the commit.
     */
    public final String email() {
        return email;
    }

    /**
     * <p>
     * The commit message you want to include in the commit. Commit messages are limited to 256 KB. If no message is
     * specified, a default message is used.
     * </p>
     * 
     * @return The commit message you want to include in the commit. Commit messages are limited to 256 KB. If no
     *         message is specified, a default message is used.
     */
    public final String commitMessage() {
        return commitMessage;
    }

    /**
     * <p>
     * If the commit contains deletions, whether to keep a folder or folder structure if the changes leave the folders
     * empty. If true, a ..gitkeep file is created for empty folders. The default is false.
     * </p>
     * 
     * @return If the commit contains deletions, whether to keep a folder or folder structure if the changes leave the
     *         folders empty. If true, a ..gitkeep file is created for empty folders. The default is false.
     */
    public final Boolean keepEmptyFolders() {
        return keepEmptyFolders;
    }

    /**
     * For responses, this returns true if the service returned a value for the PutFiles property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasPutFiles() {
        return putFiles != null && !(putFiles instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The files to add or update in this commit.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPutFiles} method.
     * </p>
     * 
     * @return The files to add or update in this commit.
     */
    public final List<PutFileEntry> putFiles() {
        return putFiles;
    }

    /**
     * For responses, this returns true if the service returned a value for the DeleteFiles property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasDeleteFiles() {
        return deleteFiles != null && !(deleteFiles instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The files to delete in this commit. These files still exist in earlier commits.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDeleteFiles} method.
     * </p>
     * 
     * @return The files to delete in this commit. These files still exist in earlier commits.
     */
    public final List<DeleteFileEntry> deleteFiles() {
        return deleteFiles;
    }

    /**
     * For responses, this returns true if the service returned a value for the SetFileModes property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasSetFileModes() {
        return setFileModes != null && !(setFileModes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The file modes to update for files in this commit.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSetFileModes} method.
     * </p>
     * 
     * @return The file modes to update for files in this commit.
     */
    public final List<SetFileModeEntry> setFileModes() {
        return setFileModes;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(repositoryName());
        hashCode = 31 * hashCode + Objects.hashCode(branchName());
        hashCode = 31 * hashCode + Objects.hashCode(parentCommitId());
        hashCode = 31 * hashCode + Objects.hashCode(authorName());
        hashCode = 31 * hashCode + Objects.hashCode(email());
        hashCode = 31 * hashCode + Objects.hashCode(commitMessage());
        hashCode = 31 * hashCode + Objects.hashCode(keepEmptyFolders());
        hashCode = 31 * hashCode + Objects.hashCode(hasPutFiles() ? putFiles() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasDeleteFiles() ? deleteFiles() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasSetFileModes() ? setFileModes() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateCommitRequest)) {
            return false;
        }
        CreateCommitRequest other = (CreateCommitRequest) obj;
        return Objects.equals(repositoryName(), other.repositoryName()) && Objects.equals(branchName(), other.branchName())
                && Objects.equals(parentCommitId(), other.parentCommitId()) && Objects.equals(authorName(), other.authorName())
                && Objects.equals(email(), other.email()) && Objects.equals(commitMessage(), other.commitMessage())
                && Objects.equals(keepEmptyFolders(), other.keepEmptyFolders()) && hasPutFiles() == other.hasPutFiles()
                && Objects.equals(putFiles(), other.putFiles()) && hasDeleteFiles() == other.hasDeleteFiles()
                && Objects.equals(deleteFiles(), other.deleteFiles()) && hasSetFileModes() == other.hasSetFileModes()
                && Objects.equals(setFileModes(), other.setFileModes());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateCommitRequest").add("RepositoryName", repositoryName()).add("BranchName", branchName())
                .add("ParentCommitId", parentCommitId()).add("AuthorName", authorName()).add("Email", email())
                .add("CommitMessage", commitMessage()).add("KeepEmptyFolders", keepEmptyFolders())
                .add("PutFiles", hasPutFiles() ? putFiles() : null).add("DeleteFiles", hasDeleteFiles() ? deleteFiles() : null)
                .add("SetFileModes", hasSetFileModes() ? setFileModes() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "repositoryName":
            return Optional.ofNullable(clazz.cast(repositoryName()));
        case "branchName":
            return Optional.ofNullable(clazz.cast(branchName()));
        case "parentCommitId":
            return Optional.ofNullable(clazz.cast(parentCommitId()));
        case "authorName":
            return Optional.ofNullable(clazz.cast(authorName()));
        case "email":
            return Optional.ofNullable(clazz.cast(email()));
        case "commitMessage":
            return Optional.ofNullable(clazz.cast(commitMessage()));
        case "keepEmptyFolders":
            return Optional.ofNullable(clazz.cast(keepEmptyFolders()));
        case "putFiles":
            return Optional.ofNullable(clazz.cast(putFiles()));
        case "deleteFiles":
            return Optional.ofNullable(clazz.cast(deleteFiles()));
        case "setFileModes":
            return Optional.ofNullable(clazz.cast(setFileModes()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("repositoryName", REPOSITORY_NAME_FIELD);
        map.put("branchName", BRANCH_NAME_FIELD);
        map.put("parentCommitId", PARENT_COMMIT_ID_FIELD);
        map.put("authorName", AUTHOR_NAME_FIELD);
        map.put("email", EMAIL_FIELD);
        map.put("commitMessage", COMMIT_MESSAGE_FIELD);
        map.put("keepEmptyFolders", KEEP_EMPTY_FOLDERS_FIELD);
        map.put("putFiles", PUT_FILES_FIELD);
        map.put("deleteFiles", DELETE_FILES_FIELD);
        map.put("setFileModes", SET_FILE_MODES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<CreateCommitRequest, T> g) {
        return obj -> g.apply((CreateCommitRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends CodeCommitRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateCommitRequest> {
        /**
         * <p>
         * The name of the repository where you create the commit.
         * </p>
         * 
         * @param repositoryName
         *        The name of the repository where you create the commit.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder repositoryName(String repositoryName);

        /**
         * <p>
         * The name of the branch where you create the commit.
         * </p>
         * 
         * @param branchName
         *        The name of the branch where you create the commit.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder branchName(String branchName);

        /**
         * <p>
         * The ID of the commit that is the parent of the commit you create. Not required if this is an empty
         * repository.
         * </p>
         * 
         * @param parentCommitId
         *        The ID of the commit that is the parent of the commit you create. Not required if this is an empty
         *        repository.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parentCommitId(String parentCommitId);

        /**
         * <p>
         * The name of the author who created the commit. This information is used as both the author and committer for
         * the commit.
         * </p>
         * 
         * @param authorName
         *        The name of the author who created the commit. This information is used as both the author and
         *        committer for the commit.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder authorName(String authorName);

        /**
         * <p>
         * The email address of the person who created the commit.
         * </p>
         * 
         * @param email
         *        The email address of the person who created the commit.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder email(String email);

        /**
         * <p>
         * The commit message you want to include in the commit. Commit messages are limited to 256 KB. If no message is
         * specified, a default message is used.
         * </p>
         * 
         * @param commitMessage
         *        The commit message you want to include in the commit. Commit messages are limited to 256 KB. If no
         *        message is specified, a default message is used.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder commitMessage(String commitMessage);

        /**
         * <p>
         * If the commit contains deletions, whether to keep a folder or folder structure if the changes leave the
         * folders empty. If true, a ..gitkeep file is created for empty folders. The default is false.
         * </p>
         * 
         * @param keepEmptyFolders
         *        If the commit contains deletions, whether to keep a folder or folder structure if the changes leave
         *        the folders empty. If true, a ..gitkeep file is created for empty folders. The default is false.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder keepEmptyFolders(Boolean keepEmptyFolders);

        /**
         * <p>
         * The files to add or update in this commit.
         * </p>
         * 
         * @param putFiles
         *        The files to add or update in this commit.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder putFiles(Collection<PutFileEntry> putFiles);

        /**
         * <p>
         * The files to add or update in this commit.
         * </p>
         * 
         * @param putFiles
         *        The files to add or update in this commit.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder putFiles(PutFileEntry... putFiles);

        /**
         * <p>
         * The files to add or update in this commit.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.codecommit.model.PutFileEntry.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.codecommit.model.PutFileEntry#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.codecommit.model.PutFileEntry.Builder#build()} is called immediately
         * and its result is passed to {@link #putFiles(List<PutFileEntry>)}.
         * 
         * @param putFiles
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.codecommit.model.PutFileEntry.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #putFiles(java.util.Collection<PutFileEntry>)
         */
        Builder putFiles(Consumer<PutFileEntry.Builder>... putFiles);

        /**
         * <p>
         * The files to delete in this commit. These files still exist in earlier commits.
         * </p>
         * 
         * @param deleteFiles
         *        The files to delete in this commit. These files still exist in earlier commits.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deleteFiles(Collection<DeleteFileEntry> deleteFiles);

        /**
         * <p>
         * The files to delete in this commit. These files still exist in earlier commits.
         * </p>
         * 
         * @param deleteFiles
         *        The files to delete in this commit. These files still exist in earlier commits.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deleteFiles(DeleteFileEntry... deleteFiles);

        /**
         * <p>
         * The files to delete in this commit. These files still exist in earlier commits.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.codecommit.model.DeleteFileEntry.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.codecommit.model.DeleteFileEntry#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.codecommit.model.DeleteFileEntry.Builder#build()} is called
         * immediately and its result is passed to {@link #deleteFiles(List<DeleteFileEntry>)}.
         * 
         * @param deleteFiles
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.codecommit.model.DeleteFileEntry.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #deleteFiles(java.util.Collection<DeleteFileEntry>)
         */
        Builder deleteFiles(Consumer<DeleteFileEntry.Builder>... deleteFiles);

        /**
         * <p>
         * The file modes to update for files in this commit.
         * </p>
         * 
         * @param setFileModes
         *        The file modes to update for files in this commit.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder setFileModes(Collection<SetFileModeEntry> setFileModes);

        /**
         * <p>
         * The file modes to update for files in this commit.
         * </p>
         * 
         * @param setFileModes
         *        The file modes to update for files in this commit.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder setFileModes(SetFileModeEntry... setFileModes);

        /**
         * <p>
         * The file modes to update for files in this commit.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.codecommit.model.SetFileModeEntry.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.codecommit.model.SetFileModeEntry#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.codecommit.model.SetFileModeEntry.Builder#build()} is called
         * immediately and its result is passed to {@link #setFileModes(List<SetFileModeEntry>)}.
         * 
         * @param setFileModes
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.codecommit.model.SetFileModeEntry.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #setFileModes(java.util.Collection<SetFileModeEntry>)
         */
        Builder setFileModes(Consumer<SetFileModeEntry.Builder>... setFileModes);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends CodeCommitRequest.BuilderImpl implements Builder {
        private String repositoryName;

        private String branchName;

        private String parentCommitId;

        private String authorName;

        private String email;

        private String commitMessage;

        private Boolean keepEmptyFolders;

        private List<PutFileEntry> putFiles = DefaultSdkAutoConstructList.getInstance();

        private List<DeleteFileEntry> deleteFiles = DefaultSdkAutoConstructList.getInstance();

        private List<SetFileModeEntry> setFileModes = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CreateCommitRequest model) {
            super(model);
            repositoryName(model.repositoryName);
            branchName(model.branchName);
            parentCommitId(model.parentCommitId);
            authorName(model.authorName);
            email(model.email);
            commitMessage(model.commitMessage);
            keepEmptyFolders(model.keepEmptyFolders);
            putFiles(model.putFiles);
            deleteFiles(model.deleteFiles);
            setFileModes(model.setFileModes);
        }

        public final String getRepositoryName() {
            return repositoryName;
        }

        public final void setRepositoryName(String repositoryName) {
            this.repositoryName = repositoryName;
        }

        @Override
        public final Builder repositoryName(String repositoryName) {
            this.repositoryName = repositoryName;
            return this;
        }

        public final String getBranchName() {
            return branchName;
        }

        public final void setBranchName(String branchName) {
            this.branchName = branchName;
        }

        @Override
        public final Builder branchName(String branchName) {
            this.branchName = branchName;
            return this;
        }

        public final String getParentCommitId() {
            return parentCommitId;
        }

        public final void setParentCommitId(String parentCommitId) {
            this.parentCommitId = parentCommitId;
        }

        @Override
        public final Builder parentCommitId(String parentCommitId) {
            this.parentCommitId = parentCommitId;
            return this;
        }

        public final String getAuthorName() {
            return authorName;
        }

        public final void setAuthorName(String authorName) {
            this.authorName = authorName;
        }

        @Override
        public final Builder authorName(String authorName) {
            this.authorName = authorName;
            return this;
        }

        public final String getEmail() {
            return email;
        }

        public final void setEmail(String email) {
            this.email = email;
        }

        @Override
        public final Builder email(String email) {
            this.email = email;
            return this;
        }

        public final String getCommitMessage() {
            return commitMessage;
        }

        public final void setCommitMessage(String commitMessage) {
            this.commitMessage = commitMessage;
        }

        @Override
        public final Builder commitMessage(String commitMessage) {
            this.commitMessage = commitMessage;
            return this;
        }

        public final Boolean getKeepEmptyFolders() {
            return keepEmptyFolders;
        }

        public final void setKeepEmptyFolders(Boolean keepEmptyFolders) {
            this.keepEmptyFolders = keepEmptyFolders;
        }

        @Override
        public final Builder keepEmptyFolders(Boolean keepEmptyFolders) {
            this.keepEmptyFolders = keepEmptyFolders;
            return this;
        }

        public final List<PutFileEntry.Builder> getPutFiles() {
            List<PutFileEntry.Builder> result = PutFileEntriesCopier.copyToBuilder(this.putFiles);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setPutFiles(Collection<PutFileEntry.BuilderImpl> putFiles) {
            this.putFiles = PutFileEntriesCopier.copyFromBuilder(putFiles);
        }

        @Override
        public final Builder putFiles(Collection<PutFileEntry> putFiles) {
            this.putFiles = PutFileEntriesCopier.copy(putFiles);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder putFiles(PutFileEntry... putFiles) {
            putFiles(Arrays.asList(putFiles));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder putFiles(Consumer<PutFileEntry.Builder>... putFiles) {
            putFiles(Stream.of(putFiles).map(c -> PutFileEntry.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final List<DeleteFileEntry.Builder> getDeleteFiles() {
            List<DeleteFileEntry.Builder> result = DeleteFileEntriesCopier.copyToBuilder(this.deleteFiles);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setDeleteFiles(Collection<DeleteFileEntry.BuilderImpl> deleteFiles) {
            this.deleteFiles = DeleteFileEntriesCopier.copyFromBuilder(deleteFiles);
        }

        @Override
        public final Builder deleteFiles(Collection<DeleteFileEntry> deleteFiles) {
            this.deleteFiles = DeleteFileEntriesCopier.copy(deleteFiles);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder deleteFiles(DeleteFileEntry... deleteFiles) {
            deleteFiles(Arrays.asList(deleteFiles));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder deleteFiles(Consumer<DeleteFileEntry.Builder>... deleteFiles) {
            deleteFiles(Stream.of(deleteFiles).map(c -> DeleteFileEntry.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final List<SetFileModeEntry.Builder> getSetFileModes() {
            List<SetFileModeEntry.Builder> result = SetFileModeEntriesCopier.copyToBuilder(this.setFileModes);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setSetFileModes(Collection<SetFileModeEntry.BuilderImpl> setFileModes) {
            this.setFileModes = SetFileModeEntriesCopier.copyFromBuilder(setFileModes);
        }

        @Override
        @Transient
        public final Builder setFileModes(Collection<SetFileModeEntry> setFileModes) {
            this.setFileModes = SetFileModeEntriesCopier.copy(setFileModes);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder setFileModes(SetFileModeEntry... setFileModes) {
            setFileModes(Arrays.asList(setFileModes));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder setFileModes(Consumer<SetFileModeEntry.Builder>... setFileModes) {
            setFileModes(Stream.of(setFileModes).map(c -> SetFileModeEntry.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateCommitRequest build() {
            return new CreateCommitRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
