/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudwatch.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * By default, a metric stream always sends the <code>MAX</code>, <code>MIN</code>, <code>SUM</code>, and
 * <code>SAMPLECOUNT</code> statistics for each metric that is streamed. This structure contains information for one
 * metric that includes additional statistics in the stream. For more information about statistics, see CloudWatch,
 * listed in <a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/Statistics-definitions.html.html">
 * CloudWatch statistics definitions</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class MetricStreamStatisticsConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<MetricStreamStatisticsConfiguration.Builder, MetricStreamStatisticsConfiguration> {
    private static final SdkField<List<MetricStreamStatisticsMetric>> INCLUDE_METRICS_FIELD = SdkField
            .<List<MetricStreamStatisticsMetric>> builder(MarshallingType.LIST)
            .memberName("IncludeMetrics")
            .getter(getter(MetricStreamStatisticsConfiguration::includeMetrics))
            .setter(setter(Builder::includeMetrics))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IncludeMetrics").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<MetricStreamStatisticsMetric> builder(MarshallingType.SDK_POJO)
                                            .constructor(MetricStreamStatisticsMetric::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> ADDITIONAL_STATISTICS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("AdditionalStatistics")
            .getter(getter(MetricStreamStatisticsConfiguration::additionalStatistics))
            .setter(setter(Builder::additionalStatistics))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AdditionalStatistics").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(INCLUDE_METRICS_FIELD,
            ADDITIONAL_STATISTICS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final List<MetricStreamStatisticsMetric> includeMetrics;

    private final List<String> additionalStatistics;

    private MetricStreamStatisticsConfiguration(BuilderImpl builder) {
        this.includeMetrics = builder.includeMetrics;
        this.additionalStatistics = builder.additionalStatistics;
    }

    /**
     * For responses, this returns true if the service returned a value for the IncludeMetrics property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasIncludeMetrics() {
        return includeMetrics != null && !(includeMetrics instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of metric name and namespace pairs that stream the additional statistics listed in the value of the
     * <code>AdditionalStatistics</code> parameter. There can be as many as 100 pairs in the array.
     * </p>
     * <p>
     * All metrics that match the combination of metric name and namespace will be streamed with the additional
     * statistics, no matter their dimensions.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasIncludeMetrics} method.
     * </p>
     * 
     * @return An array of metric name and namespace pairs that stream the additional statistics listed in the value of
     *         the <code>AdditionalStatistics</code> parameter. There can be as many as 100 pairs in the array.</p>
     *         <p>
     *         All metrics that match the combination of metric name and namespace will be streamed with the additional
     *         statistics, no matter their dimensions.
     */
    public final List<MetricStreamStatisticsMetric> includeMetrics() {
        return includeMetrics;
    }

    /**
     * For responses, this returns true if the service returned a value for the AdditionalStatistics property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAdditionalStatistics() {
        return additionalStatistics != null && !(additionalStatistics instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of additional statistics that are to be streamed for the metrics listed in the
     * <code>IncludeMetrics</code> array in this structure. This list can include as many as 20 statistics.
     * </p>
     * <p>
     * If the <code>OutputFormat</code> for the stream is <code>opentelemetry1.0</code> or <code>opentelemetry0.7</code>
     * , the only valid values are <code>p<i>??</i> </code> percentile statistics such as <code>p90</code>,
     * <code>p99</code> and so on.
     * </p>
     * <p>
     * If the <code>OutputFormat</code> for the stream is <code>json</code>, the valid values include the abbreviations
     * for all of the statistics listed in <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/Statistics-definitions.html.html">
     * CloudWatch statistics definitions</a>. For example, this includes <code>tm98, </code> <code>wm90</code>,
     * <code>PR(:300)</code>, and so on.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAdditionalStatistics} method.
     * </p>
     * 
     * @return The list of additional statistics that are to be streamed for the metrics listed in the
     *         <code>IncludeMetrics</code> array in this structure. This list can include as many as 20 statistics.</p>
     *         <p>
     *         If the <code>OutputFormat</code> for the stream is <code>opentelemetry1.0</code> or
     *         <code>opentelemetry0.7</code>, the only valid values are <code>p<i>??</i> </code> percentile statistics
     *         such as <code>p90</code>, <code>p99</code> and so on.
     *         </p>
     *         <p>
     *         If the <code>OutputFormat</code> for the stream is <code>json</code>, the valid values include the
     *         abbreviations for all of the statistics listed in <a
     *         href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/Statistics-definitions.html.html">
     *         CloudWatch statistics definitions</a>. For example, this includes <code>tm98, </code> <code>wm90</code>,
     *         <code>PR(:300)</code>, and so on.
     */
    public final List<String> additionalStatistics() {
        return additionalStatistics;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasIncludeMetrics() ? includeMetrics() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasAdditionalStatistics() ? additionalStatistics() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof MetricStreamStatisticsConfiguration)) {
            return false;
        }
        MetricStreamStatisticsConfiguration other = (MetricStreamStatisticsConfiguration) obj;
        return hasIncludeMetrics() == other.hasIncludeMetrics() && Objects.equals(includeMetrics(), other.includeMetrics())
                && hasAdditionalStatistics() == other.hasAdditionalStatistics()
                && Objects.equals(additionalStatistics(), other.additionalStatistics());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("MetricStreamStatisticsConfiguration")
                .add("IncludeMetrics", hasIncludeMetrics() ? includeMetrics() : null)
                .add("AdditionalStatistics", hasAdditionalStatistics() ? additionalStatistics() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "IncludeMetrics":
            return Optional.ofNullable(clazz.cast(includeMetrics()));
        case "AdditionalStatistics":
            return Optional.ofNullable(clazz.cast(additionalStatistics()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("IncludeMetrics", INCLUDE_METRICS_FIELD);
        map.put("AdditionalStatistics", ADDITIONAL_STATISTICS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<MetricStreamStatisticsConfiguration, T> g) {
        return obj -> g.apply((MetricStreamStatisticsConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, MetricStreamStatisticsConfiguration> {
        /**
         * <p>
         * An array of metric name and namespace pairs that stream the additional statistics listed in the value of the
         * <code>AdditionalStatistics</code> parameter. There can be as many as 100 pairs in the array.
         * </p>
         * <p>
         * All metrics that match the combination of metric name and namespace will be streamed with the additional
         * statistics, no matter their dimensions.
         * </p>
         * 
         * @param includeMetrics
         *        An array of metric name and namespace pairs that stream the additional statistics listed in the value
         *        of the <code>AdditionalStatistics</code> parameter. There can be as many as 100 pairs in the
         *        array.</p>
         *        <p>
         *        All metrics that match the combination of metric name and namespace will be streamed with the
         *        additional statistics, no matter their dimensions.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder includeMetrics(Collection<MetricStreamStatisticsMetric> includeMetrics);

        /**
         * <p>
         * An array of metric name and namespace pairs that stream the additional statistics listed in the value of the
         * <code>AdditionalStatistics</code> parameter. There can be as many as 100 pairs in the array.
         * </p>
         * <p>
         * All metrics that match the combination of metric name and namespace will be streamed with the additional
         * statistics, no matter their dimensions.
         * </p>
         * 
         * @param includeMetrics
         *        An array of metric name and namespace pairs that stream the additional statistics listed in the value
         *        of the <code>AdditionalStatistics</code> parameter. There can be as many as 100 pairs in the
         *        array.</p>
         *        <p>
         *        All metrics that match the combination of metric name and namespace will be streamed with the
         *        additional statistics, no matter their dimensions.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder includeMetrics(MetricStreamStatisticsMetric... includeMetrics);

        /**
         * <p>
         * An array of metric name and namespace pairs that stream the additional statistics listed in the value of the
         * <code>AdditionalStatistics</code> parameter. There can be as many as 100 pairs in the array.
         * </p>
         * <p>
         * All metrics that match the combination of metric name and namespace will be streamed with the additional
         * statistics, no matter their dimensions.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.cloudwatch.model.MetricStreamStatisticsMetric.Builder} avoiding the
         * need to create one manually via
         * {@link software.amazon.awssdk.services.cloudwatch.model.MetricStreamStatisticsMetric#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.cloudwatch.model.MetricStreamStatisticsMetric.Builder#build()} is
         * called immediately and its result is passed to {@link #includeMetrics(List<MetricStreamStatisticsMetric>)}.
         * 
         * @param includeMetrics
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.cloudwatch.model.MetricStreamStatisticsMetric.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #includeMetrics(java.util.Collection<MetricStreamStatisticsMetric>)
         */
        Builder includeMetrics(Consumer<MetricStreamStatisticsMetric.Builder>... includeMetrics);

        /**
         * <p>
         * The list of additional statistics that are to be streamed for the metrics listed in the
         * <code>IncludeMetrics</code> array in this structure. This list can include as many as 20 statistics.
         * </p>
         * <p>
         * If the <code>OutputFormat</code> for the stream is <code>opentelemetry1.0</code> or
         * <code>opentelemetry0.7</code>, the only valid values are <code>p<i>??</i> </code> percentile statistics such
         * as <code>p90</code>, <code>p99</code> and so on.
         * </p>
         * <p>
         * If the <code>OutputFormat</code> for the stream is <code>json</code>, the valid values include the
         * abbreviations for all of the statistics listed in <a
         * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/Statistics-definitions.html.html">
         * CloudWatch statistics definitions</a>. For example, this includes <code>tm98, </code> <code>wm90</code>,
         * <code>PR(:300)</code>, and so on.
         * </p>
         * 
         * @param additionalStatistics
         *        The list of additional statistics that are to be streamed for the metrics listed in the
         *        <code>IncludeMetrics</code> array in this structure. This list can include as many as 20
         *        statistics.</p>
         *        <p>
         *        If the <code>OutputFormat</code> for the stream is <code>opentelemetry1.0</code> or
         *        <code>opentelemetry0.7</code>, the only valid values are <code>p<i>??</i> </code> percentile
         *        statistics such as <code>p90</code>, <code>p99</code> and so on.
         *        </p>
         *        <p>
         *        If the <code>OutputFormat</code> for the stream is <code>json</code>, the valid values include the
         *        abbreviations for all of the statistics listed in <a href=
         *        "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/Statistics-definitions.html.html">
         *        CloudWatch statistics definitions</a>. For example, this includes <code>tm98, </code>
         *        <code>wm90</code>, <code>PR(:300)</code>, and so on.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder additionalStatistics(Collection<String> additionalStatistics);

        /**
         * <p>
         * The list of additional statistics that are to be streamed for the metrics listed in the
         * <code>IncludeMetrics</code> array in this structure. This list can include as many as 20 statistics.
         * </p>
         * <p>
         * If the <code>OutputFormat</code> for the stream is <code>opentelemetry1.0</code> or
         * <code>opentelemetry0.7</code>, the only valid values are <code>p<i>??</i> </code> percentile statistics such
         * as <code>p90</code>, <code>p99</code> and so on.
         * </p>
         * <p>
         * If the <code>OutputFormat</code> for the stream is <code>json</code>, the valid values include the
         * abbreviations for all of the statistics listed in <a
         * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/Statistics-definitions.html.html">
         * CloudWatch statistics definitions</a>. For example, this includes <code>tm98, </code> <code>wm90</code>,
         * <code>PR(:300)</code>, and so on.
         * </p>
         * 
         * @param additionalStatistics
         *        The list of additional statistics that are to be streamed for the metrics listed in the
         *        <code>IncludeMetrics</code> array in this structure. This list can include as many as 20
         *        statistics.</p>
         *        <p>
         *        If the <code>OutputFormat</code> for the stream is <code>opentelemetry1.0</code> or
         *        <code>opentelemetry0.7</code>, the only valid values are <code>p<i>??</i> </code> percentile
         *        statistics such as <code>p90</code>, <code>p99</code> and so on.
         *        </p>
         *        <p>
         *        If the <code>OutputFormat</code> for the stream is <code>json</code>, the valid values include the
         *        abbreviations for all of the statistics listed in <a href=
         *        "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/Statistics-definitions.html.html">
         *        CloudWatch statistics definitions</a>. For example, this includes <code>tm98, </code>
         *        <code>wm90</code>, <code>PR(:300)</code>, and so on.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder additionalStatistics(String... additionalStatistics);
    }

    static final class BuilderImpl implements Builder {
        private List<MetricStreamStatisticsMetric> includeMetrics = DefaultSdkAutoConstructList.getInstance();

        private List<String> additionalStatistics = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(MetricStreamStatisticsConfiguration model) {
            includeMetrics(model.includeMetrics);
            additionalStatistics(model.additionalStatistics);
        }

        public final List<MetricStreamStatisticsMetric.Builder> getIncludeMetrics() {
            List<MetricStreamStatisticsMetric.Builder> result = MetricStreamStatisticsIncludeMetricsCopier
                    .copyToBuilder(this.includeMetrics);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setIncludeMetrics(Collection<MetricStreamStatisticsMetric.BuilderImpl> includeMetrics) {
            this.includeMetrics = MetricStreamStatisticsIncludeMetricsCopier.copyFromBuilder(includeMetrics);
        }

        @Override
        public final Builder includeMetrics(Collection<MetricStreamStatisticsMetric> includeMetrics) {
            this.includeMetrics = MetricStreamStatisticsIncludeMetricsCopier.copy(includeMetrics);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder includeMetrics(MetricStreamStatisticsMetric... includeMetrics) {
            includeMetrics(Arrays.asList(includeMetrics));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder includeMetrics(Consumer<MetricStreamStatisticsMetric.Builder>... includeMetrics) {
            includeMetrics(Stream.of(includeMetrics).map(c -> MetricStreamStatisticsMetric.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Collection<String> getAdditionalStatistics() {
            if (additionalStatistics instanceof SdkAutoConstructList) {
                return null;
            }
            return additionalStatistics;
        }

        public final void setAdditionalStatistics(Collection<String> additionalStatistics) {
            this.additionalStatistics = MetricStreamStatisticsAdditionalStatisticsCopier.copy(additionalStatistics);
        }

        @Override
        public final Builder additionalStatistics(Collection<String> additionalStatistics) {
            this.additionalStatistics = MetricStreamStatisticsAdditionalStatisticsCopier.copy(additionalStatistics);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder additionalStatistics(String... additionalStatistics) {
            additionalStatistics(Arrays.asList(additionalStatistics));
            return this;
        }

        @Override
        public MetricStreamStatisticsConfiguration build() {
            return new MetricStreamStatisticsConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
