/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudwatch.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * This structure contains a metric namespace and optionally, a list of metric names, to either include in a metric
 * stream or exclude from a metric stream.
 * </p>
 * <p>
 * A metric stream's filters can include up to 1000 total names. This limit applies to the sum of namespace names and
 * metric names in the filters. For example, this could include 10 metric namespace filters with 99 metrics each, or 20
 * namespace filters with 49 metrics specified in each filter.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class MetricStreamFilter implements SdkPojo, Serializable,
        ToCopyableBuilder<MetricStreamFilter.Builder, MetricStreamFilter> {
    private static final SdkField<String> NAMESPACE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Namespace").getter(getter(MetricStreamFilter::namespace)).setter(setter(Builder::namespace))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Namespace").build()).build();

    private static final SdkField<List<String>> METRIC_NAMES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("MetricNames")
            .getter(getter(MetricStreamFilter::metricNames))
            .setter(setter(Builder::metricNames))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MetricNames").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAMESPACE_FIELD,
            METRIC_NAMES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String namespace;

    private final List<String> metricNames;

    private MetricStreamFilter(BuilderImpl builder) {
        this.namespace = builder.namespace;
        this.metricNames = builder.metricNames;
    }

    /**
     * <p>
     * The name of the metric namespace for this filter.
     * </p>
     * <p>
     * The namespace can contain only ASCII printable characters (ASCII range 32 through 126). It must contain at least
     * one non-whitespace character.
     * </p>
     * 
     * @return The name of the metric namespace for this filter.</p>
     *         <p>
     *         The namespace can contain only ASCII printable characters (ASCII range 32 through 126). It must contain
     *         at least one non-whitespace character.
     */
    public final String namespace() {
        return namespace;
    }

    /**
     * For responses, this returns true if the service returned a value for the MetricNames property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasMetricNames() {
        return metricNames != null && !(metricNames instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The names of the metrics to either include or exclude from the metric stream.
     * </p>
     * <p>
     * If you omit this parameter, all metrics in the namespace are included or excluded, depending on whether this
     * filter is specified as an exclude filter or an include filter.
     * </p>
     * <p>
     * Each metric name can contain only ASCII printable characters (ASCII range 32 through 126). Each metric name must
     * contain at least one non-whitespace character.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasMetricNames} method.
     * </p>
     * 
     * @return The names of the metrics to either include or exclude from the metric stream. </p>
     *         <p>
     *         If you omit this parameter, all metrics in the namespace are included or excluded, depending on whether
     *         this filter is specified as an exclude filter or an include filter.
     *         </p>
     *         <p>
     *         Each metric name can contain only ASCII printable characters (ASCII range 32 through 126). Each metric
     *         name must contain at least one non-whitespace character.
     */
    public final List<String> metricNames() {
        return metricNames;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(namespace());
        hashCode = 31 * hashCode + Objects.hashCode(hasMetricNames() ? metricNames() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof MetricStreamFilter)) {
            return false;
        }
        MetricStreamFilter other = (MetricStreamFilter) obj;
        return Objects.equals(namespace(), other.namespace()) && hasMetricNames() == other.hasMetricNames()
                && Objects.equals(metricNames(), other.metricNames());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("MetricStreamFilter").add("Namespace", namespace())
                .add("MetricNames", hasMetricNames() ? metricNames() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Namespace":
            return Optional.ofNullable(clazz.cast(namespace()));
        case "MetricNames":
            return Optional.ofNullable(clazz.cast(metricNames()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Namespace", NAMESPACE_FIELD);
        map.put("MetricNames", METRIC_NAMES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<MetricStreamFilter, T> g) {
        return obj -> g.apply((MetricStreamFilter) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, MetricStreamFilter> {
        /**
         * <p>
         * The name of the metric namespace for this filter.
         * </p>
         * <p>
         * The namespace can contain only ASCII printable characters (ASCII range 32 through 126). It must contain at
         * least one non-whitespace character.
         * </p>
         * 
         * @param namespace
         *        The name of the metric namespace for this filter.</p>
         *        <p>
         *        The namespace can contain only ASCII printable characters (ASCII range 32 through 126). It must
         *        contain at least one non-whitespace character.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder namespace(String namespace);

        /**
         * <p>
         * The names of the metrics to either include or exclude from the metric stream.
         * </p>
         * <p>
         * If you omit this parameter, all metrics in the namespace are included or excluded, depending on whether this
         * filter is specified as an exclude filter or an include filter.
         * </p>
         * <p>
         * Each metric name can contain only ASCII printable characters (ASCII range 32 through 126). Each metric name
         * must contain at least one non-whitespace character.
         * </p>
         * 
         * @param metricNames
         *        The names of the metrics to either include or exclude from the metric stream. </p>
         *        <p>
         *        If you omit this parameter, all metrics in the namespace are included or excluded, depending on
         *        whether this filter is specified as an exclude filter or an include filter.
         *        </p>
         *        <p>
         *        Each metric name can contain only ASCII printable characters (ASCII range 32 through 126). Each metric
         *        name must contain at least one non-whitespace character.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricNames(Collection<String> metricNames);

        /**
         * <p>
         * The names of the metrics to either include or exclude from the metric stream.
         * </p>
         * <p>
         * If you omit this parameter, all metrics in the namespace are included or excluded, depending on whether this
         * filter is specified as an exclude filter or an include filter.
         * </p>
         * <p>
         * Each metric name can contain only ASCII printable characters (ASCII range 32 through 126). Each metric name
         * must contain at least one non-whitespace character.
         * </p>
         * 
         * @param metricNames
         *        The names of the metrics to either include or exclude from the metric stream. </p>
         *        <p>
         *        If you omit this parameter, all metrics in the namespace are included or excluded, depending on
         *        whether this filter is specified as an exclude filter or an include filter.
         *        </p>
         *        <p>
         *        Each metric name can contain only ASCII printable characters (ASCII range 32 through 126). Each metric
         *        name must contain at least one non-whitespace character.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricNames(String... metricNames);
    }

    static final class BuilderImpl implements Builder {
        private String namespace;

        private List<String> metricNames = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(MetricStreamFilter model) {
            namespace(model.namespace);
            metricNames(model.metricNames);
        }

        public final String getNamespace() {
            return namespace;
        }

        public final void setNamespace(String namespace) {
            this.namespace = namespace;
        }

        @Override
        public final Builder namespace(String namespace) {
            this.namespace = namespace;
            return this;
        }

        public final Collection<String> getMetricNames() {
            if (metricNames instanceof SdkAutoConstructList) {
                return null;
            }
            return metricNames;
        }

        public final void setMetricNames(Collection<String> metricNames) {
            this.metricNames = MetricStreamFilterMetricNamesCopier.copy(metricNames);
        }

        @Override
        public final Builder metricNames(Collection<String> metricNames) {
            this.metricNames = MetricStreamFilterMetricNamesCopier.copy(metricNames);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder metricNames(String... metricNames) {
            metricNames(Arrays.asList(metricNames));
            return this;
        }

        @Override
        public MetricStreamFilter build() {
            return new MetricStreamFilter(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
