/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudwatch.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * One of the unique contributors found by a Contributor Insights rule. If the rule contains multiple keys, then a
 * unique contributor is a unique combination of values from all the keys in the rule.
 * </p>
 * <p>
 * If the rule contains a single key, then each unique contributor is each unique value for this key.
 * </p>
 * <p>
 * For more information, see <a
 * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_GetInsightRuleReport.html"
 * >GetInsightRuleReport</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class InsightRuleContributor implements SdkPojo, Serializable,
        ToCopyableBuilder<InsightRuleContributor.Builder, InsightRuleContributor> {
    private static final SdkField<List<String>> KEYS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(InsightRuleContributor::keys))
            .setter(setter(Builder::keys))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Keys").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Double> APPROXIMATE_AGGREGATE_VALUE_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .getter(getter(InsightRuleContributor::approximateAggregateValue)).setter(setter(Builder::approximateAggregateValue))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ApproximateAggregateValue").build())
            .build();

    private static final SdkField<List<InsightRuleContributorDatapoint>> DATAPOINTS_FIELD = SdkField
            .<List<InsightRuleContributorDatapoint>> builder(MarshallingType.LIST)
            .getter(getter(InsightRuleContributor::datapoints))
            .setter(setter(Builder::datapoints))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Datapoints").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<InsightRuleContributorDatapoint> builder(MarshallingType.SDK_POJO)
                                            .constructor(InsightRuleContributorDatapoint::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(KEYS_FIELD,
            APPROXIMATE_AGGREGATE_VALUE_FIELD, DATAPOINTS_FIELD));

    private static final long serialVersionUID = 1L;

    private final List<String> keys;

    private final Double approximateAggregateValue;

    private final List<InsightRuleContributorDatapoint> datapoints;

    private InsightRuleContributor(BuilderImpl builder) {
        this.keys = builder.keys;
        this.approximateAggregateValue = builder.approximateAggregateValue;
        this.datapoints = builder.datapoints;
    }

    /**
     * Returns true if the Keys property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasKeys() {
        return keys != null && !(keys instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * One of the log entry field keywords that is used to define contributors for this rule.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasKeys()} to see if a value was sent in this field.
     * </p>
     * 
     * @return One of the log entry field keywords that is used to define contributors for this rule.
     */
    public List<String> keys() {
        return keys;
    }

    /**
     * <p>
     * An approximation of the aggregate value that comes from this contributor.
     * </p>
     * 
     * @return An approximation of the aggregate value that comes from this contributor.
     */
    public Double approximateAggregateValue() {
        return approximateAggregateValue;
    }

    /**
     * Returns true if the Datapoints property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasDatapoints() {
        return datapoints != null && !(datapoints instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of the data points where this contributor is present. Only the data points when this contributor
     * appeared are included in the array.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasDatapoints()} to see if a value was sent in this field.
     * </p>
     * 
     * @return An array of the data points where this contributor is present. Only the data points when this contributor
     *         appeared are included in the array.
     */
    public List<InsightRuleContributorDatapoint> datapoints() {
        return datapoints;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(keys());
        hashCode = 31 * hashCode + Objects.hashCode(approximateAggregateValue());
        hashCode = 31 * hashCode + Objects.hashCode(datapoints());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof InsightRuleContributor)) {
            return false;
        }
        InsightRuleContributor other = (InsightRuleContributor) obj;
        return Objects.equals(keys(), other.keys())
                && Objects.equals(approximateAggregateValue(), other.approximateAggregateValue())
                && Objects.equals(datapoints(), other.datapoints());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("InsightRuleContributor").add("Keys", keys())
                .add("ApproximateAggregateValue", approximateAggregateValue()).add("Datapoints", datapoints()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Keys":
            return Optional.ofNullable(clazz.cast(keys()));
        case "ApproximateAggregateValue":
            return Optional.ofNullable(clazz.cast(approximateAggregateValue()));
        case "Datapoints":
            return Optional.ofNullable(clazz.cast(datapoints()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<InsightRuleContributor, T> g) {
        return obj -> g.apply((InsightRuleContributor) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, InsightRuleContributor> {
        /**
         * <p>
         * One of the log entry field keywords that is used to define contributors for this rule.
         * </p>
         * 
         * @param keys
         *        One of the log entry field keywords that is used to define contributors for this rule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder keys(Collection<String> keys);

        /**
         * <p>
         * One of the log entry field keywords that is used to define contributors for this rule.
         * </p>
         * 
         * @param keys
         *        One of the log entry field keywords that is used to define contributors for this rule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder keys(String... keys);

        /**
         * <p>
         * An approximation of the aggregate value that comes from this contributor.
         * </p>
         * 
         * @param approximateAggregateValue
         *        An approximation of the aggregate value that comes from this contributor.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder approximateAggregateValue(Double approximateAggregateValue);

        /**
         * <p>
         * An array of the data points where this contributor is present. Only the data points when this contributor
         * appeared are included in the array.
         * </p>
         * 
         * @param datapoints
         *        An array of the data points where this contributor is present. Only the data points when this
         *        contributor appeared are included in the array.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder datapoints(Collection<InsightRuleContributorDatapoint> datapoints);

        /**
         * <p>
         * An array of the data points where this contributor is present. Only the data points when this contributor
         * appeared are included in the array.
         * </p>
         * 
         * @param datapoints
         *        An array of the data points where this contributor is present. Only the data points when this
         *        contributor appeared are included in the array.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder datapoints(InsightRuleContributorDatapoint... datapoints);

        /**
         * <p>
         * An array of the data points where this contributor is present. Only the data points when this contributor
         * appeared are included in the array.
         * </p>
         * This is a convenience that creates an instance of the {@link List<InsightRuleContributorDatapoint>.Builder}
         * avoiding the need to create one manually via {@link List<InsightRuleContributorDatapoint>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<InsightRuleContributorDatapoint>.Builder#build()} is called
         * immediately and its result is passed to {@link #datapoints(List<InsightRuleContributorDatapoint>)}.
         * 
         * @param datapoints
         *        a consumer that will call methods on {@link List<InsightRuleContributorDatapoint>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #datapoints(List<InsightRuleContributorDatapoint>)
         */
        Builder datapoints(Consumer<InsightRuleContributorDatapoint.Builder>... datapoints);
    }

    static final class BuilderImpl implements Builder {
        private List<String> keys = DefaultSdkAutoConstructList.getInstance();

        private Double approximateAggregateValue;

        private List<InsightRuleContributorDatapoint> datapoints = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(InsightRuleContributor model) {
            keys(model.keys);
            approximateAggregateValue(model.approximateAggregateValue);
            datapoints(model.datapoints);
        }

        public final Collection<String> getKeys() {
            return keys;
        }

        @Override
        public final Builder keys(Collection<String> keys) {
            this.keys = InsightRuleContributorKeysCopier.copy(keys);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder keys(String... keys) {
            keys(Arrays.asList(keys));
            return this;
        }

        public final void setKeys(Collection<String> keys) {
            this.keys = InsightRuleContributorKeysCopier.copy(keys);
        }

        public final Double getApproximateAggregateValue() {
            return approximateAggregateValue;
        }

        @Override
        public final Builder approximateAggregateValue(Double approximateAggregateValue) {
            this.approximateAggregateValue = approximateAggregateValue;
            return this;
        }

        public final void setApproximateAggregateValue(Double approximateAggregateValue) {
            this.approximateAggregateValue = approximateAggregateValue;
        }

        public final Collection<InsightRuleContributorDatapoint.Builder> getDatapoints() {
            return datapoints != null ? datapoints.stream().map(InsightRuleContributorDatapoint::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder datapoints(Collection<InsightRuleContributorDatapoint> datapoints) {
            this.datapoints = InsightRuleContributorDatapointsCopier.copy(datapoints);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder datapoints(InsightRuleContributorDatapoint... datapoints) {
            datapoints(Arrays.asList(datapoints));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder datapoints(Consumer<InsightRuleContributorDatapoint.Builder>... datapoints) {
            datapoints(Stream.of(datapoints).map(c -> InsightRuleContributorDatapoint.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setDatapoints(Collection<InsightRuleContributorDatapoint.BuilderImpl> datapoints) {
            this.datapoints = InsightRuleContributorDatapointsCopier.copyFromBuilder(datapoints);
        }

        @Override
        public InsightRuleContributor build() {
            return new InsightRuleContributor(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
