/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudtrail.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Specifies the settings for each trail.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateTrailRequest extends CloudTrailRequest implements
        ToCopyableBuilder<CreateTrailRequest.Builder, CreateTrailRequest> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateTrailRequest::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> S3_BUCKET_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateTrailRequest::s3BucketName)).setter(setter(Builder::s3BucketName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S3BucketName").build()).build();

    private static final SdkField<String> S3_KEY_PREFIX_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateTrailRequest::s3KeyPrefix)).setter(setter(Builder::s3KeyPrefix))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S3KeyPrefix").build()).build();

    private static final SdkField<String> SNS_TOPIC_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateTrailRequest::snsTopicName)).setter(setter(Builder::snsTopicName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SnsTopicName").build()).build();

    private static final SdkField<Boolean> INCLUDE_GLOBAL_SERVICE_EVENTS_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(CreateTrailRequest::includeGlobalServiceEvents))
            .setter(setter(Builder::includeGlobalServiceEvents))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IncludeGlobalServiceEvents").build())
            .build();

    private static final SdkField<Boolean> IS_MULTI_REGION_TRAIL_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(CreateTrailRequest::isMultiRegionTrail)).setter(setter(Builder::isMultiRegionTrail))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IsMultiRegionTrail").build())
            .build();

    private static final SdkField<Boolean> ENABLE_LOG_FILE_VALIDATION_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(CreateTrailRequest::enableLogFileValidation)).setter(setter(Builder::enableLogFileValidation))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EnableLogFileValidation").build())
            .build();

    private static final SdkField<String> CLOUD_WATCH_LOGS_LOG_GROUP_ARN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING).getter(getter(CreateTrailRequest::cloudWatchLogsLogGroupArn))
            .setter(setter(Builder::cloudWatchLogsLogGroupArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CloudWatchLogsLogGroupArn").build())
            .build();

    private static final SdkField<String> CLOUD_WATCH_LOGS_ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateTrailRequest::cloudWatchLogsRoleArn)).setter(setter(Builder::cloudWatchLogsRoleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CloudWatchLogsRoleArn").build())
            .build();

    private static final SdkField<String> KMS_KEY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateTrailRequest::kmsKeyId)).setter(setter(Builder::kmsKeyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KmsKeyId").build()).build();

    private static final SdkField<Boolean> IS_ORGANIZATION_TRAIL_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(CreateTrailRequest::isOrganizationTrail)).setter(setter(Builder::isOrganizationTrail))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IsOrganizationTrail").build())
            .build();

    private static final SdkField<List<Tag>> TAGS_LIST_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .getter(getter(CreateTrailRequest::tagsList))
            .setter(setter(Builder::tagsList))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TagsList").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD,
            S3_BUCKET_NAME_FIELD, S3_KEY_PREFIX_FIELD, SNS_TOPIC_NAME_FIELD, INCLUDE_GLOBAL_SERVICE_EVENTS_FIELD,
            IS_MULTI_REGION_TRAIL_FIELD, ENABLE_LOG_FILE_VALIDATION_FIELD, CLOUD_WATCH_LOGS_LOG_GROUP_ARN_FIELD,
            CLOUD_WATCH_LOGS_ROLE_ARN_FIELD, KMS_KEY_ID_FIELD, IS_ORGANIZATION_TRAIL_FIELD, TAGS_LIST_FIELD));

    private final String name;

    private final String s3BucketName;

    private final String s3KeyPrefix;

    private final String snsTopicName;

    private final Boolean includeGlobalServiceEvents;

    private final Boolean isMultiRegionTrail;

    private final Boolean enableLogFileValidation;

    private final String cloudWatchLogsLogGroupArn;

    private final String cloudWatchLogsRoleArn;

    private final String kmsKeyId;

    private final Boolean isOrganizationTrail;

    private final List<Tag> tagsList;

    private CreateTrailRequest(BuilderImpl builder) {
        super(builder);
        this.name = builder.name;
        this.s3BucketName = builder.s3BucketName;
        this.s3KeyPrefix = builder.s3KeyPrefix;
        this.snsTopicName = builder.snsTopicName;
        this.includeGlobalServiceEvents = builder.includeGlobalServiceEvents;
        this.isMultiRegionTrail = builder.isMultiRegionTrail;
        this.enableLogFileValidation = builder.enableLogFileValidation;
        this.cloudWatchLogsLogGroupArn = builder.cloudWatchLogsLogGroupArn;
        this.cloudWatchLogsRoleArn = builder.cloudWatchLogsRoleArn;
        this.kmsKeyId = builder.kmsKeyId;
        this.isOrganizationTrail = builder.isOrganizationTrail;
        this.tagsList = builder.tagsList;
    }

    /**
     * <p>
     * Specifies the name of the trail. The name must meet the following requirements:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Contain only ASCII letters (a-z, A-Z), numbers (0-9), periods (.), underscores (_), or dashes (-)
     * </p>
     * </li>
     * <li>
     * <p>
     * Start with a letter or number, and end with a letter or number
     * </p>
     * </li>
     * <li>
     * <p>
     * Be between 3 and 128 characters
     * </p>
     * </li>
     * <li>
     * <p>
     * Have no adjacent periods, underscores or dashes. Names like <code>my-_namespace</code> and
     * <code>my--namespace</code> are invalid.
     * </p>
     * </li>
     * <li>
     * <p>
     * Not be in IP address format (for example, 192.168.5.4)
     * </p>
     * </li>
     * </ul>
     * 
     * @return Specifies the name of the trail. The name must meet the following requirements:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Contain only ASCII letters (a-z, A-Z), numbers (0-9), periods (.), underscores (_), or dashes (-)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Start with a letter or number, and end with a letter or number
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Be between 3 and 128 characters
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Have no adjacent periods, underscores or dashes. Names like <code>my-_namespace</code> and
     *         <code>my--namespace</code> are invalid.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Not be in IP address format (for example, 192.168.5.4)
     *         </p>
     *         </li>
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * Specifies the name of the Amazon S3 bucket designated for publishing log files. See <a
     * href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/create_trail_naming_policy.html">Amazon S3
     * Bucket Naming Requirements</a>.
     * </p>
     * 
     * @return Specifies the name of the Amazon S3 bucket designated for publishing log files. See <a
     *         href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/create_trail_naming_policy.html">Amazon
     *         S3 Bucket Naming Requirements</a>.
     */
    public String s3BucketName() {
        return s3BucketName;
    }

    /**
     * <p>
     * Specifies the Amazon S3 key prefix that comes after the name of the bucket you have designated for log file
     * delivery. For more information, see <a
     * href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/cloudtrail-find-log-files.html">Finding Your
     * CloudTrail Log Files</a>. The maximum length is 200 characters.
     * </p>
     * 
     * @return Specifies the Amazon S3 key prefix that comes after the name of the bucket you have designated for log
     *         file delivery. For more information, see <a
     *         href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/cloudtrail-find-log-files.html">Finding
     *         Your CloudTrail Log Files</a>. The maximum length is 200 characters.
     */
    public String s3KeyPrefix() {
        return s3KeyPrefix;
    }

    /**
     * <p>
     * Specifies the name of the Amazon SNS topic defined for notification of log file delivery. The maximum length is
     * 256 characters.
     * </p>
     * 
     * @return Specifies the name of the Amazon SNS topic defined for notification of log file delivery. The maximum
     *         length is 256 characters.
     */
    public String snsTopicName() {
        return snsTopicName;
    }

    /**
     * <p>
     * Specifies whether the trail is publishing events from global services such as IAM to the log files.
     * </p>
     * 
     * @return Specifies whether the trail is publishing events from global services such as IAM to the log files.
     */
    public Boolean includeGlobalServiceEvents() {
        return includeGlobalServiceEvents;
    }

    /**
     * <p>
     * Specifies whether the trail is created in the current region or in all regions. The default is false, which
     * creates a trail only in the region where you are signed in. As a best practice, consider creating trails that log
     * events in all regions.
     * </p>
     * 
     * @return Specifies whether the trail is created in the current region or in all regions. The default is false,
     *         which creates a trail only in the region where you are signed in. As a best practice, consider creating
     *         trails that log events in all regions.
     */
    public Boolean isMultiRegionTrail() {
        return isMultiRegionTrail;
    }

    /**
     * <p>
     * Specifies whether log file integrity validation is enabled. The default is false.
     * </p>
     * <note>
     * <p>
     * When you disable log file integrity validation, the chain of digest files is broken after one hour. CloudTrail
     * will not create digest files for log files that were delivered during a period in which log file integrity
     * validation was disabled. For example, if you enable log file integrity validation at noon on January 1, disable
     * it at noon on January 2, and re-enable it at noon on January 10, digest files will not be created for the log
     * files delivered from noon on January 2 to noon on January 10. The same applies whenever you stop CloudTrail
     * logging or delete a trail.
     * </p>
     * </note>
     * 
     * @return Specifies whether log file integrity validation is enabled. The default is false.</p> <note>
     *         <p>
     *         When you disable log file integrity validation, the chain of digest files is broken after one hour.
     *         CloudTrail will not create digest files for log files that were delivered during a period in which log
     *         file integrity validation was disabled. For example, if you enable log file integrity validation at noon
     *         on January 1, disable it at noon on January 2, and re-enable it at noon on January 10, digest files will
     *         not be created for the log files delivered from noon on January 2 to noon on January 10. The same applies
     *         whenever you stop CloudTrail logging or delete a trail.
     *         </p>
     */
    public Boolean enableLogFileValidation() {
        return enableLogFileValidation;
    }

    /**
     * <p>
     * Specifies a log group name using an Amazon Resource Name (ARN), a unique identifier that represents the log group
     * to which CloudTrail logs will be delivered. Not required unless you specify CloudWatchLogsRoleArn.
     * </p>
     * 
     * @return Specifies a log group name using an Amazon Resource Name (ARN), a unique identifier that represents the
     *         log group to which CloudTrail logs will be delivered. Not required unless you specify
     *         CloudWatchLogsRoleArn.
     */
    public String cloudWatchLogsLogGroupArn() {
        return cloudWatchLogsLogGroupArn;
    }

    /**
     * <p>
     * Specifies the role for the CloudWatch Logs endpoint to assume to write to a user's log group.
     * </p>
     * 
     * @return Specifies the role for the CloudWatch Logs endpoint to assume to write to a user's log group.
     */
    public String cloudWatchLogsRoleArn() {
        return cloudWatchLogsRoleArn;
    }

    /**
     * <p>
     * Specifies the KMS key ID to use to encrypt the logs delivered by CloudTrail. The value can be an alias name
     * prefixed by "alias/", a fully specified ARN to an alias, a fully specified ARN to a key, or a globally unique
     * identifier.
     * </p>
     * <p>
     * Examples:
     * </p>
     * <ul>
     * <li>
     * <p>
     * alias/MyAliasName
     * </p>
     * </li>
     * <li>
     * <p>
     * arn:aws:kms:us-east-2:123456789012:alias/MyAliasName
     * </p>
     * </li>
     * <li>
     * <p>
     * arn:aws:kms:us-east-2:123456789012:key/12345678-1234-1234-1234-123456789012
     * </p>
     * </li>
     * <li>
     * <p>
     * 12345678-1234-1234-1234-123456789012
     * </p>
     * </li>
     * </ul>
     * 
     * @return Specifies the KMS key ID to use to encrypt the logs delivered by CloudTrail. The value can be an alias
     *         name prefixed by "alias/", a fully specified ARN to an alias, a fully specified ARN to a key, or a
     *         globally unique identifier.</p>
     *         <p>
     *         Examples:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         alias/MyAliasName
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         arn:aws:kms:us-east-2:123456789012:alias/MyAliasName
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         arn:aws:kms:us-east-2:123456789012:key/12345678-1234-1234-1234-123456789012
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         12345678-1234-1234-1234-123456789012
     *         </p>
     *         </li>
     */
    public String kmsKeyId() {
        return kmsKeyId;
    }

    /**
     * <p>
     * Specifies whether the trail is created for all accounts in an organization in AWS Organizations, or only for the
     * current AWS account. The default is false, and cannot be true unless the call is made on behalf of an AWS account
     * that is the master account for an organization in AWS Organizations.
     * </p>
     * 
     * @return Specifies whether the trail is created for all accounts in an organization in AWS Organizations, or only
     *         for the current AWS account. The default is false, and cannot be true unless the call is made on behalf
     *         of an AWS account that is the master account for an organization in AWS Organizations.
     */
    public Boolean isOrganizationTrail() {
        return isOrganizationTrail;
    }

    /**
     * Returns true if the TagsList property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasTagsList() {
        return tagsList != null && !(tagsList instanceof SdkAutoConstructList);
    }

    /**
     * Returns the value of the TagsList property for this object.
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTagsList()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The value of the TagsList property for this object.
     */
    public List<Tag> tagsList() {
        return tagsList;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(s3BucketName());
        hashCode = 31 * hashCode + Objects.hashCode(s3KeyPrefix());
        hashCode = 31 * hashCode + Objects.hashCode(snsTopicName());
        hashCode = 31 * hashCode + Objects.hashCode(includeGlobalServiceEvents());
        hashCode = 31 * hashCode + Objects.hashCode(isMultiRegionTrail());
        hashCode = 31 * hashCode + Objects.hashCode(enableLogFileValidation());
        hashCode = 31 * hashCode + Objects.hashCode(cloudWatchLogsLogGroupArn());
        hashCode = 31 * hashCode + Objects.hashCode(cloudWatchLogsRoleArn());
        hashCode = 31 * hashCode + Objects.hashCode(kmsKeyId());
        hashCode = 31 * hashCode + Objects.hashCode(isOrganizationTrail());
        hashCode = 31 * hashCode + Objects.hashCode(tagsList());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateTrailRequest)) {
            return false;
        }
        CreateTrailRequest other = (CreateTrailRequest) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(s3BucketName(), other.s3BucketName())
                && Objects.equals(s3KeyPrefix(), other.s3KeyPrefix()) && Objects.equals(snsTopicName(), other.snsTopicName())
                && Objects.equals(includeGlobalServiceEvents(), other.includeGlobalServiceEvents())
                && Objects.equals(isMultiRegionTrail(), other.isMultiRegionTrail())
                && Objects.equals(enableLogFileValidation(), other.enableLogFileValidation())
                && Objects.equals(cloudWatchLogsLogGroupArn(), other.cloudWatchLogsLogGroupArn())
                && Objects.equals(cloudWatchLogsRoleArn(), other.cloudWatchLogsRoleArn())
                && Objects.equals(kmsKeyId(), other.kmsKeyId())
                && Objects.equals(isOrganizationTrail(), other.isOrganizationTrail())
                && Objects.equals(tagsList(), other.tagsList());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("CreateTrailRequest").add("Name", name()).add("S3BucketName", s3BucketName())
                .add("S3KeyPrefix", s3KeyPrefix()).add("SnsTopicName", snsTopicName())
                .add("IncludeGlobalServiceEvents", includeGlobalServiceEvents()).add("IsMultiRegionTrail", isMultiRegionTrail())
                .add("EnableLogFileValidation", enableLogFileValidation())
                .add("CloudWatchLogsLogGroupArn", cloudWatchLogsLogGroupArn())
                .add("CloudWatchLogsRoleArn", cloudWatchLogsRoleArn()).add("KmsKeyId", kmsKeyId())
                .add("IsOrganizationTrail", isOrganizationTrail()).add("TagsList", tagsList()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "S3BucketName":
            return Optional.ofNullable(clazz.cast(s3BucketName()));
        case "S3KeyPrefix":
            return Optional.ofNullable(clazz.cast(s3KeyPrefix()));
        case "SnsTopicName":
            return Optional.ofNullable(clazz.cast(snsTopicName()));
        case "IncludeGlobalServiceEvents":
            return Optional.ofNullable(clazz.cast(includeGlobalServiceEvents()));
        case "IsMultiRegionTrail":
            return Optional.ofNullable(clazz.cast(isMultiRegionTrail()));
        case "EnableLogFileValidation":
            return Optional.ofNullable(clazz.cast(enableLogFileValidation()));
        case "CloudWatchLogsLogGroupArn":
            return Optional.ofNullable(clazz.cast(cloudWatchLogsLogGroupArn()));
        case "CloudWatchLogsRoleArn":
            return Optional.ofNullable(clazz.cast(cloudWatchLogsRoleArn()));
        case "KmsKeyId":
            return Optional.ofNullable(clazz.cast(kmsKeyId()));
        case "IsOrganizationTrail":
            return Optional.ofNullable(clazz.cast(isOrganizationTrail()));
        case "TagsList":
            return Optional.ofNullable(clazz.cast(tagsList()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateTrailRequest, T> g) {
        return obj -> g.apply((CreateTrailRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends CloudTrailRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateTrailRequest> {
        /**
         * <p>
         * Specifies the name of the trail. The name must meet the following requirements:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Contain only ASCII letters (a-z, A-Z), numbers (0-9), periods (.), underscores (_), or dashes (-)
         * </p>
         * </li>
         * <li>
         * <p>
         * Start with a letter or number, and end with a letter or number
         * </p>
         * </li>
         * <li>
         * <p>
         * Be between 3 and 128 characters
         * </p>
         * </li>
         * <li>
         * <p>
         * Have no adjacent periods, underscores or dashes. Names like <code>my-_namespace</code> and
         * <code>my--namespace</code> are invalid.
         * </p>
         * </li>
         * <li>
         * <p>
         * Not be in IP address format (for example, 192.168.5.4)
         * </p>
         * </li>
         * </ul>
         * 
         * @param name
         *        Specifies the name of the trail. The name must meet the following requirements:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        Contain only ASCII letters (a-z, A-Z), numbers (0-9), periods (.), underscores (_), or dashes (-)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Start with a letter or number, and end with a letter or number
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Be between 3 and 128 characters
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Have no adjacent periods, underscores or dashes. Names like <code>my-_namespace</code> and
         *        <code>my--namespace</code> are invalid.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Not be in IP address format (for example, 192.168.5.4)
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * Specifies the name of the Amazon S3 bucket designated for publishing log files. See <a
         * href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/create_trail_naming_policy.html">Amazon S3
         * Bucket Naming Requirements</a>.
         * </p>
         * 
         * @param s3BucketName
         *        Specifies the name of the Amazon S3 bucket designated for publishing log files. See <a
         *        href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/create_trail_naming_policy.html"
         *        >Amazon S3 Bucket Naming Requirements</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3BucketName(String s3BucketName);

        /**
         * <p>
         * Specifies the Amazon S3 key prefix that comes after the name of the bucket you have designated for log file
         * delivery. For more information, see <a
         * href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/cloudtrail-find-log-files.html">Finding Your
         * CloudTrail Log Files</a>. The maximum length is 200 characters.
         * </p>
         * 
         * @param s3KeyPrefix
         *        Specifies the Amazon S3 key prefix that comes after the name of the bucket you have designated for log
         *        file delivery. For more information, see <a
         *        href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/cloudtrail-find-log-files.html"
         *        >Finding Your CloudTrail Log Files</a>. The maximum length is 200 characters.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3KeyPrefix(String s3KeyPrefix);

        /**
         * <p>
         * Specifies the name of the Amazon SNS topic defined for notification of log file delivery. The maximum length
         * is 256 characters.
         * </p>
         * 
         * @param snsTopicName
         *        Specifies the name of the Amazon SNS topic defined for notification of log file delivery. The maximum
         *        length is 256 characters.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder snsTopicName(String snsTopicName);

        /**
         * <p>
         * Specifies whether the trail is publishing events from global services such as IAM to the log files.
         * </p>
         * 
         * @param includeGlobalServiceEvents
         *        Specifies whether the trail is publishing events from global services such as IAM to the log files.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder includeGlobalServiceEvents(Boolean includeGlobalServiceEvents);

        /**
         * <p>
         * Specifies whether the trail is created in the current region or in all regions. The default is false, which
         * creates a trail only in the region where you are signed in. As a best practice, consider creating trails that
         * log events in all regions.
         * </p>
         * 
         * @param isMultiRegionTrail
         *        Specifies whether the trail is created in the current region or in all regions. The default is false,
         *        which creates a trail only in the region where you are signed in. As a best practice, consider
         *        creating trails that log events in all regions.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder isMultiRegionTrail(Boolean isMultiRegionTrail);

        /**
         * <p>
         * Specifies whether log file integrity validation is enabled. The default is false.
         * </p>
         * <note>
         * <p>
         * When you disable log file integrity validation, the chain of digest files is broken after one hour.
         * CloudTrail will not create digest files for log files that were delivered during a period in which log file
         * integrity validation was disabled. For example, if you enable log file integrity validation at noon on
         * January 1, disable it at noon on January 2, and re-enable it at noon on January 10, digest files will not be
         * created for the log files delivered from noon on January 2 to noon on January 10. The same applies whenever
         * you stop CloudTrail logging or delete a trail.
         * </p>
         * </note>
         * 
         * @param enableLogFileValidation
         *        Specifies whether log file integrity validation is enabled. The default is false.</p> <note>
         *        <p>
         *        When you disable log file integrity validation, the chain of digest files is broken after one hour.
         *        CloudTrail will not create digest files for log files that were delivered during a period in which log
         *        file integrity validation was disabled. For example, if you enable log file integrity validation at
         *        noon on January 1, disable it at noon on January 2, and re-enable it at noon on January 10, digest
         *        files will not be created for the log files delivered from noon on January 2 to noon on January 10.
         *        The same applies whenever you stop CloudTrail logging or delete a trail.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enableLogFileValidation(Boolean enableLogFileValidation);

        /**
         * <p>
         * Specifies a log group name using an Amazon Resource Name (ARN), a unique identifier that represents the log
         * group to which CloudTrail logs will be delivered. Not required unless you specify CloudWatchLogsRoleArn.
         * </p>
         * 
         * @param cloudWatchLogsLogGroupArn
         *        Specifies a log group name using an Amazon Resource Name (ARN), a unique identifier that represents
         *        the log group to which CloudTrail logs will be delivered. Not required unless you specify
         *        CloudWatchLogsRoleArn.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cloudWatchLogsLogGroupArn(String cloudWatchLogsLogGroupArn);

        /**
         * <p>
         * Specifies the role for the CloudWatch Logs endpoint to assume to write to a user's log group.
         * </p>
         * 
         * @param cloudWatchLogsRoleArn
         *        Specifies the role for the CloudWatch Logs endpoint to assume to write to a user's log group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cloudWatchLogsRoleArn(String cloudWatchLogsRoleArn);

        /**
         * <p>
         * Specifies the KMS key ID to use to encrypt the logs delivered by CloudTrail. The value can be an alias name
         * prefixed by "alias/", a fully specified ARN to an alias, a fully specified ARN to a key, or a globally unique
         * identifier.
         * </p>
         * <p>
         * Examples:
         * </p>
         * <ul>
         * <li>
         * <p>
         * alias/MyAliasName
         * </p>
         * </li>
         * <li>
         * <p>
         * arn:aws:kms:us-east-2:123456789012:alias/MyAliasName
         * </p>
         * </li>
         * <li>
         * <p>
         * arn:aws:kms:us-east-2:123456789012:key/12345678-1234-1234-1234-123456789012
         * </p>
         * </li>
         * <li>
         * <p>
         * 12345678-1234-1234-1234-123456789012
         * </p>
         * </li>
         * </ul>
         * 
         * @param kmsKeyId
         *        Specifies the KMS key ID to use to encrypt the logs delivered by CloudTrail. The value can be an alias
         *        name prefixed by "alias/", a fully specified ARN to an alias, a fully specified ARN to a key, or a
         *        globally unique identifier.</p>
         *        <p>
         *        Examples:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        alias/MyAliasName
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        arn:aws:kms:us-east-2:123456789012:alias/MyAliasName
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        arn:aws:kms:us-east-2:123456789012:key/12345678-1234-1234-1234-123456789012
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        12345678-1234-1234-1234-123456789012
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKeyId(String kmsKeyId);

        /**
         * <p>
         * Specifies whether the trail is created for all accounts in an organization in AWS Organizations, or only for
         * the current AWS account. The default is false, and cannot be true unless the call is made on behalf of an AWS
         * account that is the master account for an organization in AWS Organizations.
         * </p>
         * 
         * @param isOrganizationTrail
         *        Specifies whether the trail is created for all accounts in an organization in AWS Organizations, or
         *        only for the current AWS account. The default is false, and cannot be true unless the call is made on
         *        behalf of an AWS account that is the master account for an organization in AWS Organizations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder isOrganizationTrail(Boolean isOrganizationTrail);

        /**
         * Sets the value of the TagsList property for this object.
         *
         * @param tagsList
         *        The new value for the TagsList property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tagsList(Collection<Tag> tagsList);

        /**
         * Sets the value of the TagsList property for this object.
         *
         * @param tagsList
         *        The new value for the TagsList property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tagsList(Tag... tagsList);

        /**
         * Sets the value of the TagsList property for this object.
         *
         * This is a convenience that creates an instance of the {@link List<Tag>.Builder} avoiding the need to create
         * one manually via {@link List<Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #tagsList(List<Tag>)}.
         * 
         * @param tagsList
         *        a consumer that will call methods on {@link List<Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tagsList(List<Tag>)
         */
        Builder tagsList(Consumer<Tag.Builder>... tagsList);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends CloudTrailRequest.BuilderImpl implements Builder {
        private String name;

        private String s3BucketName;

        private String s3KeyPrefix;

        private String snsTopicName;

        private Boolean includeGlobalServiceEvents;

        private Boolean isMultiRegionTrail;

        private Boolean enableLogFileValidation;

        private String cloudWatchLogsLogGroupArn;

        private String cloudWatchLogsRoleArn;

        private String kmsKeyId;

        private Boolean isOrganizationTrail;

        private List<Tag> tagsList = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CreateTrailRequest model) {
            super(model);
            name(model.name);
            s3BucketName(model.s3BucketName);
            s3KeyPrefix(model.s3KeyPrefix);
            snsTopicName(model.snsTopicName);
            includeGlobalServiceEvents(model.includeGlobalServiceEvents);
            isMultiRegionTrail(model.isMultiRegionTrail);
            enableLogFileValidation(model.enableLogFileValidation);
            cloudWatchLogsLogGroupArn(model.cloudWatchLogsLogGroupArn);
            cloudWatchLogsRoleArn(model.cloudWatchLogsRoleArn);
            kmsKeyId(model.kmsKeyId);
            isOrganizationTrail(model.isOrganizationTrail);
            tagsList(model.tagsList);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getS3BucketName() {
            return s3BucketName;
        }

        @Override
        public final Builder s3BucketName(String s3BucketName) {
            this.s3BucketName = s3BucketName;
            return this;
        }

        public final void setS3BucketName(String s3BucketName) {
            this.s3BucketName = s3BucketName;
        }

        public final String getS3KeyPrefix() {
            return s3KeyPrefix;
        }

        @Override
        public final Builder s3KeyPrefix(String s3KeyPrefix) {
            this.s3KeyPrefix = s3KeyPrefix;
            return this;
        }

        public final void setS3KeyPrefix(String s3KeyPrefix) {
            this.s3KeyPrefix = s3KeyPrefix;
        }

        public final String getSnsTopicName() {
            return snsTopicName;
        }

        @Override
        public final Builder snsTopicName(String snsTopicName) {
            this.snsTopicName = snsTopicName;
            return this;
        }

        public final void setSnsTopicName(String snsTopicName) {
            this.snsTopicName = snsTopicName;
        }

        public final Boolean getIncludeGlobalServiceEvents() {
            return includeGlobalServiceEvents;
        }

        @Override
        public final Builder includeGlobalServiceEvents(Boolean includeGlobalServiceEvents) {
            this.includeGlobalServiceEvents = includeGlobalServiceEvents;
            return this;
        }

        public final void setIncludeGlobalServiceEvents(Boolean includeGlobalServiceEvents) {
            this.includeGlobalServiceEvents = includeGlobalServiceEvents;
        }

        public final Boolean getIsMultiRegionTrail() {
            return isMultiRegionTrail;
        }

        @Override
        public final Builder isMultiRegionTrail(Boolean isMultiRegionTrail) {
            this.isMultiRegionTrail = isMultiRegionTrail;
            return this;
        }

        public final void setIsMultiRegionTrail(Boolean isMultiRegionTrail) {
            this.isMultiRegionTrail = isMultiRegionTrail;
        }

        public final Boolean getEnableLogFileValidation() {
            return enableLogFileValidation;
        }

        @Override
        public final Builder enableLogFileValidation(Boolean enableLogFileValidation) {
            this.enableLogFileValidation = enableLogFileValidation;
            return this;
        }

        public final void setEnableLogFileValidation(Boolean enableLogFileValidation) {
            this.enableLogFileValidation = enableLogFileValidation;
        }

        public final String getCloudWatchLogsLogGroupArn() {
            return cloudWatchLogsLogGroupArn;
        }

        @Override
        public final Builder cloudWatchLogsLogGroupArn(String cloudWatchLogsLogGroupArn) {
            this.cloudWatchLogsLogGroupArn = cloudWatchLogsLogGroupArn;
            return this;
        }

        public final void setCloudWatchLogsLogGroupArn(String cloudWatchLogsLogGroupArn) {
            this.cloudWatchLogsLogGroupArn = cloudWatchLogsLogGroupArn;
        }

        public final String getCloudWatchLogsRoleArn() {
            return cloudWatchLogsRoleArn;
        }

        @Override
        public final Builder cloudWatchLogsRoleArn(String cloudWatchLogsRoleArn) {
            this.cloudWatchLogsRoleArn = cloudWatchLogsRoleArn;
            return this;
        }

        public final void setCloudWatchLogsRoleArn(String cloudWatchLogsRoleArn) {
            this.cloudWatchLogsRoleArn = cloudWatchLogsRoleArn;
        }

        public final String getKmsKeyId() {
            return kmsKeyId;
        }

        @Override
        public final Builder kmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
            return this;
        }

        public final void setKmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
        }

        public final Boolean getIsOrganizationTrail() {
            return isOrganizationTrail;
        }

        @Override
        public final Builder isOrganizationTrail(Boolean isOrganizationTrail) {
            this.isOrganizationTrail = isOrganizationTrail;
            return this;
        }

        public final void setIsOrganizationTrail(Boolean isOrganizationTrail) {
            this.isOrganizationTrail = isOrganizationTrail;
        }

        public final Collection<Tag.Builder> getTagsList() {
            return tagsList != null ? tagsList.stream().map(Tag::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder tagsList(Collection<Tag> tagsList) {
            this.tagsList = TagsListCopier.copy(tagsList);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tagsList(Tag... tagsList) {
            tagsList(Arrays.asList(tagsList));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tagsList(Consumer<Tag.Builder>... tagsList) {
            tagsList(Stream.of(tagsList).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTagsList(Collection<Tag.BuilderImpl> tagsList) {
            this.tagsList = TagsListCopier.copyFromBuilder(tagsList);
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateTrailRequest build() {
            return new CreateTrailRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
