/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudsearch;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.query.AwsQueryProtocolFactory;
import software.amazon.awssdk.services.cloudsearch.model.BaseException;
import software.amazon.awssdk.services.cloudsearch.model.BuildSuggestersRequest;
import software.amazon.awssdk.services.cloudsearch.model.BuildSuggestersResponse;
import software.amazon.awssdk.services.cloudsearch.model.CloudSearchException;
import software.amazon.awssdk.services.cloudsearch.model.CloudSearchRequest;
import software.amazon.awssdk.services.cloudsearch.model.CreateDomainRequest;
import software.amazon.awssdk.services.cloudsearch.model.CreateDomainResponse;
import software.amazon.awssdk.services.cloudsearch.model.DefineAnalysisSchemeRequest;
import software.amazon.awssdk.services.cloudsearch.model.DefineAnalysisSchemeResponse;
import software.amazon.awssdk.services.cloudsearch.model.DefineExpressionRequest;
import software.amazon.awssdk.services.cloudsearch.model.DefineExpressionResponse;
import software.amazon.awssdk.services.cloudsearch.model.DefineIndexFieldRequest;
import software.amazon.awssdk.services.cloudsearch.model.DefineIndexFieldResponse;
import software.amazon.awssdk.services.cloudsearch.model.DefineSuggesterRequest;
import software.amazon.awssdk.services.cloudsearch.model.DefineSuggesterResponse;
import software.amazon.awssdk.services.cloudsearch.model.DeleteAnalysisSchemeRequest;
import software.amazon.awssdk.services.cloudsearch.model.DeleteAnalysisSchemeResponse;
import software.amazon.awssdk.services.cloudsearch.model.DeleteDomainRequest;
import software.amazon.awssdk.services.cloudsearch.model.DeleteDomainResponse;
import software.amazon.awssdk.services.cloudsearch.model.DeleteExpressionRequest;
import software.amazon.awssdk.services.cloudsearch.model.DeleteExpressionResponse;
import software.amazon.awssdk.services.cloudsearch.model.DeleteIndexFieldRequest;
import software.amazon.awssdk.services.cloudsearch.model.DeleteIndexFieldResponse;
import software.amazon.awssdk.services.cloudsearch.model.DeleteSuggesterRequest;
import software.amazon.awssdk.services.cloudsearch.model.DeleteSuggesterResponse;
import software.amazon.awssdk.services.cloudsearch.model.DescribeAnalysisSchemesRequest;
import software.amazon.awssdk.services.cloudsearch.model.DescribeAnalysisSchemesResponse;
import software.amazon.awssdk.services.cloudsearch.model.DescribeAvailabilityOptionsRequest;
import software.amazon.awssdk.services.cloudsearch.model.DescribeAvailabilityOptionsResponse;
import software.amazon.awssdk.services.cloudsearch.model.DescribeDomainEndpointOptionsRequest;
import software.amazon.awssdk.services.cloudsearch.model.DescribeDomainEndpointOptionsResponse;
import software.amazon.awssdk.services.cloudsearch.model.DescribeDomainsRequest;
import software.amazon.awssdk.services.cloudsearch.model.DescribeDomainsResponse;
import software.amazon.awssdk.services.cloudsearch.model.DescribeExpressionsRequest;
import software.amazon.awssdk.services.cloudsearch.model.DescribeExpressionsResponse;
import software.amazon.awssdk.services.cloudsearch.model.DescribeIndexFieldsRequest;
import software.amazon.awssdk.services.cloudsearch.model.DescribeIndexFieldsResponse;
import software.amazon.awssdk.services.cloudsearch.model.DescribeScalingParametersRequest;
import software.amazon.awssdk.services.cloudsearch.model.DescribeScalingParametersResponse;
import software.amazon.awssdk.services.cloudsearch.model.DescribeServiceAccessPoliciesRequest;
import software.amazon.awssdk.services.cloudsearch.model.DescribeServiceAccessPoliciesResponse;
import software.amazon.awssdk.services.cloudsearch.model.DescribeSuggestersRequest;
import software.amazon.awssdk.services.cloudsearch.model.DescribeSuggestersResponse;
import software.amazon.awssdk.services.cloudsearch.model.DisabledOperationException;
import software.amazon.awssdk.services.cloudsearch.model.IndexDocumentsRequest;
import software.amazon.awssdk.services.cloudsearch.model.IndexDocumentsResponse;
import software.amazon.awssdk.services.cloudsearch.model.InternalException;
import software.amazon.awssdk.services.cloudsearch.model.InvalidTypeException;
import software.amazon.awssdk.services.cloudsearch.model.LimitExceededException;
import software.amazon.awssdk.services.cloudsearch.model.ListDomainNamesRequest;
import software.amazon.awssdk.services.cloudsearch.model.ListDomainNamesResponse;
import software.amazon.awssdk.services.cloudsearch.model.ResourceNotFoundException;
import software.amazon.awssdk.services.cloudsearch.model.UpdateAvailabilityOptionsRequest;
import software.amazon.awssdk.services.cloudsearch.model.UpdateAvailabilityOptionsResponse;
import software.amazon.awssdk.services.cloudsearch.model.UpdateDomainEndpointOptionsRequest;
import software.amazon.awssdk.services.cloudsearch.model.UpdateDomainEndpointOptionsResponse;
import software.amazon.awssdk.services.cloudsearch.model.UpdateScalingParametersRequest;
import software.amazon.awssdk.services.cloudsearch.model.UpdateScalingParametersResponse;
import software.amazon.awssdk.services.cloudsearch.model.UpdateServiceAccessPoliciesRequest;
import software.amazon.awssdk.services.cloudsearch.model.UpdateServiceAccessPoliciesResponse;
import software.amazon.awssdk.services.cloudsearch.model.ValidationException;
import software.amazon.awssdk.services.cloudsearch.transform.BuildSuggestersRequestMarshaller;
import software.amazon.awssdk.services.cloudsearch.transform.CreateDomainRequestMarshaller;
import software.amazon.awssdk.services.cloudsearch.transform.DefineAnalysisSchemeRequestMarshaller;
import software.amazon.awssdk.services.cloudsearch.transform.DefineExpressionRequestMarshaller;
import software.amazon.awssdk.services.cloudsearch.transform.DefineIndexFieldRequestMarshaller;
import software.amazon.awssdk.services.cloudsearch.transform.DefineSuggesterRequestMarshaller;
import software.amazon.awssdk.services.cloudsearch.transform.DeleteAnalysisSchemeRequestMarshaller;
import software.amazon.awssdk.services.cloudsearch.transform.DeleteDomainRequestMarshaller;
import software.amazon.awssdk.services.cloudsearch.transform.DeleteExpressionRequestMarshaller;
import software.amazon.awssdk.services.cloudsearch.transform.DeleteIndexFieldRequestMarshaller;
import software.amazon.awssdk.services.cloudsearch.transform.DeleteSuggesterRequestMarshaller;
import software.amazon.awssdk.services.cloudsearch.transform.DescribeAnalysisSchemesRequestMarshaller;
import software.amazon.awssdk.services.cloudsearch.transform.DescribeAvailabilityOptionsRequestMarshaller;
import software.amazon.awssdk.services.cloudsearch.transform.DescribeDomainEndpointOptionsRequestMarshaller;
import software.amazon.awssdk.services.cloudsearch.transform.DescribeDomainsRequestMarshaller;
import software.amazon.awssdk.services.cloudsearch.transform.DescribeExpressionsRequestMarshaller;
import software.amazon.awssdk.services.cloudsearch.transform.DescribeIndexFieldsRequestMarshaller;
import software.amazon.awssdk.services.cloudsearch.transform.DescribeScalingParametersRequestMarshaller;
import software.amazon.awssdk.services.cloudsearch.transform.DescribeServiceAccessPoliciesRequestMarshaller;
import software.amazon.awssdk.services.cloudsearch.transform.DescribeSuggestersRequestMarshaller;
import software.amazon.awssdk.services.cloudsearch.transform.IndexDocumentsRequestMarshaller;
import software.amazon.awssdk.services.cloudsearch.transform.ListDomainNamesRequestMarshaller;
import software.amazon.awssdk.services.cloudsearch.transform.UpdateAvailabilityOptionsRequestMarshaller;
import software.amazon.awssdk.services.cloudsearch.transform.UpdateDomainEndpointOptionsRequestMarshaller;
import software.amazon.awssdk.services.cloudsearch.transform.UpdateScalingParametersRequestMarshaller;
import software.amazon.awssdk.services.cloudsearch.transform.UpdateServiceAccessPoliciesRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link CloudSearchAsyncClient}.
 *
 * @see CloudSearchAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultCloudSearchAsyncClient implements CloudSearchAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultCloudSearchAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsQueryProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultCloudSearchAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Indexes the search suggestions. For more information, see <a href=
     * "http://docs.aws.amazon.com/cloudsearch/latest/developerguide/getting-suggestions.html#configuring-suggesters"
     * >Configuring Suggesters</a> in the <i>Amazon CloudSearch Developer Guide</i>.
     * </p>
     *
     * @param buildSuggestersRequest
     *        Container for the parameters to the <code><a>BuildSuggester</a></code> operation. Specifies the name of
     *        the domain you want to update.
     * @return A Java Future containing the result of the BuildSuggesters operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>InternalException An internal error occurred while processing the request. If this problem persists,
     *         report an issue from the <a href="http://status.aws.amazon.com/" target="_blank">Service Health
     *         Dashboard</a>.</li>
     *         <li>ResourceNotFoundException The request was rejected because it attempted to reference a resource that
     *         does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudSearchAsyncClient.BuildSuggesters
     */
    @Override
    public CompletableFuture<BuildSuggestersResponse> buildSuggesters(BuildSuggestersRequest buildSuggestersRequest) {
        try {

            HttpResponseHandler<BuildSuggestersResponse> responseHandler = protocolFactory
                    .createResponseHandler(BuildSuggestersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<BuildSuggestersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BuildSuggestersRequest, BuildSuggestersResponse>()
                            .withOperationName("BuildSuggesters")
                            .withMarshaller(new BuildSuggestersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(buildSuggestersRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new search domain. For more information, see <a
     * href="http://docs.aws.amazon.com/cloudsearch/latest/developerguide/creating-domains.html"
     * target="_blank">Creating a Search Domain</a> in the <i>Amazon CloudSearch Developer Guide</i>.
     * </p>
     *
     * @param createDomainRequest
     *        Container for the parameters to the <code><a>CreateDomain</a></code> operation. Specifies a name for the
     *        new search domain.
     * @return A Java Future containing the result of the CreateDomain operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>InternalException An internal error occurred while processing the request. If this problem persists,
     *         report an issue from the <a href="http://status.aws.amazon.com/" target="_blank">Service Health
     *         Dashboard</a>.</li>
     *         <li>LimitExceededException The request was rejected because a resource limit has already been met.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudSearchAsyncClient.CreateDomain
     */
    @Override
    public CompletableFuture<CreateDomainResponse> createDomain(CreateDomainRequest createDomainRequest) {
        try {

            HttpResponseHandler<CreateDomainResponse> responseHandler = protocolFactory
                    .createResponseHandler(CreateDomainResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<CreateDomainResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateDomainRequest, CreateDomainResponse>()
                            .withOperationName("CreateDomain").withMarshaller(new CreateDomainRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createDomainRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Configures an analysis scheme that can be applied to a <code>text</code> or <code>text-array</code> field to
     * define language-specific text processing options. For more information, see <a
     * href="http://docs.aws.amazon.com/cloudsearch/latest/developerguide/configuring-analysis-schemes.html"
     * target="_blank">Configuring Analysis Schemes</a> in the <i>Amazon CloudSearch Developer Guide</i>.
     * </p>
     *
     * @param defineAnalysisSchemeRequest
     *        Container for the parameters to the <code><a>DefineAnalysisScheme</a></code> operation. Specifies the name
     *        of the domain you want to update and the analysis scheme configuration.
     * @return A Java Future containing the result of the DefineAnalysisScheme operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>InternalException An internal error occurred while processing the request. If this problem persists,
     *         report an issue from the <a href="http://status.aws.amazon.com/" target="_blank">Service Health
     *         Dashboard</a>.</li>
     *         <li>LimitExceededException The request was rejected because a resource limit has already been met.</li>
     *         <li>InvalidTypeException The request was rejected because it specified an invalid type definition.</li>
     *         <li>ResourceNotFoundException The request was rejected because it attempted to reference a resource that
     *         does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudSearchAsyncClient.DefineAnalysisScheme
     */
    @Override
    public CompletableFuture<DefineAnalysisSchemeResponse> defineAnalysisScheme(
            DefineAnalysisSchemeRequest defineAnalysisSchemeRequest) {
        try {

            HttpResponseHandler<DefineAnalysisSchemeResponse> responseHandler = protocolFactory
                    .createResponseHandler(DefineAnalysisSchemeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<DefineAnalysisSchemeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DefineAnalysisSchemeRequest, DefineAnalysisSchemeResponse>()
                            .withOperationName("DefineAnalysisScheme")
                            .withMarshaller(new DefineAnalysisSchemeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(defineAnalysisSchemeRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Configures an <code><a>Expression</a></code> for the search domain. Used to create new expressions and modify
     * existing ones. If the expression exists, the new configuration replaces the old one. For more information, see <a
     * href="http://docs.aws.amazon.com/cloudsearch/latest/developerguide/configuring-expressions.html"
     * target="_blank">Configuring Expressions</a> in the <i>Amazon CloudSearch Developer Guide</i>.
     * </p>
     *
     * @param defineExpressionRequest
     *        Container for the parameters to the <code><a>DefineExpression</a></code> operation. Specifies the name of
     *        the domain you want to update and the expression you want to configure.
     * @return A Java Future containing the result of the DefineExpression operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>InternalException An internal error occurred while processing the request. If this problem persists,
     *         report an issue from the <a href="http://status.aws.amazon.com/" target="_blank">Service Health
     *         Dashboard</a>.</li>
     *         <li>LimitExceededException The request was rejected because a resource limit has already been met.</li>
     *         <li>InvalidTypeException The request was rejected because it specified an invalid type definition.</li>
     *         <li>ResourceNotFoundException The request was rejected because it attempted to reference a resource that
     *         does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudSearchAsyncClient.DefineExpression
     */
    @Override
    public CompletableFuture<DefineExpressionResponse> defineExpression(DefineExpressionRequest defineExpressionRequest) {
        try {

            HttpResponseHandler<DefineExpressionResponse> responseHandler = protocolFactory
                    .createResponseHandler(DefineExpressionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<DefineExpressionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DefineExpressionRequest, DefineExpressionResponse>()
                            .withOperationName("DefineExpression")
                            .withMarshaller(new DefineExpressionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(defineExpressionRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Configures an <code><a>IndexField</a></code> for the search domain. Used to create new fields and modify existing
     * ones. You must specify the name of the domain you are configuring and an index field configuration. The index
     * field configuration specifies a unique name, the index field type, and the options you want to configure for the
     * field. The options you can specify depend on the <code><a>IndexFieldType</a></code>. If the field exists, the new
     * configuration replaces the old one. For more information, see <a
     * href="http://docs.aws.amazon.com/cloudsearch/latest/developerguide/configuring-index-fields.html"
     * target="_blank">Configuring Index Fields</a> in the <i>Amazon CloudSearch Developer Guide</i>.
     * </p>
     *
     * @param defineIndexFieldRequest
     *        Container for the parameters to the <code><a>DefineIndexField</a></code> operation. Specifies the name of
     *        the domain you want to update and the index field configuration.
     * @return A Java Future containing the result of the DefineIndexField operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>InternalException An internal error occurred while processing the request. If this problem persists,
     *         report an issue from the <a href="http://status.aws.amazon.com/" target="_blank">Service Health
     *         Dashboard</a>.</li>
     *         <li>LimitExceededException The request was rejected because a resource limit has already been met.</li>
     *         <li>InvalidTypeException The request was rejected because it specified an invalid type definition.</li>
     *         <li>ResourceNotFoundException The request was rejected because it attempted to reference a resource that
     *         does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudSearchAsyncClient.DefineIndexField
     */
    @Override
    public CompletableFuture<DefineIndexFieldResponse> defineIndexField(DefineIndexFieldRequest defineIndexFieldRequest) {
        try {

            HttpResponseHandler<DefineIndexFieldResponse> responseHandler = protocolFactory
                    .createResponseHandler(DefineIndexFieldResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<DefineIndexFieldResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DefineIndexFieldRequest, DefineIndexFieldResponse>()
                            .withOperationName("DefineIndexField")
                            .withMarshaller(new DefineIndexFieldRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(defineIndexFieldRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Configures a suggester for a domain. A suggester enables you to display possible matches before users finish
     * typing their queries. When you configure a suggester, you must specify the name of the text field you want to
     * search for possible matches and a unique name for the suggester. For more information, see <a
     * href="http://docs.aws.amazon.com/cloudsearch/latest/developerguide/getting-suggestions.html"
     * target="_blank">Getting Search Suggestions</a> in the <i>Amazon CloudSearch Developer Guide</i>.
     * </p>
     *
     * @param defineSuggesterRequest
     *        Container for the parameters to the <code><a>DefineSuggester</a></code> operation. Specifies the name of
     *        the domain you want to update and the suggester configuration.
     * @return A Java Future containing the result of the DefineSuggester operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>InternalException An internal error occurred while processing the request. If this problem persists,
     *         report an issue from the <a href="http://status.aws.amazon.com/" target="_blank">Service Health
     *         Dashboard</a>.</li>
     *         <li>LimitExceededException The request was rejected because a resource limit has already been met.</li>
     *         <li>InvalidTypeException The request was rejected because it specified an invalid type definition.</li>
     *         <li>ResourceNotFoundException The request was rejected because it attempted to reference a resource that
     *         does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudSearchAsyncClient.DefineSuggester
     */
    @Override
    public CompletableFuture<DefineSuggesterResponse> defineSuggester(DefineSuggesterRequest defineSuggesterRequest) {
        try {

            HttpResponseHandler<DefineSuggesterResponse> responseHandler = protocolFactory
                    .createResponseHandler(DefineSuggesterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<DefineSuggesterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DefineSuggesterRequest, DefineSuggesterResponse>()
                            .withOperationName("DefineSuggester")
                            .withMarshaller(new DefineSuggesterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(defineSuggesterRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an analysis scheme. For more information, see <a
     * href="http://docs.aws.amazon.com/cloudsearch/latest/developerguide/configuring-analysis-schemes.html"
     * target="_blank">Configuring Analysis Schemes</a> in the <i>Amazon CloudSearch Developer Guide</i>.
     * </p>
     *
     * @param deleteAnalysisSchemeRequest
     *        Container for the parameters to the <code><a>DeleteAnalysisScheme</a></code> operation. Specifies the name
     *        of the domain you want to update and the analysis scheme you want to delete.
     * @return A Java Future containing the result of the DeleteAnalysisScheme operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>InternalException An internal error occurred while processing the request. If this problem persists,
     *         report an issue from the <a href="http://status.aws.amazon.com/" target="_blank">Service Health
     *         Dashboard</a>.</li>
     *         <li>InvalidTypeException The request was rejected because it specified an invalid type definition.</li>
     *         <li>ResourceNotFoundException The request was rejected because it attempted to reference a resource that
     *         does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudSearchAsyncClient.DeleteAnalysisScheme
     */
    @Override
    public CompletableFuture<DeleteAnalysisSchemeResponse> deleteAnalysisScheme(
            DeleteAnalysisSchemeRequest deleteAnalysisSchemeRequest) {
        try {

            HttpResponseHandler<DeleteAnalysisSchemeResponse> responseHandler = protocolFactory
                    .createResponseHandler(DeleteAnalysisSchemeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<DeleteAnalysisSchemeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteAnalysisSchemeRequest, DeleteAnalysisSchemeResponse>()
                            .withOperationName("DeleteAnalysisScheme")
                            .withMarshaller(new DeleteAnalysisSchemeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteAnalysisSchemeRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Permanently deletes a search domain and all of its data. Once a domain has been deleted, it cannot be recovered.
     * For more information, see <a
     * href="http://docs.aws.amazon.com/cloudsearch/latest/developerguide/deleting-domains.html"
     * target="_blank">Deleting a Search Domain</a> in the <i>Amazon CloudSearch Developer Guide</i>.
     * </p>
     *
     * @param deleteDomainRequest
     *        Container for the parameters to the <code><a>DeleteDomain</a></code> operation. Specifies the name of the
     *        domain you want to delete.
     * @return A Java Future containing the result of the DeleteDomain operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>InternalException An internal error occurred while processing the request. If this problem persists,
     *         report an issue from the <a href="http://status.aws.amazon.com/" target="_blank">Service Health
     *         Dashboard</a>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudSearchAsyncClient.DeleteDomain
     */
    @Override
    public CompletableFuture<DeleteDomainResponse> deleteDomain(DeleteDomainRequest deleteDomainRequest) {
        try {

            HttpResponseHandler<DeleteDomainResponse> responseHandler = protocolFactory
                    .createResponseHandler(DeleteDomainResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<DeleteDomainResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteDomainRequest, DeleteDomainResponse>()
                            .withOperationName("DeleteDomain").withMarshaller(new DeleteDomainRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteDomainRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes an <code><a>Expression</a></code> from the search domain. For more information, see <a
     * href="http://docs.aws.amazon.com/cloudsearch/latest/developerguide/configuring-expressions.html"
     * target="_blank">Configuring Expressions</a> in the <i>Amazon CloudSearch Developer Guide</i>.
     * </p>
     *
     * @param deleteExpressionRequest
     *        Container for the parameters to the <code><a>DeleteExpression</a></code> operation. Specifies the name of
     *        the domain you want to update and the name of the expression you want to delete.
     * @return A Java Future containing the result of the DeleteExpression operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>InternalException An internal error occurred while processing the request. If this problem persists,
     *         report an issue from the <a href="http://status.aws.amazon.com/" target="_blank">Service Health
     *         Dashboard</a>.</li>
     *         <li>InvalidTypeException The request was rejected because it specified an invalid type definition.</li>
     *         <li>ResourceNotFoundException The request was rejected because it attempted to reference a resource that
     *         does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudSearchAsyncClient.DeleteExpression
     */
    @Override
    public CompletableFuture<DeleteExpressionResponse> deleteExpression(DeleteExpressionRequest deleteExpressionRequest) {
        try {

            HttpResponseHandler<DeleteExpressionResponse> responseHandler = protocolFactory
                    .createResponseHandler(DeleteExpressionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<DeleteExpressionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteExpressionRequest, DeleteExpressionResponse>()
                            .withOperationName("DeleteExpression")
                            .withMarshaller(new DeleteExpressionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteExpressionRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes an <code><a>IndexField</a></code> from the search domain. For more information, see <a
     * href="http://docs.aws.amazon.com/cloudsearch/latest/developerguide/configuring-index-fields.html"
     * target="_blank">Configuring Index Fields</a> in the <i>Amazon CloudSearch Developer Guide</i>.
     * </p>
     *
     * @param deleteIndexFieldRequest
     *        Container for the parameters to the <code><a>DeleteIndexField</a></code> operation. Specifies the name of
     *        the domain you want to update and the name of the index field you want to delete.
     * @return A Java Future containing the result of the DeleteIndexField operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>InternalException An internal error occurred while processing the request. If this problem persists,
     *         report an issue from the <a href="http://status.aws.amazon.com/" target="_blank">Service Health
     *         Dashboard</a>.</li>
     *         <li>InvalidTypeException The request was rejected because it specified an invalid type definition.</li>
     *         <li>ResourceNotFoundException The request was rejected because it attempted to reference a resource that
     *         does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudSearchAsyncClient.DeleteIndexField
     */
    @Override
    public CompletableFuture<DeleteIndexFieldResponse> deleteIndexField(DeleteIndexFieldRequest deleteIndexFieldRequest) {
        try {

            HttpResponseHandler<DeleteIndexFieldResponse> responseHandler = protocolFactory
                    .createResponseHandler(DeleteIndexFieldResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<DeleteIndexFieldResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteIndexFieldRequest, DeleteIndexFieldResponse>()
                            .withOperationName("DeleteIndexField")
                            .withMarshaller(new DeleteIndexFieldRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteIndexFieldRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a suggester. For more information, see <a
     * href="http://docs.aws.amazon.com/cloudsearch/latest/developerguide/getting-suggestions.html"
     * target="_blank">Getting Search Suggestions</a> in the <i>Amazon CloudSearch Developer Guide</i>.
     * </p>
     *
     * @param deleteSuggesterRequest
     *        Container for the parameters to the <code><a>DeleteSuggester</a></code> operation. Specifies the name of
     *        the domain you want to update and name of the suggester you want to delete.
     * @return A Java Future containing the result of the DeleteSuggester operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>InternalException An internal error occurred while processing the request. If this problem persists,
     *         report an issue from the <a href="http://status.aws.amazon.com/" target="_blank">Service Health
     *         Dashboard</a>.</li>
     *         <li>InvalidTypeException The request was rejected because it specified an invalid type definition.</li>
     *         <li>ResourceNotFoundException The request was rejected because it attempted to reference a resource that
     *         does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudSearchAsyncClient.DeleteSuggester
     */
    @Override
    public CompletableFuture<DeleteSuggesterResponse> deleteSuggester(DeleteSuggesterRequest deleteSuggesterRequest) {
        try {

            HttpResponseHandler<DeleteSuggesterResponse> responseHandler = protocolFactory
                    .createResponseHandler(DeleteSuggesterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<DeleteSuggesterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteSuggesterRequest, DeleteSuggesterResponse>()
                            .withOperationName("DeleteSuggester")
                            .withMarshaller(new DeleteSuggesterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteSuggesterRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the analysis schemes configured for a domain. An analysis scheme defines language-specific text processing
     * options for a <code>text</code> field. Can be limited to specific analysis schemes by name. By default, shows all
     * analysis schemes and includes any pending changes to the configuration. Set the <code>Deployed</code> option to
     * <code>true</code> to show the active configuration and exclude pending changes. For more information, see <a
     * href="http://docs.aws.amazon.com/cloudsearch/latest/developerguide/configuring-analysis-schemes.html"
     * target="_blank">Configuring Analysis Schemes</a> in the <i>Amazon CloudSearch Developer Guide</i>.
     * </p>
     *
     * @param describeAnalysisSchemesRequest
     *        Container for the parameters to the <code><a>DescribeAnalysisSchemes</a></code> operation. Specifies the
     *        name of the domain you want to describe. To limit the response to particular analysis schemes, specify the
     *        names of the analysis schemes you want to describe. To show the active configuration and exclude any
     *        pending changes, set the <code>Deployed</code> option to <code>true</code>.
     * @return A Java Future containing the result of the DescribeAnalysisSchemes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>InternalException An internal error occurred while processing the request. If this problem persists,
     *         report an issue from the <a href="http://status.aws.amazon.com/" target="_blank">Service Health
     *         Dashboard</a>.</li>
     *         <li>ResourceNotFoundException The request was rejected because it attempted to reference a resource that
     *         does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudSearchAsyncClient.DescribeAnalysisSchemes
     */
    @Override
    public CompletableFuture<DescribeAnalysisSchemesResponse> describeAnalysisSchemes(
            DescribeAnalysisSchemesRequest describeAnalysisSchemesRequest) {
        try {

            HttpResponseHandler<DescribeAnalysisSchemesResponse> responseHandler = protocolFactory
                    .createResponseHandler(DescribeAnalysisSchemesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<DescribeAnalysisSchemesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeAnalysisSchemesRequest, DescribeAnalysisSchemesResponse>()
                            .withOperationName("DescribeAnalysisSchemes")
                            .withMarshaller(new DescribeAnalysisSchemesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeAnalysisSchemesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the availability options configured for a domain. By default, shows the configuration with any pending
     * changes. Set the <code>Deployed</code> option to <code>true</code> to show the active configuration and exclude
     * pending changes. For more information, see <a
     * href="http://docs.aws.amazon.com/cloudsearch/latest/developerguide/configuring-availability-options.html"
     * target="_blank">Configuring Availability Options</a> in the <i>Amazon CloudSearch Developer Guide</i>.
     * </p>
     *
     * @param describeAvailabilityOptionsRequest
     *        Container for the parameters to the <code><a>DescribeAvailabilityOptions</a></code> operation. Specifies
     *        the name of the domain you want to describe. To show the active configuration and exclude any pending
     *        changes, set the Deployed option to <code>true</code>.
     * @return A Java Future containing the result of the DescribeAvailabilityOptions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>InternalException An internal error occurred while processing the request. If this problem persists,
     *         report an issue from the <a href="http://status.aws.amazon.com/" target="_blank">Service Health
     *         Dashboard</a>.</li>
     *         <li>InvalidTypeException The request was rejected because it specified an invalid type definition.</li>
     *         <li>LimitExceededException The request was rejected because a resource limit has already been met.</li>
     *         <li>ResourceNotFoundException The request was rejected because it attempted to reference a resource that
     *         does not exist.</li>
     *         <li>DisabledOperationException The request was rejected because it attempted an operation which is not
     *         enabled.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudSearchAsyncClient.DescribeAvailabilityOptions
     */
    @Override
    public CompletableFuture<DescribeAvailabilityOptionsResponse> describeAvailabilityOptions(
            DescribeAvailabilityOptionsRequest describeAvailabilityOptionsRequest) {
        try {

            HttpResponseHandler<DescribeAvailabilityOptionsResponse> responseHandler = protocolFactory
                    .createResponseHandler(DescribeAvailabilityOptionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<DescribeAvailabilityOptionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeAvailabilityOptionsRequest, DescribeAvailabilityOptionsResponse>()
                            .withOperationName("DescribeAvailabilityOptions")
                            .withMarshaller(new DescribeAvailabilityOptionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeAvailabilityOptionsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the domain's endpoint options, specifically whether all requests to the domain must arrive over HTTPS.
     * For more information, see <a
     * href="http://docs.aws.amazon.com/cloudsearch/latest/developerguide/configuring-domain-endpoint-options.html"
     * target="_blank">Configuring Domain Endpoint Options</a> in the <i>Amazon CloudSearch Developer Guide</i>.
     * </p>
     *
     * @param describeDomainEndpointOptionsRequest
     *        Container for the parameters to the <code><a>DescribeDomainEndpointOptions</a></code> operation. Specify
     *        the name of the domain you want to describe. To show the active configuration and exclude any pending
     *        changes, set the Deployed option to <code>true</code>.
     * @return A Java Future containing the result of the DescribeDomainEndpointOptions operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>InternalException An internal error occurred while processing the request. If this problem persists,
     *         report an issue from the <a href="http://status.aws.amazon.com/" target="_blank">Service Health
     *         Dashboard</a>.</li>
     *         <li>LimitExceededException The request was rejected because a resource limit has already been met.</li>
     *         <li>ResourceNotFoundException The request was rejected because it attempted to reference a resource that
     *         does not exist.</li>
     *         <li>DisabledOperationException The request was rejected because it attempted an operation which is not
     *         enabled.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudSearchAsyncClient.DescribeDomainEndpointOptions
     */
    @Override
    public CompletableFuture<DescribeDomainEndpointOptionsResponse> describeDomainEndpointOptions(
            DescribeDomainEndpointOptionsRequest describeDomainEndpointOptionsRequest) {
        try {

            HttpResponseHandler<DescribeDomainEndpointOptionsResponse> responseHandler = protocolFactory
                    .createResponseHandler(DescribeDomainEndpointOptionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<DescribeDomainEndpointOptionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeDomainEndpointOptionsRequest, DescribeDomainEndpointOptionsResponse>()
                            .withOperationName("DescribeDomainEndpointOptions")
                            .withMarshaller(new DescribeDomainEndpointOptionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeDomainEndpointOptionsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about the search domains owned by this account. Can be limited to specific domains. Shows all
     * domains by default. To get the number of searchable documents in a domain, use the console or submit a
     * <code>matchall</code> request to your domain's search endpoint:
     * <code>q=matchall&amp;amp;q.parser=structured&amp;amp;size=0</code>. For more information, see <a
     * href="http://docs.aws.amazon.com/cloudsearch/latest/developerguide/getting-domain-info.html"
     * target="_blank">Getting Information about a Search Domain</a> in the <i>Amazon CloudSearch Developer Guide</i>.
     * </p>
     *
     * @param describeDomainsRequest
     *        Container for the parameters to the <code><a>DescribeDomains</a></code> operation. By default shows the
     *        status of all domains. To restrict the response to particular domains, specify the names of the domains
     *        you want to describe.
     * @return A Java Future containing the result of the DescribeDomains operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>InternalException An internal error occurred while processing the request. If this problem persists,
     *         report an issue from the <a href="http://status.aws.amazon.com/" target="_blank">Service Health
     *         Dashboard</a>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudSearchAsyncClient.DescribeDomains
     */
    @Override
    public CompletableFuture<DescribeDomainsResponse> describeDomains(DescribeDomainsRequest describeDomainsRequest) {
        try {

            HttpResponseHandler<DescribeDomainsResponse> responseHandler = protocolFactory
                    .createResponseHandler(DescribeDomainsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<DescribeDomainsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeDomainsRequest, DescribeDomainsResponse>()
                            .withOperationName("DescribeDomains")
                            .withMarshaller(new DescribeDomainsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeDomainsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the expressions configured for the search domain. Can be limited to specific expressions by name. By
     * default, shows all expressions and includes any pending changes to the configuration. Set the
     * <code>Deployed</code> option to <code>true</code> to show the active configuration and exclude pending changes.
     * For more information, see <a
     * href="http://docs.aws.amazon.com/cloudsearch/latest/developerguide/configuring-expressions.html"
     * target="_blank">Configuring Expressions</a> in the <i>Amazon CloudSearch Developer Guide</i>.
     * </p>
     *
     * @param describeExpressionsRequest
     *        Container for the parameters to the <code><a>DescribeDomains</a></code> operation. Specifies the name of
     *        the domain you want to describe. To restrict the response to particular expressions, specify the names of
     *        the expressions you want to describe. To show the active configuration and exclude any pending changes,
     *        set the <code>Deployed</code> option to <code>true</code>.
     * @return A Java Future containing the result of the DescribeExpressions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>InternalException An internal error occurred while processing the request. If this problem persists,
     *         report an issue from the <a href="http://status.aws.amazon.com/" target="_blank">Service Health
     *         Dashboard</a>.</li>
     *         <li>ResourceNotFoundException The request was rejected because it attempted to reference a resource that
     *         does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudSearchAsyncClient.DescribeExpressions
     */
    @Override
    public CompletableFuture<DescribeExpressionsResponse> describeExpressions(
            DescribeExpressionsRequest describeExpressionsRequest) {
        try {

            HttpResponseHandler<DescribeExpressionsResponse> responseHandler = protocolFactory
                    .createResponseHandler(DescribeExpressionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<DescribeExpressionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeExpressionsRequest, DescribeExpressionsResponse>()
                            .withOperationName("DescribeExpressions")
                            .withMarshaller(new DescribeExpressionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeExpressionsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about the index fields configured for the search domain. Can be limited to specific fields by
     * name. By default, shows all fields and includes any pending changes to the configuration. Set the
     * <code>Deployed</code> option to <code>true</code> to show the active configuration and exclude pending changes.
     * For more information, see <a
     * href="http://docs.aws.amazon.com/cloudsearch/latest/developerguide/getting-domain-info.html"
     * target="_blank">Getting Domain Information</a> in the <i>Amazon CloudSearch Developer Guide</i>.
     * </p>
     *
     * @param describeIndexFieldsRequest
     *        Container for the parameters to the <code><a>DescribeIndexFields</a></code> operation. Specifies the name
     *        of the domain you want to describe. To restrict the response to particular index fields, specify the names
     *        of the index fields you want to describe. To show the active configuration and exclude any pending
     *        changes, set the <code>Deployed</code> option to <code>true</code>.
     * @return A Java Future containing the result of the DescribeIndexFields operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>InternalException An internal error occurred while processing the request. If this problem persists,
     *         report an issue from the <a href="http://status.aws.amazon.com/" target="_blank">Service Health
     *         Dashboard</a>.</li>
     *         <li>ResourceNotFoundException The request was rejected because it attempted to reference a resource that
     *         does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudSearchAsyncClient.DescribeIndexFields
     */
    @Override
    public CompletableFuture<DescribeIndexFieldsResponse> describeIndexFields(
            DescribeIndexFieldsRequest describeIndexFieldsRequest) {
        try {

            HttpResponseHandler<DescribeIndexFieldsResponse> responseHandler = protocolFactory
                    .createResponseHandler(DescribeIndexFieldsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<DescribeIndexFieldsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeIndexFieldsRequest, DescribeIndexFieldsResponse>()
                            .withOperationName("DescribeIndexFields")
                            .withMarshaller(new DescribeIndexFieldsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeIndexFieldsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the scaling parameters configured for a domain. A domain's scaling parameters specify the desired search
     * instance type and replication count. For more information, see <a
     * href="http://docs.aws.amazon.com/cloudsearch/latest/developerguide/configuring-scaling-options.html"
     * target="_blank">Configuring Scaling Options</a> in the <i>Amazon CloudSearch Developer Guide</i>.
     * </p>
     *
     * @param describeScalingParametersRequest
     *        Container for the parameters to the <code><a>DescribeScalingParameters</a></code> operation. Specifies the
     *        name of the domain you want to describe.
     * @return A Java Future containing the result of the DescribeScalingParameters operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>InternalException An internal error occurred while processing the request. If this problem persists,
     *         report an issue from the <a href="http://status.aws.amazon.com/" target="_blank">Service Health
     *         Dashboard</a>.</li>
     *         <li>ResourceNotFoundException The request was rejected because it attempted to reference a resource that
     *         does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudSearchAsyncClient.DescribeScalingParameters
     */
    @Override
    public CompletableFuture<DescribeScalingParametersResponse> describeScalingParameters(
            DescribeScalingParametersRequest describeScalingParametersRequest) {
        try {

            HttpResponseHandler<DescribeScalingParametersResponse> responseHandler = protocolFactory
                    .createResponseHandler(DescribeScalingParametersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<DescribeScalingParametersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeScalingParametersRequest, DescribeScalingParametersResponse>()
                            .withOperationName("DescribeScalingParameters")
                            .withMarshaller(new DescribeScalingParametersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeScalingParametersRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about the access policies that control access to the domain's document and search endpoints. By
     * default, shows the configuration with any pending changes. Set the <code>Deployed</code> option to
     * <code>true</code> to show the active configuration and exclude pending changes. For more information, see <a
     * href="http://docs.aws.amazon.com/cloudsearch/latest/developerguide/configuring-access.html"
     * target="_blank">Configuring Access for a Search Domain</a> in the <i>Amazon CloudSearch Developer Guide</i>.
     * </p>
     *
     * @param describeServiceAccessPoliciesRequest
     *        Container for the parameters to the <code><a>DescribeServiceAccessPolicies</a></code> operation. Specifies
     *        the name of the domain you want to describe. To show the active configuration and exclude any pending
     *        changes, set the <code>Deployed</code> option to <code>true</code>.
     * @return A Java Future containing the result of the DescribeServiceAccessPolicies operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>InternalException An internal error occurred while processing the request. If this problem persists,
     *         report an issue from the <a href="http://status.aws.amazon.com/" target="_blank">Service Health
     *         Dashboard</a>.</li>
     *         <li>ResourceNotFoundException The request was rejected because it attempted to reference a resource that
     *         does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudSearchAsyncClient.DescribeServiceAccessPolicies
     */
    @Override
    public CompletableFuture<DescribeServiceAccessPoliciesResponse> describeServiceAccessPolicies(
            DescribeServiceAccessPoliciesRequest describeServiceAccessPoliciesRequest) {
        try {

            HttpResponseHandler<DescribeServiceAccessPoliciesResponse> responseHandler = protocolFactory
                    .createResponseHandler(DescribeServiceAccessPoliciesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<DescribeServiceAccessPoliciesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeServiceAccessPoliciesRequest, DescribeServiceAccessPoliciesResponse>()
                            .withOperationName("DescribeServiceAccessPolicies")
                            .withMarshaller(new DescribeServiceAccessPoliciesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeServiceAccessPoliciesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the suggesters configured for a domain. A suggester enables you to display possible matches before users
     * finish typing their queries. Can be limited to specific suggesters by name. By default, shows all suggesters and
     * includes any pending changes to the configuration. Set the <code>Deployed</code> option to <code>true</code> to
     * show the active configuration and exclude pending changes. For more information, see <a
     * href="http://docs.aws.amazon.com/cloudsearch/latest/developerguide/getting-suggestions.html"
     * target="_blank">Getting Search Suggestions</a> in the <i>Amazon CloudSearch Developer Guide</i>.
     * </p>
     *
     * @param describeSuggestersRequest
     *        Container for the parameters to the <code><a>DescribeSuggester</a></code> operation. Specifies the name of
     *        the domain you want to describe. To restrict the response to particular suggesters, specify the names of
     *        the suggesters you want to describe. To show the active configuration and exclude any pending changes, set
     *        the <code>Deployed</code> option to <code>true</code>.
     * @return A Java Future containing the result of the DescribeSuggesters operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>InternalException An internal error occurred while processing the request. If this problem persists,
     *         report an issue from the <a href="http://status.aws.amazon.com/" target="_blank">Service Health
     *         Dashboard</a>.</li>
     *         <li>ResourceNotFoundException The request was rejected because it attempted to reference a resource that
     *         does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudSearchAsyncClient.DescribeSuggesters
     */
    @Override
    public CompletableFuture<DescribeSuggestersResponse> describeSuggesters(DescribeSuggestersRequest describeSuggestersRequest) {
        try {

            HttpResponseHandler<DescribeSuggestersResponse> responseHandler = protocolFactory
                    .createResponseHandler(DescribeSuggestersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<DescribeSuggestersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeSuggestersRequest, DescribeSuggestersResponse>()
                            .withOperationName("DescribeSuggesters")
                            .withMarshaller(new DescribeSuggestersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeSuggestersRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Tells the search domain to start indexing its documents using the latest indexing options. This operation must be
     * invoked to activate options whose <a>OptionStatus</a> is <code>RequiresIndexDocuments</code>.
     * </p>
     *
     * @param indexDocumentsRequest
     *        Container for the parameters to the <code><a>IndexDocuments</a></code> operation. Specifies the name of
     *        the domain you want to re-index.
     * @return A Java Future containing the result of the IndexDocuments operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>InternalException An internal error occurred while processing the request. If this problem persists,
     *         report an issue from the <a href="http://status.aws.amazon.com/" target="_blank">Service Health
     *         Dashboard</a>.</li>
     *         <li>ResourceNotFoundException The request was rejected because it attempted to reference a resource that
     *         does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudSearchAsyncClient.IndexDocuments
     */
    @Override
    public CompletableFuture<IndexDocumentsResponse> indexDocuments(IndexDocumentsRequest indexDocumentsRequest) {
        try {

            HttpResponseHandler<IndexDocumentsResponse> responseHandler = protocolFactory
                    .createResponseHandler(IndexDocumentsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<IndexDocumentsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<IndexDocumentsRequest, IndexDocumentsResponse>()
                            .withOperationName("IndexDocuments")
                            .withMarshaller(new IndexDocumentsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(indexDocumentsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all search domains owned by an account.
     * </p>
     *
     * @param listDomainNamesRequest
     * @return A Java Future containing the result of the ListDomainNames operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudSearchAsyncClient.ListDomainNames
     */
    @Override
    public CompletableFuture<ListDomainNamesResponse> listDomainNames(ListDomainNamesRequest listDomainNamesRequest) {
        try {

            HttpResponseHandler<ListDomainNamesResponse> responseHandler = protocolFactory
                    .createResponseHandler(ListDomainNamesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<ListDomainNamesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDomainNamesRequest, ListDomainNamesResponse>()
                            .withOperationName("ListDomainNames")
                            .withMarshaller(new ListDomainNamesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listDomainNamesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Configures the availability options for a domain. Enabling the Multi-AZ option expands an Amazon CloudSearch
     * domain to an additional Availability Zone in the same Region to increase fault tolerance in the event of a
     * service disruption. Changes to the Multi-AZ option can take about half an hour to become active. For more
     * information, see <a
     * href="http://docs.aws.amazon.com/cloudsearch/latest/developerguide/configuring-availability-options.html"
     * target="_blank">Configuring Availability Options</a> in the <i>Amazon CloudSearch Developer Guide</i>.
     * </p>
     *
     * @param updateAvailabilityOptionsRequest
     *        Container for the parameters to the <code><a>UpdateAvailabilityOptions</a></code> operation. Specifies the
     *        name of the domain you want to update and the Multi-AZ availability option.
     * @return A Java Future containing the result of the UpdateAvailabilityOptions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>InternalException An internal error occurred while processing the request. If this problem persists,
     *         report an issue from the <a href="http://status.aws.amazon.com/" target="_blank">Service Health
     *         Dashboard</a>.</li>
     *         <li>InvalidTypeException The request was rejected because it specified an invalid type definition.</li>
     *         <li>LimitExceededException The request was rejected because a resource limit has already been met.</li>
     *         <li>ResourceNotFoundException The request was rejected because it attempted to reference a resource that
     *         does not exist.</li>
     *         <li>DisabledOperationException The request was rejected because it attempted an operation which is not
     *         enabled.</li>
     *         <li>ValidationException The request was rejected because it has invalid parameters.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudSearchAsyncClient.UpdateAvailabilityOptions
     */
    @Override
    public CompletableFuture<UpdateAvailabilityOptionsResponse> updateAvailabilityOptions(
            UpdateAvailabilityOptionsRequest updateAvailabilityOptionsRequest) {
        try {

            HttpResponseHandler<UpdateAvailabilityOptionsResponse> responseHandler = protocolFactory
                    .createResponseHandler(UpdateAvailabilityOptionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<UpdateAvailabilityOptionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateAvailabilityOptionsRequest, UpdateAvailabilityOptionsResponse>()
                            .withOperationName("UpdateAvailabilityOptions")
                            .withMarshaller(new UpdateAvailabilityOptionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateAvailabilityOptionsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the domain's endpoint options, specifically whether all requests to the domain must arrive over HTTPS.
     * For more information, see <a
     * href="http://docs.aws.amazon.com/cloudsearch/latest/developerguide/configuring-domain-endpoint-options.html"
     * target="_blank">Configuring Domain Endpoint Options</a> in the <i>Amazon CloudSearch Developer Guide</i>.
     * </p>
     *
     * @param updateDomainEndpointOptionsRequest
     *        Container for the parameters to the <code><a>UpdateDomainEndpointOptions</a></code> operation. Specifies
     *        the name of the domain you want to update and the domain endpoint options.
     * @return A Java Future containing the result of the UpdateDomainEndpointOptions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>InternalException An internal error occurred while processing the request. If this problem persists,
     *         report an issue from the <a href="http://status.aws.amazon.com/" target="_blank">Service Health
     *         Dashboard</a>.</li>
     *         <li>InvalidTypeException The request was rejected because it specified an invalid type definition.</li>
     *         <li>LimitExceededException The request was rejected because a resource limit has already been met.</li>
     *         <li>ResourceNotFoundException The request was rejected because it attempted to reference a resource that
     *         does not exist.</li>
     *         <li>DisabledOperationException The request was rejected because it attempted an operation which is not
     *         enabled.</li>
     *         <li>ValidationException The request was rejected because it has invalid parameters.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudSearchAsyncClient.UpdateDomainEndpointOptions
     */
    @Override
    public CompletableFuture<UpdateDomainEndpointOptionsResponse> updateDomainEndpointOptions(
            UpdateDomainEndpointOptionsRequest updateDomainEndpointOptionsRequest) {
        try {

            HttpResponseHandler<UpdateDomainEndpointOptionsResponse> responseHandler = protocolFactory
                    .createResponseHandler(UpdateDomainEndpointOptionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<UpdateDomainEndpointOptionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateDomainEndpointOptionsRequest, UpdateDomainEndpointOptionsResponse>()
                            .withOperationName("UpdateDomainEndpointOptions")
                            .withMarshaller(new UpdateDomainEndpointOptionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateDomainEndpointOptionsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Configures scaling parameters for a domain. A domain's scaling parameters specify the desired search instance
     * type and replication count. Amazon CloudSearch will still automatically scale your domain based on the volume of
     * data and traffic, but not below the desired instance type and replication count. If the Multi-AZ option is
     * enabled, these values control the resources used per Availability Zone. For more information, see <a
     * href="http://docs.aws.amazon.com/cloudsearch/latest/developerguide/configuring-scaling-options.html"
     * target="_blank">Configuring Scaling Options</a> in the <i>Amazon CloudSearch Developer Guide</i>.
     * </p>
     *
     * @param updateScalingParametersRequest
     *        Container for the parameters to the <code><a>UpdateScalingParameters</a></code> operation. Specifies the
     *        name of the domain you want to update and the scaling parameters you want to configure.
     * @return A Java Future containing the result of the UpdateScalingParameters operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>InternalException An internal error occurred while processing the request. If this problem persists,
     *         report an issue from the <a href="http://status.aws.amazon.com/" target="_blank">Service Health
     *         Dashboard</a>.</li>
     *         <li>LimitExceededException The request was rejected because a resource limit has already been met.</li>
     *         <li>ResourceNotFoundException The request was rejected because it attempted to reference a resource that
     *         does not exist.</li>
     *         <li>InvalidTypeException The request was rejected because it specified an invalid type definition.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudSearchAsyncClient.UpdateScalingParameters
     */
    @Override
    public CompletableFuture<UpdateScalingParametersResponse> updateScalingParameters(
            UpdateScalingParametersRequest updateScalingParametersRequest) {
        try {

            HttpResponseHandler<UpdateScalingParametersResponse> responseHandler = protocolFactory
                    .createResponseHandler(UpdateScalingParametersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<UpdateScalingParametersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateScalingParametersRequest, UpdateScalingParametersResponse>()
                            .withOperationName("UpdateScalingParameters")
                            .withMarshaller(new UpdateScalingParametersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateScalingParametersRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Configures the access rules that control access to the domain's document and search endpoints. For more
     * information, see <a href="http://docs.aws.amazon.com/cloudsearch/latest/developerguide/configuring-access.html"
     * target="_blank"> Configuring Access for an Amazon CloudSearch Domain</a>.
     * </p>
     *
     * @param updateServiceAccessPoliciesRequest
     *        Container for the parameters to the <code><a>UpdateServiceAccessPolicies</a></code> operation. Specifies
     *        the name of the domain you want to update and the access rules you want to configure.
     * @return A Java Future containing the result of the UpdateServiceAccessPolicies operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BaseException An error occurred while processing the request.</li>
     *         <li>InternalException An internal error occurred while processing the request. If this problem persists,
     *         report an issue from the <a href="http://status.aws.amazon.com/" target="_blank">Service Health
     *         Dashboard</a>.</li>
     *         <li>LimitExceededException The request was rejected because a resource limit has already been met.</li>
     *         <li>ResourceNotFoundException The request was rejected because it attempted to reference a resource that
     *         does not exist.</li>
     *         <li>InvalidTypeException The request was rejected because it specified an invalid type definition.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudSearchAsyncClient.UpdateServiceAccessPolicies
     */
    @Override
    public CompletableFuture<UpdateServiceAccessPoliciesResponse> updateServiceAccessPolicies(
            UpdateServiceAccessPoliciesRequest updateServiceAccessPoliciesRequest) {
        try {

            HttpResponseHandler<UpdateServiceAccessPoliciesResponse> responseHandler = protocolFactory
                    .createResponseHandler(UpdateServiceAccessPoliciesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

            CompletableFuture<UpdateServiceAccessPoliciesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateServiceAccessPoliciesRequest, UpdateServiceAccessPoliciesResponse>()
                            .withOperationName("UpdateServiceAccessPolicies")
                            .withMarshaller(new UpdateServiceAccessPoliciesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateServiceAccessPoliciesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private AwsQueryProtocolFactory init() {
        return AwsQueryProtocolFactory
                .builder()
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DisabledAction")
                                .exceptionBuilderSupplier(DisabledOperationException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceeded")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidType")
                                .exceptionBuilderSupplier(InvalidTypeException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFound")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BaseException").exceptionBuilderSupplier(BaseException::builder)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalException")
                                .exceptionBuilderSupplier(InternalException::builder).httpStatusCode(500).build())
                .clientConfiguration(clientConfiguration).defaultServiceExceptionSupplier(CloudSearchException::builder).build();
    }

    private <T extends CloudSearchRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
