/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudfront.model;

import java.beans.Transient;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkBytes;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateFunctionRequest extends CloudFrontRequest implements
        ToCopyableBuilder<CreateFunctionRequest.Builder, CreateFunctionRequest> {
    private static final SdkField<String> NAME_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Name")
            .getter(getter(CreateFunctionRequest::name))
            .setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name")
                    .unmarshallLocationName("Name").build()).build();

    private static final SdkField<FunctionConfig> FUNCTION_CONFIG_FIELD = SdkField
            .<FunctionConfig> builder(MarshallingType.SDK_POJO)
            .memberName("FunctionConfig")
            .getter(getter(CreateFunctionRequest::functionConfig))
            .setter(setter(Builder::functionConfig))
            .constructor(FunctionConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FunctionConfig")
                    .unmarshallLocationName("FunctionConfig").build()).build();

    private static final SdkField<SdkBytes> FUNCTION_CODE_FIELD = SdkField
            .<SdkBytes> builder(MarshallingType.SDK_BYTES)
            .memberName("FunctionCode")
            .getter(getter(CreateFunctionRequest::functionCode))
            .setter(setter(Builder::functionCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FunctionCode")
                    .unmarshallLocationName("FunctionCode").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD,
            FUNCTION_CONFIG_FIELD, FUNCTION_CODE_FIELD));

    private final String name;

    private final FunctionConfig functionConfig;

    private final SdkBytes functionCode;

    private CreateFunctionRequest(BuilderImpl builder) {
        super(builder);
        this.name = builder.name;
        this.functionConfig = builder.functionConfig;
        this.functionCode = builder.functionCode;
    }

    /**
     * <p>
     * A name to identify the function.
     * </p>
     * 
     * @return A name to identify the function.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * Configuration information about the function, including an optional comment and the function’s runtime.
     * </p>
     * 
     * @return Configuration information about the function, including an optional comment and the function’s runtime.
     */
    public final FunctionConfig functionConfig() {
        return functionConfig;
    }

    /**
     * <p>
     * The function code. For more information about writing a CloudFront function, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/writing-function-code.html">Writing
     * function code for CloudFront Functions</a> in the <i>Amazon CloudFront Developer Guide</i>.
     * </p>
     * 
     * @return The function code. For more information about writing a CloudFront function, see <a
     *         href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/writing-function-code.html"
     *         >Writing function code for CloudFront Functions</a> in the <i>Amazon CloudFront Developer Guide</i>.
     */
    public final SdkBytes functionCode() {
        return functionCode;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(functionConfig());
        hashCode = 31 * hashCode + Objects.hashCode(functionCode());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateFunctionRequest)) {
            return false;
        }
        CreateFunctionRequest other = (CreateFunctionRequest) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(functionConfig(), other.functionConfig())
                && Objects.equals(functionCode(), other.functionCode());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateFunctionRequest").add("Name", name()).add("FunctionConfig", functionConfig())
                .add("FunctionCode", functionCode() == null ? null : "*** Sensitive Data Redacted ***").build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "FunctionConfig":
            return Optional.ofNullable(clazz.cast(functionConfig()));
        case "FunctionCode":
            return Optional.ofNullable(clazz.cast(functionCode()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateFunctionRequest, T> g) {
        return obj -> g.apply((CreateFunctionRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends CloudFrontRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateFunctionRequest> {
        /**
         * <p>
         * A name to identify the function.
         * </p>
         * 
         * @param name
         *        A name to identify the function.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * Configuration information about the function, including an optional comment and the function’s runtime.
         * </p>
         * 
         * @param functionConfig
         *        Configuration information about the function, including an optional comment and the function’s
         *        runtime.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder functionConfig(FunctionConfig functionConfig);

        /**
         * <p>
         * Configuration information about the function, including an optional comment and the function’s runtime.
         * </p>
         * This is a convenience that creates an instance of the {@link FunctionConfig.Builder} avoiding the need to
         * create one manually via {@link FunctionConfig#builder()}.
         *
         * When the {@link Consumer} completes, {@link FunctionConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #functionConfig(FunctionConfig)}.
         * 
         * @param functionConfig
         *        a consumer that will call methods on {@link FunctionConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #functionConfig(FunctionConfig)
         */
        default Builder functionConfig(Consumer<FunctionConfig.Builder> functionConfig) {
            return functionConfig(FunctionConfig.builder().applyMutation(functionConfig).build());
        }

        /**
         * <p>
         * The function code. For more information about writing a CloudFront function, see <a
         * href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/writing-function-code.html">Writing
         * function code for CloudFront Functions</a> in the <i>Amazon CloudFront Developer Guide</i>.
         * </p>
         * 
         * @param functionCode
         *        The function code. For more information about writing a CloudFront function, see <a
         *        href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/writing-function-code.html"
         *        >Writing function code for CloudFront Functions</a> in the <i>Amazon CloudFront Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder functionCode(SdkBytes functionCode);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends CloudFrontRequest.BuilderImpl implements Builder {
        private String name;

        private FunctionConfig functionConfig;

        private SdkBytes functionCode;

        private BuilderImpl() {
        }

        private BuilderImpl(CreateFunctionRequest model) {
            super(model);
            name(model.name);
            functionConfig(model.functionConfig);
            functionCode(model.functionCode);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        @Transient
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final FunctionConfig.Builder getFunctionConfig() {
            return functionConfig != null ? functionConfig.toBuilder() : null;
        }

        public final void setFunctionConfig(FunctionConfig.BuilderImpl functionConfig) {
            this.functionConfig = functionConfig != null ? functionConfig.build() : null;
        }

        @Override
        @Transient
        public final Builder functionConfig(FunctionConfig functionConfig) {
            this.functionConfig = functionConfig;
            return this;
        }

        public final ByteBuffer getFunctionCode() {
            return functionCode == null ? null : functionCode.asByteBuffer();
        }

        public final void setFunctionCode(ByteBuffer functionCode) {
            functionCode(functionCode == null ? null : SdkBytes.fromByteBuffer(functionCode));
        }

        @Override
        @Transient
        public final Builder functionCode(SdkBytes functionCode) {
            this.functionCode = functionCode;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateFunctionRequest build() {
            return new CreateFunctionRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
