/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudfront.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A complex type that lists the AWS accounts, if any, that you included in the <code>TrustedSigners</code> complex type
 * for this distribution. These are the accounts that you want to allow to create signed URLs for private content.
 * </p>
 * <p>
 * The <code>Signer</code> complex type lists the AWS account number of the trusted signer or <code>self</code> if the
 * signer is the AWS account that created the distribution. The <code>Signer</code> element also includes the IDs of any
 * active CloudFront key pairs that are associated with the trusted signer's AWS account. If no <code>KeyPairId</code>
 * element appears for a <code>Signer</code>, that signer can't create signed URLs.
 * </p>
 * <p>
 * For more information, see <a
 * href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/PrivateContent.html">Serving Private Content
 * through CloudFront</a> in the <i>Amazon CloudFront Developer Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ActiveTrustedSigners implements SdkPojo, Serializable,
        ToCopyableBuilder<ActiveTrustedSigners.Builder, ActiveTrustedSigners> {
    private static final SdkField<Boolean> ENABLED_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(ActiveTrustedSigners::enabled))
            .setter(setter(Builder::enabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Enabled")
                    .unmarshallLocationName("Enabled").build()).build();

    private static final SdkField<Integer> QUANTITY_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(ActiveTrustedSigners::quantity))
            .setter(setter(Builder::quantity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Quantity")
                    .unmarshallLocationName("Quantity").build()).build();

    private static final SdkField<List<Signer>> ITEMS_FIELD = SdkField
            .<List<Signer>> builder(MarshallingType.LIST)
            .getter(getter(ActiveTrustedSigners::items))
            .setter(setter(Builder::items))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Items")
                    .unmarshallLocationName("Items").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("Signer")
                            .memberFieldInfo(
                                    SdkField.<Signer> builder(MarshallingType.SDK_POJO)
                                            .constructor(Signer::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Signer").unmarshallLocationName("Signer").build()).build())
                            .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ENABLED_FIELD, QUANTITY_FIELD,
            ITEMS_FIELD));

    private static final long serialVersionUID = 1L;

    private final Boolean enabled;

    private final Integer quantity;

    private final List<Signer> items;

    private ActiveTrustedSigners(BuilderImpl builder) {
        this.enabled = builder.enabled;
        this.quantity = builder.quantity;
        this.items = builder.items;
    }

    /**
     * <p>
     * Enabled is <code>true</code> if any of the AWS accounts listed in the <code>TrustedSigners</code> complex type
     * for this distribution have active CloudFront key pairs. If not, <code>Enabled</code> is <code>false</code>.
     * </p>
     * 
     * @return Enabled is <code>true</code> if any of the AWS accounts listed in the <code>TrustedSigners</code> complex
     *         type for this distribution have active CloudFront key pairs. If not, <code>Enabled</code> is
     *         <code>false</code>.
     */
    public Boolean enabled() {
        return enabled;
    }

    /**
     * <p>
     * The number of trusted signers specified in the <code>TrustedSigners</code> complex type.
     * </p>
     * 
     * @return The number of trusted signers specified in the <code>TrustedSigners</code> complex type.
     */
    public Integer quantity() {
        return quantity;
    }

    /**
     * Returns true if the Items property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasItems() {
        return items != null && !(items instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A complex type that contains one <code>Signer</code> complex type for each trusted signer that is specified in
     * the <code>TrustedSigners</code> complex type.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasItems()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A complex type that contains one <code>Signer</code> complex type for each trusted signer that is
     *         specified in the <code>TrustedSigners</code> complex type.
     */
    public List<Signer> items() {
        return items;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(enabled());
        hashCode = 31 * hashCode + Objects.hashCode(quantity());
        hashCode = 31 * hashCode + Objects.hashCode(items());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ActiveTrustedSigners)) {
            return false;
        }
        ActiveTrustedSigners other = (ActiveTrustedSigners) obj;
        return Objects.equals(enabled(), other.enabled()) && Objects.equals(quantity(), other.quantity())
                && Objects.equals(items(), other.items());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("ActiveTrustedSigners").add("Enabled", enabled()).add("Quantity", quantity())
                .add("Items", items()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Enabled":
            return Optional.ofNullable(clazz.cast(enabled()));
        case "Quantity":
            return Optional.ofNullable(clazz.cast(quantity()));
        case "Items":
            return Optional.ofNullable(clazz.cast(items()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ActiveTrustedSigners, T> g) {
        return obj -> g.apply((ActiveTrustedSigners) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ActiveTrustedSigners> {
        /**
         * <p>
         * Enabled is <code>true</code> if any of the AWS accounts listed in the <code>TrustedSigners</code> complex
         * type for this distribution have active CloudFront key pairs. If not, <code>Enabled</code> is
         * <code>false</code>.
         * </p>
         * 
         * @param enabled
         *        Enabled is <code>true</code> if any of the AWS accounts listed in the <code>TrustedSigners</code>
         *        complex type for this distribution have active CloudFront key pairs. If not, <code>Enabled</code> is
         *        <code>false</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enabled(Boolean enabled);

        /**
         * <p>
         * The number of trusted signers specified in the <code>TrustedSigners</code> complex type.
         * </p>
         * 
         * @param quantity
         *        The number of trusted signers specified in the <code>TrustedSigners</code> complex type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder quantity(Integer quantity);

        /**
         * <p>
         * A complex type that contains one <code>Signer</code> complex type for each trusted signer that is specified
         * in the <code>TrustedSigners</code> complex type.
         * </p>
         * 
         * @param items
         *        A complex type that contains one <code>Signer</code> complex type for each trusted signer that is
         *        specified in the <code>TrustedSigners</code> complex type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder items(Collection<Signer> items);

        /**
         * <p>
         * A complex type that contains one <code>Signer</code> complex type for each trusted signer that is specified
         * in the <code>TrustedSigners</code> complex type.
         * </p>
         * 
         * @param items
         *        A complex type that contains one <code>Signer</code> complex type for each trusted signer that is
         *        specified in the <code>TrustedSigners</code> complex type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder items(Signer... items);

        /**
         * <p>
         * A complex type that contains one <code>Signer</code> complex type for each trusted signer that is specified
         * in the <code>TrustedSigners</code> complex type.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Signer>.Builder} avoiding the need to
         * create one manually via {@link List<Signer>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Signer>.Builder#build()} is called immediately and its
         * result is passed to {@link #items(List<Signer>)}.
         * 
         * @param items
         *        a consumer that will call methods on {@link List<Signer>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #items(List<Signer>)
         */
        Builder items(Consumer<Signer.Builder>... items);
    }

    static final class BuilderImpl implements Builder {
        private Boolean enabled;

        private Integer quantity;

        private List<Signer> items = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ActiveTrustedSigners model) {
            enabled(model.enabled);
            quantity(model.quantity);
            items(model.items);
        }

        public final Boolean getEnabled() {
            return enabled;
        }

        @Override
        public final Builder enabled(Boolean enabled) {
            this.enabled = enabled;
            return this;
        }

        public final void setEnabled(Boolean enabled) {
            this.enabled = enabled;
        }

        public final Integer getQuantity() {
            return quantity;
        }

        @Override
        public final Builder quantity(Integer quantity) {
            this.quantity = quantity;
            return this;
        }

        public final void setQuantity(Integer quantity) {
            this.quantity = quantity;
        }

        public final Collection<Signer.Builder> getItems() {
            return items != null ? items.stream().map(Signer::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder items(Collection<Signer> items) {
            this.items = SignerListCopier.copy(items);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder items(Signer... items) {
            items(Arrays.asList(items));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder items(Consumer<Signer.Builder>... items) {
            items(Stream.of(items).map(c -> Signer.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setItems(Collection<Signer.BuilderImpl> items) {
            this.items = SignerListCopier.copyFromBuilder(items);
        }

        @Override
        public ActiveTrustedSigners build() {
            return new ActiveTrustedSigners(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
