/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.clouddirectory.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Retrieves attributes within a facet that are associated with an object inside an <a>BatchRead</a> operation. For more
 * information, see <a>GetObjectAttributes</a> and <a>BatchReadRequest$Operations</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class BatchGetObjectAttributes implements SdkPojo, Serializable,
        ToCopyableBuilder<BatchGetObjectAttributes.Builder, BatchGetObjectAttributes> {
    private static final SdkField<ObjectReference> OBJECT_REFERENCE_FIELD = SdkField
            .<ObjectReference> builder(MarshallingType.SDK_POJO).memberName("ObjectReference")
            .getter(getter(BatchGetObjectAttributes::objectReference)).setter(setter(Builder::objectReference))
            .constructor(ObjectReference::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ObjectReference").build()).build();

    private static final SdkField<SchemaFacet> SCHEMA_FACET_FIELD = SdkField.<SchemaFacet> builder(MarshallingType.SDK_POJO)
            .memberName("SchemaFacet").getter(getter(BatchGetObjectAttributes::schemaFacet)).setter(setter(Builder::schemaFacet))
            .constructor(SchemaFacet::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SchemaFacet").build()).build();

    private static final SdkField<List<String>> ATTRIBUTE_NAMES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("AttributeNames")
            .getter(getter(BatchGetObjectAttributes::attributeNames))
            .setter(setter(Builder::attributeNames))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AttributeNames").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(OBJECT_REFERENCE_FIELD,
            SCHEMA_FACET_FIELD, ATTRIBUTE_NAMES_FIELD));

    private static final long serialVersionUID = 1L;

    private final ObjectReference objectReference;

    private final SchemaFacet schemaFacet;

    private final List<String> attributeNames;

    private BatchGetObjectAttributes(BuilderImpl builder) {
        this.objectReference = builder.objectReference;
        this.schemaFacet = builder.schemaFacet;
        this.attributeNames = builder.attributeNames;
    }

    /**
     * <p>
     * Reference that identifies the object whose attributes will be retrieved.
     * </p>
     * 
     * @return Reference that identifies the object whose attributes will be retrieved.
     */
    public final ObjectReference objectReference() {
        return objectReference;
    }

    /**
     * <p>
     * Identifier for the facet whose attributes will be retrieved. See <a>SchemaFacet</a> for details.
     * </p>
     * 
     * @return Identifier for the facet whose attributes will be retrieved. See <a>SchemaFacet</a> for details.
     */
    public final SchemaFacet schemaFacet() {
        return schemaFacet;
    }

    /**
     * Returns true if the AttributeNames property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasAttributeNames() {
        return attributeNames != null && !(attributeNames instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * List of attribute names whose values will be retrieved.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasAttributeNames()} to see if a value was sent in this field.
     * </p>
     * 
     * @return List of attribute names whose values will be retrieved.
     */
    public final List<String> attributeNames() {
        return attributeNames;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(objectReference());
        hashCode = 31 * hashCode + Objects.hashCode(schemaFacet());
        hashCode = 31 * hashCode + Objects.hashCode(hasAttributeNames() ? attributeNames() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof BatchGetObjectAttributes)) {
            return false;
        }
        BatchGetObjectAttributes other = (BatchGetObjectAttributes) obj;
        return Objects.equals(objectReference(), other.objectReference()) && Objects.equals(schemaFacet(), other.schemaFacet())
                && hasAttributeNames() == other.hasAttributeNames() && Objects.equals(attributeNames(), other.attributeNames());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("BatchGetObjectAttributes").add("ObjectReference", objectReference())
                .add("SchemaFacet", schemaFacet()).add("AttributeNames", hasAttributeNames() ? attributeNames() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ObjectReference":
            return Optional.ofNullable(clazz.cast(objectReference()));
        case "SchemaFacet":
            return Optional.ofNullable(clazz.cast(schemaFacet()));
        case "AttributeNames":
            return Optional.ofNullable(clazz.cast(attributeNames()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<BatchGetObjectAttributes, T> g) {
        return obj -> g.apply((BatchGetObjectAttributes) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, BatchGetObjectAttributes> {
        /**
         * <p>
         * Reference that identifies the object whose attributes will be retrieved.
         * </p>
         * 
         * @param objectReference
         *        Reference that identifies the object whose attributes will be retrieved.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder objectReference(ObjectReference objectReference);

        /**
         * <p>
         * Reference that identifies the object whose attributes will be retrieved.
         * </p>
         * This is a convenience that creates an instance of the {@link ObjectReference.Builder} avoiding the need to
         * create one manually via {@link ObjectReference#builder()}.
         *
         * When the {@link Consumer} completes, {@link ObjectReference.Builder#build()} is called immediately and its
         * result is passed to {@link #objectReference(ObjectReference)}.
         * 
         * @param objectReference
         *        a consumer that will call methods on {@link ObjectReference.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #objectReference(ObjectReference)
         */
        default Builder objectReference(Consumer<ObjectReference.Builder> objectReference) {
            return objectReference(ObjectReference.builder().applyMutation(objectReference).build());
        }

        /**
         * <p>
         * Identifier for the facet whose attributes will be retrieved. See <a>SchemaFacet</a> for details.
         * </p>
         * 
         * @param schemaFacet
         *        Identifier for the facet whose attributes will be retrieved. See <a>SchemaFacet</a> for details.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder schemaFacet(SchemaFacet schemaFacet);

        /**
         * <p>
         * Identifier for the facet whose attributes will be retrieved. See <a>SchemaFacet</a> for details.
         * </p>
         * This is a convenience that creates an instance of the {@link SchemaFacet.Builder} avoiding the need to create
         * one manually via {@link SchemaFacet#builder()}.
         *
         * When the {@link Consumer} completes, {@link SchemaFacet.Builder#build()} is called immediately and its result
         * is passed to {@link #schemaFacet(SchemaFacet)}.
         * 
         * @param schemaFacet
         *        a consumer that will call methods on {@link SchemaFacet.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #schemaFacet(SchemaFacet)
         */
        default Builder schemaFacet(Consumer<SchemaFacet.Builder> schemaFacet) {
            return schemaFacet(SchemaFacet.builder().applyMutation(schemaFacet).build());
        }

        /**
         * <p>
         * List of attribute names whose values will be retrieved.
         * </p>
         * 
         * @param attributeNames
         *        List of attribute names whose values will be retrieved.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributeNames(Collection<String> attributeNames);

        /**
         * <p>
         * List of attribute names whose values will be retrieved.
         * </p>
         * 
         * @param attributeNames
         *        List of attribute names whose values will be retrieved.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributeNames(String... attributeNames);
    }

    static final class BuilderImpl implements Builder {
        private ObjectReference objectReference;

        private SchemaFacet schemaFacet;

        private List<String> attributeNames = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(BatchGetObjectAttributes model) {
            objectReference(model.objectReference);
            schemaFacet(model.schemaFacet);
            attributeNames(model.attributeNames);
        }

        public final ObjectReference.Builder getObjectReference() {
            return objectReference != null ? objectReference.toBuilder() : null;
        }

        @Override
        public final Builder objectReference(ObjectReference objectReference) {
            this.objectReference = objectReference;
            return this;
        }

        public final void setObjectReference(ObjectReference.BuilderImpl objectReference) {
            this.objectReference = objectReference != null ? objectReference.build() : null;
        }

        public final SchemaFacet.Builder getSchemaFacet() {
            return schemaFacet != null ? schemaFacet.toBuilder() : null;
        }

        @Override
        public final Builder schemaFacet(SchemaFacet schemaFacet) {
            this.schemaFacet = schemaFacet;
            return this;
        }

        public final void setSchemaFacet(SchemaFacet.BuilderImpl schemaFacet) {
            this.schemaFacet = schemaFacet != null ? schemaFacet.build() : null;
        }

        public final Collection<String> getAttributeNames() {
            if (attributeNames instanceof SdkAutoConstructList) {
                return null;
            }
            return attributeNames;
        }

        @Override
        public final Builder attributeNames(Collection<String> attributeNames) {
            this.attributeNames = AttributeNameListCopier.copy(attributeNames);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder attributeNames(String... attributeNames) {
            attributeNames(Arrays.asList(attributeNames));
            return this;
        }

        public final void setAttributeNames(Collection<String> attributeNames) {
            this.attributeNames = AttributeNameListCopier.copy(attributeNames);
        }

        @Override
        public BatchGetObjectAttributes build() {
            return new BatchGetObjectAttributes(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
