/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.clouddirectory.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Used when a regular object exists in a <a>Directory</a> and you want to find all of the policies that are associated
 * with that object and the parent to that object.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PolicyToPath implements SdkPojo, Serializable, ToCopyableBuilder<PolicyToPath.Builder, PolicyToPath> {
    private static final SdkField<String> PATH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(PolicyToPath::path)).setter(setter(Builder::path))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Path").build()).build();

    private static final SdkField<List<PolicyAttachment>> POLICIES_FIELD = SdkField
            .<List<PolicyAttachment>> builder(MarshallingType.LIST)
            .getter(getter(PolicyToPath::policies))
            .setter(setter(Builder::policies))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Policies").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<PolicyAttachment> builder(MarshallingType.SDK_POJO)
                                            .constructor(PolicyAttachment::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(PATH_FIELD, POLICIES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String path;

    private final List<PolicyAttachment> policies;

    private PolicyToPath(BuilderImpl builder) {
        this.path = builder.path;
        this.policies = builder.policies;
    }

    /**
     * <p>
     * The path that is referenced from the root.
     * </p>
     * 
     * @return The path that is referenced from the root.
     */
    public String path() {
        return path;
    }

    /**
     * Returns true if the Policies property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasPolicies() {
        return policies != null && !(policies instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * List of policy objects.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasPolicies()} to see if a value was sent in this field.
     * </p>
     * 
     * @return List of policy objects.
     */
    public List<PolicyAttachment> policies() {
        return policies;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(path());
        hashCode = 31 * hashCode + Objects.hashCode(policies());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PolicyToPath)) {
            return false;
        }
        PolicyToPath other = (PolicyToPath) obj;
        return Objects.equals(path(), other.path()) && Objects.equals(policies(), other.policies());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("PolicyToPath").add("Path", path()).add("Policies", policies()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Path":
            return Optional.ofNullable(clazz.cast(path()));
        case "Policies":
            return Optional.ofNullable(clazz.cast(policies()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<PolicyToPath, T> g) {
        return obj -> g.apply((PolicyToPath) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, PolicyToPath> {
        /**
         * <p>
         * The path that is referenced from the root.
         * </p>
         * 
         * @param path
         *        The path that is referenced from the root.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder path(String path);

        /**
         * <p>
         * List of policy objects.
         * </p>
         * 
         * @param policies
         *        List of policy objects.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder policies(Collection<PolicyAttachment> policies);

        /**
         * <p>
         * List of policy objects.
         * </p>
         * 
         * @param policies
         *        List of policy objects.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder policies(PolicyAttachment... policies);

        /**
         * <p>
         * List of policy objects.
         * </p>
         * This is a convenience that creates an instance of the {@link List<PolicyAttachment>.Builder} avoiding the
         * need to create one manually via {@link List<PolicyAttachment>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<PolicyAttachment>.Builder#build()} is called immediately and
         * its result is passed to {@link #policies(List<PolicyAttachment>)}.
         * 
         * @param policies
         *        a consumer that will call methods on {@link List<PolicyAttachment>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #policies(List<PolicyAttachment>)
         */
        Builder policies(Consumer<PolicyAttachment.Builder>... policies);
    }

    static final class BuilderImpl implements Builder {
        private String path;

        private List<PolicyAttachment> policies = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(PolicyToPath model) {
            path(model.path);
            policies(model.policies);
        }

        public final String getPath() {
            return path;
        }

        @Override
        public final Builder path(String path) {
            this.path = path;
            return this;
        }

        public final void setPath(String path) {
            this.path = path;
        }

        public final Collection<PolicyAttachment.Builder> getPolicies() {
            return policies != null ? policies.stream().map(PolicyAttachment::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder policies(Collection<PolicyAttachment> policies) {
            this.policies = PolicyAttachmentListCopier.copy(policies);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder policies(PolicyAttachment... policies) {
            policies(Arrays.asList(policies));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder policies(Consumer<PolicyAttachment.Builder>... policies) {
            policies(Stream.of(policies).map(c -> PolicyAttachment.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setPolicies(Collection<PolicyAttachment.BuilderImpl> policies) {
            this.policies = PolicyAttachmentListCopier.copyFromBuilder(policies);
        }

        @Override
        public PolicyToPath build() {
            return new PolicyToPath(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
