/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.clouddirectory.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.CollectionUtils;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A facet attribute definition. See <a
 * href="https://docs.aws.amazon.com/clouddirectory/latest/developerguide/schemas_attributereferences.html">Attribute
 * References</a> for more information.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class FacetAttributeDefinition implements SdkPojo, Serializable,
        ToCopyableBuilder<FacetAttributeDefinition.Builder, FacetAttributeDefinition> {
    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(FacetAttributeDefinition::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Type").build()).build();

    private static final SdkField<TypedAttributeValue> DEFAULT_VALUE_FIELD = SdkField
            .<TypedAttributeValue> builder(MarshallingType.SDK_POJO).getter(getter(FacetAttributeDefinition::defaultValue))
            .setter(setter(Builder::defaultValue)).constructor(TypedAttributeValue::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DefaultValue").build()).build();

    private static final SdkField<Boolean> IS_IMMUTABLE_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(FacetAttributeDefinition::isImmutable)).setter(setter(Builder::isImmutable))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IsImmutable").build()).build();

    private static final SdkField<Map<String, Rule>> RULES_FIELD = SdkField
            .<Map<String, Rule>> builder(MarshallingType.MAP)
            .getter(getter(FacetAttributeDefinition::rules))
            .setter(setter(Builder::rules))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Rules").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<Rule> builder(MarshallingType.SDK_POJO)
                                            .constructor(Rule::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TYPE_FIELD,
            DEFAULT_VALUE_FIELD, IS_IMMUTABLE_FIELD, RULES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String type;

    private final TypedAttributeValue defaultValue;

    private final Boolean isImmutable;

    private final Map<String, Rule> rules;

    private FacetAttributeDefinition(BuilderImpl builder) {
        this.type = builder.type;
        this.defaultValue = builder.defaultValue;
        this.isImmutable = builder.isImmutable;
        this.rules = builder.rules;
    }

    /**
     * <p>
     * The type of the attribute.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link FacetAttributeType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of the attribute.
     * @see FacetAttributeType
     */
    public FacetAttributeType type() {
        return FacetAttributeType.fromValue(type);
    }

    /**
     * <p>
     * The type of the attribute.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link FacetAttributeType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of the attribute.
     * @see FacetAttributeType
     */
    public String typeAsString() {
        return type;
    }

    /**
     * <p>
     * The default value of the attribute (if configured).
     * </p>
     * 
     * @return The default value of the attribute (if configured).
     */
    public TypedAttributeValue defaultValue() {
        return defaultValue;
    }

    /**
     * <p>
     * Whether the attribute is mutable or not.
     * </p>
     * 
     * @return Whether the attribute is mutable or not.
     */
    public Boolean isImmutable() {
        return isImmutable;
    }

    /**
     * Returns true if the Rules property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasRules() {
        return rules != null && !(rules instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Validation rules attached to the attribute definition.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasRules()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Validation rules attached to the attribute definition.
     */
    public Map<String, Rule> rules() {
        return rules;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(defaultValue());
        hashCode = 31 * hashCode + Objects.hashCode(isImmutable());
        hashCode = 31 * hashCode + Objects.hashCode(rules());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof FacetAttributeDefinition)) {
            return false;
        }
        FacetAttributeDefinition other = (FacetAttributeDefinition) obj;
        return Objects.equals(typeAsString(), other.typeAsString()) && Objects.equals(defaultValue(), other.defaultValue())
                && Objects.equals(isImmutable(), other.isImmutable()) && Objects.equals(rules(), other.rules());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("FacetAttributeDefinition").add("Type", typeAsString()).add("DefaultValue", defaultValue())
                .add("IsImmutable", isImmutable()).add("Rules", rules()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "DefaultValue":
            return Optional.ofNullable(clazz.cast(defaultValue()));
        case "IsImmutable":
            return Optional.ofNullable(clazz.cast(isImmutable()));
        case "Rules":
            return Optional.ofNullable(clazz.cast(rules()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<FacetAttributeDefinition, T> g) {
        return obj -> g.apply((FacetAttributeDefinition) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, FacetAttributeDefinition> {
        /**
         * <p>
         * The type of the attribute.
         * </p>
         * 
         * @param type
         *        The type of the attribute.
         * @see FacetAttributeType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FacetAttributeType
         */
        Builder type(String type);

        /**
         * <p>
         * The type of the attribute.
         * </p>
         * 
         * @param type
         *        The type of the attribute.
         * @see FacetAttributeType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FacetAttributeType
         */
        Builder type(FacetAttributeType type);

        /**
         * <p>
         * The default value of the attribute (if configured).
         * </p>
         * 
         * @param defaultValue
         *        The default value of the attribute (if configured).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder defaultValue(TypedAttributeValue defaultValue);

        /**
         * <p>
         * The default value of the attribute (if configured).
         * </p>
         * This is a convenience that creates an instance of the {@link TypedAttributeValue.Builder} avoiding the need
         * to create one manually via {@link TypedAttributeValue#builder()}.
         *
         * When the {@link Consumer} completes, {@link TypedAttributeValue.Builder#build()} is called immediately and
         * its result is passed to {@link #defaultValue(TypedAttributeValue)}.
         * 
         * @param defaultValue
         *        a consumer that will call methods on {@link TypedAttributeValue.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #defaultValue(TypedAttributeValue)
         */
        default Builder defaultValue(Consumer<TypedAttributeValue.Builder> defaultValue) {
            return defaultValue(TypedAttributeValue.builder().applyMutation(defaultValue).build());
        }

        /**
         * <p>
         * Whether the attribute is mutable or not.
         * </p>
         * 
         * @param isImmutable
         *        Whether the attribute is mutable or not.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder isImmutable(Boolean isImmutable);

        /**
         * <p>
         * Validation rules attached to the attribute definition.
         * </p>
         * 
         * @param rules
         *        Validation rules attached to the attribute definition.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rules(Map<String, Rule> rules);
    }

    static final class BuilderImpl implements Builder {
        private String type;

        private TypedAttributeValue defaultValue;

        private Boolean isImmutable;

        private Map<String, Rule> rules = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(FacetAttributeDefinition model) {
            type(model.type);
            defaultValue(model.defaultValue);
            isImmutable(model.isImmutable);
            rules(model.rules);
        }

        public final String getType() {
            return type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(FacetAttributeType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final void setType(String type) {
            this.type = type;
        }

        public final TypedAttributeValue.Builder getDefaultValue() {
            return defaultValue != null ? defaultValue.toBuilder() : null;
        }

        @Override
        public final Builder defaultValue(TypedAttributeValue defaultValue) {
            this.defaultValue = defaultValue;
            return this;
        }

        public final void setDefaultValue(TypedAttributeValue.BuilderImpl defaultValue) {
            this.defaultValue = defaultValue != null ? defaultValue.build() : null;
        }

        public final Boolean getIsImmutable() {
            return isImmutable;
        }

        @Override
        public final Builder isImmutable(Boolean isImmutable) {
            this.isImmutable = isImmutable;
            return this;
        }

        public final void setIsImmutable(Boolean isImmutable) {
            this.isImmutable = isImmutable;
        }

        public final Map<String, Rule.Builder> getRules() {
            return rules != null ? CollectionUtils.mapValues(rules, Rule::toBuilder) : null;
        }

        @Override
        public final Builder rules(Map<String, Rule> rules) {
            this.rules = RuleMapCopier.copy(rules);
            return this;
        }

        public final void setRules(Map<String, Rule.BuilderImpl> rules) {
            this.rules = RuleMapCopier.copyFromBuilder(rules);
        }

        @Override
        public FacetAttributeDefinition build() {
            return new FacetAttributeDefinition(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
