/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.chime;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.chime.internal.ChimeServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.chime.internal.ServiceVersionInfo;
import software.amazon.awssdk.services.chime.model.AccessDeniedException;
import software.amazon.awssdk.services.chime.model.AssociatePhoneNumberWithUserRequest;
import software.amazon.awssdk.services.chime.model.AssociatePhoneNumberWithUserResponse;
import software.amazon.awssdk.services.chime.model.AssociateSigninDelegateGroupsWithAccountRequest;
import software.amazon.awssdk.services.chime.model.AssociateSigninDelegateGroupsWithAccountResponse;
import software.amazon.awssdk.services.chime.model.BadRequestException;
import software.amazon.awssdk.services.chime.model.BatchCreateRoomMembershipRequest;
import software.amazon.awssdk.services.chime.model.BatchCreateRoomMembershipResponse;
import software.amazon.awssdk.services.chime.model.BatchDeletePhoneNumberRequest;
import software.amazon.awssdk.services.chime.model.BatchDeletePhoneNumberResponse;
import software.amazon.awssdk.services.chime.model.BatchSuspendUserRequest;
import software.amazon.awssdk.services.chime.model.BatchSuspendUserResponse;
import software.amazon.awssdk.services.chime.model.BatchUnsuspendUserRequest;
import software.amazon.awssdk.services.chime.model.BatchUnsuspendUserResponse;
import software.amazon.awssdk.services.chime.model.BatchUpdatePhoneNumberRequest;
import software.amazon.awssdk.services.chime.model.BatchUpdatePhoneNumberResponse;
import software.amazon.awssdk.services.chime.model.BatchUpdateUserRequest;
import software.amazon.awssdk.services.chime.model.BatchUpdateUserResponse;
import software.amazon.awssdk.services.chime.model.ChimeException;
import software.amazon.awssdk.services.chime.model.ConflictException;
import software.amazon.awssdk.services.chime.model.CreateAccountRequest;
import software.amazon.awssdk.services.chime.model.CreateAccountResponse;
import software.amazon.awssdk.services.chime.model.CreateBotRequest;
import software.amazon.awssdk.services.chime.model.CreateBotResponse;
import software.amazon.awssdk.services.chime.model.CreateMeetingDialOutRequest;
import software.amazon.awssdk.services.chime.model.CreateMeetingDialOutResponse;
import software.amazon.awssdk.services.chime.model.CreatePhoneNumberOrderRequest;
import software.amazon.awssdk.services.chime.model.CreatePhoneNumberOrderResponse;
import software.amazon.awssdk.services.chime.model.CreateRoomMembershipRequest;
import software.amazon.awssdk.services.chime.model.CreateRoomMembershipResponse;
import software.amazon.awssdk.services.chime.model.CreateRoomRequest;
import software.amazon.awssdk.services.chime.model.CreateRoomResponse;
import software.amazon.awssdk.services.chime.model.CreateUserRequest;
import software.amazon.awssdk.services.chime.model.CreateUserResponse;
import software.amazon.awssdk.services.chime.model.DeleteAccountRequest;
import software.amazon.awssdk.services.chime.model.DeleteAccountResponse;
import software.amazon.awssdk.services.chime.model.DeleteEventsConfigurationRequest;
import software.amazon.awssdk.services.chime.model.DeleteEventsConfigurationResponse;
import software.amazon.awssdk.services.chime.model.DeletePhoneNumberRequest;
import software.amazon.awssdk.services.chime.model.DeletePhoneNumberResponse;
import software.amazon.awssdk.services.chime.model.DeleteRoomMembershipRequest;
import software.amazon.awssdk.services.chime.model.DeleteRoomMembershipResponse;
import software.amazon.awssdk.services.chime.model.DeleteRoomRequest;
import software.amazon.awssdk.services.chime.model.DeleteRoomResponse;
import software.amazon.awssdk.services.chime.model.DisassociatePhoneNumberFromUserRequest;
import software.amazon.awssdk.services.chime.model.DisassociatePhoneNumberFromUserResponse;
import software.amazon.awssdk.services.chime.model.DisassociateSigninDelegateGroupsFromAccountRequest;
import software.amazon.awssdk.services.chime.model.DisassociateSigninDelegateGroupsFromAccountResponse;
import software.amazon.awssdk.services.chime.model.ForbiddenException;
import software.amazon.awssdk.services.chime.model.GetAccountRequest;
import software.amazon.awssdk.services.chime.model.GetAccountResponse;
import software.amazon.awssdk.services.chime.model.GetAccountSettingsRequest;
import software.amazon.awssdk.services.chime.model.GetAccountSettingsResponse;
import software.amazon.awssdk.services.chime.model.GetBotRequest;
import software.amazon.awssdk.services.chime.model.GetBotResponse;
import software.amazon.awssdk.services.chime.model.GetEventsConfigurationRequest;
import software.amazon.awssdk.services.chime.model.GetEventsConfigurationResponse;
import software.amazon.awssdk.services.chime.model.GetGlobalSettingsRequest;
import software.amazon.awssdk.services.chime.model.GetGlobalSettingsResponse;
import software.amazon.awssdk.services.chime.model.GetPhoneNumberOrderRequest;
import software.amazon.awssdk.services.chime.model.GetPhoneNumberOrderResponse;
import software.amazon.awssdk.services.chime.model.GetPhoneNumberRequest;
import software.amazon.awssdk.services.chime.model.GetPhoneNumberResponse;
import software.amazon.awssdk.services.chime.model.GetPhoneNumberSettingsRequest;
import software.amazon.awssdk.services.chime.model.GetPhoneNumberSettingsResponse;
import software.amazon.awssdk.services.chime.model.GetRetentionSettingsRequest;
import software.amazon.awssdk.services.chime.model.GetRetentionSettingsResponse;
import software.amazon.awssdk.services.chime.model.GetRoomRequest;
import software.amazon.awssdk.services.chime.model.GetRoomResponse;
import software.amazon.awssdk.services.chime.model.GetUserRequest;
import software.amazon.awssdk.services.chime.model.GetUserResponse;
import software.amazon.awssdk.services.chime.model.GetUserSettingsRequest;
import software.amazon.awssdk.services.chime.model.GetUserSettingsResponse;
import software.amazon.awssdk.services.chime.model.InviteUsersRequest;
import software.amazon.awssdk.services.chime.model.InviteUsersResponse;
import software.amazon.awssdk.services.chime.model.ListAccountsRequest;
import software.amazon.awssdk.services.chime.model.ListAccountsResponse;
import software.amazon.awssdk.services.chime.model.ListBotsRequest;
import software.amazon.awssdk.services.chime.model.ListBotsResponse;
import software.amazon.awssdk.services.chime.model.ListPhoneNumberOrdersRequest;
import software.amazon.awssdk.services.chime.model.ListPhoneNumberOrdersResponse;
import software.amazon.awssdk.services.chime.model.ListPhoneNumbersRequest;
import software.amazon.awssdk.services.chime.model.ListPhoneNumbersResponse;
import software.amazon.awssdk.services.chime.model.ListRoomMembershipsRequest;
import software.amazon.awssdk.services.chime.model.ListRoomMembershipsResponse;
import software.amazon.awssdk.services.chime.model.ListRoomsRequest;
import software.amazon.awssdk.services.chime.model.ListRoomsResponse;
import software.amazon.awssdk.services.chime.model.ListSupportedPhoneNumberCountriesRequest;
import software.amazon.awssdk.services.chime.model.ListSupportedPhoneNumberCountriesResponse;
import software.amazon.awssdk.services.chime.model.ListUsersRequest;
import software.amazon.awssdk.services.chime.model.ListUsersResponse;
import software.amazon.awssdk.services.chime.model.LogoutUserRequest;
import software.amazon.awssdk.services.chime.model.LogoutUserResponse;
import software.amazon.awssdk.services.chime.model.NotFoundException;
import software.amazon.awssdk.services.chime.model.PutEventsConfigurationRequest;
import software.amazon.awssdk.services.chime.model.PutEventsConfigurationResponse;
import software.amazon.awssdk.services.chime.model.PutRetentionSettingsRequest;
import software.amazon.awssdk.services.chime.model.PutRetentionSettingsResponse;
import software.amazon.awssdk.services.chime.model.RedactConversationMessageRequest;
import software.amazon.awssdk.services.chime.model.RedactConversationMessageResponse;
import software.amazon.awssdk.services.chime.model.RedactRoomMessageRequest;
import software.amazon.awssdk.services.chime.model.RedactRoomMessageResponse;
import software.amazon.awssdk.services.chime.model.RegenerateSecurityTokenRequest;
import software.amazon.awssdk.services.chime.model.RegenerateSecurityTokenResponse;
import software.amazon.awssdk.services.chime.model.ResetPersonalPinRequest;
import software.amazon.awssdk.services.chime.model.ResetPersonalPinResponse;
import software.amazon.awssdk.services.chime.model.ResourceLimitExceededException;
import software.amazon.awssdk.services.chime.model.RestorePhoneNumberRequest;
import software.amazon.awssdk.services.chime.model.RestorePhoneNumberResponse;
import software.amazon.awssdk.services.chime.model.SearchAvailablePhoneNumbersRequest;
import software.amazon.awssdk.services.chime.model.SearchAvailablePhoneNumbersResponse;
import software.amazon.awssdk.services.chime.model.ServiceFailureException;
import software.amazon.awssdk.services.chime.model.ServiceUnavailableException;
import software.amazon.awssdk.services.chime.model.ThrottledClientException;
import software.amazon.awssdk.services.chime.model.UnauthorizedClientException;
import software.amazon.awssdk.services.chime.model.UnprocessableEntityException;
import software.amazon.awssdk.services.chime.model.UpdateAccountRequest;
import software.amazon.awssdk.services.chime.model.UpdateAccountResponse;
import software.amazon.awssdk.services.chime.model.UpdateAccountSettingsRequest;
import software.amazon.awssdk.services.chime.model.UpdateAccountSettingsResponse;
import software.amazon.awssdk.services.chime.model.UpdateBotRequest;
import software.amazon.awssdk.services.chime.model.UpdateBotResponse;
import software.amazon.awssdk.services.chime.model.UpdateGlobalSettingsRequest;
import software.amazon.awssdk.services.chime.model.UpdateGlobalSettingsResponse;
import software.amazon.awssdk.services.chime.model.UpdatePhoneNumberRequest;
import software.amazon.awssdk.services.chime.model.UpdatePhoneNumberResponse;
import software.amazon.awssdk.services.chime.model.UpdatePhoneNumberSettingsRequest;
import software.amazon.awssdk.services.chime.model.UpdatePhoneNumberSettingsResponse;
import software.amazon.awssdk.services.chime.model.UpdateRoomMembershipRequest;
import software.amazon.awssdk.services.chime.model.UpdateRoomMembershipResponse;
import software.amazon.awssdk.services.chime.model.UpdateRoomRequest;
import software.amazon.awssdk.services.chime.model.UpdateRoomResponse;
import software.amazon.awssdk.services.chime.model.UpdateUserRequest;
import software.amazon.awssdk.services.chime.model.UpdateUserResponse;
import software.amazon.awssdk.services.chime.model.UpdateUserSettingsRequest;
import software.amazon.awssdk.services.chime.model.UpdateUserSettingsResponse;
import software.amazon.awssdk.services.chime.transform.AssociatePhoneNumberWithUserRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.AssociateSigninDelegateGroupsWithAccountRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.BatchCreateRoomMembershipRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.BatchDeletePhoneNumberRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.BatchSuspendUserRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.BatchUnsuspendUserRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.BatchUpdatePhoneNumberRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.BatchUpdateUserRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.CreateAccountRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.CreateBotRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.CreateMeetingDialOutRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.CreatePhoneNumberOrderRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.CreateRoomMembershipRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.CreateRoomRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.CreateUserRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.DeleteAccountRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.DeleteEventsConfigurationRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.DeletePhoneNumberRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.DeleteRoomMembershipRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.DeleteRoomRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.DisassociatePhoneNumberFromUserRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.DisassociateSigninDelegateGroupsFromAccountRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.GetAccountRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.GetAccountSettingsRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.GetBotRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.GetEventsConfigurationRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.GetGlobalSettingsRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.GetPhoneNumberOrderRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.GetPhoneNumberRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.GetPhoneNumberSettingsRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.GetRetentionSettingsRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.GetRoomRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.GetUserRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.GetUserSettingsRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.InviteUsersRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.ListAccountsRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.ListBotsRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.ListPhoneNumberOrdersRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.ListPhoneNumbersRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.ListRoomMembershipsRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.ListRoomsRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.ListSupportedPhoneNumberCountriesRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.ListUsersRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.LogoutUserRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.PutEventsConfigurationRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.PutRetentionSettingsRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.RedactConversationMessageRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.RedactRoomMessageRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.RegenerateSecurityTokenRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.ResetPersonalPinRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.RestorePhoneNumberRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.SearchAvailablePhoneNumbersRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.UpdateAccountRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.UpdateAccountSettingsRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.UpdateBotRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.UpdateGlobalSettingsRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.UpdatePhoneNumberRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.UpdatePhoneNumberSettingsRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.UpdateRoomMembershipRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.UpdateRoomRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.UpdateUserRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.UpdateUserSettingsRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link ChimeAsyncClient}.
 *
 * @see ChimeAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultChimeAsyncClient implements ChimeAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultChimeAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultChimeAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this)
                .option(SdkClientOption.API_METADATA, "Chime" + "#" + ServiceVersionInfo.VERSION).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Associates a phone number with the specified Amazon Chime user.
     * </p>
     *
     * @param associatePhoneNumberWithUserRequest
     * @return A Java Future containing the result of the AssociatePhoneNumberWithUser operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChimeAsyncClient.AssociatePhoneNumberWithUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/AssociatePhoneNumberWithUser"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AssociatePhoneNumberWithUserResponse> associatePhoneNumberWithUser(
            AssociatePhoneNumberWithUserRequest associatePhoneNumberWithUserRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(associatePhoneNumberWithUserRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associatePhoneNumberWithUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociatePhoneNumberWithUser");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AssociatePhoneNumberWithUserResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AssociatePhoneNumberWithUserResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "UnauthorizedClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceFailureException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "UnprocessableEntityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                            .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
                case "ThrottledClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledClientException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "ForbiddenException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                            .exceptionBuilderSupplier(ForbiddenException::builder).build());
                case "ResourceLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<AssociatePhoneNumberWithUserResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AssociatePhoneNumberWithUserRequest, AssociatePhoneNumberWithUserResponse>()
                            .withOperationName("AssociatePhoneNumberWithUser").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new AssociatePhoneNumberWithUserRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(associatePhoneNumberWithUserRequest));
            CompletableFuture<AssociatePhoneNumberWithUserResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Associates the specified sign-in delegate groups with the specified Amazon Chime account.
     * </p>
     *
     * @param associateSigninDelegateGroupsWithAccountRequest
     * @return A Java Future containing the result of the AssociateSigninDelegateGroupsWithAccount operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChimeAsyncClient.AssociateSigninDelegateGroupsWithAccount
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/AssociateSigninDelegateGroupsWithAccount"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AssociateSigninDelegateGroupsWithAccountResponse> associateSigninDelegateGroupsWithAccount(
            AssociateSigninDelegateGroupsWithAccountRequest associateSigninDelegateGroupsWithAccountRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(
                associateSigninDelegateGroupsWithAccountRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                associateSigninDelegateGroupsWithAccountRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateSigninDelegateGroupsWithAccount");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AssociateSigninDelegateGroupsWithAccountResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, AssociateSigninDelegateGroupsWithAccountResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "UnauthorizedClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceFailureException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "UnprocessableEntityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                            .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
                case "ThrottledClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledClientException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "ForbiddenException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                            .exceptionBuilderSupplier(ForbiddenException::builder).build());
                case "ResourceLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<AssociateSigninDelegateGroupsWithAccountResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AssociateSigninDelegateGroupsWithAccountRequest, AssociateSigninDelegateGroupsWithAccountResponse>()
                            .withOperationName("AssociateSigninDelegateGroupsWithAccount").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new AssociateSigninDelegateGroupsWithAccountRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(associateSigninDelegateGroupsWithAccountRequest));
            CompletableFuture<AssociateSigninDelegateGroupsWithAccountResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds up to 50 members to a chat room in an Amazon Chime Enterprise account. Members can be users or bots. The
     * member role designates whether the member is a chat room administrator or a general chat room member.
     * </p>
     *
     * @param batchCreateRoomMembershipRequest
     * @return A Java Future containing the result of the BatchCreateRoomMembership operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChimeAsyncClient.BatchCreateRoomMembership
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/BatchCreateRoomMembership"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchCreateRoomMembershipResponse> batchCreateRoomMembership(
            BatchCreateRoomMembershipRequest batchCreateRoomMembershipRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchCreateRoomMembershipRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchCreateRoomMembershipRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchCreateRoomMembership");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchCreateRoomMembershipResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchCreateRoomMembershipResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "UnauthorizedClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceFailureException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "UnprocessableEntityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                            .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
                case "ThrottledClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledClientException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "ForbiddenException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                            .exceptionBuilderSupplier(ForbiddenException::builder).build());
                case "ResourceLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<BatchCreateRoomMembershipResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchCreateRoomMembershipRequest, BatchCreateRoomMembershipResponse>()
                            .withOperationName("BatchCreateRoomMembership").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new BatchCreateRoomMembershipRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(batchCreateRoomMembershipRequest));
            CompletableFuture<BatchCreateRoomMembershipResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Moves phone numbers into the <b>Deletion queue</b>. Phone numbers must be disassociated from any users or Amazon
     * Chime Voice Connectors before they can be deleted.
     * </p>
     * <p>
     * Phone numbers remain in the <b>Deletion queue</b> for 7 days before they are deleted permanently.
     * </p>
     *
     * @param batchDeletePhoneNumberRequest
     * @return A Java Future containing the result of the BatchDeletePhoneNumber operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChimeAsyncClient.BatchDeletePhoneNumber
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/BatchDeletePhoneNumber" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchDeletePhoneNumberResponse> batchDeletePhoneNumber(
            BatchDeletePhoneNumberRequest batchDeletePhoneNumberRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchDeletePhoneNumberRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchDeletePhoneNumberRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchDeletePhoneNumber");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchDeletePhoneNumberResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchDeletePhoneNumberResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "UnauthorizedClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceFailureException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "UnprocessableEntityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                            .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
                case "ThrottledClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledClientException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "ForbiddenException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                            .exceptionBuilderSupplier(ForbiddenException::builder).build());
                case "ResourceLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<BatchDeletePhoneNumberResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchDeletePhoneNumberRequest, BatchDeletePhoneNumberResponse>()
                            .withOperationName("BatchDeletePhoneNumber").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new BatchDeletePhoneNumberRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(batchDeletePhoneNumberRequest));
            CompletableFuture<BatchDeletePhoneNumberResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Suspends up to 50 users from a <code>Team</code> or <code>EnterpriseLWA</code> Amazon Chime account. For more
     * information about different account types, see <a
     * href="https://docs.aws.amazon.com/chime/latest/ag/manage-chime-account.html">Managing Your Amazon Chime
     * Accounts</a> in the <i>Amazon Chime Administration Guide</i>.
     * </p>
     * <p>
     * Users suspended from a <code>Team</code> account are disassociated from the account,but they can continue to use
     * Amazon Chime as free users. To remove the suspension from suspended <code>Team</code> account users, invite them
     * to the <code>Team</code> account again. You can use the <a>InviteUsers</a> action to do so.
     * </p>
     * <p>
     * Users suspended from an <code>EnterpriseLWA</code> account are immediately signed out of Amazon Chime and can no
     * longer sign in. To remove the suspension from suspended <code>EnterpriseLWA</code> account users, use the
     * <a>BatchUnsuspendUser</a> action.
     * </p>
     * <p>
     * To sign out users without suspending them, use the <a>LogoutUser</a> action.
     * </p>
     *
     * @param batchSuspendUserRequest
     * @return A Java Future containing the result of the BatchSuspendUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChimeAsyncClient.BatchSuspendUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/BatchSuspendUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<BatchSuspendUserResponse> batchSuspendUser(BatchSuspendUserRequest batchSuspendUserRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchSuspendUserRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchSuspendUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchSuspendUser");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchSuspendUserResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchSuspendUserResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "UnauthorizedClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceFailureException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "UnprocessableEntityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                            .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
                case "ThrottledClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledClientException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "ForbiddenException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                            .exceptionBuilderSupplier(ForbiddenException::builder).build());
                case "ResourceLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<BatchSuspendUserResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchSuspendUserRequest, BatchSuspendUserResponse>()
                            .withOperationName("BatchSuspendUser").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new BatchSuspendUserRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(batchSuspendUserRequest));
            CompletableFuture<BatchSuspendUserResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the suspension from up to 50 previously suspended users for the specified Amazon Chime
     * <code>EnterpriseLWA</code> account. Only users on <code>EnterpriseLWA</code> accounts can be unsuspended using
     * this action. For more information about different account types, see <a
     * href="https://docs.aws.amazon.com/chime/latest/ag/manage-chime-account.html"> Managing Your Amazon Chime Accounts
     * </a> in the account types, in the <i>Amazon Chime Administration Guide</i>.
     * </p>
     * <p>
     * Previously suspended users who are unsuspended using this action are returned to <code>Registered</code> status.
     * Users who are not previously suspended are ignored.
     * </p>
     *
     * @param batchUnsuspendUserRequest
     * @return A Java Future containing the result of the BatchUnsuspendUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChimeAsyncClient.BatchUnsuspendUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/BatchUnsuspendUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<BatchUnsuspendUserResponse> batchUnsuspendUser(BatchUnsuspendUserRequest batchUnsuspendUserRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchUnsuspendUserRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchUnsuspendUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchUnsuspendUser");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchUnsuspendUserResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchUnsuspendUserResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "UnauthorizedClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceFailureException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "UnprocessableEntityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                            .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
                case "ThrottledClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledClientException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "ForbiddenException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                            .exceptionBuilderSupplier(ForbiddenException::builder).build());
                case "ResourceLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<BatchUnsuspendUserResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchUnsuspendUserRequest, BatchUnsuspendUserResponse>()
                            .withOperationName("BatchUnsuspendUser").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new BatchUnsuspendUserRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(batchUnsuspendUserRequest));
            CompletableFuture<BatchUnsuspendUserResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates phone number product types or calling names. You can update one attribute at a time for each
     * <code>UpdatePhoneNumberRequestItem</code>. For example, you can update the product type or the calling name.
     * </p>
     * <p>
     * For toll-free numbers, you cannot use the Amazon Chime Business Calling product type. For numbers outside the
     * U.S., you must use the Amazon Chime SIP Media Application Dial-In product type.
     * </p>
     * <p>
     * Updates to outbound calling names can take up to 72 hours to complete. Pending updates to outbound calling names
     * must be complete before you can request another update.
     * </p>
     *
     * @param batchUpdatePhoneNumberRequest
     * @return A Java Future containing the result of the BatchUpdatePhoneNumber operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChimeAsyncClient.BatchUpdatePhoneNumber
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/BatchUpdatePhoneNumber" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchUpdatePhoneNumberResponse> batchUpdatePhoneNumber(
            BatchUpdatePhoneNumberRequest batchUpdatePhoneNumberRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchUpdatePhoneNumberRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchUpdatePhoneNumberRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchUpdatePhoneNumber");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchUpdatePhoneNumberResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchUpdatePhoneNumberResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "UnauthorizedClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceFailureException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "UnprocessableEntityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                            .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
                case "ThrottledClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledClientException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "ForbiddenException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                            .exceptionBuilderSupplier(ForbiddenException::builder).build());
                case "ResourceLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<BatchUpdatePhoneNumberResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchUpdatePhoneNumberRequest, BatchUpdatePhoneNumberResponse>()
                            .withOperationName("BatchUpdatePhoneNumber").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new BatchUpdatePhoneNumberRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(batchUpdatePhoneNumberRequest));
            CompletableFuture<BatchUpdatePhoneNumberResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates user details within the <a>UpdateUserRequestItem</a> object for up to 20 users for the specified Amazon
     * Chime account. Currently, only <code>LicenseType</code> updates are supported for this action.
     * </p>
     *
     * @param batchUpdateUserRequest
     * @return A Java Future containing the result of the BatchUpdateUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChimeAsyncClient.BatchUpdateUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/BatchUpdateUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<BatchUpdateUserResponse> batchUpdateUser(BatchUpdateUserRequest batchUpdateUserRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchUpdateUserRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchUpdateUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchUpdateUser");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchUpdateUserResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchUpdateUserResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "UnauthorizedClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceFailureException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "UnprocessableEntityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                            .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
                case "ThrottledClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledClientException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "ForbiddenException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                            .exceptionBuilderSupplier(ForbiddenException::builder).build());
                case "ResourceLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<BatchUpdateUserResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchUpdateUserRequest, BatchUpdateUserResponse>()
                            .withOperationName("BatchUpdateUser").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new BatchUpdateUserRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(batchUpdateUserRequest));
            CompletableFuture<BatchUpdateUserResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an Amazon Chime account under the administrator's AWS account. Only <code>Team</code> account types are
     * currently supported for this action. For more information about different account types, see <a
     * href="https://docs.aws.amazon.com/chime/latest/ag/manage-chime-account.html">Managing Your Amazon Chime
     * Accounts</a> in the <i>Amazon Chime Administration Guide</i>.
     * </p>
     *
     * @param createAccountRequest
     * @return A Java Future containing the result of the CreateAccount operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChimeAsyncClient.CreateAccount
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/CreateAccount" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateAccountResponse> createAccount(CreateAccountRequest createAccountRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createAccountRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createAccountRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateAccount");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateAccountResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateAccountResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "UnauthorizedClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceFailureException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "UnprocessableEntityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                            .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
                case "ThrottledClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledClientException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "ForbiddenException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                            .exceptionBuilderSupplier(ForbiddenException::builder).build());
                case "ResourceLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<CreateAccountResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateAccountRequest, CreateAccountResponse>()
                            .withOperationName("CreateAccount").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateAccountRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createAccountRequest));
            CompletableFuture<CreateAccountResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a bot for an Amazon Chime Enterprise account.
     * </p>
     *
     * @param createBotRequest
     * @return A Java Future containing the result of the CreateBot operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ResourceLimitExceededException The request exceeds the resource limit.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChimeAsyncClient.CreateBot
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/CreateBot" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateBotResponse> createBot(CreateBotRequest createBotRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createBotRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createBotRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateBot");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateBotResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateBotResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "UnauthorizedClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceFailureException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "UnprocessableEntityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                            .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
                case "ThrottledClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledClientException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "ForbiddenException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                            .exceptionBuilderSupplier(ForbiddenException::builder).build());
                case "ResourceLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<CreateBotResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateBotRequest, CreateBotResponse>().withOperationName("CreateBot")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateBotRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(createBotRequest));
            CompletableFuture<CreateBotResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Uses the join token and call metadata in a meeting request (From number, To number, and so forth) to initiate an
     * outbound call to a public switched telephone network (PSTN) and join them into a Chime meeting. Also ensures that
     * the From number belongs to the customer.
     * </p>
     * <p>
     * To play welcome audio or implement an interactive voice response (IVR), use the
     * <code>CreateSipMediaApplicationCall</code> action with the corresponding SIP media application ID.
     * </p>
     * <important>
     * <p>
     * <b>This API is not available in a dedicated namespace.</b>
     * </p>
     * </important>
     *
     * @param createMeetingDialOutRequest
     * @return A Java Future containing the result of the CreateMeetingDialOut operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ResourceLimitExceededException The request exceeds the resource limit.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChimeAsyncClient.CreateMeetingDialOut
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/CreateMeetingDialOut" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateMeetingDialOutResponse> createMeetingDialOut(
            CreateMeetingDialOutRequest createMeetingDialOutRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createMeetingDialOutRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createMeetingDialOutRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateMeetingDialOut");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateMeetingDialOutResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateMeetingDialOutResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "UnauthorizedClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceFailureException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "UnprocessableEntityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                            .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
                case "ThrottledClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledClientException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "ForbiddenException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                            .exceptionBuilderSupplier(ForbiddenException::builder).build());
                case "ResourceLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<CreateMeetingDialOutResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateMeetingDialOutRequest, CreateMeetingDialOutResponse>()
                            .withOperationName("CreateMeetingDialOut").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateMeetingDialOutRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createMeetingDialOutRequest));
            CompletableFuture<CreateMeetingDialOutResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an order for phone numbers to be provisioned. For toll-free numbers, you cannot use the Amazon Chime
     * Business Calling product type. For numbers outside the U.S., you must use the Amazon Chime SIP Media Application
     * Dial-In product type.
     * </p>
     *
     * @param createPhoneNumberOrderRequest
     * @return A Java Future containing the result of the CreatePhoneNumberOrder operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ResourceLimitExceededException The request exceeds the resource limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChimeAsyncClient.CreatePhoneNumberOrder
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/CreatePhoneNumberOrder" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreatePhoneNumberOrderResponse> createPhoneNumberOrder(
            CreatePhoneNumberOrderRequest createPhoneNumberOrderRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createPhoneNumberOrderRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createPhoneNumberOrderRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreatePhoneNumberOrder");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreatePhoneNumberOrderResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreatePhoneNumberOrderResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "UnauthorizedClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceFailureException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "UnprocessableEntityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                            .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
                case "ThrottledClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledClientException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "ForbiddenException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                            .exceptionBuilderSupplier(ForbiddenException::builder).build());
                case "ResourceLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<CreatePhoneNumberOrderResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreatePhoneNumberOrderRequest, CreatePhoneNumberOrderResponse>()
                            .withOperationName("CreatePhoneNumberOrder").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreatePhoneNumberOrderRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createPhoneNumberOrderRequest));
            CompletableFuture<CreatePhoneNumberOrderResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a chat room for the specified Amazon Chime Enterprise account.
     * </p>
     *
     * @param createRoomRequest
     * @return A Java Future containing the result of the CreateRoom operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ResourceLimitExceededException The request exceeds the resource limit.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChimeAsyncClient.CreateRoom
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/CreateRoom" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateRoomResponse> createRoom(CreateRoomRequest createRoomRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createRoomRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createRoomRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateRoom");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateRoomResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateRoomResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "UnauthorizedClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceFailureException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "UnprocessableEntityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                            .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
                case "ThrottledClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledClientException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "ForbiddenException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                            .exceptionBuilderSupplier(ForbiddenException::builder).build());
                case "ResourceLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<CreateRoomResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateRoomRequest, CreateRoomResponse>().withOperationName("CreateRoom")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateRoomRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createRoomRequest));
            CompletableFuture<CreateRoomResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds a member to a chat room in an Amazon Chime Enterprise account. A member can be either a user or a bot. The
     * member role designates whether the member is a chat room administrator or a general chat room member.
     * </p>
     *
     * @param createRoomMembershipRequest
     * @return A Java Future containing the result of the CreateRoomMembership operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ResourceLimitExceededException The request exceeds the resource limit.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChimeAsyncClient.CreateRoomMembership
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/CreateRoomMembership" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateRoomMembershipResponse> createRoomMembership(
            CreateRoomMembershipRequest createRoomMembershipRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createRoomMembershipRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createRoomMembershipRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateRoomMembership");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateRoomMembershipResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateRoomMembershipResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "UnauthorizedClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceFailureException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "UnprocessableEntityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                            .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
                case "ThrottledClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledClientException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "ForbiddenException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                            .exceptionBuilderSupplier(ForbiddenException::builder).build());
                case "ResourceLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<CreateRoomMembershipResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateRoomMembershipRequest, CreateRoomMembershipResponse>()
                            .withOperationName("CreateRoomMembership").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateRoomMembershipRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createRoomMembershipRequest));
            CompletableFuture<CreateRoomMembershipResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a user under the specified Amazon Chime account.
     * </p>
     *
     * @param createUserRequest
     * @return A Java Future containing the result of the CreateUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChimeAsyncClient.CreateUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/CreateUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateUserResponse> createUser(CreateUserRequest createUserRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createUserRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateUser");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateUserResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "UnauthorizedClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceFailureException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "UnprocessableEntityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                            .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
                case "ThrottledClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledClientException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "ForbiddenException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                            .exceptionBuilderSupplier(ForbiddenException::builder).build());
                case "ResourceLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<CreateUserResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateUserRequest, CreateUserResponse>().withOperationName("CreateUser")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateUserRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createUserRequest));
            CompletableFuture<CreateUserResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified Amazon Chime account. You must suspend all users before deleting <code>Team</code> account.
     * You can use the <a>BatchSuspendUser</a> action to dodo.
     * </p>
     * <p>
     * For <code>EnterpriseLWA</code> and <code>EnterpriseAD</code> accounts, you must release the claimed domains for
     * your Amazon Chime account before deletion. As soon as you release the domain, all users under that account are
     * suspended.
     * </p>
     * <p>
     * Deleted accounts appear in your <code>Disabled</code> accounts list for 90 days. To restore deleted account from
     * your <code>Disabled</code> accounts list, you must contact AWS Support.
     * </p>
     * <p>
     * After 90 days, deleted accounts are permanently removed from your <code>Disabled</code> accounts list.
     * </p>
     *
     * @param deleteAccountRequest
     * @return A Java Future containing the result of the DeleteAccount operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>UnprocessableEntityException The request was well-formed but was unable to be followed due to
     *         semantic errors.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChimeAsyncClient.DeleteAccount
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/DeleteAccount" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteAccountResponse> deleteAccount(DeleteAccountRequest deleteAccountRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteAccountRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAccountRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAccount");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteAccountResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteAccountResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "UnauthorizedClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceFailureException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "UnprocessableEntityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                            .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
                case "ThrottledClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledClientException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "ForbiddenException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                            .exceptionBuilderSupplier(ForbiddenException::builder).build());
                case "ResourceLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeleteAccountResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteAccountRequest, DeleteAccountResponse>()
                            .withOperationName("DeleteAccount").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteAccountRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteAccountRequest));
            CompletableFuture<DeleteAccountResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the events configuration that allows a bot to receive outgoing events.
     * </p>
     *
     * @param deleteEventsConfigurationRequest
     * @return A Java Future containing the result of the DeleteEventsConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ResourceLimitExceededException The request exceeds the resource limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChimeAsyncClient.DeleteEventsConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/DeleteEventsConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteEventsConfigurationResponse> deleteEventsConfiguration(
            DeleteEventsConfigurationRequest deleteEventsConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteEventsConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteEventsConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteEventsConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteEventsConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteEventsConfigurationResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "UnauthorizedClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceFailureException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "UnprocessableEntityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                            .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
                case "ThrottledClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledClientException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "ForbiddenException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                            .exceptionBuilderSupplier(ForbiddenException::builder).build());
                case "ResourceLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeleteEventsConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteEventsConfigurationRequest, DeleteEventsConfigurationResponse>()
                            .withOperationName("DeleteEventsConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteEventsConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteEventsConfigurationRequest));
            CompletableFuture<DeleteEventsConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Moves the specified phone number into the <b>Deletion queue</b>. A phone number must be disassociated from any
     * users or Amazon Chime Voice Connectors before it can be deleted.
     * </p>
     * <p>
     * Deleted phone numbers remain in the <b>Deletion queue</b> for 7 days before they are deleted permanently.
     * </p>
     *
     * @param deletePhoneNumberRequest
     * @return A Java Future containing the result of the DeletePhoneNumber operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChimeAsyncClient.DeletePhoneNumber
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/DeletePhoneNumber" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeletePhoneNumberResponse> deletePhoneNumber(DeletePhoneNumberRequest deletePhoneNumberRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deletePhoneNumberRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deletePhoneNumberRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePhoneNumber");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeletePhoneNumberResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeletePhoneNumberResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "UnauthorizedClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceFailureException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "UnprocessableEntityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                            .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
                case "ThrottledClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledClientException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "ForbiddenException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                            .exceptionBuilderSupplier(ForbiddenException::builder).build());
                case "ResourceLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeletePhoneNumberResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeletePhoneNumberRequest, DeletePhoneNumberResponse>()
                            .withOperationName("DeletePhoneNumber").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeletePhoneNumberRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deletePhoneNumberRequest));
            CompletableFuture<DeletePhoneNumberResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a chat room in an Amazon Chime Enterprise account.
     * </p>
     *
     * @param deleteRoomRequest
     * @return A Java Future containing the result of the DeleteRoom operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChimeAsyncClient.DeleteRoom
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/DeleteRoom" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteRoomResponse> deleteRoom(DeleteRoomRequest deleteRoomRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteRoomRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteRoomRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteRoom");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteRoomResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteRoomResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "UnauthorizedClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceFailureException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "UnprocessableEntityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                            .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
                case "ThrottledClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledClientException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "ForbiddenException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                            .exceptionBuilderSupplier(ForbiddenException::builder).build());
                case "ResourceLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeleteRoomResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteRoomRequest, DeleteRoomResponse>().withOperationName("DeleteRoom")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteRoomRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteRoomRequest));
            CompletableFuture<DeleteRoomResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes a member from a chat room in an Amazon Chime Enterprise account.
     * </p>
     *
     * @param deleteRoomMembershipRequest
     * @return A Java Future containing the result of the DeleteRoomMembership operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChimeAsyncClient.DeleteRoomMembership
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/DeleteRoomMembership" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteRoomMembershipResponse> deleteRoomMembership(
            DeleteRoomMembershipRequest deleteRoomMembershipRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteRoomMembershipRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteRoomMembershipRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteRoomMembership");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteRoomMembershipResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteRoomMembershipResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "UnauthorizedClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceFailureException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "UnprocessableEntityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                            .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
                case "ThrottledClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledClientException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "ForbiddenException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                            .exceptionBuilderSupplier(ForbiddenException::builder).build());
                case "ResourceLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeleteRoomMembershipResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteRoomMembershipRequest, DeleteRoomMembershipResponse>()
                            .withOperationName("DeleteRoomMembership").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteRoomMembershipRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteRoomMembershipRequest));
            CompletableFuture<DeleteRoomMembershipResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Disassociates the primary provisioned phone number from the specified Amazon Chime user.
     * </p>
     *
     * @param disassociatePhoneNumberFromUserRequest
     * @return A Java Future containing the result of the DisassociatePhoneNumberFromUser operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChimeAsyncClient.DisassociatePhoneNumberFromUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/DisassociatePhoneNumberFromUser"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociatePhoneNumberFromUserResponse> disassociatePhoneNumberFromUser(
            DisassociatePhoneNumberFromUserRequest disassociatePhoneNumberFromUserRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disassociatePhoneNumberFromUserRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                disassociatePhoneNumberFromUserRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociatePhoneNumberFromUser");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociatePhoneNumberFromUserResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisassociatePhoneNumberFromUserResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "UnauthorizedClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceFailureException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "UnprocessableEntityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                            .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
                case "ThrottledClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledClientException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "ForbiddenException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                            .exceptionBuilderSupplier(ForbiddenException::builder).build());
                case "ResourceLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DisassociatePhoneNumberFromUserResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociatePhoneNumberFromUserRequest, DisassociatePhoneNumberFromUserResponse>()
                            .withOperationName("DisassociatePhoneNumberFromUser").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DisassociatePhoneNumberFromUserRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(disassociatePhoneNumberFromUserRequest));
            CompletableFuture<DisassociatePhoneNumberFromUserResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Disassociates the specified sign-in delegate groups from the specified Amazon Chime account.
     * </p>
     *
     * @param disassociateSigninDelegateGroupsFromAccountRequest
     * @return A Java Future containing the result of the DisassociateSigninDelegateGroupsFromAccount operation returned
     *         by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChimeAsyncClient.DisassociateSigninDelegateGroupsFromAccount
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/DisassociateSigninDelegateGroupsFromAccount"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateSigninDelegateGroupsFromAccountResponse> disassociateSigninDelegateGroupsFromAccount(
            DisassociateSigninDelegateGroupsFromAccountRequest disassociateSigninDelegateGroupsFromAccountRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(
                disassociateSigninDelegateGroupsFromAccountRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                disassociateSigninDelegateGroupsFromAccountRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateSigninDelegateGroupsFromAccount");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateSigninDelegateGroupsFromAccountResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DisassociateSigninDelegateGroupsFromAccountResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "UnauthorizedClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceFailureException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "UnprocessableEntityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                            .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
                case "ThrottledClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledClientException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "ForbiddenException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                            .exceptionBuilderSupplier(ForbiddenException::builder).build());
                case "ResourceLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DisassociateSigninDelegateGroupsFromAccountResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateSigninDelegateGroupsFromAccountRequest, DisassociateSigninDelegateGroupsFromAccountResponse>()
                            .withOperationName("DisassociateSigninDelegateGroupsFromAccount")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DisassociateSigninDelegateGroupsFromAccountRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(disassociateSigninDelegateGroupsFromAccountRequest));
            CompletableFuture<DisassociateSigninDelegateGroupsFromAccountResponse> whenCompleted = executeFuture.whenComplete((r,
                    e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves details for the specified Amazon Chime account, such as account type and supported licenses.
     * </p>
     *
     * @param getAccountRequest
     * @return A Java Future containing the result of the GetAccount operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChimeAsyncClient.GetAccount
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/GetAccount" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetAccountResponse> getAccount(GetAccountRequest getAccountRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getAccountRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAccountRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAccount");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetAccountResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetAccountResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "UnauthorizedClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceFailureException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "UnprocessableEntityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                            .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
                case "ThrottledClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledClientException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "ForbiddenException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                            .exceptionBuilderSupplier(ForbiddenException::builder).build());
                case "ResourceLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetAccountResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAccountRequest, GetAccountResponse>().withOperationName("GetAccount")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetAccountRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getAccountRequest));
            CompletableFuture<GetAccountResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves account settings for the specified Amazon Chime account ID, such as remote control and dialout
     * settings. For more information about these settings, see <a
     * href="https://docs.aws.amazon.com/chime/latest/ag/policies.html">Use the Policies Page</a> in the <i>Amazon Chime
     * Administration Guide</i>.
     * </p>
     *
     * @param getAccountSettingsRequest
     * @return A Java Future containing the result of the GetAccountSettings operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChimeAsyncClient.GetAccountSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/GetAccountSettings" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetAccountSettingsResponse> getAccountSettings(GetAccountSettingsRequest getAccountSettingsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getAccountSettingsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAccountSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAccountSettings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetAccountSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetAccountSettingsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "UnauthorizedClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceFailureException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "UnprocessableEntityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                            .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
                case "ThrottledClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledClientException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "ForbiddenException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                            .exceptionBuilderSupplier(ForbiddenException::builder).build());
                case "ResourceLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetAccountSettingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAccountSettingsRequest, GetAccountSettingsResponse>()
                            .withOperationName("GetAccountSettings").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetAccountSettingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getAccountSettingsRequest));
            CompletableFuture<GetAccountSettingsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves details for the specified bot, such as bot email address, bot type, status, and display name.
     * </p>
     *
     * @param getBotRequest
     * @return A Java Future containing the result of the GetBot operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChimeAsyncClient.GetBot
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/GetBot" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetBotResponse> getBot(GetBotRequest getBotRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getBotRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getBotRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetBot");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetBotResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetBotResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "UnauthorizedClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceFailureException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "UnprocessableEntityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                            .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
                case "ThrottledClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledClientException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "ForbiddenException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                            .exceptionBuilderSupplier(ForbiddenException::builder).build());
                case "ResourceLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetBotResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetBotRequest, GetBotResponse>().withOperationName("GetBot")
                            .withProtocolMetadata(protocolMetadata).withMarshaller(new GetBotRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getBotRequest));
            CompletableFuture<GetBotResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets details for an events configuration that allows a bot to receive outgoing events, such as an HTTPS endpoint
     * or Lambda function ARN.
     * </p>
     *
     * @param getEventsConfigurationRequest
     * @return A Java Future containing the result of the GetEventsConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ResourceLimitExceededException The request exceeds the resource limit.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChimeAsyncClient.GetEventsConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/GetEventsConfiguration" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetEventsConfigurationResponse> getEventsConfiguration(
            GetEventsConfigurationRequest getEventsConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getEventsConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getEventsConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetEventsConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetEventsConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetEventsConfigurationResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "UnauthorizedClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceFailureException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "UnprocessableEntityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                            .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
                case "ThrottledClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledClientException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "ForbiddenException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                            .exceptionBuilderSupplier(ForbiddenException::builder).build());
                case "ResourceLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetEventsConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetEventsConfigurationRequest, GetEventsConfigurationResponse>()
                            .withOperationName("GetEventsConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetEventsConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getEventsConfigurationRequest));
            CompletableFuture<GetEventsConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves global settings for the administrator's AWS account, such as Amazon Chime Business Calling and Amazon
     * Chime Voice Connector settings.
     * </p>
     *
     * @param getGlobalSettingsRequest
     * @return A Java Future containing the result of the GetGlobalSettings operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChimeAsyncClient.GetGlobalSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/GetGlobalSettings" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetGlobalSettingsResponse> getGlobalSettings(GetGlobalSettingsRequest getGlobalSettingsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getGlobalSettingsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getGlobalSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetGlobalSettings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetGlobalSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetGlobalSettingsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "UnauthorizedClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceFailureException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "UnprocessableEntityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                            .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
                case "ThrottledClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledClientException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "ForbiddenException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                            .exceptionBuilderSupplier(ForbiddenException::builder).build());
                case "ResourceLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetGlobalSettingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetGlobalSettingsRequest, GetGlobalSettingsResponse>()
                            .withOperationName("GetGlobalSettings").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetGlobalSettingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getGlobalSettingsRequest));
            CompletableFuture<GetGlobalSettingsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves details for the specified phone number ID, such as associations, capabilities, and product type.
     * </p>
     *
     * @param getPhoneNumberRequest
     * @return A Java Future containing the result of the GetPhoneNumber operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChimeAsyncClient.GetPhoneNumber
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/GetPhoneNumber" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetPhoneNumberResponse> getPhoneNumber(GetPhoneNumberRequest getPhoneNumberRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getPhoneNumberRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPhoneNumberRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPhoneNumber");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetPhoneNumberResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetPhoneNumberResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "UnauthorizedClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceFailureException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "UnprocessableEntityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                            .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
                case "ThrottledClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledClientException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "ForbiddenException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                            .exceptionBuilderSupplier(ForbiddenException::builder).build());
                case "ResourceLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetPhoneNumberResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetPhoneNumberRequest, GetPhoneNumberResponse>()
                            .withOperationName("GetPhoneNumber").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetPhoneNumberRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getPhoneNumberRequest));
            CompletableFuture<GetPhoneNumberResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves details for the specified phone number order, such as the order creation timestamp, phone numbers in
     * E.164 format, product type, and order status.
     * </p>
     *
     * @param getPhoneNumberOrderRequest
     * @return A Java Future containing the result of the GetPhoneNumberOrder operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChimeAsyncClient.GetPhoneNumberOrder
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/GetPhoneNumberOrder" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetPhoneNumberOrderResponse> getPhoneNumberOrder(
            GetPhoneNumberOrderRequest getPhoneNumberOrderRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getPhoneNumberOrderRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPhoneNumberOrderRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPhoneNumberOrder");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetPhoneNumberOrderResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetPhoneNumberOrderResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "UnauthorizedClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceFailureException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "UnprocessableEntityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                            .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
                case "ThrottledClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledClientException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "ForbiddenException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                            .exceptionBuilderSupplier(ForbiddenException::builder).build());
                case "ResourceLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetPhoneNumberOrderResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetPhoneNumberOrderRequest, GetPhoneNumberOrderResponse>()
                            .withOperationName("GetPhoneNumberOrder").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetPhoneNumberOrderRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getPhoneNumberOrderRequest));
            CompletableFuture<GetPhoneNumberOrderResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the phone number settings for the administrator's AWS account, such as the default outbound calling
     * name.
     * </p>
     *
     * @param getPhoneNumberSettingsRequest
     * @return A Java Future containing the result of the GetPhoneNumberSettings operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChimeAsyncClient.GetPhoneNumberSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/GetPhoneNumberSettings" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetPhoneNumberSettingsResponse> getPhoneNumberSettings(
            GetPhoneNumberSettingsRequest getPhoneNumberSettingsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getPhoneNumberSettingsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPhoneNumberSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPhoneNumberSettings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetPhoneNumberSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetPhoneNumberSettingsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "UnauthorizedClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceFailureException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "UnprocessableEntityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                            .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
                case "ThrottledClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledClientException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "ForbiddenException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                            .exceptionBuilderSupplier(ForbiddenException::builder).build());
                case "ResourceLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetPhoneNumberSettingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetPhoneNumberSettingsRequest, GetPhoneNumberSettingsResponse>()
                            .withOperationName("GetPhoneNumberSettings").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetPhoneNumberSettingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getPhoneNumberSettingsRequest));
            CompletableFuture<GetPhoneNumberSettingsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the retention settings for the specified Amazon Chime Enterprise account. For more information about
     * retention settings, see <a href="https://docs.aws.amazon.com/chime/latest/ag/chat-retention.html">Managing Chat
     * Retention Policies</a> in the <i>Amazon Chime Administration Guide</i>.
     * </p>
     *
     * @param getRetentionSettingsRequest
     * @return A Java Future containing the result of the GetRetentionSettings operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChimeAsyncClient.GetRetentionSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/GetRetentionSettings" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetRetentionSettingsResponse> getRetentionSettings(
            GetRetentionSettingsRequest getRetentionSettingsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getRetentionSettingsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getRetentionSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRetentionSettings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetRetentionSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetRetentionSettingsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "UnauthorizedClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceFailureException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "UnprocessableEntityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                            .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
                case "ThrottledClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledClientException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "ForbiddenException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                            .exceptionBuilderSupplier(ForbiddenException::builder).build());
                case "ResourceLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetRetentionSettingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetRetentionSettingsRequest, GetRetentionSettingsResponse>()
                            .withOperationName("GetRetentionSettings").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetRetentionSettingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getRetentionSettingsRequest));
            CompletableFuture<GetRetentionSettingsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves room details, such as the room name, for a room in an Amazon Chime Enterprise account.
     * </p>
     *
     * @param getRoomRequest
     * @return A Java Future containing the result of the GetRoom operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChimeAsyncClient.GetRoom
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/GetRoom" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetRoomResponse> getRoom(GetRoomRequest getRoomRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getRoomRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getRoomRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRoom");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetRoomResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetRoomResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "UnauthorizedClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceFailureException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "UnprocessableEntityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                            .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
                case "ThrottledClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledClientException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "ForbiddenException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                            .exceptionBuilderSupplier(ForbiddenException::builder).build());
                case "ResourceLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetRoomResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetRoomRequest, GetRoomResponse>().withOperationName("GetRoom")
                            .withProtocolMetadata(protocolMetadata).withMarshaller(new GetRoomRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getRoomRequest));
            CompletableFuture<GetRoomResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves details for the specified user ID, such as primary email address, license type,and personal meeting
     * PIN.
     * </p>
     * <p>
     * To retrieve user details with an email address instead of a user ID, use the <a>ListUsers</a> action, and then
     * filter by email address.
     * </p>
     *
     * @param getUserRequest
     * @return A Java Future containing the result of the GetUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChimeAsyncClient.GetUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/GetUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetUserResponse> getUser(GetUserRequest getUserRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getUserRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetUser");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetUserResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "UnauthorizedClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceFailureException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "UnprocessableEntityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                            .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
                case "ThrottledClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledClientException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "ForbiddenException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                            .exceptionBuilderSupplier(ForbiddenException::builder).build());
                case "ResourceLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetUserResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetUserRequest, GetUserResponse>().withOperationName("GetUser")
                            .withProtocolMetadata(protocolMetadata).withMarshaller(new GetUserRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getUserRequest));
            CompletableFuture<GetUserResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves settings for the specified user ID, such as any associated phone number settings.
     * </p>
     *
     * @param getUserSettingsRequest
     * @return A Java Future containing the result of the GetUserSettings operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChimeAsyncClient.GetUserSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/GetUserSettings" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetUserSettingsResponse> getUserSettings(GetUserSettingsRequest getUserSettingsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getUserSettingsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getUserSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetUserSettings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetUserSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetUserSettingsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "UnauthorizedClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceFailureException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "UnprocessableEntityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                            .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
                case "ThrottledClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledClientException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "ForbiddenException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                            .exceptionBuilderSupplier(ForbiddenException::builder).build());
                case "ResourceLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetUserSettingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetUserSettingsRequest, GetUserSettingsResponse>()
                            .withOperationName("GetUserSettings").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetUserSettingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getUserSettingsRequest));
            CompletableFuture<GetUserSettingsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Sends email to a maximum of 50 users, inviting them to the specified Amazon Chime <code>Team</code> account. Only
     * <code>Team</code> account types are currently supported for this action.
     * </p>
     *
     * @param inviteUsersRequest
     * @return A Java Future containing the result of the InviteUsers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChimeAsyncClient.InviteUsers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/InviteUsers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<InviteUsersResponse> inviteUsers(InviteUsersRequest inviteUsersRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(inviteUsersRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, inviteUsersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "InviteUsers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<InviteUsersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    InviteUsersResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "UnauthorizedClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceFailureException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "UnprocessableEntityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                            .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
                case "ThrottledClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledClientException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "ForbiddenException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                            .exceptionBuilderSupplier(ForbiddenException::builder).build());
                case "ResourceLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<InviteUsersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<InviteUsersRequest, InviteUsersResponse>()
                            .withOperationName("InviteUsers").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new InviteUsersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(inviteUsersRequest));
            CompletableFuture<InviteUsersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the Amazon Chime accounts under the administrator's AWS account. You can filter accounts by account name
     * prefix. To find out which Amazon Chime account a user belongs to, you can filter by the user's email address,
     * which returns one account result.
     * </p>
     *
     * @param listAccountsRequest
     * @return A Java Future containing the result of the ListAccounts operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChimeAsyncClient.ListAccounts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/ListAccounts" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListAccountsResponse> listAccounts(ListAccountsRequest listAccountsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listAccountsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAccountsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAccounts");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAccountsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListAccountsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "UnauthorizedClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceFailureException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "UnprocessableEntityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                            .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
                case "ThrottledClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledClientException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "ForbiddenException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                            .exceptionBuilderSupplier(ForbiddenException::builder).build());
                case "ResourceLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListAccountsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAccountsRequest, ListAccountsResponse>()
                            .withOperationName("ListAccounts").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListAccountsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listAccountsRequest));
            CompletableFuture<ListAccountsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the bots associated with the administrator's Amazon Chime Enterprise account ID.
     * </p>
     *
     * @param listBotsRequest
     * @return A Java Future containing the result of the ListBots operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChimeAsyncClient.ListBots
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/ListBots" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListBotsResponse> listBots(ListBotsRequest listBotsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listBotsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listBotsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListBots");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListBotsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListBotsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "UnauthorizedClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceFailureException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "UnprocessableEntityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                            .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
                case "ThrottledClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledClientException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "ForbiddenException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                            .exceptionBuilderSupplier(ForbiddenException::builder).build());
                case "ResourceLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListBotsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListBotsRequest, ListBotsResponse>().withOperationName("ListBots")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListBotsRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(listBotsRequest));
            CompletableFuture<ListBotsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the phone number orders for the administrator's Amazon Chime account.
     * </p>
     *
     * @param listPhoneNumberOrdersRequest
     * @return A Java Future containing the result of the ListPhoneNumberOrders operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChimeAsyncClient.ListPhoneNumberOrders
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/ListPhoneNumberOrders" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListPhoneNumberOrdersResponse> listPhoneNumberOrders(
            ListPhoneNumberOrdersRequest listPhoneNumberOrdersRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPhoneNumberOrdersRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPhoneNumberOrdersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPhoneNumberOrders");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPhoneNumberOrdersResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListPhoneNumberOrdersResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "UnauthorizedClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceFailureException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "UnprocessableEntityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                            .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
                case "ThrottledClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledClientException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "ForbiddenException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                            .exceptionBuilderSupplier(ForbiddenException::builder).build());
                case "ResourceLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListPhoneNumberOrdersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPhoneNumberOrdersRequest, ListPhoneNumberOrdersResponse>()
                            .withOperationName("ListPhoneNumberOrders").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListPhoneNumberOrdersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listPhoneNumberOrdersRequest));
            CompletableFuture<ListPhoneNumberOrdersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the phone numbers for the specified Amazon Chime account, Amazon Chime user, Amazon Chime Voice Connector,
     * or Amazon Chime Voice Connector group.
     * </p>
     *
     * @param listPhoneNumbersRequest
     * @return A Java Future containing the result of the ListPhoneNumbers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChimeAsyncClient.ListPhoneNumbers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/ListPhoneNumbers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListPhoneNumbersResponse> listPhoneNumbers(ListPhoneNumbersRequest listPhoneNumbersRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPhoneNumbersRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPhoneNumbersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPhoneNumbers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPhoneNumbersResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListPhoneNumbersResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "UnauthorizedClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceFailureException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "UnprocessableEntityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                            .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
                case "ThrottledClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledClientException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "ForbiddenException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                            .exceptionBuilderSupplier(ForbiddenException::builder).build());
                case "ResourceLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListPhoneNumbersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPhoneNumbersRequest, ListPhoneNumbersResponse>()
                            .withOperationName("ListPhoneNumbers").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListPhoneNumbersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listPhoneNumbersRequest));
            CompletableFuture<ListPhoneNumbersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the membership details for the specified room in an Amazon Chime Enterprise account, such as the members'
     * IDs, email addresses, and names.
     * </p>
     *
     * @param listRoomMembershipsRequest
     * @return A Java Future containing the result of the ListRoomMemberships operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChimeAsyncClient.ListRoomMemberships
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/ListRoomMemberships" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListRoomMembershipsResponse> listRoomMemberships(
            ListRoomMembershipsRequest listRoomMembershipsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listRoomMembershipsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRoomMembershipsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRoomMemberships");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListRoomMembershipsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListRoomMembershipsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "UnauthorizedClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceFailureException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "UnprocessableEntityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                            .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
                case "ThrottledClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledClientException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "ForbiddenException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                            .exceptionBuilderSupplier(ForbiddenException::builder).build());
                case "ResourceLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListRoomMembershipsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListRoomMembershipsRequest, ListRoomMembershipsResponse>()
                            .withOperationName("ListRoomMemberships").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListRoomMembershipsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listRoomMembershipsRequest));
            CompletableFuture<ListRoomMembershipsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the room details for the specified Amazon Chime Enterprise account. Optionally, filter the results by a
     * member ID (user ID or bot ID) to see a list of rooms that the member belongs to.
     * </p>
     *
     * @param listRoomsRequest
     * @return A Java Future containing the result of the ListRooms operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChimeAsyncClient.ListRooms
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/ListRooms" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListRoomsResponse> listRooms(ListRoomsRequest listRoomsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listRoomsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRoomsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRooms");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListRoomsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListRoomsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "UnauthorizedClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceFailureException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "UnprocessableEntityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                            .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
                case "ThrottledClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledClientException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "ForbiddenException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                            .exceptionBuilderSupplier(ForbiddenException::builder).build());
                case "ResourceLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListRoomsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListRoomsRequest, ListRoomsResponse>().withOperationName("ListRooms")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListRoomsRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(listRoomsRequest));
            CompletableFuture<ListRoomsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists supported phone number countries.
     * </p>
     *
     * @param listSupportedPhoneNumberCountriesRequest
     * @return A Java Future containing the result of the ListSupportedPhoneNumberCountries operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChimeAsyncClient.ListSupportedPhoneNumberCountries
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/ListSupportedPhoneNumberCountries"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListSupportedPhoneNumberCountriesResponse> listSupportedPhoneNumberCountries(
            ListSupportedPhoneNumberCountriesRequest listSupportedPhoneNumberCountriesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listSupportedPhoneNumberCountriesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listSupportedPhoneNumberCountriesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSupportedPhoneNumberCountries");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListSupportedPhoneNumberCountriesResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, ListSupportedPhoneNumberCountriesResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "UnauthorizedClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceFailureException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "UnprocessableEntityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                            .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
                case "ThrottledClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledClientException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "ForbiddenException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                            .exceptionBuilderSupplier(ForbiddenException::builder).build());
                case "ResourceLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListSupportedPhoneNumberCountriesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSupportedPhoneNumberCountriesRequest, ListSupportedPhoneNumberCountriesResponse>()
                            .withOperationName("ListSupportedPhoneNumberCountries").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListSupportedPhoneNumberCountriesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listSupportedPhoneNumberCountriesRequest));
            CompletableFuture<ListSupportedPhoneNumberCountriesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the users that belong to the specified Amazon Chime account. You can specify an email address to list only
     * the user that the email address belongs to.
     * </p>
     *
     * @param listUsersRequest
     * @return A Java Future containing the result of the ListUsers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChimeAsyncClient.ListUsers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/ListUsers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListUsersResponse> listUsers(ListUsersRequest listUsersRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listUsersRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listUsersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListUsers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListUsersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListUsersResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "UnauthorizedClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceFailureException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "UnprocessableEntityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                            .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
                case "ThrottledClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledClientException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "ForbiddenException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                            .exceptionBuilderSupplier(ForbiddenException::builder).build());
                case "ResourceLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListUsersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListUsersRequest, ListUsersResponse>().withOperationName("ListUsers")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListUsersRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(listUsersRequest));
            CompletableFuture<ListUsersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Logs out the specified user from all of the devices they are currently logged into.
     * </p>
     *
     * @param logoutUserRequest
     * @return A Java Future containing the result of the LogoutUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChimeAsyncClient.LogoutUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/LogoutUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<LogoutUserResponse> logoutUser(LogoutUserRequest logoutUserRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(logoutUserRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, logoutUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "LogoutUser");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<LogoutUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    LogoutUserResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "UnauthorizedClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceFailureException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "UnprocessableEntityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                            .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
                case "ThrottledClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledClientException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "ForbiddenException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                            .exceptionBuilderSupplier(ForbiddenException::builder).build());
                case "ResourceLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<LogoutUserResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<LogoutUserRequest, LogoutUserResponse>().withOperationName("LogoutUser")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new LogoutUserRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(logoutUserRequest));
            CompletableFuture<LogoutUserResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an events configuration that allows a bot to receive outgoing events sent by Amazon Chime. Choose either
     * an HTTPS endpoint or a Lambda function ARN. For more information, see <a>Bot</a>.
     * </p>
     *
     * @param putEventsConfigurationRequest
     * @return A Java Future containing the result of the PutEventsConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ResourceLimitExceededException The request exceeds the resource limit.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChimeAsyncClient.PutEventsConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/PutEventsConfiguration" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<PutEventsConfigurationResponse> putEventsConfiguration(
            PutEventsConfigurationRequest putEventsConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putEventsConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putEventsConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutEventsConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutEventsConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutEventsConfigurationResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "UnauthorizedClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceFailureException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "UnprocessableEntityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                            .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
                case "ThrottledClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledClientException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "ForbiddenException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                            .exceptionBuilderSupplier(ForbiddenException::builder).build());
                case "ResourceLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<PutEventsConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutEventsConfigurationRequest, PutEventsConfigurationResponse>()
                            .withOperationName("PutEventsConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutEventsConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putEventsConfigurationRequest));
            CompletableFuture<PutEventsConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Puts retention settings for the specified Amazon Chime Enterprise account. We recommend using AWS CloudTrail to
     * monitor usage of this API for your account. For more information, see <a
     * href="https://docs.aws.amazon.com/chime/latest/ag/cloudtrail.html">Logging Amazon Chime API Calls with AWS
     * CloudTrail</a> in the <i>Amazon Chime Administration Guide</i>.
     * </p>
     * <p>
     * To turn off existing retention settings, remove the number of days from the corresponding <b>RetentionDays</b>
     * field in the <b>RetentionSettings</b> object. For more information about retention settings, see <a
     * href="https://docs.aws.amazon.com/chime/latest/ag/chat-retention.html">Managing Chat Retention Policies</a> in
     * the <i>Amazon Chime Administration Guide</i>.
     * </p>
     *
     * @param putRetentionSettingsRequest
     * @return A Java Future containing the result of the PutRetentionSettings operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChimeAsyncClient.PutRetentionSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/PutRetentionSettings" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<PutRetentionSettingsResponse> putRetentionSettings(
            PutRetentionSettingsRequest putRetentionSettingsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putRetentionSettingsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putRetentionSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutRetentionSettings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutRetentionSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutRetentionSettingsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "UnauthorizedClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceFailureException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "UnprocessableEntityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                            .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
                case "ThrottledClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledClientException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "ForbiddenException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                            .exceptionBuilderSupplier(ForbiddenException::builder).build());
                case "ResourceLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<PutRetentionSettingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutRetentionSettingsRequest, PutRetentionSettingsResponse>()
                            .withOperationName("PutRetentionSettings").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutRetentionSettingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putRetentionSettingsRequest));
            CompletableFuture<PutRetentionSettingsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Redacts the specified message from the specified Amazon Chime conversation.
     * </p>
     *
     * @param redactConversationMessageRequest
     * @return A Java Future containing the result of the RedactConversationMessage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChimeAsyncClient.RedactConversationMessage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/RedactConversationMessage"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RedactConversationMessageResponse> redactConversationMessage(
            RedactConversationMessageRequest redactConversationMessageRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(redactConversationMessageRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, redactConversationMessageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RedactConversationMessage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RedactConversationMessageResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RedactConversationMessageResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "UnauthorizedClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceFailureException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "UnprocessableEntityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                            .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
                case "ThrottledClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledClientException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "ForbiddenException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                            .exceptionBuilderSupplier(ForbiddenException::builder).build());
                case "ResourceLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<RedactConversationMessageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RedactConversationMessageRequest, RedactConversationMessageResponse>()
                            .withOperationName("RedactConversationMessage").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new RedactConversationMessageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(redactConversationMessageRequest));
            CompletableFuture<RedactConversationMessageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Redacts the specified message from the specified Amazon Chime channel.
     * </p>
     *
     * @param redactRoomMessageRequest
     * @return A Java Future containing the result of the RedactRoomMessage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChimeAsyncClient.RedactRoomMessage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/RedactRoomMessage" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<RedactRoomMessageResponse> redactRoomMessage(RedactRoomMessageRequest redactRoomMessageRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(redactRoomMessageRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, redactRoomMessageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RedactRoomMessage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RedactRoomMessageResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RedactRoomMessageResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "UnauthorizedClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceFailureException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "UnprocessableEntityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                            .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
                case "ThrottledClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledClientException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "ForbiddenException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                            .exceptionBuilderSupplier(ForbiddenException::builder).build());
                case "ResourceLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<RedactRoomMessageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RedactRoomMessageRequest, RedactRoomMessageResponse>()
                            .withOperationName("RedactRoomMessage").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new RedactRoomMessageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(redactRoomMessageRequest));
            CompletableFuture<RedactRoomMessageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Regenerates the security token for a bot.
     * </p>
     *
     * @param regenerateSecurityTokenRequest
     * @return A Java Future containing the result of the RegenerateSecurityToken operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChimeAsyncClient.RegenerateSecurityToken
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/RegenerateSecurityToken" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<RegenerateSecurityTokenResponse> regenerateSecurityToken(
            RegenerateSecurityTokenRequest regenerateSecurityTokenRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(regenerateSecurityTokenRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, regenerateSecurityTokenRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RegenerateSecurityToken");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RegenerateSecurityTokenResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RegenerateSecurityTokenResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "UnauthorizedClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceFailureException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "UnprocessableEntityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                            .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
                case "ThrottledClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledClientException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "ForbiddenException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                            .exceptionBuilderSupplier(ForbiddenException::builder).build());
                case "ResourceLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<RegenerateSecurityTokenResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RegenerateSecurityTokenRequest, RegenerateSecurityTokenResponse>()
                            .withOperationName("RegenerateSecurityToken").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new RegenerateSecurityTokenRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(regenerateSecurityTokenRequest));
            CompletableFuture<RegenerateSecurityTokenResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Resets the personal meeting PIN for the specified user on an Amazon Chime account. Returns the <a>User</a> object
     * with the updated personal meeting PIN.
     * </p>
     *
     * @param resetPersonalPinRequest
     * @return A Java Future containing the result of the ResetPersonalPIN operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChimeAsyncClient.ResetPersonalPIN
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/ResetPersonalPIN" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ResetPersonalPinResponse> resetPersonalPIN(ResetPersonalPinRequest resetPersonalPinRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(resetPersonalPinRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, resetPersonalPinRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ResetPersonalPIN");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ResetPersonalPinResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ResetPersonalPinResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "UnauthorizedClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceFailureException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "UnprocessableEntityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                            .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
                case "ThrottledClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledClientException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "ForbiddenException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                            .exceptionBuilderSupplier(ForbiddenException::builder).build());
                case "ResourceLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ResetPersonalPinResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ResetPersonalPinRequest, ResetPersonalPinResponse>()
                            .withOperationName("ResetPersonalPIN").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ResetPersonalPinRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(resetPersonalPinRequest));
            CompletableFuture<ResetPersonalPinResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Moves a phone number from the <b>Deletion queue</b> back into the phone number <b>Inventory</b>.
     * </p>
     *
     * @param restorePhoneNumberRequest
     * @return A Java Future containing the result of the RestorePhoneNumber operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ResourceLimitExceededException The request exceeds the resource limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChimeAsyncClient.RestorePhoneNumber
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/RestorePhoneNumber" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<RestorePhoneNumberResponse> restorePhoneNumber(RestorePhoneNumberRequest restorePhoneNumberRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(restorePhoneNumberRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, restorePhoneNumberRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RestorePhoneNumber");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RestorePhoneNumberResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RestorePhoneNumberResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "UnauthorizedClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceFailureException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "UnprocessableEntityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                            .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
                case "ThrottledClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledClientException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "ForbiddenException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                            .exceptionBuilderSupplier(ForbiddenException::builder).build());
                case "ResourceLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<RestorePhoneNumberResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RestorePhoneNumberRequest, RestorePhoneNumberResponse>()
                            .withOperationName("RestorePhoneNumber").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new RestorePhoneNumberRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(restorePhoneNumberRequest));
            CompletableFuture<RestorePhoneNumberResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Searches for phone numbers that can be ordered. For US numbers, provide at least one of the following search
     * filters: <code>AreaCode</code>, <code>City</code>, <code>State</code>, or <code>TollFreePrefix</code>. If you
     * provide <code>City</code>, you must also provide <code>State</code>. Numbers outside the US only support the
     * <code>PhoneNumberType</code> filter, which you must use.
     * </p>
     *
     * @param searchAvailablePhoneNumbersRequest
     * @return A Java Future containing the result of the SearchAvailablePhoneNumbers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChimeAsyncClient.SearchAvailablePhoneNumbers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/SearchAvailablePhoneNumbers"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<SearchAvailablePhoneNumbersResponse> searchAvailablePhoneNumbers(
            SearchAvailablePhoneNumbersRequest searchAvailablePhoneNumbersRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(searchAvailablePhoneNumbersRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, searchAvailablePhoneNumbersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SearchAvailablePhoneNumbers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SearchAvailablePhoneNumbersResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, SearchAvailablePhoneNumbersResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "UnauthorizedClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceFailureException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "UnprocessableEntityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                            .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
                case "ThrottledClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledClientException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "ForbiddenException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                            .exceptionBuilderSupplier(ForbiddenException::builder).build());
                case "ResourceLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<SearchAvailablePhoneNumbersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SearchAvailablePhoneNumbersRequest, SearchAvailablePhoneNumbersResponse>()
                            .withOperationName("SearchAvailablePhoneNumbers").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new SearchAvailablePhoneNumbersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(searchAvailablePhoneNumbersRequest));
            CompletableFuture<SearchAvailablePhoneNumbersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates account details for the specified Amazon Chime account. Currently, only account name and default license
     * updates are supported for this action.
     * </p>
     *
     * @param updateAccountRequest
     * @return A Java Future containing the result of the UpdateAccount operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChimeAsyncClient.UpdateAccount
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/UpdateAccount" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateAccountResponse> updateAccount(UpdateAccountRequest updateAccountRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateAccountRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateAccountRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateAccount");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateAccountResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateAccountResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "UnauthorizedClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceFailureException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "UnprocessableEntityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                            .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
                case "ThrottledClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledClientException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "ForbiddenException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                            .exceptionBuilderSupplier(ForbiddenException::builder).build());
                case "ResourceLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UpdateAccountResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateAccountRequest, UpdateAccountResponse>()
                            .withOperationName("UpdateAccount").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateAccountRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateAccountRequest));
            CompletableFuture<UpdateAccountResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the settings for the specified Amazon Chime account. You can update settings for remote control of shared
     * screens, or for the dial-out option. For more information about these settings, see <a
     * href="https://docs.aws.amazon.com/chime/latest/ag/policies.html">Use the Policies Page</a> in the <i>Amazon Chime
     * Administration Guide</i>.
     * </p>
     *
     * @param updateAccountSettingsRequest
     * @return A Java Future containing the result of the UpdateAccountSettings operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChimeAsyncClient.UpdateAccountSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/UpdateAccountSettings" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateAccountSettingsResponse> updateAccountSettings(
            UpdateAccountSettingsRequest updateAccountSettingsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateAccountSettingsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateAccountSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateAccountSettings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateAccountSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateAccountSettingsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "UnauthorizedClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceFailureException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "UnprocessableEntityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                            .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
                case "ThrottledClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledClientException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "ForbiddenException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                            .exceptionBuilderSupplier(ForbiddenException::builder).build());
                case "ResourceLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UpdateAccountSettingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateAccountSettingsRequest, UpdateAccountSettingsResponse>()
                            .withOperationName("UpdateAccountSettings").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateAccountSettingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateAccountSettingsRequest));
            CompletableFuture<UpdateAccountSettingsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the status of the specified bot, such as starting or stopping the bot from running in your Amazon Chime
     * Enterprise account.
     * </p>
     *
     * @param updateBotRequest
     * @return A Java Future containing the result of the UpdateBot operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChimeAsyncClient.UpdateBot
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/UpdateBot" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateBotResponse> updateBot(UpdateBotRequest updateBotRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateBotRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateBotRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateBot");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateBotResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateBotResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "UnauthorizedClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceFailureException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "UnprocessableEntityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                            .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
                case "ThrottledClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledClientException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "ForbiddenException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                            .exceptionBuilderSupplier(ForbiddenException::builder).build());
                case "ResourceLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UpdateBotResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateBotRequest, UpdateBotResponse>().withOperationName("UpdateBot")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateBotRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateBotRequest));
            CompletableFuture<UpdateBotResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates global settings for the administrator's AWS account, such as Amazon Chime Business Calling and Amazon
     * Chime Voice Connector settings.
     * </p>
     *
     * @param updateGlobalSettingsRequest
     * @return A Java Future containing the result of the UpdateGlobalSettings operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChimeAsyncClient.UpdateGlobalSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/UpdateGlobalSettings" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateGlobalSettingsResponse> updateGlobalSettings(
            UpdateGlobalSettingsRequest updateGlobalSettingsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateGlobalSettingsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateGlobalSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateGlobalSettings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateGlobalSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateGlobalSettingsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "UnauthorizedClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceFailureException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "UnprocessableEntityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                            .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
                case "ThrottledClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledClientException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "ForbiddenException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                            .exceptionBuilderSupplier(ForbiddenException::builder).build());
                case "ResourceLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UpdateGlobalSettingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateGlobalSettingsRequest, UpdateGlobalSettingsResponse>()
                            .withOperationName("UpdateGlobalSettings").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateGlobalSettingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateGlobalSettingsRequest));
            CompletableFuture<UpdateGlobalSettingsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates phone number details, such as product type or calling name, for the specified phone number ID. You can
     * update one phone number detail at a time. For example, you can update either the product type or the calling name
     * in one action.
     * </p>
     * <p>
     * For toll-free numbers, you cannot use the Amazon Chime Business Calling product type. For numbers outside the
     * U.S., you must use the Amazon Chime SIP Media Application Dial-In product type.
     * </p>
     * <p>
     * Updates to outbound calling names can take 72 hours to complete. Pending updates to outbound calling names must
     * be complete before you can request another update.
     * </p>
     *
     * @param updatePhoneNumberRequest
     * @return A Java Future containing the result of the UpdatePhoneNumber operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChimeAsyncClient.UpdatePhoneNumber
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/UpdatePhoneNumber" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdatePhoneNumberResponse> updatePhoneNumber(UpdatePhoneNumberRequest updatePhoneNumberRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updatePhoneNumberRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updatePhoneNumberRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdatePhoneNumber");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdatePhoneNumberResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdatePhoneNumberResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "UnauthorizedClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceFailureException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "UnprocessableEntityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                            .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
                case "ThrottledClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledClientException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "ForbiddenException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                            .exceptionBuilderSupplier(ForbiddenException::builder).build());
                case "ResourceLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UpdatePhoneNumberResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdatePhoneNumberRequest, UpdatePhoneNumberResponse>()
                            .withOperationName("UpdatePhoneNumber").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdatePhoneNumberRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updatePhoneNumberRequest));
            CompletableFuture<UpdatePhoneNumberResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the phone number settings for the administrator's AWS account, such as the default outbound calling name.
     * You can update the default outbound calling name once every seven days. Outbound calling names can take up to 72
     * hours to update.
     * </p>
     *
     * @param updatePhoneNumberSettingsRequest
     * @return A Java Future containing the result of the UpdatePhoneNumberSettings operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChimeAsyncClient.UpdatePhoneNumberSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/UpdatePhoneNumberSettings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdatePhoneNumberSettingsResponse> updatePhoneNumberSettings(
            UpdatePhoneNumberSettingsRequest updatePhoneNumberSettingsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updatePhoneNumberSettingsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updatePhoneNumberSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdatePhoneNumberSettings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdatePhoneNumberSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdatePhoneNumberSettingsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "UnauthorizedClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceFailureException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "UnprocessableEntityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                            .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
                case "ThrottledClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledClientException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "ForbiddenException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                            .exceptionBuilderSupplier(ForbiddenException::builder).build());
                case "ResourceLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UpdatePhoneNumberSettingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdatePhoneNumberSettingsRequest, UpdatePhoneNumberSettingsResponse>()
                            .withOperationName("UpdatePhoneNumberSettings").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdatePhoneNumberSettingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updatePhoneNumberSettingsRequest));
            CompletableFuture<UpdatePhoneNumberSettingsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates room details, such as the room name, for a room in an Amazon Chime Enterprise account.
     * </p>
     *
     * @param updateRoomRequest
     * @return A Java Future containing the result of the UpdateRoom operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChimeAsyncClient.UpdateRoom
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/UpdateRoom" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateRoomResponse> updateRoom(UpdateRoomRequest updateRoomRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateRoomRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateRoomRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateRoom");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateRoomResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateRoomResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "UnauthorizedClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceFailureException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "UnprocessableEntityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                            .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
                case "ThrottledClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledClientException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "ForbiddenException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                            .exceptionBuilderSupplier(ForbiddenException::builder).build());
                case "ResourceLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UpdateRoomResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateRoomRequest, UpdateRoomResponse>().withOperationName("UpdateRoom")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateRoomRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateRoomRequest));
            CompletableFuture<UpdateRoomResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates room membership details, such as the member role, for a room in an Amazon Chime Enterprise account. The
     * member role designates whether the member is a chat room administrator or a general chat room member. The member
     * role can be updated only for user IDs.
     * </p>
     *
     * @param updateRoomMembershipRequest
     * @return A Java Future containing the result of the UpdateRoomMembership operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChimeAsyncClient.UpdateRoomMembership
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/UpdateRoomMembership" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateRoomMembershipResponse> updateRoomMembership(
            UpdateRoomMembershipRequest updateRoomMembershipRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateRoomMembershipRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateRoomMembershipRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateRoomMembership");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateRoomMembershipResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateRoomMembershipResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "UnauthorizedClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceFailureException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "UnprocessableEntityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                            .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
                case "ThrottledClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledClientException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "ForbiddenException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                            .exceptionBuilderSupplier(ForbiddenException::builder).build());
                case "ResourceLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UpdateRoomMembershipResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateRoomMembershipRequest, UpdateRoomMembershipResponse>()
                            .withOperationName("UpdateRoomMembership").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateRoomMembershipRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateRoomMembershipRequest));
            CompletableFuture<UpdateRoomMembershipResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates user details for a specified user ID. Currently, only <code>LicenseType</code> updates are supported for
     * this action.
     * </p>
     *
     * @param updateUserRequest
     * @return A Java Future containing the result of the UpdateUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChimeAsyncClient.UpdateUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/UpdateUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateUserResponse> updateUser(UpdateUserRequest updateUserRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateUserRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateUser");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateUserResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "UnauthorizedClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceFailureException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "UnprocessableEntityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                            .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
                case "ThrottledClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledClientException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "ForbiddenException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                            .exceptionBuilderSupplier(ForbiddenException::builder).build());
                case "ResourceLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UpdateUserResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateUserRequest, UpdateUserResponse>().withOperationName("UpdateUser")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateUserRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateUserRequest));
            CompletableFuture<UpdateUserResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the settings for the specified user, such as phone number settings.
     * </p>
     *
     * @param updateUserSettingsRequest
     * @return A Java Future containing the result of the UpdateUserSettings operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChimeAsyncClient.UpdateUserSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/UpdateUserSettings" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateUserSettingsResponse> updateUserSettings(UpdateUserSettingsRequest updateUserSettingsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateUserSettingsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateUserSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateUserSettings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateUserSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateUserSettingsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "UnauthorizedClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(ServiceFailureException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "UnprocessableEntityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                            .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
                case "ThrottledClientException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottledClientException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "ForbiddenException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                            .exceptionBuilderSupplier(ForbiddenException::builder).build());
                case "ResourceLimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UpdateUserSettingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateUserSettingsRequest, UpdateUserSettingsResponse>()
                            .withOperationName("UpdateUserSettings").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateUserSettingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateUserSettingsRequest));
            CompletableFuture<UpdateUserSettingsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final ChimeServiceClientConfiguration serviceClientConfiguration() {
        return new ChimeServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration).defaultServiceExceptionSupplier(ChimeException::builder)
                .protocol(AwsJsonProtocol.REST_JSON).protocolVersion("1.1");
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        ChimeServiceClientConfigurationBuilder serviceConfigBuilder = new ChimeServiceClientConfigurationBuilder(configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
