/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mediaconvert.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * Required when you choose AVC-Intra for your output video codec. For more information about the AVC-Intra settings,
 * see the relevant specification. For detailed information about SD and HD in AVC-Intra, see
 * https://ieeexplore.ieee.org/document/7290936. For information about 4K/2K in AVC-Intra, see
 * https://pro-av.panasonic.net/en/avc-ultra/AVC-ULTRAoverview.pdf.
 */
@Generated("software.amazon.awssdk:codegen")
public final class AvcIntraSettings implements SdkPojo, Serializable,
        ToCopyableBuilder<AvcIntraSettings.Builder, AvcIntraSettings> {
    private static final SdkField<String> AVC_INTRA_CLASS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AvcIntraClass").getter(getter(AvcIntraSettings::avcIntraClassAsString))
            .setter(setter(Builder::avcIntraClass))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("avcIntraClass").build()).build();

    private static final SdkField<AvcIntraUhdSettings> AVC_INTRA_UHD_SETTINGS_FIELD = SdkField
            .<AvcIntraUhdSettings> builder(MarshallingType.SDK_POJO).memberName("AvcIntraUhdSettings")
            .getter(getter(AvcIntraSettings::avcIntraUhdSettings)).setter(setter(Builder::avcIntraUhdSettings))
            .constructor(AvcIntraUhdSettings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("avcIntraUhdSettings").build())
            .build();

    private static final SdkField<String> FRAMERATE_CONTROL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FramerateControl").getter(getter(AvcIntraSettings::framerateControlAsString))
            .setter(setter(Builder::framerateControl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("framerateControl").build()).build();

    private static final SdkField<String> FRAMERATE_CONVERSION_ALGORITHM_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("FramerateConversionAlgorithm")
            .getter(getter(AvcIntraSettings::framerateConversionAlgorithmAsString))
            .setter(setter(Builder::framerateConversionAlgorithm))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("framerateConversionAlgorithm")
                    .build()).build();

    private static final SdkField<Integer> FRAMERATE_DENOMINATOR_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("FramerateDenominator").getter(getter(AvcIntraSettings::framerateDenominator))
            .setter(setter(Builder::framerateDenominator))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("framerateDenominator").build())
            .build();

    private static final SdkField<Integer> FRAMERATE_NUMERATOR_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("FramerateNumerator").getter(getter(AvcIntraSettings::framerateNumerator))
            .setter(setter(Builder::framerateNumerator))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("framerateNumerator").build())
            .build();

    private static final SdkField<String> INTERLACE_MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("InterlaceMode").getter(getter(AvcIntraSettings::interlaceModeAsString))
            .setter(setter(Builder::interlaceMode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("interlaceMode").build()).build();

    private static final SdkField<String> SCAN_TYPE_CONVERSION_MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ScanTypeConversionMode").getter(getter(AvcIntraSettings::scanTypeConversionModeAsString))
            .setter(setter(Builder::scanTypeConversionMode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("scanTypeConversionMode").build())
            .build();

    private static final SdkField<String> SLOW_PAL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SlowPal").getter(getter(AvcIntraSettings::slowPalAsString)).setter(setter(Builder::slowPal))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("slowPal").build()).build();

    private static final SdkField<String> TELECINE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Telecine").getter(getter(AvcIntraSettings::telecineAsString)).setter(setter(Builder::telecine))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("telecine").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(AVC_INTRA_CLASS_FIELD,
            AVC_INTRA_UHD_SETTINGS_FIELD, FRAMERATE_CONTROL_FIELD, FRAMERATE_CONVERSION_ALGORITHM_FIELD,
            FRAMERATE_DENOMINATOR_FIELD, FRAMERATE_NUMERATOR_FIELD, INTERLACE_MODE_FIELD, SCAN_TYPE_CONVERSION_MODE_FIELD,
            SLOW_PAL_FIELD, TELECINE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String avcIntraClass;

    private final AvcIntraUhdSettings avcIntraUhdSettings;

    private final String framerateControl;

    private final String framerateConversionAlgorithm;

    private final Integer framerateDenominator;

    private final Integer framerateNumerator;

    private final String interlaceMode;

    private final String scanTypeConversionMode;

    private final String slowPal;

    private final String telecine;

    private AvcIntraSettings(BuilderImpl builder) {
        this.avcIntraClass = builder.avcIntraClass;
        this.avcIntraUhdSettings = builder.avcIntraUhdSettings;
        this.framerateControl = builder.framerateControl;
        this.framerateConversionAlgorithm = builder.framerateConversionAlgorithm;
        this.framerateDenominator = builder.framerateDenominator;
        this.framerateNumerator = builder.framerateNumerator;
        this.interlaceMode = builder.interlaceMode;
        this.scanTypeConversionMode = builder.scanTypeConversionMode;
        this.slowPal = builder.slowPal;
        this.telecine = builder.telecine;
    }

    /**
     * Specify the AVC-Intra class of your output. The AVC-Intra class selection determines the output video bit rate
     * depending on the frame rate of the output. Outputs with higher class values have higher bitrates and improved
     * image quality. Note that for Class 4K/2K, MediaConvert supports only 4:2:2 chroma subsampling.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #avcIntraClass}
     * will return {@link AvcIntraClass#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #avcIntraClassAsString}.
     * </p>
     * 
     * @return Specify the AVC-Intra class of your output. The AVC-Intra class selection determines the output video bit
     *         rate depending on the frame rate of the output. Outputs with higher class values have higher bitrates and
     *         improved image quality. Note that for Class 4K/2K, MediaConvert supports only 4:2:2 chroma subsampling.
     * @see AvcIntraClass
     */
    public final AvcIntraClass avcIntraClass() {
        return AvcIntraClass.fromValue(avcIntraClass);
    }

    /**
     * Specify the AVC-Intra class of your output. The AVC-Intra class selection determines the output video bit rate
     * depending on the frame rate of the output. Outputs with higher class values have higher bitrates and improved
     * image quality. Note that for Class 4K/2K, MediaConvert supports only 4:2:2 chroma subsampling.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #avcIntraClass}
     * will return {@link AvcIntraClass#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #avcIntraClassAsString}.
     * </p>
     * 
     * @return Specify the AVC-Intra class of your output. The AVC-Intra class selection determines the output video bit
     *         rate depending on the frame rate of the output. Outputs with higher class values have higher bitrates and
     *         improved image quality. Note that for Class 4K/2K, MediaConvert supports only 4:2:2 chroma subsampling.
     * @see AvcIntraClass
     */
    public final String avcIntraClassAsString() {
        return avcIntraClass;
    }

    /**
     * Optional when you set AVC-Intra class to Class 4K/2K. When you set AVC-Intra class to a different value, this
     * object isn't allowed.
     * 
     * @return Optional when you set AVC-Intra class to Class 4K/2K. When you set AVC-Intra class to a different value,
     *         this object isn't allowed.
     */
    public final AvcIntraUhdSettings avcIntraUhdSettings() {
        return avcIntraUhdSettings;
    }

    /**
     * If you are using the console, use the Framerate setting to specify the frame rate for this output. If you want to
     * keep the same frame rate as the input video, choose Follow source. If you want to do frame rate conversion,
     * choose a frame rate from the dropdown list or choose Custom. The framerates shown in the dropdown list are
     * decimal approximations of fractions. If you choose Custom, specify your frame rate as a fraction.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #framerateControl}
     * will return {@link AvcIntraFramerateControl#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #framerateControlAsString}.
     * </p>
     * 
     * @return If you are using the console, use the Framerate setting to specify the frame rate for this output. If you
     *         want to keep the same frame rate as the input video, choose Follow source. If you want to do frame rate
     *         conversion, choose a frame rate from the dropdown list or choose Custom. The framerates shown in the
     *         dropdown list are decimal approximations of fractions. If you choose Custom, specify your frame rate as a
     *         fraction.
     * @see AvcIntraFramerateControl
     */
    public final AvcIntraFramerateControl framerateControl() {
        return AvcIntraFramerateControl.fromValue(framerateControl);
    }

    /**
     * If you are using the console, use the Framerate setting to specify the frame rate for this output. If you want to
     * keep the same frame rate as the input video, choose Follow source. If you want to do frame rate conversion,
     * choose a frame rate from the dropdown list or choose Custom. The framerates shown in the dropdown list are
     * decimal approximations of fractions. If you choose Custom, specify your frame rate as a fraction.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #framerateControl}
     * will return {@link AvcIntraFramerateControl#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #framerateControlAsString}.
     * </p>
     * 
     * @return If you are using the console, use the Framerate setting to specify the frame rate for this output. If you
     *         want to keep the same frame rate as the input video, choose Follow source. If you want to do frame rate
     *         conversion, choose a frame rate from the dropdown list or choose Custom. The framerates shown in the
     *         dropdown list are decimal approximations of fractions. If you choose Custom, specify your frame rate as a
     *         fraction.
     * @see AvcIntraFramerateControl
     */
    public final String framerateControlAsString() {
        return framerateControl;
    }

    /**
     * Choose the method that you want MediaConvert to use when increasing or decreasing the frame rate. For numerically
     * simple conversions, such as 60 fps to 30 fps: We recommend that you keep the default value, Drop duplicate. For
     * numerically complex conversions, to avoid stutter: Choose Interpolate. This results in a smooth picture, but
     * might introduce undesirable video artifacts. For complex frame rate conversions, especially if your source video
     * has already been converted from its original cadence: Choose FrameFormer to do motion-compensated interpolation.
     * FrameFormer uses the best conversion method frame by frame. Note that using FrameFormer increases the transcoding
     * time and incurs a significant add-on cost. When you choose FrameFormer, your input video resolution must be at
     * least 128x96.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #framerateConversionAlgorithm} will return
     * {@link AvcIntraFramerateConversionAlgorithm#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #framerateConversionAlgorithmAsString}.
     * </p>
     * 
     * @return Choose the method that you want MediaConvert to use when increasing or decreasing the frame rate. For
     *         numerically simple conversions, such as 60 fps to 30 fps: We recommend that you keep the default value,
     *         Drop duplicate. For numerically complex conversions, to avoid stutter: Choose Interpolate. This results
     *         in a smooth picture, but might introduce undesirable video artifacts. For complex frame rate conversions,
     *         especially if your source video has already been converted from its original cadence: Choose FrameFormer
     *         to do motion-compensated interpolation. FrameFormer uses the best conversion method frame by frame. Note
     *         that using FrameFormer increases the transcoding time and incurs a significant add-on cost. When you
     *         choose FrameFormer, your input video resolution must be at least 128x96.
     * @see AvcIntraFramerateConversionAlgorithm
     */
    public final AvcIntraFramerateConversionAlgorithm framerateConversionAlgorithm() {
        return AvcIntraFramerateConversionAlgorithm.fromValue(framerateConversionAlgorithm);
    }

    /**
     * Choose the method that you want MediaConvert to use when increasing or decreasing the frame rate. For numerically
     * simple conversions, such as 60 fps to 30 fps: We recommend that you keep the default value, Drop duplicate. For
     * numerically complex conversions, to avoid stutter: Choose Interpolate. This results in a smooth picture, but
     * might introduce undesirable video artifacts. For complex frame rate conversions, especially if your source video
     * has already been converted from its original cadence: Choose FrameFormer to do motion-compensated interpolation.
     * FrameFormer uses the best conversion method frame by frame. Note that using FrameFormer increases the transcoding
     * time and incurs a significant add-on cost. When you choose FrameFormer, your input video resolution must be at
     * least 128x96.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #framerateConversionAlgorithm} will return
     * {@link AvcIntraFramerateConversionAlgorithm#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #framerateConversionAlgorithmAsString}.
     * </p>
     * 
     * @return Choose the method that you want MediaConvert to use when increasing or decreasing the frame rate. For
     *         numerically simple conversions, such as 60 fps to 30 fps: We recommend that you keep the default value,
     *         Drop duplicate. For numerically complex conversions, to avoid stutter: Choose Interpolate. This results
     *         in a smooth picture, but might introduce undesirable video artifacts. For complex frame rate conversions,
     *         especially if your source video has already been converted from its original cadence: Choose FrameFormer
     *         to do motion-compensated interpolation. FrameFormer uses the best conversion method frame by frame. Note
     *         that using FrameFormer increases the transcoding time and incurs a significant add-on cost. When you
     *         choose FrameFormer, your input video resolution must be at least 128x96.
     * @see AvcIntraFramerateConversionAlgorithm
     */
    public final String framerateConversionAlgorithmAsString() {
        return framerateConversionAlgorithm;
    }

    /**
     * When you use the API for transcode jobs that use frame rate conversion, specify the frame rate as a fraction. For
     * example, 24000 / 1001 = 23.976 fps. Use FramerateDenominator to specify the denominator of this fraction. In this
     * example, use 1001 for the value of FramerateDenominator. When you use the console for transcode jobs that use
     * frame rate conversion, provide the value as a decimal number for Framerate. In this example, specify 23.976.
     * 
     * @return When you use the API for transcode jobs that use frame rate conversion, specify the frame rate as a
     *         fraction. For example, 24000 / 1001 = 23.976 fps. Use FramerateDenominator to specify the denominator of
     *         this fraction. In this example, use 1001 for the value of FramerateDenominator. When you use the console
     *         for transcode jobs that use frame rate conversion, provide the value as a decimal number for Framerate.
     *         In this example, specify 23.976.
     */
    public final Integer framerateDenominator() {
        return framerateDenominator;
    }

    /**
     * When you use the API for transcode jobs that use frame rate conversion, specify the frame rate as a fraction. For
     * example, 24000 / 1001 = 23.976 fps. Use FramerateNumerator to specify the numerator of this fraction. In this
     * example, use 24000 for the value of FramerateNumerator. When you use the console for transcode jobs that use
     * frame rate conversion, provide the value as a decimal number for Framerate. In this example, specify 23.976.
     * 
     * @return When you use the API for transcode jobs that use frame rate conversion, specify the frame rate as a
     *         fraction. For example, 24000 / 1001 = 23.976 fps. Use FramerateNumerator to specify the numerator of this
     *         fraction. In this example, use 24000 for the value of FramerateNumerator. When you use the console for
     *         transcode jobs that use frame rate conversion, provide the value as a decimal number for Framerate. In
     *         this example, specify 23.976.
     */
    public final Integer framerateNumerator() {
        return framerateNumerator;
    }

    /**
     * Choose the scan line type for the output. Keep the default value, Progressive to create a progressive output,
     * regardless of the scan type of your input. Use Top field first or Bottom field first to create an output that's
     * interlaced with the same field polarity throughout. Use Follow, default top or Follow, default bottom to produce
     * outputs with the same field polarity as the source. For jobs that have multiple inputs, the output field polarity
     * might change over the course of the output. Follow behavior depends on the input scan type. If the source is
     * interlaced, the output will be interlaced with the same polarity as the source. If the source is progressive, the
     * output will be interlaced with top field bottom field first, depending on which of the Follow options you choose.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #interlaceMode}
     * will return {@link AvcIntraInterlaceMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #interlaceModeAsString}.
     * </p>
     * 
     * @return Choose the scan line type for the output. Keep the default value, Progressive to create a progressive
     *         output, regardless of the scan type of your input. Use Top field first or Bottom field first to create an
     *         output that's interlaced with the same field polarity throughout. Use Follow, default top or Follow,
     *         default bottom to produce outputs with the same field polarity as the source. For jobs that have multiple
     *         inputs, the output field polarity might change over the course of the output. Follow behavior depends on
     *         the input scan type. If the source is interlaced, the output will be interlaced with the same polarity as
     *         the source. If the source is progressive, the output will be interlaced with top field bottom field
     *         first, depending on which of the Follow options you choose.
     * @see AvcIntraInterlaceMode
     */
    public final AvcIntraInterlaceMode interlaceMode() {
        return AvcIntraInterlaceMode.fromValue(interlaceMode);
    }

    /**
     * Choose the scan line type for the output. Keep the default value, Progressive to create a progressive output,
     * regardless of the scan type of your input. Use Top field first or Bottom field first to create an output that's
     * interlaced with the same field polarity throughout. Use Follow, default top or Follow, default bottom to produce
     * outputs with the same field polarity as the source. For jobs that have multiple inputs, the output field polarity
     * might change over the course of the output. Follow behavior depends on the input scan type. If the source is
     * interlaced, the output will be interlaced with the same polarity as the source. If the source is progressive, the
     * output will be interlaced with top field bottom field first, depending on which of the Follow options you choose.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #interlaceMode}
     * will return {@link AvcIntraInterlaceMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #interlaceModeAsString}.
     * </p>
     * 
     * @return Choose the scan line type for the output. Keep the default value, Progressive to create a progressive
     *         output, regardless of the scan type of your input. Use Top field first or Bottom field first to create an
     *         output that's interlaced with the same field polarity throughout. Use Follow, default top or Follow,
     *         default bottom to produce outputs with the same field polarity as the source. For jobs that have multiple
     *         inputs, the output field polarity might change over the course of the output. Follow behavior depends on
     *         the input scan type. If the source is interlaced, the output will be interlaced with the same polarity as
     *         the source. If the source is progressive, the output will be interlaced with top field bottom field
     *         first, depending on which of the Follow options you choose.
     * @see AvcIntraInterlaceMode
     */
    public final String interlaceModeAsString() {
        return interlaceMode;
    }

    /**
     * Use this setting for interlaced outputs, when your output frame rate is half of your input frame rate. In this
     * situation, choose Optimized interlacing to create a better quality interlaced output. In this case, each
     * progressive frame from the input corresponds to an interlaced field in the output. Keep the default value, Basic
     * interlacing, for all other output frame rates. With basic interlacing, MediaConvert performs any frame rate
     * conversion first and then interlaces the frames. When you choose Optimized interlacing and you set your output
     * frame rate to a value that isn't suitable for optimized interlacing, MediaConvert automatically falls back to
     * basic interlacing. Required settings: To use optimized interlacing, you must set Telecine to None or Soft. You
     * can't use optimized interlacing for hard telecine outputs. You must also set Interlace mode to a value other than
     * Progressive.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #scanTypeConversionMode} will return {@link AvcIntraScanTypeConversionMode#UNKNOWN_TO_SDK_VERSION}. The
     * raw value returned by the service is available from {@link #scanTypeConversionModeAsString}.
     * </p>
     * 
     * @return Use this setting for interlaced outputs, when your output frame rate is half of your input frame rate. In
     *         this situation, choose Optimized interlacing to create a better quality interlaced output. In this case,
     *         each progressive frame from the input corresponds to an interlaced field in the output. Keep the default
     *         value, Basic interlacing, for all other output frame rates. With basic interlacing, MediaConvert performs
     *         any frame rate conversion first and then interlaces the frames. When you choose Optimized interlacing and
     *         you set your output frame rate to a value that isn't suitable for optimized interlacing, MediaConvert
     *         automatically falls back to basic interlacing. Required settings: To use optimized interlacing, you must
     *         set Telecine to None or Soft. You can't use optimized interlacing for hard telecine outputs. You must
     *         also set Interlace mode to a value other than Progressive.
     * @see AvcIntraScanTypeConversionMode
     */
    public final AvcIntraScanTypeConversionMode scanTypeConversionMode() {
        return AvcIntraScanTypeConversionMode.fromValue(scanTypeConversionMode);
    }

    /**
     * Use this setting for interlaced outputs, when your output frame rate is half of your input frame rate. In this
     * situation, choose Optimized interlacing to create a better quality interlaced output. In this case, each
     * progressive frame from the input corresponds to an interlaced field in the output. Keep the default value, Basic
     * interlacing, for all other output frame rates. With basic interlacing, MediaConvert performs any frame rate
     * conversion first and then interlaces the frames. When you choose Optimized interlacing and you set your output
     * frame rate to a value that isn't suitable for optimized interlacing, MediaConvert automatically falls back to
     * basic interlacing. Required settings: To use optimized interlacing, you must set Telecine to None or Soft. You
     * can't use optimized interlacing for hard telecine outputs. You must also set Interlace mode to a value other than
     * Progressive.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #scanTypeConversionMode} will return {@link AvcIntraScanTypeConversionMode#UNKNOWN_TO_SDK_VERSION}. The
     * raw value returned by the service is available from {@link #scanTypeConversionModeAsString}.
     * </p>
     * 
     * @return Use this setting for interlaced outputs, when your output frame rate is half of your input frame rate. In
     *         this situation, choose Optimized interlacing to create a better quality interlaced output. In this case,
     *         each progressive frame from the input corresponds to an interlaced field in the output. Keep the default
     *         value, Basic interlacing, for all other output frame rates. With basic interlacing, MediaConvert performs
     *         any frame rate conversion first and then interlaces the frames. When you choose Optimized interlacing and
     *         you set your output frame rate to a value that isn't suitable for optimized interlacing, MediaConvert
     *         automatically falls back to basic interlacing. Required settings: To use optimized interlacing, you must
     *         set Telecine to None or Soft. You can't use optimized interlacing for hard telecine outputs. You must
     *         also set Interlace mode to a value other than Progressive.
     * @see AvcIntraScanTypeConversionMode
     */
    public final String scanTypeConversionModeAsString() {
        return scanTypeConversionMode;
    }

    /**
     * Ignore this setting unless your input frame rate is 23.976 or 24 frames per second (fps). Enable slow PAL to
     * create a 25 fps output. When you enable slow PAL, MediaConvert relabels the video frames to 25 fps and resamples
     * your audio to keep it synchronized with the video. Note that enabling this setting will slightly reduce the
     * duration of your video. Required settings: You must also set Framerate to 25.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #slowPal} will
     * return {@link AvcIntraSlowPal#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #slowPalAsString}.
     * </p>
     * 
     * @return Ignore this setting unless your input frame rate is 23.976 or 24 frames per second (fps). Enable slow PAL
     *         to create a 25 fps output. When you enable slow PAL, MediaConvert relabels the video frames to 25 fps and
     *         resamples your audio to keep it synchronized with the video. Note that enabling this setting will
     *         slightly reduce the duration of your video. Required settings: You must also set Framerate to 25.
     * @see AvcIntraSlowPal
     */
    public final AvcIntraSlowPal slowPal() {
        return AvcIntraSlowPal.fromValue(slowPal);
    }

    /**
     * Ignore this setting unless your input frame rate is 23.976 or 24 frames per second (fps). Enable slow PAL to
     * create a 25 fps output. When you enable slow PAL, MediaConvert relabels the video frames to 25 fps and resamples
     * your audio to keep it synchronized with the video. Note that enabling this setting will slightly reduce the
     * duration of your video. Required settings: You must also set Framerate to 25.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #slowPal} will
     * return {@link AvcIntraSlowPal#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #slowPalAsString}.
     * </p>
     * 
     * @return Ignore this setting unless your input frame rate is 23.976 or 24 frames per second (fps). Enable slow PAL
     *         to create a 25 fps output. When you enable slow PAL, MediaConvert relabels the video frames to 25 fps and
     *         resamples your audio to keep it synchronized with the video. Note that enabling this setting will
     *         slightly reduce the duration of your video. Required settings: You must also set Framerate to 25.
     * @see AvcIntraSlowPal
     */
    public final String slowPalAsString() {
        return slowPal;
    }

    /**
     * When you do frame rate conversion from 23.976 frames per second (fps) to 29.97 fps, and your output scan type is
     * interlaced, you can optionally enable hard telecine to create a smoother picture. When you keep the default
     * value, None, MediaConvert does a standard frame rate conversion to 29.97 without doing anything with the field
     * polarity to create a smoother picture.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #telecine} will
     * return {@link AvcIntraTelecine#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #telecineAsString}.
     * </p>
     * 
     * @return When you do frame rate conversion from 23.976 frames per second (fps) to 29.97 fps, and your output scan
     *         type is interlaced, you can optionally enable hard telecine to create a smoother picture. When you keep
     *         the default value, None, MediaConvert does a standard frame rate conversion to 29.97 without doing
     *         anything with the field polarity to create a smoother picture.
     * @see AvcIntraTelecine
     */
    public final AvcIntraTelecine telecine() {
        return AvcIntraTelecine.fromValue(telecine);
    }

    /**
     * When you do frame rate conversion from 23.976 frames per second (fps) to 29.97 fps, and your output scan type is
     * interlaced, you can optionally enable hard telecine to create a smoother picture. When you keep the default
     * value, None, MediaConvert does a standard frame rate conversion to 29.97 without doing anything with the field
     * polarity to create a smoother picture.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #telecine} will
     * return {@link AvcIntraTelecine#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #telecineAsString}.
     * </p>
     * 
     * @return When you do frame rate conversion from 23.976 frames per second (fps) to 29.97 fps, and your output scan
     *         type is interlaced, you can optionally enable hard telecine to create a smoother picture. When you keep
     *         the default value, None, MediaConvert does a standard frame rate conversion to 29.97 without doing
     *         anything with the field polarity to create a smoother picture.
     * @see AvcIntraTelecine
     */
    public final String telecineAsString() {
        return telecine;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(avcIntraClassAsString());
        hashCode = 31 * hashCode + Objects.hashCode(avcIntraUhdSettings());
        hashCode = 31 * hashCode + Objects.hashCode(framerateControlAsString());
        hashCode = 31 * hashCode + Objects.hashCode(framerateConversionAlgorithmAsString());
        hashCode = 31 * hashCode + Objects.hashCode(framerateDenominator());
        hashCode = 31 * hashCode + Objects.hashCode(framerateNumerator());
        hashCode = 31 * hashCode + Objects.hashCode(interlaceModeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(scanTypeConversionModeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(slowPalAsString());
        hashCode = 31 * hashCode + Objects.hashCode(telecineAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AvcIntraSettings)) {
            return false;
        }
        AvcIntraSettings other = (AvcIntraSettings) obj;
        return Objects.equals(avcIntraClassAsString(), other.avcIntraClassAsString())
                && Objects.equals(avcIntraUhdSettings(), other.avcIntraUhdSettings())
                && Objects.equals(framerateControlAsString(), other.framerateControlAsString())
                && Objects.equals(framerateConversionAlgorithmAsString(), other.framerateConversionAlgorithmAsString())
                && Objects.equals(framerateDenominator(), other.framerateDenominator())
                && Objects.equals(framerateNumerator(), other.framerateNumerator())
                && Objects.equals(interlaceModeAsString(), other.interlaceModeAsString())
                && Objects.equals(scanTypeConversionModeAsString(), other.scanTypeConversionModeAsString())
                && Objects.equals(slowPalAsString(), other.slowPalAsString())
                && Objects.equals(telecineAsString(), other.telecineAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AvcIntraSettings").add("AvcIntraClass", avcIntraClassAsString())
                .add("AvcIntraUhdSettings", avcIntraUhdSettings()).add("FramerateControl", framerateControlAsString())
                .add("FramerateConversionAlgorithm", framerateConversionAlgorithmAsString())
                .add("FramerateDenominator", framerateDenominator()).add("FramerateNumerator", framerateNumerator())
                .add("InterlaceMode", interlaceModeAsString()).add("ScanTypeConversionMode", scanTypeConversionModeAsString())
                .add("SlowPal", slowPalAsString()).add("Telecine", telecineAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AvcIntraClass":
            return Optional.ofNullable(clazz.cast(avcIntraClassAsString()));
        case "AvcIntraUhdSettings":
            return Optional.ofNullable(clazz.cast(avcIntraUhdSettings()));
        case "FramerateControl":
            return Optional.ofNullable(clazz.cast(framerateControlAsString()));
        case "FramerateConversionAlgorithm":
            return Optional.ofNullable(clazz.cast(framerateConversionAlgorithmAsString()));
        case "FramerateDenominator":
            return Optional.ofNullable(clazz.cast(framerateDenominator()));
        case "FramerateNumerator":
            return Optional.ofNullable(clazz.cast(framerateNumerator()));
        case "InterlaceMode":
            return Optional.ofNullable(clazz.cast(interlaceModeAsString()));
        case "ScanTypeConversionMode":
            return Optional.ofNullable(clazz.cast(scanTypeConversionModeAsString()));
        case "SlowPal":
            return Optional.ofNullable(clazz.cast(slowPalAsString()));
        case "Telecine":
            return Optional.ofNullable(clazz.cast(telecineAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<AvcIntraSettings, T> g) {
        return obj -> g.apply((AvcIntraSettings) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AvcIntraSettings> {
        /**
         * Specify the AVC-Intra class of your output. The AVC-Intra class selection determines the output video bit
         * rate depending on the frame rate of the output. Outputs with higher class values have higher bitrates and
         * improved image quality. Note that for Class 4K/2K, MediaConvert supports only 4:2:2 chroma subsampling.
         * 
         * @param avcIntraClass
         *        Specify the AVC-Intra class of your output. The AVC-Intra class selection determines the output video
         *        bit rate depending on the frame rate of the output. Outputs with higher class values have higher
         *        bitrates and improved image quality. Note that for Class 4K/2K, MediaConvert supports only 4:2:2
         *        chroma subsampling.
         * @see AvcIntraClass
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AvcIntraClass
         */
        Builder avcIntraClass(String avcIntraClass);

        /**
         * Specify the AVC-Intra class of your output. The AVC-Intra class selection determines the output video bit
         * rate depending on the frame rate of the output. Outputs with higher class values have higher bitrates and
         * improved image quality. Note that for Class 4K/2K, MediaConvert supports only 4:2:2 chroma subsampling.
         * 
         * @param avcIntraClass
         *        Specify the AVC-Intra class of your output. The AVC-Intra class selection determines the output video
         *        bit rate depending on the frame rate of the output. Outputs with higher class values have higher
         *        bitrates and improved image quality. Note that for Class 4K/2K, MediaConvert supports only 4:2:2
         *        chroma subsampling.
         * @see AvcIntraClass
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AvcIntraClass
         */
        Builder avcIntraClass(AvcIntraClass avcIntraClass);

        /**
         * Optional when you set AVC-Intra class to Class 4K/2K. When you set AVC-Intra class to a different value, this
         * object isn't allowed.
         * 
         * @param avcIntraUhdSettings
         *        Optional when you set AVC-Intra class to Class 4K/2K. When you set AVC-Intra class to a different
         *        value, this object isn't allowed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder avcIntraUhdSettings(AvcIntraUhdSettings avcIntraUhdSettings);

        /**
         * Optional when you set AVC-Intra class to Class 4K/2K. When you set AVC-Intra class to a different value, this
         * object isn't allowed. This is a convenience method that creates an instance of the
         * {@link AvcIntraUhdSettings.Builder} avoiding the need to create one manually via
         * {@link AvcIntraUhdSettings#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AvcIntraUhdSettings.Builder#build()} is called immediately and
         * its result is passed to {@link #avcIntraUhdSettings(AvcIntraUhdSettings)}.
         * 
         * @param avcIntraUhdSettings
         *        a consumer that will call methods on {@link AvcIntraUhdSettings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #avcIntraUhdSettings(AvcIntraUhdSettings)
         */
        default Builder avcIntraUhdSettings(Consumer<AvcIntraUhdSettings.Builder> avcIntraUhdSettings) {
            return avcIntraUhdSettings(AvcIntraUhdSettings.builder().applyMutation(avcIntraUhdSettings).build());
        }

        /**
         * If you are using the console, use the Framerate setting to specify the frame rate for this output. If you
         * want to keep the same frame rate as the input video, choose Follow source. If you want to do frame rate
         * conversion, choose a frame rate from the dropdown list or choose Custom. The framerates shown in the dropdown
         * list are decimal approximations of fractions. If you choose Custom, specify your frame rate as a fraction.
         * 
         * @param framerateControl
         *        If you are using the console, use the Framerate setting to specify the frame rate for this output. If
         *        you want to keep the same frame rate as the input video, choose Follow source. If you want to do frame
         *        rate conversion, choose a frame rate from the dropdown list or choose Custom. The framerates shown in
         *        the dropdown list are decimal approximations of fractions. If you choose Custom, specify your frame
         *        rate as a fraction.
         * @see AvcIntraFramerateControl
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AvcIntraFramerateControl
         */
        Builder framerateControl(String framerateControl);

        /**
         * If you are using the console, use the Framerate setting to specify the frame rate for this output. If you
         * want to keep the same frame rate as the input video, choose Follow source. If you want to do frame rate
         * conversion, choose a frame rate from the dropdown list or choose Custom. The framerates shown in the dropdown
         * list are decimal approximations of fractions. If you choose Custom, specify your frame rate as a fraction.
         * 
         * @param framerateControl
         *        If you are using the console, use the Framerate setting to specify the frame rate for this output. If
         *        you want to keep the same frame rate as the input video, choose Follow source. If you want to do frame
         *        rate conversion, choose a frame rate from the dropdown list or choose Custom. The framerates shown in
         *        the dropdown list are decimal approximations of fractions. If you choose Custom, specify your frame
         *        rate as a fraction.
         * @see AvcIntraFramerateControl
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AvcIntraFramerateControl
         */
        Builder framerateControl(AvcIntraFramerateControl framerateControl);

        /**
         * Choose the method that you want MediaConvert to use when increasing or decreasing the frame rate. For
         * numerically simple conversions, such as 60 fps to 30 fps: We recommend that you keep the default value, Drop
         * duplicate. For numerically complex conversions, to avoid stutter: Choose Interpolate. This results in a
         * smooth picture, but might introduce undesirable video artifacts. For complex frame rate conversions,
         * especially if your source video has already been converted from its original cadence: Choose FrameFormer to
         * do motion-compensated interpolation. FrameFormer uses the best conversion method frame by frame. Note that
         * using FrameFormer increases the transcoding time and incurs a significant add-on cost. When you choose
         * FrameFormer, your input video resolution must be at least 128x96.
         * 
         * @param framerateConversionAlgorithm
         *        Choose the method that you want MediaConvert to use when increasing or decreasing the frame rate. For
         *        numerically simple conversions, such as 60 fps to 30 fps: We recommend that you keep the default
         *        value, Drop duplicate. For numerically complex conversions, to avoid stutter: Choose Interpolate. This
         *        results in a smooth picture, but might introduce undesirable video artifacts. For complex frame rate
         *        conversions, especially if your source video has already been converted from its original cadence:
         *        Choose FrameFormer to do motion-compensated interpolation. FrameFormer uses the best conversion method
         *        frame by frame. Note that using FrameFormer increases the transcoding time and incurs a significant
         *        add-on cost. When you choose FrameFormer, your input video resolution must be at least 128x96.
         * @see AvcIntraFramerateConversionAlgorithm
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AvcIntraFramerateConversionAlgorithm
         */
        Builder framerateConversionAlgorithm(String framerateConversionAlgorithm);

        /**
         * Choose the method that you want MediaConvert to use when increasing or decreasing the frame rate. For
         * numerically simple conversions, such as 60 fps to 30 fps: We recommend that you keep the default value, Drop
         * duplicate. For numerically complex conversions, to avoid stutter: Choose Interpolate. This results in a
         * smooth picture, but might introduce undesirable video artifacts. For complex frame rate conversions,
         * especially if your source video has already been converted from its original cadence: Choose FrameFormer to
         * do motion-compensated interpolation. FrameFormer uses the best conversion method frame by frame. Note that
         * using FrameFormer increases the transcoding time and incurs a significant add-on cost. When you choose
         * FrameFormer, your input video resolution must be at least 128x96.
         * 
         * @param framerateConversionAlgorithm
         *        Choose the method that you want MediaConvert to use when increasing or decreasing the frame rate. For
         *        numerically simple conversions, such as 60 fps to 30 fps: We recommend that you keep the default
         *        value, Drop duplicate. For numerically complex conversions, to avoid stutter: Choose Interpolate. This
         *        results in a smooth picture, but might introduce undesirable video artifacts. For complex frame rate
         *        conversions, especially if your source video has already been converted from its original cadence:
         *        Choose FrameFormer to do motion-compensated interpolation. FrameFormer uses the best conversion method
         *        frame by frame. Note that using FrameFormer increases the transcoding time and incurs a significant
         *        add-on cost. When you choose FrameFormer, your input video resolution must be at least 128x96.
         * @see AvcIntraFramerateConversionAlgorithm
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AvcIntraFramerateConversionAlgorithm
         */
        Builder framerateConversionAlgorithm(AvcIntraFramerateConversionAlgorithm framerateConversionAlgorithm);

        /**
         * When you use the API for transcode jobs that use frame rate conversion, specify the frame rate as a fraction.
         * For example, 24000 / 1001 = 23.976 fps. Use FramerateDenominator to specify the denominator of this fraction.
         * In this example, use 1001 for the value of FramerateDenominator. When you use the console for transcode jobs
         * that use frame rate conversion, provide the value as a decimal number for Framerate. In this example, specify
         * 23.976.
         * 
         * @param framerateDenominator
         *        When you use the API for transcode jobs that use frame rate conversion, specify the frame rate as a
         *        fraction. For example, 24000 / 1001 = 23.976 fps. Use FramerateDenominator to specify the denominator
         *        of this fraction. In this example, use 1001 for the value of FramerateDenominator. When you use the
         *        console for transcode jobs that use frame rate conversion, provide the value as a decimal number for
         *        Framerate. In this example, specify 23.976.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder framerateDenominator(Integer framerateDenominator);

        /**
         * When you use the API for transcode jobs that use frame rate conversion, specify the frame rate as a fraction.
         * For example, 24000 / 1001 = 23.976 fps. Use FramerateNumerator to specify the numerator of this fraction. In
         * this example, use 24000 for the value of FramerateNumerator. When you use the console for transcode jobs that
         * use frame rate conversion, provide the value as a decimal number for Framerate. In this example, specify
         * 23.976.
         * 
         * @param framerateNumerator
         *        When you use the API for transcode jobs that use frame rate conversion, specify the frame rate as a
         *        fraction. For example, 24000 / 1001 = 23.976 fps. Use FramerateNumerator to specify the numerator of
         *        this fraction. In this example, use 24000 for the value of FramerateNumerator. When you use the
         *        console for transcode jobs that use frame rate conversion, provide the value as a decimal number for
         *        Framerate. In this example, specify 23.976.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder framerateNumerator(Integer framerateNumerator);

        /**
         * Choose the scan line type for the output. Keep the default value, Progressive to create a progressive output,
         * regardless of the scan type of your input. Use Top field first or Bottom field first to create an output
         * that's interlaced with the same field polarity throughout. Use Follow, default top or Follow, default bottom
         * to produce outputs with the same field polarity as the source. For jobs that have multiple inputs, the output
         * field polarity might change over the course of the output. Follow behavior depends on the input scan type. If
         * the source is interlaced, the output will be interlaced with the same polarity as the source. If the source
         * is progressive, the output will be interlaced with top field bottom field first, depending on which of the
         * Follow options you choose.
         * 
         * @param interlaceMode
         *        Choose the scan line type for the output. Keep the default value, Progressive to create a progressive
         *        output, regardless of the scan type of your input. Use Top field first or Bottom field first to create
         *        an output that's interlaced with the same field polarity throughout. Use Follow, default top or
         *        Follow, default bottom to produce outputs with the same field polarity as the source. For jobs that
         *        have multiple inputs, the output field polarity might change over the course of the output. Follow
         *        behavior depends on the input scan type. If the source is interlaced, the output will be interlaced
         *        with the same polarity as the source. If the source is progressive, the output will be interlaced with
         *        top field bottom field first, depending on which of the Follow options you choose.
         * @see AvcIntraInterlaceMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AvcIntraInterlaceMode
         */
        Builder interlaceMode(String interlaceMode);

        /**
         * Choose the scan line type for the output. Keep the default value, Progressive to create a progressive output,
         * regardless of the scan type of your input. Use Top field first or Bottom field first to create an output
         * that's interlaced with the same field polarity throughout. Use Follow, default top or Follow, default bottom
         * to produce outputs with the same field polarity as the source. For jobs that have multiple inputs, the output
         * field polarity might change over the course of the output. Follow behavior depends on the input scan type. If
         * the source is interlaced, the output will be interlaced with the same polarity as the source. If the source
         * is progressive, the output will be interlaced with top field bottom field first, depending on which of the
         * Follow options you choose.
         * 
         * @param interlaceMode
         *        Choose the scan line type for the output. Keep the default value, Progressive to create a progressive
         *        output, regardless of the scan type of your input. Use Top field first or Bottom field first to create
         *        an output that's interlaced with the same field polarity throughout. Use Follow, default top or
         *        Follow, default bottom to produce outputs with the same field polarity as the source. For jobs that
         *        have multiple inputs, the output field polarity might change over the course of the output. Follow
         *        behavior depends on the input scan type. If the source is interlaced, the output will be interlaced
         *        with the same polarity as the source. If the source is progressive, the output will be interlaced with
         *        top field bottom field first, depending on which of the Follow options you choose.
         * @see AvcIntraInterlaceMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AvcIntraInterlaceMode
         */
        Builder interlaceMode(AvcIntraInterlaceMode interlaceMode);

        /**
         * Use this setting for interlaced outputs, when your output frame rate is half of your input frame rate. In
         * this situation, choose Optimized interlacing to create a better quality interlaced output. In this case, each
         * progressive frame from the input corresponds to an interlaced field in the output. Keep the default value,
         * Basic interlacing, for all other output frame rates. With basic interlacing, MediaConvert performs any frame
         * rate conversion first and then interlaces the frames. When you choose Optimized interlacing and you set your
         * output frame rate to a value that isn't suitable for optimized interlacing, MediaConvert automatically falls
         * back to basic interlacing. Required settings: To use optimized interlacing, you must set Telecine to None or
         * Soft. You can't use optimized interlacing for hard telecine outputs. You must also set Interlace mode to a
         * value other than Progressive.
         * 
         * @param scanTypeConversionMode
         *        Use this setting for interlaced outputs, when your output frame rate is half of your input frame rate.
         *        In this situation, choose Optimized interlacing to create a better quality interlaced output. In this
         *        case, each progressive frame from the input corresponds to an interlaced field in the output. Keep the
         *        default value, Basic interlacing, for all other output frame rates. With basic interlacing,
         *        MediaConvert performs any frame rate conversion first and then interlaces the frames. When you choose
         *        Optimized interlacing and you set your output frame rate to a value that isn't suitable for optimized
         *        interlacing, MediaConvert automatically falls back to basic interlacing. Required settings: To use
         *        optimized interlacing, you must set Telecine to None or Soft. You can't use optimized interlacing for
         *        hard telecine outputs. You must also set Interlace mode to a value other than Progressive.
         * @see AvcIntraScanTypeConversionMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AvcIntraScanTypeConversionMode
         */
        Builder scanTypeConversionMode(String scanTypeConversionMode);

        /**
         * Use this setting for interlaced outputs, when your output frame rate is half of your input frame rate. In
         * this situation, choose Optimized interlacing to create a better quality interlaced output. In this case, each
         * progressive frame from the input corresponds to an interlaced field in the output. Keep the default value,
         * Basic interlacing, for all other output frame rates. With basic interlacing, MediaConvert performs any frame
         * rate conversion first and then interlaces the frames. When you choose Optimized interlacing and you set your
         * output frame rate to a value that isn't suitable for optimized interlacing, MediaConvert automatically falls
         * back to basic interlacing. Required settings: To use optimized interlacing, you must set Telecine to None or
         * Soft. You can't use optimized interlacing for hard telecine outputs. You must also set Interlace mode to a
         * value other than Progressive.
         * 
         * @param scanTypeConversionMode
         *        Use this setting for interlaced outputs, when your output frame rate is half of your input frame rate.
         *        In this situation, choose Optimized interlacing to create a better quality interlaced output. In this
         *        case, each progressive frame from the input corresponds to an interlaced field in the output. Keep the
         *        default value, Basic interlacing, for all other output frame rates. With basic interlacing,
         *        MediaConvert performs any frame rate conversion first and then interlaces the frames. When you choose
         *        Optimized interlacing and you set your output frame rate to a value that isn't suitable for optimized
         *        interlacing, MediaConvert automatically falls back to basic interlacing. Required settings: To use
         *        optimized interlacing, you must set Telecine to None or Soft. You can't use optimized interlacing for
         *        hard telecine outputs. You must also set Interlace mode to a value other than Progressive.
         * @see AvcIntraScanTypeConversionMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AvcIntraScanTypeConversionMode
         */
        Builder scanTypeConversionMode(AvcIntraScanTypeConversionMode scanTypeConversionMode);

        /**
         * Ignore this setting unless your input frame rate is 23.976 or 24 frames per second (fps). Enable slow PAL to
         * create a 25 fps output. When you enable slow PAL, MediaConvert relabels the video frames to 25 fps and
         * resamples your audio to keep it synchronized with the video. Note that enabling this setting will slightly
         * reduce the duration of your video. Required settings: You must also set Framerate to 25.
         * 
         * @param slowPal
         *        Ignore this setting unless your input frame rate is 23.976 or 24 frames per second (fps). Enable slow
         *        PAL to create a 25 fps output. When you enable slow PAL, MediaConvert relabels the video frames to 25
         *        fps and resamples your audio to keep it synchronized with the video. Note that enabling this setting
         *        will slightly reduce the duration of your video. Required settings: You must also set Framerate to 25.
         * @see AvcIntraSlowPal
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AvcIntraSlowPal
         */
        Builder slowPal(String slowPal);

        /**
         * Ignore this setting unless your input frame rate is 23.976 or 24 frames per second (fps). Enable slow PAL to
         * create a 25 fps output. When you enable slow PAL, MediaConvert relabels the video frames to 25 fps and
         * resamples your audio to keep it synchronized with the video. Note that enabling this setting will slightly
         * reduce the duration of your video. Required settings: You must also set Framerate to 25.
         * 
         * @param slowPal
         *        Ignore this setting unless your input frame rate is 23.976 or 24 frames per second (fps). Enable slow
         *        PAL to create a 25 fps output. When you enable slow PAL, MediaConvert relabels the video frames to 25
         *        fps and resamples your audio to keep it synchronized with the video. Note that enabling this setting
         *        will slightly reduce the duration of your video. Required settings: You must also set Framerate to 25.
         * @see AvcIntraSlowPal
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AvcIntraSlowPal
         */
        Builder slowPal(AvcIntraSlowPal slowPal);

        /**
         * When you do frame rate conversion from 23.976 frames per second (fps) to 29.97 fps, and your output scan type
         * is interlaced, you can optionally enable hard telecine to create a smoother picture. When you keep the
         * default value, None, MediaConvert does a standard frame rate conversion to 29.97 without doing anything with
         * the field polarity to create a smoother picture.
         * 
         * @param telecine
         *        When you do frame rate conversion from 23.976 frames per second (fps) to 29.97 fps, and your output
         *        scan type is interlaced, you can optionally enable hard telecine to create a smoother picture. When
         *        you keep the default value, None, MediaConvert does a standard frame rate conversion to 29.97 without
         *        doing anything with the field polarity to create a smoother picture.
         * @see AvcIntraTelecine
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AvcIntraTelecine
         */
        Builder telecine(String telecine);

        /**
         * When you do frame rate conversion from 23.976 frames per second (fps) to 29.97 fps, and your output scan type
         * is interlaced, you can optionally enable hard telecine to create a smoother picture. When you keep the
         * default value, None, MediaConvert does a standard frame rate conversion to 29.97 without doing anything with
         * the field polarity to create a smoother picture.
         * 
         * @param telecine
         *        When you do frame rate conversion from 23.976 frames per second (fps) to 29.97 fps, and your output
         *        scan type is interlaced, you can optionally enable hard telecine to create a smoother picture. When
         *        you keep the default value, None, MediaConvert does a standard frame rate conversion to 29.97 without
         *        doing anything with the field polarity to create a smoother picture.
         * @see AvcIntraTelecine
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AvcIntraTelecine
         */
        Builder telecine(AvcIntraTelecine telecine);
    }

    static final class BuilderImpl implements Builder {
        private String avcIntraClass;

        private AvcIntraUhdSettings avcIntraUhdSettings;

        private String framerateControl;

        private String framerateConversionAlgorithm;

        private Integer framerateDenominator;

        private Integer framerateNumerator;

        private String interlaceMode;

        private String scanTypeConversionMode;

        private String slowPal;

        private String telecine;

        private BuilderImpl() {
        }

        private BuilderImpl(AvcIntraSettings model) {
            avcIntraClass(model.avcIntraClass);
            avcIntraUhdSettings(model.avcIntraUhdSettings);
            framerateControl(model.framerateControl);
            framerateConversionAlgorithm(model.framerateConversionAlgorithm);
            framerateDenominator(model.framerateDenominator);
            framerateNumerator(model.framerateNumerator);
            interlaceMode(model.interlaceMode);
            scanTypeConversionMode(model.scanTypeConversionMode);
            slowPal(model.slowPal);
            telecine(model.telecine);
        }

        public final String getAvcIntraClass() {
            return avcIntraClass;
        }

        public final void setAvcIntraClass(String avcIntraClass) {
            this.avcIntraClass = avcIntraClass;
        }

        @Override
        public final Builder avcIntraClass(String avcIntraClass) {
            this.avcIntraClass = avcIntraClass;
            return this;
        }

        @Override
        public final Builder avcIntraClass(AvcIntraClass avcIntraClass) {
            this.avcIntraClass(avcIntraClass == null ? null : avcIntraClass.toString());
            return this;
        }

        public final AvcIntraUhdSettings.Builder getAvcIntraUhdSettings() {
            return avcIntraUhdSettings != null ? avcIntraUhdSettings.toBuilder() : null;
        }

        public final void setAvcIntraUhdSettings(AvcIntraUhdSettings.BuilderImpl avcIntraUhdSettings) {
            this.avcIntraUhdSettings = avcIntraUhdSettings != null ? avcIntraUhdSettings.build() : null;
        }

        @Override
        public final Builder avcIntraUhdSettings(AvcIntraUhdSettings avcIntraUhdSettings) {
            this.avcIntraUhdSettings = avcIntraUhdSettings;
            return this;
        }

        public final String getFramerateControl() {
            return framerateControl;
        }

        public final void setFramerateControl(String framerateControl) {
            this.framerateControl = framerateControl;
        }

        @Override
        public final Builder framerateControl(String framerateControl) {
            this.framerateControl = framerateControl;
            return this;
        }

        @Override
        public final Builder framerateControl(AvcIntraFramerateControl framerateControl) {
            this.framerateControl(framerateControl == null ? null : framerateControl.toString());
            return this;
        }

        public final String getFramerateConversionAlgorithm() {
            return framerateConversionAlgorithm;
        }

        public final void setFramerateConversionAlgorithm(String framerateConversionAlgorithm) {
            this.framerateConversionAlgorithm = framerateConversionAlgorithm;
        }

        @Override
        public final Builder framerateConversionAlgorithm(String framerateConversionAlgorithm) {
            this.framerateConversionAlgorithm = framerateConversionAlgorithm;
            return this;
        }

        @Override
        public final Builder framerateConversionAlgorithm(AvcIntraFramerateConversionAlgorithm framerateConversionAlgorithm) {
            this.framerateConversionAlgorithm(framerateConversionAlgorithm == null ? null : framerateConversionAlgorithm
                    .toString());
            return this;
        }

        public final Integer getFramerateDenominator() {
            return framerateDenominator;
        }

        public final void setFramerateDenominator(Integer framerateDenominator) {
            this.framerateDenominator = framerateDenominator;
        }

        @Override
        public final Builder framerateDenominator(Integer framerateDenominator) {
            this.framerateDenominator = framerateDenominator;
            return this;
        }

        public final Integer getFramerateNumerator() {
            return framerateNumerator;
        }

        public final void setFramerateNumerator(Integer framerateNumerator) {
            this.framerateNumerator = framerateNumerator;
        }

        @Override
        public final Builder framerateNumerator(Integer framerateNumerator) {
            this.framerateNumerator = framerateNumerator;
            return this;
        }

        public final String getInterlaceMode() {
            return interlaceMode;
        }

        public final void setInterlaceMode(String interlaceMode) {
            this.interlaceMode = interlaceMode;
        }

        @Override
        public final Builder interlaceMode(String interlaceMode) {
            this.interlaceMode = interlaceMode;
            return this;
        }

        @Override
        public final Builder interlaceMode(AvcIntraInterlaceMode interlaceMode) {
            this.interlaceMode(interlaceMode == null ? null : interlaceMode.toString());
            return this;
        }

        public final String getScanTypeConversionMode() {
            return scanTypeConversionMode;
        }

        public final void setScanTypeConversionMode(String scanTypeConversionMode) {
            this.scanTypeConversionMode = scanTypeConversionMode;
        }

        @Override
        public final Builder scanTypeConversionMode(String scanTypeConversionMode) {
            this.scanTypeConversionMode = scanTypeConversionMode;
            return this;
        }

        @Override
        public final Builder scanTypeConversionMode(AvcIntraScanTypeConversionMode scanTypeConversionMode) {
            this.scanTypeConversionMode(scanTypeConversionMode == null ? null : scanTypeConversionMode.toString());
            return this;
        }

        public final String getSlowPal() {
            return slowPal;
        }

        public final void setSlowPal(String slowPal) {
            this.slowPal = slowPal;
        }

        @Override
        public final Builder slowPal(String slowPal) {
            this.slowPal = slowPal;
            return this;
        }

        @Override
        public final Builder slowPal(AvcIntraSlowPal slowPal) {
            this.slowPal(slowPal == null ? null : slowPal.toString());
            return this;
        }

        public final String getTelecine() {
            return telecine;
        }

        public final void setTelecine(String telecine) {
            this.telecine = telecine;
        }

        @Override
        public final Builder telecine(String telecine) {
            this.telecine = telecine;
            return this;
        }

        @Override
        public final Builder telecine(AvcIntraTelecine telecine) {
            this.telecine(telecine == null ? null : telecine.toString());
            return this;
        }

        @Override
        public AvcIntraSettings build() {
            return new AvcIntraSettings(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
