/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.fsx.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A description of a specific Amazon File Cache resource, which is a response object from the
 * <code>DescribeFileCaches</code> operation.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class FileCache implements SdkPojo, Serializable, ToCopyableBuilder<FileCache.Builder, FileCache> {
    private static final SdkField<String> OWNER_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("OwnerId").getter(getter(FileCache::ownerId)).setter(setter(Builder::ownerId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OwnerId").build()).build();

    private static final SdkField<Instant> CREATION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreationTime").getter(getter(FileCache::creationTime)).setter(setter(Builder::creationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationTime").build()).build();

    private static final SdkField<String> FILE_CACHE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FileCacheId").getter(getter(FileCache::fileCacheId)).setter(setter(Builder::fileCacheId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FileCacheId").build()).build();

    private static final SdkField<String> FILE_CACHE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FileCacheType").getter(getter(FileCache::fileCacheTypeAsString)).setter(setter(Builder::fileCacheType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FileCacheType").build()).build();

    private static final SdkField<String> FILE_CACHE_TYPE_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FileCacheTypeVersion").getter(getter(FileCache::fileCacheTypeVersion))
            .setter(setter(Builder::fileCacheTypeVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FileCacheTypeVersion").build())
            .build();

    private static final SdkField<String> LIFECYCLE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Lifecycle").getter(getter(FileCache::lifecycleAsString)).setter(setter(Builder::lifecycle))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Lifecycle").build()).build();

    private static final SdkField<FileCacheFailureDetails> FAILURE_DETAILS_FIELD = SdkField
            .<FileCacheFailureDetails> builder(MarshallingType.SDK_POJO).memberName("FailureDetails")
            .getter(getter(FileCache::failureDetails)).setter(setter(Builder::failureDetails))
            .constructor(FileCacheFailureDetails::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FailureDetails").build()).build();

    private static final SdkField<Integer> STORAGE_CAPACITY_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("StorageCapacity").getter(getter(FileCache::storageCapacity)).setter(setter(Builder::storageCapacity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StorageCapacity").build()).build();

    private static final SdkField<String> VPC_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("VpcId")
            .getter(getter(FileCache::vpcId)).setter(setter(Builder::vpcId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcId").build()).build();

    private static final SdkField<List<String>> SUBNET_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("SubnetIds")
            .getter(getter(FileCache::subnetIds))
            .setter(setter(Builder::subnetIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SubnetIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> NETWORK_INTERFACE_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("NetworkInterfaceIds")
            .getter(getter(FileCache::networkInterfaceIds))
            .setter(setter(Builder::networkInterfaceIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NetworkInterfaceIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> DNS_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DNSName").getter(getter(FileCache::dnsName)).setter(setter(Builder::dnsName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DNSName").build()).build();

    private static final SdkField<String> KMS_KEY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("KmsKeyId").getter(getter(FileCache::kmsKeyId)).setter(setter(Builder::kmsKeyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KmsKeyId").build()).build();

    private static final SdkField<String> RESOURCE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceARN").getter(getter(FileCache::resourceARN)).setter(setter(Builder::resourceARN))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceARN").build()).build();

    private static final SdkField<FileCacheLustreConfiguration> LUSTRE_CONFIGURATION_FIELD = SdkField
            .<FileCacheLustreConfiguration> builder(MarshallingType.SDK_POJO).memberName("LustreConfiguration")
            .getter(getter(FileCache::lustreConfiguration)).setter(setter(Builder::lustreConfiguration))
            .constructor(FileCacheLustreConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LustreConfiguration").build())
            .build();

    private static final SdkField<List<String>> DATA_REPOSITORY_ASSOCIATION_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("DataRepositoryAssociationIds")
            .getter(getter(FileCache::dataRepositoryAssociationIds))
            .setter(setter(Builder::dataRepositoryAssociationIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DataRepositoryAssociationIds")
                    .build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(OWNER_ID_FIELD,
            CREATION_TIME_FIELD, FILE_CACHE_ID_FIELD, FILE_CACHE_TYPE_FIELD, FILE_CACHE_TYPE_VERSION_FIELD, LIFECYCLE_FIELD,
            FAILURE_DETAILS_FIELD, STORAGE_CAPACITY_FIELD, VPC_ID_FIELD, SUBNET_IDS_FIELD, NETWORK_INTERFACE_IDS_FIELD,
            DNS_NAME_FIELD, KMS_KEY_ID_FIELD, RESOURCE_ARN_FIELD, LUSTRE_CONFIGURATION_FIELD,
            DATA_REPOSITORY_ASSOCIATION_IDS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String ownerId;

    private final Instant creationTime;

    private final String fileCacheId;

    private final String fileCacheType;

    private final String fileCacheTypeVersion;

    private final String lifecycle;

    private final FileCacheFailureDetails failureDetails;

    private final Integer storageCapacity;

    private final String vpcId;

    private final List<String> subnetIds;

    private final List<String> networkInterfaceIds;

    private final String dnsName;

    private final String kmsKeyId;

    private final String resourceARN;

    private final FileCacheLustreConfiguration lustreConfiguration;

    private final List<String> dataRepositoryAssociationIds;

    private FileCache(BuilderImpl builder) {
        this.ownerId = builder.ownerId;
        this.creationTime = builder.creationTime;
        this.fileCacheId = builder.fileCacheId;
        this.fileCacheType = builder.fileCacheType;
        this.fileCacheTypeVersion = builder.fileCacheTypeVersion;
        this.lifecycle = builder.lifecycle;
        this.failureDetails = builder.failureDetails;
        this.storageCapacity = builder.storageCapacity;
        this.vpcId = builder.vpcId;
        this.subnetIds = builder.subnetIds;
        this.networkInterfaceIds = builder.networkInterfaceIds;
        this.dnsName = builder.dnsName;
        this.kmsKeyId = builder.kmsKeyId;
        this.resourceARN = builder.resourceARN;
        this.lustreConfiguration = builder.lustreConfiguration;
        this.dataRepositoryAssociationIds = builder.dataRepositoryAssociationIds;
    }

    /**
     * Returns the value of the OwnerId property for this object.
     * 
     * @return The value of the OwnerId property for this object.
     */
    public final String ownerId() {
        return ownerId;
    }

    /**
     * Returns the value of the CreationTime property for this object.
     * 
     * @return The value of the CreationTime property for this object.
     */
    public final Instant creationTime() {
        return creationTime;
    }

    /**
     * <p>
     * The system-generated, unique ID of the cache.
     * </p>
     * 
     * @return The system-generated, unique ID of the cache.
     */
    public final String fileCacheId() {
        return fileCacheId;
    }

    /**
     * <p>
     * The type of cache, which must be <code>LUSTRE</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #fileCacheType}
     * will return {@link FileCacheType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #fileCacheTypeAsString}.
     * </p>
     * 
     * @return The type of cache, which must be <code>LUSTRE</code>.
     * @see FileCacheType
     */
    public final FileCacheType fileCacheType() {
        return FileCacheType.fromValue(fileCacheType);
    }

    /**
     * <p>
     * The type of cache, which must be <code>LUSTRE</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #fileCacheType}
     * will return {@link FileCacheType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #fileCacheTypeAsString}.
     * </p>
     * 
     * @return The type of cache, which must be <code>LUSTRE</code>.
     * @see FileCacheType
     */
    public final String fileCacheTypeAsString() {
        return fileCacheType;
    }

    /**
     * <p>
     * The Lustre version of the cache, which must be <code>2.12</code>.
     * </p>
     * 
     * @return The Lustre version of the cache, which must be <code>2.12</code>.
     */
    public final String fileCacheTypeVersion() {
        return fileCacheTypeVersion;
    }

    /**
     * <p>
     * The lifecycle status of the cache. The following are the possible values and what they mean:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>AVAILABLE</code> - The cache is in a healthy state, and is reachable and available for use.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CREATING</code> - The new cache is being created.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELETING</code> - An existing cache is being deleted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>UPDATING</code> - The cache is undergoing a customer-initiated update.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FAILED</code> - An existing cache has experienced an unrecoverable failure. When creating a new cache, the
     * cache was unable to be created.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #lifecycle} will
     * return {@link FileCacheLifecycle#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #lifecycleAsString}.
     * </p>
     * 
     * @return The lifecycle status of the cache. The following are the possible values and what they mean:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>AVAILABLE</code> - The cache is in a healthy state, and is reachable and available for use.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CREATING</code> - The new cache is being created.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELETING</code> - An existing cache is being deleted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>UPDATING</code> - The cache is undergoing a customer-initiated update.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FAILED</code> - An existing cache has experienced an unrecoverable failure. When creating a new
     *         cache, the cache was unable to be created.
     *         </p>
     *         </li>
     * @see FileCacheLifecycle
     */
    public final FileCacheLifecycle lifecycle() {
        return FileCacheLifecycle.fromValue(lifecycle);
    }

    /**
     * <p>
     * The lifecycle status of the cache. The following are the possible values and what they mean:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>AVAILABLE</code> - The cache is in a healthy state, and is reachable and available for use.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CREATING</code> - The new cache is being created.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELETING</code> - An existing cache is being deleted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>UPDATING</code> - The cache is undergoing a customer-initiated update.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FAILED</code> - An existing cache has experienced an unrecoverable failure. When creating a new cache, the
     * cache was unable to be created.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #lifecycle} will
     * return {@link FileCacheLifecycle#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #lifecycleAsString}.
     * </p>
     * 
     * @return The lifecycle status of the cache. The following are the possible values and what they mean:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>AVAILABLE</code> - The cache is in a healthy state, and is reachable and available for use.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CREATING</code> - The new cache is being created.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELETING</code> - An existing cache is being deleted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>UPDATING</code> - The cache is undergoing a customer-initiated update.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FAILED</code> - An existing cache has experienced an unrecoverable failure. When creating a new
     *         cache, the cache was unable to be created.
     *         </p>
     *         </li>
     * @see FileCacheLifecycle
     */
    public final String lifecycleAsString() {
        return lifecycle;
    }

    /**
     * <p>
     * A structure providing details of any failures that occurred.
     * </p>
     * 
     * @return A structure providing details of any failures that occurred.
     */
    public final FileCacheFailureDetails failureDetails() {
        return failureDetails;
    }

    /**
     * <p>
     * The storage capacity of the cache in gibibytes (GiB).
     * </p>
     * 
     * @return The storage capacity of the cache in gibibytes (GiB).
     */
    public final Integer storageCapacity() {
        return storageCapacity;
    }

    /**
     * Returns the value of the VpcId property for this object.
     * 
     * @return The value of the VpcId property for this object.
     */
    public final String vpcId() {
        return vpcId;
    }

    /**
     * For responses, this returns true if the service returned a value for the SubnetIds property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasSubnetIds() {
        return subnetIds != null && !(subnetIds instanceof SdkAutoConstructList);
    }

    /**
     * Returns the value of the SubnetIds property for this object.
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSubnetIds} method.
     * </p>
     * 
     * @return The value of the SubnetIds property for this object.
     */
    public final List<String> subnetIds() {
        return subnetIds;
    }

    /**
     * For responses, this returns true if the service returned a value for the NetworkInterfaceIds property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasNetworkInterfaceIds() {
        return networkInterfaceIds != null && !(networkInterfaceIds instanceof SdkAutoConstructList);
    }

    /**
     * Returns the value of the NetworkInterfaceIds property for this object.
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasNetworkInterfaceIds} method.
     * </p>
     * 
     * @return The value of the NetworkInterfaceIds property for this object.
     */
    public final List<String> networkInterfaceIds() {
        return networkInterfaceIds;
    }

    /**
     * <p>
     * The Domain Name System (DNS) name for the cache.
     * </p>
     * 
     * @return The Domain Name System (DNS) name for the cache.
     */
    public final String dnsName() {
        return dnsName;
    }

    /**
     * <p>
     * Specifies the ID of the Key Management Service (KMS) key to use for encrypting data on an Amazon File Cache. If a
     * <code>KmsKeyId</code> isn't specified, the Amazon FSx-managed KMS key for your account is used. For more
     * information, see <a href="https://docs.aws.amazon.com/kms/latest/APIReference/API_Encrypt.html">Encrypt</a> in
     * the <i>Key Management Service API Reference</i>.
     * </p>
     * 
     * @return Specifies the ID of the Key Management Service (KMS) key to use for encrypting data on an Amazon File
     *         Cache. If a <code>KmsKeyId</code> isn't specified, the Amazon FSx-managed KMS key for your account is
     *         used. For more information, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/APIReference/API_Encrypt.html">Encrypt</a> in the <i>Key
     *         Management Service API Reference</i>.
     */
    public final String kmsKeyId() {
        return kmsKeyId;
    }

    /**
     * Returns the value of the ResourceARN property for this object.
     * 
     * @return The value of the ResourceARN property for this object.
     */
    public final String resourceARN() {
        return resourceARN;
    }

    /**
     * <p>
     * The configuration for the Amazon File Cache resource.
     * </p>
     * 
     * @return The configuration for the Amazon File Cache resource.
     */
    public final FileCacheLustreConfiguration lustreConfiguration() {
        return lustreConfiguration;
    }

    /**
     * For responses, this returns true if the service returned a value for the DataRepositoryAssociationIds property.
     * This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasDataRepositoryAssociationIds() {
        return dataRepositoryAssociationIds != null && !(dataRepositoryAssociationIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of IDs of data repository associations that are associated with this cache.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDataRepositoryAssociationIds} method.
     * </p>
     * 
     * @return A list of IDs of data repository associations that are associated with this cache.
     */
    public final List<String> dataRepositoryAssociationIds() {
        return dataRepositoryAssociationIds;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(ownerId());
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        hashCode = 31 * hashCode + Objects.hashCode(fileCacheId());
        hashCode = 31 * hashCode + Objects.hashCode(fileCacheTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(fileCacheTypeVersion());
        hashCode = 31 * hashCode + Objects.hashCode(lifecycleAsString());
        hashCode = 31 * hashCode + Objects.hashCode(failureDetails());
        hashCode = 31 * hashCode + Objects.hashCode(storageCapacity());
        hashCode = 31 * hashCode + Objects.hashCode(vpcId());
        hashCode = 31 * hashCode + Objects.hashCode(hasSubnetIds() ? subnetIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasNetworkInterfaceIds() ? networkInterfaceIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(dnsName());
        hashCode = 31 * hashCode + Objects.hashCode(kmsKeyId());
        hashCode = 31 * hashCode + Objects.hashCode(resourceARN());
        hashCode = 31 * hashCode + Objects.hashCode(lustreConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(hasDataRepositoryAssociationIds() ? dataRepositoryAssociationIds() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof FileCache)) {
            return false;
        }
        FileCache other = (FileCache) obj;
        return Objects.equals(ownerId(), other.ownerId()) && Objects.equals(creationTime(), other.creationTime())
                && Objects.equals(fileCacheId(), other.fileCacheId())
                && Objects.equals(fileCacheTypeAsString(), other.fileCacheTypeAsString())
                && Objects.equals(fileCacheTypeVersion(), other.fileCacheTypeVersion())
                && Objects.equals(lifecycleAsString(), other.lifecycleAsString())
                && Objects.equals(failureDetails(), other.failureDetails())
                && Objects.equals(storageCapacity(), other.storageCapacity()) && Objects.equals(vpcId(), other.vpcId())
                && hasSubnetIds() == other.hasSubnetIds() && Objects.equals(subnetIds(), other.subnetIds())
                && hasNetworkInterfaceIds() == other.hasNetworkInterfaceIds()
                && Objects.equals(networkInterfaceIds(), other.networkInterfaceIds())
                && Objects.equals(dnsName(), other.dnsName()) && Objects.equals(kmsKeyId(), other.kmsKeyId())
                && Objects.equals(resourceARN(), other.resourceARN())
                && Objects.equals(lustreConfiguration(), other.lustreConfiguration())
                && hasDataRepositoryAssociationIds() == other.hasDataRepositoryAssociationIds()
                && Objects.equals(dataRepositoryAssociationIds(), other.dataRepositoryAssociationIds());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("FileCache").add("OwnerId", ownerId()).add("CreationTime", creationTime())
                .add("FileCacheId", fileCacheId()).add("FileCacheType", fileCacheTypeAsString())
                .add("FileCacheTypeVersion", fileCacheTypeVersion()).add("Lifecycle", lifecycleAsString())
                .add("FailureDetails", failureDetails()).add("StorageCapacity", storageCapacity()).add("VpcId", vpcId())
                .add("SubnetIds", hasSubnetIds() ? subnetIds() : null)
                .add("NetworkInterfaceIds", hasNetworkInterfaceIds() ? networkInterfaceIds() : null).add("DNSName", dnsName())
                .add("KmsKeyId", kmsKeyId()).add("ResourceARN", resourceARN()).add("LustreConfiguration", lustreConfiguration())
                .add("DataRepositoryAssociationIds", hasDataRepositoryAssociationIds() ? dataRepositoryAssociationIds() : null)
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "OwnerId":
            return Optional.ofNullable(clazz.cast(ownerId()));
        case "CreationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        case "FileCacheId":
            return Optional.ofNullable(clazz.cast(fileCacheId()));
        case "FileCacheType":
            return Optional.ofNullable(clazz.cast(fileCacheTypeAsString()));
        case "FileCacheTypeVersion":
            return Optional.ofNullable(clazz.cast(fileCacheTypeVersion()));
        case "Lifecycle":
            return Optional.ofNullable(clazz.cast(lifecycleAsString()));
        case "FailureDetails":
            return Optional.ofNullable(clazz.cast(failureDetails()));
        case "StorageCapacity":
            return Optional.ofNullable(clazz.cast(storageCapacity()));
        case "VpcId":
            return Optional.ofNullable(clazz.cast(vpcId()));
        case "SubnetIds":
            return Optional.ofNullable(clazz.cast(subnetIds()));
        case "NetworkInterfaceIds":
            return Optional.ofNullable(clazz.cast(networkInterfaceIds()));
        case "DNSName":
            return Optional.ofNullable(clazz.cast(dnsName()));
        case "KmsKeyId":
            return Optional.ofNullable(clazz.cast(kmsKeyId()));
        case "ResourceARN":
            return Optional.ofNullable(clazz.cast(resourceARN()));
        case "LustreConfiguration":
            return Optional.ofNullable(clazz.cast(lustreConfiguration()));
        case "DataRepositoryAssociationIds":
            return Optional.ofNullable(clazz.cast(dataRepositoryAssociationIds()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<FileCache, T> g) {
        return obj -> g.apply((FileCache) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, FileCache> {
        /**
         * Sets the value of the OwnerId property for this object.
         *
         * @param ownerId
         *        The new value for the OwnerId property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ownerId(String ownerId);

        /**
         * Sets the value of the CreationTime property for this object.
         *
         * @param creationTime
         *        The new value for the CreationTime property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(Instant creationTime);

        /**
         * <p>
         * The system-generated, unique ID of the cache.
         * </p>
         * 
         * @param fileCacheId
         *        The system-generated, unique ID of the cache.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fileCacheId(String fileCacheId);

        /**
         * <p>
         * The type of cache, which must be <code>LUSTRE</code>.
         * </p>
         * 
         * @param fileCacheType
         *        The type of cache, which must be <code>LUSTRE</code>.
         * @see FileCacheType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FileCacheType
         */
        Builder fileCacheType(String fileCacheType);

        /**
         * <p>
         * The type of cache, which must be <code>LUSTRE</code>.
         * </p>
         * 
         * @param fileCacheType
         *        The type of cache, which must be <code>LUSTRE</code>.
         * @see FileCacheType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FileCacheType
         */
        Builder fileCacheType(FileCacheType fileCacheType);

        /**
         * <p>
         * The Lustre version of the cache, which must be <code>2.12</code>.
         * </p>
         * 
         * @param fileCacheTypeVersion
         *        The Lustre version of the cache, which must be <code>2.12</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fileCacheTypeVersion(String fileCacheTypeVersion);

        /**
         * <p>
         * The lifecycle status of the cache. The following are the possible values and what they mean:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>AVAILABLE</code> - The cache is in a healthy state, and is reachable and available for use.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CREATING</code> - The new cache is being created.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELETING</code> - An existing cache is being deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>UPDATING</code> - The cache is undergoing a customer-initiated update.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FAILED</code> - An existing cache has experienced an unrecoverable failure. When creating a new cache,
         * the cache was unable to be created.
         * </p>
         * </li>
         * </ul>
         * 
         * @param lifecycle
         *        The lifecycle status of the cache. The following are the possible values and what they mean:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>AVAILABLE</code> - The cache is in a healthy state, and is reachable and available for use.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CREATING</code> - The new cache is being created.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETING</code> - An existing cache is being deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>UPDATING</code> - The cache is undergoing a customer-initiated update.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FAILED</code> - An existing cache has experienced an unrecoverable failure. When creating a new
         *        cache, the cache was unable to be created.
         *        </p>
         *        </li>
         * @see FileCacheLifecycle
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FileCacheLifecycle
         */
        Builder lifecycle(String lifecycle);

        /**
         * <p>
         * The lifecycle status of the cache. The following are the possible values and what they mean:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>AVAILABLE</code> - The cache is in a healthy state, and is reachable and available for use.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CREATING</code> - The new cache is being created.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELETING</code> - An existing cache is being deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>UPDATING</code> - The cache is undergoing a customer-initiated update.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FAILED</code> - An existing cache has experienced an unrecoverable failure. When creating a new cache,
         * the cache was unable to be created.
         * </p>
         * </li>
         * </ul>
         * 
         * @param lifecycle
         *        The lifecycle status of the cache. The following are the possible values and what they mean:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>AVAILABLE</code> - The cache is in a healthy state, and is reachable and available for use.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CREATING</code> - The new cache is being created.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETING</code> - An existing cache is being deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>UPDATING</code> - The cache is undergoing a customer-initiated update.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FAILED</code> - An existing cache has experienced an unrecoverable failure. When creating a new
         *        cache, the cache was unable to be created.
         *        </p>
         *        </li>
         * @see FileCacheLifecycle
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FileCacheLifecycle
         */
        Builder lifecycle(FileCacheLifecycle lifecycle);

        /**
         * <p>
         * A structure providing details of any failures that occurred.
         * </p>
         * 
         * @param failureDetails
         *        A structure providing details of any failures that occurred.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failureDetails(FileCacheFailureDetails failureDetails);

        /**
         * <p>
         * A structure providing details of any failures that occurred.
         * </p>
         * This is a convenience method that creates an instance of the {@link FileCacheFailureDetails.Builder} avoiding
         * the need to create one manually via {@link FileCacheFailureDetails#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link FileCacheFailureDetails.Builder#build()} is called immediately
         * and its result is passed to {@link #failureDetails(FileCacheFailureDetails)}.
         * 
         * @param failureDetails
         *        a consumer that will call methods on {@link FileCacheFailureDetails.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #failureDetails(FileCacheFailureDetails)
         */
        default Builder failureDetails(Consumer<FileCacheFailureDetails.Builder> failureDetails) {
            return failureDetails(FileCacheFailureDetails.builder().applyMutation(failureDetails).build());
        }

        /**
         * <p>
         * The storage capacity of the cache in gibibytes (GiB).
         * </p>
         * 
         * @param storageCapacity
         *        The storage capacity of the cache in gibibytes (GiB).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder storageCapacity(Integer storageCapacity);

        /**
         * Sets the value of the VpcId property for this object.
         *
         * @param vpcId
         *        The new value for the VpcId property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcId(String vpcId);

        /**
         * Sets the value of the SubnetIds property for this object.
         *
         * @param subnetIds
         *        The new value for the SubnetIds property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetIds(Collection<String> subnetIds);

        /**
         * Sets the value of the SubnetIds property for this object.
         *
         * @param subnetIds
         *        The new value for the SubnetIds property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetIds(String... subnetIds);

        /**
         * Sets the value of the NetworkInterfaceIds property for this object.
         *
         * @param networkInterfaceIds
         *        The new value for the NetworkInterfaceIds property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkInterfaceIds(Collection<String> networkInterfaceIds);

        /**
         * Sets the value of the NetworkInterfaceIds property for this object.
         *
         * @param networkInterfaceIds
         *        The new value for the NetworkInterfaceIds property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkInterfaceIds(String... networkInterfaceIds);

        /**
         * <p>
         * The Domain Name System (DNS) name for the cache.
         * </p>
         * 
         * @param dnsName
         *        The Domain Name System (DNS) name for the cache.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dnsName(String dnsName);

        /**
         * <p>
         * Specifies the ID of the Key Management Service (KMS) key to use for encrypting data on an Amazon File Cache.
         * If a <code>KmsKeyId</code> isn't specified, the Amazon FSx-managed KMS key for your account is used. For more
         * information, see <a href="https://docs.aws.amazon.com/kms/latest/APIReference/API_Encrypt.html">Encrypt</a>
         * in the <i>Key Management Service API Reference</i>.
         * </p>
         * 
         * @param kmsKeyId
         *        Specifies the ID of the Key Management Service (KMS) key to use for encrypting data on an Amazon File
         *        Cache. If a <code>KmsKeyId</code> isn't specified, the Amazon FSx-managed KMS key for your account is
         *        used. For more information, see <a
         *        href="https://docs.aws.amazon.com/kms/latest/APIReference/API_Encrypt.html">Encrypt</a> in the <i>Key
         *        Management Service API Reference</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKeyId(String kmsKeyId);

        /**
         * Sets the value of the ResourceARN property for this object.
         *
         * @param resourceARN
         *        The new value for the ResourceARN property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceARN(String resourceARN);

        /**
         * <p>
         * The configuration for the Amazon File Cache resource.
         * </p>
         * 
         * @param lustreConfiguration
         *        The configuration for the Amazon File Cache resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lustreConfiguration(FileCacheLustreConfiguration lustreConfiguration);

        /**
         * <p>
         * The configuration for the Amazon File Cache resource.
         * </p>
         * This is a convenience method that creates an instance of the {@link FileCacheLustreConfiguration.Builder}
         * avoiding the need to create one manually via {@link FileCacheLustreConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link FileCacheLustreConfiguration.Builder#build()} is called
         * immediately and its result is passed to {@link #lustreConfiguration(FileCacheLustreConfiguration)}.
         * 
         * @param lustreConfiguration
         *        a consumer that will call methods on {@link FileCacheLustreConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #lustreConfiguration(FileCacheLustreConfiguration)
         */
        default Builder lustreConfiguration(Consumer<FileCacheLustreConfiguration.Builder> lustreConfiguration) {
            return lustreConfiguration(FileCacheLustreConfiguration.builder().applyMutation(lustreConfiguration).build());
        }

        /**
         * <p>
         * A list of IDs of data repository associations that are associated with this cache.
         * </p>
         * 
         * @param dataRepositoryAssociationIds
         *        A list of IDs of data repository associations that are associated with this cache.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataRepositoryAssociationIds(Collection<String> dataRepositoryAssociationIds);

        /**
         * <p>
         * A list of IDs of data repository associations that are associated with this cache.
         * </p>
         * 
         * @param dataRepositoryAssociationIds
         *        A list of IDs of data repository associations that are associated with this cache.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataRepositoryAssociationIds(String... dataRepositoryAssociationIds);
    }

    static final class BuilderImpl implements Builder {
        private String ownerId;

        private Instant creationTime;

        private String fileCacheId;

        private String fileCacheType;

        private String fileCacheTypeVersion;

        private String lifecycle;

        private FileCacheFailureDetails failureDetails;

        private Integer storageCapacity;

        private String vpcId;

        private List<String> subnetIds = DefaultSdkAutoConstructList.getInstance();

        private List<String> networkInterfaceIds = DefaultSdkAutoConstructList.getInstance();

        private String dnsName;

        private String kmsKeyId;

        private String resourceARN;

        private FileCacheLustreConfiguration lustreConfiguration;

        private List<String> dataRepositoryAssociationIds = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(FileCache model) {
            ownerId(model.ownerId);
            creationTime(model.creationTime);
            fileCacheId(model.fileCacheId);
            fileCacheType(model.fileCacheType);
            fileCacheTypeVersion(model.fileCacheTypeVersion);
            lifecycle(model.lifecycle);
            failureDetails(model.failureDetails);
            storageCapacity(model.storageCapacity);
            vpcId(model.vpcId);
            subnetIds(model.subnetIds);
            networkInterfaceIds(model.networkInterfaceIds);
            dnsName(model.dnsName);
            kmsKeyId(model.kmsKeyId);
            resourceARN(model.resourceARN);
            lustreConfiguration(model.lustreConfiguration);
            dataRepositoryAssociationIds(model.dataRepositoryAssociationIds);
        }

        public final String getOwnerId() {
            return ownerId;
        }

        public final void setOwnerId(String ownerId) {
            this.ownerId = ownerId;
        }

        @Override
        public final Builder ownerId(String ownerId) {
            this.ownerId = ownerId;
            return this;
        }

        public final Instant getCreationTime() {
            return creationTime;
        }

        public final void setCreationTime(Instant creationTime) {
            this.creationTime = creationTime;
        }

        @Override
        public final Builder creationTime(Instant creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final String getFileCacheId() {
            return fileCacheId;
        }

        public final void setFileCacheId(String fileCacheId) {
            this.fileCacheId = fileCacheId;
        }

        @Override
        public final Builder fileCacheId(String fileCacheId) {
            this.fileCacheId = fileCacheId;
            return this;
        }

        public final String getFileCacheType() {
            return fileCacheType;
        }

        public final void setFileCacheType(String fileCacheType) {
            this.fileCacheType = fileCacheType;
        }

        @Override
        public final Builder fileCacheType(String fileCacheType) {
            this.fileCacheType = fileCacheType;
            return this;
        }

        @Override
        public final Builder fileCacheType(FileCacheType fileCacheType) {
            this.fileCacheType(fileCacheType == null ? null : fileCacheType.toString());
            return this;
        }

        public final String getFileCacheTypeVersion() {
            return fileCacheTypeVersion;
        }

        public final void setFileCacheTypeVersion(String fileCacheTypeVersion) {
            this.fileCacheTypeVersion = fileCacheTypeVersion;
        }

        @Override
        public final Builder fileCacheTypeVersion(String fileCacheTypeVersion) {
            this.fileCacheTypeVersion = fileCacheTypeVersion;
            return this;
        }

        public final String getLifecycle() {
            return lifecycle;
        }

        public final void setLifecycle(String lifecycle) {
            this.lifecycle = lifecycle;
        }

        @Override
        public final Builder lifecycle(String lifecycle) {
            this.lifecycle = lifecycle;
            return this;
        }

        @Override
        public final Builder lifecycle(FileCacheLifecycle lifecycle) {
            this.lifecycle(lifecycle == null ? null : lifecycle.toString());
            return this;
        }

        public final FileCacheFailureDetails.Builder getFailureDetails() {
            return failureDetails != null ? failureDetails.toBuilder() : null;
        }

        public final void setFailureDetails(FileCacheFailureDetails.BuilderImpl failureDetails) {
            this.failureDetails = failureDetails != null ? failureDetails.build() : null;
        }

        @Override
        public final Builder failureDetails(FileCacheFailureDetails failureDetails) {
            this.failureDetails = failureDetails;
            return this;
        }

        public final Integer getStorageCapacity() {
            return storageCapacity;
        }

        public final void setStorageCapacity(Integer storageCapacity) {
            this.storageCapacity = storageCapacity;
        }

        @Override
        public final Builder storageCapacity(Integer storageCapacity) {
            this.storageCapacity = storageCapacity;
            return this;
        }

        public final String getVpcId() {
            return vpcId;
        }

        public final void setVpcId(String vpcId) {
            this.vpcId = vpcId;
        }

        @Override
        public final Builder vpcId(String vpcId) {
            this.vpcId = vpcId;
            return this;
        }

        public final Collection<String> getSubnetIds() {
            if (subnetIds instanceof SdkAutoConstructList) {
                return null;
            }
            return subnetIds;
        }

        public final void setSubnetIds(Collection<String> subnetIds) {
            this.subnetIds = SubnetIdsCopier.copy(subnetIds);
        }

        @Override
        public final Builder subnetIds(Collection<String> subnetIds) {
            this.subnetIds = SubnetIdsCopier.copy(subnetIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder subnetIds(String... subnetIds) {
            subnetIds(Arrays.asList(subnetIds));
            return this;
        }

        public final Collection<String> getNetworkInterfaceIds() {
            if (networkInterfaceIds instanceof SdkAutoConstructList) {
                return null;
            }
            return networkInterfaceIds;
        }

        public final void setNetworkInterfaceIds(Collection<String> networkInterfaceIds) {
            this.networkInterfaceIds = NetworkInterfaceIdsCopier.copy(networkInterfaceIds);
        }

        @Override
        public final Builder networkInterfaceIds(Collection<String> networkInterfaceIds) {
            this.networkInterfaceIds = NetworkInterfaceIdsCopier.copy(networkInterfaceIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder networkInterfaceIds(String... networkInterfaceIds) {
            networkInterfaceIds(Arrays.asList(networkInterfaceIds));
            return this;
        }

        public final String getDnsName() {
            return dnsName;
        }

        public final void setDnsName(String dnsName) {
            this.dnsName = dnsName;
        }

        @Override
        public final Builder dnsName(String dnsName) {
            this.dnsName = dnsName;
            return this;
        }

        public final String getKmsKeyId() {
            return kmsKeyId;
        }

        public final void setKmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
        }

        @Override
        public final Builder kmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
            return this;
        }

        public final String getResourceARN() {
            return resourceARN;
        }

        public final void setResourceARN(String resourceARN) {
            this.resourceARN = resourceARN;
        }

        @Override
        public final Builder resourceARN(String resourceARN) {
            this.resourceARN = resourceARN;
            return this;
        }

        public final FileCacheLustreConfiguration.Builder getLustreConfiguration() {
            return lustreConfiguration != null ? lustreConfiguration.toBuilder() : null;
        }

        public final void setLustreConfiguration(FileCacheLustreConfiguration.BuilderImpl lustreConfiguration) {
            this.lustreConfiguration = lustreConfiguration != null ? lustreConfiguration.build() : null;
        }

        @Override
        public final Builder lustreConfiguration(FileCacheLustreConfiguration lustreConfiguration) {
            this.lustreConfiguration = lustreConfiguration;
            return this;
        }

        public final Collection<String> getDataRepositoryAssociationIds() {
            if (dataRepositoryAssociationIds instanceof SdkAutoConstructList) {
                return null;
            }
            return dataRepositoryAssociationIds;
        }

        public final void setDataRepositoryAssociationIds(Collection<String> dataRepositoryAssociationIds) {
            this.dataRepositoryAssociationIds = DataRepositoryAssociationIdsCopier.copy(dataRepositoryAssociationIds);
        }

        @Override
        public final Builder dataRepositoryAssociationIds(Collection<String> dataRepositoryAssociationIds) {
            this.dataRepositoryAssociationIds = DataRepositoryAssociationIdsCopier.copy(dataRepositoryAssociationIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dataRepositoryAssociationIds(String... dataRepositoryAssociationIds) {
            dataRepositoryAssociationIds(Arrays.asList(dataRepositoryAssociationIds));
            return this;
        }

        @Override
        public FileCache build() {
            return new FileCache(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
