/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.bedrockruntime;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsClient;
import software.amazon.awssdk.services.bedrockruntime.model.ApplyGuardrailRequest;
import software.amazon.awssdk.services.bedrockruntime.model.ApplyGuardrailResponse;
import software.amazon.awssdk.services.bedrockruntime.model.ConverseRequest;
import software.amazon.awssdk.services.bedrockruntime.model.ConverseResponse;
import software.amazon.awssdk.services.bedrockruntime.model.ConverseStreamRequest;
import software.amazon.awssdk.services.bedrockruntime.model.ConverseStreamResponseHandler;
import software.amazon.awssdk.services.bedrockruntime.model.InvokeModelRequest;
import software.amazon.awssdk.services.bedrockruntime.model.InvokeModelResponse;
import software.amazon.awssdk.services.bedrockruntime.model.InvokeModelWithResponseStreamRequest;
import software.amazon.awssdk.services.bedrockruntime.model.InvokeModelWithResponseStreamResponseHandler;

/**
 * Service client for accessing Amazon Bedrock Runtime asynchronously. This can be created using the static
 * {@link #builder()} method.The asynchronous client performs non-blocking I/O when configured with any
 * {@code SdkAsyncHttpClient} supported in the SDK. However, full non-blocking is not guaranteed as the async client may
 * perform blocking calls in some cases such as credentials retrieval and endpoint discovery as part of the async API
 * call.
 *
 * <p>
 * Describes the API operations for running inference using Amazon Bedrock models.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
public interface BedrockRuntimeAsyncClient extends AwsClient {
    String SERVICE_NAME = "bedrock";

    /**
     * Value for looking up the service's metadata from the
     * {@link software.amazon.awssdk.regions.ServiceMetadataProvider}.
     */
    String SERVICE_METADATA_ID = "bedrock-runtime";

    /**
     * <p>
     * The action to apply a guardrail.
     * </p>
     *
     * @param applyGuardrailRequest
     * @return A Java Future containing the result of the ApplyGuardrail operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException The request is denied because of missing access permissions.</li>
     *         <li>ResourceNotFoundException The specified resource ARN was not found. Check the ARN and try your
     *         request again.</li>
     *         <li>ThrottlingException Your request was throttled because of service-wide limitations. Resubmit your
     *         request later or in a different region. You can also purchase <a
     *         href="https://docs.aws.amazon.com/bedrock/latest/userguide/prov-throughput.html">Provisioned
     *         Throughput</a> to increase the rate or number of tokens you can process.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ValidationException Input validation failed. Check your request parameters and retry the request.</li>
     *         <li>ServiceQuotaExceededException Your request exceeds the service quota for your account. You can view
     *         your quotas at <a
     *         href="https://docs.aws.amazon.com/servicequotas/latest/userguide/gs-request-quota.html">Viewing service
     *         quotas</a>. You can resubmit your request later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockRuntimeException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample BedrockRuntimeAsyncClient.ApplyGuardrail
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-runtime-2023-09-30/ApplyGuardrail"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<ApplyGuardrailResponse> applyGuardrail(ApplyGuardrailRequest applyGuardrailRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * The action to apply a guardrail.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ApplyGuardrailRequest.Builder} avoiding the need to
     * create one manually via {@link ApplyGuardrailRequest#builder()}
     * </p>
     *
     * @param applyGuardrailRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.bedrockruntime.model.ApplyGuardrailRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the ApplyGuardrail operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException The request is denied because of missing access permissions.</li>
     *         <li>ResourceNotFoundException The specified resource ARN was not found. Check the ARN and try your
     *         request again.</li>
     *         <li>ThrottlingException Your request was throttled because of service-wide limitations. Resubmit your
     *         request later or in a different region. You can also purchase <a
     *         href="https://docs.aws.amazon.com/bedrock/latest/userguide/prov-throughput.html">Provisioned
     *         Throughput</a> to increase the rate or number of tokens you can process.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ValidationException Input validation failed. Check your request parameters and retry the request.</li>
     *         <li>ServiceQuotaExceededException Your request exceeds the service quota for your account. You can view
     *         your quotas at <a
     *         href="https://docs.aws.amazon.com/servicequotas/latest/userguide/gs-request-quota.html">Viewing service
     *         quotas</a>. You can resubmit your request later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockRuntimeException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample BedrockRuntimeAsyncClient.ApplyGuardrail
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-runtime-2023-09-30/ApplyGuardrail"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<ApplyGuardrailResponse> applyGuardrail(Consumer<ApplyGuardrailRequest.Builder> applyGuardrailRequest) {
        return applyGuardrail(ApplyGuardrailRequest.builder().applyMutation(applyGuardrailRequest).build());
    }

    /**
     * <p>
     * Sends messages to the specified Amazon Bedrock model. <code>Converse</code> provides a consistent interface that
     * works with all models that support messages. This allows you to write code once and use it with different models.
     * If a model has unique inference parameters, you can also pass those unique parameters to the model.
     * </p>
     * <p>
     * Amazon Bedrock doesn't store any text, images, or documents that you provide as content. The data is only used to
     * generate the response.
     * </p>
     * <p>
     * For information about the Converse API, see <i>Use the Converse API</i> in the <i>Amazon Bedrock User Guide</i>.
     * To use a guardrail, see <i>Use a guardrail with the Converse API</i> in the <i>Amazon Bedrock User Guide</i>. To
     * use a tool with a model, see <i>Tool use (Function calling)</i> in the <i>Amazon Bedrock User Guide</i>
     * </p>
     * <p>
     * For example code, see <i>Converse API examples</i> in the <i>Amazon Bedrock User Guide</i>.
     * </p>
     * <p>
     * This operation requires permission for the <code>bedrock:InvokeModel</code> action.
     * </p>
     *
     * @param converseRequest
     * @return A Java Future containing the result of the Converse operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException The request is denied because of missing access permissions.</li>
     *         <li>ResourceNotFoundException The specified resource ARN was not found. Check the ARN and try your
     *         request again.</li>
     *         <li>ThrottlingException Your request was throttled because of service-wide limitations. Resubmit your
     *         request later or in a different region. You can also purchase <a
     *         href="https://docs.aws.amazon.com/bedrock/latest/userguide/prov-throughput.html">Provisioned
     *         Throughput</a> to increase the rate or number of tokens you can process.</li>
     *         <li>ModelTimeoutException The request took too long to process. Processing time exceeded the model
     *         timeout length.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ServiceUnavailableException The service isn't currently available. Try again later.</li>
     *         <li>ValidationException Input validation failed. Check your request parameters and retry the request.</li>
     *         <li>ModelNotReadyException The model specified in the request is not ready to serve inference requests.
     *         The AWS SDK will automatically retry the operation up to 5 times. For information about configuring
     *         automatic retries, see <a
     *         href="https://docs.aws.amazon.com/sdkref/latest/guide/feature-retry-behavior.html">Retry behavior</a> in
     *         the <i>AWS SDKs and Tools</i> reference guide.</li>
     *         <li>ModelErrorException The request failed due to an error while processing the model.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockRuntimeException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample BedrockRuntimeAsyncClient.Converse
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-runtime-2023-09-30/Converse" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ConverseResponse> converse(ConverseRequest converseRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Sends messages to the specified Amazon Bedrock model. <code>Converse</code> provides a consistent interface that
     * works with all models that support messages. This allows you to write code once and use it with different models.
     * If a model has unique inference parameters, you can also pass those unique parameters to the model.
     * </p>
     * <p>
     * Amazon Bedrock doesn't store any text, images, or documents that you provide as content. The data is only used to
     * generate the response.
     * </p>
     * <p>
     * For information about the Converse API, see <i>Use the Converse API</i> in the <i>Amazon Bedrock User Guide</i>.
     * To use a guardrail, see <i>Use a guardrail with the Converse API</i> in the <i>Amazon Bedrock User Guide</i>. To
     * use a tool with a model, see <i>Tool use (Function calling)</i> in the <i>Amazon Bedrock User Guide</i>
     * </p>
     * <p>
     * For example code, see <i>Converse API examples</i> in the <i>Amazon Bedrock User Guide</i>.
     * </p>
     * <p>
     * This operation requires permission for the <code>bedrock:InvokeModel</code> action.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ConverseRequest.Builder} avoiding the need to
     * create one manually via {@link ConverseRequest#builder()}
     * </p>
     *
     * @param converseRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.bedrockruntime.model.ConverseRequest.Builder} to create a request.
     * @return A Java Future containing the result of the Converse operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException The request is denied because of missing access permissions.</li>
     *         <li>ResourceNotFoundException The specified resource ARN was not found. Check the ARN and try your
     *         request again.</li>
     *         <li>ThrottlingException Your request was throttled because of service-wide limitations. Resubmit your
     *         request later or in a different region. You can also purchase <a
     *         href="https://docs.aws.amazon.com/bedrock/latest/userguide/prov-throughput.html">Provisioned
     *         Throughput</a> to increase the rate or number of tokens you can process.</li>
     *         <li>ModelTimeoutException The request took too long to process. Processing time exceeded the model
     *         timeout length.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ServiceUnavailableException The service isn't currently available. Try again later.</li>
     *         <li>ValidationException Input validation failed. Check your request parameters and retry the request.</li>
     *         <li>ModelNotReadyException The model specified in the request is not ready to serve inference requests.
     *         The AWS SDK will automatically retry the operation up to 5 times. For information about configuring
     *         automatic retries, see <a
     *         href="https://docs.aws.amazon.com/sdkref/latest/guide/feature-retry-behavior.html">Retry behavior</a> in
     *         the <i>AWS SDKs and Tools</i> reference guide.</li>
     *         <li>ModelErrorException The request failed due to an error while processing the model.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockRuntimeException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample BedrockRuntimeAsyncClient.Converse
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-runtime-2023-09-30/Converse" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ConverseResponse> converse(Consumer<ConverseRequest.Builder> converseRequest) {
        return converse(ConverseRequest.builder().applyMutation(converseRequest).build());
    }

    /**
     * <p>
     * Sends messages to the specified Amazon Bedrock model and returns the response in a stream.
     * <code>ConverseStream</code> provides a consistent API that works with all Amazon Bedrock models that support
     * messages. This allows you to write code once and use it with different models. Should a model have unique
     * inference parameters, you can also pass those unique parameters to the model.
     * </p>
     * <p>
     * To find out if a model supports streaming, call <a
     * href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_GetFoundationModel.html">GetFoundationModel</a>
     * and check the <code>responseStreamingSupported</code> field in the response.
     * </p>
     * <note>
     * <p>
     * The CLI doesn't support streaming operations in Amazon Bedrock, including <code>ConverseStream</code>.
     * </p>
     * </note>
     * <p>
     * Amazon Bedrock doesn't store any text, images, or documents that you provide as content. The data is only used to
     * generate the response.
     * </p>
     * <p>
     * For information about the Converse API, see <i>Use the Converse API</i> in the <i>Amazon Bedrock User Guide</i>.
     * To use a guardrail, see <i>Use a guardrail with the Converse API</i> in the <i>Amazon Bedrock User Guide</i>. To
     * use a tool with a model, see <i>Tool use (Function calling)</i> in the <i>Amazon Bedrock User Guide</i>
     * </p>
     * <p>
     * For example code, see <i>Conversation streaming example</i> in the <i>Amazon Bedrock User Guide</i>.
     * </p>
     * <p>
     * This operation requires permission for the <code>bedrock:InvokeModelWithResponseStream</code> action.
     * </p>
     *
     * @param converseStreamRequest
     * @return A Java Future containing the result of the ConverseStream operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException The request is denied because of missing access permissions.</li>
     *         <li>ResourceNotFoundException The specified resource ARN was not found. Check the ARN and try your
     *         request again.</li>
     *         <li>ThrottlingException Your request was throttled because of service-wide limitations. Resubmit your
     *         request later or in a different region. You can also purchase <a
     *         href="https://docs.aws.amazon.com/bedrock/latest/userguide/prov-throughput.html">Provisioned
     *         Throughput</a> to increase the rate or number of tokens you can process.</li>
     *         <li>ModelTimeoutException The request took too long to process. Processing time exceeded the model
     *         timeout length.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ServiceUnavailableException The service isn't currently available. Try again later.</li>
     *         <li>ValidationException Input validation failed. Check your request parameters and retry the request.</li>
     *         <li>ModelNotReadyException The model specified in the request is not ready to serve inference requests.
     *         The AWS SDK will automatically retry the operation up to 5 times. For information about configuring
     *         automatic retries, see <a
     *         href="https://docs.aws.amazon.com/sdkref/latest/guide/feature-retry-behavior.html">Retry behavior</a> in
     *         the <i>AWS SDKs and Tools</i> reference guide.</li>
     *         <li>ModelErrorException The request failed due to an error while processing the model.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockRuntimeException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample BedrockRuntimeAsyncClient.ConverseStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-runtime-2023-09-30/ConverseStream"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<Void> converseStream(ConverseStreamRequest converseStreamRequest,
            ConverseStreamResponseHandler asyncResponseHandler) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Sends messages to the specified Amazon Bedrock model and returns the response in a stream.
     * <code>ConverseStream</code> provides a consistent API that works with all Amazon Bedrock models that support
     * messages. This allows you to write code once and use it with different models. Should a model have unique
     * inference parameters, you can also pass those unique parameters to the model.
     * </p>
     * <p>
     * To find out if a model supports streaming, call <a
     * href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_GetFoundationModel.html">GetFoundationModel</a>
     * and check the <code>responseStreamingSupported</code> field in the response.
     * </p>
     * <note>
     * <p>
     * The CLI doesn't support streaming operations in Amazon Bedrock, including <code>ConverseStream</code>.
     * </p>
     * </note>
     * <p>
     * Amazon Bedrock doesn't store any text, images, or documents that you provide as content. The data is only used to
     * generate the response.
     * </p>
     * <p>
     * For information about the Converse API, see <i>Use the Converse API</i> in the <i>Amazon Bedrock User Guide</i>.
     * To use a guardrail, see <i>Use a guardrail with the Converse API</i> in the <i>Amazon Bedrock User Guide</i>. To
     * use a tool with a model, see <i>Tool use (Function calling)</i> in the <i>Amazon Bedrock User Guide</i>
     * </p>
     * <p>
     * For example code, see <i>Conversation streaming example</i> in the <i>Amazon Bedrock User Guide</i>.
     * </p>
     * <p>
     * This operation requires permission for the <code>bedrock:InvokeModelWithResponseStream</code> action.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ConverseStreamRequest.Builder} avoiding the need to
     * create one manually via {@link ConverseStreamRequest#builder()}
     * </p>
     *
     * @param converseStreamRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.bedrockruntime.model.ConverseStreamRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the ConverseStream operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException The request is denied because of missing access permissions.</li>
     *         <li>ResourceNotFoundException The specified resource ARN was not found. Check the ARN and try your
     *         request again.</li>
     *         <li>ThrottlingException Your request was throttled because of service-wide limitations. Resubmit your
     *         request later or in a different region. You can also purchase <a
     *         href="https://docs.aws.amazon.com/bedrock/latest/userguide/prov-throughput.html">Provisioned
     *         Throughput</a> to increase the rate or number of tokens you can process.</li>
     *         <li>ModelTimeoutException The request took too long to process. Processing time exceeded the model
     *         timeout length.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ServiceUnavailableException The service isn't currently available. Try again later.</li>
     *         <li>ValidationException Input validation failed. Check your request parameters and retry the request.</li>
     *         <li>ModelNotReadyException The model specified in the request is not ready to serve inference requests.
     *         The AWS SDK will automatically retry the operation up to 5 times. For information about configuring
     *         automatic retries, see <a
     *         href="https://docs.aws.amazon.com/sdkref/latest/guide/feature-retry-behavior.html">Retry behavior</a> in
     *         the <i>AWS SDKs and Tools</i> reference guide.</li>
     *         <li>ModelErrorException The request failed due to an error while processing the model.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockRuntimeException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample BedrockRuntimeAsyncClient.ConverseStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-runtime-2023-09-30/ConverseStream"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<Void> converseStream(Consumer<ConverseStreamRequest.Builder> converseStreamRequest,
            ConverseStreamResponseHandler asyncResponseHandler) {
        return converseStream(ConverseStreamRequest.builder().applyMutation(converseStreamRequest).build(), asyncResponseHandler);
    }

    /**
     * <p>
     * Invokes the specified Amazon Bedrock model to run inference using the prompt and inference parameters provided in
     * the request body. You use model inference to generate text, images, and embeddings.
     * </p>
     * <p>
     * For example code, see <i>Invoke model code examples</i> in the <i>Amazon Bedrock User Guide</i>.
     * </p>
     * <p>
     * This operation requires permission for the <code>bedrock:InvokeModel</code> action.
     * </p>
     *
     * @param invokeModelRequest
     * @return A Java Future containing the result of the InvokeModel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException The request is denied because of missing access permissions.</li>
     *         <li>ResourceNotFoundException The specified resource ARN was not found. Check the ARN and try your
     *         request again.</li>
     *         <li>ThrottlingException Your request was throttled because of service-wide limitations. Resubmit your
     *         request later or in a different region. You can also purchase <a
     *         href="https://docs.aws.amazon.com/bedrock/latest/userguide/prov-throughput.html">Provisioned
     *         Throughput</a> to increase the rate or number of tokens you can process.</li>
     *         <li>ModelTimeoutException The request took too long to process. Processing time exceeded the model
     *         timeout length.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ServiceUnavailableException The service isn't currently available. Try again later.</li>
     *         <li>ValidationException Input validation failed. Check your request parameters and retry the request.</li>
     *         <li>ModelNotReadyException The model specified in the request is not ready to serve inference requests.
     *         The AWS SDK will automatically retry the operation up to 5 times. For information about configuring
     *         automatic retries, see <a
     *         href="https://docs.aws.amazon.com/sdkref/latest/guide/feature-retry-behavior.html">Retry behavior</a> in
     *         the <i>AWS SDKs and Tools</i> reference guide.</li>
     *         <li>ServiceQuotaExceededException Your request exceeds the service quota for your account. You can view
     *         your quotas at <a
     *         href="https://docs.aws.amazon.com/servicequotas/latest/userguide/gs-request-quota.html">Viewing service
     *         quotas</a>. You can resubmit your request later.</li>
     *         <li>ModelErrorException The request failed due to an error while processing the model.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockRuntimeException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample BedrockRuntimeAsyncClient.InvokeModel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-runtime-2023-09-30/InvokeModel" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<InvokeModelResponse> invokeModel(InvokeModelRequest invokeModelRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Invokes the specified Amazon Bedrock model to run inference using the prompt and inference parameters provided in
     * the request body. You use model inference to generate text, images, and embeddings.
     * </p>
     * <p>
     * For example code, see <i>Invoke model code examples</i> in the <i>Amazon Bedrock User Guide</i>.
     * </p>
     * <p>
     * This operation requires permission for the <code>bedrock:InvokeModel</code> action.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link InvokeModelRequest.Builder} avoiding the need to
     * create one manually via {@link InvokeModelRequest#builder()}
     * </p>
     *
     * @param invokeModelRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.bedrockruntime.model.InvokeModelRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the InvokeModel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException The request is denied because of missing access permissions.</li>
     *         <li>ResourceNotFoundException The specified resource ARN was not found. Check the ARN and try your
     *         request again.</li>
     *         <li>ThrottlingException Your request was throttled because of service-wide limitations. Resubmit your
     *         request later or in a different region. You can also purchase <a
     *         href="https://docs.aws.amazon.com/bedrock/latest/userguide/prov-throughput.html">Provisioned
     *         Throughput</a> to increase the rate or number of tokens you can process.</li>
     *         <li>ModelTimeoutException The request took too long to process. Processing time exceeded the model
     *         timeout length.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ServiceUnavailableException The service isn't currently available. Try again later.</li>
     *         <li>ValidationException Input validation failed. Check your request parameters and retry the request.</li>
     *         <li>ModelNotReadyException The model specified in the request is not ready to serve inference requests.
     *         The AWS SDK will automatically retry the operation up to 5 times. For information about configuring
     *         automatic retries, see <a
     *         href="https://docs.aws.amazon.com/sdkref/latest/guide/feature-retry-behavior.html">Retry behavior</a> in
     *         the <i>AWS SDKs and Tools</i> reference guide.</li>
     *         <li>ServiceQuotaExceededException Your request exceeds the service quota for your account. You can view
     *         your quotas at <a
     *         href="https://docs.aws.amazon.com/servicequotas/latest/userguide/gs-request-quota.html">Viewing service
     *         quotas</a>. You can resubmit your request later.</li>
     *         <li>ModelErrorException The request failed due to an error while processing the model.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockRuntimeException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample BedrockRuntimeAsyncClient.InvokeModel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-runtime-2023-09-30/InvokeModel" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<InvokeModelResponse> invokeModel(Consumer<InvokeModelRequest.Builder> invokeModelRequest) {
        return invokeModel(InvokeModelRequest.builder().applyMutation(invokeModelRequest).build());
    }

    /**
     * <p>
     * Invoke the specified Amazon Bedrock model to run inference using the prompt and inference parameters provided in
     * the request body. The response is returned in a stream.
     * </p>
     * <p>
     * To see if a model supports streaming, call <a
     * href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_GetFoundationModel.html">GetFoundationModel</a>
     * and check the <code>responseStreamingSupported</code> field in the response.
     * </p>
     * <note>
     * <p>
     * The CLI doesn't support streaming operations in Amazon Bedrock, including
     * <code>InvokeModelWithResponseStream</code>.
     * </p>
     * </note>
     * <p>
     * For example code, see <i>Invoke model with streaming code example</i> in the <i>Amazon Bedrock User Guide</i>.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>bedrock:InvokeModelWithResponseStream</code> action.
     * </p>
     *
     * @param invokeModelWithResponseStreamRequest
     * @return A Java Future containing the result of the InvokeModelWithResponseStream operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException The request is denied because of missing access permissions.</li>
     *         <li>ResourceNotFoundException The specified resource ARN was not found. Check the ARN and try your
     *         request again.</li>
     *         <li>ThrottlingException Your request was throttled because of service-wide limitations. Resubmit your
     *         request later or in a different region. You can also purchase <a
     *         href="https://docs.aws.amazon.com/bedrock/latest/userguide/prov-throughput.html">Provisioned
     *         Throughput</a> to increase the rate or number of tokens you can process.</li>
     *         <li>ModelTimeoutException The request took too long to process. Processing time exceeded the model
     *         timeout length.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ServiceUnavailableException The service isn't currently available. Try again later.</li>
     *         <li>ModelStreamErrorException An error occurred while streaming the response. Retry your request.</li>
     *         <li>ValidationException Input validation failed. Check your request parameters and retry the request.</li>
     *         <li>ModelNotReadyException The model specified in the request is not ready to serve inference requests.
     *         The AWS SDK will automatically retry the operation up to 5 times. For information about configuring
     *         automatic retries, see <a
     *         href="https://docs.aws.amazon.com/sdkref/latest/guide/feature-retry-behavior.html">Retry behavior</a> in
     *         the <i>AWS SDKs and Tools</i> reference guide.</li>
     *         <li>ServiceQuotaExceededException Your request exceeds the service quota for your account. You can view
     *         your quotas at <a
     *         href="https://docs.aws.amazon.com/servicequotas/latest/userguide/gs-request-quota.html">Viewing service
     *         quotas</a>. You can resubmit your request later.</li>
     *         <li>ModelErrorException The request failed due to an error while processing the model.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockRuntimeException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample BedrockRuntimeAsyncClient.InvokeModelWithResponseStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-runtime-2023-09-30/InvokeModelWithResponseStream"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<Void> invokeModelWithResponseStream(
            InvokeModelWithResponseStreamRequest invokeModelWithResponseStreamRequest,
            InvokeModelWithResponseStreamResponseHandler asyncResponseHandler) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Invoke the specified Amazon Bedrock model to run inference using the prompt and inference parameters provided in
     * the request body. The response is returned in a stream.
     * </p>
     * <p>
     * To see if a model supports streaming, call <a
     * href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_GetFoundationModel.html">GetFoundationModel</a>
     * and check the <code>responseStreamingSupported</code> field in the response.
     * </p>
     * <note>
     * <p>
     * The CLI doesn't support streaming operations in Amazon Bedrock, including
     * <code>InvokeModelWithResponseStream</code>.
     * </p>
     * </note>
     * <p>
     * For example code, see <i>Invoke model with streaming code example</i> in the <i>Amazon Bedrock User Guide</i>.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>bedrock:InvokeModelWithResponseStream</code> action.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link InvokeModelWithResponseStreamRequest.Builder}
     * avoiding the need to create one manually via {@link InvokeModelWithResponseStreamRequest#builder()}
     * </p>
     *
     * @param invokeModelWithResponseStreamRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.bedrockruntime.model.InvokeModelWithResponseStreamRequest.Builder}
     *        to create a request.
     * @return A Java Future containing the result of the InvokeModelWithResponseStream operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException The request is denied because of missing access permissions.</li>
     *         <li>ResourceNotFoundException The specified resource ARN was not found. Check the ARN and try your
     *         request again.</li>
     *         <li>ThrottlingException Your request was throttled because of service-wide limitations. Resubmit your
     *         request later or in a different region. You can also purchase <a
     *         href="https://docs.aws.amazon.com/bedrock/latest/userguide/prov-throughput.html">Provisioned
     *         Throughput</a> to increase the rate or number of tokens you can process.</li>
     *         <li>ModelTimeoutException The request took too long to process. Processing time exceeded the model
     *         timeout length.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ServiceUnavailableException The service isn't currently available. Try again later.</li>
     *         <li>ModelStreamErrorException An error occurred while streaming the response. Retry your request.</li>
     *         <li>ValidationException Input validation failed. Check your request parameters and retry the request.</li>
     *         <li>ModelNotReadyException The model specified in the request is not ready to serve inference requests.
     *         The AWS SDK will automatically retry the operation up to 5 times. For information about configuring
     *         automatic retries, see <a
     *         href="https://docs.aws.amazon.com/sdkref/latest/guide/feature-retry-behavior.html">Retry behavior</a> in
     *         the <i>AWS SDKs and Tools</i> reference guide.</li>
     *         <li>ServiceQuotaExceededException Your request exceeds the service quota for your account. You can view
     *         your quotas at <a
     *         href="https://docs.aws.amazon.com/servicequotas/latest/userguide/gs-request-quota.html">Viewing service
     *         quotas</a>. You can resubmit your request later.</li>
     *         <li>ModelErrorException The request failed due to an error while processing the model.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockRuntimeException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample BedrockRuntimeAsyncClient.InvokeModelWithResponseStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-runtime-2023-09-30/InvokeModelWithResponseStream"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<Void> invokeModelWithResponseStream(
            Consumer<InvokeModelWithResponseStreamRequest.Builder> invokeModelWithResponseStreamRequest,
            InvokeModelWithResponseStreamResponseHandler asyncResponseHandler) {
        return invokeModelWithResponseStream(
                InvokeModelWithResponseStreamRequest.builder().applyMutation(invokeModelWithResponseStreamRequest).build(),
                asyncResponseHandler);
    }

    @Override
    default BedrockRuntimeServiceClientConfiguration serviceClientConfiguration() {
        throw new UnsupportedOperationException();
    }

    /**
     * Create a {@link BedrockRuntimeAsyncClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static BedrockRuntimeAsyncClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link BedrockRuntimeAsyncClient}.
     */
    static BedrockRuntimeAsyncClientBuilder builder() {
        return new DefaultBedrockRuntimeAsyncClientBuilder();
    }
}
