/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.bedrockruntime;

import static software.amazon.awssdk.utils.FunctionalUtils.runAndLogError;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.eventstream.EventStreamAsyncResponseTransformer;
import software.amazon.awssdk.awscore.eventstream.EventStreamTaggedUnionPojoSupplier;
import software.amazon.awssdk.awscore.eventstream.RestEventStreamAsyncResponseTransformer;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkPojoBuilder;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.SdkResponse;
import software.amazon.awssdk.core.client.config.SdkAdvancedAsyncClientOption;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.AttachHttpMetadataResponseHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.protocol.VoidSdkResponse;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.bedrockruntime.internal.BedrockRuntimeServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.bedrockruntime.model.AccessDeniedException;
import software.amazon.awssdk.services.bedrockruntime.model.BedrockRuntimeException;
import software.amazon.awssdk.services.bedrockruntime.model.InternalServerException;
import software.amazon.awssdk.services.bedrockruntime.model.InvokeModelRequest;
import software.amazon.awssdk.services.bedrockruntime.model.InvokeModelResponse;
import software.amazon.awssdk.services.bedrockruntime.model.InvokeModelWithResponseStreamRequest;
import software.amazon.awssdk.services.bedrockruntime.model.InvokeModelWithResponseStreamResponse;
import software.amazon.awssdk.services.bedrockruntime.model.InvokeModelWithResponseStreamResponseHandler;
import software.amazon.awssdk.services.bedrockruntime.model.ModelErrorException;
import software.amazon.awssdk.services.bedrockruntime.model.ModelNotReadyException;
import software.amazon.awssdk.services.bedrockruntime.model.ModelStreamErrorException;
import software.amazon.awssdk.services.bedrockruntime.model.ModelTimeoutException;
import software.amazon.awssdk.services.bedrockruntime.model.ResourceNotFoundException;
import software.amazon.awssdk.services.bedrockruntime.model.ResponseStream;
import software.amazon.awssdk.services.bedrockruntime.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.bedrockruntime.model.ThrottlingException;
import software.amazon.awssdk.services.bedrockruntime.model.ValidationException;
import software.amazon.awssdk.services.bedrockruntime.transform.InvokeModelRequestMarshaller;
import software.amazon.awssdk.services.bedrockruntime.transform.InvokeModelWithResponseStreamRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link BedrockRuntimeAsyncClient}.
 *
 * @see BedrockRuntimeAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultBedrockRuntimeAsyncClient implements BedrockRuntimeAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultBedrockRuntimeAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final Executor executor;

    protected DefaultBedrockRuntimeAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
        this.executor = clientConfiguration.option(SdkAdvancedAsyncClientOption.FUTURE_COMPLETION_EXECUTOR);
    }

    /**
     * <p>
     * Invokes the specified Bedrock model to run inference using the input provided in the request body. You use
     * InvokeModel to run inference for text models, image models, and embedding models.
     * </p>
     * <p>
     * For more information, see <a href="https://docs.aws.amazon.com/bedrock/latest/userguide/api-methods-run.html">Run
     * inference</a> in the Bedrock User Guide.
     * </p>
     * <p>
     * For example requests, see Examples (after the Errors section).
     * </p>
     *
     * @param invokeModelRequest
     * @return A Java Future containing the result of the InvokeModel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The request is denied because of missing access permissions.</li>
     *         <li>ResourceNotFoundException The specified resource ARN was not found. Check the ARN and try your
     *         request again.</li>
     *         <li>ThrottlingException The number of requests exceeds the limit. Resubmit your request later.</li>
     *         <li>ModelTimeoutException The request took too long to process. Processing time exceeded the model
     *         timeout length.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ValidationException Input validation failed. Check your request parameters and retry the request.</li>
     *         <li>ModelNotReadyException The model specified in the request is not ready to serve inference requests.</li>
     *         <li>ServiceQuotaExceededException The number of requests exceeds the service quota. Resubmit your request
     *         later.</li>
     *         <li>ModelErrorException The request failed due to an error while processing the model.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockRuntimeException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample BedrockRuntimeAsyncClient.InvokeModel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-runtime-2023-09-30/InvokeModel" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<InvokeModelResponse> invokeModel(InvokeModelRequest invokeModelRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(invokeModelRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, invokeModelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "InvokeModel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(false).build();

            HttpResponseHandler<InvokeModelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    InvokeModelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<InvokeModelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<InvokeModelRequest, InvokeModelResponse>()
                            .withOperationName("InvokeModel").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new InvokeModelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(invokeModelRequest));
            CompletableFuture<InvokeModelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Invoke the specified Bedrock model to run inference using the input provided. Return the response in a stream.
     * </p>
     * <p>
     * For more information, see <a href="https://docs.aws.amazon.com/bedrock/latest/userguide/api-methods-run.html">Run
     * inference</a> in the Bedrock User Guide.
     * </p>
     * <p>
     * For an example request and response, see Examples (after the Errors section).
     * </p>
     *
     * @param invokeModelWithResponseStreamRequest
     * @return A Java Future containing the result of the InvokeModelWithResponseStream operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The request is denied because of missing access permissions.</li>
     *         <li>ResourceNotFoundException The specified resource ARN was not found. Check the ARN and try your
     *         request again.</li>
     *         <li>ThrottlingException The number of requests exceeds the limit. Resubmit your request later.</li>
     *         <li>ModelTimeoutException The request took too long to process. Processing time exceeded the model
     *         timeout length.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ModelStreamErrorException An error occurred while streaming the response.</li>
     *         <li>ValidationException Input validation failed. Check your request parameters and retry the request.</li>
     *         <li>ModelNotReadyException The model specified in the request is not ready to serve inference requests.</li>
     *         <li>ServiceQuotaExceededException The number of requests exceeds the service quota. Resubmit your request
     *         later.</li>
     *         <li>ModelErrorException The request failed due to an error while processing the model.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockRuntimeException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample BedrockRuntimeAsyncClient.InvokeModelWithResponseStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-runtime-2023-09-30/InvokeModelWithResponseStream"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<Void> invokeModelWithResponseStream(
            InvokeModelWithResponseStreamRequest invokeModelWithResponseStreamRequest,
            InvokeModelWithResponseStreamResponseHandler asyncResponseHandler) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(invokeModelWithResponseStreamRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                invokeModelWithResponseStreamRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "InvokeModelWithResponseStream");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<InvokeModelWithResponseStreamResponse> responseHandler = new AttachHttpMetadataResponseHandler(
                    protocolFactory.createResponseHandler(operationMetadata, InvokeModelWithResponseStreamResponse::builder));

            HttpResponseHandler<SdkResponse> voidResponseHandler = protocolFactory.createResponseHandler(JsonOperationMetadata
                    .builder().isPayloadJson(false).hasStreamingSuccessResponse(true).build(), VoidSdkResponse::builder);

            HttpResponseHandler<? extends ResponseStream> eventResponseHandler = protocolFactory.createResponseHandler(
                    JsonOperationMetadata.builder().isPayloadJson(true).hasStreamingSuccessResponse(false).build(),
                    EventStreamTaggedUnionPojoSupplier.builder().putSdkPojoSupplier("chunk", ResponseStream::chunkBuilder)
                            .defaultSdkPojoSupplier(() -> new SdkPojoBuilder(ResponseStream.UNKNOWN)).build());

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            CompletableFuture<Void> future = new CompletableFuture<>();
            EventStreamAsyncResponseTransformer<InvokeModelWithResponseStreamResponse, ResponseStream> asyncResponseTransformer = EventStreamAsyncResponseTransformer
                    .<InvokeModelWithResponseStreamResponse, ResponseStream> builder()
                    .eventStreamResponseHandler(asyncResponseHandler).eventResponseHandler(eventResponseHandler)
                    .initialResponseHandler(responseHandler).exceptionResponseHandler(errorResponseHandler).future(future)
                    .executor(executor).serviceName(serviceName()).build();
            RestEventStreamAsyncResponseTransformer<InvokeModelWithResponseStreamResponse, ResponseStream> restAsyncResponseTransformer = RestEventStreamAsyncResponseTransformer
                    .<InvokeModelWithResponseStreamResponse, ResponseStream> builder()
                    .eventStreamAsyncResponseTransformer(asyncResponseTransformer)
                    .eventStreamResponseHandler(asyncResponseHandler).build();

            CompletableFuture<Void> executeFuture = clientHandler.execute(
                    new ClientExecutionParams<InvokeModelWithResponseStreamRequest, InvokeModelWithResponseStreamResponse>()
                            .withOperationName("InvokeModelWithResponseStream").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new InvokeModelWithResponseStreamRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(invokeModelWithResponseStreamRequest), restAsyncResponseTransformer);
            CompletableFuture<Void> whenCompleted = executeFuture.whenComplete((r, e) -> {
                if (e != null) {
                    try {
                        asyncResponseHandler.exceptionOccurred(e);
                    } finally {
                        future.completeExceptionally(e);
                    }
                }
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return CompletableFutureUtils.forwardExceptionTo(future, executeFuture);
        } catch (Throwable t) {
            runAndLogError(log, "Exception thrown in exceptionOccurred callback, ignoring",
                    () -> asyncResponseHandler.exceptionOccurred(t));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final BedrockRuntimeServiceClientConfiguration serviceClientConfiguration() {
        return new BedrockRuntimeServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(BedrockRuntimeException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ModelStreamErrorException")
                                .exceptionBuilderSupplier(ModelStreamErrorException::builder).httpStatusCode(424).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ModelErrorException")
                                .exceptionBuilderSupplier(ModelErrorException::builder).httpStatusCode(424).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ModelTimeoutException")
                                .exceptionBuilderSupplier(ModelTimeoutException::builder).httpStatusCode(408).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ModelNotReadyException")
                                .exceptionBuilderSupplier(ModelNotReadyException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        BedrockRuntimeServiceClientConfigurationBuilder serviceConfigBuilder = new BedrockRuntimeServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
