/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.bedrockagentcore.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A condensed representation of a code interpreter session in Amazon Bedrock. This structure contains key information
 * about a code interpreter session, including identifiers, status, and timestamps, without the full details of the
 * session configuration.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CodeInterpreterSessionSummary implements SdkPojo, Serializable,
        ToCopyableBuilder<CodeInterpreterSessionSummary.Builder, CodeInterpreterSessionSummary> {
    private static final SdkField<String> CODE_INTERPRETER_IDENTIFIER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("codeInterpreterIdentifier").getter(getter(CodeInterpreterSessionSummary::codeInterpreterIdentifier))
            .setter(setter(Builder::codeInterpreterIdentifier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("codeInterpreterIdentifier").build())
            .build();

    private static final SdkField<String> SESSION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("sessionId").getter(getter(CodeInterpreterSessionSummary::sessionId)).setter(setter(Builder::sessionId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sessionId").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(CodeInterpreterSessionSummary::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("status")
            .getter(getter(CodeInterpreterSessionSummary::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("createdAt")
            .getter(getter(CodeInterpreterSessionSummary::createdAt))
            .setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<Instant> LAST_UPDATED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("lastUpdatedAt")
            .getter(getter(CodeInterpreterSessionSummary::lastUpdatedAt))
            .setter(setter(Builder::lastUpdatedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lastUpdatedAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            CODE_INTERPRETER_IDENTIFIER_FIELD, SESSION_ID_FIELD, NAME_FIELD, STATUS_FIELD, CREATED_AT_FIELD,
            LAST_UPDATED_AT_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String codeInterpreterIdentifier;

    private final String sessionId;

    private final String name;

    private final String status;

    private final Instant createdAt;

    private final Instant lastUpdatedAt;

    private CodeInterpreterSessionSummary(BuilderImpl builder) {
        this.codeInterpreterIdentifier = builder.codeInterpreterIdentifier;
        this.sessionId = builder.sessionId;
        this.name = builder.name;
        this.status = builder.status;
        this.createdAt = builder.createdAt;
        this.lastUpdatedAt = builder.lastUpdatedAt;
    }

    /**
     * <p>
     * The unique identifier of the code interpreter associated with the session. This identifier specifies which code
     * interpreter environment is used for the session.
     * </p>
     * 
     * @return The unique identifier of the code interpreter associated with the session. This identifier specifies
     *         which code interpreter environment is used for the session.
     */
    public final String codeInterpreterIdentifier() {
        return codeInterpreterIdentifier;
    }

    /**
     * <p>
     * The unique identifier of the code interpreter session. This identifier is used in operations that interact with
     * the session.
     * </p>
     * 
     * @return The unique identifier of the code interpreter session. This identifier is used in operations that
     *         interact with the session.
     */
    public final String sessionId() {
        return sessionId;
    }

    /**
     * <p>
     * The name of the code interpreter session. This name helps identify and manage the session.
     * </p>
     * 
     * @return The name of the code interpreter session. This name helps identify and manage the session.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The current status of the code interpreter session. Possible values include ACTIVE, STOPPING, and STOPPED.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link CodeInterpreterSessionStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of the code interpreter session. Possible values include ACTIVE, STOPPING, and
     *         STOPPED.
     * @see CodeInterpreterSessionStatus
     */
    public final CodeInterpreterSessionStatus status() {
        return CodeInterpreterSessionStatus.fromValue(status);
    }

    /**
     * <p>
     * The current status of the code interpreter session. Possible values include ACTIVE, STOPPING, and STOPPED.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link CodeInterpreterSessionStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of the code interpreter session. Possible values include ACTIVE, STOPPING, and
     *         STOPPED.
     * @see CodeInterpreterSessionStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The timestamp when the code interpreter session was created. This value is in ISO 8601 format.
     * </p>
     * 
     * @return The timestamp when the code interpreter session was created. This value is in ISO 8601 format.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The timestamp when the code interpreter session was last updated. This value is in ISO 8601 format.
     * </p>
     * 
     * @return The timestamp when the code interpreter session was last updated. This value is in ISO 8601 format.
     */
    public final Instant lastUpdatedAt() {
        return lastUpdatedAt;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(codeInterpreterIdentifier());
        hashCode = 31 * hashCode + Objects.hashCode(sessionId());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdatedAt());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CodeInterpreterSessionSummary)) {
            return false;
        }
        CodeInterpreterSessionSummary other = (CodeInterpreterSessionSummary) obj;
        return Objects.equals(codeInterpreterIdentifier(), other.codeInterpreterIdentifier())
                && Objects.equals(sessionId(), other.sessionId()) && Objects.equals(name(), other.name())
                && Objects.equals(statusAsString(), other.statusAsString()) && Objects.equals(createdAt(), other.createdAt())
                && Objects.equals(lastUpdatedAt(), other.lastUpdatedAt());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CodeInterpreterSessionSummary").add("CodeInterpreterIdentifier", codeInterpreterIdentifier())
                .add("SessionId", sessionId()).add("Name", name()).add("Status", statusAsString()).add("CreatedAt", createdAt())
                .add("LastUpdatedAt", lastUpdatedAt()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "codeInterpreterIdentifier":
            return Optional.ofNullable(clazz.cast(codeInterpreterIdentifier()));
        case "sessionId":
            return Optional.ofNullable(clazz.cast(sessionId()));
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "lastUpdatedAt":
            return Optional.ofNullable(clazz.cast(lastUpdatedAt()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("codeInterpreterIdentifier", CODE_INTERPRETER_IDENTIFIER_FIELD);
        map.put("sessionId", SESSION_ID_FIELD);
        map.put("name", NAME_FIELD);
        map.put("status", STATUS_FIELD);
        map.put("createdAt", CREATED_AT_FIELD);
        map.put("lastUpdatedAt", LAST_UPDATED_AT_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<CodeInterpreterSessionSummary, T> g) {
        return obj -> g.apply((CodeInterpreterSessionSummary) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CodeInterpreterSessionSummary> {
        /**
         * <p>
         * The unique identifier of the code interpreter associated with the session. This identifier specifies which
         * code interpreter environment is used for the session.
         * </p>
         * 
         * @param codeInterpreterIdentifier
         *        The unique identifier of the code interpreter associated with the session. This identifier specifies
         *        which code interpreter environment is used for the session.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder codeInterpreterIdentifier(String codeInterpreterIdentifier);

        /**
         * <p>
         * The unique identifier of the code interpreter session. This identifier is used in operations that interact
         * with the session.
         * </p>
         * 
         * @param sessionId
         *        The unique identifier of the code interpreter session. This identifier is used in operations that
         *        interact with the session.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sessionId(String sessionId);

        /**
         * <p>
         * The name of the code interpreter session. This name helps identify and manage the session.
         * </p>
         * 
         * @param name
         *        The name of the code interpreter session. This name helps identify and manage the session.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The current status of the code interpreter session. Possible values include ACTIVE, STOPPING, and STOPPED.
         * </p>
         * 
         * @param status
         *        The current status of the code interpreter session. Possible values include ACTIVE, STOPPING, and
         *        STOPPED.
         * @see CodeInterpreterSessionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CodeInterpreterSessionStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The current status of the code interpreter session. Possible values include ACTIVE, STOPPING, and STOPPED.
         * </p>
         * 
         * @param status
         *        The current status of the code interpreter session. Possible values include ACTIVE, STOPPING, and
         *        STOPPED.
         * @see CodeInterpreterSessionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CodeInterpreterSessionStatus
         */
        Builder status(CodeInterpreterSessionStatus status);

        /**
         * <p>
         * The timestamp when the code interpreter session was created. This value is in ISO 8601 format.
         * </p>
         * 
         * @param createdAt
         *        The timestamp when the code interpreter session was created. This value is in ISO 8601 format.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The timestamp when the code interpreter session was last updated. This value is in ISO 8601 format.
         * </p>
         * 
         * @param lastUpdatedAt
         *        The timestamp when the code interpreter session was last updated. This value is in ISO 8601 format.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdatedAt(Instant lastUpdatedAt);
    }

    static final class BuilderImpl implements Builder {
        private String codeInterpreterIdentifier;

        private String sessionId;

        private String name;

        private String status;

        private Instant createdAt;

        private Instant lastUpdatedAt;

        private BuilderImpl() {
        }

        private BuilderImpl(CodeInterpreterSessionSummary model) {
            codeInterpreterIdentifier(model.codeInterpreterIdentifier);
            sessionId(model.sessionId);
            name(model.name);
            status(model.status);
            createdAt(model.createdAt);
            lastUpdatedAt(model.lastUpdatedAt);
        }

        public final String getCodeInterpreterIdentifier() {
            return codeInterpreterIdentifier;
        }

        public final void setCodeInterpreterIdentifier(String codeInterpreterIdentifier) {
            this.codeInterpreterIdentifier = codeInterpreterIdentifier;
        }

        @Override
        public final Builder codeInterpreterIdentifier(String codeInterpreterIdentifier) {
            this.codeInterpreterIdentifier = codeInterpreterIdentifier;
            return this;
        }

        public final String getSessionId() {
            return sessionId;
        }

        public final void setSessionId(String sessionId) {
            this.sessionId = sessionId;
        }

        @Override
        public final Builder sessionId(String sessionId) {
            this.sessionId = sessionId;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(CodeInterpreterSessionStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final Instant getLastUpdatedAt() {
            return lastUpdatedAt;
        }

        public final void setLastUpdatedAt(Instant lastUpdatedAt) {
            this.lastUpdatedAt = lastUpdatedAt;
        }

        @Override
        public final Builder lastUpdatedAt(Instant lastUpdatedAt) {
            this.lastUpdatedAt = lastUpdatedAt;
            return this;
        }

        @Override
        public CodeInterpreterSessionSummary build() {
            return new CodeInterpreterSessionSummary(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
