/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.bedrockagentcore.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about an event in an AgentCore Memory resource.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Event implements SdkPojo, Serializable, ToCopyableBuilder<Event.Builder, Event> {
    private static final SdkField<String> MEMORY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("memoryId").getter(getter(Event::memoryId)).setter(setter(Builder::memoryId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("memoryId").build()).build();

    private static final SdkField<String> ACTOR_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("actorId").getter(getter(Event::actorId)).setter(setter(Builder::actorId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("actorId").build()).build();

    private static final SdkField<String> SESSION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("sessionId").getter(getter(Event::sessionId)).setter(setter(Builder::sessionId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sessionId").build()).build();

    private static final SdkField<String> EVENT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("eventId").getter(getter(Event::eventId)).setter(setter(Builder::eventId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("eventId").build()).build();

    private static final SdkField<Instant> EVENT_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("eventTimestamp").getter(getter(Event::eventTimestamp)).setter(setter(Builder::eventTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("eventTimestamp").build()).build();

    private static final SdkField<List<PayloadType>> PAYLOAD_FIELD = SdkField
            .<List<PayloadType>> builder(MarshallingType.LIST)
            .memberName("payload")
            .getter(getter(Event::payload))
            .setter(setter(Builder::payload))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("payload").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<PayloadType> builder(MarshallingType.SDK_POJO)
                                            .constructor(PayloadType::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Branch> BRANCH_FIELD = SdkField.<Branch> builder(MarshallingType.SDK_POJO).memberName("branch")
            .getter(getter(Event::branch)).setter(setter(Builder::branch)).constructor(Branch::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("branch").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(MEMORY_ID_FIELD,
            ACTOR_ID_FIELD, SESSION_ID_FIELD, EVENT_ID_FIELD, EVENT_TIMESTAMP_FIELD, PAYLOAD_FIELD, BRANCH_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String memoryId;

    private final String actorId;

    private final String sessionId;

    private final String eventId;

    private final Instant eventTimestamp;

    private final List<PayloadType> payload;

    private final Branch branch;

    private Event(BuilderImpl builder) {
        this.memoryId = builder.memoryId;
        this.actorId = builder.actorId;
        this.sessionId = builder.sessionId;
        this.eventId = builder.eventId;
        this.eventTimestamp = builder.eventTimestamp;
        this.payload = builder.payload;
        this.branch = builder.branch;
    }

    /**
     * <p>
     * The identifier of the AgentCore Memory resource containing the event.
     * </p>
     * 
     * @return The identifier of the AgentCore Memory resource containing the event.
     */
    public final String memoryId() {
        return memoryId;
    }

    /**
     * <p>
     * The identifier of the actor associated with the event.
     * </p>
     * 
     * @return The identifier of the actor associated with the event.
     */
    public final String actorId() {
        return actorId;
    }

    /**
     * <p>
     * The identifier of the session containing the event.
     * </p>
     * 
     * @return The identifier of the session containing the event.
     */
    public final String sessionId() {
        return sessionId;
    }

    /**
     * <p>
     * The unique identifier of the event.
     * </p>
     * 
     * @return The unique identifier of the event.
     */
    public final String eventId() {
        return eventId;
    }

    /**
     * <p>
     * The timestamp when the event occurred.
     * </p>
     * 
     * @return The timestamp when the event occurred.
     */
    public final Instant eventTimestamp() {
        return eventTimestamp;
    }

    /**
     * For responses, this returns true if the service returned a value for the Payload property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasPayload() {
        return payload != null && !(payload instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The content payload of the event.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPayload} method.
     * </p>
     * 
     * @return The content payload of the event.
     */
    public final List<PayloadType> payload() {
        return payload;
    }

    /**
     * <p>
     * The branch information for the event.
     * </p>
     * 
     * @return The branch information for the event.
     */
    public final Branch branch() {
        return branch;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(memoryId());
        hashCode = 31 * hashCode + Objects.hashCode(actorId());
        hashCode = 31 * hashCode + Objects.hashCode(sessionId());
        hashCode = 31 * hashCode + Objects.hashCode(eventId());
        hashCode = 31 * hashCode + Objects.hashCode(eventTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(hasPayload() ? payload() : null);
        hashCode = 31 * hashCode + Objects.hashCode(branch());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Event)) {
            return false;
        }
        Event other = (Event) obj;
        return Objects.equals(memoryId(), other.memoryId()) && Objects.equals(actorId(), other.actorId())
                && Objects.equals(sessionId(), other.sessionId()) && Objects.equals(eventId(), other.eventId())
                && Objects.equals(eventTimestamp(), other.eventTimestamp()) && hasPayload() == other.hasPayload()
                && Objects.equals(payload(), other.payload()) && Objects.equals(branch(), other.branch());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Event").add("MemoryId", memoryId()).add("ActorId", actorId()).add("SessionId", sessionId())
                .add("EventId", eventId()).add("EventTimestamp", eventTimestamp())
                .add("Payload", hasPayload() ? payload() : null).add("Branch", branch()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "memoryId":
            return Optional.ofNullable(clazz.cast(memoryId()));
        case "actorId":
            return Optional.ofNullable(clazz.cast(actorId()));
        case "sessionId":
            return Optional.ofNullable(clazz.cast(sessionId()));
        case "eventId":
            return Optional.ofNullable(clazz.cast(eventId()));
        case "eventTimestamp":
            return Optional.ofNullable(clazz.cast(eventTimestamp()));
        case "payload":
            return Optional.ofNullable(clazz.cast(payload()));
        case "branch":
            return Optional.ofNullable(clazz.cast(branch()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("memoryId", MEMORY_ID_FIELD);
        map.put("actorId", ACTOR_ID_FIELD);
        map.put("sessionId", SESSION_ID_FIELD);
        map.put("eventId", EVENT_ID_FIELD);
        map.put("eventTimestamp", EVENT_TIMESTAMP_FIELD);
        map.put("payload", PAYLOAD_FIELD);
        map.put("branch", BRANCH_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Event, T> g) {
        return obj -> g.apply((Event) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Event> {
        /**
         * <p>
         * The identifier of the AgentCore Memory resource containing the event.
         * </p>
         * 
         * @param memoryId
         *        The identifier of the AgentCore Memory resource containing the event.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder memoryId(String memoryId);

        /**
         * <p>
         * The identifier of the actor associated with the event.
         * </p>
         * 
         * @param actorId
         *        The identifier of the actor associated with the event.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder actorId(String actorId);

        /**
         * <p>
         * The identifier of the session containing the event.
         * </p>
         * 
         * @param sessionId
         *        The identifier of the session containing the event.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sessionId(String sessionId);

        /**
         * <p>
         * The unique identifier of the event.
         * </p>
         * 
         * @param eventId
         *        The unique identifier of the event.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventId(String eventId);

        /**
         * <p>
         * The timestamp when the event occurred.
         * </p>
         * 
         * @param eventTimestamp
         *        The timestamp when the event occurred.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventTimestamp(Instant eventTimestamp);

        /**
         * <p>
         * The content payload of the event.
         * </p>
         * 
         * @param payload
         *        The content payload of the event.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder payload(Collection<PayloadType> payload);

        /**
         * <p>
         * The content payload of the event.
         * </p>
         * 
         * @param payload
         *        The content payload of the event.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder payload(PayloadType... payload);

        /**
         * <p>
         * The content payload of the event.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.bedrockagentcore.model.PayloadType.Builder} avoiding the need to
         * create one manually via {@link software.amazon.awssdk.services.bedrockagentcore.model.PayloadType#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.bedrockagentcore.model.PayloadType.Builder#build()} is called
         * immediately and its result is passed to {@link #payload(List<PayloadType>)}.
         * 
         * @param payload
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.bedrockagentcore.model.PayloadType.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #payload(java.util.Collection<PayloadType>)
         */
        Builder payload(Consumer<PayloadType.Builder>... payload);

        /**
         * <p>
         * The branch information for the event.
         * </p>
         * 
         * @param branch
         *        The branch information for the event.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder branch(Branch branch);

        /**
         * <p>
         * The branch information for the event.
         * </p>
         * This is a convenience method that creates an instance of the {@link Branch.Builder} avoiding the need to
         * create one manually via {@link Branch#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Branch.Builder#build()} is called immediately and its result is
         * passed to {@link #branch(Branch)}.
         * 
         * @param branch
         *        a consumer that will call methods on {@link Branch.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #branch(Branch)
         */
        default Builder branch(Consumer<Branch.Builder> branch) {
            return branch(Branch.builder().applyMutation(branch).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String memoryId;

        private String actorId;

        private String sessionId;

        private String eventId;

        private Instant eventTimestamp;

        private List<PayloadType> payload = DefaultSdkAutoConstructList.getInstance();

        private Branch branch;

        private BuilderImpl() {
        }

        private BuilderImpl(Event model) {
            memoryId(model.memoryId);
            actorId(model.actorId);
            sessionId(model.sessionId);
            eventId(model.eventId);
            eventTimestamp(model.eventTimestamp);
            payload(model.payload);
            branch(model.branch);
        }

        public final String getMemoryId() {
            return memoryId;
        }

        public final void setMemoryId(String memoryId) {
            this.memoryId = memoryId;
        }

        @Override
        public final Builder memoryId(String memoryId) {
            this.memoryId = memoryId;
            return this;
        }

        public final String getActorId() {
            return actorId;
        }

        public final void setActorId(String actorId) {
            this.actorId = actorId;
        }

        @Override
        public final Builder actorId(String actorId) {
            this.actorId = actorId;
            return this;
        }

        public final String getSessionId() {
            return sessionId;
        }

        public final void setSessionId(String sessionId) {
            this.sessionId = sessionId;
        }

        @Override
        public final Builder sessionId(String sessionId) {
            this.sessionId = sessionId;
            return this;
        }

        public final String getEventId() {
            return eventId;
        }

        public final void setEventId(String eventId) {
            this.eventId = eventId;
        }

        @Override
        public final Builder eventId(String eventId) {
            this.eventId = eventId;
            return this;
        }

        public final Instant getEventTimestamp() {
            return eventTimestamp;
        }

        public final void setEventTimestamp(Instant eventTimestamp) {
            this.eventTimestamp = eventTimestamp;
        }

        @Override
        public final Builder eventTimestamp(Instant eventTimestamp) {
            this.eventTimestamp = eventTimestamp;
            return this;
        }

        public final List<PayloadType.Builder> getPayload() {
            List<PayloadType.Builder> result = PayloadTypeListCopier.copyToBuilder(this.payload);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setPayload(Collection<PayloadType.BuilderImpl> payload) {
            this.payload = PayloadTypeListCopier.copyFromBuilder(payload);
        }

        @Override
        public final Builder payload(Collection<PayloadType> payload) {
            this.payload = PayloadTypeListCopier.copy(payload);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder payload(PayloadType... payload) {
            payload(Arrays.asList(payload));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder payload(Consumer<PayloadType.Builder>... payload) {
            payload(Stream.of(payload).map(c -> PayloadType.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final Branch.Builder getBranch() {
            return branch != null ? branch.toBuilder() : null;
        }

        public final void setBranch(Branch.BuilderImpl branch) {
            this.branch = branch != null ? branch.build() : null;
        }

        @Override
        public final Builder branch(Branch branch) {
            this.branch = branch;
            return this;
        }

        @Override
        public Event build() {
            return new Event(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
