/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.bedrockagentcore.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains summary information about a memory record.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class MemoryRecordSummary implements SdkPojo, Serializable,
        ToCopyableBuilder<MemoryRecordSummary.Builder, MemoryRecordSummary> {
    private static final SdkField<String> MEMORY_RECORD_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("memoryRecordId").getter(getter(MemoryRecordSummary::memoryRecordId))
            .setter(setter(Builder::memoryRecordId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("memoryRecordId").build()).build();

    private static final SdkField<MemoryContent> CONTENT_FIELD = SdkField.<MemoryContent> builder(MarshallingType.SDK_POJO)
            .memberName("content").getter(getter(MemoryRecordSummary::content)).setter(setter(Builder::content))
            .constructor(MemoryContent::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("content").build()).build();

    private static final SdkField<String> MEMORY_STRATEGY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("memoryStrategyId").getter(getter(MemoryRecordSummary::memoryStrategyId))
            .setter(setter(Builder::memoryStrategyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("memoryStrategyId").build()).build();

    private static final SdkField<List<String>> NAMESPACES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("namespaces")
            .getter(getter(MemoryRecordSummary::namespaces))
            .setter(setter(Builder::namespaces))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("namespaces").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("createdAt").getter(getter(MemoryRecordSummary::createdAt)).setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build()).build();

    private static final SdkField<Double> SCORE_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE).memberName("score")
            .getter(getter(MemoryRecordSummary::score)).setter(setter(Builder::score))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("score").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(MEMORY_RECORD_ID_FIELD,
            CONTENT_FIELD, MEMORY_STRATEGY_ID_FIELD, NAMESPACES_FIELD, CREATED_AT_FIELD, SCORE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String memoryRecordId;

    private final MemoryContent content;

    private final String memoryStrategyId;

    private final List<String> namespaces;

    private final Instant createdAt;

    private final Double score;

    private MemoryRecordSummary(BuilderImpl builder) {
        this.memoryRecordId = builder.memoryRecordId;
        this.content = builder.content;
        this.memoryStrategyId = builder.memoryStrategyId;
        this.namespaces = builder.namespaces;
        this.createdAt = builder.createdAt;
        this.score = builder.score;
    }

    /**
     * <p>
     * The unique identifier of the memory record.
     * </p>
     * 
     * @return The unique identifier of the memory record.
     */
    public final String memoryRecordId() {
        return memoryRecordId;
    }

    /**
     * <p>
     * The content of the memory record.
     * </p>
     * 
     * @return The content of the memory record.
     */
    public final MemoryContent content() {
        return content;
    }

    /**
     * <p>
     * The identifier of the memory strategy associated with this record.
     * </p>
     * 
     * @return The identifier of the memory strategy associated with this record.
     */
    public final String memoryStrategyId() {
        return memoryStrategyId;
    }

    /**
     * For responses, this returns true if the service returned a value for the Namespaces property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasNamespaces() {
        return namespaces != null && !(namespaces instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The namespaces associated with this memory record.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasNamespaces} method.
     * </p>
     * 
     * @return The namespaces associated with this memory record.
     */
    public final List<String> namespaces() {
        return namespaces;
    }

    /**
     * <p>
     * The timestamp when the memory record was created.
     * </p>
     * 
     * @return The timestamp when the memory record was created.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The relevance score of the memory record when returned as part of a search result. Higher values indicate greater
     * relevance to the search query.
     * </p>
     * 
     * @return The relevance score of the memory record when returned as part of a search result. Higher values indicate
     *         greater relevance to the search query.
     */
    public final Double score() {
        return score;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(memoryRecordId());
        hashCode = 31 * hashCode + Objects.hashCode(content());
        hashCode = 31 * hashCode + Objects.hashCode(memoryStrategyId());
        hashCode = 31 * hashCode + Objects.hashCode(hasNamespaces() ? namespaces() : null);
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(score());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof MemoryRecordSummary)) {
            return false;
        }
        MemoryRecordSummary other = (MemoryRecordSummary) obj;
        return Objects.equals(memoryRecordId(), other.memoryRecordId()) && Objects.equals(content(), other.content())
                && Objects.equals(memoryStrategyId(), other.memoryStrategyId()) && hasNamespaces() == other.hasNamespaces()
                && Objects.equals(namespaces(), other.namespaces()) && Objects.equals(createdAt(), other.createdAt())
                && Objects.equals(score(), other.score());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("MemoryRecordSummary").add("MemoryRecordId", memoryRecordId()).add("Content", content())
                .add("MemoryStrategyId", memoryStrategyId()).add("Namespaces", hasNamespaces() ? namespaces() : null)
                .add("CreatedAt", createdAt()).add("Score", score()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "memoryRecordId":
            return Optional.ofNullable(clazz.cast(memoryRecordId()));
        case "content":
            return Optional.ofNullable(clazz.cast(content()));
        case "memoryStrategyId":
            return Optional.ofNullable(clazz.cast(memoryStrategyId()));
        case "namespaces":
            return Optional.ofNullable(clazz.cast(namespaces()));
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "score":
            return Optional.ofNullable(clazz.cast(score()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("memoryRecordId", MEMORY_RECORD_ID_FIELD);
        map.put("content", CONTENT_FIELD);
        map.put("memoryStrategyId", MEMORY_STRATEGY_ID_FIELD);
        map.put("namespaces", NAMESPACES_FIELD);
        map.put("createdAt", CREATED_AT_FIELD);
        map.put("score", SCORE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<MemoryRecordSummary, T> g) {
        return obj -> g.apply((MemoryRecordSummary) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, MemoryRecordSummary> {
        /**
         * <p>
         * The unique identifier of the memory record.
         * </p>
         * 
         * @param memoryRecordId
         *        The unique identifier of the memory record.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder memoryRecordId(String memoryRecordId);

        /**
         * <p>
         * The content of the memory record.
         * </p>
         * 
         * @param content
         *        The content of the memory record.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder content(MemoryContent content);

        /**
         * <p>
         * The content of the memory record.
         * </p>
         * This is a convenience method that creates an instance of the {@link MemoryContent.Builder} avoiding the need
         * to create one manually via {@link MemoryContent#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link MemoryContent.Builder#build()} is called immediately and its
         * result is passed to {@link #content(MemoryContent)}.
         * 
         * @param content
         *        a consumer that will call methods on {@link MemoryContent.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #content(MemoryContent)
         */
        default Builder content(Consumer<MemoryContent.Builder> content) {
            return content(MemoryContent.builder().applyMutation(content).build());
        }

        /**
         * <p>
         * The identifier of the memory strategy associated with this record.
         * </p>
         * 
         * @param memoryStrategyId
         *        The identifier of the memory strategy associated with this record.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder memoryStrategyId(String memoryStrategyId);

        /**
         * <p>
         * The namespaces associated with this memory record.
         * </p>
         * 
         * @param namespaces
         *        The namespaces associated with this memory record.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder namespaces(Collection<String> namespaces);

        /**
         * <p>
         * The namespaces associated with this memory record.
         * </p>
         * 
         * @param namespaces
         *        The namespaces associated with this memory record.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder namespaces(String... namespaces);

        /**
         * <p>
         * The timestamp when the memory record was created.
         * </p>
         * 
         * @param createdAt
         *        The timestamp when the memory record was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The relevance score of the memory record when returned as part of a search result. Higher values indicate
         * greater relevance to the search query.
         * </p>
         * 
         * @param score
         *        The relevance score of the memory record when returned as part of a search result. Higher values
         *        indicate greater relevance to the search query.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder score(Double score);
    }

    static final class BuilderImpl implements Builder {
        private String memoryRecordId;

        private MemoryContent content;

        private String memoryStrategyId;

        private List<String> namespaces = DefaultSdkAutoConstructList.getInstance();

        private Instant createdAt;

        private Double score;

        private BuilderImpl() {
        }

        private BuilderImpl(MemoryRecordSummary model) {
            memoryRecordId(model.memoryRecordId);
            content(model.content);
            memoryStrategyId(model.memoryStrategyId);
            namespaces(model.namespaces);
            createdAt(model.createdAt);
            score(model.score);
        }

        public final String getMemoryRecordId() {
            return memoryRecordId;
        }

        public final void setMemoryRecordId(String memoryRecordId) {
            this.memoryRecordId = memoryRecordId;
        }

        @Override
        public final Builder memoryRecordId(String memoryRecordId) {
            this.memoryRecordId = memoryRecordId;
            return this;
        }

        public final MemoryContent.Builder getContent() {
            return content != null ? content.toBuilder() : null;
        }

        public final void setContent(MemoryContent.BuilderImpl content) {
            this.content = content != null ? content.build() : null;
        }

        @Override
        public final Builder content(MemoryContent content) {
            this.content = content;
            return this;
        }

        public final String getMemoryStrategyId() {
            return memoryStrategyId;
        }

        public final void setMemoryStrategyId(String memoryStrategyId) {
            this.memoryStrategyId = memoryStrategyId;
        }

        @Override
        public final Builder memoryStrategyId(String memoryStrategyId) {
            this.memoryStrategyId = memoryStrategyId;
            return this;
        }

        public final Collection<String> getNamespaces() {
            if (namespaces instanceof SdkAutoConstructList) {
                return null;
            }
            return namespaces;
        }

        public final void setNamespaces(Collection<String> namespaces) {
            this.namespaces = NamespacesListCopier.copy(namespaces);
        }

        @Override
        public final Builder namespaces(Collection<String> namespaces) {
            this.namespaces = NamespacesListCopier.copy(namespaces);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder namespaces(String... namespaces) {
            namespaces(Arrays.asList(namespaces));
            return this;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final Double getScore() {
            return score;
        }

        public final void setScore(Double score) {
            this.score = score;
        }

        @Override
        public final Builder score(Double score) {
            this.score = score;
            return this;
        }

        @Override
        public MemoryRecordSummary build() {
            return new MemoryRecordSummary(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
