/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.bedrockagent.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Determines how multiple nodes in a flow can run in parallel. Running nodes concurrently can improve your flow's
 * performance.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class FlowAliasConcurrencyConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<FlowAliasConcurrencyConfiguration.Builder, FlowAliasConcurrencyConfiguration> {
    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("type")
            .getter(getter(FlowAliasConcurrencyConfiguration::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("type").build()).build();

    private static final SdkField<Integer> MAX_CONCURRENCY_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("maxConcurrency").getter(getter(FlowAliasConcurrencyConfiguration::maxConcurrency))
            .setter(setter(Builder::maxConcurrency))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("maxConcurrency").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TYPE_FIELD,
            MAX_CONCURRENCY_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String type;

    private final Integer maxConcurrency;

    private FlowAliasConcurrencyConfiguration(BuilderImpl builder) {
        this.type = builder.type;
        this.maxConcurrency = builder.maxConcurrency;
    }

    /**
     * <p>
     * The type of concurrency to use for parallel node execution. Specify one of the following options:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Automatic</code> - Amazon Bedrock determines which nodes can be executed in parallel based on the flow
     * definition and its dependencies.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Manual</code> - You specify which nodes can be executed in parallel.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link ConcurrencyType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of concurrency to use for parallel node execution. Specify one of the following options:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Automatic</code> - Amazon Bedrock determines which nodes can be executed in parallel based on the
     *         flow definition and its dependencies.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Manual</code> - You specify which nodes can be executed in parallel.
     *         </p>
     *         </li>
     * @see ConcurrencyType
     */
    public final ConcurrencyType type() {
        return ConcurrencyType.fromValue(type);
    }

    /**
     * <p>
     * The type of concurrency to use for parallel node execution. Specify one of the following options:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Automatic</code> - Amazon Bedrock determines which nodes can be executed in parallel based on the flow
     * definition and its dependencies.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Manual</code> - You specify which nodes can be executed in parallel.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link ConcurrencyType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of concurrency to use for parallel node execution. Specify one of the following options:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Automatic</code> - Amazon Bedrock determines which nodes can be executed in parallel based on the
     *         flow definition and its dependencies.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Manual</code> - You specify which nodes can be executed in parallel.
     *         </p>
     *         </li>
     * @see ConcurrencyType
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * <p>
     * The maximum number of nodes that can be executed concurrently in the flow.
     * </p>
     * 
     * @return The maximum number of nodes that can be executed concurrently in the flow.
     */
    public final Integer maxConcurrency() {
        return maxConcurrency;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(maxConcurrency());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof FlowAliasConcurrencyConfiguration)) {
            return false;
        }
        FlowAliasConcurrencyConfiguration other = (FlowAliasConcurrencyConfiguration) obj;
        return Objects.equals(typeAsString(), other.typeAsString()) && Objects.equals(maxConcurrency(), other.maxConcurrency());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("FlowAliasConcurrencyConfiguration").add("Type", typeAsString())
                .add("MaxConcurrency", maxConcurrency()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "maxConcurrency":
            return Optional.ofNullable(clazz.cast(maxConcurrency()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("type", TYPE_FIELD);
        map.put("maxConcurrency", MAX_CONCURRENCY_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<FlowAliasConcurrencyConfiguration, T> g) {
        return obj -> g.apply((FlowAliasConcurrencyConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, FlowAliasConcurrencyConfiguration> {
        /**
         * <p>
         * The type of concurrency to use for parallel node execution. Specify one of the following options:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Automatic</code> - Amazon Bedrock determines which nodes can be executed in parallel based on the flow
         * definition and its dependencies.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Manual</code> - You specify which nodes can be executed in parallel.
         * </p>
         * </li>
         * </ul>
         * 
         * @param type
         *        The type of concurrency to use for parallel node execution. Specify one of the following options:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Automatic</code> - Amazon Bedrock determines which nodes can be executed in parallel based on
         *        the flow definition and its dependencies.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Manual</code> - You specify which nodes can be executed in parallel.
         *        </p>
         *        </li>
         * @see ConcurrencyType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ConcurrencyType
         */
        Builder type(String type);

        /**
         * <p>
         * The type of concurrency to use for parallel node execution. Specify one of the following options:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Automatic</code> - Amazon Bedrock determines which nodes can be executed in parallel based on the flow
         * definition and its dependencies.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Manual</code> - You specify which nodes can be executed in parallel.
         * </p>
         * </li>
         * </ul>
         * 
         * @param type
         *        The type of concurrency to use for parallel node execution. Specify one of the following options:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Automatic</code> - Amazon Bedrock determines which nodes can be executed in parallel based on
         *        the flow definition and its dependencies.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Manual</code> - You specify which nodes can be executed in parallel.
         *        </p>
         *        </li>
         * @see ConcurrencyType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ConcurrencyType
         */
        Builder type(ConcurrencyType type);

        /**
         * <p>
         * The maximum number of nodes that can be executed concurrently in the flow.
         * </p>
         * 
         * @param maxConcurrency
         *        The maximum number of nodes that can be executed concurrently in the flow.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxConcurrency(Integer maxConcurrency);
    }

    static final class BuilderImpl implements Builder {
        private String type;

        private Integer maxConcurrency;

        private BuilderImpl() {
        }

        private BuilderImpl(FlowAliasConcurrencyConfiguration model) {
            type(model.type);
            maxConcurrency(model.maxConcurrency);
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(ConcurrencyType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final Integer getMaxConcurrency() {
            return maxConcurrency;
        }

        public final void setMaxConcurrency(Integer maxConcurrency) {
            this.maxConcurrency = maxConcurrency;
        }

        @Override
        public final Builder maxConcurrency(Integer maxConcurrency) {
            this.maxConcurrency = maxConcurrency;
            return this;
        }

        @Override
        public FlowAliasConcurrencyConfiguration build() {
            return new FlowAliasConcurrencyConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
