/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.bedrockagent.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains details about a data source.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DataSource implements SdkPojo, Serializable, ToCopyableBuilder<DataSource.Builder, DataSource> {
    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("createdAt")
            .getter(getter(DataSource::createdAt))
            .setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<String> DATA_DELETION_POLICY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("dataDeletionPolicy").getter(getter(DataSource::dataDeletionPolicyAsString))
            .setter(setter(Builder::dataDeletionPolicy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("dataDeletionPolicy").build())
            .build();

    private static final SdkField<DataSourceConfiguration> DATA_SOURCE_CONFIGURATION_FIELD = SdkField
            .<DataSourceConfiguration> builder(MarshallingType.SDK_POJO).memberName("dataSourceConfiguration")
            .getter(getter(DataSource::dataSourceConfiguration)).setter(setter(Builder::dataSourceConfiguration))
            .constructor(DataSourceConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("dataSourceConfiguration").build())
            .build();

    private static final SdkField<String> DATA_SOURCE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("dataSourceId").getter(getter(DataSource::dataSourceId)).setter(setter(Builder::dataSourceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("dataSourceId").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("description").getter(getter(DataSource::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("description").build()).build();

    private static final SdkField<List<String>> FAILURE_REASONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("failureReasons")
            .getter(getter(DataSource::failureReasons))
            .setter(setter(Builder::failureReasons))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("failureReasons").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> KNOWLEDGE_BASE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("knowledgeBaseId").getter(getter(DataSource::knowledgeBaseId)).setter(setter(Builder::knowledgeBaseId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("knowledgeBaseId").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(DataSource::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<ServerSideEncryptionConfiguration> SERVER_SIDE_ENCRYPTION_CONFIGURATION_FIELD = SdkField
            .<ServerSideEncryptionConfiguration> builder(MarshallingType.SDK_POJO)
            .memberName("serverSideEncryptionConfiguration")
            .getter(getter(DataSource::serverSideEncryptionConfiguration))
            .setter(setter(Builder::serverSideEncryptionConfiguration))
            .constructor(ServerSideEncryptionConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("serverSideEncryptionConfiguration")
                    .build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("status")
            .getter(getter(DataSource::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<Instant> UPDATED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("updatedAt")
            .getter(getter(DataSource::updatedAt))
            .setter(setter(Builder::updatedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("updatedAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<VectorIngestionConfiguration> VECTOR_INGESTION_CONFIGURATION_FIELD = SdkField
            .<VectorIngestionConfiguration> builder(MarshallingType.SDK_POJO)
            .memberName("vectorIngestionConfiguration")
            .getter(getter(DataSource::vectorIngestionConfiguration))
            .setter(setter(Builder::vectorIngestionConfiguration))
            .constructor(VectorIngestionConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("vectorIngestionConfiguration")
                    .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CREATED_AT_FIELD,
            DATA_DELETION_POLICY_FIELD, DATA_SOURCE_CONFIGURATION_FIELD, DATA_SOURCE_ID_FIELD, DESCRIPTION_FIELD,
            FAILURE_REASONS_FIELD, KNOWLEDGE_BASE_ID_FIELD, NAME_FIELD, SERVER_SIDE_ENCRYPTION_CONFIGURATION_FIELD, STATUS_FIELD,
            UPDATED_AT_FIELD, VECTOR_INGESTION_CONFIGURATION_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final Instant createdAt;

    private final String dataDeletionPolicy;

    private final DataSourceConfiguration dataSourceConfiguration;

    private final String dataSourceId;

    private final String description;

    private final List<String> failureReasons;

    private final String knowledgeBaseId;

    private final String name;

    private final ServerSideEncryptionConfiguration serverSideEncryptionConfiguration;

    private final String status;

    private final Instant updatedAt;

    private final VectorIngestionConfiguration vectorIngestionConfiguration;

    private DataSource(BuilderImpl builder) {
        this.createdAt = builder.createdAt;
        this.dataDeletionPolicy = builder.dataDeletionPolicy;
        this.dataSourceConfiguration = builder.dataSourceConfiguration;
        this.dataSourceId = builder.dataSourceId;
        this.description = builder.description;
        this.failureReasons = builder.failureReasons;
        this.knowledgeBaseId = builder.knowledgeBaseId;
        this.name = builder.name;
        this.serverSideEncryptionConfiguration = builder.serverSideEncryptionConfiguration;
        this.status = builder.status;
        this.updatedAt = builder.updatedAt;
        this.vectorIngestionConfiguration = builder.vectorIngestionConfiguration;
    }

    /**
     * <p>
     * The time at which the data source was created.
     * </p>
     * 
     * @return The time at which the data source was created.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The data deletion policy for the data source.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #dataDeletionPolicy} will return {@link DataDeletionPolicy#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #dataDeletionPolicyAsString}.
     * </p>
     * 
     * @return The data deletion policy for the data source.
     * @see DataDeletionPolicy
     */
    public final DataDeletionPolicy dataDeletionPolicy() {
        return DataDeletionPolicy.fromValue(dataDeletionPolicy);
    }

    /**
     * <p>
     * The data deletion policy for the data source.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #dataDeletionPolicy} will return {@link DataDeletionPolicy#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #dataDeletionPolicyAsString}.
     * </p>
     * 
     * @return The data deletion policy for the data source.
     * @see DataDeletionPolicy
     */
    public final String dataDeletionPolicyAsString() {
        return dataDeletionPolicy;
    }

    /**
     * <p>
     * The connection configuration for the data source.
     * </p>
     * 
     * @return The connection configuration for the data source.
     */
    public final DataSourceConfiguration dataSourceConfiguration() {
        return dataSourceConfiguration;
    }

    /**
     * <p>
     * The unique identifier of the data source.
     * </p>
     * 
     * @return The unique identifier of the data source.
     */
    public final String dataSourceId() {
        return dataSourceId;
    }

    /**
     * <p>
     * The description of the data source.
     * </p>
     * 
     * @return The description of the data source.
     */
    public final String description() {
        return description;
    }

    /**
     * For responses, this returns true if the service returned a value for the FailureReasons property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasFailureReasons() {
        return failureReasons != null && !(failureReasons instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The detailed reasons on the failure to delete a data source.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFailureReasons} method.
     * </p>
     * 
     * @return The detailed reasons on the failure to delete a data source.
     */
    public final List<String> failureReasons() {
        return failureReasons;
    }

    /**
     * <p>
     * The unique identifier of the knowledge base to which the data source belongs.
     * </p>
     * 
     * @return The unique identifier of the knowledge base to which the data source belongs.
     */
    public final String knowledgeBaseId() {
        return knowledgeBaseId;
    }

    /**
     * <p>
     * The name of the data source.
     * </p>
     * 
     * @return The name of the data source.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * Contains details about the configuration of the server-side encryption.
     * </p>
     * 
     * @return Contains details about the configuration of the server-side encryption.
     */
    public final ServerSideEncryptionConfiguration serverSideEncryptionConfiguration() {
        return serverSideEncryptionConfiguration;
    }

    /**
     * <p>
     * The status of the data source. The following statuses are possible:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Available – The data source has been created and is ready for ingestion into the knowledge base.
     * </p>
     * </li>
     * <li>
     * <p>
     * Deleting – The data source is being deleted.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link DataSourceStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the data source. The following statuses are possible:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Available – The data source has been created and is ready for ingestion into the knowledge base.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Deleting – The data source is being deleted.
     *         </p>
     *         </li>
     * @see DataSourceStatus
     */
    public final DataSourceStatus status() {
        return DataSourceStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the data source. The following statuses are possible:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Available – The data source has been created and is ready for ingestion into the knowledge base.
     * </p>
     * </li>
     * <li>
     * <p>
     * Deleting – The data source is being deleted.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link DataSourceStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the data source. The following statuses are possible:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Available – The data source has been created and is ready for ingestion into the knowledge base.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Deleting – The data source is being deleted.
     *         </p>
     *         </li>
     * @see DataSourceStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The time at which the data source was last updated.
     * </p>
     * 
     * @return The time at which the data source was last updated.
     */
    public final Instant updatedAt() {
        return updatedAt;
    }

    /**
     * <p>
     * Contains details about how to ingest the documents in the data source.
     * </p>
     * 
     * @return Contains details about how to ingest the documents in the data source.
     */
    public final VectorIngestionConfiguration vectorIngestionConfiguration() {
        return vectorIngestionConfiguration;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(dataDeletionPolicyAsString());
        hashCode = 31 * hashCode + Objects.hashCode(dataSourceConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(dataSourceId());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(hasFailureReasons() ? failureReasons() : null);
        hashCode = 31 * hashCode + Objects.hashCode(knowledgeBaseId());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(serverSideEncryptionConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(updatedAt());
        hashCode = 31 * hashCode + Objects.hashCode(vectorIngestionConfiguration());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DataSource)) {
            return false;
        }
        DataSource other = (DataSource) obj;
        return Objects.equals(createdAt(), other.createdAt())
                && Objects.equals(dataDeletionPolicyAsString(), other.dataDeletionPolicyAsString())
                && Objects.equals(dataSourceConfiguration(), other.dataSourceConfiguration())
                && Objects.equals(dataSourceId(), other.dataSourceId()) && Objects.equals(description(), other.description())
                && hasFailureReasons() == other.hasFailureReasons() && Objects.equals(failureReasons(), other.failureReasons())
                && Objects.equals(knowledgeBaseId(), other.knowledgeBaseId()) && Objects.equals(name(), other.name())
                && Objects.equals(serverSideEncryptionConfiguration(), other.serverSideEncryptionConfiguration())
                && Objects.equals(statusAsString(), other.statusAsString()) && Objects.equals(updatedAt(), other.updatedAt())
                && Objects.equals(vectorIngestionConfiguration(), other.vectorIngestionConfiguration());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DataSource").add("CreatedAt", createdAt())
                .add("DataDeletionPolicy", dataDeletionPolicyAsString())
                .add("DataSourceConfiguration", dataSourceConfiguration()).add("DataSourceId", dataSourceId())
                .add("Description", description()).add("FailureReasons", hasFailureReasons() ? failureReasons() : null)
                .add("KnowledgeBaseId", knowledgeBaseId()).add("Name", name())
                .add("ServerSideEncryptionConfiguration", serverSideEncryptionConfiguration()).add("Status", statusAsString())
                .add("UpdatedAt", updatedAt()).add("VectorIngestionConfiguration", vectorIngestionConfiguration()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "dataDeletionPolicy":
            return Optional.ofNullable(clazz.cast(dataDeletionPolicyAsString()));
        case "dataSourceConfiguration":
            return Optional.ofNullable(clazz.cast(dataSourceConfiguration()));
        case "dataSourceId":
            return Optional.ofNullable(clazz.cast(dataSourceId()));
        case "description":
            return Optional.ofNullable(clazz.cast(description()));
        case "failureReasons":
            return Optional.ofNullable(clazz.cast(failureReasons()));
        case "knowledgeBaseId":
            return Optional.ofNullable(clazz.cast(knowledgeBaseId()));
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "serverSideEncryptionConfiguration":
            return Optional.ofNullable(clazz.cast(serverSideEncryptionConfiguration()));
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "updatedAt":
            return Optional.ofNullable(clazz.cast(updatedAt()));
        case "vectorIngestionConfiguration":
            return Optional.ofNullable(clazz.cast(vectorIngestionConfiguration()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("createdAt", CREATED_AT_FIELD);
        map.put("dataDeletionPolicy", DATA_DELETION_POLICY_FIELD);
        map.put("dataSourceConfiguration", DATA_SOURCE_CONFIGURATION_FIELD);
        map.put("dataSourceId", DATA_SOURCE_ID_FIELD);
        map.put("description", DESCRIPTION_FIELD);
        map.put("failureReasons", FAILURE_REASONS_FIELD);
        map.put("knowledgeBaseId", KNOWLEDGE_BASE_ID_FIELD);
        map.put("name", NAME_FIELD);
        map.put("serverSideEncryptionConfiguration", SERVER_SIDE_ENCRYPTION_CONFIGURATION_FIELD);
        map.put("status", STATUS_FIELD);
        map.put("updatedAt", UPDATED_AT_FIELD);
        map.put("vectorIngestionConfiguration", VECTOR_INGESTION_CONFIGURATION_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<DataSource, T> g) {
        return obj -> g.apply((DataSource) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DataSource> {
        /**
         * <p>
         * The time at which the data source was created.
         * </p>
         * 
         * @param createdAt
         *        The time at which the data source was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The data deletion policy for the data source.
         * </p>
         * 
         * @param dataDeletionPolicy
         *        The data deletion policy for the data source.
         * @see DataDeletionPolicy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DataDeletionPolicy
         */
        Builder dataDeletionPolicy(String dataDeletionPolicy);

        /**
         * <p>
         * The data deletion policy for the data source.
         * </p>
         * 
         * @param dataDeletionPolicy
         *        The data deletion policy for the data source.
         * @see DataDeletionPolicy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DataDeletionPolicy
         */
        Builder dataDeletionPolicy(DataDeletionPolicy dataDeletionPolicy);

        /**
         * <p>
         * The connection configuration for the data source.
         * </p>
         * 
         * @param dataSourceConfiguration
         *        The connection configuration for the data source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataSourceConfiguration(DataSourceConfiguration dataSourceConfiguration);

        /**
         * <p>
         * The connection configuration for the data source.
         * </p>
         * This is a convenience method that creates an instance of the {@link DataSourceConfiguration.Builder} avoiding
         * the need to create one manually via {@link DataSourceConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DataSourceConfiguration.Builder#build()} is called immediately
         * and its result is passed to {@link #dataSourceConfiguration(DataSourceConfiguration)}.
         * 
         * @param dataSourceConfiguration
         *        a consumer that will call methods on {@link DataSourceConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #dataSourceConfiguration(DataSourceConfiguration)
         */
        default Builder dataSourceConfiguration(Consumer<DataSourceConfiguration.Builder> dataSourceConfiguration) {
            return dataSourceConfiguration(DataSourceConfiguration.builder().applyMutation(dataSourceConfiguration).build());
        }

        /**
         * <p>
         * The unique identifier of the data source.
         * </p>
         * 
         * @param dataSourceId
         *        The unique identifier of the data source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataSourceId(String dataSourceId);

        /**
         * <p>
         * The description of the data source.
         * </p>
         * 
         * @param description
         *        The description of the data source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The detailed reasons on the failure to delete a data source.
         * </p>
         * 
         * @param failureReasons
         *        The detailed reasons on the failure to delete a data source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failureReasons(Collection<String> failureReasons);

        /**
         * <p>
         * The detailed reasons on the failure to delete a data source.
         * </p>
         * 
         * @param failureReasons
         *        The detailed reasons on the failure to delete a data source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failureReasons(String... failureReasons);

        /**
         * <p>
         * The unique identifier of the knowledge base to which the data source belongs.
         * </p>
         * 
         * @param knowledgeBaseId
         *        The unique identifier of the knowledge base to which the data source belongs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder knowledgeBaseId(String knowledgeBaseId);

        /**
         * <p>
         * The name of the data source.
         * </p>
         * 
         * @param name
         *        The name of the data source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * Contains details about the configuration of the server-side encryption.
         * </p>
         * 
         * @param serverSideEncryptionConfiguration
         *        Contains details about the configuration of the server-side encryption.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serverSideEncryptionConfiguration(ServerSideEncryptionConfiguration serverSideEncryptionConfiguration);

        /**
         * <p>
         * Contains details about the configuration of the server-side encryption.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link ServerSideEncryptionConfiguration.Builder} avoiding the need to create one manually via
         * {@link ServerSideEncryptionConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ServerSideEncryptionConfiguration.Builder#build()} is called
         * immediately and its result is passed to
         * {@link #serverSideEncryptionConfiguration(ServerSideEncryptionConfiguration)}.
         * 
         * @param serverSideEncryptionConfiguration
         *        a consumer that will call methods on {@link ServerSideEncryptionConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #serverSideEncryptionConfiguration(ServerSideEncryptionConfiguration)
         */
        default Builder serverSideEncryptionConfiguration(
                Consumer<ServerSideEncryptionConfiguration.Builder> serverSideEncryptionConfiguration) {
            return serverSideEncryptionConfiguration(ServerSideEncryptionConfiguration.builder()
                    .applyMutation(serverSideEncryptionConfiguration).build());
        }

        /**
         * <p>
         * The status of the data source. The following statuses are possible:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Available – The data source has been created and is ready for ingestion into the knowledge base.
         * </p>
         * </li>
         * <li>
         * <p>
         * Deleting – The data source is being deleted.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The status of the data source. The following statuses are possible:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        Available – The data source has been created and is ready for ingestion into the knowledge base.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Deleting – The data source is being deleted.
         *        </p>
         *        </li>
         * @see DataSourceStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DataSourceStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the data source. The following statuses are possible:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Available – The data source has been created and is ready for ingestion into the knowledge base.
         * </p>
         * </li>
         * <li>
         * <p>
         * Deleting – The data source is being deleted.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The status of the data source. The following statuses are possible:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        Available – The data source has been created and is ready for ingestion into the knowledge base.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Deleting – The data source is being deleted.
         *        </p>
         *        </li>
         * @see DataSourceStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DataSourceStatus
         */
        Builder status(DataSourceStatus status);

        /**
         * <p>
         * The time at which the data source was last updated.
         * </p>
         * 
         * @param updatedAt
         *        The time at which the data source was last updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder updatedAt(Instant updatedAt);

        /**
         * <p>
         * Contains details about how to ingest the documents in the data source.
         * </p>
         * 
         * @param vectorIngestionConfiguration
         *        Contains details about how to ingest the documents in the data source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vectorIngestionConfiguration(VectorIngestionConfiguration vectorIngestionConfiguration);

        /**
         * <p>
         * Contains details about how to ingest the documents in the data source.
         * </p>
         * This is a convenience method that creates an instance of the {@link VectorIngestionConfiguration.Builder}
         * avoiding the need to create one manually via {@link VectorIngestionConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link VectorIngestionConfiguration.Builder#build()} is called
         * immediately and its result is passed to {@link #vectorIngestionConfiguration(VectorIngestionConfiguration)}.
         * 
         * @param vectorIngestionConfiguration
         *        a consumer that will call methods on {@link VectorIngestionConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #vectorIngestionConfiguration(VectorIngestionConfiguration)
         */
        default Builder vectorIngestionConfiguration(Consumer<VectorIngestionConfiguration.Builder> vectorIngestionConfiguration) {
            return vectorIngestionConfiguration(VectorIngestionConfiguration.builder()
                    .applyMutation(vectorIngestionConfiguration).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private Instant createdAt;

        private String dataDeletionPolicy;

        private DataSourceConfiguration dataSourceConfiguration;

        private String dataSourceId;

        private String description;

        private List<String> failureReasons = DefaultSdkAutoConstructList.getInstance();

        private String knowledgeBaseId;

        private String name;

        private ServerSideEncryptionConfiguration serverSideEncryptionConfiguration;

        private String status;

        private Instant updatedAt;

        private VectorIngestionConfiguration vectorIngestionConfiguration;

        private BuilderImpl() {
        }

        private BuilderImpl(DataSource model) {
            createdAt(model.createdAt);
            dataDeletionPolicy(model.dataDeletionPolicy);
            dataSourceConfiguration(model.dataSourceConfiguration);
            dataSourceId(model.dataSourceId);
            description(model.description);
            failureReasons(model.failureReasons);
            knowledgeBaseId(model.knowledgeBaseId);
            name(model.name);
            serverSideEncryptionConfiguration(model.serverSideEncryptionConfiguration);
            status(model.status);
            updatedAt(model.updatedAt);
            vectorIngestionConfiguration(model.vectorIngestionConfiguration);
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final String getDataDeletionPolicy() {
            return dataDeletionPolicy;
        }

        public final void setDataDeletionPolicy(String dataDeletionPolicy) {
            this.dataDeletionPolicy = dataDeletionPolicy;
        }

        @Override
        public final Builder dataDeletionPolicy(String dataDeletionPolicy) {
            this.dataDeletionPolicy = dataDeletionPolicy;
            return this;
        }

        @Override
        public final Builder dataDeletionPolicy(DataDeletionPolicy dataDeletionPolicy) {
            this.dataDeletionPolicy(dataDeletionPolicy == null ? null : dataDeletionPolicy.toString());
            return this;
        }

        public final DataSourceConfiguration.Builder getDataSourceConfiguration() {
            return dataSourceConfiguration != null ? dataSourceConfiguration.toBuilder() : null;
        }

        public final void setDataSourceConfiguration(DataSourceConfiguration.BuilderImpl dataSourceConfiguration) {
            this.dataSourceConfiguration = dataSourceConfiguration != null ? dataSourceConfiguration.build() : null;
        }

        @Override
        public final Builder dataSourceConfiguration(DataSourceConfiguration dataSourceConfiguration) {
            this.dataSourceConfiguration = dataSourceConfiguration;
            return this;
        }

        public final String getDataSourceId() {
            return dataSourceId;
        }

        public final void setDataSourceId(String dataSourceId) {
            this.dataSourceId = dataSourceId;
        }

        @Override
        public final Builder dataSourceId(String dataSourceId) {
            this.dataSourceId = dataSourceId;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final Collection<String> getFailureReasons() {
            if (failureReasons instanceof SdkAutoConstructList) {
                return null;
            }
            return failureReasons;
        }

        public final void setFailureReasons(Collection<String> failureReasons) {
            this.failureReasons = FailureReasonsCopier.copy(failureReasons);
        }

        @Override
        public final Builder failureReasons(Collection<String> failureReasons) {
            this.failureReasons = FailureReasonsCopier.copy(failureReasons);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder failureReasons(String... failureReasons) {
            failureReasons(Arrays.asList(failureReasons));
            return this;
        }

        public final String getKnowledgeBaseId() {
            return knowledgeBaseId;
        }

        public final void setKnowledgeBaseId(String knowledgeBaseId) {
            this.knowledgeBaseId = knowledgeBaseId;
        }

        @Override
        public final Builder knowledgeBaseId(String knowledgeBaseId) {
            this.knowledgeBaseId = knowledgeBaseId;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final ServerSideEncryptionConfiguration.Builder getServerSideEncryptionConfiguration() {
            return serverSideEncryptionConfiguration != null ? serverSideEncryptionConfiguration.toBuilder() : null;
        }

        public final void setServerSideEncryptionConfiguration(
                ServerSideEncryptionConfiguration.BuilderImpl serverSideEncryptionConfiguration) {
            this.serverSideEncryptionConfiguration = serverSideEncryptionConfiguration != null ? serverSideEncryptionConfiguration
                    .build() : null;
        }

        @Override
        public final Builder serverSideEncryptionConfiguration(ServerSideEncryptionConfiguration serverSideEncryptionConfiguration) {
            this.serverSideEncryptionConfiguration = serverSideEncryptionConfiguration;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(DataSourceStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final Instant getUpdatedAt() {
            return updatedAt;
        }

        public final void setUpdatedAt(Instant updatedAt) {
            this.updatedAt = updatedAt;
        }

        @Override
        public final Builder updatedAt(Instant updatedAt) {
            this.updatedAt = updatedAt;
            return this;
        }

        public final VectorIngestionConfiguration.Builder getVectorIngestionConfiguration() {
            return vectorIngestionConfiguration != null ? vectorIngestionConfiguration.toBuilder() : null;
        }

        public final void setVectorIngestionConfiguration(VectorIngestionConfiguration.BuilderImpl vectorIngestionConfiguration) {
            this.vectorIngestionConfiguration = vectorIngestionConfiguration != null ? vectorIngestionConfiguration.build()
                    : null;
        }

        @Override
        public final Builder vectorIngestionConfiguration(VectorIngestionConfiguration vectorIngestionConfiguration) {
            this.vectorIngestionConfiguration = vectorIngestionConfiguration;
            return this;
        }

        @Override
        public DataSource build() {
            return new DataSource(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
