/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.bedrock.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The configuration details for returning the results from the knowledge base vector search.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class KnowledgeBaseVectorSearchConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<KnowledgeBaseVectorSearchConfiguration.Builder, KnowledgeBaseVectorSearchConfiguration> {
    private static final SdkField<Integer> NUMBER_OF_RESULTS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("numberOfResults").getter(getter(KnowledgeBaseVectorSearchConfiguration::numberOfResults))
            .setter(setter(Builder::numberOfResults))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("numberOfResults").build()).build();

    private static final SdkField<String> OVERRIDE_SEARCH_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("overrideSearchType").getter(getter(KnowledgeBaseVectorSearchConfiguration::overrideSearchTypeAsString))
            .setter(setter(Builder::overrideSearchType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("overrideSearchType").build())
            .build();

    private static final SdkField<RetrievalFilter> FILTER_FIELD = SdkField.<RetrievalFilter> builder(MarshallingType.SDK_POJO)
            .memberName("filter").getter(getter(KnowledgeBaseVectorSearchConfiguration::filter)).setter(setter(Builder::filter))
            .constructor(RetrievalFilter::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("filter").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NUMBER_OF_RESULTS_FIELD,
            OVERRIDE_SEARCH_TYPE_FIELD, FILTER_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final Integer numberOfResults;

    private final String overrideSearchType;

    private final RetrievalFilter filter;

    private KnowledgeBaseVectorSearchConfiguration(BuilderImpl builder) {
        this.numberOfResults = builder.numberOfResults;
        this.overrideSearchType = builder.overrideSearchType;
        this.filter = builder.filter;
    }

    /**
     * <p>
     * The number of text chunks to retrieve; the number of results to return.
     * </p>
     * 
     * @return The number of text chunks to retrieve; the number of results to return.
     */
    public final Integer numberOfResults() {
        return numberOfResults;
    }

    /**
     * <p>
     * By default, Amazon Bedrock decides a search strategy for you. If you're using an Amazon OpenSearch Serverless
     * vector store that contains a filterable text field, you can specify whether to query the knowledge base with a
     * <code>HYBRID</code> search using both vector embeddings and raw text, or <code>SEMANTIC</code> search using only
     * vector embeddings. For other vector store configurations, only <code>SEMANTIC</code> search is available.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #overrideSearchType} will return {@link SearchType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the
     * service is available from {@link #overrideSearchTypeAsString}.
     * </p>
     * 
     * @return By default, Amazon Bedrock decides a search strategy for you. If you're using an Amazon OpenSearch
     *         Serverless vector store that contains a filterable text field, you can specify whether to query the
     *         knowledge base with a <code>HYBRID</code> search using both vector embeddings and raw text, or
     *         <code>SEMANTIC</code> search using only vector embeddings. For other vector store configurations, only
     *         <code>SEMANTIC</code> search is available.
     * @see SearchType
     */
    public final SearchType overrideSearchType() {
        return SearchType.fromValue(overrideSearchType);
    }

    /**
     * <p>
     * By default, Amazon Bedrock decides a search strategy for you. If you're using an Amazon OpenSearch Serverless
     * vector store that contains a filterable text field, you can specify whether to query the knowledge base with a
     * <code>HYBRID</code> search using both vector embeddings and raw text, or <code>SEMANTIC</code> search using only
     * vector embeddings. For other vector store configurations, only <code>SEMANTIC</code> search is available.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #overrideSearchType} will return {@link SearchType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the
     * service is available from {@link #overrideSearchTypeAsString}.
     * </p>
     * 
     * @return By default, Amazon Bedrock decides a search strategy for you. If you're using an Amazon OpenSearch
     *         Serverless vector store that contains a filterable text field, you can specify whether to query the
     *         knowledge base with a <code>HYBRID</code> search using both vector embeddings and raw text, or
     *         <code>SEMANTIC</code> search using only vector embeddings. For other vector store configurations, only
     *         <code>SEMANTIC</code> search is available.
     * @see SearchType
     */
    public final String overrideSearchTypeAsString() {
        return overrideSearchType;
    }

    /**
     * <p>
     * Specifies the filters to use on the metadata fields in the knowledge base data sources before returning results.
     * </p>
     * 
     * @return Specifies the filters to use on the metadata fields in the knowledge base data sources before returning
     *         results.
     */
    public final RetrievalFilter filter() {
        return filter;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(numberOfResults());
        hashCode = 31 * hashCode + Objects.hashCode(overrideSearchTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(filter());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof KnowledgeBaseVectorSearchConfiguration)) {
            return false;
        }
        KnowledgeBaseVectorSearchConfiguration other = (KnowledgeBaseVectorSearchConfiguration) obj;
        return Objects.equals(numberOfResults(), other.numberOfResults())
                && Objects.equals(overrideSearchTypeAsString(), other.overrideSearchTypeAsString())
                && Objects.equals(filter(), other.filter());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("KnowledgeBaseVectorSearchConfiguration").add("NumberOfResults", numberOfResults())
                .add("OverrideSearchType", overrideSearchTypeAsString())
                .add("Filter", filter() == null ? null : "*** Sensitive Data Redacted ***").build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "numberOfResults":
            return Optional.ofNullable(clazz.cast(numberOfResults()));
        case "overrideSearchType":
            return Optional.ofNullable(clazz.cast(overrideSearchTypeAsString()));
        case "filter":
            return Optional.ofNullable(clazz.cast(filter()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("numberOfResults", NUMBER_OF_RESULTS_FIELD);
        map.put("overrideSearchType", OVERRIDE_SEARCH_TYPE_FIELD);
        map.put("filter", FILTER_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<KnowledgeBaseVectorSearchConfiguration, T> g) {
        return obj -> g.apply((KnowledgeBaseVectorSearchConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, KnowledgeBaseVectorSearchConfiguration> {
        /**
         * <p>
         * The number of text chunks to retrieve; the number of results to return.
         * </p>
         * 
         * @param numberOfResults
         *        The number of text chunks to retrieve; the number of results to return.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder numberOfResults(Integer numberOfResults);

        /**
         * <p>
         * By default, Amazon Bedrock decides a search strategy for you. If you're using an Amazon OpenSearch Serverless
         * vector store that contains a filterable text field, you can specify whether to query the knowledge base with
         * a <code>HYBRID</code> search using both vector embeddings and raw text, or <code>SEMANTIC</code> search using
         * only vector embeddings. For other vector store configurations, only <code>SEMANTIC</code> search is
         * available.
         * </p>
         * 
         * @param overrideSearchType
         *        By default, Amazon Bedrock decides a search strategy for you. If you're using an Amazon OpenSearch
         *        Serverless vector store that contains a filterable text field, you can specify whether to query the
         *        knowledge base with a <code>HYBRID</code> search using both vector embeddings and raw text, or
         *        <code>SEMANTIC</code> search using only vector embeddings. For other vector store configurations, only
         *        <code>SEMANTIC</code> search is available.
         * @see SearchType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SearchType
         */
        Builder overrideSearchType(String overrideSearchType);

        /**
         * <p>
         * By default, Amazon Bedrock decides a search strategy for you. If you're using an Amazon OpenSearch Serverless
         * vector store that contains a filterable text field, you can specify whether to query the knowledge base with
         * a <code>HYBRID</code> search using both vector embeddings and raw text, or <code>SEMANTIC</code> search using
         * only vector embeddings. For other vector store configurations, only <code>SEMANTIC</code> search is
         * available.
         * </p>
         * 
         * @param overrideSearchType
         *        By default, Amazon Bedrock decides a search strategy for you. If you're using an Amazon OpenSearch
         *        Serverless vector store that contains a filterable text field, you can specify whether to query the
         *        knowledge base with a <code>HYBRID</code> search using both vector embeddings and raw text, or
         *        <code>SEMANTIC</code> search using only vector embeddings. For other vector store configurations, only
         *        <code>SEMANTIC</code> search is available.
         * @see SearchType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SearchType
         */
        Builder overrideSearchType(SearchType overrideSearchType);

        /**
         * <p>
         * Specifies the filters to use on the metadata fields in the knowledge base data sources before returning
         * results.
         * </p>
         * 
         * @param filter
         *        Specifies the filters to use on the metadata fields in the knowledge base data sources before
         *        returning results.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder filter(RetrievalFilter filter);

        /**
         * <p>
         * Specifies the filters to use on the metadata fields in the knowledge base data sources before returning
         * results.
         * </p>
         * This is a convenience method that creates an instance of the {@link RetrievalFilter.Builder} avoiding the
         * need to create one manually via {@link RetrievalFilter#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RetrievalFilter.Builder#build()} is called immediately and its
         * result is passed to {@link #filter(RetrievalFilter)}.
         * 
         * @param filter
         *        a consumer that will call methods on {@link RetrievalFilter.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #filter(RetrievalFilter)
         */
        default Builder filter(Consumer<RetrievalFilter.Builder> filter) {
            return filter(RetrievalFilter.builder().applyMutation(filter).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private Integer numberOfResults;

        private String overrideSearchType;

        private RetrievalFilter filter;

        private BuilderImpl() {
        }

        private BuilderImpl(KnowledgeBaseVectorSearchConfiguration model) {
            numberOfResults(model.numberOfResults);
            overrideSearchType(model.overrideSearchType);
            filter(model.filter);
        }

        public final Integer getNumberOfResults() {
            return numberOfResults;
        }

        public final void setNumberOfResults(Integer numberOfResults) {
            this.numberOfResults = numberOfResults;
        }

        @Override
        public final Builder numberOfResults(Integer numberOfResults) {
            this.numberOfResults = numberOfResults;
            return this;
        }

        public final String getOverrideSearchType() {
            return overrideSearchType;
        }

        public final void setOverrideSearchType(String overrideSearchType) {
            this.overrideSearchType = overrideSearchType;
        }

        @Override
        public final Builder overrideSearchType(String overrideSearchType) {
            this.overrideSearchType = overrideSearchType;
            return this;
        }

        @Override
        public final Builder overrideSearchType(SearchType overrideSearchType) {
            this.overrideSearchType(overrideSearchType == null ? null : overrideSearchType.toString());
            return this;
        }

        public final RetrievalFilter.Builder getFilter() {
            return filter != null ? filter.toBuilder() : null;
        }

        public final void setFilter(RetrievalFilter.BuilderImpl filter) {
            this.filter = filter != null ? filter.build() : null;
        }

        @Override
        public final Builder filter(RetrievalFilter filter) {
            this.filter = filter;
            return this;
        }

        @Override
        public KnowledgeBaseVectorSearchConfiguration build() {
            return new KnowledgeBaseVectorSearchConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
