/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.bedrock;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.bedrock.internal.BedrockServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.bedrock.model.AccessDeniedException;
import software.amazon.awssdk.services.bedrock.model.BedrockException;
import software.amazon.awssdk.services.bedrock.model.ConflictException;
import software.amazon.awssdk.services.bedrock.model.CreateModelCustomizationJobRequest;
import software.amazon.awssdk.services.bedrock.model.CreateModelCustomizationJobResponse;
import software.amazon.awssdk.services.bedrock.model.CreateProvisionedModelThroughputRequest;
import software.amazon.awssdk.services.bedrock.model.CreateProvisionedModelThroughputResponse;
import software.amazon.awssdk.services.bedrock.model.DeleteCustomModelRequest;
import software.amazon.awssdk.services.bedrock.model.DeleteCustomModelResponse;
import software.amazon.awssdk.services.bedrock.model.DeleteModelInvocationLoggingConfigurationRequest;
import software.amazon.awssdk.services.bedrock.model.DeleteModelInvocationLoggingConfigurationResponse;
import software.amazon.awssdk.services.bedrock.model.DeleteProvisionedModelThroughputRequest;
import software.amazon.awssdk.services.bedrock.model.DeleteProvisionedModelThroughputResponse;
import software.amazon.awssdk.services.bedrock.model.GetCustomModelRequest;
import software.amazon.awssdk.services.bedrock.model.GetCustomModelResponse;
import software.amazon.awssdk.services.bedrock.model.GetFoundationModelRequest;
import software.amazon.awssdk.services.bedrock.model.GetFoundationModelResponse;
import software.amazon.awssdk.services.bedrock.model.GetModelCustomizationJobRequest;
import software.amazon.awssdk.services.bedrock.model.GetModelCustomizationJobResponse;
import software.amazon.awssdk.services.bedrock.model.GetModelInvocationLoggingConfigurationRequest;
import software.amazon.awssdk.services.bedrock.model.GetModelInvocationLoggingConfigurationResponse;
import software.amazon.awssdk.services.bedrock.model.GetProvisionedModelThroughputRequest;
import software.amazon.awssdk.services.bedrock.model.GetProvisionedModelThroughputResponse;
import software.amazon.awssdk.services.bedrock.model.InternalServerException;
import software.amazon.awssdk.services.bedrock.model.ListCustomModelsRequest;
import software.amazon.awssdk.services.bedrock.model.ListCustomModelsResponse;
import software.amazon.awssdk.services.bedrock.model.ListFoundationModelsRequest;
import software.amazon.awssdk.services.bedrock.model.ListFoundationModelsResponse;
import software.amazon.awssdk.services.bedrock.model.ListModelCustomizationJobsRequest;
import software.amazon.awssdk.services.bedrock.model.ListModelCustomizationJobsResponse;
import software.amazon.awssdk.services.bedrock.model.ListProvisionedModelThroughputsRequest;
import software.amazon.awssdk.services.bedrock.model.ListProvisionedModelThroughputsResponse;
import software.amazon.awssdk.services.bedrock.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.bedrock.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.bedrock.model.PutModelInvocationLoggingConfigurationRequest;
import software.amazon.awssdk.services.bedrock.model.PutModelInvocationLoggingConfigurationResponse;
import software.amazon.awssdk.services.bedrock.model.ResourceNotFoundException;
import software.amazon.awssdk.services.bedrock.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.bedrock.model.StopModelCustomizationJobRequest;
import software.amazon.awssdk.services.bedrock.model.StopModelCustomizationJobResponse;
import software.amazon.awssdk.services.bedrock.model.TagResourceRequest;
import software.amazon.awssdk.services.bedrock.model.TagResourceResponse;
import software.amazon.awssdk.services.bedrock.model.ThrottlingException;
import software.amazon.awssdk.services.bedrock.model.TooManyTagsException;
import software.amazon.awssdk.services.bedrock.model.UntagResourceRequest;
import software.amazon.awssdk.services.bedrock.model.UntagResourceResponse;
import software.amazon.awssdk.services.bedrock.model.UpdateProvisionedModelThroughputRequest;
import software.amazon.awssdk.services.bedrock.model.UpdateProvisionedModelThroughputResponse;
import software.amazon.awssdk.services.bedrock.model.ValidationException;
import software.amazon.awssdk.services.bedrock.transform.CreateModelCustomizationJobRequestMarshaller;
import software.amazon.awssdk.services.bedrock.transform.CreateProvisionedModelThroughputRequestMarshaller;
import software.amazon.awssdk.services.bedrock.transform.DeleteCustomModelRequestMarshaller;
import software.amazon.awssdk.services.bedrock.transform.DeleteModelInvocationLoggingConfigurationRequestMarshaller;
import software.amazon.awssdk.services.bedrock.transform.DeleteProvisionedModelThroughputRequestMarshaller;
import software.amazon.awssdk.services.bedrock.transform.GetCustomModelRequestMarshaller;
import software.amazon.awssdk.services.bedrock.transform.GetFoundationModelRequestMarshaller;
import software.amazon.awssdk.services.bedrock.transform.GetModelCustomizationJobRequestMarshaller;
import software.amazon.awssdk.services.bedrock.transform.GetModelInvocationLoggingConfigurationRequestMarshaller;
import software.amazon.awssdk.services.bedrock.transform.GetProvisionedModelThroughputRequestMarshaller;
import software.amazon.awssdk.services.bedrock.transform.ListCustomModelsRequestMarshaller;
import software.amazon.awssdk.services.bedrock.transform.ListFoundationModelsRequestMarshaller;
import software.amazon.awssdk.services.bedrock.transform.ListModelCustomizationJobsRequestMarshaller;
import software.amazon.awssdk.services.bedrock.transform.ListProvisionedModelThroughputsRequestMarshaller;
import software.amazon.awssdk.services.bedrock.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.bedrock.transform.PutModelInvocationLoggingConfigurationRequestMarshaller;
import software.amazon.awssdk.services.bedrock.transform.StopModelCustomizationJobRequestMarshaller;
import software.amazon.awssdk.services.bedrock.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.bedrock.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.bedrock.transform.UpdateProvisionedModelThroughputRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link BedrockAsyncClient}.
 *
 * @see BedrockAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultBedrockAsyncClient implements BedrockAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultBedrockAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultBedrockAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a fine-tuning job to customize a base model.
     * </p>
     * <p>
     * You specify the base foundation model and the location of the training data. After the model-customization job
     * completes successfully, your custom model resource will be ready to use. Training data contains input and output
     * text for each record in a JSONL format. Optionally, you can specify validation data in the same format as the
     * training data. Amazon Bedrock returns validation loss metrics and output generations after the job completes.
     * </p>
     * <p>
     * Model-customization jobs are asynchronous and the completion time depends on the base model and the
     * training/validation data size. To monitor a job, use the <code>GetModelCustomizationJob</code> operation to
     * retrieve the job status.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/custom-models.html">Custom models</a> in the Bedrock
     * User Guide.
     * </p>
     *
     * @param createModelCustomizationJobRequest
     * @return A Java Future containing the result of the CreateModelCustomizationJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource ARN was not found. Check the ARN and try your
     *         request again.</li>
     *         <li>AccessDeniedException The request is denied because of missing access permissions.</li>
     *         <li>ValidationException Input validation failed. Check your request parameters and retry the request.</li>
     *         <li>ConflictException Error occurred because of a conflict while performing an operation.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>TooManyTagsException The request contains more tags than can be associated with a resource (50 tags
     *         per resource). The maximum number of tags includes both existing tags and those included in your current
     *         request.</li>
     *         <li>ServiceQuotaExceededException The number of requests exceeds the service quota. Resubmit your request
     *         later.</li>
     *         <li>ThrottlingException The number of requests exceeds the limit. Resubmit your request later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BedrockAsyncClient.CreateModelCustomizationJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-2023-04-20/CreateModelCustomizationJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateModelCustomizationJobResponse> createModelCustomizationJob(
            CreateModelCustomizationJobRequest createModelCustomizationJobRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createModelCustomizationJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createModelCustomizationJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateModelCustomizationJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateModelCustomizationJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateModelCustomizationJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateModelCustomizationJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateModelCustomizationJobRequest, CreateModelCustomizationJobResponse>()
                            .withOperationName("CreateModelCustomizationJob").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateModelCustomizationJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createModelCustomizationJobRequest));
            CompletableFuture<CreateModelCustomizationJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a provisioned throughput with dedicated capacity for a foundation model or a fine-tuned model.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/what-is-service.html">Provisioned throughput</a> in
     * the Bedrock User Guide.
     * </p>
     *
     * @param createProvisionedModelThroughputRequest
     * @return A Java Future containing the result of the CreateProvisionedModelThroughput operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource ARN was not found. Check the ARN and try your
     *         request again.</li>
     *         <li>AccessDeniedException The request is denied because of missing access permissions.</li>
     *         <li>ValidationException Input validation failed. Check your request parameters and retry the request.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>TooManyTagsException The request contains more tags than can be associated with a resource (50 tags
     *         per resource). The maximum number of tags includes both existing tags and those included in your current
     *         request.</li>
     *         <li>ServiceQuotaExceededException The number of requests exceeds the service quota. Resubmit your request
     *         later.</li>
     *         <li>ThrottlingException The number of requests exceeds the limit. Resubmit your request later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BedrockAsyncClient.CreateProvisionedModelThroughput
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-2023-04-20/CreateProvisionedModelThroughput"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateProvisionedModelThroughputResponse> createProvisionedModelThroughput(
            CreateProvisionedModelThroughputRequest createProvisionedModelThroughputRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createProvisionedModelThroughputRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createProvisionedModelThroughputRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateProvisionedModelThroughput");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateProvisionedModelThroughputResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, CreateProvisionedModelThroughputResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateProvisionedModelThroughputResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateProvisionedModelThroughputRequest, CreateProvisionedModelThroughputResponse>()
                            .withOperationName("CreateProvisionedModelThroughput").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateProvisionedModelThroughputRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createProvisionedModelThroughputRequest));
            CompletableFuture<CreateProvisionedModelThroughputResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a custom model that you created earlier. For more information, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/custom-models.html">Custom models</a> in the Bedrock
     * User Guide.
     * </p>
     *
     * @param deleteCustomModelRequest
     * @return A Java Future containing the result of the DeleteCustomModel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource ARN was not found. Check the ARN and try your
     *         request again.</li>
     *         <li>AccessDeniedException The request is denied because of missing access permissions.</li>
     *         <li>ValidationException Input validation failed. Check your request parameters and retry the request.</li>
     *         <li>ConflictException Error occurred because of a conflict while performing an operation.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ThrottlingException The number of requests exceeds the limit. Resubmit your request later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BedrockAsyncClient.DeleteCustomModel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-2023-04-20/DeleteCustomModel" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteCustomModelResponse> deleteCustomModel(DeleteCustomModelRequest deleteCustomModelRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteCustomModelRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteCustomModelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteCustomModel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteCustomModelResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteCustomModelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteCustomModelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteCustomModelRequest, DeleteCustomModelResponse>()
                            .withOperationName("DeleteCustomModel").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteCustomModelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteCustomModelRequest));
            CompletableFuture<DeleteCustomModelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Delete the invocation logging.
     * </p>
     *
     * @param deleteModelInvocationLoggingConfigurationRequest
     * @return A Java Future containing the result of the DeleteModelInvocationLoggingConfiguration operation returned
     *         by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The request is denied because of missing access permissions.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ThrottlingException The number of requests exceeds the limit. Resubmit your request later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BedrockAsyncClient.DeleteModelInvocationLoggingConfiguration
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-2023-04-20/DeleteModelInvocationLoggingConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteModelInvocationLoggingConfigurationResponse> deleteModelInvocationLoggingConfiguration(
            DeleteModelInvocationLoggingConfigurationRequest deleteModelInvocationLoggingConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(
                deleteModelInvocationLoggingConfigurationRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteModelInvocationLoggingConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteModelInvocationLoggingConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteModelInvocationLoggingConfigurationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DeleteModelInvocationLoggingConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteModelInvocationLoggingConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteModelInvocationLoggingConfigurationRequest, DeleteModelInvocationLoggingConfigurationResponse>()
                            .withOperationName("DeleteModelInvocationLoggingConfiguration")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteModelInvocationLoggingConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteModelInvocationLoggingConfigurationRequest));
            CompletableFuture<DeleteModelInvocationLoggingConfigurationResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a provisioned throughput. For more information, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/what-is-service.html">Provisioned throughput</a> in
     * the Bedrock User Guide.
     * </p>
     *
     * @param deleteProvisionedModelThroughputRequest
     * @return A Java Future containing the result of the DeleteProvisionedModelThroughput operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource ARN was not found. Check the ARN and try your
     *         request again.</li>
     *         <li>AccessDeniedException The request is denied because of missing access permissions.</li>
     *         <li>ValidationException Input validation failed. Check your request parameters and retry the request.</li>
     *         <li>ConflictException Error occurred because of a conflict while performing an operation.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ThrottlingException The number of requests exceeds the limit. Resubmit your request later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BedrockAsyncClient.DeleteProvisionedModelThroughput
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-2023-04-20/DeleteProvisionedModelThroughput"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteProvisionedModelThroughputResponse> deleteProvisionedModelThroughput(
            DeleteProvisionedModelThroughputRequest deleteProvisionedModelThroughputRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteProvisionedModelThroughputRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteProvisionedModelThroughputRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteProvisionedModelThroughput");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteProvisionedModelThroughputResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DeleteProvisionedModelThroughputResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteProvisionedModelThroughputResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteProvisionedModelThroughputRequest, DeleteProvisionedModelThroughputResponse>()
                            .withOperationName("DeleteProvisionedModelThroughput").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteProvisionedModelThroughputRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteProvisionedModelThroughputRequest));
            CompletableFuture<DeleteProvisionedModelThroughputResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Get the properties associated with a Amazon Bedrock custom model that you have created.For more information, see
     * <a href="https://docs.aws.amazon.com/bedrock/latest/userguide/custom-models.html">Custom models</a> in the
     * Bedrock User Guide.
     * </p>
     *
     * @param getCustomModelRequest
     * @return A Java Future containing the result of the GetCustomModel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource ARN was not found. Check the ARN and try your
     *         request again.</li>
     *         <li>AccessDeniedException The request is denied because of missing access permissions.</li>
     *         <li>ValidationException Input validation failed. Check your request parameters and retry the request.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ThrottlingException The number of requests exceeds the limit. Resubmit your request later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BedrockAsyncClient.GetCustomModel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-2023-04-20/GetCustomModel" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetCustomModelResponse> getCustomModel(GetCustomModelRequest getCustomModelRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getCustomModelRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getCustomModelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetCustomModel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetCustomModelResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetCustomModelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetCustomModelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetCustomModelRequest, GetCustomModelResponse>()
                            .withOperationName("GetCustomModel").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetCustomModelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getCustomModelRequest));
            CompletableFuture<GetCustomModelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Get details about a Amazon Bedrock foundation model.
     * </p>
     *
     * @param getFoundationModelRequest
     * @return A Java Future containing the result of the GetFoundationModel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource ARN was not found. Check the ARN and try your
     *         request again.</li>
     *         <li>AccessDeniedException The request is denied because of missing access permissions.</li>
     *         <li>ValidationException Input validation failed. Check your request parameters and retry the request.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ThrottlingException The number of requests exceeds the limit. Resubmit your request later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BedrockAsyncClient.GetFoundationModel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-2023-04-20/GetFoundationModel" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetFoundationModelResponse> getFoundationModel(GetFoundationModelRequest getFoundationModelRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getFoundationModelRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getFoundationModelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetFoundationModel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetFoundationModelResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetFoundationModelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetFoundationModelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetFoundationModelRequest, GetFoundationModelResponse>()
                            .withOperationName("GetFoundationModel").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetFoundationModelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getFoundationModelRequest));
            CompletableFuture<GetFoundationModelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the properties associated with a model-customization job, including the status of the job. For more
     * information, see <a href="https://docs.aws.amazon.com/bedrock/latest/userguide/custom-models.html">Custom
     * models</a> in the Bedrock User Guide.
     * </p>
     *
     * @param getModelCustomizationJobRequest
     * @return A Java Future containing the result of the GetModelCustomizationJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource ARN was not found. Check the ARN and try your
     *         request again.</li>
     *         <li>AccessDeniedException The request is denied because of missing access permissions.</li>
     *         <li>ValidationException Input validation failed. Check your request parameters and retry the request.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ThrottlingException The number of requests exceeds the limit. Resubmit your request later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BedrockAsyncClient.GetModelCustomizationJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-2023-04-20/GetModelCustomizationJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetModelCustomizationJobResponse> getModelCustomizationJob(
            GetModelCustomizationJobRequest getModelCustomizationJobRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getModelCustomizationJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getModelCustomizationJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetModelCustomizationJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetModelCustomizationJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetModelCustomizationJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetModelCustomizationJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetModelCustomizationJobRequest, GetModelCustomizationJobResponse>()
                            .withOperationName("GetModelCustomizationJob").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetModelCustomizationJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getModelCustomizationJobRequest));
            CompletableFuture<GetModelCustomizationJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Get the current configuration values for model invocation logging.
     * </p>
     *
     * @param getModelInvocationLoggingConfigurationRequest
     * @return A Java Future containing the result of the GetModelInvocationLoggingConfiguration operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The request is denied because of missing access permissions.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ThrottlingException The number of requests exceeds the limit. Resubmit your request later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BedrockAsyncClient.GetModelInvocationLoggingConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-2023-04-20/GetModelInvocationLoggingConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetModelInvocationLoggingConfigurationResponse> getModelInvocationLoggingConfiguration(
            GetModelInvocationLoggingConfigurationRequest getModelInvocationLoggingConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getModelInvocationLoggingConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getModelInvocationLoggingConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetModelInvocationLoggingConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetModelInvocationLoggingConfigurationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, GetModelInvocationLoggingConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetModelInvocationLoggingConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetModelInvocationLoggingConfigurationRequest, GetModelInvocationLoggingConfigurationResponse>()
                            .withOperationName("GetModelInvocationLoggingConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetModelInvocationLoggingConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getModelInvocationLoggingConfigurationRequest));
            CompletableFuture<GetModelInvocationLoggingConfigurationResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Get details for a provisioned throughput. For more information, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/what-is-service.html">Provisioned throughput</a> in
     * the Bedrock User Guide.
     * </p>
     *
     * @param getProvisionedModelThroughputRequest
     * @return A Java Future containing the result of the GetProvisionedModelThroughput operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource ARN was not found. Check the ARN and try your
     *         request again.</li>
     *         <li>AccessDeniedException The request is denied because of missing access permissions.</li>
     *         <li>ValidationException Input validation failed. Check your request parameters and retry the request.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ThrottlingException The number of requests exceeds the limit. Resubmit your request later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BedrockAsyncClient.GetProvisionedModelThroughput
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-2023-04-20/GetProvisionedModelThroughput"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetProvisionedModelThroughputResponse> getProvisionedModelThroughput(
            GetProvisionedModelThroughputRequest getProvisionedModelThroughputRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getProvisionedModelThroughputRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getProvisionedModelThroughputRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetProvisionedModelThroughput");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetProvisionedModelThroughputResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetProvisionedModelThroughputResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetProvisionedModelThroughputResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetProvisionedModelThroughputRequest, GetProvisionedModelThroughputResponse>()
                            .withOperationName("GetProvisionedModelThroughput").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetProvisionedModelThroughputRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getProvisionedModelThroughputRequest));
            CompletableFuture<GetProvisionedModelThroughputResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of the custom models that you have created with the <code>CreateModelCustomizationJob</code>
     * operation.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/custom-models.html">Custom models</a> in the Bedrock
     * User Guide.
     * </p>
     *
     * @param listCustomModelsRequest
     * @return A Java Future containing the result of the ListCustomModels operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The request is denied because of missing access permissions.</li>
     *         <li>ValidationException Input validation failed. Check your request parameters and retry the request.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ThrottlingException The number of requests exceeds the limit. Resubmit your request later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BedrockAsyncClient.ListCustomModels
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-2023-04-20/ListCustomModels" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListCustomModelsResponse> listCustomModels(ListCustomModelsRequest listCustomModelsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listCustomModelsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listCustomModelsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListCustomModels");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListCustomModelsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListCustomModelsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListCustomModelsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListCustomModelsRequest, ListCustomModelsResponse>()
                            .withOperationName("ListCustomModels").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListCustomModelsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listCustomModelsRequest));
            CompletableFuture<ListCustomModelsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List of Amazon Bedrock foundation models that you can use. For more information, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/foundation-models.html">Foundation models</a> in the
     * Bedrock User Guide.
     * </p>
     *
     * @param listFoundationModelsRequest
     * @return A Java Future containing the result of the ListFoundationModels operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The request is denied because of missing access permissions.</li>
     *         <li>ValidationException Input validation failed. Check your request parameters and retry the request.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ThrottlingException The number of requests exceeds the limit. Resubmit your request later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BedrockAsyncClient.ListFoundationModels
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-2023-04-20/ListFoundationModels" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListFoundationModelsResponse> listFoundationModels(
            ListFoundationModelsRequest listFoundationModelsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listFoundationModelsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listFoundationModelsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListFoundationModels");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListFoundationModelsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListFoundationModelsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListFoundationModelsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListFoundationModelsRequest, ListFoundationModelsResponse>()
                            .withOperationName("ListFoundationModels").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListFoundationModelsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listFoundationModelsRequest));
            CompletableFuture<ListFoundationModelsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of model customization jobs that you have submitted. You can filter the jobs to return based on
     * one or more criteria.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/custom-models.html">Custom models</a> in the Bedrock
     * User Guide.
     * </p>
     *
     * @param listModelCustomizationJobsRequest
     * @return A Java Future containing the result of the ListModelCustomizationJobs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The request is denied because of missing access permissions.</li>
     *         <li>ValidationException Input validation failed. Check your request parameters and retry the request.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ThrottlingException The number of requests exceeds the limit. Resubmit your request later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BedrockAsyncClient.ListModelCustomizationJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-2023-04-20/ListModelCustomizationJobs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListModelCustomizationJobsResponse> listModelCustomizationJobs(
            ListModelCustomizationJobsRequest listModelCustomizationJobsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listModelCustomizationJobsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listModelCustomizationJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListModelCustomizationJobs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListModelCustomizationJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListModelCustomizationJobsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListModelCustomizationJobsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListModelCustomizationJobsRequest, ListModelCustomizationJobsResponse>()
                            .withOperationName("ListModelCustomizationJobs").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListModelCustomizationJobsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listModelCustomizationJobsRequest));
            CompletableFuture<ListModelCustomizationJobsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List the provisioned capacities. For more information, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/what-is-service.html">Provisioned throughput</a> in
     * the Bedrock User Guide.
     * </p>
     *
     * @param listProvisionedModelThroughputsRequest
     * @return A Java Future containing the result of the ListProvisionedModelThroughputs operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The request is denied because of missing access permissions.</li>
     *         <li>ValidationException Input validation failed. Check your request parameters and retry the request.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ThrottlingException The number of requests exceeds the limit. Resubmit your request later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BedrockAsyncClient.ListProvisionedModelThroughputs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-2023-04-20/ListProvisionedModelThroughputs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListProvisionedModelThroughputsResponse> listProvisionedModelThroughputs(
            ListProvisionedModelThroughputsRequest listProvisionedModelThroughputsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listProvisionedModelThroughputsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listProvisionedModelThroughputsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListProvisionedModelThroughputs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListProvisionedModelThroughputsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListProvisionedModelThroughputsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListProvisionedModelThroughputsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListProvisionedModelThroughputsRequest, ListProvisionedModelThroughputsResponse>()
                            .withOperationName("ListProvisionedModelThroughputs").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListProvisionedModelThroughputsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listProvisionedModelThroughputsRequest));
            CompletableFuture<ListProvisionedModelThroughputsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List the tags associated with the specified resource.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/what-is-service.html">Tagging resources</a> in the
     * Bedrock User Guide.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource ARN was not found. Check the ARN and try your
     *         request again.</li>
     *         <li>AccessDeniedException The request is denied because of missing access permissions.</li>
     *         <li>ValidationException Input validation failed. Check your request parameters and retry the request.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ThrottlingException The number of requests exceeds the limit. Resubmit your request later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BedrockAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-2023-04-20/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Set the configuration values for model invocation logging.
     * </p>
     *
     * @param putModelInvocationLoggingConfigurationRequest
     * @return A Java Future containing the result of the PutModelInvocationLoggingConfiguration operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The request is denied because of missing access permissions.</li>
     *         <li>ValidationException Input validation failed. Check your request parameters and retry the request.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ThrottlingException The number of requests exceeds the limit. Resubmit your request later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BedrockAsyncClient.PutModelInvocationLoggingConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-2023-04-20/PutModelInvocationLoggingConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutModelInvocationLoggingConfigurationResponse> putModelInvocationLoggingConfiguration(
            PutModelInvocationLoggingConfigurationRequest putModelInvocationLoggingConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putModelInvocationLoggingConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                putModelInvocationLoggingConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutModelInvocationLoggingConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutModelInvocationLoggingConfigurationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, PutModelInvocationLoggingConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutModelInvocationLoggingConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutModelInvocationLoggingConfigurationRequest, PutModelInvocationLoggingConfigurationResponse>()
                            .withOperationName("PutModelInvocationLoggingConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutModelInvocationLoggingConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putModelInvocationLoggingConfigurationRequest));
            CompletableFuture<PutModelInvocationLoggingConfigurationResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Stops an active model customization job. For more information, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/custom-models.html">Custom models</a> in the Bedrock
     * User Guide.
     * </p>
     *
     * @param stopModelCustomizationJobRequest
     * @return A Java Future containing the result of the StopModelCustomizationJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource ARN was not found. Check the ARN and try your
     *         request again.</li>
     *         <li>AccessDeniedException The request is denied because of missing access permissions.</li>
     *         <li>ValidationException Input validation failed. Check your request parameters and retry the request.</li>
     *         <li>ConflictException Error occurred because of a conflict while performing an operation.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ThrottlingException The number of requests exceeds the limit. Resubmit your request later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BedrockAsyncClient.StopModelCustomizationJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-2023-04-20/StopModelCustomizationJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StopModelCustomizationJobResponse> stopModelCustomizationJob(
            StopModelCustomizationJobRequest stopModelCustomizationJobRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(stopModelCustomizationJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopModelCustomizationJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopModelCustomizationJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopModelCustomizationJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StopModelCustomizationJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StopModelCustomizationJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopModelCustomizationJobRequest, StopModelCustomizationJobResponse>()
                            .withOperationName("StopModelCustomizationJob").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StopModelCustomizationJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(stopModelCustomizationJobRequest));
            CompletableFuture<StopModelCustomizationJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Associate tags with a resource. For more information, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/what-is-service.html">Tagging resources</a> in the
     * Bedrock User Guide.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource ARN was not found. Check the ARN and try your
     *         request again.</li>
     *         <li>AccessDeniedException The request is denied because of missing access permissions.</li>
     *         <li>ValidationException Input validation failed. Check your request parameters and retry the request.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>TooManyTagsException The request contains more tags than can be associated with a resource (50 tags
     *         per resource). The maximum number of tags includes both existing tags and those included in your current
     *         request.</li>
     *         <li>ThrottlingException The number of requests exceeds the limit. Resubmit your request later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BedrockAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-2023-04-20/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Remove one or more tags from a resource. For more information, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/what-is-service.html">Tagging resources</a> in the
     * Bedrock User Guide.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource ARN was not found. Check the ARN and try your
     *         request again.</li>
     *         <li>AccessDeniedException The request is denied because of missing access permissions.</li>
     *         <li>ValidationException Input validation failed. Check your request parameters and retry the request.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ThrottlingException The number of requests exceeds the limit. Resubmit your request later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BedrockAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-2023-04-20/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Update a provisioned throughput. For more information, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/what-is-service.html">Provisioned throughput</a> in
     * the Bedrock User Guide.
     * </p>
     *
     * @param updateProvisionedModelThroughputRequest
     * @return A Java Future containing the result of the UpdateProvisionedModelThroughput operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource ARN was not found. Check the ARN and try your
     *         request again.</li>
     *         <li>AccessDeniedException The request is denied because of missing access permissions.</li>
     *         <li>ValidationException Input validation failed. Check your request parameters and retry the request.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ThrottlingException The number of requests exceeds the limit. Resubmit your request later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BedrockAsyncClient.UpdateProvisionedModelThroughput
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-2023-04-20/UpdateProvisionedModelThroughput"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateProvisionedModelThroughputResponse> updateProvisionedModelThroughput(
            UpdateProvisionedModelThroughputRequest updateProvisionedModelThroughputRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateProvisionedModelThroughputRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateProvisionedModelThroughputRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateProvisionedModelThroughput");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateProvisionedModelThroughputResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, UpdateProvisionedModelThroughputResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateProvisionedModelThroughputResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateProvisionedModelThroughputRequest, UpdateProvisionedModelThroughputResponse>()
                            .withOperationName("UpdateProvisionedModelThroughput").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateProvisionedModelThroughputRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateProvisionedModelThroughputRequest));
            CompletableFuture<UpdateProvisionedModelThroughputResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final BedrockServiceClientConfiguration serviceClientConfiguration() {
        return new BedrockServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(BedrockException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyTagsException")
                                .exceptionBuilderSupplier(TooManyTagsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        BedrockServiceClientConfigurationBuilder serviceConfigBuilder = new BedrockServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
