/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.bcmdashboards;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.bcmdashboards.internal.BcmDashboardsServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.bcmdashboards.internal.ServiceVersionInfo;
import software.amazon.awssdk.services.bcmdashboards.model.AccessDeniedException;
import software.amazon.awssdk.services.bcmdashboards.model.BcmDashboardsException;
import software.amazon.awssdk.services.bcmdashboards.model.CreateDashboardRequest;
import software.amazon.awssdk.services.bcmdashboards.model.CreateDashboardResponse;
import software.amazon.awssdk.services.bcmdashboards.model.DeleteDashboardRequest;
import software.amazon.awssdk.services.bcmdashboards.model.DeleteDashboardResponse;
import software.amazon.awssdk.services.bcmdashboards.model.GetDashboardRequest;
import software.amazon.awssdk.services.bcmdashboards.model.GetDashboardResponse;
import software.amazon.awssdk.services.bcmdashboards.model.GetResourcePolicyRequest;
import software.amazon.awssdk.services.bcmdashboards.model.GetResourcePolicyResponse;
import software.amazon.awssdk.services.bcmdashboards.model.InternalServerException;
import software.amazon.awssdk.services.bcmdashboards.model.ListDashboardsRequest;
import software.amazon.awssdk.services.bcmdashboards.model.ListDashboardsResponse;
import software.amazon.awssdk.services.bcmdashboards.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.bcmdashboards.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.bcmdashboards.model.ResourceNotFoundException;
import software.amazon.awssdk.services.bcmdashboards.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.bcmdashboards.model.TagResourceRequest;
import software.amazon.awssdk.services.bcmdashboards.model.TagResourceResponse;
import software.amazon.awssdk.services.bcmdashboards.model.ThrottlingException;
import software.amazon.awssdk.services.bcmdashboards.model.UntagResourceRequest;
import software.amazon.awssdk.services.bcmdashboards.model.UntagResourceResponse;
import software.amazon.awssdk.services.bcmdashboards.model.UpdateDashboardRequest;
import software.amazon.awssdk.services.bcmdashboards.model.UpdateDashboardResponse;
import software.amazon.awssdk.services.bcmdashboards.model.ValidationException;
import software.amazon.awssdk.services.bcmdashboards.transform.CreateDashboardRequestMarshaller;
import software.amazon.awssdk.services.bcmdashboards.transform.DeleteDashboardRequestMarshaller;
import software.amazon.awssdk.services.bcmdashboards.transform.GetDashboardRequestMarshaller;
import software.amazon.awssdk.services.bcmdashboards.transform.GetResourcePolicyRequestMarshaller;
import software.amazon.awssdk.services.bcmdashboards.transform.ListDashboardsRequestMarshaller;
import software.amazon.awssdk.services.bcmdashboards.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.bcmdashboards.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.bcmdashboards.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.bcmdashboards.transform.UpdateDashboardRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link BcmDashboardsClient}.
 *
 * @see BcmDashboardsClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultBcmDashboardsClient implements BcmDashboardsClient {
    private static final Logger log = Logger.loggerFor(DefaultBcmDashboardsClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultBcmDashboardsClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this)
                .option(SdkClientOption.API_METADATA, "BCM_Dashboards" + "#" + ServiceVersionInfo.VERSION).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a new dashboard that can contain multiple widgets displaying cost and usage data. You can add custom
     * widgets or use predefined widgets, arranging them in your preferred layout.
     * </p>
     *
     * @param createDashboardRequest
     * @return Result of the CreateDashboard operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Reduce the frequency of requests and use exponential
     *         backoff.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action. Verify your IAM permissions and any
     *         resource policies.
     * @throws InternalServerException
     *         An internal error occurred while processing the request. Retry your request. If the problem persists,
     *         contact Amazon Web Services Support.
     * @throws ValidationException
     *         The input parameters do not satisfy the requirements. Check the error message for specific validation
     *         details.
     * @throws ServiceQuotaExceededException
     *         The request would exceed service quotas. For example, attempting to create more than 20 widgets in a
     *         dashboard or exceeding the maximum number of dashboards per account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BcmDashboardsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BcmDashboardsClient.CreateDashboard
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bcm-dashboards-2025-08-18/CreateDashboard"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateDashboardResponse createDashboard(CreateDashboardRequest createDashboardRequest) throws ThrottlingException,
            AccessDeniedException, InternalServerException, ValidationException, ServiceQuotaExceededException,
            AwsServiceException, SdkClientException, BcmDashboardsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateDashboardResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateDashboardResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createDashboardRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDashboardRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "BCM Dashboards");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDashboard");

            return clientHandler.execute(new ClientExecutionParams<CreateDashboardRequest, CreateDashboardResponse>()
                    .withOperationName("CreateDashboard").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createDashboardRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateDashboardRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a specified dashboard. This action cannot be undone.
     * </p>
     *
     * @param deleteDashboardRequest
     * @return Result of the DeleteDashboard operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Reduce the frequency of requests and use exponential
     *         backoff.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action. Verify your IAM permissions and any
     *         resource policies.
     * @throws InternalServerException
     *         An internal error occurred while processing the request. Retry your request. If the problem persists,
     *         contact Amazon Web Services Support.
     * @throws ValidationException
     *         The input parameters do not satisfy the requirements. Check the error message for specific validation
     *         details.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BcmDashboardsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BcmDashboardsClient.DeleteDashboard
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bcm-dashboards-2025-08-18/DeleteDashboard"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteDashboardResponse deleteDashboard(DeleteDashboardRequest deleteDashboardRequest) throws ThrottlingException,
            AccessDeniedException, InternalServerException, ValidationException, AwsServiceException, SdkClientException,
            BcmDashboardsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteDashboardResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteDashboardResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteDashboardRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDashboardRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "BCM Dashboards");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDashboard");

            return clientHandler.execute(new ClientExecutionParams<DeleteDashboardRequest, DeleteDashboardResponse>()
                    .withOperationName("DeleteDashboard").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteDashboardRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteDashboardRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the configuration and metadata of a specified dashboard, including its widgets and layout settings.
     * </p>
     *
     * @param getDashboardRequest
     * @return Result of the GetDashboard operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Reduce the frequency of requests and use exponential
     *         backoff.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action. Verify your IAM permissions and any
     *         resource policies.
     * @throws InternalServerException
     *         An internal error occurred while processing the request. Retry your request. If the problem persists,
     *         contact Amazon Web Services Support.
     * @throws ValidationException
     *         The input parameters do not satisfy the requirements. Check the error message for specific validation
     *         details.
     * @throws ResourceNotFoundException
     *         The specified resource (dashboard, policy, or widget) was not found. Verify the ARN and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BcmDashboardsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BcmDashboardsClient.GetDashboard
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bcm-dashboards-2025-08-18/GetDashboard" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetDashboardResponse getDashboard(GetDashboardRequest getDashboardRequest) throws ThrottlingException,
            AccessDeniedException, InternalServerException, ValidationException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, BcmDashboardsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetDashboardResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetDashboardResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getDashboardRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDashboardRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "BCM Dashboards");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDashboard");

            return clientHandler.execute(new ClientExecutionParams<GetDashboardRequest, GetDashboardResponse>()
                    .withOperationName("GetDashboard").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getDashboardRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetDashboardRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the resource-based policy attached to a dashboard, showing sharing configurations and permissions.
     * </p>
     *
     * @param getResourcePolicyRequest
     * @return Result of the GetResourcePolicy operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Reduce the frequency of requests and use exponential
     *         backoff.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action. Verify your IAM permissions and any
     *         resource policies.
     * @throws InternalServerException
     *         An internal error occurred while processing the request. Retry your request. If the problem persists,
     *         contact Amazon Web Services Support.
     * @throws ValidationException
     *         The input parameters do not satisfy the requirements. Check the error message for specific validation
     *         details.
     * @throws ResourceNotFoundException
     *         The specified resource (dashboard, policy, or widget) was not found. Verify the ARN and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BcmDashboardsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BcmDashboardsClient.GetResourcePolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bcm-dashboards-2025-08-18/GetResourcePolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetResourcePolicyResponse getResourcePolicy(GetResourcePolicyRequest getResourcePolicyRequest)
            throws ThrottlingException, AccessDeniedException, InternalServerException, ValidationException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, BcmDashboardsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetResourcePolicyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetResourcePolicyResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getResourcePolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResourcePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "BCM Dashboards");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResourcePolicy");

            return clientHandler.execute(new ClientExecutionParams<GetResourcePolicyRequest, GetResourcePolicyResponse>()
                    .withOperationName("GetResourcePolicy").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getResourcePolicyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetResourcePolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of all dashboards in your account.
     * </p>
     *
     * @param listDashboardsRequest
     * @return Result of the ListDashboards operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Reduce the frequency of requests and use exponential
     *         backoff.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action. Verify your IAM permissions and any
     *         resource policies.
     * @throws InternalServerException
     *         An internal error occurred while processing the request. Retry your request. If the problem persists,
     *         contact Amazon Web Services Support.
     * @throws ValidationException
     *         The input parameters do not satisfy the requirements. Check the error message for specific validation
     *         details.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BcmDashboardsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BcmDashboardsClient.ListDashboards
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bcm-dashboards-2025-08-18/ListDashboards" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListDashboardsResponse listDashboards(ListDashboardsRequest listDashboardsRequest) throws ThrottlingException,
            AccessDeniedException, InternalServerException, ValidationException, AwsServiceException, SdkClientException,
            BcmDashboardsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListDashboardsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListDashboardsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listDashboardsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDashboardsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "BCM Dashboards");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDashboards");

            return clientHandler.execute(new ClientExecutionParams<ListDashboardsRequest, ListDashboardsResponse>()
                    .withOperationName("ListDashboards").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listDashboardsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListDashboardsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of all tags associated with a specified dashboard resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Reduce the frequency of requests and use exponential
     *         backoff.
     * @throws InternalServerException
     *         An internal error occurred while processing the request. Retry your request. If the problem persists,
     *         contact Amazon Web Services Support.
     * @throws ValidationException
     *         The input parameters do not satisfy the requirements. Check the error message for specific validation
     *         details.
     * @throws ResourceNotFoundException
     *         The specified resource (dashboard, policy, or widget) was not found. Verify the ARN and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BcmDashboardsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BcmDashboardsClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bcm-dashboards-2025-08-18/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ThrottlingException, InternalServerException, ValidationException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, BcmDashboardsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "BCM Dashboards");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds or updates tags for a specified dashboard resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Reduce the frequency of requests and use exponential
     *         backoff.
     * @throws InternalServerException
     *         An internal error occurred while processing the request. Retry your request. If the problem persists,
     *         contact Amazon Web Services Support.
     * @throws ValidationException
     *         The input parameters do not satisfy the requirements. Check the error message for specific validation
     *         details.
     * @throws ResourceNotFoundException
     *         The specified resource (dashboard, policy, or widget) was not found. Verify the ARN and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BcmDashboardsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BcmDashboardsClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bcm-dashboards-2025-08-18/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ThrottlingException,
            InternalServerException, ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            BcmDashboardsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "BCM Dashboards");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes specified tags from a dashboard resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Reduce the frequency of requests and use exponential
     *         backoff.
     * @throws InternalServerException
     *         An internal error occurred while processing the request. Retry your request. If the problem persists,
     *         contact Amazon Web Services Support.
     * @throws ValidationException
     *         The input parameters do not satisfy the requirements. Check the error message for specific validation
     *         details.
     * @throws ResourceNotFoundException
     *         The specified resource (dashboard, policy, or widget) was not found. Verify the ARN and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BcmDashboardsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BcmDashboardsClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bcm-dashboards-2025-08-18/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ThrottlingException,
            InternalServerException, ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            BcmDashboardsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "BCM Dashboards");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates an existing dashboard's properties, including its name, description, and widget configurations.
     * </p>
     *
     * @param updateDashboardRequest
     * @return Result of the UpdateDashboard operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Reduce the frequency of requests and use exponential
     *         backoff.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action. Verify your IAM permissions and any
     *         resource policies.
     * @throws InternalServerException
     *         An internal error occurred while processing the request. Retry your request. If the problem persists,
     *         contact Amazon Web Services Support.
     * @throws ValidationException
     *         The input parameters do not satisfy the requirements. Check the error message for specific validation
     *         details.
     * @throws ResourceNotFoundException
     *         The specified resource (dashboard, policy, or widget) was not found. Verify the ARN and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BcmDashboardsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BcmDashboardsClient.UpdateDashboard
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bcm-dashboards-2025-08-18/UpdateDashboard"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateDashboardResponse updateDashboard(UpdateDashboardRequest updateDashboardRequest) throws ThrottlingException,
            AccessDeniedException, InternalServerException, ValidationException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, BcmDashboardsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateDashboardResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateDashboardResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateDashboardRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateDashboardRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "BCM Dashboards");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDashboard");

            return clientHandler.execute(new ClientExecutionParams<UpdateDashboardRequest, UpdateDashboardResponse>()
                    .withOperationName("UpdateDashboard").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateDashboardRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateDashboardRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        BcmDashboardsServiceClientConfigurationBuilder serviceConfigBuilder = new BcmDashboardsServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration).defaultServiceExceptionSupplier(BcmDashboardsException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON).protocolVersion("1.0");
    }

    @Override
    public final BcmDashboardsServiceClientConfiguration serviceClientConfiguration() {
        return new BcmDashboardsServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
