/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.bcmdashboards;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.bcmdashboards.internal.BcmDashboardsServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.bcmdashboards.internal.ServiceVersionInfo;
import software.amazon.awssdk.services.bcmdashboards.model.AccessDeniedException;
import software.amazon.awssdk.services.bcmdashboards.model.BcmDashboardsException;
import software.amazon.awssdk.services.bcmdashboards.model.CreateDashboardRequest;
import software.amazon.awssdk.services.bcmdashboards.model.CreateDashboardResponse;
import software.amazon.awssdk.services.bcmdashboards.model.DeleteDashboardRequest;
import software.amazon.awssdk.services.bcmdashboards.model.DeleteDashboardResponse;
import software.amazon.awssdk.services.bcmdashboards.model.GetDashboardRequest;
import software.amazon.awssdk.services.bcmdashboards.model.GetDashboardResponse;
import software.amazon.awssdk.services.bcmdashboards.model.GetResourcePolicyRequest;
import software.amazon.awssdk.services.bcmdashboards.model.GetResourcePolicyResponse;
import software.amazon.awssdk.services.bcmdashboards.model.InternalServerException;
import software.amazon.awssdk.services.bcmdashboards.model.ListDashboardsRequest;
import software.amazon.awssdk.services.bcmdashboards.model.ListDashboardsResponse;
import software.amazon.awssdk.services.bcmdashboards.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.bcmdashboards.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.bcmdashboards.model.ResourceNotFoundException;
import software.amazon.awssdk.services.bcmdashboards.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.bcmdashboards.model.TagResourceRequest;
import software.amazon.awssdk.services.bcmdashboards.model.TagResourceResponse;
import software.amazon.awssdk.services.bcmdashboards.model.ThrottlingException;
import software.amazon.awssdk.services.bcmdashboards.model.UntagResourceRequest;
import software.amazon.awssdk.services.bcmdashboards.model.UntagResourceResponse;
import software.amazon.awssdk.services.bcmdashboards.model.UpdateDashboardRequest;
import software.amazon.awssdk.services.bcmdashboards.model.UpdateDashboardResponse;
import software.amazon.awssdk.services.bcmdashboards.model.ValidationException;
import software.amazon.awssdk.services.bcmdashboards.transform.CreateDashboardRequestMarshaller;
import software.amazon.awssdk.services.bcmdashboards.transform.DeleteDashboardRequestMarshaller;
import software.amazon.awssdk.services.bcmdashboards.transform.GetDashboardRequestMarshaller;
import software.amazon.awssdk.services.bcmdashboards.transform.GetResourcePolicyRequestMarshaller;
import software.amazon.awssdk.services.bcmdashboards.transform.ListDashboardsRequestMarshaller;
import software.amazon.awssdk.services.bcmdashboards.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.bcmdashboards.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.bcmdashboards.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.bcmdashboards.transform.UpdateDashboardRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link BcmDashboardsAsyncClient}.
 *
 * @see BcmDashboardsAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultBcmDashboardsAsyncClient implements BcmDashboardsAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultBcmDashboardsAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultBcmDashboardsAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this)
                .option(SdkClientOption.API_METADATA, "BCM_Dashboards" + "#" + ServiceVersionInfo.VERSION).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a new dashboard that can contain multiple widgets displaying cost and usage data. You can add custom
     * widgets or use predefined widgets, arranging them in your preferred layout.
     * </p>
     *
     * @param createDashboardRequest
     * @return A Java Future containing the result of the CreateDashboard operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to request throttling. Reduce the frequency of
     *         requests and use exponential backoff.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action. Verify your IAM
     *         permissions and any resource policies.</li>
     *         <li>InternalServerException An internal error occurred while processing the request. Retry your request.
     *         If the problem persists, contact Amazon Web Services Support.</li>
     *         <li>ValidationException The input parameters do not satisfy the requirements. Check the error message for
     *         specific validation details.</li>
     *         <li>ServiceQuotaExceededException The request would exceed service quotas. For example, attempting to
     *         create more than 20 widgets in a dashboard or exceeding the maximum number of dashboards per account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BcmDashboardsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BcmDashboardsAsyncClient.CreateDashboard
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bcm-dashboards-2025-08-18/CreateDashboard"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateDashboardResponse> createDashboard(CreateDashboardRequest createDashboardRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createDashboardRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDashboardRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "BCM Dashboards");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDashboard");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateDashboardResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateDashboardResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<CreateDashboardResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateDashboardRequest, CreateDashboardResponse>()
                            .withOperationName("CreateDashboard").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateDashboardRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createDashboardRequest));
            CompletableFuture<CreateDashboardResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a specified dashboard. This action cannot be undone.
     * </p>
     *
     * @param deleteDashboardRequest
     * @return A Java Future containing the result of the DeleteDashboard operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to request throttling. Reduce the frequency of
     *         requests and use exponential backoff.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action. Verify your IAM
     *         permissions and any resource policies.</li>
     *         <li>InternalServerException An internal error occurred while processing the request. Retry your request.
     *         If the problem persists, contact Amazon Web Services Support.</li>
     *         <li>ValidationException The input parameters do not satisfy the requirements. Check the error message for
     *         specific validation details.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BcmDashboardsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BcmDashboardsAsyncClient.DeleteDashboard
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bcm-dashboards-2025-08-18/DeleteDashboard"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteDashboardResponse> deleteDashboard(DeleteDashboardRequest deleteDashboardRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteDashboardRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDashboardRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "BCM Dashboards");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDashboard");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteDashboardResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteDashboardResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeleteDashboardResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteDashboardRequest, DeleteDashboardResponse>()
                            .withOperationName("DeleteDashboard").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteDashboardRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteDashboardRequest));
            CompletableFuture<DeleteDashboardResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the configuration and metadata of a specified dashboard, including its widgets and layout settings.
     * </p>
     *
     * @param getDashboardRequest
     * @return A Java Future containing the result of the GetDashboard operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to request throttling. Reduce the frequency of
     *         requests and use exponential backoff.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action. Verify your IAM
     *         permissions and any resource policies.</li>
     *         <li>InternalServerException An internal error occurred while processing the request. Retry your request.
     *         If the problem persists, contact Amazon Web Services Support.</li>
     *         <li>ValidationException The input parameters do not satisfy the requirements. Check the error message for
     *         specific validation details.</li>
     *         <li>ResourceNotFoundException The specified resource (dashboard, policy, or widget) was not found. Verify
     *         the ARN and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BcmDashboardsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BcmDashboardsAsyncClient.GetDashboard
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bcm-dashboards-2025-08-18/GetDashboard" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetDashboardResponse> getDashboard(GetDashboardRequest getDashboardRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getDashboardRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDashboardRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "BCM Dashboards");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDashboard");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDashboardResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetDashboardResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetDashboardResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDashboardRequest, GetDashboardResponse>()
                            .withOperationName("GetDashboard").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetDashboardRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getDashboardRequest));
            CompletableFuture<GetDashboardResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the resource-based policy attached to a dashboard, showing sharing configurations and permissions.
     * </p>
     *
     * @param getResourcePolicyRequest
     * @return A Java Future containing the result of the GetResourcePolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to request throttling. Reduce the frequency of
     *         requests and use exponential backoff.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action. Verify your IAM
     *         permissions and any resource policies.</li>
     *         <li>InternalServerException An internal error occurred while processing the request. Retry your request.
     *         If the problem persists, contact Amazon Web Services Support.</li>
     *         <li>ValidationException The input parameters do not satisfy the requirements. Check the error message for
     *         specific validation details.</li>
     *         <li>ResourceNotFoundException The specified resource (dashboard, policy, or widget) was not found. Verify
     *         the ARN and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BcmDashboardsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BcmDashboardsAsyncClient.GetResourcePolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bcm-dashboards-2025-08-18/GetResourcePolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetResourcePolicyResponse> getResourcePolicy(GetResourcePolicyRequest getResourcePolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getResourcePolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResourcePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "BCM Dashboards");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResourcePolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetResourcePolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetResourcePolicyResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetResourcePolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetResourcePolicyRequest, GetResourcePolicyResponse>()
                            .withOperationName("GetResourcePolicy").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetResourcePolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getResourcePolicyRequest));
            CompletableFuture<GetResourcePolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of all dashboards in your account.
     * </p>
     *
     * @param listDashboardsRequest
     * @return A Java Future containing the result of the ListDashboards operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to request throttling. Reduce the frequency of
     *         requests and use exponential backoff.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action. Verify your IAM
     *         permissions and any resource policies.</li>
     *         <li>InternalServerException An internal error occurred while processing the request. Retry your request.
     *         If the problem persists, contact Amazon Web Services Support.</li>
     *         <li>ValidationException The input parameters do not satisfy the requirements. Check the error message for
     *         specific validation details.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BcmDashboardsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BcmDashboardsAsyncClient.ListDashboards
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bcm-dashboards-2025-08-18/ListDashboards" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListDashboardsResponse> listDashboards(ListDashboardsRequest listDashboardsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listDashboardsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDashboardsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "BCM Dashboards");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDashboards");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDashboardsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListDashboardsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListDashboardsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDashboardsRequest, ListDashboardsResponse>()
                            .withOperationName("ListDashboards").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListDashboardsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listDashboardsRequest));
            CompletableFuture<ListDashboardsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of all tags associated with a specified dashboard resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to request throttling. Reduce the frequency of
     *         requests and use exponential backoff.</li>
     *         <li>InternalServerException An internal error occurred while processing the request. Retry your request.
     *         If the problem persists, contact Amazon Web Services Support.</li>
     *         <li>ValidationException The input parameters do not satisfy the requirements. Check the error message for
     *         specific validation details.</li>
     *         <li>ResourceNotFoundException The specified resource (dashboard, policy, or widget) was not found. Verify
     *         the ARN and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BcmDashboardsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BcmDashboardsAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bcm-dashboards-2025-08-18/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "BCM Dashboards");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds or updates tags for a specified dashboard resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to request throttling. Reduce the frequency of
     *         requests and use exponential backoff.</li>
     *         <li>InternalServerException An internal error occurred while processing the request. Retry your request.
     *         If the problem persists, contact Amazon Web Services Support.</li>
     *         <li>ValidationException The input parameters do not satisfy the requirements. Check the error message for
     *         specific validation details.</li>
     *         <li>ResourceNotFoundException The specified resource (dashboard, policy, or widget) was not found. Verify
     *         the ARN and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BcmDashboardsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BcmDashboardsAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bcm-dashboards-2025-08-18/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "BCM Dashboards");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes specified tags from a dashboard resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to request throttling. Reduce the frequency of
     *         requests and use exponential backoff.</li>
     *         <li>InternalServerException An internal error occurred while processing the request. Retry your request.
     *         If the problem persists, contact Amazon Web Services Support.</li>
     *         <li>ValidationException The input parameters do not satisfy the requirements. Check the error message for
     *         specific validation details.</li>
     *         <li>ResourceNotFoundException The specified resource (dashboard, policy, or widget) was not found. Verify
     *         the ARN and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BcmDashboardsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BcmDashboardsAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bcm-dashboards-2025-08-18/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "BCM Dashboards");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an existing dashboard's properties, including its name, description, and widget configurations.
     * </p>
     *
     * @param updateDashboardRequest
     * @return A Java Future containing the result of the UpdateDashboard operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to request throttling. Reduce the frequency of
     *         requests and use exponential backoff.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action. Verify your IAM
     *         permissions and any resource policies.</li>
     *         <li>InternalServerException An internal error occurred while processing the request. Retry your request.
     *         If the problem persists, contact Amazon Web Services Support.</li>
     *         <li>ValidationException The input parameters do not satisfy the requirements. Check the error message for
     *         specific validation details.</li>
     *         <li>ResourceNotFoundException The specified resource (dashboard, policy, or widget) was not found. Verify
     *         the ARN and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BcmDashboardsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BcmDashboardsAsyncClient.UpdateDashboard
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bcm-dashboards-2025-08-18/UpdateDashboard"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateDashboardResponse> updateDashboard(UpdateDashboardRequest updateDashboardRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateDashboardRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateDashboardRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "BCM Dashboards");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDashboard");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateDashboardResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateDashboardResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UpdateDashboardResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateDashboardRequest, UpdateDashboardResponse>()
                            .withOperationName("UpdateDashboard").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateDashboardRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateDashboardRequest));
            CompletableFuture<UpdateDashboardResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final BcmDashboardsServiceClientConfiguration serviceClientConfiguration() {
        return new BcmDashboardsServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration).defaultServiceExceptionSupplier(BcmDashboardsException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON).protocolVersion("1.0");
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        BcmDashboardsServiceClientConfigurationBuilder serviceConfigBuilder = new BcmDashboardsServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
