/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.batch.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides information used to select Amazon Machine Images (AMIs) for instances in the compute environment. If
 * <code>Ec2Configuration</code> isn't specified, the default is <code>ECS_AL2</code> (<a
 * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html#al2ami">Amazon Linux 2</a>).
 * </p>
 * <note>
 * <p>
 * This object isn't applicable to jobs that are running on Fargate resources.
 * </p>
 * </note>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Ec2Configuration implements SdkPojo, Serializable,
        ToCopyableBuilder<Ec2Configuration.Builder, Ec2Configuration> {
    private static final SdkField<String> IMAGE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("imageType").getter(getter(Ec2Configuration::imageType)).setter(setter(Builder::imageType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("imageType").build()).build();

    private static final SdkField<String> IMAGE_ID_OVERRIDE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("imageIdOverride").getter(getter(Ec2Configuration::imageIdOverride))
            .setter(setter(Builder::imageIdOverride))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("imageIdOverride").build()).build();

    private static final SdkField<String> IMAGE_KUBERNETES_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("imageKubernetesVersion").getter(getter(Ec2Configuration::imageKubernetesVersion))
            .setter(setter(Builder::imageKubernetesVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("imageKubernetesVersion").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(IMAGE_TYPE_FIELD,
            IMAGE_ID_OVERRIDE_FIELD, IMAGE_KUBERNETES_VERSION_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String imageType;

    private final String imageIdOverride;

    private final String imageKubernetesVersion;

    private Ec2Configuration(BuilderImpl builder) {
        this.imageType = builder.imageType;
        this.imageIdOverride = builder.imageIdOverride;
        this.imageKubernetesVersion = builder.imageKubernetesVersion;
    }

    /**
     * <p>
     * The image type to match with the instance type to select an AMI. The supported values are different for
     * <code>ECS</code> and <code>EKS</code> resources.
     * </p>
     * <dl>
     * <dt>ECS</dt>
     * <dd>
     * <p>
     * If the <code>imageIdOverride</code> parameter isn't specified, then a recent <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html#al2ami">Amazon
     * ECS-optimized Amazon Linux 2 AMI</a> (<code>ECS_AL2</code>) is used. If a new image type is specified in an
     * update, but neither an <code>imageId</code> nor a <code>imageIdOverride</code> parameter is specified, then the
     * latest Amazon ECS optimized AMI for that image type that's supported by Batch is used.
     * </p>
     * <important>
     * <p>
     * Amazon Web Services will end support for Amazon ECS optimized AL2-optimized and AL2-accelerated AMIs. Starting in
     * January 2026, Batch will change the default AMI for new Amazon ECS compute environments from Amazon Linux 2 to
     * Amazon Linux 2023. We recommend migrating Batch Amazon ECS compute environments to Amazon Linux 2023 to maintain
     * optimal performance and security. For more information on upgrading from AL2 to AL2023, see <a
     * href="https://docs.aws.amazon.com/batch/latest/userguide/ecs-migration-2023.html">How to migrate from ECS AL2 to
     * ECS AL2023</a> in the <i>Batch User Guide</i>.
     * </p>
     * </important>
     * <dl>
     * <dt>ECS_AL2</dt>
     * <dd>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html#al2ami">Amazon Linux
     * 2</a>: Default for all non-GPU instance families.
     * </p>
     * </dd>
     * <dt>ECS_AL2_NVIDIA</dt>
     * <dd>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html#gpuami">Amazon Linux
     * 2 (GPU)</a>: Default for all GPU instance families (for example <code>P4</code> and <code>G4</code>) and can be
     * used for all non Amazon Web Services Graviton-based instance types.
     * </p>
     * </dd>
     * <dt>ECS_AL2023</dt>
     * <dd>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html">Amazon Linux
     * 2023</a>: Batch supports Amazon Linux 2023.
     * </p>
     * <note>
     * <p>
     * Amazon Linux 2023 does not support <code>A1</code> instances.
     * </p>
     * </note></dd>
     * <dt>ECS_AL2023_NVIDIA</dt>
     * <dd>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html#gpuami">Amazon Linux
     * 2023 (GPU)</a>: For all GPU instance families and can be used for all non Amazon Web Services Graviton-based
     * instance types.
     * </p>
     * <note>
     * <p>
     * ECS_AL2023_NVIDIA doesn't support <code>p3</code> and <code>g3</code> instance types.
     * </p>
     * </note></dd>
     * </dl>
     * </dd>
     * <dt>EKS</dt>
     * <dd>
     * <p>
     * If the <code>imageIdOverride</code> parameter isn't specified, then a recent <a
     * href="https://docs.aws.amazon.com/eks/latest/userguide/eks-optimized-ami.html">Amazon EKS-optimized Amazon Linux
     * AMI</a> (<code>EKS_AL2</code>) is used. If a new image type is specified in an update, but neither an
     * <code>imageId</code> nor a <code>imageIdOverride</code> parameter is specified, then the latest Amazon EKS
     * optimized AMI for that image type that Batch supports is used.
     * </p>
     * <important>
     * <p>
     * Starting end of October 2025 Amazon EKS optimized Amazon Linux 2023 AMIs will be the default on Batch for EKS
     * versions prior to 1.33. Starting from Kubernetes version 1.33, EKS optimized Amazon Linux 2023 AMIs will be the
     * default when it becomes supported on Batch.
     * </p>
     * <p>
     * Amazon Web Services will end support for Amazon EKS AL2-optimized and AL2-accelerated AMIs, starting 11/26/25.
     * You can continue using Batch-provided Amazon EKS optimized Amazon Linux 2 AMIs on your Amazon EKS compute
     * environments beyond the 11/26/25 end-of-support date, these compute environments will no longer receive any new
     * software updates, security patches, or bug fixes from Amazon Web Services. For more information on upgrading from
     * AL2 to AL2023, see <a href="https://docs.aws.amazon.com/batch/latest/userguide/eks-migration-2023.html">How to
     * upgrade from EKS AL2 to EKS AL2023</a> in the <i>Batch User Guide</i>.
     * </p>
     * </important>
     * <dl>
     * <dt>EKS_AL2</dt>
     * <dd>
     * <p>
     * <a href="https://docs.aws.amazon.com/eks/latest/userguide/eks-optimized-ami.html">Amazon Linux 2</a>: Default for
     * all non-GPU instance families.
     * </p>
     * </dd>
     * <dt>EKS_AL2_NVIDIA</dt>
     * <dd>
     * <p>
     * <a href="https://docs.aws.amazon.com/eks/latest/userguide/eks-optimized-ami.html">Amazon Linux 2
     * (accelerated)</a>: Default for all GPU instance families (for example, <code>P4</code> and <code>G4</code>) and
     * can be used for all non Amazon Web Services Graviton-based instance types.
     * </p>
     * </dd>
     * <dt>EKS_AL2023</dt>
     * <dd>
     * <p>
     * <a href="https://docs.aws.amazon.com/eks/latest/userguide/eks-optimized-ami.html">Amazon Linux 2023</a>: Batch
     * supports Amazon Linux 2023.
     * </p>
     * <note>
     * <p>
     * Amazon Linux 2023 does not support <code>A1</code> instances.
     * </p>
     * </note></dd>
     * <dt>EKS_AL2023_NVIDIA</dt>
     * <dd>
     * <p>
     * <a href="https://docs.aws.amazon.com/eks/latest/userguide/eks-optimized-ami.html">Amazon Linux 2023
     * (accelerated)</a>: GPU instance families and can be used for all non Amazon Web Services Graviton-based instance
     * types.
     * </p>
     * </dd>
     * </dl>
     * </dd>
     * </dl>
     * 
     * @return The image type to match with the instance type to select an AMI. The supported values are different for
     *         <code>ECS</code> and <code>EKS</code> resources.</p>
     *         <dl>
     *         <dt>ECS</dt>
     *         <dd>
     *         <p>
     *         If the <code>imageIdOverride</code> parameter isn't specified, then a recent <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html#al2ami">Amazon
     *         ECS-optimized Amazon Linux 2 AMI</a> (<code>ECS_AL2</code>) is used. If a new image type is specified in
     *         an update, but neither an <code>imageId</code> nor a <code>imageIdOverride</code> parameter is specified,
     *         then the latest Amazon ECS optimized AMI for that image type that's supported by Batch is used.
     *         </p>
     *         <important>
     *         <p>
     *         Amazon Web Services will end support for Amazon ECS optimized AL2-optimized and AL2-accelerated AMIs.
     *         Starting in January 2026, Batch will change the default AMI for new Amazon ECS compute environments from
     *         Amazon Linux 2 to Amazon Linux 2023. We recommend migrating Batch Amazon ECS compute environments to
     *         Amazon Linux 2023 to maintain optimal performance and security. For more information on upgrading from
     *         AL2 to AL2023, see <a
     *         href="https://docs.aws.amazon.com/batch/latest/userguide/ecs-migration-2023.html">How to migrate from ECS
     *         AL2 to ECS AL2023</a> in the <i>Batch User Guide</i>.
     *         </p>
     *         </important>
     *         <dl>
     *         <dt>ECS_AL2</dt>
     *         <dd>
     *         <p>
     *         <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html#al2ami">Amazon
     *         Linux 2</a>: Default for all non-GPU instance families.
     *         </p>
     *         </dd>
     *         <dt>ECS_AL2_NVIDIA</dt>
     *         <dd>
     *         <p>
     *         <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html#gpuami">Amazon
     *         Linux 2 (GPU)</a>: Default for all GPU instance families (for example <code>P4</code> and <code>G4</code>
     *         ) and can be used for all non Amazon Web Services Graviton-based instance types.
     *         </p>
     *         </dd>
     *         <dt>ECS_AL2023</dt>
     *         <dd>
     *         <p>
     *         <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html">Amazon Linux
     *         2023</a>: Batch supports Amazon Linux 2023.
     *         </p>
     *         <note>
     *         <p>
     *         Amazon Linux 2023 does not support <code>A1</code> instances.
     *         </p>
     *         </note></dd>
     *         <dt>ECS_AL2023_NVIDIA</dt>
     *         <dd>
     *         <p>
     *         <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html#gpuami">Amazon
     *         Linux 2023 (GPU)</a>: For all GPU instance families and can be used for all non Amazon Web Services
     *         Graviton-based instance types.
     *         </p>
     *         <note>
     *         <p>
     *         ECS_AL2023_NVIDIA doesn't support <code>p3</code> and <code>g3</code> instance types.
     *         </p>
     *         </note></dd>
     *         </dl>
     *         </dd>
     *         <dt>EKS</dt>
     *         <dd>
     *         <p>
     *         If the <code>imageIdOverride</code> parameter isn't specified, then a recent <a
     *         href="https://docs.aws.amazon.com/eks/latest/userguide/eks-optimized-ami.html">Amazon EKS-optimized
     *         Amazon Linux AMI</a> (<code>EKS_AL2</code>) is used. If a new image type is specified in an update, but
     *         neither an <code>imageId</code> nor a <code>imageIdOverride</code> parameter is specified, then the
     *         latest Amazon EKS optimized AMI for that image type that Batch supports is used.
     *         </p>
     *         <important>
     *         <p>
     *         Starting end of October 2025 Amazon EKS optimized Amazon Linux 2023 AMIs will be the default on Batch for
     *         EKS versions prior to 1.33. Starting from Kubernetes version 1.33, EKS optimized Amazon Linux 2023 AMIs
     *         will be the default when it becomes supported on Batch.
     *         </p>
     *         <p>
     *         Amazon Web Services will end support for Amazon EKS AL2-optimized and AL2-accelerated AMIs, starting
     *         11/26/25. You can continue using Batch-provided Amazon EKS optimized Amazon Linux 2 AMIs on your Amazon
     *         EKS compute environments beyond the 11/26/25 end-of-support date, these compute environments will no
     *         longer receive any new software updates, security patches, or bug fixes from Amazon Web Services. For
     *         more information on upgrading from AL2 to AL2023, see <a
     *         href="https://docs.aws.amazon.com/batch/latest/userguide/eks-migration-2023.html">How to upgrade from EKS
     *         AL2 to EKS AL2023</a> in the <i>Batch User Guide</i>.
     *         </p>
     *         </important>
     *         <dl>
     *         <dt>EKS_AL2</dt>
     *         <dd>
     *         <p>
     *         <a href="https://docs.aws.amazon.com/eks/latest/userguide/eks-optimized-ami.html">Amazon Linux 2</a>:
     *         Default for all non-GPU instance families.
     *         </p>
     *         </dd>
     *         <dt>EKS_AL2_NVIDIA</dt>
     *         <dd>
     *         <p>
     *         <a href="https://docs.aws.amazon.com/eks/latest/userguide/eks-optimized-ami.html">Amazon Linux 2
     *         (accelerated)</a>: Default for all GPU instance families (for example, <code>P4</code> and
     *         <code>G4</code>) and can be used for all non Amazon Web Services Graviton-based instance types.
     *         </p>
     *         </dd>
     *         <dt>EKS_AL2023</dt>
     *         <dd>
     *         <p>
     *         <a href="https://docs.aws.amazon.com/eks/latest/userguide/eks-optimized-ami.html">Amazon Linux 2023</a>:
     *         Batch supports Amazon Linux 2023.
     *         </p>
     *         <note>
     *         <p>
     *         Amazon Linux 2023 does not support <code>A1</code> instances.
     *         </p>
     *         </note></dd>
     *         <dt>EKS_AL2023_NVIDIA</dt>
     *         <dd>
     *         <p>
     *         <a href="https://docs.aws.amazon.com/eks/latest/userguide/eks-optimized-ami.html">Amazon Linux 2023
     *         (accelerated)</a>: GPU instance families and can be used for all non Amazon Web Services Graviton-based
     *         instance types.
     *         </p>
     *         </dd>
     *         </dl>
     *         </dd>
     */
    public final String imageType() {
        return imageType;
    }

    /**
     * <p>
     * The AMI ID used for instances launched in the compute environment that match the image type. This setting
     * overrides the <code>imageId</code> set in the <code>computeResource</code> object.
     * </p>
     * <note>
     * <p>
     * The AMI that you choose for a compute environment must match the architecture of the instance types that you
     * intend to use for that compute environment. For example, if your compute environment uses A1 instance types, the
     * compute resource AMI that you choose must support ARM instances. Amazon ECS vends both x86 and ARM versions of
     * the Amazon ECS-optimized Amazon Linux 2 AMI. For more information, see <a href=
     * "https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html#ecs-optimized-ami-linux-variants.html"
     * >Amazon ECS-optimized Amazon Linux 2 AMI</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * </note>
     * 
     * @return The AMI ID used for instances launched in the compute environment that match the image type. This setting
     *         overrides the <code>imageId</code> set in the <code>computeResource</code> object.</p> <note>
     *         <p>
     *         The AMI that you choose for a compute environment must match the architecture of the instance types that
     *         you intend to use for that compute environment. For example, if your compute environment uses A1 instance
     *         types, the compute resource AMI that you choose must support ARM instances. Amazon ECS vends both x86 and
     *         ARM versions of the Amazon ECS-optimized Amazon Linux 2 AMI. For more information, see <a href=
     *         "https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html#ecs-optimized-ami-linux-variants.html"
     *         >Amazon ECS-optimized Amazon Linux 2 AMI</a> in the <i>Amazon Elastic Container Service Developer
     *         Guide</i>.
     *         </p>
     */
    public final String imageIdOverride() {
        return imageIdOverride;
    }

    /**
     * <p>
     * The Kubernetes version for the compute environment. If you don't specify a value, the latest version that Batch
     * supports is used.
     * </p>
     * 
     * @return The Kubernetes version for the compute environment. If you don't specify a value, the latest version that
     *         Batch supports is used.
     */
    public final String imageKubernetesVersion() {
        return imageKubernetesVersion;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(imageType());
        hashCode = 31 * hashCode + Objects.hashCode(imageIdOverride());
        hashCode = 31 * hashCode + Objects.hashCode(imageKubernetesVersion());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Ec2Configuration)) {
            return false;
        }
        Ec2Configuration other = (Ec2Configuration) obj;
        return Objects.equals(imageType(), other.imageType()) && Objects.equals(imageIdOverride(), other.imageIdOverride())
                && Objects.equals(imageKubernetesVersion(), other.imageKubernetesVersion());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Ec2Configuration").add("ImageType", imageType()).add("ImageIdOverride", imageIdOverride())
                .add("ImageKubernetesVersion", imageKubernetesVersion()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "imageType":
            return Optional.ofNullable(clazz.cast(imageType()));
        case "imageIdOverride":
            return Optional.ofNullable(clazz.cast(imageIdOverride()));
        case "imageKubernetesVersion":
            return Optional.ofNullable(clazz.cast(imageKubernetesVersion()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("imageType", IMAGE_TYPE_FIELD);
        map.put("imageIdOverride", IMAGE_ID_OVERRIDE_FIELD);
        map.put("imageKubernetesVersion", IMAGE_KUBERNETES_VERSION_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Ec2Configuration, T> g) {
        return obj -> g.apply((Ec2Configuration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Ec2Configuration> {
        /**
         * <p>
         * The image type to match with the instance type to select an AMI. The supported values are different for
         * <code>ECS</code> and <code>EKS</code> resources.
         * </p>
         * <dl>
         * <dt>ECS</dt>
         * <dd>
         * <p>
         * If the <code>imageIdOverride</code> parameter isn't specified, then a recent <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html#al2ami">Amazon
         * ECS-optimized Amazon Linux 2 AMI</a> (<code>ECS_AL2</code>) is used. If a new image type is specified in an
         * update, but neither an <code>imageId</code> nor a <code>imageIdOverride</code> parameter is specified, then
         * the latest Amazon ECS optimized AMI for that image type that's supported by Batch is used.
         * </p>
         * <important>
         * <p>
         * Amazon Web Services will end support for Amazon ECS optimized AL2-optimized and AL2-accelerated AMIs.
         * Starting in January 2026, Batch will change the default AMI for new Amazon ECS compute environments from
         * Amazon Linux 2 to Amazon Linux 2023. We recommend migrating Batch Amazon ECS compute environments to Amazon
         * Linux 2023 to maintain optimal performance and security. For more information on upgrading from AL2 to
         * AL2023, see <a href="https://docs.aws.amazon.com/batch/latest/userguide/ecs-migration-2023.html">How to
         * migrate from ECS AL2 to ECS AL2023</a> in the <i>Batch User Guide</i>.
         * </p>
         * </important>
         * <dl>
         * <dt>ECS_AL2</dt>
         * <dd>
         * <p>
         * <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html#al2ami">Amazon
         * Linux 2</a>: Default for all non-GPU instance families.
         * </p>
         * </dd>
         * <dt>ECS_AL2_NVIDIA</dt>
         * <dd>
         * <p>
         * <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html#gpuami">Amazon
         * Linux 2 (GPU)</a>: Default for all GPU instance families (for example <code>P4</code> and <code>G4</code>)
         * and can be used for all non Amazon Web Services Graviton-based instance types.
         * </p>
         * </dd>
         * <dt>ECS_AL2023</dt>
         * <dd>
         * <p>
         * <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html">Amazon Linux
         * 2023</a>: Batch supports Amazon Linux 2023.
         * </p>
         * <note>
         * <p>
         * Amazon Linux 2023 does not support <code>A1</code> instances.
         * </p>
         * </note></dd>
         * <dt>ECS_AL2023_NVIDIA</dt>
         * <dd>
         * <p>
         * <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html#gpuami">Amazon
         * Linux 2023 (GPU)</a>: For all GPU instance families and can be used for all non Amazon Web Services
         * Graviton-based instance types.
         * </p>
         * <note>
         * <p>
         * ECS_AL2023_NVIDIA doesn't support <code>p3</code> and <code>g3</code> instance types.
         * </p>
         * </note></dd>
         * </dl>
         * </dd>
         * <dt>EKS</dt>
         * <dd>
         * <p>
         * If the <code>imageIdOverride</code> parameter isn't specified, then a recent <a
         * href="https://docs.aws.amazon.com/eks/latest/userguide/eks-optimized-ami.html">Amazon EKS-optimized Amazon
         * Linux AMI</a> (<code>EKS_AL2</code>) is used. If a new image type is specified in an update, but neither an
         * <code>imageId</code> nor a <code>imageIdOverride</code> parameter is specified, then the latest Amazon EKS
         * optimized AMI for that image type that Batch supports is used.
         * </p>
         * <important>
         * <p>
         * Starting end of October 2025 Amazon EKS optimized Amazon Linux 2023 AMIs will be the default on Batch for EKS
         * versions prior to 1.33. Starting from Kubernetes version 1.33, EKS optimized Amazon Linux 2023 AMIs will be
         * the default when it becomes supported on Batch.
         * </p>
         * <p>
         * Amazon Web Services will end support for Amazon EKS AL2-optimized and AL2-accelerated AMIs, starting
         * 11/26/25. You can continue using Batch-provided Amazon EKS optimized Amazon Linux 2 AMIs on your Amazon EKS
         * compute environments beyond the 11/26/25 end-of-support date, these compute environments will no longer
         * receive any new software updates, security patches, or bug fixes from Amazon Web Services. For more
         * information on upgrading from AL2 to AL2023, see <a
         * href="https://docs.aws.amazon.com/batch/latest/userguide/eks-migration-2023.html">How to upgrade from EKS AL2
         * to EKS AL2023</a> in the <i>Batch User Guide</i>.
         * </p>
         * </important>
         * <dl>
         * <dt>EKS_AL2</dt>
         * <dd>
         * <p>
         * <a href="https://docs.aws.amazon.com/eks/latest/userguide/eks-optimized-ami.html">Amazon Linux 2</a>: Default
         * for all non-GPU instance families.
         * </p>
         * </dd>
         * <dt>EKS_AL2_NVIDIA</dt>
         * <dd>
         * <p>
         * <a href="https://docs.aws.amazon.com/eks/latest/userguide/eks-optimized-ami.html">Amazon Linux 2
         * (accelerated)</a>: Default for all GPU instance families (for example, <code>P4</code> and <code>G4</code>)
         * and can be used for all non Amazon Web Services Graviton-based instance types.
         * </p>
         * </dd>
         * <dt>EKS_AL2023</dt>
         * <dd>
         * <p>
         * <a href="https://docs.aws.amazon.com/eks/latest/userguide/eks-optimized-ami.html">Amazon Linux 2023</a>:
         * Batch supports Amazon Linux 2023.
         * </p>
         * <note>
         * <p>
         * Amazon Linux 2023 does not support <code>A1</code> instances.
         * </p>
         * </note></dd>
         * <dt>EKS_AL2023_NVIDIA</dt>
         * <dd>
         * <p>
         * <a href="https://docs.aws.amazon.com/eks/latest/userguide/eks-optimized-ami.html">Amazon Linux 2023
         * (accelerated)</a>: GPU instance families and can be used for all non Amazon Web Services Graviton-based
         * instance types.
         * </p>
         * </dd>
         * </dl>
         * </dd>
         * </dl>
         * 
         * @param imageType
         *        The image type to match with the instance type to select an AMI. The supported values are different
         *        for <code>ECS</code> and <code>EKS</code> resources.</p>
         *        <dl>
         *        <dt>ECS</dt>
         *        <dd>
         *        <p>
         *        If the <code>imageIdOverride</code> parameter isn't specified, then a recent <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html#al2ami"
         *        >Amazon ECS-optimized Amazon Linux 2 AMI</a> (<code>ECS_AL2</code>) is used. If a new image type is
         *        specified in an update, but neither an <code>imageId</code> nor a <code>imageIdOverride</code>
         *        parameter is specified, then the latest Amazon ECS optimized AMI for that image type that's supported
         *        by Batch is used.
         *        </p>
         *        <important>
         *        <p>
         *        Amazon Web Services will end support for Amazon ECS optimized AL2-optimized and AL2-accelerated AMIs.
         *        Starting in January 2026, Batch will change the default AMI for new Amazon ECS compute environments
         *        from Amazon Linux 2 to Amazon Linux 2023. We recommend migrating Batch Amazon ECS compute environments
         *        to Amazon Linux 2023 to maintain optimal performance and security. For more information on upgrading
         *        from AL2 to AL2023, see <a
         *        href="https://docs.aws.amazon.com/batch/latest/userguide/ecs-migration-2023.html">How to migrate from
         *        ECS AL2 to ECS AL2023</a> in the <i>Batch User Guide</i>.
         *        </p>
         *        </important>
         *        <dl>
         *        <dt>ECS_AL2</dt>
         *        <dd>
         *        <p>
         *        <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html#al2ami">
         *        Amazon Linux 2</a>: Default for all non-GPU instance families.
         *        </p>
         *        </dd>
         *        <dt>ECS_AL2_NVIDIA</dt>
         *        <dd>
         *        <p>
         *        <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html#gpuami">
         *        Amazon Linux 2 (GPU)</a>: Default for all GPU instance families (for example <code>P4</code> and
         *        <code>G4</code>) and can be used for all non Amazon Web Services Graviton-based instance types.
         *        </p>
         *        </dd>
         *        <dt>ECS_AL2023</dt>
         *        <dd>
         *        <p>
         *        <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html">Amazon
         *        Linux 2023</a>: Batch supports Amazon Linux 2023.
         *        </p>
         *        <note>
         *        <p>
         *        Amazon Linux 2023 does not support <code>A1</code> instances.
         *        </p>
         *        </note></dd>
         *        <dt>ECS_AL2023_NVIDIA</dt>
         *        <dd>
         *        <p>
         *        <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html#gpuami">
         *        Amazon Linux 2023 (GPU)</a>: For all GPU instance families and can be used for all non Amazon Web
         *        Services Graviton-based instance types.
         *        </p>
         *        <note>
         *        <p>
         *        ECS_AL2023_NVIDIA doesn't support <code>p3</code> and <code>g3</code> instance types.
         *        </p>
         *        </note></dd>
         *        </dl>
         *        </dd>
         *        <dt>EKS</dt>
         *        <dd>
         *        <p>
         *        If the <code>imageIdOverride</code> parameter isn't specified, then a recent <a
         *        href="https://docs.aws.amazon.com/eks/latest/userguide/eks-optimized-ami.html">Amazon EKS-optimized
         *        Amazon Linux AMI</a> (<code>EKS_AL2</code>) is used. If a new image type is specified in an update,
         *        but neither an <code>imageId</code> nor a <code>imageIdOverride</code> parameter is specified, then
         *        the latest Amazon EKS optimized AMI for that image type that Batch supports is used.
         *        </p>
         *        <important>
         *        <p>
         *        Starting end of October 2025 Amazon EKS optimized Amazon Linux 2023 AMIs will be the default on Batch
         *        for EKS versions prior to 1.33. Starting from Kubernetes version 1.33, EKS optimized Amazon Linux 2023
         *        AMIs will be the default when it becomes supported on Batch.
         *        </p>
         *        <p>
         *        Amazon Web Services will end support for Amazon EKS AL2-optimized and AL2-accelerated AMIs, starting
         *        11/26/25. You can continue using Batch-provided Amazon EKS optimized Amazon Linux 2 AMIs on your
         *        Amazon EKS compute environments beyond the 11/26/25 end-of-support date, these compute environments
         *        will no longer receive any new software updates, security patches, or bug fixes from Amazon Web
         *        Services. For more information on upgrading from AL2 to AL2023, see <a
         *        href="https://docs.aws.amazon.com/batch/latest/userguide/eks-migration-2023.html">How to upgrade from
         *        EKS AL2 to EKS AL2023</a> in the <i>Batch User Guide</i>.
         *        </p>
         *        </important>
         *        <dl>
         *        <dt>EKS_AL2</dt>
         *        <dd>
         *        <p>
         *        <a href="https://docs.aws.amazon.com/eks/latest/userguide/eks-optimized-ami.html">Amazon Linux 2</a>:
         *        Default for all non-GPU instance families.
         *        </p>
         *        </dd>
         *        <dt>EKS_AL2_NVIDIA</dt>
         *        <dd>
         *        <p>
         *        <a href="https://docs.aws.amazon.com/eks/latest/userguide/eks-optimized-ami.html">Amazon Linux 2
         *        (accelerated)</a>: Default for all GPU instance families (for example, <code>P4</code> and
         *        <code>G4</code>) and can be used for all non Amazon Web Services Graviton-based instance types.
         *        </p>
         *        </dd>
         *        <dt>EKS_AL2023</dt>
         *        <dd>
         *        <p>
         *        <a href="https://docs.aws.amazon.com/eks/latest/userguide/eks-optimized-ami.html">Amazon Linux
         *        2023</a>: Batch supports Amazon Linux 2023.
         *        </p>
         *        <note>
         *        <p>
         *        Amazon Linux 2023 does not support <code>A1</code> instances.
         *        </p>
         *        </note></dd>
         *        <dt>EKS_AL2023_NVIDIA</dt>
         *        <dd>
         *        <p>
         *        <a href="https://docs.aws.amazon.com/eks/latest/userguide/eks-optimized-ami.html">Amazon Linux 2023
         *        (accelerated)</a>: GPU instance families and can be used for all non Amazon Web Services
         *        Graviton-based instance types.
         *        </p>
         *        </dd>
         *        </dl>
         *        </dd>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder imageType(String imageType);

        /**
         * <p>
         * The AMI ID used for instances launched in the compute environment that match the image type. This setting
         * overrides the <code>imageId</code> set in the <code>computeResource</code> object.
         * </p>
         * <note>
         * <p>
         * The AMI that you choose for a compute environment must match the architecture of the instance types that you
         * intend to use for that compute environment. For example, if your compute environment uses A1 instance types,
         * the compute resource AMI that you choose must support ARM instances. Amazon ECS vends both x86 and ARM
         * versions of the Amazon ECS-optimized Amazon Linux 2 AMI. For more information, see <a href=
         * "https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html#ecs-optimized-ami-linux-variants.html"
         * >Amazon ECS-optimized Amazon Linux 2 AMI</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * </note>
         * 
         * @param imageIdOverride
         *        The AMI ID used for instances launched in the compute environment that match the image type. This
         *        setting overrides the <code>imageId</code> set in the <code>computeResource</code> object.</p> <note>
         *        <p>
         *        The AMI that you choose for a compute environment must match the architecture of the instance types
         *        that you intend to use for that compute environment. For example, if your compute environment uses A1
         *        instance types, the compute resource AMI that you choose must support ARM instances. Amazon ECS vends
         *        both x86 and ARM versions of the Amazon ECS-optimized Amazon Linux 2 AMI. For more information, see <a
         *        href=
         *        "https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html#ecs-optimized-ami-linux-variants.html"
         *        >Amazon ECS-optimized Amazon Linux 2 AMI</a> in the <i>Amazon Elastic Container Service Developer
         *        Guide</i>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder imageIdOverride(String imageIdOverride);

        /**
         * <p>
         * The Kubernetes version for the compute environment. If you don't specify a value, the latest version that
         * Batch supports is used.
         * </p>
         * 
         * @param imageKubernetesVersion
         *        The Kubernetes version for the compute environment. If you don't specify a value, the latest version
         *        that Batch supports is used.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder imageKubernetesVersion(String imageKubernetesVersion);
    }

    static final class BuilderImpl implements Builder {
        private String imageType;

        private String imageIdOverride;

        private String imageKubernetesVersion;

        private BuilderImpl() {
        }

        private BuilderImpl(Ec2Configuration model) {
            imageType(model.imageType);
            imageIdOverride(model.imageIdOverride);
            imageKubernetesVersion(model.imageKubernetesVersion);
        }

        public final String getImageType() {
            return imageType;
        }

        public final void setImageType(String imageType) {
            this.imageType = imageType;
        }

        @Override
        public final Builder imageType(String imageType) {
            this.imageType = imageType;
            return this;
        }

        public final String getImageIdOverride() {
            return imageIdOverride;
        }

        public final void setImageIdOverride(String imageIdOverride) {
            this.imageIdOverride = imageIdOverride;
        }

        @Override
        public final Builder imageIdOverride(String imageIdOverride) {
            this.imageIdOverride = imageIdOverride;
            return this;
        }

        public final String getImageKubernetesVersion() {
            return imageKubernetesVersion;
        }

        public final void setImageKubernetesVersion(String imageKubernetesVersion) {
            this.imageKubernetesVersion = imageKubernetesVersion;
        }

        @Override
        public final Builder imageKubernetesVersion(String imageKubernetesVersion) {
            this.imageKubernetesVersion = imageKubernetesVersion;
            return this;
        }

        @Override
        public Ec2Configuration build() {
            return new Ec2Configuration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
