/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.autoscaling.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains details about a collection of instances launched in the Auto Scaling group.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class InstanceCollection implements SdkPojo, Serializable,
        ToCopyableBuilder<InstanceCollection.Builder, InstanceCollection> {
    private static final SdkField<String> INSTANCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("InstanceType").getter(getter(InstanceCollection::instanceType)).setter(setter(Builder::instanceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceType").build()).build();

    private static final SdkField<String> MARKET_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MarketType").getter(getter(InstanceCollection::marketType)).setter(setter(Builder::marketType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MarketType").build()).build();

    private static final SdkField<String> SUBNET_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SubnetId").getter(getter(InstanceCollection::subnetId)).setter(setter(Builder::subnetId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SubnetId").build()).build();

    private static final SdkField<String> AVAILABILITY_ZONE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AvailabilityZone").getter(getter(InstanceCollection::availabilityZone))
            .setter(setter(Builder::availabilityZone))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AvailabilityZone").build()).build();

    private static final SdkField<String> AVAILABILITY_ZONE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AvailabilityZoneId").getter(getter(InstanceCollection::availabilityZoneId))
            .setter(setter(Builder::availabilityZoneId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AvailabilityZoneId").build())
            .build();

    private static final SdkField<List<String>> INSTANCE_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("InstanceIds")
            .getter(getter(InstanceCollection::instanceIds))
            .setter(setter(Builder::instanceIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(INSTANCE_TYPE_FIELD,
            MARKET_TYPE_FIELD, SUBNET_ID_FIELD, AVAILABILITY_ZONE_FIELD, AVAILABILITY_ZONE_ID_FIELD, INSTANCE_IDS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String instanceType;

    private final String marketType;

    private final String subnetId;

    private final String availabilityZone;

    private final String availabilityZoneId;

    private final List<String> instanceIds;

    private InstanceCollection(BuilderImpl builder) {
        this.instanceType = builder.instanceType;
        this.marketType = builder.marketType;
        this.subnetId = builder.subnetId;
        this.availabilityZone = builder.availabilityZone;
        this.availabilityZoneId = builder.availabilityZoneId;
        this.instanceIds = builder.instanceIds;
    }

    /**
     * <p>
     * The instance type of the launched instances.
     * </p>
     * 
     * @return The instance type of the launched instances.
     */
    public final String instanceType() {
        return instanceType;
    }

    /**
     * <p>
     * The market type for the instances (On-Demand or Spot).
     * </p>
     * 
     * @return The market type for the instances (On-Demand or Spot).
     */
    public final String marketType() {
        return marketType;
    }

    /**
     * <p>
     * The ID of the subnet where the instances were launched.
     * </p>
     * 
     * @return The ID of the subnet where the instances were launched.
     */
    public final String subnetId() {
        return subnetId;
    }

    /**
     * <p>
     * The Availability Zone where the instances were launched.
     * </p>
     * 
     * @return The Availability Zone where the instances were launched.
     */
    public final String availabilityZone() {
        return availabilityZone;
    }

    /**
     * <p>
     * The Availability Zone ID where the instances in this collection were launched.
     * </p>
     * 
     * @return The Availability Zone ID where the instances in this collection were launched.
     */
    public final String availabilityZoneId() {
        return availabilityZoneId;
    }

    /**
     * For responses, this returns true if the service returned a value for the InstanceIds property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasInstanceIds() {
        return instanceIds != null && !(instanceIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of instance IDs for the successfully launched instances.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasInstanceIds} method.
     * </p>
     * 
     * @return A list of instance IDs for the successfully launched instances.
     */
    public final List<String> instanceIds() {
        return instanceIds;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(instanceType());
        hashCode = 31 * hashCode + Objects.hashCode(marketType());
        hashCode = 31 * hashCode + Objects.hashCode(subnetId());
        hashCode = 31 * hashCode + Objects.hashCode(availabilityZone());
        hashCode = 31 * hashCode + Objects.hashCode(availabilityZoneId());
        hashCode = 31 * hashCode + Objects.hashCode(hasInstanceIds() ? instanceIds() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof InstanceCollection)) {
            return false;
        }
        InstanceCollection other = (InstanceCollection) obj;
        return Objects.equals(instanceType(), other.instanceType()) && Objects.equals(marketType(), other.marketType())
                && Objects.equals(subnetId(), other.subnetId()) && Objects.equals(availabilityZone(), other.availabilityZone())
                && Objects.equals(availabilityZoneId(), other.availabilityZoneId()) && hasInstanceIds() == other.hasInstanceIds()
                && Objects.equals(instanceIds(), other.instanceIds());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("InstanceCollection").add("InstanceType", instanceType()).add("MarketType", marketType())
                .add("SubnetId", subnetId()).add("AvailabilityZone", availabilityZone())
                .add("AvailabilityZoneId", availabilityZoneId()).add("InstanceIds", hasInstanceIds() ? instanceIds() : null)
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "InstanceType":
            return Optional.ofNullable(clazz.cast(instanceType()));
        case "MarketType":
            return Optional.ofNullable(clazz.cast(marketType()));
        case "SubnetId":
            return Optional.ofNullable(clazz.cast(subnetId()));
        case "AvailabilityZone":
            return Optional.ofNullable(clazz.cast(availabilityZone()));
        case "AvailabilityZoneId":
            return Optional.ofNullable(clazz.cast(availabilityZoneId()));
        case "InstanceIds":
            return Optional.ofNullable(clazz.cast(instanceIds()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("InstanceType", INSTANCE_TYPE_FIELD);
        map.put("MarketType", MARKET_TYPE_FIELD);
        map.put("SubnetId", SUBNET_ID_FIELD);
        map.put("AvailabilityZone", AVAILABILITY_ZONE_FIELD);
        map.put("AvailabilityZoneId", AVAILABILITY_ZONE_ID_FIELD);
        map.put("InstanceIds", INSTANCE_IDS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<InstanceCollection, T> g) {
        return obj -> g.apply((InstanceCollection) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, InstanceCollection> {
        /**
         * <p>
         * The instance type of the launched instances.
         * </p>
         * 
         * @param instanceType
         *        The instance type of the launched instances.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceType(String instanceType);

        /**
         * <p>
         * The market type for the instances (On-Demand or Spot).
         * </p>
         * 
         * @param marketType
         *        The market type for the instances (On-Demand or Spot).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder marketType(String marketType);

        /**
         * <p>
         * The ID of the subnet where the instances were launched.
         * </p>
         * 
         * @param subnetId
         *        The ID of the subnet where the instances were launched.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetId(String subnetId);

        /**
         * <p>
         * The Availability Zone where the instances were launched.
         * </p>
         * 
         * @param availabilityZone
         *        The Availability Zone where the instances were launched.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availabilityZone(String availabilityZone);

        /**
         * <p>
         * The Availability Zone ID where the instances in this collection were launched.
         * </p>
         * 
         * @param availabilityZoneId
         *        The Availability Zone ID where the instances in this collection were launched.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availabilityZoneId(String availabilityZoneId);

        /**
         * <p>
         * A list of instance IDs for the successfully launched instances.
         * </p>
         * 
         * @param instanceIds
         *        A list of instance IDs for the successfully launched instances.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceIds(Collection<String> instanceIds);

        /**
         * <p>
         * A list of instance IDs for the successfully launched instances.
         * </p>
         * 
         * @param instanceIds
         *        A list of instance IDs for the successfully launched instances.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceIds(String... instanceIds);
    }

    static final class BuilderImpl implements Builder {
        private String instanceType;

        private String marketType;

        private String subnetId;

        private String availabilityZone;

        private String availabilityZoneId;

        private List<String> instanceIds = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(InstanceCollection model) {
            instanceType(model.instanceType);
            marketType(model.marketType);
            subnetId(model.subnetId);
            availabilityZone(model.availabilityZone);
            availabilityZoneId(model.availabilityZoneId);
            instanceIds(model.instanceIds);
        }

        public final String getInstanceType() {
            return instanceType;
        }

        public final void setInstanceType(String instanceType) {
            this.instanceType = instanceType;
        }

        @Override
        public final Builder instanceType(String instanceType) {
            this.instanceType = instanceType;
            return this;
        }

        public final String getMarketType() {
            return marketType;
        }

        public final void setMarketType(String marketType) {
            this.marketType = marketType;
        }

        @Override
        public final Builder marketType(String marketType) {
            this.marketType = marketType;
            return this;
        }

        public final String getSubnetId() {
            return subnetId;
        }

        public final void setSubnetId(String subnetId) {
            this.subnetId = subnetId;
        }

        @Override
        public final Builder subnetId(String subnetId) {
            this.subnetId = subnetId;
            return this;
        }

        public final String getAvailabilityZone() {
            return availabilityZone;
        }

        public final void setAvailabilityZone(String availabilityZone) {
            this.availabilityZone = availabilityZone;
        }

        @Override
        public final Builder availabilityZone(String availabilityZone) {
            this.availabilityZone = availabilityZone;
            return this;
        }

        public final String getAvailabilityZoneId() {
            return availabilityZoneId;
        }

        public final void setAvailabilityZoneId(String availabilityZoneId) {
            this.availabilityZoneId = availabilityZoneId;
        }

        @Override
        public final Builder availabilityZoneId(String availabilityZoneId) {
            this.availabilityZoneId = availabilityZoneId;
            return this;
        }

        public final Collection<String> getInstanceIds() {
            if (instanceIds instanceof SdkAutoConstructList) {
                return null;
            }
            return instanceIds;
        }

        public final void setInstanceIds(Collection<String> instanceIds) {
            this.instanceIds = InstanceIdsCopier.copy(instanceIds);
        }

        @Override
        public final Builder instanceIds(Collection<String> instanceIds) {
            this.instanceIds = InstanceIdsCopier.copy(instanceIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder instanceIds(String... instanceIds) {
            instanceIds(Arrays.asList(instanceIds));
            return this;
        }

        @Override
        public InstanceCollection build() {
            return new InstanceCollection(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
