/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.athena;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.athena.model.AthenaException;
import software.amazon.awssdk.services.athena.model.AthenaRequest;
import software.amazon.awssdk.services.athena.model.BatchGetNamedQueryRequest;
import software.amazon.awssdk.services.athena.model.BatchGetNamedQueryResponse;
import software.amazon.awssdk.services.athena.model.BatchGetQueryExecutionRequest;
import software.amazon.awssdk.services.athena.model.BatchGetQueryExecutionResponse;
import software.amazon.awssdk.services.athena.model.CreateDataCatalogRequest;
import software.amazon.awssdk.services.athena.model.CreateDataCatalogResponse;
import software.amazon.awssdk.services.athena.model.CreateNamedQueryRequest;
import software.amazon.awssdk.services.athena.model.CreateNamedQueryResponse;
import software.amazon.awssdk.services.athena.model.CreateWorkGroupRequest;
import software.amazon.awssdk.services.athena.model.CreateWorkGroupResponse;
import software.amazon.awssdk.services.athena.model.DeleteDataCatalogRequest;
import software.amazon.awssdk.services.athena.model.DeleteDataCatalogResponse;
import software.amazon.awssdk.services.athena.model.DeleteNamedQueryRequest;
import software.amazon.awssdk.services.athena.model.DeleteNamedQueryResponse;
import software.amazon.awssdk.services.athena.model.DeleteWorkGroupRequest;
import software.amazon.awssdk.services.athena.model.DeleteWorkGroupResponse;
import software.amazon.awssdk.services.athena.model.GetDataCatalogRequest;
import software.amazon.awssdk.services.athena.model.GetDataCatalogResponse;
import software.amazon.awssdk.services.athena.model.GetDatabaseRequest;
import software.amazon.awssdk.services.athena.model.GetDatabaseResponse;
import software.amazon.awssdk.services.athena.model.GetNamedQueryRequest;
import software.amazon.awssdk.services.athena.model.GetNamedQueryResponse;
import software.amazon.awssdk.services.athena.model.GetQueryExecutionRequest;
import software.amazon.awssdk.services.athena.model.GetQueryExecutionResponse;
import software.amazon.awssdk.services.athena.model.GetQueryResultsRequest;
import software.amazon.awssdk.services.athena.model.GetQueryResultsResponse;
import software.amazon.awssdk.services.athena.model.GetTableMetadataRequest;
import software.amazon.awssdk.services.athena.model.GetTableMetadataResponse;
import software.amazon.awssdk.services.athena.model.GetWorkGroupRequest;
import software.amazon.awssdk.services.athena.model.GetWorkGroupResponse;
import software.amazon.awssdk.services.athena.model.InternalServerException;
import software.amazon.awssdk.services.athena.model.InvalidRequestException;
import software.amazon.awssdk.services.athena.model.ListDataCatalogsRequest;
import software.amazon.awssdk.services.athena.model.ListDataCatalogsResponse;
import software.amazon.awssdk.services.athena.model.ListDatabasesRequest;
import software.amazon.awssdk.services.athena.model.ListDatabasesResponse;
import software.amazon.awssdk.services.athena.model.ListNamedQueriesRequest;
import software.amazon.awssdk.services.athena.model.ListNamedQueriesResponse;
import software.amazon.awssdk.services.athena.model.ListQueryExecutionsRequest;
import software.amazon.awssdk.services.athena.model.ListQueryExecutionsResponse;
import software.amazon.awssdk.services.athena.model.ListTableMetadataRequest;
import software.amazon.awssdk.services.athena.model.ListTableMetadataResponse;
import software.amazon.awssdk.services.athena.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.athena.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.athena.model.ListWorkGroupsRequest;
import software.amazon.awssdk.services.athena.model.ListWorkGroupsResponse;
import software.amazon.awssdk.services.athena.model.MetadataException;
import software.amazon.awssdk.services.athena.model.ResourceNotFoundException;
import software.amazon.awssdk.services.athena.model.StartQueryExecutionRequest;
import software.amazon.awssdk.services.athena.model.StartQueryExecutionResponse;
import software.amazon.awssdk.services.athena.model.StopQueryExecutionRequest;
import software.amazon.awssdk.services.athena.model.StopQueryExecutionResponse;
import software.amazon.awssdk.services.athena.model.TagResourceRequest;
import software.amazon.awssdk.services.athena.model.TagResourceResponse;
import software.amazon.awssdk.services.athena.model.TooManyRequestsException;
import software.amazon.awssdk.services.athena.model.UntagResourceRequest;
import software.amazon.awssdk.services.athena.model.UntagResourceResponse;
import software.amazon.awssdk.services.athena.model.UpdateDataCatalogRequest;
import software.amazon.awssdk.services.athena.model.UpdateDataCatalogResponse;
import software.amazon.awssdk.services.athena.model.UpdateWorkGroupRequest;
import software.amazon.awssdk.services.athena.model.UpdateWorkGroupResponse;
import software.amazon.awssdk.services.athena.paginators.GetQueryResultsPublisher;
import software.amazon.awssdk.services.athena.paginators.ListDataCatalogsPublisher;
import software.amazon.awssdk.services.athena.paginators.ListDatabasesPublisher;
import software.amazon.awssdk.services.athena.paginators.ListNamedQueriesPublisher;
import software.amazon.awssdk.services.athena.paginators.ListQueryExecutionsPublisher;
import software.amazon.awssdk.services.athena.paginators.ListTableMetadataPublisher;
import software.amazon.awssdk.services.athena.paginators.ListTagsForResourcePublisher;
import software.amazon.awssdk.services.athena.paginators.ListWorkGroupsPublisher;
import software.amazon.awssdk.services.athena.transform.BatchGetNamedQueryRequestMarshaller;
import software.amazon.awssdk.services.athena.transform.BatchGetQueryExecutionRequestMarshaller;
import software.amazon.awssdk.services.athena.transform.CreateDataCatalogRequestMarshaller;
import software.amazon.awssdk.services.athena.transform.CreateNamedQueryRequestMarshaller;
import software.amazon.awssdk.services.athena.transform.CreateWorkGroupRequestMarshaller;
import software.amazon.awssdk.services.athena.transform.DeleteDataCatalogRequestMarshaller;
import software.amazon.awssdk.services.athena.transform.DeleteNamedQueryRequestMarshaller;
import software.amazon.awssdk.services.athena.transform.DeleteWorkGroupRequestMarshaller;
import software.amazon.awssdk.services.athena.transform.GetDataCatalogRequestMarshaller;
import software.amazon.awssdk.services.athena.transform.GetDatabaseRequestMarshaller;
import software.amazon.awssdk.services.athena.transform.GetNamedQueryRequestMarshaller;
import software.amazon.awssdk.services.athena.transform.GetQueryExecutionRequestMarshaller;
import software.amazon.awssdk.services.athena.transform.GetQueryResultsRequestMarshaller;
import software.amazon.awssdk.services.athena.transform.GetTableMetadataRequestMarshaller;
import software.amazon.awssdk.services.athena.transform.GetWorkGroupRequestMarshaller;
import software.amazon.awssdk.services.athena.transform.ListDataCatalogsRequestMarshaller;
import software.amazon.awssdk.services.athena.transform.ListDatabasesRequestMarshaller;
import software.amazon.awssdk.services.athena.transform.ListNamedQueriesRequestMarshaller;
import software.amazon.awssdk.services.athena.transform.ListQueryExecutionsRequestMarshaller;
import software.amazon.awssdk.services.athena.transform.ListTableMetadataRequestMarshaller;
import software.amazon.awssdk.services.athena.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.athena.transform.ListWorkGroupsRequestMarshaller;
import software.amazon.awssdk.services.athena.transform.StartQueryExecutionRequestMarshaller;
import software.amazon.awssdk.services.athena.transform.StopQueryExecutionRequestMarshaller;
import software.amazon.awssdk.services.athena.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.athena.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.athena.transform.UpdateDataCatalogRequestMarshaller;
import software.amazon.awssdk.services.athena.transform.UpdateWorkGroupRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link AthenaAsyncClient}.
 *
 * @see AthenaAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultAthenaAsyncClient implements AthenaAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultAthenaAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultAthenaAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Returns the details of a single named query or a list of up to 50 queries, which you provide as an array of query
     * ID strings. Requires you to have access to the workgroup in which the queries were saved. Use
     * <a>ListNamedQueriesInput</a> to get the list of named query IDs in the specified workgroup. If information could
     * not be retrieved for a submitted query ID, information about the query ID submitted is listed under
     * <a>UnprocessedNamedQueryId</a>. Named queries differ from executed queries. Use
     * <a>BatchGetQueryExecutionInput</a> to get details about each unique query execution, and
     * <a>ListQueryExecutionsInput</a> to get a list of query execution IDs.
     * </p>
     *
     * @param batchGetNamedQueryRequest
     * @return A Java Future containing the result of the BatchGetNamedQuery operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.BatchGetNamedQuery
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/BatchGetNamedQuery" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<BatchGetNamedQueryResponse> batchGetNamedQuery(BatchGetNamedQueryRequest batchGetNamedQueryRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchGetNamedQueryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchGetNamedQueryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchGetNamedQueryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchGetNamedQueryRequest, BatchGetNamedQueryResponse>()
                            .withOperationName("BatchGetNamedQuery")
                            .withMarshaller(new BatchGetNamedQueryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(batchGetNamedQueryRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the details of a single query execution or a list of up to 50 query executions, which you provide as an
     * array of query execution ID strings. Requires you to have access to the workgroup in which the queries ran. To
     * get a list of query execution IDs, use <a>ListQueryExecutionsInput&#36WorkGroup</a>. Query executions differ from
     * named (saved) queries. Use <a>BatchGetNamedQueryInput</a> to get details about named queries.
     * </p>
     *
     * @param batchGetQueryExecutionRequest
     * @return A Java Future containing the result of the BatchGetQueryExecution operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.BatchGetQueryExecution
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/BatchGetQueryExecution" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchGetQueryExecutionResponse> batchGetQueryExecution(
            BatchGetQueryExecutionRequest batchGetQueryExecutionRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchGetQueryExecutionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchGetQueryExecutionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchGetQueryExecutionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchGetQueryExecutionRequest, BatchGetQueryExecutionResponse>()
                            .withOperationName("BatchGetQueryExecution")
                            .withMarshaller(new BatchGetQueryExecutionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(batchGetQueryExecutionRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates (registers) a data catalog with the specified name and properties. Catalogs created are visible to all
     * users of the same AWS account.
     * </p>
     *
     * @param createDataCatalogRequest
     * @return A Java Future containing the result of the CreateDataCatalog operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.CreateDataCatalog
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/CreateDataCatalog" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateDataCatalogResponse> createDataCatalog(CreateDataCatalogRequest createDataCatalogRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateDataCatalogResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateDataCatalogResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateDataCatalogResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateDataCatalogRequest, CreateDataCatalogResponse>()
                            .withOperationName("CreateDataCatalog")
                            .withMarshaller(new CreateDataCatalogRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createDataCatalogRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a named query in the specified workgroup. Requires that you have access to the workgroup.
     * </p>
     * <p>
     * For code samples using the AWS SDK for Java, see <a
     * href="http://docs.aws.amazon.com/athena/latest/ug/code-samples.html">Examples and Code Samples</a> in the
     * <i>Amazon Athena User Guide</i>.
     * </p>
     *
     * @param createNamedQueryRequest
     * @return A Java Future containing the result of the CreateNamedQuery operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.CreateNamedQuery
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/CreateNamedQuery" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateNamedQueryResponse> createNamedQuery(CreateNamedQueryRequest createNamedQueryRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateNamedQueryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateNamedQueryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateNamedQueryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateNamedQueryRequest, CreateNamedQueryResponse>()
                            .withOperationName("CreateNamedQuery")
                            .withMarshaller(new CreateNamedQueryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createNamedQueryRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a workgroup with the specified name.
     * </p>
     *
     * @param createWorkGroupRequest
     * @return A Java Future containing the result of the CreateWorkGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.CreateWorkGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/CreateWorkGroup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateWorkGroupResponse> createWorkGroup(CreateWorkGroupRequest createWorkGroupRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateWorkGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateWorkGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateWorkGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateWorkGroupRequest, CreateWorkGroupResponse>()
                            .withOperationName("CreateWorkGroup")
                            .withMarshaller(new CreateWorkGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createWorkGroupRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a data catalog.
     * </p>
     *
     * @param deleteDataCatalogRequest
     * @return A Java Future containing the result of the DeleteDataCatalog operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.DeleteDataCatalog
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/DeleteDataCatalog" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteDataCatalogResponse> deleteDataCatalog(DeleteDataCatalogRequest deleteDataCatalogRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteDataCatalogResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteDataCatalogResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteDataCatalogResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteDataCatalogRequest, DeleteDataCatalogResponse>()
                            .withOperationName("DeleteDataCatalog")
                            .withMarshaller(new DeleteDataCatalogRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteDataCatalogRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the named query if you have access to the workgroup in which the query was saved.
     * </p>
     * <p>
     * For code samples using the AWS SDK for Java, see <a
     * href="http://docs.aws.amazon.com/athena/latest/ug/code-samples.html">Examples and Code Samples</a> in the
     * <i>Amazon Athena User Guide</i>.
     * </p>
     *
     * @param deleteNamedQueryRequest
     * @return A Java Future containing the result of the DeleteNamedQuery operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.DeleteNamedQuery
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/DeleteNamedQuery" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteNamedQueryResponse> deleteNamedQuery(DeleteNamedQueryRequest deleteNamedQueryRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteNamedQueryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteNamedQueryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteNamedQueryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteNamedQueryRequest, DeleteNamedQueryResponse>()
                            .withOperationName("DeleteNamedQuery")
                            .withMarshaller(new DeleteNamedQueryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteNamedQueryRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the workgroup with the specified name. The primary workgroup cannot be deleted.
     * </p>
     *
     * @param deleteWorkGroupRequest
     * @return A Java Future containing the result of the DeleteWorkGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.DeleteWorkGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/DeleteWorkGroup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteWorkGroupResponse> deleteWorkGroup(DeleteWorkGroupRequest deleteWorkGroupRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteWorkGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteWorkGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteWorkGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteWorkGroupRequest, DeleteWorkGroupResponse>()
                            .withOperationName("DeleteWorkGroup")
                            .withMarshaller(new DeleteWorkGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteWorkGroupRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the specified data catalog.
     * </p>
     *
     * @param getDataCatalogRequest
     * @return A Java Future containing the result of the GetDataCatalog operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.GetDataCatalog
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/GetDataCatalog" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetDataCatalogResponse> getDataCatalog(GetDataCatalogRequest getDataCatalogRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDataCatalogResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetDataCatalogResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDataCatalogResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDataCatalogRequest, GetDataCatalogResponse>()
                            .withOperationName("GetDataCatalog")
                            .withMarshaller(new GetDataCatalogRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getDataCatalogRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a database object for the specfied database and data catalog.
     * </p>
     *
     * @param getDatabaseRequest
     * @return A Java Future containing the result of the GetDatabase operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>MetadataException An exception that Athena received when it called a custom metastore. Occurs if the
     *         error is not caused by user input (<code>InvalidRequestException</code>) or from the Athena platform (
     *         <code>InternalServerException</code>). For example, if a user-created Lambda function is missing
     *         permissions, the Lambda <code>4XX</code> exception is returned in a <code>MetadataException</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.GetDatabase
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/GetDatabase" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetDatabaseResponse> getDatabase(GetDatabaseRequest getDatabaseRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDatabaseResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetDatabaseResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDatabaseResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDatabaseRequest, GetDatabaseResponse>()
                            .withOperationName("GetDatabase").withMarshaller(new GetDatabaseRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getDatabaseRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about a single query. Requires that you have access to the workgroup in which the query was
     * saved.
     * </p>
     *
     * @param getNamedQueryRequest
     * @return A Java Future containing the result of the GetNamedQuery operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.GetNamedQuery
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/GetNamedQuery" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetNamedQueryResponse> getNamedQuery(GetNamedQueryRequest getNamedQueryRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetNamedQueryResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetNamedQueryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetNamedQueryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetNamedQueryRequest, GetNamedQueryResponse>()
                            .withOperationName("GetNamedQuery")
                            .withMarshaller(new GetNamedQueryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getNamedQueryRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about a single execution of a query if you have access to the workgroup in which the query
     * ran. Each time a query executes, information about the query execution is saved with a unique ID.
     * </p>
     *
     * @param getQueryExecutionRequest
     * @return A Java Future containing the result of the GetQueryExecution operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.GetQueryExecution
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/GetQueryExecution" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetQueryExecutionResponse> getQueryExecution(GetQueryExecutionRequest getQueryExecutionRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetQueryExecutionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetQueryExecutionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetQueryExecutionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetQueryExecutionRequest, GetQueryExecutionResponse>()
                            .withOperationName("GetQueryExecution")
                            .withMarshaller(new GetQueryExecutionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getQueryExecutionRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Streams the results of a single query execution specified by <code>QueryExecutionId</code> from the Athena query
     * results location in Amazon S3. For more information, see <a
     * href="https://docs.aws.amazon.com/athena/latest/ug/querying.html">Query Results</a> in the <i>Amazon Athena User
     * Guide</i>. This request does not execute the query but returns results. Use <a>StartQueryExecution</a> to run a
     * query.
     * </p>
     * <p>
     * To stream query results successfully, the IAM principal with permission to call <code>GetQueryResults</code> also
     * must have permissions to the Amazon S3 <code>GetObject</code> action for the Athena query results location.
     * </p>
     * <important>
     * <p>
     * IAM principals with permission to the Amazon S3 <code>GetObject</code> action for the query results location are
     * able to retrieve query results from Amazon S3 even if permission to the <code>GetQueryResults</code> action is
     * denied. To restrict user or role access, ensure that Amazon S3 permissions to the Athena query location are
     * denied.
     * </p>
     * </important>
     *
     * @param getQueryResultsRequest
     * @return A Java Future containing the result of the GetQueryResults operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.GetQueryResults
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/GetQueryResults" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetQueryResultsResponse> getQueryResults(GetQueryResultsRequest getQueryResultsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetQueryResultsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetQueryResultsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetQueryResultsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetQueryResultsRequest, GetQueryResultsResponse>()
                            .withOperationName("GetQueryResults")
                            .withMarshaller(new GetQueryResultsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getQueryResultsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Streams the results of a single query execution specified by <code>QueryExecutionId</code> from the Athena query
     * results location in Amazon S3. For more information, see <a
     * href="https://docs.aws.amazon.com/athena/latest/ug/querying.html">Query Results</a> in the <i>Amazon Athena User
     * Guide</i>. This request does not execute the query but returns results. Use <a>StartQueryExecution</a> to run a
     * query.
     * </p>
     * <p>
     * To stream query results successfully, the IAM principal with permission to call <code>GetQueryResults</code> also
     * must have permissions to the Amazon S3 <code>GetObject</code> action for the Athena query results location.
     * </p>
     * <important>
     * <p>
     * IAM principals with permission to the Amazon S3 <code>GetObject</code> action for the query results location are
     * able to retrieve query results from Amazon S3 even if permission to the <code>GetQueryResults</code> action is
     * denied. To restrict user or role access, ensure that Amazon S3 permissions to the Athena query location are
     * denied.
     * </p>
     * </important><br/>
     * <p>
     * This is a variant of
     * {@link #getQueryResults(software.amazon.awssdk.services.athena.model.GetQueryResultsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.athena.paginators.GetQueryResultsPublisher publisher = client.getQueryResultsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.athena.paginators.GetQueryResultsPublisher publisher = client.getQueryResultsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.athena.model.GetQueryResultsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.athena.model.GetQueryResultsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getQueryResults(software.amazon.awssdk.services.athena.model.GetQueryResultsRequest)} operation.</b>
     * </p>
     *
     * @param getQueryResultsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.GetQueryResults
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/GetQueryResults" target="_top">AWS API
     *      Documentation</a>
     */
    public GetQueryResultsPublisher getQueryResultsPaginator(GetQueryResultsRequest getQueryResultsRequest) {
        return new GetQueryResultsPublisher(this, applyPaginatorUserAgent(getQueryResultsRequest));
    }

    /**
     * <p>
     * Returns table metadata for the specified catalog, database, and table.
     * </p>
     *
     * @param getTableMetadataRequest
     * @return A Java Future containing the result of the GetTableMetadata operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>MetadataException An exception that Athena received when it called a custom metastore. Occurs if the
     *         error is not caused by user input (<code>InvalidRequestException</code>) or from the Athena platform (
     *         <code>InternalServerException</code>). For example, if a user-created Lambda function is missing
     *         permissions, the Lambda <code>4XX</code> exception is returned in a <code>MetadataException</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.GetTableMetadata
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/GetTableMetadata" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetTableMetadataResponse> getTableMetadata(GetTableMetadataRequest getTableMetadataRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetTableMetadataResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetTableMetadataResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetTableMetadataResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetTableMetadataRequest, GetTableMetadataResponse>()
                            .withOperationName("GetTableMetadata")
                            .withMarshaller(new GetTableMetadataRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getTableMetadataRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about the workgroup with the specified name.
     * </p>
     *
     * @param getWorkGroupRequest
     * @return A Java Future containing the result of the GetWorkGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.GetWorkGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/GetWorkGroup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetWorkGroupResponse> getWorkGroup(GetWorkGroupRequest getWorkGroupRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetWorkGroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetWorkGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetWorkGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetWorkGroupRequest, GetWorkGroupResponse>()
                            .withOperationName("GetWorkGroup").withMarshaller(new GetWorkGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getWorkGroupRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the data catalogs in the current AWS account.
     * </p>
     *
     * @param listDataCatalogsRequest
     * @return A Java Future containing the result of the ListDataCatalogs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.ListDataCatalogs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/ListDataCatalogs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListDataCatalogsResponse> listDataCatalogs(ListDataCatalogsRequest listDataCatalogsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDataCatalogsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListDataCatalogsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDataCatalogsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDataCatalogsRequest, ListDataCatalogsResponse>()
                            .withOperationName("ListDataCatalogs")
                            .withMarshaller(new ListDataCatalogsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listDataCatalogsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the data catalogs in the current AWS account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listDataCatalogs(software.amazon.awssdk.services.athena.model.ListDataCatalogsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.athena.paginators.ListDataCatalogsPublisher publisher = client.listDataCatalogsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.athena.paginators.ListDataCatalogsPublisher publisher = client.listDataCatalogsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.athena.model.ListDataCatalogsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.athena.model.ListDataCatalogsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listDataCatalogs(software.amazon.awssdk.services.athena.model.ListDataCatalogsRequest)} operation.</b>
     * </p>
     *
     * @param listDataCatalogsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.ListDataCatalogs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/ListDataCatalogs" target="_top">AWS API
     *      Documentation</a>
     */
    public ListDataCatalogsPublisher listDataCatalogsPaginator(ListDataCatalogsRequest listDataCatalogsRequest) {
        return new ListDataCatalogsPublisher(this, applyPaginatorUserAgent(listDataCatalogsRequest));
    }

    /**
     * <p>
     * Lists the databases in the specified data catalog.
     * </p>
     *
     * @param listDatabasesRequest
     * @return A Java Future containing the result of the ListDatabases operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>MetadataException An exception that Athena received when it called a custom metastore. Occurs if the
     *         error is not caused by user input (<code>InvalidRequestException</code>) or from the Athena platform (
     *         <code>InternalServerException</code>). For example, if a user-created Lambda function is missing
     *         permissions, the Lambda <code>4XX</code> exception is returned in a <code>MetadataException</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.ListDatabases
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/ListDatabases" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListDatabasesResponse> listDatabases(ListDatabasesRequest listDatabasesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDatabasesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListDatabasesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDatabasesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDatabasesRequest, ListDatabasesResponse>()
                            .withOperationName("ListDatabases")
                            .withMarshaller(new ListDatabasesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listDatabasesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the databases in the specified data catalog.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listDatabases(software.amazon.awssdk.services.athena.model.ListDatabasesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.athena.paginators.ListDatabasesPublisher publisher = client.listDatabasesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.athena.paginators.ListDatabasesPublisher publisher = client.listDatabasesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.athena.model.ListDatabasesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.athena.model.ListDatabasesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listDatabases(software.amazon.awssdk.services.athena.model.ListDatabasesRequest)} operation.</b>
     * </p>
     *
     * @param listDatabasesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>MetadataException An exception that Athena received when it called a custom metastore. Occurs if the
     *         error is not caused by user input (<code>InvalidRequestException</code>) or from the Athena platform (
     *         <code>InternalServerException</code>). For example, if a user-created Lambda function is missing
     *         permissions, the Lambda <code>4XX</code> exception is returned in a <code>MetadataException</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.ListDatabases
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/ListDatabases" target="_top">AWS API
     *      Documentation</a>
     */
    public ListDatabasesPublisher listDatabasesPaginator(ListDatabasesRequest listDatabasesRequest) {
        return new ListDatabasesPublisher(this, applyPaginatorUserAgent(listDatabasesRequest));
    }

    /**
     * <p>
     * Provides a list of available query IDs only for queries saved in the specified workgroup. Requires that you have
     * access to the specified workgroup. If a workgroup is not specified, lists the saved queries for the primary
     * workgroup.
     * </p>
     * <p>
     * For code samples using the AWS SDK for Java, see <a
     * href="http://docs.aws.amazon.com/athena/latest/ug/code-samples.html">Examples and Code Samples</a> in the
     * <i>Amazon Athena User Guide</i>.
     * </p>
     *
     * @param listNamedQueriesRequest
     * @return A Java Future containing the result of the ListNamedQueries operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.ListNamedQueries
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/ListNamedQueries" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListNamedQueriesResponse> listNamedQueries(ListNamedQueriesRequest listNamedQueriesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListNamedQueriesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListNamedQueriesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListNamedQueriesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListNamedQueriesRequest, ListNamedQueriesResponse>()
                            .withOperationName("ListNamedQueries")
                            .withMarshaller(new ListNamedQueriesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listNamedQueriesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides a list of available query IDs only for queries saved in the specified workgroup. Requires that you have
     * access to the specified workgroup. If a workgroup is not specified, lists the saved queries for the primary
     * workgroup.
     * </p>
     * <p>
     * For code samples using the AWS SDK for Java, see <a
     * href="http://docs.aws.amazon.com/athena/latest/ug/code-samples.html">Examples and Code Samples</a> in the
     * <i>Amazon Athena User Guide</i>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listNamedQueries(software.amazon.awssdk.services.athena.model.ListNamedQueriesRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.athena.paginators.ListNamedQueriesPublisher publisher = client.listNamedQueriesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.athena.paginators.ListNamedQueriesPublisher publisher = client.listNamedQueriesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.athena.model.ListNamedQueriesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.athena.model.ListNamedQueriesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listNamedQueries(software.amazon.awssdk.services.athena.model.ListNamedQueriesRequest)} operation.</b>
     * </p>
     *
     * @param listNamedQueriesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.ListNamedQueries
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/ListNamedQueries" target="_top">AWS API
     *      Documentation</a>
     */
    public ListNamedQueriesPublisher listNamedQueriesPaginator(ListNamedQueriesRequest listNamedQueriesRequest) {
        return new ListNamedQueriesPublisher(this, applyPaginatorUserAgent(listNamedQueriesRequest));
    }

    /**
     * <p>
     * Provides a list of available query execution IDs for the queries in the specified workgroup. If a workgroup is
     * not specified, returns a list of query execution IDs for the primary workgroup. Requires you to have access to
     * the workgroup in which the queries ran.
     * </p>
     * <p>
     * For code samples using the AWS SDK for Java, see <a
     * href="http://docs.aws.amazon.com/athena/latest/ug/code-samples.html">Examples and Code Samples</a> in the
     * <i>Amazon Athena User Guide</i>.
     * </p>
     *
     * @param listQueryExecutionsRequest
     * @return A Java Future containing the result of the ListQueryExecutions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.ListQueryExecutions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/ListQueryExecutions" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListQueryExecutionsResponse> listQueryExecutions(
            ListQueryExecutionsRequest listQueryExecutionsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListQueryExecutionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListQueryExecutionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListQueryExecutionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListQueryExecutionsRequest, ListQueryExecutionsResponse>()
                            .withOperationName("ListQueryExecutions")
                            .withMarshaller(new ListQueryExecutionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listQueryExecutionsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides a list of available query execution IDs for the queries in the specified workgroup. If a workgroup is
     * not specified, returns a list of query execution IDs for the primary workgroup. Requires you to have access to
     * the workgroup in which the queries ran.
     * </p>
     * <p>
     * For code samples using the AWS SDK for Java, see <a
     * href="http://docs.aws.amazon.com/athena/latest/ug/code-samples.html">Examples and Code Samples</a> in the
     * <i>Amazon Athena User Guide</i>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listQueryExecutions(software.amazon.awssdk.services.athena.model.ListQueryExecutionsRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.athena.paginators.ListQueryExecutionsPublisher publisher = client.listQueryExecutionsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.athena.paginators.ListQueryExecutionsPublisher publisher = client.listQueryExecutionsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.athena.model.ListQueryExecutionsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.athena.model.ListQueryExecutionsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listQueryExecutions(software.amazon.awssdk.services.athena.model.ListQueryExecutionsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listQueryExecutionsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.ListQueryExecutions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/ListQueryExecutions" target="_top">AWS API
     *      Documentation</a>
     */
    public ListQueryExecutionsPublisher listQueryExecutionsPaginator(ListQueryExecutionsRequest listQueryExecutionsRequest) {
        return new ListQueryExecutionsPublisher(this, applyPaginatorUserAgent(listQueryExecutionsRequest));
    }

    /**
     * <p>
     * Lists the metadata for the tables in the specified data catalog database.
     * </p>
     *
     * @param listTableMetadataRequest
     * @return A Java Future containing the result of the ListTableMetadata operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>MetadataException An exception that Athena received when it called a custom metastore. Occurs if the
     *         error is not caused by user input (<code>InvalidRequestException</code>) or from the Athena platform (
     *         <code>InternalServerException</code>). For example, if a user-created Lambda function is missing
     *         permissions, the Lambda <code>4XX</code> exception is returned in a <code>MetadataException</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.ListTableMetadata
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/ListTableMetadata" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListTableMetadataResponse> listTableMetadata(ListTableMetadataRequest listTableMetadataRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTableMetadataResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTableMetadataResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTableMetadataResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTableMetadataRequest, ListTableMetadataResponse>()
                            .withOperationName("ListTableMetadata")
                            .withMarshaller(new ListTableMetadataRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listTableMetadataRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the metadata for the tables in the specified data catalog database.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listTableMetadata(software.amazon.awssdk.services.athena.model.ListTableMetadataRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.athena.paginators.ListTableMetadataPublisher publisher = client.listTableMetadataPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.athena.paginators.ListTableMetadataPublisher publisher = client.listTableMetadataPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.athena.model.ListTableMetadataResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.athena.model.ListTableMetadataResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listTableMetadata(software.amazon.awssdk.services.athena.model.ListTableMetadataRequest)} operation.</b>
     * </p>
     *
     * @param listTableMetadataRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>MetadataException An exception that Athena received when it called a custom metastore. Occurs if the
     *         error is not caused by user input (<code>InvalidRequestException</code>) or from the Athena platform (
     *         <code>InternalServerException</code>). For example, if a user-created Lambda function is missing
     *         permissions, the Lambda <code>4XX</code> exception is returned in a <code>MetadataException</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.ListTableMetadata
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/ListTableMetadata" target="_top">AWS API
     *      Documentation</a>
     */
    public ListTableMetadataPublisher listTableMetadataPaginator(ListTableMetadataRequest listTableMetadataRequest) {
        return new ListTableMetadataPublisher(this, applyPaginatorUserAgent(listTableMetadataRequest));
    }

    /**
     * <p>
     * Lists the tags associated with an Athena workgroup or data catalog resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>ResourceNotFoundException A resource, such as a workgroup, was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.ListTagsForResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listTagsForResourceRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the tags associated with an Athena workgroup or data catalog resource.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listTagsForResource(software.amazon.awssdk.services.athena.model.ListTagsForResourceRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.athena.paginators.ListTagsForResourcePublisher publisher = client.listTagsForResourcePaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.athena.paginators.ListTagsForResourcePublisher publisher = client.listTagsForResourcePaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.athena.model.ListTagsForResourceResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.athena.model.ListTagsForResourceResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listTagsForResource(software.amazon.awssdk.services.athena.model.ListTagsForResourceRequest)}
     * operation.</b>
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>ResourceNotFoundException A resource, such as a workgroup, was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.ListTagsForResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    public ListTagsForResourcePublisher listTagsForResourcePaginator(ListTagsForResourceRequest listTagsForResourceRequest) {
        return new ListTagsForResourcePublisher(this, applyPaginatorUserAgent(listTagsForResourceRequest));
    }

    /**
     * <p>
     * Lists available workgroups for the account.
     * </p>
     *
     * @param listWorkGroupsRequest
     * @return A Java Future containing the result of the ListWorkGroups operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.ListWorkGroups
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/ListWorkGroups" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListWorkGroupsResponse> listWorkGroups(ListWorkGroupsRequest listWorkGroupsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListWorkGroupsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListWorkGroupsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListWorkGroupsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListWorkGroupsRequest, ListWorkGroupsResponse>()
                            .withOperationName("ListWorkGroups")
                            .withMarshaller(new ListWorkGroupsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listWorkGroupsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists available workgroups for the account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listWorkGroups(software.amazon.awssdk.services.athena.model.ListWorkGroupsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.athena.paginators.ListWorkGroupsPublisher publisher = client.listWorkGroupsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.athena.paginators.ListWorkGroupsPublisher publisher = client.listWorkGroupsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.athena.model.ListWorkGroupsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.athena.model.ListWorkGroupsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listWorkGroups(software.amazon.awssdk.services.athena.model.ListWorkGroupsRequest)} operation.</b>
     * </p>
     *
     * @param listWorkGroupsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.ListWorkGroups
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/ListWorkGroups" target="_top">AWS API
     *      Documentation</a>
     */
    public ListWorkGroupsPublisher listWorkGroupsPaginator(ListWorkGroupsRequest listWorkGroupsRequest) {
        return new ListWorkGroupsPublisher(this, applyPaginatorUserAgent(listWorkGroupsRequest));
    }

    /**
     * <p>
     * Runs the SQL query statements contained in the <code>Query</code>. Requires you to have access to the workgroup
     * in which the query ran. Running queries against an external catalog requires <a>GetDataCatalog</a> permission to
     * the catalog. For code samples using the AWS SDK for Java, see <a
     * href="http://docs.aws.amazon.com/athena/latest/ug/code-samples.html">Examples and Code Samples</a> in the
     * <i>Amazon Athena User Guide</i>.
     * </p>
     *
     * @param startQueryExecutionRequest
     * @return A Java Future containing the result of the StartQueryExecution operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>TooManyRequestsException Indicates that the request was throttled.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.StartQueryExecution
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/StartQueryExecution" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StartQueryExecutionResponse> startQueryExecution(
            StartQueryExecutionRequest startQueryExecutionRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartQueryExecutionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartQueryExecutionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartQueryExecutionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartQueryExecutionRequest, StartQueryExecutionResponse>()
                            .withOperationName("StartQueryExecution")
                            .withMarshaller(new StartQueryExecutionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(startQueryExecutionRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Stops a query execution. Requires you to have access to the workgroup in which the query ran.
     * </p>
     * <p>
     * For code samples using the AWS SDK for Java, see <a
     * href="http://docs.aws.amazon.com/athena/latest/ug/code-samples.html">Examples and Code Samples</a> in the
     * <i>Amazon Athena User Guide</i>.
     * </p>
     *
     * @param stopQueryExecutionRequest
     * @return A Java Future containing the result of the StopQueryExecution operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.StopQueryExecution
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/StopQueryExecution" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StopQueryExecutionResponse> stopQueryExecution(StopQueryExecutionRequest stopQueryExecutionRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopQueryExecutionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StopQueryExecutionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StopQueryExecutionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopQueryExecutionRequest, StopQueryExecutionResponse>()
                            .withOperationName("StopQueryExecution")
                            .withMarshaller(new StopQueryExecutionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(stopQueryExecutionRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds one or more tags to an Athena resource. A tag is a label that you assign to a resource. In Athena, a
     * resource can be a workgroup or data catalog. Each tag consists of a key and an optional value, both of which you
     * define. For example, you can use tags to categorize Athena workgroups or data catalogs by purpose, owner, or
     * environment. Use a consistent set of tag keys to make it easier to search and filter workgroups or data catalogs
     * in your account. For best practices, see <a
     * href="https://aws.amazon.com/answers/account-management/aws-tagging-strategies/">Tagging Best Practices</a>. Tag
     * keys can be from 1 to 128 UTF-8 Unicode characters, and tag values can be from 0 to 256 UTF-8 Unicode characters.
     * Tags can use letters and numbers representable in UTF-8, and the following characters: + - = . _ : / @. Tag keys
     * and values are case-sensitive. Tag keys must be unique per resource. If you specify more than one tag, separate
     * them by commas.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>ResourceNotFoundException A resource, such as a workgroup, was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.TagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(tagResourceRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes one or more tags from a data catalog or workgroup resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>ResourceNotFoundException A resource, such as a workgroup, was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.UntagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(untagResourceRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the data catalog that has the specified name.
     * </p>
     *
     * @param updateDataCatalogRequest
     * @return A Java Future containing the result of the UpdateDataCatalog operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.UpdateDataCatalog
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/UpdateDataCatalog" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateDataCatalogResponse> updateDataCatalog(UpdateDataCatalogRequest updateDataCatalogRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateDataCatalogResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateDataCatalogResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateDataCatalogResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateDataCatalogRequest, UpdateDataCatalogResponse>()
                            .withOperationName("UpdateDataCatalog")
                            .withMarshaller(new UpdateDataCatalogRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateDataCatalogRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the workgroup with the specified name. The workgroup's name cannot be changed.
     * </p>
     *
     * @param updateWorkGroupRequest
     * @return A Java Future containing the result of the UpdateWorkGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Indicates a platform issue, which may be due to a transient condition or
     *         outage.</li>
     *         <li>InvalidRequestException Indicates that something is wrong with the input to the request. For example,
     *         a required parameter may be missing or out of range.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AthenaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AthenaAsyncClient.UpdateWorkGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/athena-2017-05-18/UpdateWorkGroup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateWorkGroupResponse> updateWorkGroup(UpdateWorkGroupRequest updateWorkGroupRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateWorkGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateWorkGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateWorkGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateWorkGroupRequest, UpdateWorkGroupResponse>()
                            .withOperationName("UpdateWorkGroup")
                            .withMarshaller(new UpdateWorkGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateWorkGroupRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(AthenaException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MetadataException")
                                .exceptionBuilderSupplier(MetadataException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestException")
                                .exceptionBuilderSupplier(InvalidRequestException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyRequestsException")
                                .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
    }

    private <T extends AthenaRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
