/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.arcregionswitch;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.Consumer;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.protocols.rpcv2.SmithyRpcV2CborProtocolFactory;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.arcregionswitch.internal.ArcRegionSwitchServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.arcregionswitch.internal.ServiceVersionInfo;
import software.amazon.awssdk.services.arcregionswitch.model.AccessDeniedException;
import software.amazon.awssdk.services.arcregionswitch.model.ApprovePlanExecutionStepRequest;
import software.amazon.awssdk.services.arcregionswitch.model.ApprovePlanExecutionStepResponse;
import software.amazon.awssdk.services.arcregionswitch.model.ArcRegionSwitchException;
import software.amazon.awssdk.services.arcregionswitch.model.CancelPlanExecutionRequest;
import software.amazon.awssdk.services.arcregionswitch.model.CancelPlanExecutionResponse;
import software.amazon.awssdk.services.arcregionswitch.model.CreatePlanRequest;
import software.amazon.awssdk.services.arcregionswitch.model.CreatePlanResponse;
import software.amazon.awssdk.services.arcregionswitch.model.DeletePlanRequest;
import software.amazon.awssdk.services.arcregionswitch.model.DeletePlanResponse;
import software.amazon.awssdk.services.arcregionswitch.model.GetPlanEvaluationStatusRequest;
import software.amazon.awssdk.services.arcregionswitch.model.GetPlanEvaluationStatusResponse;
import software.amazon.awssdk.services.arcregionswitch.model.GetPlanExecutionRequest;
import software.amazon.awssdk.services.arcregionswitch.model.GetPlanExecutionResponse;
import software.amazon.awssdk.services.arcregionswitch.model.GetPlanInRegionRequest;
import software.amazon.awssdk.services.arcregionswitch.model.GetPlanInRegionResponse;
import software.amazon.awssdk.services.arcregionswitch.model.GetPlanRequest;
import software.amazon.awssdk.services.arcregionswitch.model.GetPlanResponse;
import software.amazon.awssdk.services.arcregionswitch.model.IllegalArgumentException;
import software.amazon.awssdk.services.arcregionswitch.model.IllegalStateException;
import software.amazon.awssdk.services.arcregionswitch.model.InternalServerException;
import software.amazon.awssdk.services.arcregionswitch.model.ListPlanExecutionEventsRequest;
import software.amazon.awssdk.services.arcregionswitch.model.ListPlanExecutionEventsResponse;
import software.amazon.awssdk.services.arcregionswitch.model.ListPlanExecutionsRequest;
import software.amazon.awssdk.services.arcregionswitch.model.ListPlanExecutionsResponse;
import software.amazon.awssdk.services.arcregionswitch.model.ListPlansInRegionRequest;
import software.amazon.awssdk.services.arcregionswitch.model.ListPlansInRegionResponse;
import software.amazon.awssdk.services.arcregionswitch.model.ListPlansRequest;
import software.amazon.awssdk.services.arcregionswitch.model.ListPlansResponse;
import software.amazon.awssdk.services.arcregionswitch.model.ListRoute53HealthChecksRequest;
import software.amazon.awssdk.services.arcregionswitch.model.ListRoute53HealthChecksResponse;
import software.amazon.awssdk.services.arcregionswitch.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.arcregionswitch.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.arcregionswitch.model.ResourceNotFoundException;
import software.amazon.awssdk.services.arcregionswitch.model.StartPlanExecutionRequest;
import software.amazon.awssdk.services.arcregionswitch.model.StartPlanExecutionResponse;
import software.amazon.awssdk.services.arcregionswitch.model.TagResourceRequest;
import software.amazon.awssdk.services.arcregionswitch.model.TagResourceResponse;
import software.amazon.awssdk.services.arcregionswitch.model.UntagResourceRequest;
import software.amazon.awssdk.services.arcregionswitch.model.UntagResourceResponse;
import software.amazon.awssdk.services.arcregionswitch.model.UpdatePlanExecutionRequest;
import software.amazon.awssdk.services.arcregionswitch.model.UpdatePlanExecutionResponse;
import software.amazon.awssdk.services.arcregionswitch.model.UpdatePlanExecutionStepRequest;
import software.amazon.awssdk.services.arcregionswitch.model.UpdatePlanExecutionStepResponse;
import software.amazon.awssdk.services.arcregionswitch.model.UpdatePlanRequest;
import software.amazon.awssdk.services.arcregionswitch.model.UpdatePlanResponse;
import software.amazon.awssdk.services.arcregionswitch.transform.ApprovePlanExecutionStepRequestMarshaller;
import software.amazon.awssdk.services.arcregionswitch.transform.CancelPlanExecutionRequestMarshaller;
import software.amazon.awssdk.services.arcregionswitch.transform.CreatePlanRequestMarshaller;
import software.amazon.awssdk.services.arcregionswitch.transform.DeletePlanRequestMarshaller;
import software.amazon.awssdk.services.arcregionswitch.transform.GetPlanEvaluationStatusRequestMarshaller;
import software.amazon.awssdk.services.arcregionswitch.transform.GetPlanExecutionRequestMarshaller;
import software.amazon.awssdk.services.arcregionswitch.transform.GetPlanInRegionRequestMarshaller;
import software.amazon.awssdk.services.arcregionswitch.transform.GetPlanRequestMarshaller;
import software.amazon.awssdk.services.arcregionswitch.transform.ListPlanExecutionEventsRequestMarshaller;
import software.amazon.awssdk.services.arcregionswitch.transform.ListPlanExecutionsRequestMarshaller;
import software.amazon.awssdk.services.arcregionswitch.transform.ListPlansInRegionRequestMarshaller;
import software.amazon.awssdk.services.arcregionswitch.transform.ListPlansRequestMarshaller;
import software.amazon.awssdk.services.arcregionswitch.transform.ListRoute53HealthChecksRequestMarshaller;
import software.amazon.awssdk.services.arcregionswitch.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.arcregionswitch.transform.StartPlanExecutionRequestMarshaller;
import software.amazon.awssdk.services.arcregionswitch.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.arcregionswitch.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.arcregionswitch.transform.UpdatePlanExecutionRequestMarshaller;
import software.amazon.awssdk.services.arcregionswitch.transform.UpdatePlanExecutionStepRequestMarshaller;
import software.amazon.awssdk.services.arcregionswitch.transform.UpdatePlanRequestMarshaller;
import software.amazon.awssdk.services.arcregionswitch.waiters.ArcRegionSwitchAsyncWaiter;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link ArcRegionSwitchAsyncClient}.
 *
 * @see ArcRegionSwitchAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultArcRegionSwitchAsyncClient implements ArcRegionSwitchAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultArcRegionSwitchAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.SMITHY_RPC_V2_CBOR).build();

    private final AsyncClientHandler clientHandler;

    private final SmithyRpcV2CborProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final ScheduledExecutorService executorService;

    protected DefaultArcRegionSwitchAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this)
                .option(SdkClientOption.API_METADATA, "ARC_Region_switch" + "#" + ServiceVersionInfo.VERSION).build();
        this.protocolFactory = init(SmithyRpcV2CborProtocolFactory.builder()).build();
        this.executorService = clientConfiguration.option(SdkClientOption.SCHEDULED_EXECUTOR_SERVICE);
    }

    /**
     * <p>
     * Approves a step in a plan execution that requires manual approval. When you create a plan, you can include
     * approval steps that require manual intervention before the execution can proceed. This operation allows you to
     * provide that approval.
     * </p>
     * <p>
     * You must specify the plan ARN, execution ID, step name, and approval status. You can also provide an optional
     * comment explaining the approval decision.
     * </p>
     *
     * @param approvePlanExecutionStepRequest
     * @return A Java Future containing the result of the ApprovePlanExecutionStep operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</p>
     *         <p>
     *         HTTP Status Code: 404</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</p>
     *         <p>
     *         HTTP Status Code: 403</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ArcRegionSwitchException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ArcRegionSwitchAsyncClient.ApprovePlanExecutionStep
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/ApprovePlanExecutionStep"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ApprovePlanExecutionStepResponse> approvePlanExecutionStep(
            ApprovePlanExecutionStepRequest approvePlanExecutionStepRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(approvePlanExecutionStepRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, approvePlanExecutionStepRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ARC Region switch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ApprovePlanExecutionStep");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ApprovePlanExecutionStepResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ApprovePlanExecutionStepResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "IllegalArgumentException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("IllegalArgumentException").httpStatusCode(400)
                            .exceptionBuilderSupplier(IllegalArgumentException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "IllegalStateException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("IllegalStateException").httpStatusCode(400)
                            .exceptionBuilderSupplier(IllegalStateException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ApprovePlanExecutionStepResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ApprovePlanExecutionStepRequest, ApprovePlanExecutionStepResponse>()
                            .withOperationName("ApprovePlanExecutionStep").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ApprovePlanExecutionStepRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(approvePlanExecutionStepRequest));
            CompletableFuture<ApprovePlanExecutionStepResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Cancels an in-progress plan execution. This operation stops the execution of the plan and prevents any further
     * steps from being processed.
     * </p>
     * <p>
     * You must specify the plan ARN and execution ID. You can also provide an optional comment explaining why the
     * execution was canceled.
     * </p>
     *
     * @param cancelPlanExecutionRequest
     * @return A Java Future containing the result of the CancelPlanExecution operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</p>
     *         <p>
     *         HTTP Status Code: 404</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</p>
     *         <p>
     *         HTTP Status Code: 403</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ArcRegionSwitchException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ArcRegionSwitchAsyncClient.CancelPlanExecution
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/CancelPlanExecution"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CancelPlanExecutionResponse> cancelPlanExecution(
            CancelPlanExecutionRequest cancelPlanExecutionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(cancelPlanExecutionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelPlanExecutionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ARC Region switch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelPlanExecution");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CancelPlanExecutionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CancelPlanExecutionResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "IllegalArgumentException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("IllegalArgumentException").httpStatusCode(400)
                            .exceptionBuilderSupplier(IllegalArgumentException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "IllegalStateException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("IllegalStateException").httpStatusCode(400)
                            .exceptionBuilderSupplier(IllegalStateException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<CancelPlanExecutionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CancelPlanExecutionRequest, CancelPlanExecutionResponse>()
                            .withOperationName("CancelPlanExecution").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CancelPlanExecutionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(cancelPlanExecutionRequest));
            CompletableFuture<CancelPlanExecutionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new Region switch plan. A plan defines the steps required to shift traffic from one Amazon Web Services
     * Region to another.
     * </p>
     * <p>
     * You must specify a name for the plan, the primary Region, and at least one additional Region. You can also
     * provide a description, execution role, recovery time objective, associated alarms, triggers, and workflows that
     * define the steps to execute during a Region switch.
     * </p>
     *
     * @param createPlanRequest
     * @return A Java Future containing the result of the CreatePlan operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ArcRegionSwitchException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ArcRegionSwitchAsyncClient.CreatePlan
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/CreatePlan" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreatePlanResponse> createPlan(CreatePlanRequest createPlanRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createPlanRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createPlanRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ARC Region switch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreatePlan");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreatePlanResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreatePlanResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "IllegalArgumentException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("IllegalArgumentException").httpStatusCode(400)
                            .exceptionBuilderSupplier(IllegalArgumentException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "IllegalStateException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("IllegalStateException").httpStatusCode(400)
                            .exceptionBuilderSupplier(IllegalStateException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<CreatePlanResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreatePlanRequest, CreatePlanResponse>().withOperationName("CreatePlan")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreatePlanRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createPlanRequest));
            CompletableFuture<CreatePlanResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a Region switch plan. You must specify the ARN of the plan to delete.
     * </p>
     * <p>
     * You cannot delete a plan that has an active execution in progress.
     * </p>
     *
     * @param deletePlanRequest
     * @return A Java Future containing the result of the DeletePlan operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</p>
     *         <p>
     *         HTTP Status Code: 404</li>
     *         <li>IllegalStateException The operation failed because the current state of the resource doesn't allow
     *         the operation to proceed.</p>
     *         <p>
     *         HTTP Status Code: 400</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ArcRegionSwitchException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ArcRegionSwitchAsyncClient.DeletePlan
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/DeletePlan" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeletePlanResponse> deletePlan(DeletePlanRequest deletePlanRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deletePlanRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deletePlanRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ARC Region switch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePlan");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeletePlanResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeletePlanResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "IllegalArgumentException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("IllegalArgumentException").httpStatusCode(400)
                            .exceptionBuilderSupplier(IllegalArgumentException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "IllegalStateException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("IllegalStateException").httpStatusCode(400)
                            .exceptionBuilderSupplier(IllegalStateException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeletePlanResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeletePlanRequest, DeletePlanResponse>().withOperationName("DeletePlan")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeletePlanRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deletePlanRequest));
            CompletableFuture<DeletePlanResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves detailed information about a Region switch plan. You must specify the ARN of the plan.
     * </p>
     *
     * @param getPlanRequest
     * @return A Java Future containing the result of the GetPlan operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</p>
     *         <p>
     *         HTTP Status Code: 404</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ArcRegionSwitchException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ArcRegionSwitchAsyncClient.GetPlan
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/GetPlan" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetPlanResponse> getPlan(GetPlanRequest getPlanRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getPlanRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPlanRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ARC Region switch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPlan");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetPlanResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetPlanResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "IllegalArgumentException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("IllegalArgumentException").httpStatusCode(400)
                            .exceptionBuilderSupplier(IllegalArgumentException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "IllegalStateException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("IllegalStateException").httpStatusCode(400)
                            .exceptionBuilderSupplier(IllegalStateException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetPlanResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetPlanRequest, GetPlanResponse>().withOperationName("GetPlan")
                            .withProtocolMetadata(protocolMetadata).withMarshaller(new GetPlanRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getPlanRequest));
            CompletableFuture<GetPlanResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the evaluation status of a Region switch plan. The evaluation status provides information about the
     * last time the plan was evaluated and any warnings or issues detected.
     * </p>
     *
     * @param getPlanEvaluationStatusRequest
     * @return A Java Future containing the result of the GetPlanEvaluationStatus operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</p>
     *         <p>
     *         HTTP Status Code: 404</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</p>
     *         <p>
     *         HTTP Status Code: 403</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ArcRegionSwitchException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ArcRegionSwitchAsyncClient.GetPlanEvaluationStatus
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/GetPlanEvaluationStatus"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetPlanEvaluationStatusResponse> getPlanEvaluationStatus(
            GetPlanEvaluationStatusRequest getPlanEvaluationStatusRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getPlanEvaluationStatusRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPlanEvaluationStatusRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ARC Region switch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPlanEvaluationStatus");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetPlanEvaluationStatusResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetPlanEvaluationStatusResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "IllegalArgumentException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("IllegalArgumentException").httpStatusCode(400)
                            .exceptionBuilderSupplier(IllegalArgumentException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "IllegalStateException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("IllegalStateException").httpStatusCode(400)
                            .exceptionBuilderSupplier(IllegalStateException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetPlanEvaluationStatusResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetPlanEvaluationStatusRequest, GetPlanEvaluationStatusResponse>()
                            .withOperationName("GetPlanEvaluationStatus").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetPlanEvaluationStatusRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getPlanEvaluationStatusRequest));
            CompletableFuture<GetPlanEvaluationStatusResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves detailed information about a specific plan execution. You must specify the plan ARN and execution ID.
     * </p>
     *
     * @param getPlanExecutionRequest
     * @return A Java Future containing the result of the GetPlanExecution operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</p>
     *         <p>
     *         HTTP Status Code: 404</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</p>
     *         <p>
     *         HTTP Status Code: 403</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ArcRegionSwitchException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ArcRegionSwitchAsyncClient.GetPlanExecution
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/GetPlanExecution"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetPlanExecutionResponse> getPlanExecution(GetPlanExecutionRequest getPlanExecutionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getPlanExecutionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPlanExecutionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ARC Region switch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPlanExecution");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetPlanExecutionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetPlanExecutionResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "IllegalArgumentException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("IllegalArgumentException").httpStatusCode(400)
                            .exceptionBuilderSupplier(IllegalArgumentException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "IllegalStateException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("IllegalStateException").httpStatusCode(400)
                            .exceptionBuilderSupplier(IllegalStateException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetPlanExecutionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetPlanExecutionRequest, GetPlanExecutionResponse>()
                            .withOperationName("GetPlanExecution").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetPlanExecutionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getPlanExecutionRequest));
            CompletableFuture<GetPlanExecutionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves information about a Region switch plan in a specific Amazon Web Services Region. This operation is
     * useful for getting Region-specific information about a plan.
     * </p>
     *
     * @param getPlanInRegionRequest
     * @return A Java Future containing the result of the GetPlanInRegion operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</p>
     *         <p>
     *         HTTP Status Code: 404</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</p>
     *         <p>
     *         HTTP Status Code: 403</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ArcRegionSwitchException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ArcRegionSwitchAsyncClient.GetPlanInRegion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/GetPlanInRegion"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetPlanInRegionResponse> getPlanInRegion(GetPlanInRegionRequest getPlanInRegionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getPlanInRegionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPlanInRegionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ARC Region switch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPlanInRegion");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetPlanInRegionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetPlanInRegionResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "IllegalArgumentException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("IllegalArgumentException").httpStatusCode(400)
                            .exceptionBuilderSupplier(IllegalArgumentException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "IllegalStateException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("IllegalStateException").httpStatusCode(400)
                            .exceptionBuilderSupplier(IllegalStateException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetPlanInRegionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetPlanInRegionRequest, GetPlanInRegionResponse>()
                            .withOperationName("GetPlanInRegion").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetPlanInRegionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getPlanInRegionRequest));
            CompletableFuture<GetPlanInRegionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the events that occurred during a plan execution. These events provide a detailed timeline of the execution
     * process.
     * </p>
     *
     * @param listPlanExecutionEventsRequest
     * @return A Java Future containing the result of the ListPlanExecutionEvents operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</p>
     *         <p>
     *         HTTP Status Code: 404</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</p>
     *         <p>
     *         HTTP Status Code: 403</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ArcRegionSwitchException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ArcRegionSwitchAsyncClient.ListPlanExecutionEvents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/ListPlanExecutionEvents"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListPlanExecutionEventsResponse> listPlanExecutionEvents(
            ListPlanExecutionEventsRequest listPlanExecutionEventsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPlanExecutionEventsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPlanExecutionEventsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ARC Region switch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPlanExecutionEvents");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPlanExecutionEventsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListPlanExecutionEventsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "IllegalArgumentException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("IllegalArgumentException").httpStatusCode(400)
                            .exceptionBuilderSupplier(IllegalArgumentException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "IllegalStateException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("IllegalStateException").httpStatusCode(400)
                            .exceptionBuilderSupplier(IllegalStateException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListPlanExecutionEventsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPlanExecutionEventsRequest, ListPlanExecutionEventsResponse>()
                            .withOperationName("ListPlanExecutionEvents").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListPlanExecutionEventsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listPlanExecutionEventsRequest));
            CompletableFuture<ListPlanExecutionEventsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the executions of a Region switch plan. This operation returns information about both current and
     * historical executions.
     * </p>
     *
     * @param listPlanExecutionsRequest
     * @return A Java Future containing the result of the ListPlanExecutions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</p>
     *         <p>
     *         HTTP Status Code: 404</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</p>
     *         <p>
     *         HTTP Status Code: 403</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ArcRegionSwitchException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ArcRegionSwitchAsyncClient.ListPlanExecutions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/ListPlanExecutions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListPlanExecutionsResponse> listPlanExecutions(ListPlanExecutionsRequest listPlanExecutionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPlanExecutionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPlanExecutionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ARC Region switch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPlanExecutions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPlanExecutionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListPlanExecutionsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "IllegalArgumentException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("IllegalArgumentException").httpStatusCode(400)
                            .exceptionBuilderSupplier(IllegalArgumentException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "IllegalStateException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("IllegalStateException").httpStatusCode(400)
                            .exceptionBuilderSupplier(IllegalStateException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListPlanExecutionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPlanExecutionsRequest, ListPlanExecutionsResponse>()
                            .withOperationName("ListPlanExecutions").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListPlanExecutionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listPlanExecutionsRequest));
            CompletableFuture<ListPlanExecutionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all Region switch plans in your Amazon Web Services account.
     * </p>
     *
     * @param listPlansRequest
     * @return A Java Future containing the result of the ListPlans operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ArcRegionSwitchException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ArcRegionSwitchAsyncClient.ListPlans
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/ListPlans" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListPlansResponse> listPlans(ListPlansRequest listPlansRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPlansRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPlansRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ARC Region switch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPlans");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPlansResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListPlansResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "IllegalArgumentException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("IllegalArgumentException").httpStatusCode(400)
                            .exceptionBuilderSupplier(IllegalArgumentException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "IllegalStateException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("IllegalStateException").httpStatusCode(400)
                            .exceptionBuilderSupplier(IllegalStateException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListPlansResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPlansRequest, ListPlansResponse>().withOperationName("ListPlans")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListPlansRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(listPlansRequest));
            CompletableFuture<ListPlansResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all Region switch plans in your Amazon Web Services account that are available in the current Amazon Web
     * Services Region.
     * </p>
     *
     * @param listPlansInRegionRequest
     * @return A Java Future containing the result of the ListPlansInRegion operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</p>
     *         <p>
     *         HTTP Status Code: 403</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ArcRegionSwitchException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ArcRegionSwitchAsyncClient.ListPlansInRegion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/ListPlansInRegion"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListPlansInRegionResponse> listPlansInRegion(ListPlansInRegionRequest listPlansInRegionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPlansInRegionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPlansInRegionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ARC Region switch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPlansInRegion");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPlansInRegionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListPlansInRegionResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "IllegalArgumentException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("IllegalArgumentException").httpStatusCode(400)
                            .exceptionBuilderSupplier(IllegalArgumentException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "IllegalStateException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("IllegalStateException").httpStatusCode(400)
                            .exceptionBuilderSupplier(IllegalStateException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListPlansInRegionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPlansInRegionRequest, ListPlansInRegionResponse>()
                            .withOperationName("ListPlansInRegion").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListPlansInRegionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listPlansInRegionRequest));
            CompletableFuture<ListPlansInRegionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List the Amazon Route 53 health checks.
     * </p>
     *
     * @param listRoute53HealthChecksRequest
     * @return A Java Future containing the result of the ListRoute53HealthChecks operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</p>
     *         <p>
     *         HTTP Status Code: 404</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</p>
     *         <p>
     *         HTTP Status Code: 403</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</p>
     *         <p>
     *         HTTP Status Code: 500</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ArcRegionSwitchException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ArcRegionSwitchAsyncClient.ListRoute53HealthChecks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/ListRoute53HealthChecks"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListRoute53HealthChecksResponse> listRoute53HealthChecks(
            ListRoute53HealthChecksRequest listRoute53HealthChecksRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listRoute53HealthChecksRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRoute53HealthChecksRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ARC Region switch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRoute53HealthChecks");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListRoute53HealthChecksResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListRoute53HealthChecksResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "IllegalArgumentException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("IllegalArgumentException").httpStatusCode(400)
                            .exceptionBuilderSupplier(IllegalArgumentException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "IllegalStateException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("IllegalStateException").httpStatusCode(400)
                            .exceptionBuilderSupplier(IllegalStateException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListRoute53HealthChecksResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListRoute53HealthChecksRequest, ListRoute53HealthChecksResponse>()
                            .withOperationName("ListRoute53HealthChecks").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListRoute53HealthChecksRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listRoute53HealthChecksRequest));
            CompletableFuture<ListRoute53HealthChecksResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the tags attached to a Region switch resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</p>
     *         <p>
     *         HTTP Status Code: 404</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</p>
     *         <p>
     *         HTTP Status Code: 500</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ArcRegionSwitchException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ArcRegionSwitchAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ARC Region switch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "IllegalArgumentException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("IllegalArgumentException").httpStatusCode(400)
                            .exceptionBuilderSupplier(IllegalArgumentException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "IllegalStateException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("IllegalStateException").httpStatusCode(400)
                            .exceptionBuilderSupplier(IllegalStateException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts the execution of a Region switch plan. You can execute a plan in either PRACTICE or RECOVERY mode.
     * </p>
     * <p>
     * In PRACTICE mode, the execution simulates the steps without making actual changes to your application's traffic
     * routing. In RECOVERY mode, the execution performs actual changes to shift traffic between Regions.
     * </p>
     *
     * @param startPlanExecutionRequest
     * @return A Java Future containing the result of the StartPlanExecution operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</p>
     *         <p>
     *         HTTP Status Code: 404</li>
     *         <li>IllegalStateException The operation failed because the current state of the resource doesn't allow
     *         the operation to proceed.</p>
     *         <p>
     *         HTTP Status Code: 400</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</p>
     *         <p>
     *         HTTP Status Code: 403</li>
     *         <li>IllegalArgumentException The request processing has an invalid argument.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ArcRegionSwitchException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ArcRegionSwitchAsyncClient.StartPlanExecution
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/StartPlanExecution"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartPlanExecutionResponse> startPlanExecution(StartPlanExecutionRequest startPlanExecutionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startPlanExecutionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startPlanExecutionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ARC Region switch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartPlanExecution");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartPlanExecutionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartPlanExecutionResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "IllegalArgumentException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("IllegalArgumentException").httpStatusCode(400)
                            .exceptionBuilderSupplier(IllegalArgumentException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "IllegalStateException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("IllegalStateException").httpStatusCode(400)
                            .exceptionBuilderSupplier(IllegalStateException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<StartPlanExecutionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartPlanExecutionRequest, StartPlanExecutionResponse>()
                            .withOperationName("StartPlanExecution").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartPlanExecutionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startPlanExecutionRequest));
            CompletableFuture<StartPlanExecutionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds or updates tags for a Region switch resource. You can assign metadata to your resources in the form of tags,
     * which are key-value pairs.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</p>
     *         <p>
     *         HTTP Status Code: 404</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</p>
     *         <p>
     *         HTTP Status Code: 500</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ArcRegionSwitchException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ArcRegionSwitchAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ARC Region switch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "IllegalArgumentException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("IllegalArgumentException").httpStatusCode(400)
                            .exceptionBuilderSupplier(IllegalArgumentException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "IllegalStateException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("IllegalStateException").httpStatusCode(400)
                            .exceptionBuilderSupplier(IllegalStateException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes tags from a Region switch resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</p>
     *         <p>
     *         HTTP Status Code: 404</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</p>
     *         <p>
     *         HTTP Status Code: 500</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ArcRegionSwitchException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ArcRegionSwitchAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ARC Region switch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "IllegalArgumentException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("IllegalArgumentException").httpStatusCode(400)
                            .exceptionBuilderSupplier(IllegalArgumentException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "IllegalStateException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("IllegalStateException").httpStatusCode(400)
                            .exceptionBuilderSupplier(IllegalStateException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an existing Region switch plan. You can modify the plan's description, workflows, execution role,
     * recovery time objective, associated alarms, and triggers.
     * </p>
     *
     * @param updatePlanRequest
     * @return A Java Future containing the result of the UpdatePlan operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</p>
     *         <p>
     *         HTTP Status Code: 404</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ArcRegionSwitchException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ArcRegionSwitchAsyncClient.UpdatePlan
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/UpdatePlan" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdatePlanResponse> updatePlan(UpdatePlanRequest updatePlanRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updatePlanRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updatePlanRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ARC Region switch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdatePlan");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdatePlanResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdatePlanResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "IllegalArgumentException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("IllegalArgumentException").httpStatusCode(400)
                            .exceptionBuilderSupplier(IllegalArgumentException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "IllegalStateException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("IllegalStateException").httpStatusCode(400)
                            .exceptionBuilderSupplier(IllegalStateException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UpdatePlanResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdatePlanRequest, UpdatePlanResponse>().withOperationName("UpdatePlan")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdatePlanRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updatePlanRequest));
            CompletableFuture<UpdatePlanResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an in-progress plan execution. This operation allows you to modify certain aspects of the execution, such
     * as adding a comment or changing the action.
     * </p>
     *
     * @param updatePlanExecutionRequest
     * @return A Java Future containing the result of the UpdatePlanExecution operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</p>
     *         <p>
     *         HTTP Status Code: 404</li>
     *         <li>IllegalStateException The operation failed because the current state of the resource doesn't allow
     *         the operation to proceed.</p>
     *         <p>
     *         HTTP Status Code: 400</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</p>
     *         <p>
     *         HTTP Status Code: 403</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ArcRegionSwitchException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ArcRegionSwitchAsyncClient.UpdatePlanExecution
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/UpdatePlanExecution"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdatePlanExecutionResponse> updatePlanExecution(
            UpdatePlanExecutionRequest updatePlanExecutionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updatePlanExecutionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updatePlanExecutionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ARC Region switch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdatePlanExecution");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdatePlanExecutionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdatePlanExecutionResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "IllegalArgumentException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("IllegalArgumentException").httpStatusCode(400)
                            .exceptionBuilderSupplier(IllegalArgumentException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "IllegalStateException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("IllegalStateException").httpStatusCode(400)
                            .exceptionBuilderSupplier(IllegalStateException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UpdatePlanExecutionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdatePlanExecutionRequest, UpdatePlanExecutionResponse>()
                            .withOperationName("UpdatePlanExecution").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdatePlanExecutionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updatePlanExecutionRequest));
            CompletableFuture<UpdatePlanExecutionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a specific step in an in-progress plan execution. This operation allows you to modify the step's comment
     * or action.
     * </p>
     *
     * @param updatePlanExecutionStepRequest
     * @return A Java Future containing the result of the UpdatePlanExecutionStep operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</p>
     *         <p>
     *         HTTP Status Code: 404</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</p>
     *         <p>
     *         HTTP Status Code: 403</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ArcRegionSwitchException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ArcRegionSwitchAsyncClient.UpdatePlanExecutionStep
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/UpdatePlanExecutionStep"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdatePlanExecutionStepResponse> updatePlanExecutionStep(
            UpdatePlanExecutionStepRequest updatePlanExecutionStepRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updatePlanExecutionStepRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updatePlanExecutionStepRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ARC Region switch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdatePlanExecutionStep");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdatePlanExecutionStepResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdatePlanExecutionStepResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "IllegalArgumentException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("IllegalArgumentException").httpStatusCode(400)
                            .exceptionBuilderSupplier(IllegalArgumentException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "IllegalStateException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("IllegalStateException").httpStatusCode(400)
                            .exceptionBuilderSupplier(IllegalStateException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UpdatePlanExecutionStepResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdatePlanExecutionStepRequest, UpdatePlanExecutionStepResponse>()
                            .withOperationName("UpdatePlanExecutionStep").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdatePlanExecutionStepRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updatePlanExecutionStepRequest));
            CompletableFuture<UpdatePlanExecutionStepResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public ArcRegionSwitchAsyncWaiter waiter() {
        return ArcRegionSwitchAsyncWaiter.builder().client(this).scheduledExecutorService(executorService).build();
    }

    @Override
    public final ArcRegionSwitchServiceClientConfiguration serviceClientConfiguration() {
        return new ArcRegionSwitchServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ArcRegionSwitchException::builder).protocol(AwsJsonProtocol.SMITHY_RPC_V2_CBOR)
                .protocolVersion("1.0");
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        ArcRegionSwitchServiceClientConfigurationBuilder serviceConfigBuilder = new ArcRegionSwitchServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
