/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.arcregionswitch;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsClient;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.regions.ServiceMetadata;
import software.amazon.awssdk.services.arcregionswitch.model.AccessDeniedException;
import software.amazon.awssdk.services.arcregionswitch.model.ApprovePlanExecutionStepRequest;
import software.amazon.awssdk.services.arcregionswitch.model.ApprovePlanExecutionStepResponse;
import software.amazon.awssdk.services.arcregionswitch.model.ArcRegionSwitchException;
import software.amazon.awssdk.services.arcregionswitch.model.CancelPlanExecutionRequest;
import software.amazon.awssdk.services.arcregionswitch.model.CancelPlanExecutionResponse;
import software.amazon.awssdk.services.arcregionswitch.model.CreatePlanRequest;
import software.amazon.awssdk.services.arcregionswitch.model.CreatePlanResponse;
import software.amazon.awssdk.services.arcregionswitch.model.DeletePlanRequest;
import software.amazon.awssdk.services.arcregionswitch.model.DeletePlanResponse;
import software.amazon.awssdk.services.arcregionswitch.model.GetPlanEvaluationStatusRequest;
import software.amazon.awssdk.services.arcregionswitch.model.GetPlanEvaluationStatusResponse;
import software.amazon.awssdk.services.arcregionswitch.model.GetPlanExecutionRequest;
import software.amazon.awssdk.services.arcregionswitch.model.GetPlanExecutionResponse;
import software.amazon.awssdk.services.arcregionswitch.model.GetPlanInRegionRequest;
import software.amazon.awssdk.services.arcregionswitch.model.GetPlanInRegionResponse;
import software.amazon.awssdk.services.arcregionswitch.model.GetPlanRequest;
import software.amazon.awssdk.services.arcregionswitch.model.GetPlanResponse;
import software.amazon.awssdk.services.arcregionswitch.model.IllegalArgumentException;
import software.amazon.awssdk.services.arcregionswitch.model.IllegalStateException;
import software.amazon.awssdk.services.arcregionswitch.model.InternalServerException;
import software.amazon.awssdk.services.arcregionswitch.model.ListPlanExecutionEventsRequest;
import software.amazon.awssdk.services.arcregionswitch.model.ListPlanExecutionEventsResponse;
import software.amazon.awssdk.services.arcregionswitch.model.ListPlanExecutionsRequest;
import software.amazon.awssdk.services.arcregionswitch.model.ListPlanExecutionsResponse;
import software.amazon.awssdk.services.arcregionswitch.model.ListPlansInRegionRequest;
import software.amazon.awssdk.services.arcregionswitch.model.ListPlansInRegionResponse;
import software.amazon.awssdk.services.arcregionswitch.model.ListPlansRequest;
import software.amazon.awssdk.services.arcregionswitch.model.ListPlansResponse;
import software.amazon.awssdk.services.arcregionswitch.model.ListRoute53HealthChecksRequest;
import software.amazon.awssdk.services.arcregionswitch.model.ListRoute53HealthChecksResponse;
import software.amazon.awssdk.services.arcregionswitch.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.arcregionswitch.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.arcregionswitch.model.ResourceNotFoundException;
import software.amazon.awssdk.services.arcregionswitch.model.StartPlanExecutionRequest;
import software.amazon.awssdk.services.arcregionswitch.model.StartPlanExecutionResponse;
import software.amazon.awssdk.services.arcregionswitch.model.TagResourceRequest;
import software.amazon.awssdk.services.arcregionswitch.model.TagResourceResponse;
import software.amazon.awssdk.services.arcregionswitch.model.UntagResourceRequest;
import software.amazon.awssdk.services.arcregionswitch.model.UntagResourceResponse;
import software.amazon.awssdk.services.arcregionswitch.model.UpdatePlanExecutionRequest;
import software.amazon.awssdk.services.arcregionswitch.model.UpdatePlanExecutionResponse;
import software.amazon.awssdk.services.arcregionswitch.model.UpdatePlanExecutionStepRequest;
import software.amazon.awssdk.services.arcregionswitch.model.UpdatePlanExecutionStepResponse;
import software.amazon.awssdk.services.arcregionswitch.model.UpdatePlanRequest;
import software.amazon.awssdk.services.arcregionswitch.model.UpdatePlanResponse;
import software.amazon.awssdk.services.arcregionswitch.paginators.GetPlanEvaluationStatusIterable;
import software.amazon.awssdk.services.arcregionswitch.paginators.GetPlanExecutionIterable;
import software.amazon.awssdk.services.arcregionswitch.paginators.ListPlanExecutionEventsIterable;
import software.amazon.awssdk.services.arcregionswitch.paginators.ListPlanExecutionsIterable;
import software.amazon.awssdk.services.arcregionswitch.paginators.ListPlansInRegionIterable;
import software.amazon.awssdk.services.arcregionswitch.paginators.ListPlansIterable;
import software.amazon.awssdk.services.arcregionswitch.paginators.ListRoute53HealthChecksIterable;
import software.amazon.awssdk.services.arcregionswitch.waiters.ArcRegionSwitchWaiter;

/**
 * Service client for accessing ARC - Region switch. This can be created using the static {@link #builder()} method.
 *
 * <p>
 * Amazon Application Recovery Controller (ARC) Region switch helps you to quickly and reliably shift traffic away from
 * an impaired Amazon Web Services Region to a healthy Region. With Region switch, you can create plans that define the
 * steps to shift traffic for your application from one Amazon Web Services Region to another. You can test your plans
 * in practice mode before using them in a real recovery scenario.
 * </p>
 * <p>
 * Region switch provides a structured approach to multi-Region failover, helping you to meet your recovery time
 * objectives (RTOs) and maintain business continuity during regional disruptions.
 * </p>
 * <p>
 * For more information, see <a href="https://docs.aws.amazon.com/r53recovery/latest/dg/region-switch.html">Region
 * switch in ARC</a> in the <i>Amazon Application Recovery Controller User Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
public interface ArcRegionSwitchClient extends AwsClient {
    String SERVICE_NAME = "arc-region-switch";

    /**
     * Value for looking up the service's metadata from the
     * {@link software.amazon.awssdk.regions.ServiceMetadataProvider}.
     */
    String SERVICE_METADATA_ID = "arc-region-switch";

    /**
     * <p>
     * Approves a step in a plan execution that requires manual approval. When you create a plan, you can include
     * approval steps that require manual intervention before the execution can proceed. This operation allows you to
     * provide that approval.
     * </p>
     * <p>
     * You must specify the plan ARN, execution ID, step name, and approval status. You can also provide an optional
     * comment explaining the approval decision.
     * </p>
     *
     * @param approvePlanExecutionStepRequest
     * @return Result of the ApprovePlanExecutionStep operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.</p>
     *         <p>
     *         HTTP Status Code: 404
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     *         </p>
     *         <p>
     *         HTTP Status Code: 403
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcRegionSwitchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcRegionSwitchClient.ApprovePlanExecutionStep
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/ApprovePlanExecutionStep"
     *      target="_top">AWS API Documentation</a>
     */
    default ApprovePlanExecutionStepResponse approvePlanExecutionStep(
            ApprovePlanExecutionStepRequest approvePlanExecutionStepRequest) throws ResourceNotFoundException,
            AccessDeniedException, AwsServiceException, SdkClientException, ArcRegionSwitchException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Approves a step in a plan execution that requires manual approval. When you create a plan, you can include
     * approval steps that require manual intervention before the execution can proceed. This operation allows you to
     * provide that approval.
     * </p>
     * <p>
     * You must specify the plan ARN, execution ID, step name, and approval status. You can also provide an optional
     * comment explaining the approval decision.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ApprovePlanExecutionStepRequest.Builder} avoiding
     * the need to create one manually via {@link ApprovePlanExecutionStepRequest#builder()}
     * </p>
     *
     * @param approvePlanExecutionStepRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.arcregionswitch.model.ApprovePlanExecutionStepRequest.Builder} to
     *        create a request.
     * @return Result of the ApprovePlanExecutionStep operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.</p>
     *         <p>
     *         HTTP Status Code: 404
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     *         </p>
     *         <p>
     *         HTTP Status Code: 403
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcRegionSwitchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcRegionSwitchClient.ApprovePlanExecutionStep
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/ApprovePlanExecutionStep"
     *      target="_top">AWS API Documentation</a>
     */
    default ApprovePlanExecutionStepResponse approvePlanExecutionStep(
            Consumer<ApprovePlanExecutionStepRequest.Builder> approvePlanExecutionStepRequest) throws ResourceNotFoundException,
            AccessDeniedException, AwsServiceException, SdkClientException, ArcRegionSwitchException {
        return approvePlanExecutionStep(ApprovePlanExecutionStepRequest.builder().applyMutation(approvePlanExecutionStepRequest)
                .build());
    }

    /**
     * <p>
     * Cancels an in-progress plan execution. This operation stops the execution of the plan and prevents any further
     * steps from being processed.
     * </p>
     * <p>
     * You must specify the plan ARN and execution ID. You can also provide an optional comment explaining why the
     * execution was canceled.
     * </p>
     *
     * @param cancelPlanExecutionRequest
     * @return Result of the CancelPlanExecution operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.</p>
     *         <p>
     *         HTTP Status Code: 404
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     *         </p>
     *         <p>
     *         HTTP Status Code: 403
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcRegionSwitchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcRegionSwitchClient.CancelPlanExecution
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/CancelPlanExecution"
     *      target="_top">AWS API Documentation</a>
     */
    default CancelPlanExecutionResponse cancelPlanExecution(CancelPlanExecutionRequest cancelPlanExecutionRequest)
            throws ResourceNotFoundException, AccessDeniedException, AwsServiceException, SdkClientException,
            ArcRegionSwitchException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Cancels an in-progress plan execution. This operation stops the execution of the plan and prevents any further
     * steps from being processed.
     * </p>
     * <p>
     * You must specify the plan ARN and execution ID. You can also provide an optional comment explaining why the
     * execution was canceled.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CancelPlanExecutionRequest.Builder} avoiding the
     * need to create one manually via {@link CancelPlanExecutionRequest#builder()}
     * </p>
     *
     * @param cancelPlanExecutionRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.arcregionswitch.model.CancelPlanExecutionRequest.Builder} to create
     *        a request.
     * @return Result of the CancelPlanExecution operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.</p>
     *         <p>
     *         HTTP Status Code: 404
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     *         </p>
     *         <p>
     *         HTTP Status Code: 403
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcRegionSwitchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcRegionSwitchClient.CancelPlanExecution
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/CancelPlanExecution"
     *      target="_top">AWS API Documentation</a>
     */
    default CancelPlanExecutionResponse cancelPlanExecution(
            Consumer<CancelPlanExecutionRequest.Builder> cancelPlanExecutionRequest) throws ResourceNotFoundException,
            AccessDeniedException, AwsServiceException, SdkClientException, ArcRegionSwitchException {
        return cancelPlanExecution(CancelPlanExecutionRequest.builder().applyMutation(cancelPlanExecutionRequest).build());
    }

    /**
     * <p>
     * Creates a new Region switch plan. A plan defines the steps required to shift traffic from one Amazon Web Services
     * Region to another.
     * </p>
     * <p>
     * You must specify a name for the plan, the primary Region, and at least one additional Region. You can also
     * provide a description, execution role, recovery time objective, associated alarms, triggers, and workflows that
     * define the steps to execute during a Region switch.
     * </p>
     *
     * @param createPlanRequest
     * @return Result of the CreatePlan operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcRegionSwitchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcRegionSwitchClient.CreatePlan
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/CreatePlan" target="_top">AWS
     *      API Documentation</a>
     */
    default CreatePlanResponse createPlan(CreatePlanRequest createPlanRequest) throws AwsServiceException, SdkClientException,
            ArcRegionSwitchException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates a new Region switch plan. A plan defines the steps required to shift traffic from one Amazon Web Services
     * Region to another.
     * </p>
     * <p>
     * You must specify a name for the plan, the primary Region, and at least one additional Region. You can also
     * provide a description, execution role, recovery time objective, associated alarms, triggers, and workflows that
     * define the steps to execute during a Region switch.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreatePlanRequest.Builder} avoiding the need to
     * create one manually via {@link CreatePlanRequest#builder()}
     * </p>
     *
     * @param createPlanRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.arcregionswitch.model.CreatePlanRequest.Builder} to create a
     *        request.
     * @return Result of the CreatePlan operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcRegionSwitchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcRegionSwitchClient.CreatePlan
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/CreatePlan" target="_top">AWS
     *      API Documentation</a>
     */
    default CreatePlanResponse createPlan(Consumer<CreatePlanRequest.Builder> createPlanRequest) throws AwsServiceException,
            SdkClientException, ArcRegionSwitchException {
        return createPlan(CreatePlanRequest.builder().applyMutation(createPlanRequest).build());
    }

    /**
     * <p>
     * Deletes a Region switch plan. You must specify the ARN of the plan to delete.
     * </p>
     * <p>
     * You cannot delete a plan that has an active execution in progress.
     * </p>
     *
     * @param deletePlanRequest
     * @return Result of the DeletePlan operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.</p>
     *         <p>
     *         HTTP Status Code: 404
     * @throws IllegalStateException
     *         The operation failed because the current state of the resource doesn't allow the operation to proceed.
     *         </p>
     *         <p>
     *         HTTP Status Code: 400
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcRegionSwitchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcRegionSwitchClient.DeletePlan
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/DeletePlan" target="_top">AWS
     *      API Documentation</a>
     */
    default DeletePlanResponse deletePlan(DeletePlanRequest deletePlanRequest) throws ResourceNotFoundException,
            IllegalStateException, AwsServiceException, SdkClientException, ArcRegionSwitchException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes a Region switch plan. You must specify the ARN of the plan to delete.
     * </p>
     * <p>
     * You cannot delete a plan that has an active execution in progress.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeletePlanRequest.Builder} avoiding the need to
     * create one manually via {@link DeletePlanRequest#builder()}
     * </p>
     *
     * @param deletePlanRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.arcregionswitch.model.DeletePlanRequest.Builder} to create a
     *        request.
     * @return Result of the DeletePlan operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.</p>
     *         <p>
     *         HTTP Status Code: 404
     * @throws IllegalStateException
     *         The operation failed because the current state of the resource doesn't allow the operation to proceed.
     *         </p>
     *         <p>
     *         HTTP Status Code: 400
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcRegionSwitchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcRegionSwitchClient.DeletePlan
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/DeletePlan" target="_top">AWS
     *      API Documentation</a>
     */
    default DeletePlanResponse deletePlan(Consumer<DeletePlanRequest.Builder> deletePlanRequest)
            throws ResourceNotFoundException, IllegalStateException, AwsServiceException, SdkClientException,
            ArcRegionSwitchException {
        return deletePlan(DeletePlanRequest.builder().applyMutation(deletePlanRequest).build());
    }

    /**
     * <p>
     * Retrieves detailed information about a Region switch plan. You must specify the ARN of the plan.
     * </p>
     *
     * @param getPlanRequest
     * @return Result of the GetPlan operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.</p>
     *         <p>
     *         HTTP Status Code: 404
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcRegionSwitchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcRegionSwitchClient.GetPlan
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/GetPlan" target="_top">AWS API
     *      Documentation</a>
     */
    default GetPlanResponse getPlan(GetPlanRequest getPlanRequest) throws ResourceNotFoundException, AwsServiceException,
            SdkClientException, ArcRegionSwitchException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Retrieves detailed information about a Region switch plan. You must specify the ARN of the plan.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetPlanRequest.Builder} avoiding the need to create
     * one manually via {@link GetPlanRequest#builder()}
     * </p>
     *
     * @param getPlanRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.arcregionswitch.model.GetPlanRequest.Builder} to create a request.
     * @return Result of the GetPlan operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.</p>
     *         <p>
     *         HTTP Status Code: 404
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcRegionSwitchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcRegionSwitchClient.GetPlan
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/GetPlan" target="_top">AWS API
     *      Documentation</a>
     */
    default GetPlanResponse getPlan(Consumer<GetPlanRequest.Builder> getPlanRequest) throws ResourceNotFoundException,
            AwsServiceException, SdkClientException, ArcRegionSwitchException {
        return getPlan(GetPlanRequest.builder().applyMutation(getPlanRequest).build());
    }

    /**
     * <p>
     * Retrieves the evaluation status of a Region switch plan. The evaluation status provides information about the
     * last time the plan was evaluated and any warnings or issues detected.
     * </p>
     *
     * @param getPlanEvaluationStatusRequest
     * @return Result of the GetPlanEvaluationStatus operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.</p>
     *         <p>
     *         HTTP Status Code: 404
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     *         </p>
     *         <p>
     *         HTTP Status Code: 403
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcRegionSwitchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcRegionSwitchClient.GetPlanEvaluationStatus
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/GetPlanEvaluationStatus"
     *      target="_top">AWS API Documentation</a>
     */
    default GetPlanEvaluationStatusResponse getPlanEvaluationStatus(GetPlanEvaluationStatusRequest getPlanEvaluationStatusRequest)
            throws ResourceNotFoundException, AccessDeniedException, AwsServiceException, SdkClientException,
            ArcRegionSwitchException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Retrieves the evaluation status of a Region switch plan. The evaluation status provides information about the
     * last time the plan was evaluated and any warnings or issues detected.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetPlanEvaluationStatusRequest.Builder} avoiding
     * the need to create one manually via {@link GetPlanEvaluationStatusRequest#builder()}
     * </p>
     *
     * @param getPlanEvaluationStatusRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.arcregionswitch.model.GetPlanEvaluationStatusRequest.Builder} to
     *        create a request.
     * @return Result of the GetPlanEvaluationStatus operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.</p>
     *         <p>
     *         HTTP Status Code: 404
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     *         </p>
     *         <p>
     *         HTTP Status Code: 403
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcRegionSwitchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcRegionSwitchClient.GetPlanEvaluationStatus
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/GetPlanEvaluationStatus"
     *      target="_top">AWS API Documentation</a>
     */
    default GetPlanEvaluationStatusResponse getPlanEvaluationStatus(
            Consumer<GetPlanEvaluationStatusRequest.Builder> getPlanEvaluationStatusRequest) throws ResourceNotFoundException,
            AccessDeniedException, AwsServiceException, SdkClientException, ArcRegionSwitchException {
        return getPlanEvaluationStatus(GetPlanEvaluationStatusRequest.builder().applyMutation(getPlanEvaluationStatusRequest)
                .build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #getPlanEvaluationStatus(software.amazon.awssdk.services.arcregionswitch.model.GetPlanEvaluationStatusRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.arcregionswitch.paginators.GetPlanEvaluationStatusIterable responses = client.getPlanEvaluationStatusPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.arcregionswitch.paginators.GetPlanEvaluationStatusIterable responses = client
     *             .getPlanEvaluationStatusPaginator(request);
     *     for (software.amazon.awssdk.services.arcregionswitch.model.GetPlanEvaluationStatusResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.arcregionswitch.paginators.GetPlanEvaluationStatusIterable responses = client.getPlanEvaluationStatusPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getPlanEvaluationStatus(software.amazon.awssdk.services.arcregionswitch.model.GetPlanEvaluationStatusRequest)}
     * operation.</b>
     * </p>
     *
     * @param getPlanEvaluationStatusRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.</p>
     *         <p>
     *         HTTP Status Code: 404
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     *         </p>
     *         <p>
     *         HTTP Status Code: 403
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcRegionSwitchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcRegionSwitchClient.GetPlanEvaluationStatus
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/GetPlanEvaluationStatus"
     *      target="_top">AWS API Documentation</a>
     */
    default GetPlanEvaluationStatusIterable getPlanEvaluationStatusPaginator(
            GetPlanEvaluationStatusRequest getPlanEvaluationStatusRequest) throws ResourceNotFoundException,
            AccessDeniedException, AwsServiceException, SdkClientException, ArcRegionSwitchException {
        return new GetPlanEvaluationStatusIterable(this, getPlanEvaluationStatusRequest);
    }

    /**
     * <p>
     * This is a variant of
     * {@link #getPlanEvaluationStatus(software.amazon.awssdk.services.arcregionswitch.model.GetPlanEvaluationStatusRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.arcregionswitch.paginators.GetPlanEvaluationStatusIterable responses = client.getPlanEvaluationStatusPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.arcregionswitch.paginators.GetPlanEvaluationStatusIterable responses = client
     *             .getPlanEvaluationStatusPaginator(request);
     *     for (software.amazon.awssdk.services.arcregionswitch.model.GetPlanEvaluationStatusResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.arcregionswitch.paginators.GetPlanEvaluationStatusIterable responses = client.getPlanEvaluationStatusPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getPlanEvaluationStatus(software.amazon.awssdk.services.arcregionswitch.model.GetPlanEvaluationStatusRequest)}
     * operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetPlanEvaluationStatusRequest.Builder} avoiding
     * the need to create one manually via {@link GetPlanEvaluationStatusRequest#builder()}
     * </p>
     *
     * @param getPlanEvaluationStatusRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.arcregionswitch.model.GetPlanEvaluationStatusRequest.Builder} to
     *        create a request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.</p>
     *         <p>
     *         HTTP Status Code: 404
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     *         </p>
     *         <p>
     *         HTTP Status Code: 403
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcRegionSwitchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcRegionSwitchClient.GetPlanEvaluationStatus
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/GetPlanEvaluationStatus"
     *      target="_top">AWS API Documentation</a>
     */
    default GetPlanEvaluationStatusIterable getPlanEvaluationStatusPaginator(
            Consumer<GetPlanEvaluationStatusRequest.Builder> getPlanEvaluationStatusRequest) throws ResourceNotFoundException,
            AccessDeniedException, AwsServiceException, SdkClientException, ArcRegionSwitchException {
        return getPlanEvaluationStatusPaginator(GetPlanEvaluationStatusRequest.builder()
                .applyMutation(getPlanEvaluationStatusRequest).build());
    }

    /**
     * <p>
     * Retrieves detailed information about a specific plan execution. You must specify the plan ARN and execution ID.
     * </p>
     *
     * @param getPlanExecutionRequest
     * @return Result of the GetPlanExecution operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.</p>
     *         <p>
     *         HTTP Status Code: 404
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     *         </p>
     *         <p>
     *         HTTP Status Code: 403
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcRegionSwitchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcRegionSwitchClient.GetPlanExecution
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/GetPlanExecution"
     *      target="_top">AWS API Documentation</a>
     */
    default GetPlanExecutionResponse getPlanExecution(GetPlanExecutionRequest getPlanExecutionRequest)
            throws ResourceNotFoundException, AccessDeniedException, AwsServiceException, SdkClientException,
            ArcRegionSwitchException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Retrieves detailed information about a specific plan execution. You must specify the plan ARN and execution ID.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetPlanExecutionRequest.Builder} avoiding the need
     * to create one manually via {@link GetPlanExecutionRequest#builder()}
     * </p>
     *
     * @param getPlanExecutionRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.arcregionswitch.model.GetPlanExecutionRequest.Builder} to create a
     *        request.
     * @return Result of the GetPlanExecution operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.</p>
     *         <p>
     *         HTTP Status Code: 404
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     *         </p>
     *         <p>
     *         HTTP Status Code: 403
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcRegionSwitchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcRegionSwitchClient.GetPlanExecution
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/GetPlanExecution"
     *      target="_top">AWS API Documentation</a>
     */
    default GetPlanExecutionResponse getPlanExecution(Consumer<GetPlanExecutionRequest.Builder> getPlanExecutionRequest)
            throws ResourceNotFoundException, AccessDeniedException, AwsServiceException, SdkClientException,
            ArcRegionSwitchException {
        return getPlanExecution(GetPlanExecutionRequest.builder().applyMutation(getPlanExecutionRequest).build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #getPlanExecution(software.amazon.awssdk.services.arcregionswitch.model.GetPlanExecutionRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.arcregionswitch.paginators.GetPlanExecutionIterable responses = client.getPlanExecutionPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.arcregionswitch.paginators.GetPlanExecutionIterable responses = client
     *             .getPlanExecutionPaginator(request);
     *     for (software.amazon.awssdk.services.arcregionswitch.model.GetPlanExecutionResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.arcregionswitch.paginators.GetPlanExecutionIterable responses = client.getPlanExecutionPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getPlanExecution(software.amazon.awssdk.services.arcregionswitch.model.GetPlanExecutionRequest)}
     * operation.</b>
     * </p>
     *
     * @param getPlanExecutionRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.</p>
     *         <p>
     *         HTTP Status Code: 404
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     *         </p>
     *         <p>
     *         HTTP Status Code: 403
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcRegionSwitchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcRegionSwitchClient.GetPlanExecution
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/GetPlanExecution"
     *      target="_top">AWS API Documentation</a>
     */
    default GetPlanExecutionIterable getPlanExecutionPaginator(GetPlanExecutionRequest getPlanExecutionRequest)
            throws ResourceNotFoundException, AccessDeniedException, AwsServiceException, SdkClientException,
            ArcRegionSwitchException {
        return new GetPlanExecutionIterable(this, getPlanExecutionRequest);
    }

    /**
     * <p>
     * This is a variant of
     * {@link #getPlanExecution(software.amazon.awssdk.services.arcregionswitch.model.GetPlanExecutionRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.arcregionswitch.paginators.GetPlanExecutionIterable responses = client.getPlanExecutionPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.arcregionswitch.paginators.GetPlanExecutionIterable responses = client
     *             .getPlanExecutionPaginator(request);
     *     for (software.amazon.awssdk.services.arcregionswitch.model.GetPlanExecutionResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.arcregionswitch.paginators.GetPlanExecutionIterable responses = client.getPlanExecutionPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getPlanExecution(software.amazon.awssdk.services.arcregionswitch.model.GetPlanExecutionRequest)}
     * operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetPlanExecutionRequest.Builder} avoiding the need
     * to create one manually via {@link GetPlanExecutionRequest#builder()}
     * </p>
     *
     * @param getPlanExecutionRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.arcregionswitch.model.GetPlanExecutionRequest.Builder} to create a
     *        request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.</p>
     *         <p>
     *         HTTP Status Code: 404
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     *         </p>
     *         <p>
     *         HTTP Status Code: 403
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcRegionSwitchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcRegionSwitchClient.GetPlanExecution
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/GetPlanExecution"
     *      target="_top">AWS API Documentation</a>
     */
    default GetPlanExecutionIterable getPlanExecutionPaginator(Consumer<GetPlanExecutionRequest.Builder> getPlanExecutionRequest)
            throws ResourceNotFoundException, AccessDeniedException, AwsServiceException, SdkClientException,
            ArcRegionSwitchException {
        return getPlanExecutionPaginator(GetPlanExecutionRequest.builder().applyMutation(getPlanExecutionRequest).build());
    }

    /**
     * <p>
     * Retrieves information about a Region switch plan in a specific Amazon Web Services Region. This operation is
     * useful for getting Region-specific information about a plan.
     * </p>
     *
     * @param getPlanInRegionRequest
     * @return Result of the GetPlanInRegion operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.</p>
     *         <p>
     *         HTTP Status Code: 404
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     *         </p>
     *         <p>
     *         HTTP Status Code: 403
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcRegionSwitchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcRegionSwitchClient.GetPlanInRegion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/GetPlanInRegion"
     *      target="_top">AWS API Documentation</a>
     */
    default GetPlanInRegionResponse getPlanInRegion(GetPlanInRegionRequest getPlanInRegionRequest)
            throws ResourceNotFoundException, AccessDeniedException, AwsServiceException, SdkClientException,
            ArcRegionSwitchException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Retrieves information about a Region switch plan in a specific Amazon Web Services Region. This operation is
     * useful for getting Region-specific information about a plan.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetPlanInRegionRequest.Builder} avoiding the need
     * to create one manually via {@link GetPlanInRegionRequest#builder()}
     * </p>
     *
     * @param getPlanInRegionRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.arcregionswitch.model.GetPlanInRegionRequest.Builder} to create a
     *        request.
     * @return Result of the GetPlanInRegion operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.</p>
     *         <p>
     *         HTTP Status Code: 404
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     *         </p>
     *         <p>
     *         HTTP Status Code: 403
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcRegionSwitchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcRegionSwitchClient.GetPlanInRegion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/GetPlanInRegion"
     *      target="_top">AWS API Documentation</a>
     */
    default GetPlanInRegionResponse getPlanInRegion(Consumer<GetPlanInRegionRequest.Builder> getPlanInRegionRequest)
            throws ResourceNotFoundException, AccessDeniedException, AwsServiceException, SdkClientException,
            ArcRegionSwitchException {
        return getPlanInRegion(GetPlanInRegionRequest.builder().applyMutation(getPlanInRegionRequest).build());
    }

    /**
     * <p>
     * Lists the events that occurred during a plan execution. These events provide a detailed timeline of the execution
     * process.
     * </p>
     *
     * @param listPlanExecutionEventsRequest
     * @return Result of the ListPlanExecutionEvents operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.</p>
     *         <p>
     *         HTTP Status Code: 404
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     *         </p>
     *         <p>
     *         HTTP Status Code: 403
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcRegionSwitchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcRegionSwitchClient.ListPlanExecutionEvents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/ListPlanExecutionEvents"
     *      target="_top">AWS API Documentation</a>
     */
    default ListPlanExecutionEventsResponse listPlanExecutionEvents(ListPlanExecutionEventsRequest listPlanExecutionEventsRequest)
            throws ResourceNotFoundException, AccessDeniedException, AwsServiceException, SdkClientException,
            ArcRegionSwitchException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists the events that occurred during a plan execution. These events provide a detailed timeline of the execution
     * process.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListPlanExecutionEventsRequest.Builder} avoiding
     * the need to create one manually via {@link ListPlanExecutionEventsRequest#builder()}
     * </p>
     *
     * @param listPlanExecutionEventsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.arcregionswitch.model.ListPlanExecutionEventsRequest.Builder} to
     *        create a request.
     * @return Result of the ListPlanExecutionEvents operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.</p>
     *         <p>
     *         HTTP Status Code: 404
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     *         </p>
     *         <p>
     *         HTTP Status Code: 403
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcRegionSwitchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcRegionSwitchClient.ListPlanExecutionEvents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/ListPlanExecutionEvents"
     *      target="_top">AWS API Documentation</a>
     */
    default ListPlanExecutionEventsResponse listPlanExecutionEvents(
            Consumer<ListPlanExecutionEventsRequest.Builder> listPlanExecutionEventsRequest) throws ResourceNotFoundException,
            AccessDeniedException, AwsServiceException, SdkClientException, ArcRegionSwitchException {
        return listPlanExecutionEvents(ListPlanExecutionEventsRequest.builder().applyMutation(listPlanExecutionEventsRequest)
                .build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listPlanExecutionEvents(software.amazon.awssdk.services.arcregionswitch.model.ListPlanExecutionEventsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.arcregionswitch.paginators.ListPlanExecutionEventsIterable responses = client.listPlanExecutionEventsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.arcregionswitch.paginators.ListPlanExecutionEventsIterable responses = client
     *             .listPlanExecutionEventsPaginator(request);
     *     for (software.amazon.awssdk.services.arcregionswitch.model.ListPlanExecutionEventsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.arcregionswitch.paginators.ListPlanExecutionEventsIterable responses = client.listPlanExecutionEventsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listPlanExecutionEvents(software.amazon.awssdk.services.arcregionswitch.model.ListPlanExecutionEventsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listPlanExecutionEventsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.</p>
     *         <p>
     *         HTTP Status Code: 404
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     *         </p>
     *         <p>
     *         HTTP Status Code: 403
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcRegionSwitchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcRegionSwitchClient.ListPlanExecutionEvents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/ListPlanExecutionEvents"
     *      target="_top">AWS API Documentation</a>
     */
    default ListPlanExecutionEventsIterable listPlanExecutionEventsPaginator(
            ListPlanExecutionEventsRequest listPlanExecutionEventsRequest) throws ResourceNotFoundException,
            AccessDeniedException, AwsServiceException, SdkClientException, ArcRegionSwitchException {
        return new ListPlanExecutionEventsIterable(this, listPlanExecutionEventsRequest);
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listPlanExecutionEvents(software.amazon.awssdk.services.arcregionswitch.model.ListPlanExecutionEventsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.arcregionswitch.paginators.ListPlanExecutionEventsIterable responses = client.listPlanExecutionEventsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.arcregionswitch.paginators.ListPlanExecutionEventsIterable responses = client
     *             .listPlanExecutionEventsPaginator(request);
     *     for (software.amazon.awssdk.services.arcregionswitch.model.ListPlanExecutionEventsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.arcregionswitch.paginators.ListPlanExecutionEventsIterable responses = client.listPlanExecutionEventsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listPlanExecutionEvents(software.amazon.awssdk.services.arcregionswitch.model.ListPlanExecutionEventsRequest)}
     * operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListPlanExecutionEventsRequest.Builder} avoiding
     * the need to create one manually via {@link ListPlanExecutionEventsRequest#builder()}
     * </p>
     *
     * @param listPlanExecutionEventsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.arcregionswitch.model.ListPlanExecutionEventsRequest.Builder} to
     *        create a request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.</p>
     *         <p>
     *         HTTP Status Code: 404
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     *         </p>
     *         <p>
     *         HTTP Status Code: 403
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcRegionSwitchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcRegionSwitchClient.ListPlanExecutionEvents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/ListPlanExecutionEvents"
     *      target="_top">AWS API Documentation</a>
     */
    default ListPlanExecutionEventsIterable listPlanExecutionEventsPaginator(
            Consumer<ListPlanExecutionEventsRequest.Builder> listPlanExecutionEventsRequest) throws ResourceNotFoundException,
            AccessDeniedException, AwsServiceException, SdkClientException, ArcRegionSwitchException {
        return listPlanExecutionEventsPaginator(ListPlanExecutionEventsRequest.builder()
                .applyMutation(listPlanExecutionEventsRequest).build());
    }

    /**
     * <p>
     * Lists the executions of a Region switch plan. This operation returns information about both current and
     * historical executions.
     * </p>
     *
     * @param listPlanExecutionsRequest
     * @return Result of the ListPlanExecutions operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.</p>
     *         <p>
     *         HTTP Status Code: 404
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     *         </p>
     *         <p>
     *         HTTP Status Code: 403
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcRegionSwitchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcRegionSwitchClient.ListPlanExecutions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/ListPlanExecutions"
     *      target="_top">AWS API Documentation</a>
     */
    default ListPlanExecutionsResponse listPlanExecutions(ListPlanExecutionsRequest listPlanExecutionsRequest)
            throws ResourceNotFoundException, AccessDeniedException, AwsServiceException, SdkClientException,
            ArcRegionSwitchException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists the executions of a Region switch plan. This operation returns information about both current and
     * historical executions.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListPlanExecutionsRequest.Builder} avoiding the
     * need to create one manually via {@link ListPlanExecutionsRequest#builder()}
     * </p>
     *
     * @param listPlanExecutionsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.arcregionswitch.model.ListPlanExecutionsRequest.Builder} to create
     *        a request.
     * @return Result of the ListPlanExecutions operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.</p>
     *         <p>
     *         HTTP Status Code: 404
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     *         </p>
     *         <p>
     *         HTTP Status Code: 403
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcRegionSwitchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcRegionSwitchClient.ListPlanExecutions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/ListPlanExecutions"
     *      target="_top">AWS API Documentation</a>
     */
    default ListPlanExecutionsResponse listPlanExecutions(Consumer<ListPlanExecutionsRequest.Builder> listPlanExecutionsRequest)
            throws ResourceNotFoundException, AccessDeniedException, AwsServiceException, SdkClientException,
            ArcRegionSwitchException {
        return listPlanExecutions(ListPlanExecutionsRequest.builder().applyMutation(listPlanExecutionsRequest).build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listPlanExecutions(software.amazon.awssdk.services.arcregionswitch.model.ListPlanExecutionsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.arcregionswitch.paginators.ListPlanExecutionsIterable responses = client.listPlanExecutionsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.arcregionswitch.paginators.ListPlanExecutionsIterable responses = client
     *             .listPlanExecutionsPaginator(request);
     *     for (software.amazon.awssdk.services.arcregionswitch.model.ListPlanExecutionsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.arcregionswitch.paginators.ListPlanExecutionsIterable responses = client.listPlanExecutionsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listPlanExecutions(software.amazon.awssdk.services.arcregionswitch.model.ListPlanExecutionsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listPlanExecutionsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.</p>
     *         <p>
     *         HTTP Status Code: 404
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     *         </p>
     *         <p>
     *         HTTP Status Code: 403
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcRegionSwitchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcRegionSwitchClient.ListPlanExecutions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/ListPlanExecutions"
     *      target="_top">AWS API Documentation</a>
     */
    default ListPlanExecutionsIterable listPlanExecutionsPaginator(ListPlanExecutionsRequest listPlanExecutionsRequest)
            throws ResourceNotFoundException, AccessDeniedException, AwsServiceException, SdkClientException,
            ArcRegionSwitchException {
        return new ListPlanExecutionsIterable(this, listPlanExecutionsRequest);
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listPlanExecutions(software.amazon.awssdk.services.arcregionswitch.model.ListPlanExecutionsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.arcregionswitch.paginators.ListPlanExecutionsIterable responses = client.listPlanExecutionsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.arcregionswitch.paginators.ListPlanExecutionsIterable responses = client
     *             .listPlanExecutionsPaginator(request);
     *     for (software.amazon.awssdk.services.arcregionswitch.model.ListPlanExecutionsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.arcregionswitch.paginators.ListPlanExecutionsIterable responses = client.listPlanExecutionsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listPlanExecutions(software.amazon.awssdk.services.arcregionswitch.model.ListPlanExecutionsRequest)}
     * operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListPlanExecutionsRequest.Builder} avoiding the
     * need to create one manually via {@link ListPlanExecutionsRequest#builder()}
     * </p>
     *
     * @param listPlanExecutionsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.arcregionswitch.model.ListPlanExecutionsRequest.Builder} to create
     *        a request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.</p>
     *         <p>
     *         HTTP Status Code: 404
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     *         </p>
     *         <p>
     *         HTTP Status Code: 403
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcRegionSwitchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcRegionSwitchClient.ListPlanExecutions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/ListPlanExecutions"
     *      target="_top">AWS API Documentation</a>
     */
    default ListPlanExecutionsIterable listPlanExecutionsPaginator(
            Consumer<ListPlanExecutionsRequest.Builder> listPlanExecutionsRequest) throws ResourceNotFoundException,
            AccessDeniedException, AwsServiceException, SdkClientException, ArcRegionSwitchException {
        return listPlanExecutionsPaginator(ListPlanExecutionsRequest.builder().applyMutation(listPlanExecutionsRequest).build());
    }

    /**
     * <p>
     * Lists all Region switch plans in your Amazon Web Services account.
     * </p>
     *
     * @param listPlansRequest
     * @return Result of the ListPlans operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcRegionSwitchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcRegionSwitchClient.ListPlans
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/ListPlans" target="_top">AWS
     *      API Documentation</a>
     */
    default ListPlansResponse listPlans(ListPlansRequest listPlansRequest) throws AwsServiceException, SdkClientException,
            ArcRegionSwitchException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists all Region switch plans in your Amazon Web Services account.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListPlansRequest.Builder} avoiding the need to
     * create one manually via {@link ListPlansRequest#builder()}
     * </p>
     *
     * @param listPlansRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.arcregionswitch.model.ListPlansRequest.Builder} to create a
     *        request.
     * @return Result of the ListPlans operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcRegionSwitchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcRegionSwitchClient.ListPlans
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/ListPlans" target="_top">AWS
     *      API Documentation</a>
     */
    default ListPlansResponse listPlans(Consumer<ListPlansRequest.Builder> listPlansRequest) throws AwsServiceException,
            SdkClientException, ArcRegionSwitchException {
        return listPlans(ListPlansRequest.builder().applyMutation(listPlansRequest).build());
    }

    /**
     * <p>
     * This is a variant of {@link #listPlans(software.amazon.awssdk.services.arcregionswitch.model.ListPlansRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.arcregionswitch.paginators.ListPlansIterable responses = client.listPlansPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.arcregionswitch.paginators.ListPlansIterable responses = client.listPlansPaginator(request);
     *     for (software.amazon.awssdk.services.arcregionswitch.model.ListPlansResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.arcregionswitch.paginators.ListPlansIterable responses = client.listPlansPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listPlans(software.amazon.awssdk.services.arcregionswitch.model.ListPlansRequest)} operation.</b>
     * </p>
     *
     * @param listPlansRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcRegionSwitchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcRegionSwitchClient.ListPlans
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/ListPlans" target="_top">AWS
     *      API Documentation</a>
     */
    default ListPlansIterable listPlansPaginator(ListPlansRequest listPlansRequest) throws AwsServiceException,
            SdkClientException, ArcRegionSwitchException {
        return new ListPlansIterable(this, listPlansRequest);
    }

    /**
     * <p>
     * This is a variant of {@link #listPlans(software.amazon.awssdk.services.arcregionswitch.model.ListPlansRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.arcregionswitch.paginators.ListPlansIterable responses = client.listPlansPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.arcregionswitch.paginators.ListPlansIterable responses = client.listPlansPaginator(request);
     *     for (software.amazon.awssdk.services.arcregionswitch.model.ListPlansResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.arcregionswitch.paginators.ListPlansIterable responses = client.listPlansPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listPlans(software.amazon.awssdk.services.arcregionswitch.model.ListPlansRequest)} operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListPlansRequest.Builder} avoiding the need to
     * create one manually via {@link ListPlansRequest#builder()}
     * </p>
     *
     * @param listPlansRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.arcregionswitch.model.ListPlansRequest.Builder} to create a
     *        request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcRegionSwitchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcRegionSwitchClient.ListPlans
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/ListPlans" target="_top">AWS
     *      API Documentation</a>
     */
    default ListPlansIterable listPlansPaginator(Consumer<ListPlansRequest.Builder> listPlansRequest) throws AwsServiceException,
            SdkClientException, ArcRegionSwitchException {
        return listPlansPaginator(ListPlansRequest.builder().applyMutation(listPlansRequest).build());
    }

    /**
     * <p>
     * Lists all Region switch plans in your Amazon Web Services account that are available in the current Amazon Web
     * Services Region.
     * </p>
     *
     * @param listPlansInRegionRequest
     * @return Result of the ListPlansInRegion operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.</p>
     *         <p>
     *         HTTP Status Code: 403
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcRegionSwitchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcRegionSwitchClient.ListPlansInRegion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/ListPlansInRegion"
     *      target="_top">AWS API Documentation</a>
     */
    default ListPlansInRegionResponse listPlansInRegion(ListPlansInRegionRequest listPlansInRegionRequest)
            throws AccessDeniedException, AwsServiceException, SdkClientException, ArcRegionSwitchException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists all Region switch plans in your Amazon Web Services account that are available in the current Amazon Web
     * Services Region.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListPlansInRegionRequest.Builder} avoiding the need
     * to create one manually via {@link ListPlansInRegionRequest#builder()}
     * </p>
     *
     * @param listPlansInRegionRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.arcregionswitch.model.ListPlansInRegionRequest.Builder} to create a
     *        request.
     * @return Result of the ListPlansInRegion operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.</p>
     *         <p>
     *         HTTP Status Code: 403
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcRegionSwitchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcRegionSwitchClient.ListPlansInRegion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/ListPlansInRegion"
     *      target="_top">AWS API Documentation</a>
     */
    default ListPlansInRegionResponse listPlansInRegion(Consumer<ListPlansInRegionRequest.Builder> listPlansInRegionRequest)
            throws AccessDeniedException, AwsServiceException, SdkClientException, ArcRegionSwitchException {
        return listPlansInRegion(ListPlansInRegionRequest.builder().applyMutation(listPlansInRegionRequest).build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listPlansInRegion(software.amazon.awssdk.services.arcregionswitch.model.ListPlansInRegionRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.arcregionswitch.paginators.ListPlansInRegionIterable responses = client.listPlansInRegionPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.arcregionswitch.paginators.ListPlansInRegionIterable responses = client
     *             .listPlansInRegionPaginator(request);
     *     for (software.amazon.awssdk.services.arcregionswitch.model.ListPlansInRegionResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.arcregionswitch.paginators.ListPlansInRegionIterable responses = client.listPlansInRegionPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listPlansInRegion(software.amazon.awssdk.services.arcregionswitch.model.ListPlansInRegionRequest)}
     * operation.</b>
     * </p>
     *
     * @param listPlansInRegionRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.</p>
     *         <p>
     *         HTTP Status Code: 403
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcRegionSwitchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcRegionSwitchClient.ListPlansInRegion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/ListPlansInRegion"
     *      target="_top">AWS API Documentation</a>
     */
    default ListPlansInRegionIterable listPlansInRegionPaginator(ListPlansInRegionRequest listPlansInRegionRequest)
            throws AccessDeniedException, AwsServiceException, SdkClientException, ArcRegionSwitchException {
        return new ListPlansInRegionIterable(this, listPlansInRegionRequest);
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listPlansInRegion(software.amazon.awssdk.services.arcregionswitch.model.ListPlansInRegionRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.arcregionswitch.paginators.ListPlansInRegionIterable responses = client.listPlansInRegionPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.arcregionswitch.paginators.ListPlansInRegionIterable responses = client
     *             .listPlansInRegionPaginator(request);
     *     for (software.amazon.awssdk.services.arcregionswitch.model.ListPlansInRegionResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.arcregionswitch.paginators.ListPlansInRegionIterable responses = client.listPlansInRegionPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listPlansInRegion(software.amazon.awssdk.services.arcregionswitch.model.ListPlansInRegionRequest)}
     * operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListPlansInRegionRequest.Builder} avoiding the need
     * to create one manually via {@link ListPlansInRegionRequest#builder()}
     * </p>
     *
     * @param listPlansInRegionRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.arcregionswitch.model.ListPlansInRegionRequest.Builder} to create a
     *        request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.</p>
     *         <p>
     *         HTTP Status Code: 403
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcRegionSwitchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcRegionSwitchClient.ListPlansInRegion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/ListPlansInRegion"
     *      target="_top">AWS API Documentation</a>
     */
    default ListPlansInRegionIterable listPlansInRegionPaginator(
            Consumer<ListPlansInRegionRequest.Builder> listPlansInRegionRequest) throws AccessDeniedException,
            AwsServiceException, SdkClientException, ArcRegionSwitchException {
        return listPlansInRegionPaginator(ListPlansInRegionRequest.builder().applyMutation(listPlansInRegionRequest).build());
    }

    /**
     * <p>
     * List the Amazon Route 53 health checks.
     * </p>
     *
     * @param listRoute53HealthChecksRequest
     * @return Result of the ListRoute53HealthChecks operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.</p>
     *         <p>
     *         HTTP Status Code: 404
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     *         </p>
     *         <p>
     *         HTTP Status Code: 403
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception, or failure.
     *         </p>
     *         <p>
     *         HTTP Status Code: 500
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcRegionSwitchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcRegionSwitchClient.ListRoute53HealthChecks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/ListRoute53HealthChecks"
     *      target="_top">AWS API Documentation</a>
     */
    default ListRoute53HealthChecksResponse listRoute53HealthChecks(ListRoute53HealthChecksRequest listRoute53HealthChecksRequest)
            throws ResourceNotFoundException, AccessDeniedException, InternalServerException, AwsServiceException,
            SdkClientException, ArcRegionSwitchException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * List the Amazon Route 53 health checks.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListRoute53HealthChecksRequest.Builder} avoiding
     * the need to create one manually via {@link ListRoute53HealthChecksRequest#builder()}
     * </p>
     *
     * @param listRoute53HealthChecksRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.arcregionswitch.model.ListRoute53HealthChecksRequest.Builder} to
     *        create a request.
     * @return Result of the ListRoute53HealthChecks operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.</p>
     *         <p>
     *         HTTP Status Code: 404
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     *         </p>
     *         <p>
     *         HTTP Status Code: 403
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception, or failure.
     *         </p>
     *         <p>
     *         HTTP Status Code: 500
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcRegionSwitchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcRegionSwitchClient.ListRoute53HealthChecks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/ListRoute53HealthChecks"
     *      target="_top">AWS API Documentation</a>
     */
    default ListRoute53HealthChecksResponse listRoute53HealthChecks(
            Consumer<ListRoute53HealthChecksRequest.Builder> listRoute53HealthChecksRequest) throws ResourceNotFoundException,
            AccessDeniedException, InternalServerException, AwsServiceException, SdkClientException, ArcRegionSwitchException {
        return listRoute53HealthChecks(ListRoute53HealthChecksRequest.builder().applyMutation(listRoute53HealthChecksRequest)
                .build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listRoute53HealthChecks(software.amazon.awssdk.services.arcregionswitch.model.ListRoute53HealthChecksRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.arcregionswitch.paginators.ListRoute53HealthChecksIterable responses = client.listRoute53HealthChecksPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.arcregionswitch.paginators.ListRoute53HealthChecksIterable responses = client
     *             .listRoute53HealthChecksPaginator(request);
     *     for (software.amazon.awssdk.services.arcregionswitch.model.ListRoute53HealthChecksResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.arcregionswitch.paginators.ListRoute53HealthChecksIterable responses = client.listRoute53HealthChecksPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listRoute53HealthChecks(software.amazon.awssdk.services.arcregionswitch.model.ListRoute53HealthChecksRequest)}
     * operation.</b>
     * </p>
     *
     * @param listRoute53HealthChecksRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.</p>
     *         <p>
     *         HTTP Status Code: 404
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     *         </p>
     *         <p>
     *         HTTP Status Code: 403
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception, or failure.
     *         </p>
     *         <p>
     *         HTTP Status Code: 500
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcRegionSwitchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcRegionSwitchClient.ListRoute53HealthChecks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/ListRoute53HealthChecks"
     *      target="_top">AWS API Documentation</a>
     */
    default ListRoute53HealthChecksIterable listRoute53HealthChecksPaginator(
            ListRoute53HealthChecksRequest listRoute53HealthChecksRequest) throws ResourceNotFoundException,
            AccessDeniedException, InternalServerException, AwsServiceException, SdkClientException, ArcRegionSwitchException {
        return new ListRoute53HealthChecksIterable(this, listRoute53HealthChecksRequest);
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listRoute53HealthChecks(software.amazon.awssdk.services.arcregionswitch.model.ListRoute53HealthChecksRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.arcregionswitch.paginators.ListRoute53HealthChecksIterable responses = client.listRoute53HealthChecksPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.arcregionswitch.paginators.ListRoute53HealthChecksIterable responses = client
     *             .listRoute53HealthChecksPaginator(request);
     *     for (software.amazon.awssdk.services.arcregionswitch.model.ListRoute53HealthChecksResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.arcregionswitch.paginators.ListRoute53HealthChecksIterable responses = client.listRoute53HealthChecksPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listRoute53HealthChecks(software.amazon.awssdk.services.arcregionswitch.model.ListRoute53HealthChecksRequest)}
     * operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListRoute53HealthChecksRequest.Builder} avoiding
     * the need to create one manually via {@link ListRoute53HealthChecksRequest#builder()}
     * </p>
     *
     * @param listRoute53HealthChecksRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.arcregionswitch.model.ListRoute53HealthChecksRequest.Builder} to
     *        create a request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.</p>
     *         <p>
     *         HTTP Status Code: 404
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     *         </p>
     *         <p>
     *         HTTP Status Code: 403
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception, or failure.
     *         </p>
     *         <p>
     *         HTTP Status Code: 500
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcRegionSwitchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcRegionSwitchClient.ListRoute53HealthChecks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/ListRoute53HealthChecks"
     *      target="_top">AWS API Documentation</a>
     */
    default ListRoute53HealthChecksIterable listRoute53HealthChecksPaginator(
            Consumer<ListRoute53HealthChecksRequest.Builder> listRoute53HealthChecksRequest) throws ResourceNotFoundException,
            AccessDeniedException, InternalServerException, AwsServiceException, SdkClientException, ArcRegionSwitchException {
        return listRoute53HealthChecksPaginator(ListRoute53HealthChecksRequest.builder()
                .applyMutation(listRoute53HealthChecksRequest).build());
    }

    /**
     * <p>
     * Lists the tags attached to a Region switch resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.</p>
     *         <p>
     *         HTTP Status Code: 404
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception, or failure.
     *         </p>
     *         <p>
     *         HTTP Status Code: 500
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcRegionSwitchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcRegionSwitchClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    default ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ResourceNotFoundException, InternalServerException, AwsServiceException, SdkClientException,
            ArcRegionSwitchException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists the tags attached to a Region switch resource.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListTagsForResourceRequest.Builder} avoiding the
     * need to create one manually via {@link ListTagsForResourceRequest#builder()}
     * </p>
     *
     * @param listTagsForResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.arcregionswitch.model.ListTagsForResourceRequest.Builder} to create
     *        a request.
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.</p>
     *         <p>
     *         HTTP Status Code: 404
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception, or failure.
     *         </p>
     *         <p>
     *         HTTP Status Code: 500
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcRegionSwitchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcRegionSwitchClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    default ListTagsForResourceResponse listTagsForResource(
            Consumer<ListTagsForResourceRequest.Builder> listTagsForResourceRequest) throws ResourceNotFoundException,
            InternalServerException, AwsServiceException, SdkClientException, ArcRegionSwitchException {
        return listTagsForResource(ListTagsForResourceRequest.builder().applyMutation(listTagsForResourceRequest).build());
    }

    /**
     * <p>
     * Starts the execution of a Region switch plan. You can execute a plan in either PRACTICE or RECOVERY mode.
     * </p>
     * <p>
     * In PRACTICE mode, the execution simulates the steps without making actual changes to your application's traffic
     * routing. In RECOVERY mode, the execution performs actual changes to shift traffic between Regions.
     * </p>
     *
     * @param startPlanExecutionRequest
     * @return Result of the StartPlanExecution operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.</p>
     *         <p>
     *         HTTP Status Code: 404
     * @throws IllegalStateException
     *         The operation failed because the current state of the resource doesn't allow the operation to proceed.
     *         </p>
     *         <p>
     *         HTTP Status Code: 400
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     *         </p>
     *         <p>
     *         HTTP Status Code: 403
     * @throws IllegalArgumentException
     *         The request processing has an invalid argument.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcRegionSwitchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcRegionSwitchClient.StartPlanExecution
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/StartPlanExecution"
     *      target="_top">AWS API Documentation</a>
     */
    default StartPlanExecutionResponse startPlanExecution(StartPlanExecutionRequest startPlanExecutionRequest)
            throws ResourceNotFoundException, IllegalStateException, AccessDeniedException, IllegalArgumentException,
            AwsServiceException, SdkClientException, ArcRegionSwitchException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Starts the execution of a Region switch plan. You can execute a plan in either PRACTICE or RECOVERY mode.
     * </p>
     * <p>
     * In PRACTICE mode, the execution simulates the steps without making actual changes to your application's traffic
     * routing. In RECOVERY mode, the execution performs actual changes to shift traffic between Regions.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link StartPlanExecutionRequest.Builder} avoiding the
     * need to create one manually via {@link StartPlanExecutionRequest#builder()}
     * </p>
     *
     * @param startPlanExecutionRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.arcregionswitch.model.StartPlanExecutionRequest.Builder} to create
     *        a request.
     * @return Result of the StartPlanExecution operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.</p>
     *         <p>
     *         HTTP Status Code: 404
     * @throws IllegalStateException
     *         The operation failed because the current state of the resource doesn't allow the operation to proceed.
     *         </p>
     *         <p>
     *         HTTP Status Code: 400
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     *         </p>
     *         <p>
     *         HTTP Status Code: 403
     * @throws IllegalArgumentException
     *         The request processing has an invalid argument.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcRegionSwitchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcRegionSwitchClient.StartPlanExecution
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/StartPlanExecution"
     *      target="_top">AWS API Documentation</a>
     */
    default StartPlanExecutionResponse startPlanExecution(Consumer<StartPlanExecutionRequest.Builder> startPlanExecutionRequest)
            throws ResourceNotFoundException, IllegalStateException, AccessDeniedException, IllegalArgumentException,
            AwsServiceException, SdkClientException, ArcRegionSwitchException {
        return startPlanExecution(StartPlanExecutionRequest.builder().applyMutation(startPlanExecutionRequest).build());
    }

    /**
     * <p>
     * Adds or updates tags for a Region switch resource. You can assign metadata to your resources in the form of tags,
     * which are key-value pairs.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.</p>
     *         <p>
     *         HTTP Status Code: 404
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception, or failure.
     *         </p>
     *         <p>
     *         HTTP Status Code: 500
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcRegionSwitchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcRegionSwitchClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    default TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ResourceNotFoundException,
            InternalServerException, AwsServiceException, SdkClientException, ArcRegionSwitchException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Adds or updates tags for a Region switch resource. You can assign metadata to your resources in the form of tags,
     * which are key-value pairs.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link TagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link TagResourceRequest#builder()}
     * </p>
     *
     * @param tagResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.arcregionswitch.model.TagResourceRequest.Builder} to create a
     *        request.
     * @return Result of the TagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.</p>
     *         <p>
     *         HTTP Status Code: 404
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception, or failure.
     *         </p>
     *         <p>
     *         HTTP Status Code: 500
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcRegionSwitchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcRegionSwitchClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    default TagResourceResponse tagResource(Consumer<TagResourceRequest.Builder> tagResourceRequest)
            throws ResourceNotFoundException, InternalServerException, AwsServiceException, SdkClientException,
            ArcRegionSwitchException {
        return tagResource(TagResourceRequest.builder().applyMutation(tagResourceRequest).build());
    }

    /**
     * <p>
     * Removes tags from a Region switch resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.</p>
     *         <p>
     *         HTTP Status Code: 404
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception, or failure.
     *         </p>
     *         <p>
     *         HTTP Status Code: 500
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcRegionSwitchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcRegionSwitchClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    default UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ResourceNotFoundException,
            InternalServerException, AwsServiceException, SdkClientException, ArcRegionSwitchException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Removes tags from a Region switch resource.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UntagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link UntagResourceRequest#builder()}
     * </p>
     *
     * @param untagResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.arcregionswitch.model.UntagResourceRequest.Builder} to create a
     *        request.
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.</p>
     *         <p>
     *         HTTP Status Code: 404
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception, or failure.
     *         </p>
     *         <p>
     *         HTTP Status Code: 500
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcRegionSwitchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcRegionSwitchClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    default UntagResourceResponse untagResource(Consumer<UntagResourceRequest.Builder> untagResourceRequest)
            throws ResourceNotFoundException, InternalServerException, AwsServiceException, SdkClientException,
            ArcRegionSwitchException {
        return untagResource(UntagResourceRequest.builder().applyMutation(untagResourceRequest).build());
    }

    /**
     * <p>
     * Updates an existing Region switch plan. You can modify the plan's description, workflows, execution role,
     * recovery time objective, associated alarms, and triggers.
     * </p>
     *
     * @param updatePlanRequest
     * @return Result of the UpdatePlan operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.</p>
     *         <p>
     *         HTTP Status Code: 404
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcRegionSwitchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcRegionSwitchClient.UpdatePlan
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/UpdatePlan" target="_top">AWS
     *      API Documentation</a>
     */
    default UpdatePlanResponse updatePlan(UpdatePlanRequest updatePlanRequest) throws ResourceNotFoundException,
            AwsServiceException, SdkClientException, ArcRegionSwitchException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Updates an existing Region switch plan. You can modify the plan's description, workflows, execution role,
     * recovery time objective, associated alarms, and triggers.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UpdatePlanRequest.Builder} avoiding the need to
     * create one manually via {@link UpdatePlanRequest#builder()}
     * </p>
     *
     * @param updatePlanRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.arcregionswitch.model.UpdatePlanRequest.Builder} to create a
     *        request.
     * @return Result of the UpdatePlan operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.</p>
     *         <p>
     *         HTTP Status Code: 404
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcRegionSwitchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcRegionSwitchClient.UpdatePlan
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/UpdatePlan" target="_top">AWS
     *      API Documentation</a>
     */
    default UpdatePlanResponse updatePlan(Consumer<UpdatePlanRequest.Builder> updatePlanRequest)
            throws ResourceNotFoundException, AwsServiceException, SdkClientException, ArcRegionSwitchException {
        return updatePlan(UpdatePlanRequest.builder().applyMutation(updatePlanRequest).build());
    }

    /**
     * <p>
     * Updates an in-progress plan execution. This operation allows you to modify certain aspects of the execution, such
     * as adding a comment or changing the action.
     * </p>
     *
     * @param updatePlanExecutionRequest
     * @return Result of the UpdatePlanExecution operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.</p>
     *         <p>
     *         HTTP Status Code: 404
     * @throws IllegalStateException
     *         The operation failed because the current state of the resource doesn't allow the operation to proceed.
     *         </p>
     *         <p>
     *         HTTP Status Code: 400
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     *         </p>
     *         <p>
     *         HTTP Status Code: 403
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcRegionSwitchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcRegionSwitchClient.UpdatePlanExecution
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/UpdatePlanExecution"
     *      target="_top">AWS API Documentation</a>
     */
    default UpdatePlanExecutionResponse updatePlanExecution(UpdatePlanExecutionRequest updatePlanExecutionRequest)
            throws ResourceNotFoundException, IllegalStateException, AccessDeniedException, AwsServiceException,
            SdkClientException, ArcRegionSwitchException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Updates an in-progress plan execution. This operation allows you to modify certain aspects of the execution, such
     * as adding a comment or changing the action.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UpdatePlanExecutionRequest.Builder} avoiding the
     * need to create one manually via {@link UpdatePlanExecutionRequest#builder()}
     * </p>
     *
     * @param updatePlanExecutionRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.arcregionswitch.model.UpdatePlanExecutionRequest.Builder} to create
     *        a request.
     * @return Result of the UpdatePlanExecution operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.</p>
     *         <p>
     *         HTTP Status Code: 404
     * @throws IllegalStateException
     *         The operation failed because the current state of the resource doesn't allow the operation to proceed.
     *         </p>
     *         <p>
     *         HTTP Status Code: 400
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     *         </p>
     *         <p>
     *         HTTP Status Code: 403
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcRegionSwitchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcRegionSwitchClient.UpdatePlanExecution
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/UpdatePlanExecution"
     *      target="_top">AWS API Documentation</a>
     */
    default UpdatePlanExecutionResponse updatePlanExecution(
            Consumer<UpdatePlanExecutionRequest.Builder> updatePlanExecutionRequest) throws ResourceNotFoundException,
            IllegalStateException, AccessDeniedException, AwsServiceException, SdkClientException, ArcRegionSwitchException {
        return updatePlanExecution(UpdatePlanExecutionRequest.builder().applyMutation(updatePlanExecutionRequest).build());
    }

    /**
     * <p>
     * Updates a specific step in an in-progress plan execution. This operation allows you to modify the step's comment
     * or action.
     * </p>
     *
     * @param updatePlanExecutionStepRequest
     * @return Result of the UpdatePlanExecutionStep operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.</p>
     *         <p>
     *         HTTP Status Code: 404
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     *         </p>
     *         <p>
     *         HTTP Status Code: 403
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcRegionSwitchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcRegionSwitchClient.UpdatePlanExecutionStep
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/UpdatePlanExecutionStep"
     *      target="_top">AWS API Documentation</a>
     */
    default UpdatePlanExecutionStepResponse updatePlanExecutionStep(UpdatePlanExecutionStepRequest updatePlanExecutionStepRequest)
            throws ResourceNotFoundException, AccessDeniedException, AwsServiceException, SdkClientException,
            ArcRegionSwitchException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Updates a specific step in an in-progress plan execution. This operation allows you to modify the step's comment
     * or action.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UpdatePlanExecutionStepRequest.Builder} avoiding
     * the need to create one manually via {@link UpdatePlanExecutionStepRequest#builder()}
     * </p>
     *
     * @param updatePlanExecutionStepRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.arcregionswitch.model.UpdatePlanExecutionStepRequest.Builder} to
     *        create a request.
     * @return Result of the UpdatePlanExecutionStep operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.</p>
     *         <p>
     *         HTTP Status Code: 404
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     *         </p>
     *         <p>
     *         HTTP Status Code: 403
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcRegionSwitchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcRegionSwitchClient.UpdatePlanExecutionStep
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-region-switch-2022-07-26/UpdatePlanExecutionStep"
     *      target="_top">AWS API Documentation</a>
     */
    default UpdatePlanExecutionStepResponse updatePlanExecutionStep(
            Consumer<UpdatePlanExecutionStepRequest.Builder> updatePlanExecutionStepRequest) throws ResourceNotFoundException,
            AccessDeniedException, AwsServiceException, SdkClientException, ArcRegionSwitchException {
        return updatePlanExecutionStep(UpdatePlanExecutionStepRequest.builder().applyMutation(updatePlanExecutionStepRequest)
                .build());
    }

    /**
     * Create an instance of {@link ArcRegionSwitchWaiter} using this client.
     * <p>
     * Waiters created via this method are managed by the SDK and resources will be released when the service client is
     * closed.
     *
     * @return an instance of {@link ArcRegionSwitchWaiter}
     */
    default ArcRegionSwitchWaiter waiter() {
        throw new UnsupportedOperationException();
    }

    /**
     * Create a {@link ArcRegionSwitchClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static ArcRegionSwitchClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link ArcRegionSwitchClient}.
     */
    static ArcRegionSwitchClientBuilder builder() {
        return new DefaultArcRegionSwitchClientBuilder();
    }

    static ServiceMetadata serviceMetadata() {
        return ServiceMetadata.of(SERVICE_METADATA_ID);
    }

    @Override
    default ArcRegionSwitchServiceClientConfiguration serviceClientConfiguration() {
        throw new UnsupportedOperationException();
    }
}
