/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.applicationinsights;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.applicationinsights.model.ApplicationInsightsException;
import software.amazon.awssdk.services.applicationinsights.model.ApplicationInsightsRequest;
import software.amazon.awssdk.services.applicationinsights.model.BadRequestException;
import software.amazon.awssdk.services.applicationinsights.model.CreateApplicationRequest;
import software.amazon.awssdk.services.applicationinsights.model.CreateApplicationResponse;
import software.amazon.awssdk.services.applicationinsights.model.CreateComponentRequest;
import software.amazon.awssdk.services.applicationinsights.model.CreateComponentResponse;
import software.amazon.awssdk.services.applicationinsights.model.CreateLogPatternRequest;
import software.amazon.awssdk.services.applicationinsights.model.CreateLogPatternResponse;
import software.amazon.awssdk.services.applicationinsights.model.DeleteApplicationRequest;
import software.amazon.awssdk.services.applicationinsights.model.DeleteApplicationResponse;
import software.amazon.awssdk.services.applicationinsights.model.DeleteComponentRequest;
import software.amazon.awssdk.services.applicationinsights.model.DeleteComponentResponse;
import software.amazon.awssdk.services.applicationinsights.model.DeleteLogPatternRequest;
import software.amazon.awssdk.services.applicationinsights.model.DeleteLogPatternResponse;
import software.amazon.awssdk.services.applicationinsights.model.DescribeApplicationRequest;
import software.amazon.awssdk.services.applicationinsights.model.DescribeApplicationResponse;
import software.amazon.awssdk.services.applicationinsights.model.DescribeComponentConfigurationRecommendationRequest;
import software.amazon.awssdk.services.applicationinsights.model.DescribeComponentConfigurationRecommendationResponse;
import software.amazon.awssdk.services.applicationinsights.model.DescribeComponentConfigurationRequest;
import software.amazon.awssdk.services.applicationinsights.model.DescribeComponentConfigurationResponse;
import software.amazon.awssdk.services.applicationinsights.model.DescribeComponentRequest;
import software.amazon.awssdk.services.applicationinsights.model.DescribeComponentResponse;
import software.amazon.awssdk.services.applicationinsights.model.DescribeLogPatternRequest;
import software.amazon.awssdk.services.applicationinsights.model.DescribeLogPatternResponse;
import software.amazon.awssdk.services.applicationinsights.model.DescribeObservationRequest;
import software.amazon.awssdk.services.applicationinsights.model.DescribeObservationResponse;
import software.amazon.awssdk.services.applicationinsights.model.DescribeProblemObservationsRequest;
import software.amazon.awssdk.services.applicationinsights.model.DescribeProblemObservationsResponse;
import software.amazon.awssdk.services.applicationinsights.model.DescribeProblemRequest;
import software.amazon.awssdk.services.applicationinsights.model.DescribeProblemResponse;
import software.amazon.awssdk.services.applicationinsights.model.InternalServerException;
import software.amazon.awssdk.services.applicationinsights.model.ListApplicationsRequest;
import software.amazon.awssdk.services.applicationinsights.model.ListApplicationsResponse;
import software.amazon.awssdk.services.applicationinsights.model.ListComponentsRequest;
import software.amazon.awssdk.services.applicationinsights.model.ListComponentsResponse;
import software.amazon.awssdk.services.applicationinsights.model.ListConfigurationHistoryRequest;
import software.amazon.awssdk.services.applicationinsights.model.ListConfigurationHistoryResponse;
import software.amazon.awssdk.services.applicationinsights.model.ListLogPatternSetsRequest;
import software.amazon.awssdk.services.applicationinsights.model.ListLogPatternSetsResponse;
import software.amazon.awssdk.services.applicationinsights.model.ListLogPatternsRequest;
import software.amazon.awssdk.services.applicationinsights.model.ListLogPatternsResponse;
import software.amazon.awssdk.services.applicationinsights.model.ListProblemsRequest;
import software.amazon.awssdk.services.applicationinsights.model.ListProblemsResponse;
import software.amazon.awssdk.services.applicationinsights.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.applicationinsights.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.applicationinsights.model.ResourceInUseException;
import software.amazon.awssdk.services.applicationinsights.model.ResourceNotFoundException;
import software.amazon.awssdk.services.applicationinsights.model.TagResourceRequest;
import software.amazon.awssdk.services.applicationinsights.model.TagResourceResponse;
import software.amazon.awssdk.services.applicationinsights.model.TagsAlreadyExistException;
import software.amazon.awssdk.services.applicationinsights.model.TooManyTagsException;
import software.amazon.awssdk.services.applicationinsights.model.UntagResourceRequest;
import software.amazon.awssdk.services.applicationinsights.model.UntagResourceResponse;
import software.amazon.awssdk.services.applicationinsights.model.UpdateApplicationRequest;
import software.amazon.awssdk.services.applicationinsights.model.UpdateApplicationResponse;
import software.amazon.awssdk.services.applicationinsights.model.UpdateComponentConfigurationRequest;
import software.amazon.awssdk.services.applicationinsights.model.UpdateComponentConfigurationResponse;
import software.amazon.awssdk.services.applicationinsights.model.UpdateComponentRequest;
import software.amazon.awssdk.services.applicationinsights.model.UpdateComponentResponse;
import software.amazon.awssdk.services.applicationinsights.model.UpdateLogPatternRequest;
import software.amazon.awssdk.services.applicationinsights.model.UpdateLogPatternResponse;
import software.amazon.awssdk.services.applicationinsights.model.ValidationException;
import software.amazon.awssdk.services.applicationinsights.paginators.ListApplicationsPublisher;
import software.amazon.awssdk.services.applicationinsights.paginators.ListComponentsPublisher;
import software.amazon.awssdk.services.applicationinsights.paginators.ListConfigurationHistoryPublisher;
import software.amazon.awssdk.services.applicationinsights.paginators.ListLogPatternSetsPublisher;
import software.amazon.awssdk.services.applicationinsights.paginators.ListLogPatternsPublisher;
import software.amazon.awssdk.services.applicationinsights.paginators.ListProblemsPublisher;
import software.amazon.awssdk.services.applicationinsights.transform.CreateApplicationRequestMarshaller;
import software.amazon.awssdk.services.applicationinsights.transform.CreateComponentRequestMarshaller;
import software.amazon.awssdk.services.applicationinsights.transform.CreateLogPatternRequestMarshaller;
import software.amazon.awssdk.services.applicationinsights.transform.DeleteApplicationRequestMarshaller;
import software.amazon.awssdk.services.applicationinsights.transform.DeleteComponentRequestMarshaller;
import software.amazon.awssdk.services.applicationinsights.transform.DeleteLogPatternRequestMarshaller;
import software.amazon.awssdk.services.applicationinsights.transform.DescribeApplicationRequestMarshaller;
import software.amazon.awssdk.services.applicationinsights.transform.DescribeComponentConfigurationRecommendationRequestMarshaller;
import software.amazon.awssdk.services.applicationinsights.transform.DescribeComponentConfigurationRequestMarshaller;
import software.amazon.awssdk.services.applicationinsights.transform.DescribeComponentRequestMarshaller;
import software.amazon.awssdk.services.applicationinsights.transform.DescribeLogPatternRequestMarshaller;
import software.amazon.awssdk.services.applicationinsights.transform.DescribeObservationRequestMarshaller;
import software.amazon.awssdk.services.applicationinsights.transform.DescribeProblemObservationsRequestMarshaller;
import software.amazon.awssdk.services.applicationinsights.transform.DescribeProblemRequestMarshaller;
import software.amazon.awssdk.services.applicationinsights.transform.ListApplicationsRequestMarshaller;
import software.amazon.awssdk.services.applicationinsights.transform.ListComponentsRequestMarshaller;
import software.amazon.awssdk.services.applicationinsights.transform.ListConfigurationHistoryRequestMarshaller;
import software.amazon.awssdk.services.applicationinsights.transform.ListLogPatternSetsRequestMarshaller;
import software.amazon.awssdk.services.applicationinsights.transform.ListLogPatternsRequestMarshaller;
import software.amazon.awssdk.services.applicationinsights.transform.ListProblemsRequestMarshaller;
import software.amazon.awssdk.services.applicationinsights.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.applicationinsights.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.applicationinsights.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.applicationinsights.transform.UpdateApplicationRequestMarshaller;
import software.amazon.awssdk.services.applicationinsights.transform.UpdateComponentConfigurationRequestMarshaller;
import software.amazon.awssdk.services.applicationinsights.transform.UpdateComponentRequestMarshaller;
import software.amazon.awssdk.services.applicationinsights.transform.UpdateLogPatternRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link ApplicationInsightsAsyncClient}.
 *
 * @see ApplicationInsightsAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultApplicationInsightsAsyncClient implements ApplicationInsightsAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultApplicationInsightsAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultApplicationInsightsAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Adds an application that is created from a resource group.
     * </p>
     *
     * @param createApplicationRequest
     * @return A Java Future containing the result of the CreateApplication operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceInUseException The resource is already created or in use.</li>
     *         <li>ResourceNotFoundException The resource does not exist in the customer account.</li>
     *         <li>ValidationException The parameter is not valid.</li>
     *         <li>InternalServerException The server encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>TagsAlreadyExistException Tags are already registered for the specified application ARN.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationInsightsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationInsightsAsyncClient.CreateApplication
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/application-insights-2018-11-25/CreateApplication"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateApplicationResponse> createApplication(CreateApplicationRequest createApplicationRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateApplicationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateApplicationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateApplicationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateApplicationRequest, CreateApplicationResponse>()
                            .withOperationName("CreateApplication")
                            .withMarshaller(new CreateApplicationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createApplicationRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a custom component by grouping similar standalone instances to monitor.
     * </p>
     *
     * @param createComponentRequest
     * @return A Java Future containing the result of the CreateComponent operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceInUseException The resource is already created or in use.</li>
     *         <li>ResourceNotFoundException The resource does not exist in the customer account.</li>
     *         <li>ValidationException The parameter is not valid.</li>
     *         <li>InternalServerException The server encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationInsightsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationInsightsAsyncClient.CreateComponent
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/application-insights-2018-11-25/CreateComponent"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateComponentResponse> createComponent(CreateComponentRequest createComponentRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateComponentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateComponentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateComponentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateComponentRequest, CreateComponentResponse>()
                            .withOperationName("CreateComponent")
                            .withMarshaller(new CreateComponentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createComponentRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds an log pattern to a <code>LogPatternSet</code>.
     * </p>
     *
     * @param createLogPatternRequest
     * @return A Java Future containing the result of the CreateLogPattern operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceInUseException The resource is already created or in use.</li>
     *         <li>ResourceNotFoundException The resource does not exist in the customer account.</li>
     *         <li>ValidationException The parameter is not valid.</li>
     *         <li>InternalServerException The server encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationInsightsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationInsightsAsyncClient.CreateLogPattern
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/application-insights-2018-11-25/CreateLogPattern"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateLogPatternResponse> createLogPattern(CreateLogPatternRequest createLogPatternRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateLogPatternResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateLogPatternResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateLogPatternResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateLogPatternRequest, CreateLogPatternResponse>()
                            .withOperationName("CreateLogPattern")
                            .withMarshaller(new CreateLogPatternRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createLogPatternRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the specified application from monitoring. Does not delete the application.
     * </p>
     *
     * @param deleteApplicationRequest
     * @return A Java Future containing the result of the DeleteApplication operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource does not exist in the customer account.</li>
     *         <li>ValidationException The parameter is not valid.</li>
     *         <li>BadRequestException The request is not understood by the server.</li>
     *         <li>InternalServerException The server encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationInsightsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationInsightsAsyncClient.DeleteApplication
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/application-insights-2018-11-25/DeleteApplication"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteApplicationResponse> deleteApplication(DeleteApplicationRequest deleteApplicationRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteApplicationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteApplicationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteApplicationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteApplicationRequest, DeleteApplicationResponse>()
                            .withOperationName("DeleteApplication")
                            .withMarshaller(new DeleteApplicationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteApplicationRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Ungroups a custom component. When you ungroup custom components, all applicable monitors that are set up for the
     * component are removed and the instances revert to their standalone status.
     * </p>
     *
     * @param deleteComponentRequest
     * @return A Java Future containing the result of the DeleteComponent operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource does not exist in the customer account.</li>
     *         <li>ValidationException The parameter is not valid.</li>
     *         <li>InternalServerException The server encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationInsightsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationInsightsAsyncClient.DeleteComponent
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/application-insights-2018-11-25/DeleteComponent"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteComponentResponse> deleteComponent(DeleteComponentRequest deleteComponentRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteComponentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteComponentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteComponentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteComponentRequest, DeleteComponentResponse>()
                            .withOperationName("DeleteComponent")
                            .withMarshaller(new DeleteComponentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteComponentRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the specified log pattern from a <code>LogPatternSet</code>.
     * </p>
     *
     * @param deleteLogPatternRequest
     * @return A Java Future containing the result of the DeleteLogPattern operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource does not exist in the customer account.</li>
     *         <li>ValidationException The parameter is not valid.</li>
     *         <li>BadRequestException The request is not understood by the server.</li>
     *         <li>InternalServerException The server encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationInsightsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationInsightsAsyncClient.DeleteLogPattern
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/application-insights-2018-11-25/DeleteLogPattern"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteLogPatternResponse> deleteLogPattern(DeleteLogPatternRequest deleteLogPatternRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteLogPatternResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteLogPatternResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteLogPatternResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteLogPatternRequest, DeleteLogPatternResponse>()
                            .withOperationName("DeleteLogPattern")
                            .withMarshaller(new DeleteLogPatternRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteLogPatternRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the application.
     * </p>
     *
     * @param describeApplicationRequest
     * @return A Java Future containing the result of the DescribeApplication operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource does not exist in the customer account.</li>
     *         <li>ValidationException The parameter is not valid.</li>
     *         <li>InternalServerException The server encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationInsightsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationInsightsAsyncClient.DescribeApplication
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/application-insights-2018-11-25/DescribeApplication"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeApplicationResponse> describeApplication(
            DescribeApplicationRequest describeApplicationRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeApplicationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeApplicationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeApplicationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeApplicationRequest, DescribeApplicationResponse>()
                            .withOperationName("DescribeApplication")
                            .withMarshaller(new DescribeApplicationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeApplicationRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes a component and lists the resources that are grouped together in a component.
     * </p>
     *
     * @param describeComponentRequest
     * @return A Java Future containing the result of the DescribeComponent operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource does not exist in the customer account.</li>
     *         <li>ValidationException The parameter is not valid.</li>
     *         <li>InternalServerException The server encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationInsightsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationInsightsAsyncClient.DescribeComponent
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/application-insights-2018-11-25/DescribeComponent"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeComponentResponse> describeComponent(DescribeComponentRequest describeComponentRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeComponentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeComponentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeComponentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeComponentRequest, DescribeComponentResponse>()
                            .withOperationName("DescribeComponent")
                            .withMarshaller(new DescribeComponentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeComponentRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the monitoring configuration of the component.
     * </p>
     *
     * @param describeComponentConfigurationRequest
     * @return A Java Future containing the result of the DescribeComponentConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource does not exist in the customer account.</li>
     *         <li>ValidationException The parameter is not valid.</li>
     *         <li>InternalServerException The server encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationInsightsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationInsightsAsyncClient.DescribeComponentConfiguration
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/application-insights-2018-11-25/DescribeComponentConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeComponentConfigurationResponse> describeComponentConfiguration(
            DescribeComponentConfigurationRequest describeComponentConfigurationRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeComponentConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeComponentConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeComponentConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeComponentConfigurationRequest, DescribeComponentConfigurationResponse>()
                            .withOperationName("DescribeComponentConfiguration")
                            .withMarshaller(new DescribeComponentConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeComponentConfigurationRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the recommended monitoring configuration of the component.
     * </p>
     *
     * @param describeComponentConfigurationRecommendationRequest
     * @return A Java Future containing the result of the DescribeComponentConfigurationRecommendation operation
     *         returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource does not exist in the customer account.</li>
     *         <li>ValidationException The parameter is not valid.</li>
     *         <li>InternalServerException The server encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationInsightsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationInsightsAsyncClient.DescribeComponentConfigurationRecommendation
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/application-insights-2018-11-25/DescribeComponentConfigurationRecommendation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeComponentConfigurationRecommendationResponse> describeComponentConfigurationRecommendation(
            DescribeComponentConfigurationRecommendationRequest describeComponentConfigurationRecommendationRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeComponentConfigurationRecommendationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DescribeComponentConfigurationRecommendationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeComponentConfigurationRecommendationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeComponentConfigurationRecommendationRequest, DescribeComponentConfigurationRecommendationResponse>()
                            .withOperationName("DescribeComponentConfigurationRecommendation")
                            .withMarshaller(new DescribeComponentConfigurationRecommendationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeComponentConfigurationRecommendationRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describe a specific log pattern from a <code>LogPatternSet</code>.
     * </p>
     *
     * @param describeLogPatternRequest
     * @return A Java Future containing the result of the DescribeLogPattern operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource does not exist in the customer account.</li>
     *         <li>ValidationException The parameter is not valid.</li>
     *         <li>InternalServerException The server encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationInsightsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationInsightsAsyncClient.DescribeLogPattern
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/application-insights-2018-11-25/DescribeLogPattern"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeLogPatternResponse> describeLogPattern(DescribeLogPatternRequest describeLogPatternRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeLogPatternResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeLogPatternResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeLogPatternResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeLogPatternRequest, DescribeLogPatternResponse>()
                            .withOperationName("DescribeLogPattern")
                            .withMarshaller(new DescribeLogPatternRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeLogPatternRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes an anomaly or error with the application.
     * </p>
     *
     * @param describeObservationRequest
     * @return A Java Future containing the result of the DescribeObservation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The server encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>ValidationException The parameter is not valid.</li>
     *         <li>ResourceNotFoundException The resource does not exist in the customer account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationInsightsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationInsightsAsyncClient.DescribeObservation
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/application-insights-2018-11-25/DescribeObservation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeObservationResponse> describeObservation(
            DescribeObservationRequest describeObservationRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeObservationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeObservationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeObservationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeObservationRequest, DescribeObservationResponse>()
                            .withOperationName("DescribeObservation")
                            .withMarshaller(new DescribeObservationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeObservationRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes an application problem.
     * </p>
     *
     * @param describeProblemRequest
     * @return A Java Future containing the result of the DescribeProblem operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The server encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>ValidationException The parameter is not valid.</li>
     *         <li>ResourceNotFoundException The resource does not exist in the customer account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationInsightsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationInsightsAsyncClient.DescribeProblem
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/application-insights-2018-11-25/DescribeProblem"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeProblemResponse> describeProblem(DescribeProblemRequest describeProblemRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeProblemResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeProblemResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeProblemResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeProblemRequest, DescribeProblemResponse>()
                            .withOperationName("DescribeProblem")
                            .withMarshaller(new DescribeProblemRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeProblemRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the anomalies or errors associated with the problem.
     * </p>
     *
     * @param describeProblemObservationsRequest
     * @return A Java Future containing the result of the DescribeProblemObservations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The server encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>ValidationException The parameter is not valid.</li>
     *         <li>ResourceNotFoundException The resource does not exist in the customer account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationInsightsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationInsightsAsyncClient.DescribeProblemObservations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/application-insights-2018-11-25/DescribeProblemObservations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeProblemObservationsResponse> describeProblemObservations(
            DescribeProblemObservationsRequest describeProblemObservationsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeProblemObservationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeProblemObservationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeProblemObservationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeProblemObservationsRequest, DescribeProblemObservationsResponse>()
                            .withOperationName("DescribeProblemObservations")
                            .withMarshaller(new DescribeProblemObservationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeProblemObservationsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the IDs of the applications that you are monitoring.
     * </p>
     *
     * @param listApplicationsRequest
     * @return A Java Future containing the result of the ListApplications operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The parameter is not valid.</li>
     *         <li>InternalServerException The server encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationInsightsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationInsightsAsyncClient.ListApplications
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/application-insights-2018-11-25/ListApplications"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListApplicationsResponse> listApplications(ListApplicationsRequest listApplicationsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListApplicationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListApplicationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListApplicationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListApplicationsRequest, ListApplicationsResponse>()
                            .withOperationName("ListApplications")
                            .withMarshaller(new ListApplicationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listApplicationsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the IDs of the applications that you are monitoring.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listApplications(software.amazon.awssdk.services.applicationinsights.model.ListApplicationsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.applicationinsights.paginators.ListApplicationsPublisher publisher = client.listApplicationsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.applicationinsights.paginators.ListApplicationsPublisher publisher = client.listApplicationsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.applicationinsights.model.ListApplicationsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.applicationinsights.model.ListApplicationsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listApplications(software.amazon.awssdk.services.applicationinsights.model.ListApplicationsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listApplicationsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The parameter is not valid.</li>
     *         <li>InternalServerException The server encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationInsightsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationInsightsAsyncClient.ListApplications
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/application-insights-2018-11-25/ListApplications"
     *      target="_top">AWS API Documentation</a>
     */
    public ListApplicationsPublisher listApplicationsPaginator(ListApplicationsRequest listApplicationsRequest) {
        return new ListApplicationsPublisher(this, applyPaginatorUserAgent(listApplicationsRequest));
    }

    /**
     * <p>
     * Lists the auto-grouped, standalone, and custom components of the application.
     * </p>
     *
     * @param listComponentsRequest
     * @return A Java Future containing the result of the ListComponents operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource does not exist in the customer account.</li>
     *         <li>ValidationException The parameter is not valid.</li>
     *         <li>InternalServerException The server encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationInsightsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationInsightsAsyncClient.ListComponents
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/application-insights-2018-11-25/ListComponents"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListComponentsResponse> listComponents(ListComponentsRequest listComponentsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListComponentsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListComponentsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListComponentsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListComponentsRequest, ListComponentsResponse>()
                            .withOperationName("ListComponents")
                            .withMarshaller(new ListComponentsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listComponentsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the auto-grouped, standalone, and custom components of the application.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listComponents(software.amazon.awssdk.services.applicationinsights.model.ListComponentsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.applicationinsights.paginators.ListComponentsPublisher publisher = client.listComponentsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.applicationinsights.paginators.ListComponentsPublisher publisher = client.listComponentsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.applicationinsights.model.ListComponentsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.applicationinsights.model.ListComponentsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listComponents(software.amazon.awssdk.services.applicationinsights.model.ListComponentsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listComponentsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource does not exist in the customer account.</li>
     *         <li>ValidationException The parameter is not valid.</li>
     *         <li>InternalServerException The server encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationInsightsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationInsightsAsyncClient.ListComponents
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/application-insights-2018-11-25/ListComponents"
     *      target="_top">AWS API Documentation</a>
     */
    public ListComponentsPublisher listComponentsPaginator(ListComponentsRequest listComponentsRequest) {
        return new ListComponentsPublisher(this, applyPaginatorUserAgent(listComponentsRequest));
    }

    /**
     * <p>
     * Lists the INFO, WARN, and ERROR events for periodic configuration updates performed by Application Insights.
     * Examples of events represented are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * INFO: creating a new alarm or updating an alarm threshold.
     * </p>
     * </li>
     * <li>
     * <p>
     * WARN: alarm not created due to insufficient data points used to predict thresholds.
     * </p>
     * </li>
     * <li>
     * <p>
     * ERROR: alarm not created due to permission errors or exceeding quotas.
     * </p>
     * </li>
     * </ul>
     *
     * @param listConfigurationHistoryRequest
     * @return A Java Future containing the result of the ListConfigurationHistory operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The parameter is not valid.</li>
     *         <li>ResourceNotFoundException The resource does not exist in the customer account.</li>
     *         <li>InternalServerException The server encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationInsightsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationInsightsAsyncClient.ListConfigurationHistory
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/application-insights-2018-11-25/ListConfigurationHistory"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListConfigurationHistoryResponse> listConfigurationHistory(
            ListConfigurationHistoryRequest listConfigurationHistoryRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListConfigurationHistoryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListConfigurationHistoryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListConfigurationHistoryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListConfigurationHistoryRequest, ListConfigurationHistoryResponse>()
                            .withOperationName("ListConfigurationHistory")
                            .withMarshaller(new ListConfigurationHistoryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listConfigurationHistoryRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the INFO, WARN, and ERROR events for periodic configuration updates performed by Application Insights.
     * Examples of events represented are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * INFO: creating a new alarm or updating an alarm threshold.
     * </p>
     * </li>
     * <li>
     * <p>
     * WARN: alarm not created due to insufficient data points used to predict thresholds.
     * </p>
     * </li>
     * <li>
     * <p>
     * ERROR: alarm not created due to permission errors or exceeding quotas.
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listConfigurationHistory(software.amazon.awssdk.services.applicationinsights.model.ListConfigurationHistoryRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.applicationinsights.paginators.ListConfigurationHistoryPublisher publisher = client.listConfigurationHistoryPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.applicationinsights.paginators.ListConfigurationHistoryPublisher publisher = client.listConfigurationHistoryPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.applicationinsights.model.ListConfigurationHistoryResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.applicationinsights.model.ListConfigurationHistoryResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listConfigurationHistory(software.amazon.awssdk.services.applicationinsights.model.ListConfigurationHistoryRequest)}
     * operation.</b>
     * </p>
     *
     * @param listConfigurationHistoryRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The parameter is not valid.</li>
     *         <li>ResourceNotFoundException The resource does not exist in the customer account.</li>
     *         <li>InternalServerException The server encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationInsightsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationInsightsAsyncClient.ListConfigurationHistory
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/application-insights-2018-11-25/ListConfigurationHistory"
     *      target="_top">AWS API Documentation</a>
     */
    public ListConfigurationHistoryPublisher listConfigurationHistoryPaginator(
            ListConfigurationHistoryRequest listConfigurationHistoryRequest) {
        return new ListConfigurationHistoryPublisher(this, applyPaginatorUserAgent(listConfigurationHistoryRequest));
    }

    /**
     * <p>
     * Lists the log pattern sets in the specific application.
     * </p>
     *
     * @param listLogPatternSetsRequest
     * @return A Java Future containing the result of the ListLogPatternSets operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource does not exist in the customer account.</li>
     *         <li>ValidationException The parameter is not valid.</li>
     *         <li>InternalServerException The server encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationInsightsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationInsightsAsyncClient.ListLogPatternSets
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/application-insights-2018-11-25/ListLogPatternSets"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListLogPatternSetsResponse> listLogPatternSets(ListLogPatternSetsRequest listLogPatternSetsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListLogPatternSetsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListLogPatternSetsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListLogPatternSetsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListLogPatternSetsRequest, ListLogPatternSetsResponse>()
                            .withOperationName("ListLogPatternSets")
                            .withMarshaller(new ListLogPatternSetsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listLogPatternSetsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the log pattern sets in the specific application.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listLogPatternSets(software.amazon.awssdk.services.applicationinsights.model.ListLogPatternSetsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.applicationinsights.paginators.ListLogPatternSetsPublisher publisher = client.listLogPatternSetsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.applicationinsights.paginators.ListLogPatternSetsPublisher publisher = client.listLogPatternSetsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.applicationinsights.model.ListLogPatternSetsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.applicationinsights.model.ListLogPatternSetsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listLogPatternSets(software.amazon.awssdk.services.applicationinsights.model.ListLogPatternSetsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listLogPatternSetsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource does not exist in the customer account.</li>
     *         <li>ValidationException The parameter is not valid.</li>
     *         <li>InternalServerException The server encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationInsightsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationInsightsAsyncClient.ListLogPatternSets
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/application-insights-2018-11-25/ListLogPatternSets"
     *      target="_top">AWS API Documentation</a>
     */
    public ListLogPatternSetsPublisher listLogPatternSetsPaginator(ListLogPatternSetsRequest listLogPatternSetsRequest) {
        return new ListLogPatternSetsPublisher(this, applyPaginatorUserAgent(listLogPatternSetsRequest));
    }

    /**
     * <p>
     * Lists the log patterns in the specific log <code>LogPatternSet</code>.
     * </p>
     *
     * @param listLogPatternsRequest
     * @return A Java Future containing the result of the ListLogPatterns operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource does not exist in the customer account.</li>
     *         <li>ValidationException The parameter is not valid.</li>
     *         <li>InternalServerException The server encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationInsightsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationInsightsAsyncClient.ListLogPatterns
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/application-insights-2018-11-25/ListLogPatterns"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListLogPatternsResponse> listLogPatterns(ListLogPatternsRequest listLogPatternsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListLogPatternsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListLogPatternsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListLogPatternsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListLogPatternsRequest, ListLogPatternsResponse>()
                            .withOperationName("ListLogPatterns")
                            .withMarshaller(new ListLogPatternsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listLogPatternsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the log patterns in the specific log <code>LogPatternSet</code>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listLogPatterns(software.amazon.awssdk.services.applicationinsights.model.ListLogPatternsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.applicationinsights.paginators.ListLogPatternsPublisher publisher = client.listLogPatternsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.applicationinsights.paginators.ListLogPatternsPublisher publisher = client.listLogPatternsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.applicationinsights.model.ListLogPatternsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.applicationinsights.model.ListLogPatternsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listLogPatterns(software.amazon.awssdk.services.applicationinsights.model.ListLogPatternsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listLogPatternsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource does not exist in the customer account.</li>
     *         <li>ValidationException The parameter is not valid.</li>
     *         <li>InternalServerException The server encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationInsightsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationInsightsAsyncClient.ListLogPatterns
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/application-insights-2018-11-25/ListLogPatterns"
     *      target="_top">AWS API Documentation</a>
     */
    public ListLogPatternsPublisher listLogPatternsPaginator(ListLogPatternsRequest listLogPatternsRequest) {
        return new ListLogPatternsPublisher(this, applyPaginatorUserAgent(listLogPatternsRequest));
    }

    /**
     * <p>
     * Lists the problems with your application.
     * </p>
     *
     * @param listProblemsRequest
     * @return A Java Future containing the result of the ListProblems operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The parameter is not valid.</li>
     *         <li>ResourceNotFoundException The resource does not exist in the customer account.</li>
     *         <li>InternalServerException The server encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationInsightsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationInsightsAsyncClient.ListProblems
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/application-insights-2018-11-25/ListProblems"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListProblemsResponse> listProblems(ListProblemsRequest listProblemsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListProblemsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListProblemsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListProblemsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListProblemsRequest, ListProblemsResponse>()
                            .withOperationName("ListProblems").withMarshaller(new ListProblemsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listProblemsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the problems with your application.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listProblems(software.amazon.awssdk.services.applicationinsights.model.ListProblemsRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.applicationinsights.paginators.ListProblemsPublisher publisher = client.listProblemsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.applicationinsights.paginators.ListProblemsPublisher publisher = client.listProblemsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.applicationinsights.model.ListProblemsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.applicationinsights.model.ListProblemsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listProblems(software.amazon.awssdk.services.applicationinsights.model.ListProblemsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listProblemsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The parameter is not valid.</li>
     *         <li>ResourceNotFoundException The resource does not exist in the customer account.</li>
     *         <li>InternalServerException The server encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationInsightsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationInsightsAsyncClient.ListProblems
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/application-insights-2018-11-25/ListProblems"
     *      target="_top">AWS API Documentation</a>
     */
    public ListProblemsPublisher listProblemsPaginator(ListProblemsRequest listProblemsRequest) {
        return new ListProblemsPublisher(this, applyPaginatorUserAgent(listProblemsRequest));
    }

    /**
     * <p>
     * Retrieve a list of the tags (keys and values) that are associated with a specified application. A <i>tag</i> is a
     * label that you optionally define and associate with an application. Each tag consists of a required <i>tag
     * key</i> and an optional associated <i>tag value</i>. A tag key is a general label that acts as a category for
     * more specific tag values. A tag value acts as a descriptor within a tag key.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource does not exist in the customer account.</li>
     *         <li>ValidationException The parameter is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationInsightsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationInsightsAsyncClient.ListTagsForResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/application-insights-2018-11-25/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listTagsForResourceRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Add one or more tags (keys and values) to a specified application. A <i>tag</i> is a label that you optionally
     * define and associate with an application. Tags can help you categorize and manage application in different ways,
     * such as by purpose, owner, environment, or other criteria.
     * </p>
     * <p>
     * Each tag consists of a required <i>tag key</i> and an associated <i>tag value</i>, both of which you define. A
     * tag key is a general label that acts as a category for more specific tag values. A tag value acts as a descriptor
     * within a tag key.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource does not exist in the customer account.</li>
     *         <li>TooManyTagsException The number of the provided tags is beyond the limit, or the number of total tags
     *         you are trying to attach to the specified resource exceeds the limit.</li>
     *         <li>ValidationException The parameter is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationInsightsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationInsightsAsyncClient.TagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/application-insights-2018-11-25/TagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(tagResourceRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Remove one or more tags (keys and values) from a specified application.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource does not exist in the customer account.</li>
     *         <li>ValidationException The parameter is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationInsightsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationInsightsAsyncClient.UntagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/application-insights-2018-11-25/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(untagResourceRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the application.
     * </p>
     *
     * @param updateApplicationRequest
     * @return A Java Future containing the result of the UpdateApplication operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The server encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>ResourceNotFoundException The resource does not exist in the customer account.</li>
     *         <li>ValidationException The parameter is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationInsightsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationInsightsAsyncClient.UpdateApplication
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/application-insights-2018-11-25/UpdateApplication"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateApplicationResponse> updateApplication(UpdateApplicationRequest updateApplicationRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateApplicationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateApplicationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateApplicationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateApplicationRequest, UpdateApplicationResponse>()
                            .withOperationName("UpdateApplication")
                            .withMarshaller(new UpdateApplicationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateApplicationRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the custom component name and/or the list of resources that make up the component.
     * </p>
     *
     * @param updateComponentRequest
     * @return A Java Future containing the result of the UpdateComponent operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceInUseException The resource is already created or in use.</li>
     *         <li>ResourceNotFoundException The resource does not exist in the customer account.</li>
     *         <li>ValidationException The parameter is not valid.</li>
     *         <li>InternalServerException The server encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationInsightsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationInsightsAsyncClient.UpdateComponent
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/application-insights-2018-11-25/UpdateComponent"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateComponentResponse> updateComponent(UpdateComponentRequest updateComponentRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateComponentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateComponentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateComponentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateComponentRequest, UpdateComponentResponse>()
                            .withOperationName("UpdateComponent")
                            .withMarshaller(new UpdateComponentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateComponentRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the monitoring configurations for the component. The configuration input parameter is an escaped JSON of
     * the configuration and should match the schema of what is returned by
     * <code>DescribeComponentConfigurationRecommendation</code>.
     * </p>
     *
     * @param updateComponentConfigurationRequest
     * @return A Java Future containing the result of the UpdateComponentConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource does not exist in the customer account.</li>
     *         <li>ValidationException The parameter is not valid.</li>
     *         <li>InternalServerException The server encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationInsightsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationInsightsAsyncClient.UpdateComponentConfiguration
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/application-insights-2018-11-25/UpdateComponentConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateComponentConfigurationResponse> updateComponentConfiguration(
            UpdateComponentConfigurationRequest updateComponentConfigurationRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateComponentConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateComponentConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateComponentConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateComponentConfigurationRequest, UpdateComponentConfigurationResponse>()
                            .withOperationName("UpdateComponentConfiguration")
                            .withMarshaller(new UpdateComponentConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateComponentConfigurationRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds a log pattern to a <code>LogPatternSet</code>.
     * </p>
     *
     * @param updateLogPatternRequest
     * @return A Java Future containing the result of the UpdateLogPattern operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceInUseException The resource is already created or in use.</li>
     *         <li>ResourceNotFoundException The resource does not exist in the customer account.</li>
     *         <li>ValidationException The parameter is not valid.</li>
     *         <li>InternalServerException The server encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationInsightsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationInsightsAsyncClient.UpdateLogPattern
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/application-insights-2018-11-25/UpdateLogPattern"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateLogPatternResponse> updateLogPattern(UpdateLogPatternRequest updateLogPatternRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateLogPatternResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateLogPatternResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateLogPatternResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateLogPatternRequest, UpdateLogPatternResponse>()
                            .withOperationName("UpdateLogPattern")
                            .withMarshaller(new UpdateLogPatternRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateLogPatternRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ApplicationInsightsException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyTagsException")
                                .exceptionBuilderSupplier(TooManyTagsException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceInUseException")
                                .exceptionBuilderSupplier(ResourceInUseException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TagsAlreadyExistException")
                                .exceptionBuilderSupplier(TagsAlreadyExistException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).build());
    }

    private <T extends ApplicationInsightsRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
