/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.applicationautoscaling.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents a step scaling policy configuration to use with Application Auto Scaling.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class StepScalingPolicyConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<StepScalingPolicyConfiguration.Builder, StepScalingPolicyConfiguration> {
    private static final SdkField<String> ADJUSTMENT_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(StepScalingPolicyConfiguration::adjustmentTypeAsString)).setter(setter(Builder::adjustmentType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AdjustmentType").build()).build();

    private static final SdkField<List<StepAdjustment>> STEP_ADJUSTMENTS_FIELD = SdkField
            .<List<StepAdjustment>> builder(MarshallingType.LIST)
            .getter(getter(StepScalingPolicyConfiguration::stepAdjustments))
            .setter(setter(Builder::stepAdjustments))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StepAdjustments").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<StepAdjustment> builder(MarshallingType.SDK_POJO)
                                            .constructor(StepAdjustment::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Integer> MIN_ADJUSTMENT_MAGNITUDE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(StepScalingPolicyConfiguration::minAdjustmentMagnitude))
            .setter(setter(Builder::minAdjustmentMagnitude))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MinAdjustmentMagnitude").build())
            .build();

    private static final SdkField<Integer> COOLDOWN_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(StepScalingPolicyConfiguration::cooldown)).setter(setter(Builder::cooldown))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Cooldown").build()).build();

    private static final SdkField<String> METRIC_AGGREGATION_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(StepScalingPolicyConfiguration::metricAggregationTypeAsString))
            .setter(setter(Builder::metricAggregationType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MetricAggregationType").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ADJUSTMENT_TYPE_FIELD,
            STEP_ADJUSTMENTS_FIELD, MIN_ADJUSTMENT_MAGNITUDE_FIELD, COOLDOWN_FIELD, METRIC_AGGREGATION_TYPE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String adjustmentType;

    private final List<StepAdjustment> stepAdjustments;

    private final Integer minAdjustmentMagnitude;

    private final Integer cooldown;

    private final String metricAggregationType;

    private StepScalingPolicyConfiguration(BuilderImpl builder) {
        this.adjustmentType = builder.adjustmentType;
        this.stepAdjustments = builder.stepAdjustments;
        this.minAdjustmentMagnitude = builder.minAdjustmentMagnitude;
        this.cooldown = builder.cooldown;
        this.metricAggregationType = builder.metricAggregationType;
    }

    /**
     * <p>
     * Specifies whether the <code>ScalingAdjustment</code> value in a <a>StepAdjustment</a> is an absolute number or a
     * percentage of the current capacity.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #adjustmentType}
     * will return {@link AdjustmentType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #adjustmentTypeAsString}.
     * </p>
     * 
     * @return Specifies whether the <code>ScalingAdjustment</code> value in a <a>StepAdjustment</a> is an absolute
     *         number or a percentage of the current capacity.
     * @see AdjustmentType
     */
    public AdjustmentType adjustmentType() {
        return AdjustmentType.fromValue(adjustmentType);
    }

    /**
     * <p>
     * Specifies whether the <code>ScalingAdjustment</code> value in a <a>StepAdjustment</a> is an absolute number or a
     * percentage of the current capacity.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #adjustmentType}
     * will return {@link AdjustmentType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #adjustmentTypeAsString}.
     * </p>
     * 
     * @return Specifies whether the <code>ScalingAdjustment</code> value in a <a>StepAdjustment</a> is an absolute
     *         number or a percentage of the current capacity.
     * @see AdjustmentType
     */
    public String adjustmentTypeAsString() {
        return adjustmentType;
    }

    /**
     * Returns true if the StepAdjustments property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasStepAdjustments() {
        return stepAdjustments != null && !(stepAdjustments instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A set of adjustments that enable you to scale based on the size of the alarm breach.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasStepAdjustments()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A set of adjustments that enable you to scale based on the size of the alarm breach.
     */
    public List<StepAdjustment> stepAdjustments() {
        return stepAdjustments;
    }

    /**
     * <p>
     * The minimum number to adjust your scalable dimension as a result of a scaling activity. If the adjustment type is
     * <code>PercentChangeInCapacity</code>, the scaling policy changes the scalable dimension of the scalable target by
     * this amount.
     * </p>
     * <p>
     * For example, suppose that you create a step scaling policy to scale out an Amazon ECS service by 25 percent and
     * you specify a <code>MinAdjustmentMagnitude</code> of 2. If the service has 4 tasks and the scaling policy is
     * performed, 25 percent of 4 is 1. However, because you specified a <code>MinAdjustmentMagnitude</code> of 2,
     * Application Auto Scaling scales out the service by 2 tasks.
     * </p>
     * 
     * @return The minimum number to adjust your scalable dimension as a result of a scaling activity. If the adjustment
     *         type is <code>PercentChangeInCapacity</code>, the scaling policy changes the scalable dimension of the
     *         scalable target by this amount.</p>
     *         <p>
     *         For example, suppose that you create a step scaling policy to scale out an Amazon ECS service by 25
     *         percent and you specify a <code>MinAdjustmentMagnitude</code> of 2. If the service has 4 tasks and the
     *         scaling policy is performed, 25 percent of 4 is 1. However, because you specified a
     *         <code>MinAdjustmentMagnitude</code> of 2, Application Auto Scaling scales out the service by 2 tasks.
     */
    public Integer minAdjustmentMagnitude() {
        return minAdjustmentMagnitude;
    }

    /**
     * <p>
     * The amount of time, in seconds, after a scaling activity completes where previous trigger-related scaling
     * activities can influence future scaling events.
     * </p>
     * <p>
     * For scale-out policies, while the cooldown period is in effect, the capacity that has been added by the previous
     * scale-out event that initiated the cooldown is calculated as part of the desired capacity for the next scale out.
     * The intention is to continuously (but not excessively) scale out. For example, an alarm triggers a step scaling
     * policy to scale out an Amazon ECS service by 2 tasks, the scaling activity completes successfully, and a cooldown
     * period of 5 minutes starts. During the cooldown period, if the alarm triggers the same policy again but at a more
     * aggressive step adjustment to scale out the service by 3 tasks, the 2 tasks that were added in the previous
     * scale-out event are considered part of that capacity and only 1 additional task is added to the desired count.
     * </p>
     * <p>
     * For scale-in policies, the cooldown period is used to block subsequent scale-in requests until it has expired.
     * The intention is to scale in conservatively to protect your application's availability. However, if another alarm
     * triggers a scale-out policy during the cooldown period after a scale-in, Application Auto Scaling scales out your
     * scalable target immediately.
     * </p>
     * 
     * @return The amount of time, in seconds, after a scaling activity completes where previous trigger-related scaling
     *         activities can influence future scaling events.</p>
     *         <p>
     *         For scale-out policies, while the cooldown period is in effect, the capacity that has been added by the
     *         previous scale-out event that initiated the cooldown is calculated as part of the desired capacity for
     *         the next scale out. The intention is to continuously (but not excessively) scale out. For example, an
     *         alarm triggers a step scaling policy to scale out an Amazon ECS service by 2 tasks, the scaling activity
     *         completes successfully, and a cooldown period of 5 minutes starts. During the cooldown period, if the
     *         alarm triggers the same policy again but at a more aggressive step adjustment to scale out the service by
     *         3 tasks, the 2 tasks that were added in the previous scale-out event are considered part of that capacity
     *         and only 1 additional task is added to the desired count.
     *         </p>
     *         <p>
     *         For scale-in policies, the cooldown period is used to block subsequent scale-in requests until it has
     *         expired. The intention is to scale in conservatively to protect your application's availability. However,
     *         if another alarm triggers a scale-out policy during the cooldown period after a scale-in, Application
     *         Auto Scaling scales out your scalable target immediately.
     */
    public Integer cooldown() {
        return cooldown;
    }

    /**
     * <p>
     * The aggregation type for the CloudWatch metrics. Valid values are <code>Minimum</code>, <code>Maximum</code>, and
     * <code>Average</code>. If the aggregation type is null, the value is treated as <code>Average</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #metricAggregationType} will return {@link MetricAggregationType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #metricAggregationTypeAsString}.
     * </p>
     * 
     * @return The aggregation type for the CloudWatch metrics. Valid values are <code>Minimum</code>,
     *         <code>Maximum</code>, and <code>Average</code>. If the aggregation type is null, the value is treated as
     *         <code>Average</code>.
     * @see MetricAggregationType
     */
    public MetricAggregationType metricAggregationType() {
        return MetricAggregationType.fromValue(metricAggregationType);
    }

    /**
     * <p>
     * The aggregation type for the CloudWatch metrics. Valid values are <code>Minimum</code>, <code>Maximum</code>, and
     * <code>Average</code>. If the aggregation type is null, the value is treated as <code>Average</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #metricAggregationType} will return {@link MetricAggregationType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #metricAggregationTypeAsString}.
     * </p>
     * 
     * @return The aggregation type for the CloudWatch metrics. Valid values are <code>Minimum</code>,
     *         <code>Maximum</code>, and <code>Average</code>. If the aggregation type is null, the value is treated as
     *         <code>Average</code>.
     * @see MetricAggregationType
     */
    public String metricAggregationTypeAsString() {
        return metricAggregationType;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(adjustmentTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(stepAdjustments());
        hashCode = 31 * hashCode + Objects.hashCode(minAdjustmentMagnitude());
        hashCode = 31 * hashCode + Objects.hashCode(cooldown());
        hashCode = 31 * hashCode + Objects.hashCode(metricAggregationTypeAsString());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof StepScalingPolicyConfiguration)) {
            return false;
        }
        StepScalingPolicyConfiguration other = (StepScalingPolicyConfiguration) obj;
        return Objects.equals(adjustmentTypeAsString(), other.adjustmentTypeAsString())
                && Objects.equals(stepAdjustments(), other.stepAdjustments())
                && Objects.equals(minAdjustmentMagnitude(), other.minAdjustmentMagnitude())
                && Objects.equals(cooldown(), other.cooldown())
                && Objects.equals(metricAggregationTypeAsString(), other.metricAggregationTypeAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("StepScalingPolicyConfiguration").add("AdjustmentType", adjustmentTypeAsString())
                .add("StepAdjustments", stepAdjustments()).add("MinAdjustmentMagnitude", minAdjustmentMagnitude())
                .add("Cooldown", cooldown()).add("MetricAggregationType", metricAggregationTypeAsString()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AdjustmentType":
            return Optional.ofNullable(clazz.cast(adjustmentTypeAsString()));
        case "StepAdjustments":
            return Optional.ofNullable(clazz.cast(stepAdjustments()));
        case "MinAdjustmentMagnitude":
            return Optional.ofNullable(clazz.cast(minAdjustmentMagnitude()));
        case "Cooldown":
            return Optional.ofNullable(clazz.cast(cooldown()));
        case "MetricAggregationType":
            return Optional.ofNullable(clazz.cast(metricAggregationTypeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<StepScalingPolicyConfiguration, T> g) {
        return obj -> g.apply((StepScalingPolicyConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, StepScalingPolicyConfiguration> {
        /**
         * <p>
         * Specifies whether the <code>ScalingAdjustment</code> value in a <a>StepAdjustment</a> is an absolute number
         * or a percentage of the current capacity.
         * </p>
         * 
         * @param adjustmentType
         *        Specifies whether the <code>ScalingAdjustment</code> value in a <a>StepAdjustment</a> is an absolute
         *        number or a percentage of the current capacity.
         * @see AdjustmentType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AdjustmentType
         */
        Builder adjustmentType(String adjustmentType);

        /**
         * <p>
         * Specifies whether the <code>ScalingAdjustment</code> value in a <a>StepAdjustment</a> is an absolute number
         * or a percentage of the current capacity.
         * </p>
         * 
         * @param adjustmentType
         *        Specifies whether the <code>ScalingAdjustment</code> value in a <a>StepAdjustment</a> is an absolute
         *        number or a percentage of the current capacity.
         * @see AdjustmentType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AdjustmentType
         */
        Builder adjustmentType(AdjustmentType adjustmentType);

        /**
         * <p>
         * A set of adjustments that enable you to scale based on the size of the alarm breach.
         * </p>
         * 
         * @param stepAdjustments
         *        A set of adjustments that enable you to scale based on the size of the alarm breach.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stepAdjustments(Collection<StepAdjustment> stepAdjustments);

        /**
         * <p>
         * A set of adjustments that enable you to scale based on the size of the alarm breach.
         * </p>
         * 
         * @param stepAdjustments
         *        A set of adjustments that enable you to scale based on the size of the alarm breach.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stepAdjustments(StepAdjustment... stepAdjustments);

        /**
         * <p>
         * A set of adjustments that enable you to scale based on the size of the alarm breach.
         * </p>
         * This is a convenience that creates an instance of the {@link List<StepAdjustment>.Builder} avoiding the need
         * to create one manually via {@link List<StepAdjustment>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<StepAdjustment>.Builder#build()} is called immediately and
         * its result is passed to {@link #stepAdjustments(List<StepAdjustment>)}.
         * 
         * @param stepAdjustments
         *        a consumer that will call methods on {@link List<StepAdjustment>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #stepAdjustments(List<StepAdjustment>)
         */
        Builder stepAdjustments(Consumer<StepAdjustment.Builder>... stepAdjustments);

        /**
         * <p>
         * The minimum number to adjust your scalable dimension as a result of a scaling activity. If the adjustment
         * type is <code>PercentChangeInCapacity</code>, the scaling policy changes the scalable dimension of the
         * scalable target by this amount.
         * </p>
         * <p>
         * For example, suppose that you create a step scaling policy to scale out an Amazon ECS service by 25 percent
         * and you specify a <code>MinAdjustmentMagnitude</code> of 2. If the service has 4 tasks and the scaling policy
         * is performed, 25 percent of 4 is 1. However, because you specified a <code>MinAdjustmentMagnitude</code> of
         * 2, Application Auto Scaling scales out the service by 2 tasks.
         * </p>
         * 
         * @param minAdjustmentMagnitude
         *        The minimum number to adjust your scalable dimension as a result of a scaling activity. If the
         *        adjustment type is <code>PercentChangeInCapacity</code>, the scaling policy changes the scalable
         *        dimension of the scalable target by this amount.</p>
         *        <p>
         *        For example, suppose that you create a step scaling policy to scale out an Amazon ECS service by 25
         *        percent and you specify a <code>MinAdjustmentMagnitude</code> of 2. If the service has 4 tasks and the
         *        scaling policy is performed, 25 percent of 4 is 1. However, because you specified a
         *        <code>MinAdjustmentMagnitude</code> of 2, Application Auto Scaling scales out the service by 2 tasks.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder minAdjustmentMagnitude(Integer minAdjustmentMagnitude);

        /**
         * <p>
         * The amount of time, in seconds, after a scaling activity completes where previous trigger-related scaling
         * activities can influence future scaling events.
         * </p>
         * <p>
         * For scale-out policies, while the cooldown period is in effect, the capacity that has been added by the
         * previous scale-out event that initiated the cooldown is calculated as part of the desired capacity for the
         * next scale out. The intention is to continuously (but not excessively) scale out. For example, an alarm
         * triggers a step scaling policy to scale out an Amazon ECS service by 2 tasks, the scaling activity completes
         * successfully, and a cooldown period of 5 minutes starts. During the cooldown period, if the alarm triggers
         * the same policy again but at a more aggressive step adjustment to scale out the service by 3 tasks, the 2
         * tasks that were added in the previous scale-out event are considered part of that capacity and only 1
         * additional task is added to the desired count.
         * </p>
         * <p>
         * For scale-in policies, the cooldown period is used to block subsequent scale-in requests until it has
         * expired. The intention is to scale in conservatively to protect your application's availability. However, if
         * another alarm triggers a scale-out policy during the cooldown period after a scale-in, Application Auto
         * Scaling scales out your scalable target immediately.
         * </p>
         * 
         * @param cooldown
         *        The amount of time, in seconds, after a scaling activity completes where previous trigger-related
         *        scaling activities can influence future scaling events.</p>
         *        <p>
         *        For scale-out policies, while the cooldown period is in effect, the capacity that has been added by
         *        the previous scale-out event that initiated the cooldown is calculated as part of the desired capacity
         *        for the next scale out. The intention is to continuously (but not excessively) scale out. For example,
         *        an alarm triggers a step scaling policy to scale out an Amazon ECS service by 2 tasks, the scaling
         *        activity completes successfully, and a cooldown period of 5 minutes starts. During the cooldown
         *        period, if the alarm triggers the same policy again but at a more aggressive step adjustment to scale
         *        out the service by 3 tasks, the 2 tasks that were added in the previous scale-out event are considered
         *        part of that capacity and only 1 additional task is added to the desired count.
         *        </p>
         *        <p>
         *        For scale-in policies, the cooldown period is used to block subsequent scale-in requests until it has
         *        expired. The intention is to scale in conservatively to protect your application's availability.
         *        However, if another alarm triggers a scale-out policy during the cooldown period after a scale-in,
         *        Application Auto Scaling scales out your scalable target immediately.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cooldown(Integer cooldown);

        /**
         * <p>
         * The aggregation type for the CloudWatch metrics. Valid values are <code>Minimum</code>, <code>Maximum</code>,
         * and <code>Average</code>. If the aggregation type is null, the value is treated as <code>Average</code>.
         * </p>
         * 
         * @param metricAggregationType
         *        The aggregation type for the CloudWatch metrics. Valid values are <code>Minimum</code>,
         *        <code>Maximum</code>, and <code>Average</code>. If the aggregation type is null, the value is treated
         *        as <code>Average</code>.
         * @see MetricAggregationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MetricAggregationType
         */
        Builder metricAggregationType(String metricAggregationType);

        /**
         * <p>
         * The aggregation type for the CloudWatch metrics. Valid values are <code>Minimum</code>, <code>Maximum</code>,
         * and <code>Average</code>. If the aggregation type is null, the value is treated as <code>Average</code>.
         * </p>
         * 
         * @param metricAggregationType
         *        The aggregation type for the CloudWatch metrics. Valid values are <code>Minimum</code>,
         *        <code>Maximum</code>, and <code>Average</code>. If the aggregation type is null, the value is treated
         *        as <code>Average</code>.
         * @see MetricAggregationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MetricAggregationType
         */
        Builder metricAggregationType(MetricAggregationType metricAggregationType);
    }

    static final class BuilderImpl implements Builder {
        private String adjustmentType;

        private List<StepAdjustment> stepAdjustments = DefaultSdkAutoConstructList.getInstance();

        private Integer minAdjustmentMagnitude;

        private Integer cooldown;

        private String metricAggregationType;

        private BuilderImpl() {
        }

        private BuilderImpl(StepScalingPolicyConfiguration model) {
            adjustmentType(model.adjustmentType);
            stepAdjustments(model.stepAdjustments);
            minAdjustmentMagnitude(model.minAdjustmentMagnitude);
            cooldown(model.cooldown);
            metricAggregationType(model.metricAggregationType);
        }

        public final String getAdjustmentTypeAsString() {
            return adjustmentType;
        }

        @Override
        public final Builder adjustmentType(String adjustmentType) {
            this.adjustmentType = adjustmentType;
            return this;
        }

        @Override
        public final Builder adjustmentType(AdjustmentType adjustmentType) {
            this.adjustmentType(adjustmentType == null ? null : adjustmentType.toString());
            return this;
        }

        public final void setAdjustmentType(String adjustmentType) {
            this.adjustmentType = adjustmentType;
        }

        public final Collection<StepAdjustment.Builder> getStepAdjustments() {
            return stepAdjustments != null ? stepAdjustments.stream().map(StepAdjustment::toBuilder).collect(Collectors.toList())
                    : null;
        }

        @Override
        public final Builder stepAdjustments(Collection<StepAdjustment> stepAdjustments) {
            this.stepAdjustments = StepAdjustmentsCopier.copy(stepAdjustments);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder stepAdjustments(StepAdjustment... stepAdjustments) {
            stepAdjustments(Arrays.asList(stepAdjustments));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder stepAdjustments(Consumer<StepAdjustment.Builder>... stepAdjustments) {
            stepAdjustments(Stream.of(stepAdjustments).map(c -> StepAdjustment.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setStepAdjustments(Collection<StepAdjustment.BuilderImpl> stepAdjustments) {
            this.stepAdjustments = StepAdjustmentsCopier.copyFromBuilder(stepAdjustments);
        }

        public final Integer getMinAdjustmentMagnitude() {
            return minAdjustmentMagnitude;
        }

        @Override
        public final Builder minAdjustmentMagnitude(Integer minAdjustmentMagnitude) {
            this.minAdjustmentMagnitude = minAdjustmentMagnitude;
            return this;
        }

        public final void setMinAdjustmentMagnitude(Integer minAdjustmentMagnitude) {
            this.minAdjustmentMagnitude = minAdjustmentMagnitude;
        }

        public final Integer getCooldown() {
            return cooldown;
        }

        @Override
        public final Builder cooldown(Integer cooldown) {
            this.cooldown = cooldown;
            return this;
        }

        public final void setCooldown(Integer cooldown) {
            this.cooldown = cooldown;
        }

        public final String getMetricAggregationTypeAsString() {
            return metricAggregationType;
        }

        @Override
        public final Builder metricAggregationType(String metricAggregationType) {
            this.metricAggregationType = metricAggregationType;
            return this;
        }

        @Override
        public final Builder metricAggregationType(MetricAggregationType metricAggregationType) {
            this.metricAggregationType(metricAggregationType == null ? null : metricAggregationType.toString());
            return this;
        }

        public final void setMetricAggregationType(String metricAggregationType) {
            this.metricAggregationType = metricAggregationType;
        }

        @Override
        public StepScalingPolicyConfiguration build() {
            return new StepScalingPolicyConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
