/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.applicationautoscaling.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents a scaling activity.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ScalingActivity implements SdkPojo, Serializable, ToCopyableBuilder<ScalingActivity.Builder, ScalingActivity> {
    private static final SdkField<String> ACTIVITY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ScalingActivity::activityId)).setter(setter(Builder::activityId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ActivityId").build()).build();

    private static final SdkField<String> SERVICE_NAMESPACE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ScalingActivity::serviceNamespaceAsString)).setter(setter(Builder::serviceNamespace))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServiceNamespace").build()).build();

    private static final SdkField<String> RESOURCE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ScalingActivity::resourceId)).setter(setter(Builder::resourceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceId").build()).build();

    private static final SdkField<String> SCALABLE_DIMENSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ScalingActivity::scalableDimensionAsString)).setter(setter(Builder::scalableDimension))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ScalableDimension").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ScalingActivity::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<String> CAUSE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ScalingActivity::cause)).setter(setter(Builder::cause))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Cause").build()).build();

    private static final SdkField<Instant> START_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(ScalingActivity::startTime)).setter(setter(Builder::startTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StartTime").build()).build();

    private static final SdkField<Instant> END_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(ScalingActivity::endTime)).setter(setter(Builder::endTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndTime").build()).build();

    private static final SdkField<String> STATUS_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ScalingActivity::statusCodeAsString)).setter(setter(Builder::statusCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StatusCode").build()).build();

    private static final SdkField<String> STATUS_MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ScalingActivity::statusMessage)).setter(setter(Builder::statusMessage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StatusMessage").build()).build();

    private static final SdkField<String> DETAILS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ScalingActivity::details)).setter(setter(Builder::details))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Details").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ACTIVITY_ID_FIELD,
            SERVICE_NAMESPACE_FIELD, RESOURCE_ID_FIELD, SCALABLE_DIMENSION_FIELD, DESCRIPTION_FIELD, CAUSE_FIELD,
            START_TIME_FIELD, END_TIME_FIELD, STATUS_CODE_FIELD, STATUS_MESSAGE_FIELD, DETAILS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String activityId;

    private final String serviceNamespace;

    private final String resourceId;

    private final String scalableDimension;

    private final String description;

    private final String cause;

    private final Instant startTime;

    private final Instant endTime;

    private final String statusCode;

    private final String statusMessage;

    private final String details;

    private ScalingActivity(BuilderImpl builder) {
        this.activityId = builder.activityId;
        this.serviceNamespace = builder.serviceNamespace;
        this.resourceId = builder.resourceId;
        this.scalableDimension = builder.scalableDimension;
        this.description = builder.description;
        this.cause = builder.cause;
        this.startTime = builder.startTime;
        this.endTime = builder.endTime;
        this.statusCode = builder.statusCode;
        this.statusMessage = builder.statusMessage;
        this.details = builder.details;
    }

    /**
     * <p>
     * The unique identifier of the scaling activity.
     * </p>
     * 
     * @return The unique identifier of the scaling activity.
     */
    public String activityId() {
        return activityId;
    }

    /**
     * <p>
     * The namespace of the AWS service that provides the resource or <code>custom-resource</code> for a resource
     * provided by your own application or service. For more information, see <a
     * href="http://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html#genref-aws-service-namespaces"
     * >AWS Service Namespaces</a> in the <i>Amazon Web Services General Reference</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #serviceNamespace}
     * will return {@link ServiceNamespace#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #serviceNamespaceAsString}.
     * </p>
     * 
     * @return The namespace of the AWS service that provides the resource or <code>custom-resource</code> for a
     *         resource provided by your own application or service. For more information, see <a href=
     *         "http://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html#genref-aws-service-namespaces"
     *         >AWS Service Namespaces</a> in the <i>Amazon Web Services General Reference</i>.
     * @see ServiceNamespace
     */
    public ServiceNamespace serviceNamespace() {
        return ServiceNamespace.fromValue(serviceNamespace);
    }

    /**
     * <p>
     * The namespace of the AWS service that provides the resource or <code>custom-resource</code> for a resource
     * provided by your own application or service. For more information, see <a
     * href="http://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html#genref-aws-service-namespaces"
     * >AWS Service Namespaces</a> in the <i>Amazon Web Services General Reference</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #serviceNamespace}
     * will return {@link ServiceNamespace#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #serviceNamespaceAsString}.
     * </p>
     * 
     * @return The namespace of the AWS service that provides the resource or <code>custom-resource</code> for a
     *         resource provided by your own application or service. For more information, see <a href=
     *         "http://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html#genref-aws-service-namespaces"
     *         >AWS Service Namespaces</a> in the <i>Amazon Web Services General Reference</i>.
     * @see ServiceNamespace
     */
    public String serviceNamespaceAsString() {
        return serviceNamespace;
    }

    /**
     * <p>
     * The identifier of the resource associated with the scaling activity. This string consists of the resource type
     * and unique identifier.
     * </p>
     * <ul>
     * <li>
     * <p>
     * ECS service - The resource type is <code>service</code> and the unique identifier is the cluster name and service
     * name. Example: <code>service/default/sample-webapp</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Spot Fleet request - The resource type is <code>spot-fleet-request</code> and the unique identifier is the Spot
     * Fleet request ID. Example: <code>spot-fleet-request/sfr-73fbd2ce-aa30-494c-8788-1cee4EXAMPLE</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * EMR cluster - The resource type is <code>instancegroup</code> and the unique identifier is the cluster ID and
     * instance group ID. Example: <code>instancegroup/j-2EEZNYKUA1NTV/ig-1791Y4E1L8YI0</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * AppStream 2.0 fleet - The resource type is <code>fleet</code> and the unique identifier is the fleet name.
     * Example: <code>fleet/sample-fleet</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * DynamoDB table - The resource type is <code>table</code> and the unique identifier is the table name. Example:
     * <code>table/my-table</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * DynamoDB global secondary index - The resource type is <code>index</code> and the unique identifier is the index
     * name. Example: <code>table/my-table/index/my-table-index</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Aurora DB cluster - The resource type is <code>cluster</code> and the unique identifier is the cluster name.
     * Example: <code>cluster:my-db-cluster</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon SageMaker endpoint variant - The resource type is <code>variant</code> and the unique identifier is the
     * resource ID. Example: <code>endpoint/my-end-point/variant/KMeansClustering</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Custom resources are not supported with a resource type. This parameter must specify the <code>OutputValue</code>
     * from the CloudFormation template stack used to access the resources. The unique identifier is defined by the
     * service provider. More information is available in our <a
     * href="https://github.com/aws/aws-auto-scaling-custom-resource">GitHub repository</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon Comprehend document classification endpoint - The resource type and unique identifier are specified using
     * the endpoint ARN. Example:
     * <code>arn:aws:comprehend:us-west-2:123456789012:document-classifier-endpoint/EXAMPLE</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Lambda provisioned concurrency - The resource type is <code>function</code> and the unique identifier is the
     * function name with a function version or alias name suffix that is not <code>$LATEST</code>. Example:
     * <code>function:my-function:prod</code> or <code>function:my-function:1</code>.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The identifier of the resource associated with the scaling activity. This string consists of the resource
     *         type and unique identifier.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         ECS service - The resource type is <code>service</code> and the unique identifier is the cluster name and
     *         service name. Example: <code>service/default/sample-webapp</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Spot Fleet request - The resource type is <code>spot-fleet-request</code> and the unique identifier is
     *         the Spot Fleet request ID. Example:
     *         <code>spot-fleet-request/sfr-73fbd2ce-aa30-494c-8788-1cee4EXAMPLE</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         EMR cluster - The resource type is <code>instancegroup</code> and the unique identifier is the cluster ID
     *         and instance group ID. Example: <code>instancegroup/j-2EEZNYKUA1NTV/ig-1791Y4E1L8YI0</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AppStream 2.0 fleet - The resource type is <code>fleet</code> and the unique identifier is the fleet
     *         name. Example: <code>fleet/sample-fleet</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         DynamoDB table - The resource type is <code>table</code> and the unique identifier is the table name.
     *         Example: <code>table/my-table</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         DynamoDB global secondary index - The resource type is <code>index</code> and the unique identifier is
     *         the index name. Example: <code>table/my-table/index/my-table-index</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Aurora DB cluster - The resource type is <code>cluster</code> and the unique identifier is the cluster
     *         name. Example: <code>cluster:my-db-cluster</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Amazon SageMaker endpoint variant - The resource type is <code>variant</code> and the unique identifier
     *         is the resource ID. Example: <code>endpoint/my-end-point/variant/KMeansClustering</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Custom resources are not supported with a resource type. This parameter must specify the
     *         <code>OutputValue</code> from the CloudFormation template stack used to access the resources. The unique
     *         identifier is defined by the service provider. More information is available in our <a
     *         href="https://github.com/aws/aws-auto-scaling-custom-resource">GitHub repository</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Amazon Comprehend document classification endpoint - The resource type and unique identifier are
     *         specified using the endpoint ARN. Example:
     *         <code>arn:aws:comprehend:us-west-2:123456789012:document-classifier-endpoint/EXAMPLE</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Lambda provisioned concurrency - The resource type is <code>function</code> and the unique identifier is
     *         the function name with a function version or alias name suffix that is not <code>$LATEST</code>. Example:
     *         <code>function:my-function:prod</code> or <code>function:my-function:1</code>.
     *         </p>
     *         </li>
     */
    public String resourceId() {
        return resourceId;
    }

    /**
     * <p>
     * The scalable dimension. This string consists of the service namespace, resource type, and scaling property.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ecs:service:DesiredCount</code> - The desired task count of an ECS service.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ec2:spot-fleet-request:TargetCapacity</code> - The target capacity of a Spot Fleet request.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>elasticmapreduce:instancegroup:InstanceCount</code> - The instance count of an EMR Instance Group.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>appstream:fleet:DesiredCapacity</code> - The desired capacity of an AppStream 2.0 fleet.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>dynamodb:table:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB table.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>dynamodb:table:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB table.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>dynamodb:index:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB global secondary
     * index.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>dynamodb:index:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB global secondary
     * index.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>rds:cluster:ReadReplicaCount</code> - The count of Aurora Replicas in an Aurora DB cluster. Available for
     * Aurora MySQL-compatible edition and Aurora PostgreSQL-compatible edition.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>sagemaker:variant:DesiredInstanceCount</code> - The number of EC2 instances for an Amazon SageMaker model
     * endpoint variant.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>custom-resource:ResourceType:Property</code> - The scalable dimension for a custom resource provided by
     * your own application or service.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>comprehend:document-classifier-endpoint:DesiredInferenceUnits</code> - The number of inference units for an
     * Amazon Comprehend document classification endpoint.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>lambda:function:ProvisionedConcurrency</code> - The provisioned concurrency for a Lambda function.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #scalableDimension}
     * will return {@link ScalableDimension#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #scalableDimensionAsString}.
     * </p>
     * 
     * @return The scalable dimension. This string consists of the service namespace, resource type, and scaling
     *         property.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ecs:service:DesiredCount</code> - The desired task count of an ECS service.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ec2:spot-fleet-request:TargetCapacity</code> - The target capacity of a Spot Fleet request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>elasticmapreduce:instancegroup:InstanceCount</code> - The instance count of an EMR Instance Group.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>appstream:fleet:DesiredCapacity</code> - The desired capacity of an AppStream 2.0 fleet.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>dynamodb:table:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB table.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>dynamodb:table:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB table.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>dynamodb:index:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB global
     *         secondary index.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>dynamodb:index:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB global
     *         secondary index.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>rds:cluster:ReadReplicaCount</code> - The count of Aurora Replicas in an Aurora DB cluster.
     *         Available for Aurora MySQL-compatible edition and Aurora PostgreSQL-compatible edition.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>sagemaker:variant:DesiredInstanceCount</code> - The number of EC2 instances for an Amazon SageMaker
     *         model endpoint variant.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>custom-resource:ResourceType:Property</code> - The scalable dimension for a custom resource
     *         provided by your own application or service.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>comprehend:document-classifier-endpoint:DesiredInferenceUnits</code> - The number of inference
     *         units for an Amazon Comprehend document classification endpoint.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>lambda:function:ProvisionedConcurrency</code> - The provisioned concurrency for a Lambda function.
     *         </p>
     *         </li>
     * @see ScalableDimension
     */
    public ScalableDimension scalableDimension() {
        return ScalableDimension.fromValue(scalableDimension);
    }

    /**
     * <p>
     * The scalable dimension. This string consists of the service namespace, resource type, and scaling property.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ecs:service:DesiredCount</code> - The desired task count of an ECS service.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ec2:spot-fleet-request:TargetCapacity</code> - The target capacity of a Spot Fleet request.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>elasticmapreduce:instancegroup:InstanceCount</code> - The instance count of an EMR Instance Group.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>appstream:fleet:DesiredCapacity</code> - The desired capacity of an AppStream 2.0 fleet.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>dynamodb:table:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB table.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>dynamodb:table:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB table.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>dynamodb:index:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB global secondary
     * index.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>dynamodb:index:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB global secondary
     * index.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>rds:cluster:ReadReplicaCount</code> - The count of Aurora Replicas in an Aurora DB cluster. Available for
     * Aurora MySQL-compatible edition and Aurora PostgreSQL-compatible edition.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>sagemaker:variant:DesiredInstanceCount</code> - The number of EC2 instances for an Amazon SageMaker model
     * endpoint variant.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>custom-resource:ResourceType:Property</code> - The scalable dimension for a custom resource provided by
     * your own application or service.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>comprehend:document-classifier-endpoint:DesiredInferenceUnits</code> - The number of inference units for an
     * Amazon Comprehend document classification endpoint.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>lambda:function:ProvisionedConcurrency</code> - The provisioned concurrency for a Lambda function.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #scalableDimension}
     * will return {@link ScalableDimension#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #scalableDimensionAsString}.
     * </p>
     * 
     * @return The scalable dimension. This string consists of the service namespace, resource type, and scaling
     *         property.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ecs:service:DesiredCount</code> - The desired task count of an ECS service.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ec2:spot-fleet-request:TargetCapacity</code> - The target capacity of a Spot Fleet request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>elasticmapreduce:instancegroup:InstanceCount</code> - The instance count of an EMR Instance Group.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>appstream:fleet:DesiredCapacity</code> - The desired capacity of an AppStream 2.0 fleet.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>dynamodb:table:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB table.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>dynamodb:table:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB table.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>dynamodb:index:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB global
     *         secondary index.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>dynamodb:index:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB global
     *         secondary index.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>rds:cluster:ReadReplicaCount</code> - The count of Aurora Replicas in an Aurora DB cluster.
     *         Available for Aurora MySQL-compatible edition and Aurora PostgreSQL-compatible edition.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>sagemaker:variant:DesiredInstanceCount</code> - The number of EC2 instances for an Amazon SageMaker
     *         model endpoint variant.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>custom-resource:ResourceType:Property</code> - The scalable dimension for a custom resource
     *         provided by your own application or service.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>comprehend:document-classifier-endpoint:DesiredInferenceUnits</code> - The number of inference
     *         units for an Amazon Comprehend document classification endpoint.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>lambda:function:ProvisionedConcurrency</code> - The provisioned concurrency for a Lambda function.
     *         </p>
     *         </li>
     * @see ScalableDimension
     */
    public String scalableDimensionAsString() {
        return scalableDimension;
    }

    /**
     * <p>
     * A simple description of what action the scaling activity intends to accomplish.
     * </p>
     * 
     * @return A simple description of what action the scaling activity intends to accomplish.
     */
    public String description() {
        return description;
    }

    /**
     * <p>
     * A simple description of what caused the scaling activity to happen.
     * </p>
     * 
     * @return A simple description of what caused the scaling activity to happen.
     */
    public String cause() {
        return cause;
    }

    /**
     * <p>
     * The Unix timestamp for when the scaling activity began.
     * </p>
     * 
     * @return The Unix timestamp for when the scaling activity began.
     */
    public Instant startTime() {
        return startTime;
    }

    /**
     * <p>
     * The Unix timestamp for when the scaling activity ended.
     * </p>
     * 
     * @return The Unix timestamp for when the scaling activity ended.
     */
    public Instant endTime() {
        return endTime;
    }

    /**
     * <p>
     * Indicates the status of the scaling activity.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #statusCode} will
     * return {@link ScalingActivityStatusCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #statusCodeAsString}.
     * </p>
     * 
     * @return Indicates the status of the scaling activity.
     * @see ScalingActivityStatusCode
     */
    public ScalingActivityStatusCode statusCode() {
        return ScalingActivityStatusCode.fromValue(statusCode);
    }

    /**
     * <p>
     * Indicates the status of the scaling activity.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #statusCode} will
     * return {@link ScalingActivityStatusCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #statusCodeAsString}.
     * </p>
     * 
     * @return Indicates the status of the scaling activity.
     * @see ScalingActivityStatusCode
     */
    public String statusCodeAsString() {
        return statusCode;
    }

    /**
     * <p>
     * A simple message about the current status of the scaling activity.
     * </p>
     * 
     * @return A simple message about the current status of the scaling activity.
     */
    public String statusMessage() {
        return statusMessage;
    }

    /**
     * <p>
     * The details about the scaling activity.
     * </p>
     * 
     * @return The details about the scaling activity.
     */
    public String details() {
        return details;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(activityId());
        hashCode = 31 * hashCode + Objects.hashCode(serviceNamespaceAsString());
        hashCode = 31 * hashCode + Objects.hashCode(resourceId());
        hashCode = 31 * hashCode + Objects.hashCode(scalableDimensionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(cause());
        hashCode = 31 * hashCode + Objects.hashCode(startTime());
        hashCode = 31 * hashCode + Objects.hashCode(endTime());
        hashCode = 31 * hashCode + Objects.hashCode(statusCodeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusMessage());
        hashCode = 31 * hashCode + Objects.hashCode(details());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ScalingActivity)) {
            return false;
        }
        ScalingActivity other = (ScalingActivity) obj;
        return Objects.equals(activityId(), other.activityId())
                && Objects.equals(serviceNamespaceAsString(), other.serviceNamespaceAsString())
                && Objects.equals(resourceId(), other.resourceId())
                && Objects.equals(scalableDimensionAsString(), other.scalableDimensionAsString())
                && Objects.equals(description(), other.description()) && Objects.equals(cause(), other.cause())
                && Objects.equals(startTime(), other.startTime()) && Objects.equals(endTime(), other.endTime())
                && Objects.equals(statusCodeAsString(), other.statusCodeAsString())
                && Objects.equals(statusMessage(), other.statusMessage()) && Objects.equals(details(), other.details());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("ScalingActivity").add("ActivityId", activityId())
                .add("ServiceNamespace", serviceNamespaceAsString()).add("ResourceId", resourceId())
                .add("ScalableDimension", scalableDimensionAsString()).add("Description", description()).add("Cause", cause())
                .add("StartTime", startTime()).add("EndTime", endTime()).add("StatusCode", statusCodeAsString())
                .add("StatusMessage", statusMessage()).add("Details", details()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ActivityId":
            return Optional.ofNullable(clazz.cast(activityId()));
        case "ServiceNamespace":
            return Optional.ofNullable(clazz.cast(serviceNamespaceAsString()));
        case "ResourceId":
            return Optional.ofNullable(clazz.cast(resourceId()));
        case "ScalableDimension":
            return Optional.ofNullable(clazz.cast(scalableDimensionAsString()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "Cause":
            return Optional.ofNullable(clazz.cast(cause()));
        case "StartTime":
            return Optional.ofNullable(clazz.cast(startTime()));
        case "EndTime":
            return Optional.ofNullable(clazz.cast(endTime()));
        case "StatusCode":
            return Optional.ofNullable(clazz.cast(statusCodeAsString()));
        case "StatusMessage":
            return Optional.ofNullable(clazz.cast(statusMessage()));
        case "Details":
            return Optional.ofNullable(clazz.cast(details()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ScalingActivity, T> g) {
        return obj -> g.apply((ScalingActivity) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ScalingActivity> {
        /**
         * <p>
         * The unique identifier of the scaling activity.
         * </p>
         * 
         * @param activityId
         *        The unique identifier of the scaling activity.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder activityId(String activityId);

        /**
         * <p>
         * The namespace of the AWS service that provides the resource or <code>custom-resource</code> for a resource
         * provided by your own application or service. For more information, see <a href=
         * "http://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html#genref-aws-service-namespaces">AWS
         * Service Namespaces</a> in the <i>Amazon Web Services General Reference</i>.
         * </p>
         * 
         * @param serviceNamespace
         *        The namespace of the AWS service that provides the resource or <code>custom-resource</code> for a
         *        resource provided by your own application or service. For more information, see <a href=
         *        "http://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html#genref-aws-service-namespaces"
         *        >AWS Service Namespaces</a> in the <i>Amazon Web Services General Reference</i>.
         * @see ServiceNamespace
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ServiceNamespace
         */
        Builder serviceNamespace(String serviceNamespace);

        /**
         * <p>
         * The namespace of the AWS service that provides the resource or <code>custom-resource</code> for a resource
         * provided by your own application or service. For more information, see <a href=
         * "http://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html#genref-aws-service-namespaces">AWS
         * Service Namespaces</a> in the <i>Amazon Web Services General Reference</i>.
         * </p>
         * 
         * @param serviceNamespace
         *        The namespace of the AWS service that provides the resource or <code>custom-resource</code> for a
         *        resource provided by your own application or service. For more information, see <a href=
         *        "http://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html#genref-aws-service-namespaces"
         *        >AWS Service Namespaces</a> in the <i>Amazon Web Services General Reference</i>.
         * @see ServiceNamespace
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ServiceNamespace
         */
        Builder serviceNamespace(ServiceNamespace serviceNamespace);

        /**
         * <p>
         * The identifier of the resource associated with the scaling activity. This string consists of the resource
         * type and unique identifier.
         * </p>
         * <ul>
         * <li>
         * <p>
         * ECS service - The resource type is <code>service</code> and the unique identifier is the cluster name and
         * service name. Example: <code>service/default/sample-webapp</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Spot Fleet request - The resource type is <code>spot-fleet-request</code> and the unique identifier is the
         * Spot Fleet request ID. Example: <code>spot-fleet-request/sfr-73fbd2ce-aa30-494c-8788-1cee4EXAMPLE</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * EMR cluster - The resource type is <code>instancegroup</code> and the unique identifier is the cluster ID and
         * instance group ID. Example: <code>instancegroup/j-2EEZNYKUA1NTV/ig-1791Y4E1L8YI0</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * AppStream 2.0 fleet - The resource type is <code>fleet</code> and the unique identifier is the fleet name.
         * Example: <code>fleet/sample-fleet</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * DynamoDB table - The resource type is <code>table</code> and the unique identifier is the table name.
         * Example: <code>table/my-table</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * DynamoDB global secondary index - The resource type is <code>index</code> and the unique identifier is the
         * index name. Example: <code>table/my-table/index/my-table-index</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Aurora DB cluster - The resource type is <code>cluster</code> and the unique identifier is the cluster name.
         * Example: <code>cluster:my-db-cluster</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Amazon SageMaker endpoint variant - The resource type is <code>variant</code> and the unique identifier is
         * the resource ID. Example: <code>endpoint/my-end-point/variant/KMeansClustering</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Custom resources are not supported with a resource type. This parameter must specify the
         * <code>OutputValue</code> from the CloudFormation template stack used to access the resources. The unique
         * identifier is defined by the service provider. More information is available in our <a
         * href="https://github.com/aws/aws-auto-scaling-custom-resource">GitHub repository</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Amazon Comprehend document classification endpoint - The resource type and unique identifier are specified
         * using the endpoint ARN. Example:
         * <code>arn:aws:comprehend:us-west-2:123456789012:document-classifier-endpoint/EXAMPLE</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Lambda provisioned concurrency - The resource type is <code>function</code> and the unique identifier is the
         * function name with a function version or alias name suffix that is not <code>$LATEST</code>. Example:
         * <code>function:my-function:prod</code> or <code>function:my-function:1</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param resourceId
         *        The identifier of the resource associated with the scaling activity. This string consists of the
         *        resource type and unique identifier.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        ECS service - The resource type is <code>service</code> and the unique identifier is the cluster name
         *        and service name. Example: <code>service/default/sample-webapp</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Spot Fleet request - The resource type is <code>spot-fleet-request</code> and the unique identifier is
         *        the Spot Fleet request ID. Example:
         *        <code>spot-fleet-request/sfr-73fbd2ce-aa30-494c-8788-1cee4EXAMPLE</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        EMR cluster - The resource type is <code>instancegroup</code> and the unique identifier is the cluster
         *        ID and instance group ID. Example: <code>instancegroup/j-2EEZNYKUA1NTV/ig-1791Y4E1L8YI0</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        AppStream 2.0 fleet - The resource type is <code>fleet</code> and the unique identifier is the fleet
         *        name. Example: <code>fleet/sample-fleet</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        DynamoDB table - The resource type is <code>table</code> and the unique identifier is the table name.
         *        Example: <code>table/my-table</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        DynamoDB global secondary index - The resource type is <code>index</code> and the unique identifier is
         *        the index name. Example: <code>table/my-table/index/my-table-index</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Aurora DB cluster - The resource type is <code>cluster</code> and the unique identifier is the cluster
         *        name. Example: <code>cluster:my-db-cluster</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Amazon SageMaker endpoint variant - The resource type is <code>variant</code> and the unique
         *        identifier is the resource ID. Example: <code>endpoint/my-end-point/variant/KMeansClustering</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Custom resources are not supported with a resource type. This parameter must specify the
         *        <code>OutputValue</code> from the CloudFormation template stack used to access the resources. The
         *        unique identifier is defined by the service provider. More information is available in our <a
         *        href="https://github.com/aws/aws-auto-scaling-custom-resource">GitHub repository</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Amazon Comprehend document classification endpoint - The resource type and unique identifier are
         *        specified using the endpoint ARN. Example:
         *        <code>arn:aws:comprehend:us-west-2:123456789012:document-classifier-endpoint/EXAMPLE</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Lambda provisioned concurrency - The resource type is <code>function</code> and the unique identifier
         *        is the function name with a function version or alias name suffix that is not <code>$LATEST</code>.
         *        Example: <code>function:my-function:prod</code> or <code>function:my-function:1</code>.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceId(String resourceId);

        /**
         * <p>
         * The scalable dimension. This string consists of the service namespace, resource type, and scaling property.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ecs:service:DesiredCount</code> - The desired task count of an ECS service.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ec2:spot-fleet-request:TargetCapacity</code> - The target capacity of a Spot Fleet request.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>elasticmapreduce:instancegroup:InstanceCount</code> - The instance count of an EMR Instance Group.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>appstream:fleet:DesiredCapacity</code> - The desired capacity of an AppStream 2.0 fleet.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>dynamodb:table:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB table.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>dynamodb:table:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB table.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>dynamodb:index:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB global secondary
         * index.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>dynamodb:index:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB global
         * secondary index.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>rds:cluster:ReadReplicaCount</code> - The count of Aurora Replicas in an Aurora DB cluster. Available
         * for Aurora MySQL-compatible edition and Aurora PostgreSQL-compatible edition.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>sagemaker:variant:DesiredInstanceCount</code> - The number of EC2 instances for an Amazon SageMaker
         * model endpoint variant.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>custom-resource:ResourceType:Property</code> - The scalable dimension for a custom resource provided by
         * your own application or service.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>comprehend:document-classifier-endpoint:DesiredInferenceUnits</code> - The number of inference units
         * for an Amazon Comprehend document classification endpoint.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>lambda:function:ProvisionedConcurrency</code> - The provisioned concurrency for a Lambda function.
         * </p>
         * </li>
         * </ul>
         * 
         * @param scalableDimension
         *        The scalable dimension. This string consists of the service namespace, resource type, and scaling
         *        property.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ecs:service:DesiredCount</code> - The desired task count of an ECS service.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ec2:spot-fleet-request:TargetCapacity</code> - The target capacity of a Spot Fleet request.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>elasticmapreduce:instancegroup:InstanceCount</code> - The instance count of an EMR Instance
         *        Group.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>appstream:fleet:DesiredCapacity</code> - The desired capacity of an AppStream 2.0 fleet.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>dynamodb:table:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB table.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>dynamodb:table:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB table.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>dynamodb:index:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB global
         *        secondary index.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>dynamodb:index:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB global
         *        secondary index.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>rds:cluster:ReadReplicaCount</code> - The count of Aurora Replicas in an Aurora DB cluster.
         *        Available for Aurora MySQL-compatible edition and Aurora PostgreSQL-compatible edition.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>sagemaker:variant:DesiredInstanceCount</code> - The number of EC2 instances for an Amazon
         *        SageMaker model endpoint variant.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>custom-resource:ResourceType:Property</code> - The scalable dimension for a custom resource
         *        provided by your own application or service.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>comprehend:document-classifier-endpoint:DesiredInferenceUnits</code> - The number of inference
         *        units for an Amazon Comprehend document classification endpoint.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>lambda:function:ProvisionedConcurrency</code> - The provisioned concurrency for a Lambda
         *        function.
         *        </p>
         *        </li>
         * @see ScalableDimension
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ScalableDimension
         */
        Builder scalableDimension(String scalableDimension);

        /**
         * <p>
         * The scalable dimension. This string consists of the service namespace, resource type, and scaling property.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ecs:service:DesiredCount</code> - The desired task count of an ECS service.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ec2:spot-fleet-request:TargetCapacity</code> - The target capacity of a Spot Fleet request.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>elasticmapreduce:instancegroup:InstanceCount</code> - The instance count of an EMR Instance Group.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>appstream:fleet:DesiredCapacity</code> - The desired capacity of an AppStream 2.0 fleet.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>dynamodb:table:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB table.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>dynamodb:table:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB table.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>dynamodb:index:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB global secondary
         * index.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>dynamodb:index:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB global
         * secondary index.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>rds:cluster:ReadReplicaCount</code> - The count of Aurora Replicas in an Aurora DB cluster. Available
         * for Aurora MySQL-compatible edition and Aurora PostgreSQL-compatible edition.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>sagemaker:variant:DesiredInstanceCount</code> - The number of EC2 instances for an Amazon SageMaker
         * model endpoint variant.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>custom-resource:ResourceType:Property</code> - The scalable dimension for a custom resource provided by
         * your own application or service.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>comprehend:document-classifier-endpoint:DesiredInferenceUnits</code> - The number of inference units
         * for an Amazon Comprehend document classification endpoint.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>lambda:function:ProvisionedConcurrency</code> - The provisioned concurrency for a Lambda function.
         * </p>
         * </li>
         * </ul>
         * 
         * @param scalableDimension
         *        The scalable dimension. This string consists of the service namespace, resource type, and scaling
         *        property.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ecs:service:DesiredCount</code> - The desired task count of an ECS service.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ec2:spot-fleet-request:TargetCapacity</code> - The target capacity of a Spot Fleet request.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>elasticmapreduce:instancegroup:InstanceCount</code> - The instance count of an EMR Instance
         *        Group.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>appstream:fleet:DesiredCapacity</code> - The desired capacity of an AppStream 2.0 fleet.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>dynamodb:table:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB table.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>dynamodb:table:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB table.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>dynamodb:index:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB global
         *        secondary index.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>dynamodb:index:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB global
         *        secondary index.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>rds:cluster:ReadReplicaCount</code> - The count of Aurora Replicas in an Aurora DB cluster.
         *        Available for Aurora MySQL-compatible edition and Aurora PostgreSQL-compatible edition.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>sagemaker:variant:DesiredInstanceCount</code> - The number of EC2 instances for an Amazon
         *        SageMaker model endpoint variant.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>custom-resource:ResourceType:Property</code> - The scalable dimension for a custom resource
         *        provided by your own application or service.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>comprehend:document-classifier-endpoint:DesiredInferenceUnits</code> - The number of inference
         *        units for an Amazon Comprehend document classification endpoint.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>lambda:function:ProvisionedConcurrency</code> - The provisioned concurrency for a Lambda
         *        function.
         *        </p>
         *        </li>
         * @see ScalableDimension
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ScalableDimension
         */
        Builder scalableDimension(ScalableDimension scalableDimension);

        /**
         * <p>
         * A simple description of what action the scaling activity intends to accomplish.
         * </p>
         * 
         * @param description
         *        A simple description of what action the scaling activity intends to accomplish.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * A simple description of what caused the scaling activity to happen.
         * </p>
         * 
         * @param cause
         *        A simple description of what caused the scaling activity to happen.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cause(String cause);

        /**
         * <p>
         * The Unix timestamp for when the scaling activity began.
         * </p>
         * 
         * @param startTime
         *        The Unix timestamp for when the scaling activity began.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTime(Instant startTime);

        /**
         * <p>
         * The Unix timestamp for when the scaling activity ended.
         * </p>
         * 
         * @param endTime
         *        The Unix timestamp for when the scaling activity ended.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endTime(Instant endTime);

        /**
         * <p>
         * Indicates the status of the scaling activity.
         * </p>
         * 
         * @param statusCode
         *        Indicates the status of the scaling activity.
         * @see ScalingActivityStatusCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ScalingActivityStatusCode
         */
        Builder statusCode(String statusCode);

        /**
         * <p>
         * Indicates the status of the scaling activity.
         * </p>
         * 
         * @param statusCode
         *        Indicates the status of the scaling activity.
         * @see ScalingActivityStatusCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ScalingActivityStatusCode
         */
        Builder statusCode(ScalingActivityStatusCode statusCode);

        /**
         * <p>
         * A simple message about the current status of the scaling activity.
         * </p>
         * 
         * @param statusMessage
         *        A simple message about the current status of the scaling activity.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusMessage(String statusMessage);

        /**
         * <p>
         * The details about the scaling activity.
         * </p>
         * 
         * @param details
         *        The details about the scaling activity.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder details(String details);
    }

    static final class BuilderImpl implements Builder {
        private String activityId;

        private String serviceNamespace;

        private String resourceId;

        private String scalableDimension;

        private String description;

        private String cause;

        private Instant startTime;

        private Instant endTime;

        private String statusCode;

        private String statusMessage;

        private String details;

        private BuilderImpl() {
        }

        private BuilderImpl(ScalingActivity model) {
            activityId(model.activityId);
            serviceNamespace(model.serviceNamespace);
            resourceId(model.resourceId);
            scalableDimension(model.scalableDimension);
            description(model.description);
            cause(model.cause);
            startTime(model.startTime);
            endTime(model.endTime);
            statusCode(model.statusCode);
            statusMessage(model.statusMessage);
            details(model.details);
        }

        public final String getActivityId() {
            return activityId;
        }

        @Override
        public final Builder activityId(String activityId) {
            this.activityId = activityId;
            return this;
        }

        public final void setActivityId(String activityId) {
            this.activityId = activityId;
        }

        public final String getServiceNamespaceAsString() {
            return serviceNamespace;
        }

        @Override
        public final Builder serviceNamespace(String serviceNamespace) {
            this.serviceNamespace = serviceNamespace;
            return this;
        }

        @Override
        public final Builder serviceNamespace(ServiceNamespace serviceNamespace) {
            this.serviceNamespace(serviceNamespace == null ? null : serviceNamespace.toString());
            return this;
        }

        public final void setServiceNamespace(String serviceNamespace) {
            this.serviceNamespace = serviceNamespace;
        }

        public final String getResourceId() {
            return resourceId;
        }

        @Override
        public final Builder resourceId(String resourceId) {
            this.resourceId = resourceId;
            return this;
        }

        public final void setResourceId(String resourceId) {
            this.resourceId = resourceId;
        }

        public final String getScalableDimensionAsString() {
            return scalableDimension;
        }

        @Override
        public final Builder scalableDimension(String scalableDimension) {
            this.scalableDimension = scalableDimension;
            return this;
        }

        @Override
        public final Builder scalableDimension(ScalableDimension scalableDimension) {
            this.scalableDimension(scalableDimension == null ? null : scalableDimension.toString());
            return this;
        }

        public final void setScalableDimension(String scalableDimension) {
            this.scalableDimension = scalableDimension;
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final String getCause() {
            return cause;
        }

        @Override
        public final Builder cause(String cause) {
            this.cause = cause;
            return this;
        }

        public final void setCause(String cause) {
            this.cause = cause;
        }

        public final Instant getStartTime() {
            return startTime;
        }

        @Override
        public final Builder startTime(Instant startTime) {
            this.startTime = startTime;
            return this;
        }

        public final void setStartTime(Instant startTime) {
            this.startTime = startTime;
        }

        public final Instant getEndTime() {
            return endTime;
        }

        @Override
        public final Builder endTime(Instant endTime) {
            this.endTime = endTime;
            return this;
        }

        public final void setEndTime(Instant endTime) {
            this.endTime = endTime;
        }

        public final String getStatusCodeAsString() {
            return statusCode;
        }

        @Override
        public final Builder statusCode(String statusCode) {
            this.statusCode = statusCode;
            return this;
        }

        @Override
        public final Builder statusCode(ScalingActivityStatusCode statusCode) {
            this.statusCode(statusCode == null ? null : statusCode.toString());
            return this;
        }

        public final void setStatusCode(String statusCode) {
            this.statusCode = statusCode;
        }

        public final String getStatusMessage() {
            return statusMessage;
        }

        @Override
        public final Builder statusMessage(String statusMessage) {
            this.statusMessage = statusMessage;
            return this;
        }

        public final void setStatusMessage(String statusMessage) {
            this.statusMessage = statusMessage;
        }

        public final String getDetails() {
            return details;
        }

        @Override
        public final Builder details(String details) {
            this.details = details;
            return this;
        }

        public final void setDetails(String details) {
            this.details = details;
        }

        @Override
        public ScalingActivity build() {
            return new ScalingActivity(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
