/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.applicationautoscaling;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.applicationautoscaling.model.ApplicationAutoScalingException;
import software.amazon.awssdk.services.applicationautoscaling.model.ApplicationAutoScalingRequest;
import software.amazon.awssdk.services.applicationautoscaling.model.ConcurrentUpdateException;
import software.amazon.awssdk.services.applicationautoscaling.model.DeleteScalingPolicyRequest;
import software.amazon.awssdk.services.applicationautoscaling.model.DeleteScalingPolicyResponse;
import software.amazon.awssdk.services.applicationautoscaling.model.DeleteScheduledActionRequest;
import software.amazon.awssdk.services.applicationautoscaling.model.DeleteScheduledActionResponse;
import software.amazon.awssdk.services.applicationautoscaling.model.DeregisterScalableTargetRequest;
import software.amazon.awssdk.services.applicationautoscaling.model.DeregisterScalableTargetResponse;
import software.amazon.awssdk.services.applicationautoscaling.model.DescribeScalableTargetsRequest;
import software.amazon.awssdk.services.applicationautoscaling.model.DescribeScalableTargetsResponse;
import software.amazon.awssdk.services.applicationautoscaling.model.DescribeScalingActivitiesRequest;
import software.amazon.awssdk.services.applicationautoscaling.model.DescribeScalingActivitiesResponse;
import software.amazon.awssdk.services.applicationautoscaling.model.DescribeScalingPoliciesRequest;
import software.amazon.awssdk.services.applicationautoscaling.model.DescribeScalingPoliciesResponse;
import software.amazon.awssdk.services.applicationautoscaling.model.DescribeScheduledActionsRequest;
import software.amazon.awssdk.services.applicationautoscaling.model.DescribeScheduledActionsResponse;
import software.amazon.awssdk.services.applicationautoscaling.model.FailedResourceAccessException;
import software.amazon.awssdk.services.applicationautoscaling.model.InternalServiceException;
import software.amazon.awssdk.services.applicationautoscaling.model.InvalidNextTokenException;
import software.amazon.awssdk.services.applicationautoscaling.model.LimitExceededException;
import software.amazon.awssdk.services.applicationautoscaling.model.ObjectNotFoundException;
import software.amazon.awssdk.services.applicationautoscaling.model.PutScalingPolicyRequest;
import software.amazon.awssdk.services.applicationautoscaling.model.PutScalingPolicyResponse;
import software.amazon.awssdk.services.applicationautoscaling.model.PutScheduledActionRequest;
import software.amazon.awssdk.services.applicationautoscaling.model.PutScheduledActionResponse;
import software.amazon.awssdk.services.applicationautoscaling.model.RegisterScalableTargetRequest;
import software.amazon.awssdk.services.applicationautoscaling.model.RegisterScalableTargetResponse;
import software.amazon.awssdk.services.applicationautoscaling.model.ValidationException;
import software.amazon.awssdk.services.applicationautoscaling.paginators.DescribeScalableTargetsIterable;
import software.amazon.awssdk.services.applicationautoscaling.paginators.DescribeScalingActivitiesIterable;
import software.amazon.awssdk.services.applicationautoscaling.paginators.DescribeScalingPoliciesIterable;
import software.amazon.awssdk.services.applicationautoscaling.paginators.DescribeScheduledActionsIterable;
import software.amazon.awssdk.services.applicationautoscaling.transform.DeleteScalingPolicyRequestMarshaller;
import software.amazon.awssdk.services.applicationautoscaling.transform.DeleteScheduledActionRequestMarshaller;
import software.amazon.awssdk.services.applicationautoscaling.transform.DeregisterScalableTargetRequestMarshaller;
import software.amazon.awssdk.services.applicationautoscaling.transform.DescribeScalableTargetsRequestMarshaller;
import software.amazon.awssdk.services.applicationautoscaling.transform.DescribeScalingActivitiesRequestMarshaller;
import software.amazon.awssdk.services.applicationautoscaling.transform.DescribeScalingPoliciesRequestMarshaller;
import software.amazon.awssdk.services.applicationautoscaling.transform.DescribeScheduledActionsRequestMarshaller;
import software.amazon.awssdk.services.applicationautoscaling.transform.PutScalingPolicyRequestMarshaller;
import software.amazon.awssdk.services.applicationautoscaling.transform.PutScheduledActionRequestMarshaller;
import software.amazon.awssdk.services.applicationautoscaling.transform.RegisterScalableTargetRequestMarshaller;

/**
 * Internal implementation of {@link ApplicationAutoScalingClient}.
 *
 * @see ApplicationAutoScalingClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultApplicationAutoScalingClient implements ApplicationAutoScalingClient {
    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultApplicationAutoScalingClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Deletes the specified scaling policy for an Application Auto Scaling scalable target.
     * </p>
     * <p>
     * Deleting a step scaling policy deletes the underlying alarm action, but does not delete the CloudWatch alarm
     * associated with the scaling policy, even if it no longer has an associated action.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/autoscaling/application/userguide/application-auto-scaling-step-scaling-policies.html#delete-step-scaling-policy"
     * >Delete a Step Scaling Policy</a> and <a href=
     * "https://docs.aws.amazon.com/autoscaling/application/userguide/application-auto-scaling-target-tracking.html#delete-target-tracking-policy"
     * >Delete a Target Tracking Scaling Policy</a> in the <i>Application Auto Scaling User Guide</i>.
     * </p>
     * <p>
     * To create a scaling policy or update an existing one, see <a>PutScalingPolicy</a>.
     * </p>
     *
     * @param deleteScalingPolicyRequest
     * @return Result of the DeleteScalingPolicy operation returned by the service.
     * @throws ValidationException
     *         An exception was thrown for a validation issue. Review the available parameters for the API request.
     * @throws ObjectNotFoundException
     *         The specified object could not be found. For any operation that depends on the existence of a scalable
     *         target, this exception is thrown if the scalable target with the specified service namespace, resource
     *         ID, and scalable dimension does not exist. For any operation that deletes or deregisters a resource, this
     *         exception is thrown if the resource cannot be found.
     * @throws ConcurrentUpdateException
     *         Concurrent updates caused an exception, for example, if you request an update to an Application Auto
     *         Scaling resource that already has a pending update.
     * @throws InternalServiceException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationAutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationAutoScalingClient.DeleteScalingPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/application-autoscaling-2016-02-06/DeleteScalingPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteScalingPolicyResponse deleteScalingPolicy(DeleteScalingPolicyRequest deleteScalingPolicyRequest)
            throws ValidationException, ObjectNotFoundException, ConcurrentUpdateException, InternalServiceException,
            AwsServiceException, SdkClientException, ApplicationAutoScalingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteScalingPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteScalingPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeleteScalingPolicyRequest, DeleteScalingPolicyResponse>()
                .withOperationName("DeleteScalingPolicy").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteScalingPolicyRequest)
                .withMarshaller(new DeleteScalingPolicyRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes the specified scheduled action for an Application Auto Scaling scalable target.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/autoscaling/application/userguide/application-auto-scaling-scheduled-scaling.html#delete-scheduled-action"
     * >Delete a Scheduled Action</a> in the <i>Application Auto Scaling User Guide</i>.
     * </p>
     *
     * @param deleteScheduledActionRequest
     * @return Result of the DeleteScheduledAction operation returned by the service.
     * @throws ValidationException
     *         An exception was thrown for a validation issue. Review the available parameters for the API request.
     * @throws ObjectNotFoundException
     *         The specified object could not be found. For any operation that depends on the existence of a scalable
     *         target, this exception is thrown if the scalable target with the specified service namespace, resource
     *         ID, and scalable dimension does not exist. For any operation that deletes or deregisters a resource, this
     *         exception is thrown if the resource cannot be found.
     * @throws ConcurrentUpdateException
     *         Concurrent updates caused an exception, for example, if you request an update to an Application Auto
     *         Scaling resource that already has a pending update.
     * @throws InternalServiceException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationAutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationAutoScalingClient.DeleteScheduledAction
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/application-autoscaling-2016-02-06/DeleteScheduledAction"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteScheduledActionResponse deleteScheduledAction(DeleteScheduledActionRequest deleteScheduledActionRequest)
            throws ValidationException, ObjectNotFoundException, ConcurrentUpdateException, InternalServiceException,
            AwsServiceException, SdkClientException, ApplicationAutoScalingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteScheduledActionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteScheduledActionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeleteScheduledActionRequest, DeleteScheduledActionResponse>()
                .withOperationName("DeleteScheduledAction").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteScheduledActionRequest)
                .withMarshaller(new DeleteScheduledActionRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deregisters an Application Auto Scaling scalable target.
     * </p>
     * <p>
     * Deregistering a scalable target deletes the scaling policies that are associated with it.
     * </p>
     * <p>
     * To create a scalable target or update an existing one, see <a>RegisterScalableTarget</a>.
     * </p>
     *
     * @param deregisterScalableTargetRequest
     * @return Result of the DeregisterScalableTarget operation returned by the service.
     * @throws ValidationException
     *         An exception was thrown for a validation issue. Review the available parameters for the API request.
     * @throws ObjectNotFoundException
     *         The specified object could not be found. For any operation that depends on the existence of a scalable
     *         target, this exception is thrown if the scalable target with the specified service namespace, resource
     *         ID, and scalable dimension does not exist. For any operation that deletes or deregisters a resource, this
     *         exception is thrown if the resource cannot be found.
     * @throws ConcurrentUpdateException
     *         Concurrent updates caused an exception, for example, if you request an update to an Application Auto
     *         Scaling resource that already has a pending update.
     * @throws InternalServiceException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationAutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationAutoScalingClient.DeregisterScalableTarget
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/application-autoscaling-2016-02-06/DeregisterScalableTarget"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeregisterScalableTargetResponse deregisterScalableTarget(
            DeregisterScalableTargetRequest deregisterScalableTargetRequest) throws ValidationException, ObjectNotFoundException,
            ConcurrentUpdateException, InternalServiceException, AwsServiceException, SdkClientException,
            ApplicationAutoScalingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeregisterScalableTargetResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeregisterScalableTargetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<DeregisterScalableTargetRequest, DeregisterScalableTargetResponse>()
                        .withOperationName("DeregisterScalableTarget").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(deregisterScalableTargetRequest)
                        .withMarshaller(new DeregisterScalableTargetRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Gets information about the scalable targets in the specified namespace.
     * </p>
     * <p>
     * You can filter the results using <code>ResourceIds</code> and <code>ScalableDimension</code>.
     * </p>
     * <p>
     * To create a scalable target or update an existing one, see <a>RegisterScalableTarget</a>. If you are no longer
     * using a scalable target, you can deregister it using <a>DeregisterScalableTarget</a>.
     * </p>
     *
     * @param describeScalableTargetsRequest
     * @return Result of the DescribeScalableTargets operation returned by the service.
     * @throws ValidationException
     *         An exception was thrown for a validation issue. Review the available parameters for the API request.
     * @throws InvalidNextTokenException
     *         The next token supplied was invalid.
     * @throws ConcurrentUpdateException
     *         Concurrent updates caused an exception, for example, if you request an update to an Application Auto
     *         Scaling resource that already has a pending update.
     * @throws InternalServiceException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationAutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationAutoScalingClient.DescribeScalableTargets
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/application-autoscaling-2016-02-06/DescribeScalableTargets"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeScalableTargetsResponse describeScalableTargets(DescribeScalableTargetsRequest describeScalableTargetsRequest)
            throws ValidationException, InvalidNextTokenException, ConcurrentUpdateException, InternalServiceException,
            AwsServiceException, SdkClientException, ApplicationAutoScalingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeScalableTargetsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeScalableTargetsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DescribeScalableTargetsRequest, DescribeScalableTargetsResponse>()
                .withOperationName("DescribeScalableTargets").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeScalableTargetsRequest)
                .withMarshaller(new DescribeScalableTargetsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Gets information about the scalable targets in the specified namespace.
     * </p>
     * <p>
     * You can filter the results using <code>ResourceIds</code> and <code>ScalableDimension</code>.
     * </p>
     * <p>
     * To create a scalable target or update an existing one, see <a>RegisterScalableTarget</a>. If you are no longer
     * using a scalable target, you can deregister it using <a>DeregisterScalableTarget</a>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeScalableTargets(software.amazon.awssdk.services.applicationautoscaling.model.DescribeScalableTargetsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.applicationautoscaling.paginators.DescribeScalableTargetsIterable responses = client.describeScalableTargetsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.applicationautoscaling.paginators.DescribeScalableTargetsIterable responses = client
     *             .describeScalableTargetsPaginator(request);
     *     for (software.amazon.awssdk.services.applicationautoscaling.model.DescribeScalableTargetsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.applicationautoscaling.paginators.DescribeScalableTargetsIterable responses = client.describeScalableTargetsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeScalableTargets(software.amazon.awssdk.services.applicationautoscaling.model.DescribeScalableTargetsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeScalableTargetsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         An exception was thrown for a validation issue. Review the available parameters for the API request.
     * @throws InvalidNextTokenException
     *         The next token supplied was invalid.
     * @throws ConcurrentUpdateException
     *         Concurrent updates caused an exception, for example, if you request an update to an Application Auto
     *         Scaling resource that already has a pending update.
     * @throws InternalServiceException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationAutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationAutoScalingClient.DescribeScalableTargets
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/application-autoscaling-2016-02-06/DescribeScalableTargets"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeScalableTargetsIterable describeScalableTargetsPaginator(
            DescribeScalableTargetsRequest describeScalableTargetsRequest) throws ValidationException, InvalidNextTokenException,
            ConcurrentUpdateException, InternalServiceException, AwsServiceException, SdkClientException,
            ApplicationAutoScalingException {
        return new DescribeScalableTargetsIterable(this, applyPaginatorUserAgent(describeScalableTargetsRequest));
    }

    /**
     * <p>
     * Provides descriptive information about the scaling activities in the specified namespace from the previous six
     * weeks.
     * </p>
     * <p>
     * You can filter the results using <code>ResourceId</code> and <code>ScalableDimension</code>.
     * </p>
     * <p>
     * Scaling activities are triggered by CloudWatch alarms that are associated with scaling policies. To view the
     * scaling policies for a service namespace, see <a>DescribeScalingPolicies</a>. To create a scaling policy or
     * update an existing one, see <a>PutScalingPolicy</a>.
     * </p>
     *
     * @param describeScalingActivitiesRequest
     * @return Result of the DescribeScalingActivities operation returned by the service.
     * @throws ValidationException
     *         An exception was thrown for a validation issue. Review the available parameters for the API request.
     * @throws InvalidNextTokenException
     *         The next token supplied was invalid.
     * @throws ConcurrentUpdateException
     *         Concurrent updates caused an exception, for example, if you request an update to an Application Auto
     *         Scaling resource that already has a pending update.
     * @throws InternalServiceException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationAutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationAutoScalingClient.DescribeScalingActivities
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/application-autoscaling-2016-02-06/DescribeScalingActivities"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeScalingActivitiesResponse describeScalingActivities(
            DescribeScalingActivitiesRequest describeScalingActivitiesRequest) throws ValidationException,
            InvalidNextTokenException, ConcurrentUpdateException, InternalServiceException, AwsServiceException,
            SdkClientException, ApplicationAutoScalingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeScalingActivitiesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeScalingActivitiesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<DescribeScalingActivitiesRequest, DescribeScalingActivitiesResponse>()
                        .withOperationName("DescribeScalingActivities").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(describeScalingActivitiesRequest)
                        .withMarshaller(new DescribeScalingActivitiesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Provides descriptive information about the scaling activities in the specified namespace from the previous six
     * weeks.
     * </p>
     * <p>
     * You can filter the results using <code>ResourceId</code> and <code>ScalableDimension</code>.
     * </p>
     * <p>
     * Scaling activities are triggered by CloudWatch alarms that are associated with scaling policies. To view the
     * scaling policies for a service namespace, see <a>DescribeScalingPolicies</a>. To create a scaling policy or
     * update an existing one, see <a>PutScalingPolicy</a>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeScalingActivities(software.amazon.awssdk.services.applicationautoscaling.model.DescribeScalingActivitiesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.applicationautoscaling.paginators.DescribeScalingActivitiesIterable responses = client.describeScalingActivitiesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.applicationautoscaling.paginators.DescribeScalingActivitiesIterable responses = client
     *             .describeScalingActivitiesPaginator(request);
     *     for (software.amazon.awssdk.services.applicationautoscaling.model.DescribeScalingActivitiesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.applicationautoscaling.paginators.DescribeScalingActivitiesIterable responses = client.describeScalingActivitiesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeScalingActivities(software.amazon.awssdk.services.applicationautoscaling.model.DescribeScalingActivitiesRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeScalingActivitiesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         An exception was thrown for a validation issue. Review the available parameters for the API request.
     * @throws InvalidNextTokenException
     *         The next token supplied was invalid.
     * @throws ConcurrentUpdateException
     *         Concurrent updates caused an exception, for example, if you request an update to an Application Auto
     *         Scaling resource that already has a pending update.
     * @throws InternalServiceException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationAutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationAutoScalingClient.DescribeScalingActivities
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/application-autoscaling-2016-02-06/DescribeScalingActivities"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeScalingActivitiesIterable describeScalingActivitiesPaginator(
            DescribeScalingActivitiesRequest describeScalingActivitiesRequest) throws ValidationException,
            InvalidNextTokenException, ConcurrentUpdateException, InternalServiceException, AwsServiceException,
            SdkClientException, ApplicationAutoScalingException {
        return new DescribeScalingActivitiesIterable(this, applyPaginatorUserAgent(describeScalingActivitiesRequest));
    }

    /**
     * <p>
     * Describes the Application Auto Scaling scaling policies for the specified service namespace.
     * </p>
     * <p>
     * You can filter the results using <code>ResourceId</code>, <code>ScalableDimension</code>, and
     * <code>PolicyNames</code>.
     * </p>
     * <p>
     * To create a scaling policy or update an existing one, see <a>PutScalingPolicy</a>. If you are no longer using a
     * scaling policy, you can delete it using <a>DeleteScalingPolicy</a>.
     * </p>
     *
     * @param describeScalingPoliciesRequest
     * @return Result of the DescribeScalingPolicies operation returned by the service.
     * @throws ValidationException
     *         An exception was thrown for a validation issue. Review the available parameters for the API request.
     * @throws FailedResourceAccessException
     *         Failed access to resources caused an exception. This exception is thrown when Application Auto Scaling is
     *         unable to retrieve the alarms associated with a scaling policy due to a client error, for example, if the
     *         role ARN specified for a scalable target does not have permission to call the CloudWatch <a
     *         href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_DescribeAlarms.html"
     *         >DescribeAlarms</a> on your behalf.
     * @throws InvalidNextTokenException
     *         The next token supplied was invalid.
     * @throws ConcurrentUpdateException
     *         Concurrent updates caused an exception, for example, if you request an update to an Application Auto
     *         Scaling resource that already has a pending update.
     * @throws InternalServiceException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationAutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationAutoScalingClient.DescribeScalingPolicies
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/application-autoscaling-2016-02-06/DescribeScalingPolicies"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeScalingPoliciesResponse describeScalingPolicies(DescribeScalingPoliciesRequest describeScalingPoliciesRequest)
            throws ValidationException, FailedResourceAccessException, InvalidNextTokenException, ConcurrentUpdateException,
            InternalServiceException, AwsServiceException, SdkClientException, ApplicationAutoScalingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeScalingPoliciesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeScalingPoliciesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DescribeScalingPoliciesRequest, DescribeScalingPoliciesResponse>()
                .withOperationName("DescribeScalingPolicies").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeScalingPoliciesRequest)
                .withMarshaller(new DescribeScalingPoliciesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Describes the Application Auto Scaling scaling policies for the specified service namespace.
     * </p>
     * <p>
     * You can filter the results using <code>ResourceId</code>, <code>ScalableDimension</code>, and
     * <code>PolicyNames</code>.
     * </p>
     * <p>
     * To create a scaling policy or update an existing one, see <a>PutScalingPolicy</a>. If you are no longer using a
     * scaling policy, you can delete it using <a>DeleteScalingPolicy</a>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeScalingPolicies(software.amazon.awssdk.services.applicationautoscaling.model.DescribeScalingPoliciesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.applicationautoscaling.paginators.DescribeScalingPoliciesIterable responses = client.describeScalingPoliciesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.applicationautoscaling.paginators.DescribeScalingPoliciesIterable responses = client
     *             .describeScalingPoliciesPaginator(request);
     *     for (software.amazon.awssdk.services.applicationautoscaling.model.DescribeScalingPoliciesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.applicationautoscaling.paginators.DescribeScalingPoliciesIterable responses = client.describeScalingPoliciesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeScalingPolicies(software.amazon.awssdk.services.applicationautoscaling.model.DescribeScalingPoliciesRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeScalingPoliciesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         An exception was thrown for a validation issue. Review the available parameters for the API request.
     * @throws FailedResourceAccessException
     *         Failed access to resources caused an exception. This exception is thrown when Application Auto Scaling is
     *         unable to retrieve the alarms associated with a scaling policy due to a client error, for example, if the
     *         role ARN specified for a scalable target does not have permission to call the CloudWatch <a
     *         href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_DescribeAlarms.html"
     *         >DescribeAlarms</a> on your behalf.
     * @throws InvalidNextTokenException
     *         The next token supplied was invalid.
     * @throws ConcurrentUpdateException
     *         Concurrent updates caused an exception, for example, if you request an update to an Application Auto
     *         Scaling resource that already has a pending update.
     * @throws InternalServiceException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationAutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationAutoScalingClient.DescribeScalingPolicies
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/application-autoscaling-2016-02-06/DescribeScalingPolicies"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeScalingPoliciesIterable describeScalingPoliciesPaginator(
            DescribeScalingPoliciesRequest describeScalingPoliciesRequest) throws ValidationException,
            FailedResourceAccessException, InvalidNextTokenException, ConcurrentUpdateException, InternalServiceException,
            AwsServiceException, SdkClientException, ApplicationAutoScalingException {
        return new DescribeScalingPoliciesIterable(this, applyPaginatorUserAgent(describeScalingPoliciesRequest));
    }

    /**
     * <p>
     * Describes the Application Auto Scaling scheduled actions for the specified service namespace.
     * </p>
     * <p>
     * You can filter the results using the <code>ResourceId</code>, <code>ScalableDimension</code>, and
     * <code>ScheduledActionNames</code> parameters.
     * </p>
     * <p>
     * To create a scheduled action or update an existing one, see <a>PutScheduledAction</a>. If you are no longer using
     * a scheduled action, you can delete it using <a>DeleteScheduledAction</a>.
     * </p>
     *
     * @param describeScheduledActionsRequest
     * @return Result of the DescribeScheduledActions operation returned by the service.
     * @throws ValidationException
     *         An exception was thrown for a validation issue. Review the available parameters for the API request.
     * @throws InvalidNextTokenException
     *         The next token supplied was invalid.
     * @throws ConcurrentUpdateException
     *         Concurrent updates caused an exception, for example, if you request an update to an Application Auto
     *         Scaling resource that already has a pending update.
     * @throws InternalServiceException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationAutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationAutoScalingClient.DescribeScheduledActions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/application-autoscaling-2016-02-06/DescribeScheduledActions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeScheduledActionsResponse describeScheduledActions(
            DescribeScheduledActionsRequest describeScheduledActionsRequest) throws ValidationException,
            InvalidNextTokenException, ConcurrentUpdateException, InternalServiceException, AwsServiceException,
            SdkClientException, ApplicationAutoScalingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeScheduledActionsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeScheduledActionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<DescribeScheduledActionsRequest, DescribeScheduledActionsResponse>()
                        .withOperationName("DescribeScheduledActions").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(describeScheduledActionsRequest)
                        .withMarshaller(new DescribeScheduledActionsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Describes the Application Auto Scaling scheduled actions for the specified service namespace.
     * </p>
     * <p>
     * You can filter the results using the <code>ResourceId</code>, <code>ScalableDimension</code>, and
     * <code>ScheduledActionNames</code> parameters.
     * </p>
     * <p>
     * To create a scheduled action or update an existing one, see <a>PutScheduledAction</a>. If you are no longer using
     * a scheduled action, you can delete it using <a>DeleteScheduledAction</a>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeScheduledActions(software.amazon.awssdk.services.applicationautoscaling.model.DescribeScheduledActionsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.applicationautoscaling.paginators.DescribeScheduledActionsIterable responses = client.describeScheduledActionsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.applicationautoscaling.paginators.DescribeScheduledActionsIterable responses = client
     *             .describeScheduledActionsPaginator(request);
     *     for (software.amazon.awssdk.services.applicationautoscaling.model.DescribeScheduledActionsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.applicationautoscaling.paginators.DescribeScheduledActionsIterable responses = client.describeScheduledActionsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeScheduledActions(software.amazon.awssdk.services.applicationautoscaling.model.DescribeScheduledActionsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeScheduledActionsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         An exception was thrown for a validation issue. Review the available parameters for the API request.
     * @throws InvalidNextTokenException
     *         The next token supplied was invalid.
     * @throws ConcurrentUpdateException
     *         Concurrent updates caused an exception, for example, if you request an update to an Application Auto
     *         Scaling resource that already has a pending update.
     * @throws InternalServiceException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationAutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationAutoScalingClient.DescribeScheduledActions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/application-autoscaling-2016-02-06/DescribeScheduledActions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeScheduledActionsIterable describeScheduledActionsPaginator(
            DescribeScheduledActionsRequest describeScheduledActionsRequest) throws ValidationException,
            InvalidNextTokenException, ConcurrentUpdateException, InternalServiceException, AwsServiceException,
            SdkClientException, ApplicationAutoScalingException {
        return new DescribeScheduledActionsIterable(this, applyPaginatorUserAgent(describeScheduledActionsRequest));
    }

    /**
     * <p>
     * Creates or updates a policy for an Application Auto Scaling scalable target.
     * </p>
     * <p>
     * Each scalable target is identified by a service namespace, resource ID, and scalable dimension. A scaling policy
     * applies to the scalable target identified by those three attributes. You cannot create a scaling policy until you
     * have registered the resource as a scalable target using <a>RegisterScalableTarget</a>.
     * </p>
     * <p>
     * To update a policy, specify its policy name and the parameters that you want to change. Any parameters that you
     * don't specify are not changed by this update request.
     * </p>
     * <p>
     * You can view the scaling policies for a service namespace using <a>DescribeScalingPolicies</a>. If you are no
     * longer using a scaling policy, you can delete it using <a>DeleteScalingPolicy</a>.
     * </p>
     * <p>
     * Multiple scaling policies can be in force at the same time for the same scalable target. You can have one or more
     * target tracking scaling policies, one or more step scaling policies, or both. However, there is a chance that
     * multiple policies could conflict, instructing the scalable target to scale out or in at the same time.
     * Application Auto Scaling gives precedence to the policy that provides the largest capacity for both scale out and
     * scale in. For example, if one policy increases capacity by 3, another policy increases capacity by 200 percent,
     * and the current capacity is 10, Application Auto Scaling uses the policy with the highest calculated capacity
     * (200% of 10 = 20) and scales out to 30.
     * </p>
     * <p>
     * Learn more about how to work with scaling policies in the <a
     * href="https://docs.aws.amazon.com/autoscaling/application/userguide/what-is-application-auto-scaling.html"
     * >Application Auto Scaling User Guide</a>.
     * </p>
     *
     * @param putScalingPolicyRequest
     * @return Result of the PutScalingPolicy operation returned by the service.
     * @throws ValidationException
     *         An exception was thrown for a validation issue. Review the available parameters for the API request.
     * @throws LimitExceededException
     *         A per-account resource limit is exceeded. For more information, see <a href=
     *         "https://docs.aws.amazon.com/ApplicationAutoScaling/latest/userguide/application-auto-scaling-limits.html"
     *         >Application Auto Scaling Limits</a>.
     * @throws ObjectNotFoundException
     *         The specified object could not be found. For any operation that depends on the existence of a scalable
     *         target, this exception is thrown if the scalable target with the specified service namespace, resource
     *         ID, and scalable dimension does not exist. For any operation that deletes or deregisters a resource, this
     *         exception is thrown if the resource cannot be found.
     * @throws ConcurrentUpdateException
     *         Concurrent updates caused an exception, for example, if you request an update to an Application Auto
     *         Scaling resource that already has a pending update.
     * @throws FailedResourceAccessException
     *         Failed access to resources caused an exception. This exception is thrown when Application Auto Scaling is
     *         unable to retrieve the alarms associated with a scaling policy due to a client error, for example, if the
     *         role ARN specified for a scalable target does not have permission to call the CloudWatch <a
     *         href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_DescribeAlarms.html"
     *         >DescribeAlarms</a> on your behalf.
     * @throws InternalServiceException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationAutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationAutoScalingClient.PutScalingPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/application-autoscaling-2016-02-06/PutScalingPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PutScalingPolicyResponse putScalingPolicy(PutScalingPolicyRequest putScalingPolicyRequest) throws ValidationException,
            LimitExceededException, ObjectNotFoundException, ConcurrentUpdateException, FailedResourceAccessException,
            InternalServiceException, AwsServiceException, SdkClientException, ApplicationAutoScalingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutScalingPolicyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PutScalingPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<PutScalingPolicyRequest, PutScalingPolicyResponse>()
                .withOperationName("PutScalingPolicy").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(putScalingPolicyRequest)
                .withMarshaller(new PutScalingPolicyRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Creates or updates a scheduled action for an Application Auto Scaling scalable target.
     * </p>
     * <p>
     * Each scalable target is identified by a service namespace, resource ID, and scalable dimension. A scheduled
     * action applies to the scalable target identified by those three attributes. You cannot create a scheduled action
     * until you have registered the resource as a scalable target using <a>RegisterScalableTarget</a>.
     * </p>
     * <p>
     * To update an action, specify its name and the parameters that you want to change. If you don't specify start and
     * end times, the old values are deleted. Any other parameters that you don't specify are not changed by this update
     * request.
     * </p>
     * <p>
     * You can view the scheduled actions using <a>DescribeScheduledActions</a>. If you are no longer using a scheduled
     * action, you can delete it using <a>DeleteScheduledAction</a>.
     * </p>
     * <p>
     * Learn more about how to work with scheduled actions in the <a
     * href="https://docs.aws.amazon.com/autoscaling/application/userguide/what-is-application-auto-scaling.html"
     * >Application Auto Scaling User Guide</a>.
     * </p>
     *
     * @param putScheduledActionRequest
     * @return Result of the PutScheduledAction operation returned by the service.
     * @throws ValidationException
     *         An exception was thrown for a validation issue. Review the available parameters for the API request.
     * @throws LimitExceededException
     *         A per-account resource limit is exceeded. For more information, see <a href=
     *         "https://docs.aws.amazon.com/ApplicationAutoScaling/latest/userguide/application-auto-scaling-limits.html"
     *         >Application Auto Scaling Limits</a>.
     * @throws ObjectNotFoundException
     *         The specified object could not be found. For any operation that depends on the existence of a scalable
     *         target, this exception is thrown if the scalable target with the specified service namespace, resource
     *         ID, and scalable dimension does not exist. For any operation that deletes or deregisters a resource, this
     *         exception is thrown if the resource cannot be found.
     * @throws ConcurrentUpdateException
     *         Concurrent updates caused an exception, for example, if you request an update to an Application Auto
     *         Scaling resource that already has a pending update.
     * @throws InternalServiceException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationAutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationAutoScalingClient.PutScheduledAction
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/application-autoscaling-2016-02-06/PutScheduledAction"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PutScheduledActionResponse putScheduledAction(PutScheduledActionRequest putScheduledActionRequest)
            throws ValidationException, LimitExceededException, ObjectNotFoundException, ConcurrentUpdateException,
            InternalServiceException, AwsServiceException, SdkClientException, ApplicationAutoScalingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutScheduledActionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutScheduledActionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<PutScheduledActionRequest, PutScheduledActionResponse>()
                .withOperationName("PutScheduledAction").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(putScheduledActionRequest)
                .withMarshaller(new PutScheduledActionRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Registers or updates a scalable target. A scalable target is a resource that Application Auto Scaling can scale
     * out and scale in. Scalable targets are uniquely identified by the combination of resource ID, scalable dimension,
     * and namespace.
     * </p>
     * <p>
     * When you register a new scalable target, you must specify values for minimum and maximum capacity. Application
     * Auto Scaling will not scale capacity to values that are outside of this range.
     * </p>
     * <p>
     * To update a scalable target, specify the parameter that you want to change as well as the following parameters
     * that identify the scalable target: resource ID, scalable dimension, and namespace. Any parameters that you don't
     * specify are not changed by this update request.
     * </p>
     * <p>
     * After you register a scalable target, you do not need to register it again to use other Application Auto Scaling
     * operations. To see which resources have been registered, use <a>DescribeScalableTargets</a>. You can also view
     * the scaling policies for a service namespace by using <a>DescribeScalableTargets</a>.
     * </p>
     * <p>
     * If you no longer need a scalable target, you can deregister it by using <a>DeregisterScalableTarget</a>.
     * </p>
     *
     * @param registerScalableTargetRequest
     * @return Result of the RegisterScalableTarget operation returned by the service.
     * @throws ValidationException
     *         An exception was thrown for a validation issue. Review the available parameters for the API request.
     * @throws LimitExceededException
     *         A per-account resource limit is exceeded. For more information, see <a href=
     *         "https://docs.aws.amazon.com/ApplicationAutoScaling/latest/userguide/application-auto-scaling-limits.html"
     *         >Application Auto Scaling Limits</a>.
     * @throws ConcurrentUpdateException
     *         Concurrent updates caused an exception, for example, if you request an update to an Application Auto
     *         Scaling resource that already has a pending update.
     * @throws InternalServiceException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationAutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationAutoScalingClient.RegisterScalableTarget
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/application-autoscaling-2016-02-06/RegisterScalableTarget"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public RegisterScalableTargetResponse registerScalableTarget(RegisterScalableTargetRequest registerScalableTargetRequest)
            throws ValidationException, LimitExceededException, ConcurrentUpdateException, InternalServiceException,
            AwsServiceException, SdkClientException, ApplicationAutoScalingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RegisterScalableTargetResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, RegisterScalableTargetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<RegisterScalableTargetRequest, RegisterScalableTargetResponse>()
                .withOperationName("RegisterScalableTarget").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(registerScalableTargetRequest)
                .withMarshaller(new RegisterScalableTargetRequestMarshaller(protocolFactory)));
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ApplicationAutoScalingException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServiceException")
                                .exceptionBuilderSupplier(InternalServiceException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidNextTokenException")
                                .exceptionBuilderSupplier(InvalidNextTokenException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ObjectNotFoundException")
                                .exceptionBuilderSupplier(ObjectNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("FailedResourceAccessException")
                                .exceptionBuilderSupplier(FailedResourceAccessException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConcurrentUpdateException")
                                .exceptionBuilderSupplier(ConcurrentUpdateException::builder).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends ApplicationAutoScalingRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
