/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.applicationautoscaling;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.applicationautoscaling.model.ApplicationAutoScalingException;
import software.amazon.awssdk.services.applicationautoscaling.model.ApplicationAutoScalingRequest;
import software.amazon.awssdk.services.applicationautoscaling.model.ConcurrentUpdateException;
import software.amazon.awssdk.services.applicationautoscaling.model.DeleteScalingPolicyRequest;
import software.amazon.awssdk.services.applicationautoscaling.model.DeleteScalingPolicyResponse;
import software.amazon.awssdk.services.applicationautoscaling.model.DeleteScheduledActionRequest;
import software.amazon.awssdk.services.applicationautoscaling.model.DeleteScheduledActionResponse;
import software.amazon.awssdk.services.applicationautoscaling.model.DeregisterScalableTargetRequest;
import software.amazon.awssdk.services.applicationautoscaling.model.DeregisterScalableTargetResponse;
import software.amazon.awssdk.services.applicationautoscaling.model.DescribeScalableTargetsRequest;
import software.amazon.awssdk.services.applicationautoscaling.model.DescribeScalableTargetsResponse;
import software.amazon.awssdk.services.applicationautoscaling.model.DescribeScalingActivitiesRequest;
import software.amazon.awssdk.services.applicationautoscaling.model.DescribeScalingActivitiesResponse;
import software.amazon.awssdk.services.applicationautoscaling.model.DescribeScalingPoliciesRequest;
import software.amazon.awssdk.services.applicationautoscaling.model.DescribeScalingPoliciesResponse;
import software.amazon.awssdk.services.applicationautoscaling.model.DescribeScheduledActionsRequest;
import software.amazon.awssdk.services.applicationautoscaling.model.DescribeScheduledActionsResponse;
import software.amazon.awssdk.services.applicationautoscaling.model.FailedResourceAccessException;
import software.amazon.awssdk.services.applicationautoscaling.model.InternalServiceException;
import software.amazon.awssdk.services.applicationautoscaling.model.InvalidNextTokenException;
import software.amazon.awssdk.services.applicationautoscaling.model.LimitExceededException;
import software.amazon.awssdk.services.applicationautoscaling.model.ObjectNotFoundException;
import software.amazon.awssdk.services.applicationautoscaling.model.PutScalingPolicyRequest;
import software.amazon.awssdk.services.applicationautoscaling.model.PutScalingPolicyResponse;
import software.amazon.awssdk.services.applicationautoscaling.model.PutScheduledActionRequest;
import software.amazon.awssdk.services.applicationautoscaling.model.PutScheduledActionResponse;
import software.amazon.awssdk.services.applicationautoscaling.model.RegisterScalableTargetRequest;
import software.amazon.awssdk.services.applicationautoscaling.model.RegisterScalableTargetResponse;
import software.amazon.awssdk.services.applicationautoscaling.model.ValidationException;
import software.amazon.awssdk.services.applicationautoscaling.paginators.DescribeScalableTargetsPublisher;
import software.amazon.awssdk.services.applicationautoscaling.paginators.DescribeScalingActivitiesPublisher;
import software.amazon.awssdk.services.applicationautoscaling.paginators.DescribeScalingPoliciesPublisher;
import software.amazon.awssdk.services.applicationautoscaling.paginators.DescribeScheduledActionsPublisher;
import software.amazon.awssdk.services.applicationautoscaling.transform.DeleteScalingPolicyRequestMarshaller;
import software.amazon.awssdk.services.applicationautoscaling.transform.DeleteScheduledActionRequestMarshaller;
import software.amazon.awssdk.services.applicationautoscaling.transform.DeregisterScalableTargetRequestMarshaller;
import software.amazon.awssdk.services.applicationautoscaling.transform.DescribeScalableTargetsRequestMarshaller;
import software.amazon.awssdk.services.applicationautoscaling.transform.DescribeScalingActivitiesRequestMarshaller;
import software.amazon.awssdk.services.applicationautoscaling.transform.DescribeScalingPoliciesRequestMarshaller;
import software.amazon.awssdk.services.applicationautoscaling.transform.DescribeScheduledActionsRequestMarshaller;
import software.amazon.awssdk.services.applicationautoscaling.transform.PutScalingPolicyRequestMarshaller;
import software.amazon.awssdk.services.applicationautoscaling.transform.PutScheduledActionRequestMarshaller;
import software.amazon.awssdk.services.applicationautoscaling.transform.RegisterScalableTargetRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link ApplicationAutoScalingAsyncClient}.
 *
 * @see ApplicationAutoScalingAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultApplicationAutoScalingAsyncClient implements ApplicationAutoScalingAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultApplicationAutoScalingAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultApplicationAutoScalingAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Deletes the specified scaling policy for an Application Auto Scaling scalable target.
     * </p>
     * <p>
     * Deleting a step scaling policy deletes the underlying alarm action, but does not delete the CloudWatch alarm
     * associated with the scaling policy, even if it no longer has an associated action.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/autoscaling/application/userguide/application-auto-scaling-step-scaling-policies.html#delete-step-scaling-policy"
     * >Delete a Step Scaling Policy</a> and <a href=
     * "https://docs.aws.amazon.com/autoscaling/application/userguide/application-auto-scaling-target-tracking.html#delete-target-tracking-policy"
     * >Delete a Target Tracking Scaling Policy</a> in the <i>Application Auto Scaling User Guide</i>.
     * </p>
     * <p>
     * To create a scaling policy or update an existing one, see <a>PutScalingPolicy</a>.
     * </p>
     *
     * @param deleteScalingPolicyRequest
     * @return A Java Future containing the result of the DeleteScalingPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException An exception was thrown for a validation issue. Review the available parameters
     *         for the API request.</li>
     *         <li>ObjectNotFoundException The specified object could not be found. For any operation that depends on
     *         the existence of a scalable target, this exception is thrown if the scalable target with the specified
     *         service namespace, resource ID, and scalable dimension does not exist. For any operation that deletes or
     *         deregisters a resource, this exception is thrown if the resource cannot be found.</li>
     *         <li>ConcurrentUpdateException Concurrent updates caused an exception, for example, if you request an
     *         update to an Application Auto Scaling resource that already has a pending update.</li>
     *         <li>InternalServiceException The service encountered an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationAutoScalingException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationAutoScalingAsyncClient.DeleteScalingPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/application-autoscaling-2016-02-06/DeleteScalingPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteScalingPolicyResponse> deleteScalingPolicy(
            DeleteScalingPolicyRequest deleteScalingPolicyRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteScalingPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteScalingPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteScalingPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteScalingPolicyRequest, DeleteScalingPolicyResponse>()
                            .withOperationName("DeleteScalingPolicy")
                            .withMarshaller(new DeleteScalingPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteScalingPolicyRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified scheduled action for an Application Auto Scaling scalable target.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/autoscaling/application/userguide/application-auto-scaling-scheduled-scaling.html#delete-scheduled-action"
     * >Delete a Scheduled Action</a> in the <i>Application Auto Scaling User Guide</i>.
     * </p>
     *
     * @param deleteScheduledActionRequest
     * @return A Java Future containing the result of the DeleteScheduledAction operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException An exception was thrown for a validation issue. Review the available parameters
     *         for the API request.</li>
     *         <li>ObjectNotFoundException The specified object could not be found. For any operation that depends on
     *         the existence of a scalable target, this exception is thrown if the scalable target with the specified
     *         service namespace, resource ID, and scalable dimension does not exist. For any operation that deletes or
     *         deregisters a resource, this exception is thrown if the resource cannot be found.</li>
     *         <li>ConcurrentUpdateException Concurrent updates caused an exception, for example, if you request an
     *         update to an Application Auto Scaling resource that already has a pending update.</li>
     *         <li>InternalServiceException The service encountered an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationAutoScalingException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationAutoScalingAsyncClient.DeleteScheduledAction
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/application-autoscaling-2016-02-06/DeleteScheduledAction"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteScheduledActionResponse> deleteScheduledAction(
            DeleteScheduledActionRequest deleteScheduledActionRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteScheduledActionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteScheduledActionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteScheduledActionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteScheduledActionRequest, DeleteScheduledActionResponse>()
                            .withOperationName("DeleteScheduledAction")
                            .withMarshaller(new DeleteScheduledActionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteScheduledActionRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deregisters an Application Auto Scaling scalable target.
     * </p>
     * <p>
     * Deregistering a scalable target deletes the scaling policies that are associated with it.
     * </p>
     * <p>
     * To create a scalable target or update an existing one, see <a>RegisterScalableTarget</a>.
     * </p>
     *
     * @param deregisterScalableTargetRequest
     * @return A Java Future containing the result of the DeregisterScalableTarget operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException An exception was thrown for a validation issue. Review the available parameters
     *         for the API request.</li>
     *         <li>ObjectNotFoundException The specified object could not be found. For any operation that depends on
     *         the existence of a scalable target, this exception is thrown if the scalable target with the specified
     *         service namespace, resource ID, and scalable dimension does not exist. For any operation that deletes or
     *         deregisters a resource, this exception is thrown if the resource cannot be found.</li>
     *         <li>ConcurrentUpdateException Concurrent updates caused an exception, for example, if you request an
     *         update to an Application Auto Scaling resource that already has a pending update.</li>
     *         <li>InternalServiceException The service encountered an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationAutoScalingException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationAutoScalingAsyncClient.DeregisterScalableTarget
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/application-autoscaling-2016-02-06/DeregisterScalableTarget"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeregisterScalableTargetResponse> deregisterScalableTarget(
            DeregisterScalableTargetRequest deregisterScalableTargetRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeregisterScalableTargetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeregisterScalableTargetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeregisterScalableTargetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeregisterScalableTargetRequest, DeregisterScalableTargetResponse>()
                            .withOperationName("DeregisterScalableTarget")
                            .withMarshaller(new DeregisterScalableTargetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deregisterScalableTargetRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about the scalable targets in the specified namespace.
     * </p>
     * <p>
     * You can filter the results using <code>ResourceIds</code> and <code>ScalableDimension</code>.
     * </p>
     * <p>
     * To create a scalable target or update an existing one, see <a>RegisterScalableTarget</a>. If you are no longer
     * using a scalable target, you can deregister it using <a>DeregisterScalableTarget</a>.
     * </p>
     *
     * @param describeScalableTargetsRequest
     * @return A Java Future containing the result of the DescribeScalableTargets operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException An exception was thrown for a validation issue. Review the available parameters
     *         for the API request.</li>
     *         <li>InvalidNextTokenException The next token supplied was invalid.</li>
     *         <li>ConcurrentUpdateException Concurrent updates caused an exception, for example, if you request an
     *         update to an Application Auto Scaling resource that already has a pending update.</li>
     *         <li>InternalServiceException The service encountered an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationAutoScalingException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationAutoScalingAsyncClient.DescribeScalableTargets
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/application-autoscaling-2016-02-06/DescribeScalableTargets"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeScalableTargetsResponse> describeScalableTargets(
            DescribeScalableTargetsRequest describeScalableTargetsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeScalableTargetsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeScalableTargetsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeScalableTargetsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeScalableTargetsRequest, DescribeScalableTargetsResponse>()
                            .withOperationName("DescribeScalableTargets")
                            .withMarshaller(new DescribeScalableTargetsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeScalableTargetsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about the scalable targets in the specified namespace.
     * </p>
     * <p>
     * You can filter the results using <code>ResourceIds</code> and <code>ScalableDimension</code>.
     * </p>
     * <p>
     * To create a scalable target or update an existing one, see <a>RegisterScalableTarget</a>. If you are no longer
     * using a scalable target, you can deregister it using <a>DeregisterScalableTarget</a>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeScalableTargets(software.amazon.awssdk.services.applicationautoscaling.model.DescribeScalableTargetsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.applicationautoscaling.paginators.DescribeScalableTargetsPublisher publisher = client.describeScalableTargetsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.applicationautoscaling.paginators.DescribeScalableTargetsPublisher publisher = client.describeScalableTargetsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.applicationautoscaling.model.DescribeScalableTargetsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.applicationautoscaling.model.DescribeScalableTargetsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeScalableTargets(software.amazon.awssdk.services.applicationautoscaling.model.DescribeScalableTargetsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeScalableTargetsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException An exception was thrown for a validation issue. Review the available parameters
     *         for the API request.</li>
     *         <li>InvalidNextTokenException The next token supplied was invalid.</li>
     *         <li>ConcurrentUpdateException Concurrent updates caused an exception, for example, if you request an
     *         update to an Application Auto Scaling resource that already has a pending update.</li>
     *         <li>InternalServiceException The service encountered an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationAutoScalingException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationAutoScalingAsyncClient.DescribeScalableTargets
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/application-autoscaling-2016-02-06/DescribeScalableTargets"
     *      target="_top">AWS API Documentation</a>
     */
    public DescribeScalableTargetsPublisher describeScalableTargetsPaginator(
            DescribeScalableTargetsRequest describeScalableTargetsRequest) {
        return new DescribeScalableTargetsPublisher(this, applyPaginatorUserAgent(describeScalableTargetsRequest));
    }

    /**
     * <p>
     * Provides descriptive information about the scaling activities in the specified namespace from the previous six
     * weeks.
     * </p>
     * <p>
     * You can filter the results using <code>ResourceId</code> and <code>ScalableDimension</code>.
     * </p>
     * <p>
     * Scaling activities are triggered by CloudWatch alarms that are associated with scaling policies. To view the
     * scaling policies for a service namespace, see <a>DescribeScalingPolicies</a>. To create a scaling policy or
     * update an existing one, see <a>PutScalingPolicy</a>.
     * </p>
     *
     * @param describeScalingActivitiesRequest
     * @return A Java Future containing the result of the DescribeScalingActivities operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException An exception was thrown for a validation issue. Review the available parameters
     *         for the API request.</li>
     *         <li>InvalidNextTokenException The next token supplied was invalid.</li>
     *         <li>ConcurrentUpdateException Concurrent updates caused an exception, for example, if you request an
     *         update to an Application Auto Scaling resource that already has a pending update.</li>
     *         <li>InternalServiceException The service encountered an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationAutoScalingException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationAutoScalingAsyncClient.DescribeScalingActivities
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/application-autoscaling-2016-02-06/DescribeScalingActivities"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeScalingActivitiesResponse> describeScalingActivities(
            DescribeScalingActivitiesRequest describeScalingActivitiesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeScalingActivitiesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeScalingActivitiesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeScalingActivitiesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeScalingActivitiesRequest, DescribeScalingActivitiesResponse>()
                            .withOperationName("DescribeScalingActivities")
                            .withMarshaller(new DescribeScalingActivitiesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeScalingActivitiesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides descriptive information about the scaling activities in the specified namespace from the previous six
     * weeks.
     * </p>
     * <p>
     * You can filter the results using <code>ResourceId</code> and <code>ScalableDimension</code>.
     * </p>
     * <p>
     * Scaling activities are triggered by CloudWatch alarms that are associated with scaling policies. To view the
     * scaling policies for a service namespace, see <a>DescribeScalingPolicies</a>. To create a scaling policy or
     * update an existing one, see <a>PutScalingPolicy</a>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeScalingActivities(software.amazon.awssdk.services.applicationautoscaling.model.DescribeScalingActivitiesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.applicationautoscaling.paginators.DescribeScalingActivitiesPublisher publisher = client.describeScalingActivitiesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.applicationautoscaling.paginators.DescribeScalingActivitiesPublisher publisher = client.describeScalingActivitiesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.applicationautoscaling.model.DescribeScalingActivitiesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.applicationautoscaling.model.DescribeScalingActivitiesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeScalingActivities(software.amazon.awssdk.services.applicationautoscaling.model.DescribeScalingActivitiesRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeScalingActivitiesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException An exception was thrown for a validation issue. Review the available parameters
     *         for the API request.</li>
     *         <li>InvalidNextTokenException The next token supplied was invalid.</li>
     *         <li>ConcurrentUpdateException Concurrent updates caused an exception, for example, if you request an
     *         update to an Application Auto Scaling resource that already has a pending update.</li>
     *         <li>InternalServiceException The service encountered an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationAutoScalingException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationAutoScalingAsyncClient.DescribeScalingActivities
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/application-autoscaling-2016-02-06/DescribeScalingActivities"
     *      target="_top">AWS API Documentation</a>
     */
    public DescribeScalingActivitiesPublisher describeScalingActivitiesPaginator(
            DescribeScalingActivitiesRequest describeScalingActivitiesRequest) {
        return new DescribeScalingActivitiesPublisher(this, applyPaginatorUserAgent(describeScalingActivitiesRequest));
    }

    /**
     * <p>
     * Describes the Application Auto Scaling scaling policies for the specified service namespace.
     * </p>
     * <p>
     * You can filter the results using <code>ResourceId</code>, <code>ScalableDimension</code>, and
     * <code>PolicyNames</code>.
     * </p>
     * <p>
     * To create a scaling policy or update an existing one, see <a>PutScalingPolicy</a>. If you are no longer using a
     * scaling policy, you can delete it using <a>DeleteScalingPolicy</a>.
     * </p>
     *
     * @param describeScalingPoliciesRequest
     * @return A Java Future containing the result of the DescribeScalingPolicies operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException An exception was thrown for a validation issue. Review the available parameters
     *         for the API request.</li>
     *         <li>FailedResourceAccessException Failed access to resources caused an exception. This exception is
     *         thrown when Application Auto Scaling is unable to retrieve the alarms associated with a scaling policy
     *         due to a client error, for example, if the role ARN specified for a scalable target does not have
     *         permission to call the CloudWatch <a
     *         href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_DescribeAlarms.html"
     *         >DescribeAlarms</a> on your behalf.</li>
     *         <li>InvalidNextTokenException The next token supplied was invalid.</li>
     *         <li>ConcurrentUpdateException Concurrent updates caused an exception, for example, if you request an
     *         update to an Application Auto Scaling resource that already has a pending update.</li>
     *         <li>InternalServiceException The service encountered an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationAutoScalingException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationAutoScalingAsyncClient.DescribeScalingPolicies
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/application-autoscaling-2016-02-06/DescribeScalingPolicies"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeScalingPoliciesResponse> describeScalingPolicies(
            DescribeScalingPoliciesRequest describeScalingPoliciesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeScalingPoliciesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeScalingPoliciesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeScalingPoliciesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeScalingPoliciesRequest, DescribeScalingPoliciesResponse>()
                            .withOperationName("DescribeScalingPolicies")
                            .withMarshaller(new DescribeScalingPoliciesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeScalingPoliciesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the Application Auto Scaling scaling policies for the specified service namespace.
     * </p>
     * <p>
     * You can filter the results using <code>ResourceId</code>, <code>ScalableDimension</code>, and
     * <code>PolicyNames</code>.
     * </p>
     * <p>
     * To create a scaling policy or update an existing one, see <a>PutScalingPolicy</a>. If you are no longer using a
     * scaling policy, you can delete it using <a>DeleteScalingPolicy</a>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeScalingPolicies(software.amazon.awssdk.services.applicationautoscaling.model.DescribeScalingPoliciesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.applicationautoscaling.paginators.DescribeScalingPoliciesPublisher publisher = client.describeScalingPoliciesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.applicationautoscaling.paginators.DescribeScalingPoliciesPublisher publisher = client.describeScalingPoliciesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.applicationautoscaling.model.DescribeScalingPoliciesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.applicationautoscaling.model.DescribeScalingPoliciesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeScalingPolicies(software.amazon.awssdk.services.applicationautoscaling.model.DescribeScalingPoliciesRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeScalingPoliciesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException An exception was thrown for a validation issue. Review the available parameters
     *         for the API request.</li>
     *         <li>FailedResourceAccessException Failed access to resources caused an exception. This exception is
     *         thrown when Application Auto Scaling is unable to retrieve the alarms associated with a scaling policy
     *         due to a client error, for example, if the role ARN specified for a scalable target does not have
     *         permission to call the CloudWatch <a
     *         href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_DescribeAlarms.html"
     *         >DescribeAlarms</a> on your behalf.</li>
     *         <li>InvalidNextTokenException The next token supplied was invalid.</li>
     *         <li>ConcurrentUpdateException Concurrent updates caused an exception, for example, if you request an
     *         update to an Application Auto Scaling resource that already has a pending update.</li>
     *         <li>InternalServiceException The service encountered an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationAutoScalingException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationAutoScalingAsyncClient.DescribeScalingPolicies
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/application-autoscaling-2016-02-06/DescribeScalingPolicies"
     *      target="_top">AWS API Documentation</a>
     */
    public DescribeScalingPoliciesPublisher describeScalingPoliciesPaginator(
            DescribeScalingPoliciesRequest describeScalingPoliciesRequest) {
        return new DescribeScalingPoliciesPublisher(this, applyPaginatorUserAgent(describeScalingPoliciesRequest));
    }

    /**
     * <p>
     * Describes the Application Auto Scaling scheduled actions for the specified service namespace.
     * </p>
     * <p>
     * You can filter the results using the <code>ResourceId</code>, <code>ScalableDimension</code>, and
     * <code>ScheduledActionNames</code> parameters.
     * </p>
     * <p>
     * To create a scheduled action or update an existing one, see <a>PutScheduledAction</a>. If you are no longer using
     * a scheduled action, you can delete it using <a>DeleteScheduledAction</a>.
     * </p>
     *
     * @param describeScheduledActionsRequest
     * @return A Java Future containing the result of the DescribeScheduledActions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException An exception was thrown for a validation issue. Review the available parameters
     *         for the API request.</li>
     *         <li>InvalidNextTokenException The next token supplied was invalid.</li>
     *         <li>ConcurrentUpdateException Concurrent updates caused an exception, for example, if you request an
     *         update to an Application Auto Scaling resource that already has a pending update.</li>
     *         <li>InternalServiceException The service encountered an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationAutoScalingException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationAutoScalingAsyncClient.DescribeScheduledActions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/application-autoscaling-2016-02-06/DescribeScheduledActions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeScheduledActionsResponse> describeScheduledActions(
            DescribeScheduledActionsRequest describeScheduledActionsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeScheduledActionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeScheduledActionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeScheduledActionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeScheduledActionsRequest, DescribeScheduledActionsResponse>()
                            .withOperationName("DescribeScheduledActions")
                            .withMarshaller(new DescribeScheduledActionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeScheduledActionsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the Application Auto Scaling scheduled actions for the specified service namespace.
     * </p>
     * <p>
     * You can filter the results using the <code>ResourceId</code>, <code>ScalableDimension</code>, and
     * <code>ScheduledActionNames</code> parameters.
     * </p>
     * <p>
     * To create a scheduled action or update an existing one, see <a>PutScheduledAction</a>. If you are no longer using
     * a scheduled action, you can delete it using <a>DeleteScheduledAction</a>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeScheduledActions(software.amazon.awssdk.services.applicationautoscaling.model.DescribeScheduledActionsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.applicationautoscaling.paginators.DescribeScheduledActionsPublisher publisher = client.describeScheduledActionsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.applicationautoscaling.paginators.DescribeScheduledActionsPublisher publisher = client.describeScheduledActionsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.applicationautoscaling.model.DescribeScheduledActionsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.applicationautoscaling.model.DescribeScheduledActionsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeScheduledActions(software.amazon.awssdk.services.applicationautoscaling.model.DescribeScheduledActionsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeScheduledActionsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException An exception was thrown for a validation issue. Review the available parameters
     *         for the API request.</li>
     *         <li>InvalidNextTokenException The next token supplied was invalid.</li>
     *         <li>ConcurrentUpdateException Concurrent updates caused an exception, for example, if you request an
     *         update to an Application Auto Scaling resource that already has a pending update.</li>
     *         <li>InternalServiceException The service encountered an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationAutoScalingException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationAutoScalingAsyncClient.DescribeScheduledActions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/application-autoscaling-2016-02-06/DescribeScheduledActions"
     *      target="_top">AWS API Documentation</a>
     */
    public DescribeScheduledActionsPublisher describeScheduledActionsPaginator(
            DescribeScheduledActionsRequest describeScheduledActionsRequest) {
        return new DescribeScheduledActionsPublisher(this, applyPaginatorUserAgent(describeScheduledActionsRequest));
    }

    /**
     * <p>
     * Creates or updates a policy for an Application Auto Scaling scalable target.
     * </p>
     * <p>
     * Each scalable target is identified by a service namespace, resource ID, and scalable dimension. A scaling policy
     * applies to the scalable target identified by those three attributes. You cannot create a scaling policy until you
     * have registered the resource as a scalable target using <a>RegisterScalableTarget</a>.
     * </p>
     * <p>
     * To update a policy, specify its policy name and the parameters that you want to change. Any parameters that you
     * don't specify are not changed by this update request.
     * </p>
     * <p>
     * You can view the scaling policies for a service namespace using <a>DescribeScalingPolicies</a>. If you are no
     * longer using a scaling policy, you can delete it using <a>DeleteScalingPolicy</a>.
     * </p>
     * <p>
     * Multiple scaling policies can be in force at the same time for the same scalable target. You can have one or more
     * target tracking scaling policies, one or more step scaling policies, or both. However, there is a chance that
     * multiple policies could conflict, instructing the scalable target to scale out or in at the same time.
     * Application Auto Scaling gives precedence to the policy that provides the largest capacity for both scale out and
     * scale in. For example, if one policy increases capacity by 3, another policy increases capacity by 200 percent,
     * and the current capacity is 10, Application Auto Scaling uses the policy with the highest calculated capacity
     * (200% of 10 = 20) and scales out to 30.
     * </p>
     * <p>
     * Learn more about how to work with scaling policies in the <a
     * href="https://docs.aws.amazon.com/autoscaling/application/userguide/what-is-application-auto-scaling.html"
     * >Application Auto Scaling User Guide</a>.
     * </p>
     *
     * @param putScalingPolicyRequest
     * @return A Java Future containing the result of the PutScalingPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException An exception was thrown for a validation issue. Review the available parameters
     *         for the API request.</li>
     *         <li>LimitExceededException A per-account resource limit is exceeded. For more information, see <a
     *         href="https://docs.aws.amazon.com/ApplicationAutoScaling/latest/userguide/application-auto-scaling-limits.html"
     *         >Application Auto Scaling Limits</a>.</li>
     *         <li>ObjectNotFoundException The specified object could not be found. For any operation that depends on
     *         the existence of a scalable target, this exception is thrown if the scalable target with the specified
     *         service namespace, resource ID, and scalable dimension does not exist. For any operation that deletes or
     *         deregisters a resource, this exception is thrown if the resource cannot be found.</li>
     *         <li>ConcurrentUpdateException Concurrent updates caused an exception, for example, if you request an
     *         update to an Application Auto Scaling resource that already has a pending update.</li>
     *         <li>FailedResourceAccessException Failed access to resources caused an exception. This exception is
     *         thrown when Application Auto Scaling is unable to retrieve the alarms associated with a scaling policy
     *         due to a client error, for example, if the role ARN specified for a scalable target does not have
     *         permission to call the CloudWatch <a
     *         href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_DescribeAlarms.html"
     *         >DescribeAlarms</a> on your behalf.</li>
     *         <li>InternalServiceException The service encountered an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationAutoScalingException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationAutoScalingAsyncClient.PutScalingPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/application-autoscaling-2016-02-06/PutScalingPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutScalingPolicyResponse> putScalingPolicy(PutScalingPolicyRequest putScalingPolicyRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutScalingPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutScalingPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutScalingPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutScalingPolicyRequest, PutScalingPolicyResponse>()
                            .withOperationName("PutScalingPolicy")
                            .withMarshaller(new PutScalingPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(putScalingPolicyRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates or updates a scheduled action for an Application Auto Scaling scalable target.
     * </p>
     * <p>
     * Each scalable target is identified by a service namespace, resource ID, and scalable dimension. A scheduled
     * action applies to the scalable target identified by those three attributes. You cannot create a scheduled action
     * until you have registered the resource as a scalable target using <a>RegisterScalableTarget</a>.
     * </p>
     * <p>
     * To update an action, specify its name and the parameters that you want to change. If you don't specify start and
     * end times, the old values are deleted. Any other parameters that you don't specify are not changed by this update
     * request.
     * </p>
     * <p>
     * You can view the scheduled actions using <a>DescribeScheduledActions</a>. If you are no longer using a scheduled
     * action, you can delete it using <a>DeleteScheduledAction</a>.
     * </p>
     * <p>
     * Learn more about how to work with scheduled actions in the <a
     * href="https://docs.aws.amazon.com/autoscaling/application/userguide/what-is-application-auto-scaling.html"
     * >Application Auto Scaling User Guide</a>.
     * </p>
     *
     * @param putScheduledActionRequest
     * @return A Java Future containing the result of the PutScheduledAction operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException An exception was thrown for a validation issue. Review the available parameters
     *         for the API request.</li>
     *         <li>LimitExceededException A per-account resource limit is exceeded. For more information, see <a
     *         href="https://docs.aws.amazon.com/ApplicationAutoScaling/latest/userguide/application-auto-scaling-limits.html"
     *         >Application Auto Scaling Limits</a>.</li>
     *         <li>ObjectNotFoundException The specified object could not be found. For any operation that depends on
     *         the existence of a scalable target, this exception is thrown if the scalable target with the specified
     *         service namespace, resource ID, and scalable dimension does not exist. For any operation that deletes or
     *         deregisters a resource, this exception is thrown if the resource cannot be found.</li>
     *         <li>ConcurrentUpdateException Concurrent updates caused an exception, for example, if you request an
     *         update to an Application Auto Scaling resource that already has a pending update.</li>
     *         <li>InternalServiceException The service encountered an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationAutoScalingException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationAutoScalingAsyncClient.PutScheduledAction
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/application-autoscaling-2016-02-06/PutScheduledAction"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutScheduledActionResponse> putScheduledAction(PutScheduledActionRequest putScheduledActionRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutScheduledActionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutScheduledActionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutScheduledActionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutScheduledActionRequest, PutScheduledActionResponse>()
                            .withOperationName("PutScheduledAction")
                            .withMarshaller(new PutScheduledActionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(putScheduledActionRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Registers or updates a scalable target. A scalable target is a resource that Application Auto Scaling can scale
     * out and scale in. Scalable targets are uniquely identified by the combination of resource ID, scalable dimension,
     * and namespace.
     * </p>
     * <p>
     * When you register a new scalable target, you must specify values for minimum and maximum capacity. Application
     * Auto Scaling will not scale capacity to values that are outside of this range.
     * </p>
     * <p>
     * To update a scalable target, specify the parameter that you want to change as well as the following parameters
     * that identify the scalable target: resource ID, scalable dimension, and namespace. Any parameters that you don't
     * specify are not changed by this update request.
     * </p>
     * <p>
     * After you register a scalable target, you do not need to register it again to use other Application Auto Scaling
     * operations. To see which resources have been registered, use <a>DescribeScalableTargets</a>. You can also view
     * the scaling policies for a service namespace by using <a>DescribeScalableTargets</a>.
     * </p>
     * <p>
     * If you no longer need a scalable target, you can deregister it by using <a>DeregisterScalableTarget</a>.
     * </p>
     *
     * @param registerScalableTargetRequest
     * @return A Java Future containing the result of the RegisterScalableTarget operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException An exception was thrown for a validation issue. Review the available parameters
     *         for the API request.</li>
     *         <li>LimitExceededException A per-account resource limit is exceeded. For more information, see <a
     *         href="https://docs.aws.amazon.com/ApplicationAutoScaling/latest/userguide/application-auto-scaling-limits.html"
     *         >Application Auto Scaling Limits</a>.</li>
     *         <li>ConcurrentUpdateException Concurrent updates caused an exception, for example, if you request an
     *         update to an Application Auto Scaling resource that already has a pending update.</li>
     *         <li>InternalServiceException The service encountered an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationAutoScalingException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationAutoScalingAsyncClient.RegisterScalableTarget
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/application-autoscaling-2016-02-06/RegisterScalableTarget"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RegisterScalableTargetResponse> registerScalableTarget(
            RegisterScalableTargetRequest registerScalableTargetRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RegisterScalableTargetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RegisterScalableTargetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RegisterScalableTargetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RegisterScalableTargetRequest, RegisterScalableTargetResponse>()
                            .withOperationName("RegisterScalableTarget")
                            .withMarshaller(new RegisterScalableTargetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(registerScalableTargetRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ApplicationAutoScalingException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServiceException")
                                .exceptionBuilderSupplier(InternalServiceException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidNextTokenException")
                                .exceptionBuilderSupplier(InvalidNextTokenException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ObjectNotFoundException")
                                .exceptionBuilderSupplier(ObjectNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("FailedResourceAccessException")
                                .exceptionBuilderSupplier(FailedResourceAccessException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConcurrentUpdateException")
                                .exceptionBuilderSupplier(ConcurrentUpdateException::builder).build());
    }

    private <T extends ApplicationAutoScalingRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
