/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.appintegrations;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.appintegrations.model.AccessDeniedException;
import software.amazon.awssdk.services.appintegrations.model.AppIntegrationsException;
import software.amazon.awssdk.services.appintegrations.model.CreateEventIntegrationRequest;
import software.amazon.awssdk.services.appintegrations.model.CreateEventIntegrationResponse;
import software.amazon.awssdk.services.appintegrations.model.DeleteEventIntegrationRequest;
import software.amazon.awssdk.services.appintegrations.model.DeleteEventIntegrationResponse;
import software.amazon.awssdk.services.appintegrations.model.DuplicateResourceException;
import software.amazon.awssdk.services.appintegrations.model.GetEventIntegrationRequest;
import software.amazon.awssdk.services.appintegrations.model.GetEventIntegrationResponse;
import software.amazon.awssdk.services.appintegrations.model.InternalServiceErrorException;
import software.amazon.awssdk.services.appintegrations.model.InvalidRequestException;
import software.amazon.awssdk.services.appintegrations.model.ListEventIntegrationAssociationsRequest;
import software.amazon.awssdk.services.appintegrations.model.ListEventIntegrationAssociationsResponse;
import software.amazon.awssdk.services.appintegrations.model.ListEventIntegrationsRequest;
import software.amazon.awssdk.services.appintegrations.model.ListEventIntegrationsResponse;
import software.amazon.awssdk.services.appintegrations.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.appintegrations.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.appintegrations.model.ResourceNotFoundException;
import software.amazon.awssdk.services.appintegrations.model.ResourceQuotaExceededException;
import software.amazon.awssdk.services.appintegrations.model.TagResourceRequest;
import software.amazon.awssdk.services.appintegrations.model.TagResourceResponse;
import software.amazon.awssdk.services.appintegrations.model.ThrottlingException;
import software.amazon.awssdk.services.appintegrations.model.UntagResourceRequest;
import software.amazon.awssdk.services.appintegrations.model.UntagResourceResponse;
import software.amazon.awssdk.services.appintegrations.model.UpdateEventIntegrationRequest;
import software.amazon.awssdk.services.appintegrations.model.UpdateEventIntegrationResponse;
import software.amazon.awssdk.services.appintegrations.transform.CreateEventIntegrationRequestMarshaller;
import software.amazon.awssdk.services.appintegrations.transform.DeleteEventIntegrationRequestMarshaller;
import software.amazon.awssdk.services.appintegrations.transform.GetEventIntegrationRequestMarshaller;
import software.amazon.awssdk.services.appintegrations.transform.ListEventIntegrationAssociationsRequestMarshaller;
import software.amazon.awssdk.services.appintegrations.transform.ListEventIntegrationsRequestMarshaller;
import software.amazon.awssdk.services.appintegrations.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.appintegrations.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.appintegrations.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.appintegrations.transform.UpdateEventIntegrationRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link AppIntegrationsAsyncClient}.
 *
 * @see AppIntegrationsAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultAppIntegrationsAsyncClient implements AppIntegrationsAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultAppIntegrationsAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultAppIntegrationsAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * The Amazon AppIntegrations APIs are in preview release and are subject to change.
     * </p>
     * <p>
     * Creates an EventIntegration, given a specified name, description, and a reference to an Amazon Eventbridge bus in
     * your account and a partner event source that will push events to that bus. No objects are created in the your
     * account, only metadata that is persisted on the EventIntegration control plane.
     * </p>
     *
     * @param createEventIntegrationRequest
     * @return A Java Future containing the result of the CreateEventIntegration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceErrorException Request processing failed due to an error or failure with the service.</li>
     *         <li>ResourceQuotaExceededException The allowed quota for the resource has been exceeded.</li>
     *         <li>DuplicateResourceException A resource with the specified name already exists.</li>
     *         <li>ThrottlingException The throttling limit has been exceeded.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppIntegrationsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AppIntegrationsAsyncClient.CreateEventIntegration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appintegrations-2020-07-29/CreateEventIntegration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateEventIntegrationResponse> createEventIntegration(
            CreateEventIntegrationRequest createEventIntegrationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createEventIntegrationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppIntegrations");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateEventIntegration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateEventIntegrationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateEventIntegrationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateEventIntegrationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateEventIntegrationRequest, CreateEventIntegrationResponse>()
                            .withOperationName("CreateEventIntegration")
                            .withMarshaller(new CreateEventIntegrationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createEventIntegrationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createEventIntegrationRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<CreateEventIntegrationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The Amazon AppIntegrations APIs are in preview release and are subject to change.
     * </p>
     * <p>
     * Deletes the specified existing event integration. If the event integration is associated with clients, the
     * request is rejected.
     * </p>
     *
     * @param deleteEventIntegrationRequest
     * @return A Java Future containing the result of the DeleteEventIntegration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceErrorException Request processing failed due to an error or failure with the service.</li>
     *         <li>ThrottlingException The throttling limit has been exceeded.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppIntegrationsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AppIntegrationsAsyncClient.DeleteEventIntegration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appintegrations-2020-07-29/DeleteEventIntegration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteEventIntegrationResponse> deleteEventIntegration(
            DeleteEventIntegrationRequest deleteEventIntegrationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteEventIntegrationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppIntegrations");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteEventIntegration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteEventIntegrationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteEventIntegrationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteEventIntegrationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteEventIntegrationRequest, DeleteEventIntegrationResponse>()
                            .withOperationName("DeleteEventIntegration")
                            .withMarshaller(new DeleteEventIntegrationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteEventIntegrationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteEventIntegrationRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<DeleteEventIntegrationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The Amazon AppIntegrations APIs are in preview release and are subject to change.
     * </p>
     * <p>
     * Return information about the event integration.
     * </p>
     *
     * @param getEventIntegrationRequest
     * @return A Java Future containing the result of the GetEventIntegration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceErrorException Request processing failed due to an error or failure with the service.</li>
     *         <li>ThrottlingException The throttling limit has been exceeded.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppIntegrationsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AppIntegrationsAsyncClient.GetEventIntegration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appintegrations-2020-07-29/GetEventIntegration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetEventIntegrationResponse> getEventIntegration(
            GetEventIntegrationRequest getEventIntegrationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getEventIntegrationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppIntegrations");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetEventIntegration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetEventIntegrationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetEventIntegrationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetEventIntegrationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetEventIntegrationRequest, GetEventIntegrationResponse>()
                            .withOperationName("GetEventIntegration")
                            .withMarshaller(new GetEventIntegrationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getEventIntegrationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getEventIntegrationRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<GetEventIntegrationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The Amazon AppIntegrations APIs are in preview release and are subject to change.
     * </p>
     * <p>
     * Returns a paginated list of event integration associations in the account.
     * </p>
     *
     * @param listEventIntegrationAssociationsRequest
     * @return A Java Future containing the result of the ListEventIntegrationAssociations operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceErrorException Request processing failed due to an error or failure with the service.</li>
     *         <li>ThrottlingException The throttling limit has been exceeded.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppIntegrationsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AppIntegrationsAsyncClient.ListEventIntegrationAssociations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appintegrations-2020-07-29/ListEventIntegrationAssociations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListEventIntegrationAssociationsResponse> listEventIntegrationAssociations(
            ListEventIntegrationAssociationsRequest listEventIntegrationAssociationsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listEventIntegrationAssociationsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppIntegrations");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListEventIntegrationAssociations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListEventIntegrationAssociationsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, ListEventIntegrationAssociationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListEventIntegrationAssociationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListEventIntegrationAssociationsRequest, ListEventIntegrationAssociationsResponse>()
                            .withOperationName("ListEventIntegrationAssociations")
                            .withMarshaller(new ListEventIntegrationAssociationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listEventIntegrationAssociationsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listEventIntegrationAssociationsRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<ListEventIntegrationAssociationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The Amazon AppIntegrations APIs are in preview release and are subject to change.
     * </p>
     * <p>
     * Returns a paginated list of event integrations in the account.
     * </p>
     *
     * @param listEventIntegrationsRequest
     * @return A Java Future containing the result of the ListEventIntegrations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceErrorException Request processing failed due to an error or failure with the service.</li>
     *         <li>ThrottlingException The throttling limit has been exceeded.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppIntegrationsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AppIntegrationsAsyncClient.ListEventIntegrations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appintegrations-2020-07-29/ListEventIntegrations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListEventIntegrationsResponse> listEventIntegrations(
            ListEventIntegrationsRequest listEventIntegrationsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listEventIntegrationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppIntegrations");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListEventIntegrations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListEventIntegrationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListEventIntegrationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListEventIntegrationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListEventIntegrationsRequest, ListEventIntegrationsResponse>()
                            .withOperationName("ListEventIntegrations")
                            .withMarshaller(new ListEventIntegrationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listEventIntegrationsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listEventIntegrationsRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<ListEventIntegrationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The Amazon AppIntegrations APIs are in preview release and are subject to change.
     * </p>
     * <p>
     * Lists the tags for the specified resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>InternalServiceErrorException Request processing failed due to an error or failure with the service.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>ThrottlingException The throttling limit has been exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppIntegrationsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AppIntegrationsAsyncClient.ListTagsForResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appintegrations-2020-07-29/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppIntegrations");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsForResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listTagsForResourceRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The Amazon AppIntegrations APIs are in preview release and are subject to change.
     * </p>
     * <p>
     * Adds the specified tags to the specified resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>InternalServiceErrorException Request processing failed due to an error or failure with the service.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>ThrottlingException The throttling limit has been exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppIntegrationsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AppIntegrationsAsyncClient.TagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appintegrations-2020-07-29/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppIntegrations");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = tagResourceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The Amazon AppIntegrations APIs are in preview release and are subject to change.
     * </p>
     * <p>
     * Removes the specified tags from the specified resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>InternalServiceErrorException Request processing failed due to an error or failure with the service.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>ThrottlingException The throttling limit has been exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppIntegrationsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AppIntegrationsAsyncClient.UntagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appintegrations-2020-07-29/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppIntegrations");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = untagResourceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The Amazon AppIntegrations APIs are in preview release and are subject to change.
     * </p>
     * <p>
     * Updates the description of an event integration.
     * </p>
     *
     * @param updateEventIntegrationRequest
     * @return A Java Future containing the result of the UpdateEventIntegration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceErrorException Request processing failed due to an error or failure with the service.</li>
     *         <li>ThrottlingException The throttling limit has been exceeded.</li>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppIntegrationsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AppIntegrationsAsyncClient.UpdateEventIntegration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appintegrations-2020-07-29/UpdateEventIntegration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateEventIntegrationResponse> updateEventIntegration(
            UpdateEventIntegrationRequest updateEventIntegrationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateEventIntegrationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppIntegrations");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateEventIntegration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateEventIntegrationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateEventIntegrationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateEventIntegrationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateEventIntegrationRequest, UpdateEventIntegrationResponse>()
                            .withOperationName("UpdateEventIntegration")
                            .withMarshaller(new UpdateEventIntegrationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateEventIntegrationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateEventIntegrationRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<UpdateEventIntegrationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(AppIntegrationsException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServiceError")
                                .exceptionBuilderSupplier(InternalServiceErrorException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestException")
                                .exceptionBuilderSupplier(InvalidRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DuplicateResourceException")
                                .exceptionBuilderSupplier(DuplicateResourceException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceQuotaExceededException")
                                .exceptionBuilderSupplier(ResourceQuotaExceededException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
